/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* 
 * e-contact-editor-categories.c
 * Copyright (C) 2000  Helix Code, Inc.
 * Author: Chris Lahey <clahey@helixcode.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include "gal/util/e-i18n.h"
#include <string.h>
#include <gdk/gdkkeysyms.h>
#include <libgnomeui/gnome-stock.h>

#include "gal/e-table/e-table-scrolled.h"
#include "gal/e-table/e-table.h"
#include "gal/e-table/e-table-simple.h"
#include "gal/widgets/e-unicode.h"
#include "gal/util/e-i18n.h"

#include "e-categories.h"

static void e_categories_init		(ECategories		 *card);
static void e_categories_class_init	(ECategoriesClass	 *klass);
static void e_categories_set_arg         (GtkObject *o, GtkArg *arg, guint arg_id);
static void e_categories_get_arg         (GtkObject *object, GtkArg *arg, guint arg_id);
static void e_categories_destroy         (GtkObject *object);
static int e_categories_col_count (ETableModel *etc, gpointer data);
static int e_categories_row_count (ETableModel *etc, gpointer data);
static void *e_categories_value_at (ETableModel *etc, int col, int row, gpointer data);
static void e_categories_set_value_at (ETableModel *etc, int col, int row, const void *val, gpointer data);
static gboolean e_categories_is_cell_editable (ETableModel *etc, int col, int row, gpointer data);
static void *e_categories_duplicate_value (ETableModel *etc, int col, const void *value, gpointer data);
static void e_categories_free_value (ETableModel *etc, int col, void *value, gpointer data);
static void *e_categories_initialize_value (ETableModel *etc, int col, gpointer data);
static gboolean e_categories_value_is_empty (ETableModel *etc, int col, const void *value, gpointer data);
static char * e_categories_value_to_string (ETableModel *etc, int col, const void *value, gpointer data);
static void e_categories_toggle (ECategories *categories, int row);

static GnomeDialogClass *parent_class = NULL;

/* The arguments we take */
enum {
	ARG_0,
	ARG_CATEGORIES,
	ARG_HEADER,
};

GtkType
e_categories_get_type (void)
{
	static GtkType contact_editor_categories_type = 0;

	if (!contact_editor_categories_type)
		{
			static const GtkTypeInfo contact_editor_categories_info =
			{
				"ECategories",
				sizeof (ECategories),
				sizeof (ECategoriesClass),
				(GtkClassInitFunc) e_categories_class_init,
				(GtkObjectInitFunc) e_categories_init,
				/* reserved_1 */ NULL,
				/* reserved_2 */ NULL,
				(GtkClassInitFunc) NULL,
			};

			contact_editor_categories_type = gtk_type_unique (gnome_dialog_get_type (), &contact_editor_categories_info);
		}

	return contact_editor_categories_type;
}

static void
e_categories_class_init (ECategoriesClass *klass)
{
	GtkObjectClass *object_class;
	GnomeDialogClass *dialog_class;

	object_class = (GtkObjectClass*) klass;
	dialog_class = (GnomeDialogClass *) klass;

	parent_class = gtk_type_class (gnome_dialog_get_type ());

	gtk_object_add_arg_type ("ECategories::categories", GTK_TYPE_STRING,
				 GTK_ARG_READWRITE, ARG_CATEGORIES);
	gtk_object_add_arg_type ("ECategories::header", GTK_TYPE_STRING,
				 GTK_ARG_READWRITE, ARG_HEADER);
 
	object_class->set_arg = e_categories_set_arg;
	object_class->get_arg = e_categories_get_arg;
	object_class->destroy = e_categories_destroy;
}

gchar *builtin_categories[] = {
	N_("Business"),
	N_("Competition"),
	N_("Favorites"),
	N_("Gifts"),
	N_("Goals/Objectives"),
	N_("Holiday"),
	N_("Holiday Cards"),
	N_("Hot Contacts"),
	N_("Ideas"),
	N_("International"),
	N_("Key Customer"),
	N_("Miscellaneous"),
	N_("Personal"),
	N_("Phone Calls"),
	N_("Status"),
	N_("Strategies"),
	N_("Suppliers"),
	N_("Time & Expenses"),
	N_("VIP"),
	N_("Waiting"),
};

#define BUILTIN_CATEGORY_COUNT (sizeof(builtin_categories) / sizeof(builtin_categories[0]))

static void
add_list_unique(ECategories *categories, char *string)
{
	int k;
	char *temp = e_strdup_strip(string);
	char **list = categories->category_list;

	if (!*temp) {
		g_free(temp);
		return;
	}
	for (k = 0; k < categories->list_length; k++) {
		if (!strcmp(list[k], temp)) {
			categories->selected_list[k] = TRUE;
			break;
		}
	}
	if (k == categories->list_length) {
		categories->selected_list[categories->list_length] = TRUE;
		list[categories->list_length++] = temp;
	} else {
		g_free(temp);
	}
}

static void
do_parse_categories(ECategories *categories)
{
	char *str = categories->categories;
	int length = strlen(str);
	char *copy = g_new(char, length + 1);
	int i, j;
	char **list;
	int count = 1;

	e_table_model_pre_change(categories->model);

	for (i = 0; i < categories->list_length; i++)
		g_free(categories->category_list[i]);
	g_free(categories->category_list);
	g_free(categories->selected_list);

	for (i = 0; str[i]; i++) {
		switch (str[i]) {
		case '\\':
			i++;
			if (!str[i])
				i--;
			break;
		case ',':
			count ++;
			break;
		}
	}
	list = g_new(char *, count + 1 + BUILTIN_CATEGORY_COUNT);
	categories->category_list = list;

	categories->selected_list = g_new(gboolean, count + 1 + BUILTIN_CATEGORY_COUNT);

	for (count = 0; count < BUILTIN_CATEGORY_COUNT; count++) {
		list[count] = e_utf8_from_locale_string(_(builtin_categories[count]));
		categories->selected_list[count] = 0;
	}

	categories->list_length = count;

	for (i = 0, j = 0; str[i]; i++, j++) {
		switch (str[i]) {
		case '\\':
			i++;
			if (str[i]) {
				copy[j] = str[i];
			} else
				i--;
			break;
		case ',':
			copy[j] = 0;
			add_list_unique(categories, copy);
			j = -1;
			break;
		default:
			copy[j] = str[i];
			break;
		}
	}
	copy[j] = 0;
	add_list_unique(categories, copy);
	g_free(copy);
	e_table_model_changed(categories->model);
}

static void
e_categories_entry_change (GtkWidget *entry, 
			   ECategories *categories)
{
	g_free(categories->categories);
	categories->categories = e_utf8_gtk_entry_get_text(GTK_ENTRY(entry));
	do_parse_categories(categories);
}

static void 
ec_set_categories (GtkWidget *entry, const char *categories)
{
	e_utf8_gtk_entry_set_text(GTK_ENTRY(entry), categories);
}

static void 
ec_set_header (ECategories *e_categories, const char *header)
{
	GtkWidget *widget;
	widget = glade_xml_get_widget(e_categories->gui, "label-header");
	if (widget && GTK_IS_LABEL(widget))
		gtk_label_set_text(GTK_LABEL(widget), header);
}


#define INITIAL_SPEC "<ETableSpecification no-headers=\"true\" draw-grid=\"true\" cursor-mode=\"line\">\
  <ETableColumn model_col=\"0\" _title=\" \" expansion=\"0.0\" minimum_width=\"20\" resizable=\"false\" cell=\"checkbox\"       compare=\"integer\"/> \
  <ETableColumn model_col=\"1\" _title=\"Category\" expansion=\"1.0\" minimum_width=\"20\" resizable=\"true\" cell=\"string\" compare=\"string\"/> \
        <ETableState> \
		<column source=\"0\"/>						\
		<column source=\"1\"/>						\
        	<grouping> <leaf column=\"1\" ascending=\"true\"/> </grouping>	\
        </ETableState> \
</ETableSpecification>"

static gint
table_key_press (ETable *table, int row, int col, GdkEvent *event, ECategories *categories)
{
	switch (event->key.keyval) {
	case GDK_space:
		e_categories_toggle(categories, row);
		return TRUE;
		break;
	}
	return FALSE;
}

static void
e_categories_init (ECategories *categories)
{
	GladeXML *gui;
	GtkWidget *table;
	GtkWidget *e_table;
	ETable *table_object;

	categories->list_length = 0;
	categories->category_list = NULL;
	categories->selected_list = NULL;
	categories->categories = NULL;

	gnome_dialog_append_button ( GNOME_DIALOG(categories),
				     GNOME_STOCK_BUTTON_OK);
	
	gnome_dialog_append_button ( GNOME_DIALOG(categories),
				     GNOME_STOCK_BUTTON_CANCEL);

	gnome_dialog_set_default (GNOME_DIALOG (categories), 0);

	gtk_window_set_policy(GTK_WINDOW(categories), FALSE, TRUE, FALSE);

	gui = glade_xml_new_with_domain (GAL_GLADEDIR "/gal-categories.glade", NULL, PACKAGE);
	categories->gui = gui;

	table = glade_xml_get_widget(gui, "table-categories");
	gtk_widget_ref(table);
	gtk_container_remove(GTK_CONTAINER(table->parent), table);
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (categories)->vbox), table, TRUE, TRUE, 0);
	gtk_widget_unref(table);

	categories->entry = glade_xml_get_widget(gui, "entry-categories");
	
	gtk_signal_connect(GTK_OBJECT(categories->entry), "changed",
			   GTK_SIGNAL_FUNC(e_categories_entry_change), categories);

	categories->model = e_table_simple_new(e_categories_col_count, 
					       e_categories_row_count,
					       e_categories_value_at,
					       e_categories_set_value_at,
					       e_categories_is_cell_editable,
					       e_categories_duplicate_value,
					       e_categories_free_value,
					       e_categories_initialize_value,
					       e_categories_value_is_empty,
					       e_categories_value_to_string,
					       categories);

	e_table = e_table_scrolled_new (categories->model, NULL, INITIAL_SPEC, NULL);

	table_object = e_table_scrolled_get_table(E_TABLE_SCROLLED(e_table));
	gtk_signal_connect(GTK_OBJECT(table_object), "key_press",
			   GTK_SIGNAL_FUNC(table_key_press), categories);

	gtk_object_sink(GTK_OBJECT(categories->model));
	
	gtk_widget_show(e_table);

	gtk_table_attach_defaults(GTK_TABLE(table),
				  e_table, 
				  0, 1,
				  3, 4);

	gtk_widget_grab_focus (categories->entry);
	gnome_dialog_editable_enters (GNOME_DIALOG (categories), GTK_EDITABLE (categories->entry));
}

void
e_categories_destroy (GtkObject *object)
{
	ECategories *categories = E_CATEGORIES(object);
	int i;

	if (categories->gui)
		gtk_object_unref(GTK_OBJECT(categories->gui));

	g_free(categories->categories);
	for (i = 0; i < categories->list_length; i++)
		g_free(categories->category_list[i]);
	g_free(categories->category_list);
	g_free(categories->selected_list);
}

/**
 * e_categories_construct:
 * @categories: An uninitialized %ECategories widget.
 * @initial_category_list: Comma-separated list of initial categories.
 * 
 * Construct the @categories object.
 **/
void
e_categories_construct (ECategories *categories,
			const char *initial_category_list)
{
	g_return_if_fail (categories != NULL);
	g_return_if_fail (E_IS_CATEGORIES (categories));
	g_return_if_fail (initial_category_list != NULL);

	ec_set_categories (categories->entry, initial_category_list);

	gtk_window_set_default_size (GTK_WINDOW (categories), 200, 400);
}

/**
 * e_categories_new:
 * @initial_category_list: Comma-separated list of initial categories.
 * 
 * Create a new %ECategories widget.
 * 
 * Return value: A pointer to the newly created %ECategories widget.
 **/
GtkWidget*
e_categories_new (const char *initial_category_list)
{
	GtkWidget *widget;

	widget = GTK_WIDGET (gtk_type_new (e_categories_get_type ()));

	e_categories_construct (E_CATEGORIES (widget), initial_category_list);

	return widget;
}

static void
e_categories_set_arg (GtkObject *o, GtkArg *arg, guint arg_id)
{
	ECategories *e_categories;

	e_categories = E_CATEGORIES (o);
	
	switch (arg_id){
	case ARG_CATEGORIES:
		ec_set_categories (e_categories->entry, GTK_VALUE_STRING (*arg));
		break;

	case ARG_HEADER:
		ec_set_header (e_categories, GTK_VALUE_STRING (*arg));
		break;
	}
}

static void
e_categories_get_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	ECategories *e_categories;
	GtkWidget *widget;
	char *header;

	e_categories = E_CATEGORIES (object);

	switch (arg_id) {
	case ARG_CATEGORIES:
		GTK_VALUE_STRING (*arg) = g_strdup(e_categories->categories);
		break;

	case ARG_HEADER:
		widget = glade_xml_get_widget(e_categories->gui, "label-header");
		if (widget && GTK_IS_LABEL(widget)) {
			gtk_object_get(GTK_OBJECT(widget),
				       "label", &header,
				       NULL);
			GTK_VALUE_STRING (*arg) = header;
		} else
			GTK_VALUE_STRING (*arg) = NULL;
		break;
	default:
		arg->type = GTK_TYPE_INVALID;
		break;
	}
}

/* This function returns the number of columns in our ETableModel. */
static int
e_categories_col_count (ETableModel *etc, gpointer data)
{
	return 2;
}

/* This function returns the number of rows in our ETableModel. */
static int
e_categories_row_count (ETableModel *etc, gpointer data)
{
	ECategories *categories = E_CATEGORIES(data);
	return categories->list_length;
}

/* This function returns the value at a particular point in our ETableModel. */
static void *
e_categories_value_at (ETableModel *etc, int col, int row, gpointer data)
{
	ECategories *categories = E_CATEGORIES(data);
	if (col == 0)
		return (void *) categories->selected_list[row];
	else
		return categories->category_list[row];
}

static void
e_categories_rebuild (ECategories *categories)
{
	char **strs;
	int i, j;
	char *string;

	strs = g_new(char *, categories->list_length + 1);
	for (i = 0, j = 0; i < categories->list_length; i++) {
		if (categories->selected_list[i])
			strs[j++] = categories->category_list[i];
	}
	strs[j] = 0;
	string = g_strjoinv(", ", strs);
	e_utf8_gtk_entry_set_text(GTK_ENTRY(categories->entry), string);
	g_free(string);
	g_free(strs);
}

static void
e_categories_toggle (ECategories *categories, int row)
{
	categories->selected_list[row] = !categories->selected_list[row];
	e_categories_rebuild (categories);
}

/* This function sets the value at a particular point in our ETableModel. */
static void
e_categories_set_value_at (ETableModel *etc, int col, int row, const void *val, gpointer data)
{
	ECategories *categories = E_CATEGORIES(data);
	if ( col == 0 ) {
		categories->selected_list[row] = (gboolean) val;
		e_categories_rebuild (categories);
	}
	if ( col == 1 )
		return;
}

/* This function returns whether a particular cell is editable. */
static gboolean
e_categories_is_cell_editable (ETableModel *etc, int col, int row, gpointer data)
{
	return col == 0;
}

/* This function duplicates the value passed to it. */
static void *
e_categories_duplicate_value (ETableModel *etc, int col, const void *value, gpointer data)
{
	if (col == 0)
		return (void *)value;
	else
		return g_strdup(value);
}

/* This function frees the value passed to it. */
static void
e_categories_free_value (ETableModel *etc, int col, void *value, gpointer data)
{
	if (col == 0)
		return;
	else
		g_free(value);
}

static void *
e_categories_initialize_value (ETableModel *etc, int col, gpointer data)
{
	if (col == 0)
		return NULL;
	else
		return g_strdup("");
}

static gboolean
e_categories_value_is_empty (ETableModel *etc, int col, const void *value, gpointer data)
{
	if (col == 0)
		return value == NULL;
	else
		return !(value && *(char *)value);
}

static char *
e_categories_value_to_string (ETableModel *etc, int col, const void *value, gpointer data)
{
	if (col == 0)
		return g_strdup_printf("%d", (int) value);
	else
		return g_strdup(value);
}
