/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * e-categories-master-list.c: the master list of categories.
 *
 * Authors:
 *   Chris Lahey <clahey@ximian.com>
 *
 * (C) 1999, 2000 Ximian, Inc.
 */
#include <config.h>

#include "e-categories-master-list-array.h"

#include <gal/util/e-i18n.h>
#include <gal/util/e-xml-utils.h>
#include <gnome-xml/tree.h>
#include <gnome-xml/parser.h>
#include <gnome-xml/xmlmemory.h>
#include "e-unicode.h"
#include <string.h>

#define PARENT_TYPE e_categories_master_list_get_type ()

#define d(x)

typedef enum {
	E_CATEGORIES_MASTER_LIST_ARRAY_NORMAL,
	E_CATEGORIES_MASTER_LIST_ARRAY_DELETED,
} ECategoriesMasterListArrayState;

struct _ECategoriesMasterListArrayPriv {
	char                            **categories;
	int                               count;

	ECategoriesMasterListArrayState  *states;
	char                            **appends;
	int                               appends_count;
};

static ECategoriesMasterListClass *parent_class;

gchar *builtin_categories[] = {
	N_("Business"),
	N_("Competition"),
	N_("Favorites"),
	N_("Gifts"),
	N_("Goals/Objectives"),
	N_("Holiday"),
	N_("Holiday Cards"),
	N_("Hot Contacts"),
	N_("Ideas"),
	N_("International"),
	N_("Key Customer"),
	N_("Miscellaneous"),
	N_("Personal"),
	N_("Phone Calls"),
	N_("Status"),
	N_("Strategies"),
	N_("Suppliers"),
	N_("Time & Expenses"),
	N_("VIP"),
	N_("Waiting"),
};

#define BUILTIN_CATEGORY_COUNT (sizeof(builtin_categories) / sizeof(builtin_categories[0]))

static void
ecmla_clear (ECategoriesMasterListArray *ecmla)
{
	int i;
	g_free (ecmla->priv->states);
	ecmla->priv->states = NULL;

	for (i = 0; i < ecmla->priv->appends_count; i++)
		g_free (ecmla->priv->appends[i]);
	g_free (ecmla->priv->appends);
	ecmla->priv->appends = NULL;
	ecmla->priv->appends_count = 0;

	for (i = 0; i < ecmla->priv->count; i++)
		g_free (ecmla->priv->categories[i]);
	g_free (ecmla->priv->categories);
	ecmla->priv->categories = NULL;
	ecmla->priv->count = 0;
}

/**
 * ecmla_count:
 * @ecmla: The e-categories-master-list to operate on
 *
 * Returns: the number of categories in the list.
 */
static int
ecmla_count (ECategoriesMasterList *ecml)
{
	ECategoriesMasterListArray *ecmla = E_CATEGORIES_MASTER_LIST_ARRAY (ecml);

	return ecmla->priv->count;
}

/**
 * ecmla_nth:
 * @ecml: the e-categories-master-list to operate on
 * @n: The category to return.
 *
 * Return value: This function returns the nth category in the list.
 */
static char *
ecmla_nth (ECategoriesMasterList *ecml, int n)
{
	ECategoriesMasterListArray *ecmla = E_CATEGORIES_MASTER_LIST_ARRAY (ecml);

	g_return_val_if_fail (n < ecmla->priv->count, NULL);
	g_return_val_if_fail (n >= 0, NULL);

	return ecmla->priv->categories[n];
}

/**
 * ecmla_append:
 * @ecml: the master list to append to
 * @category: The category to append
 */
static void
ecmla_append (ECategoriesMasterList *ecml, const char *category)
{
	ECategoriesMasterListArray *ecmla = E_CATEGORIES_MASTER_LIST_ARRAY (ecml);

	ecmla->priv->appends = g_renew(char *, ecmla->priv->appends, ecmla->priv->appends_count + 1);
	ecmla->priv->appends[ecmla->priv->appends_count] = g_strdup (category);
	ecmla->priv->appends_count ++;
}

/**
 * ecmla_delete:
 * @ecml: the master list to remove from.
 * @n: the item to delete.
 */
static void
ecmla_delete (ECategoriesMasterList *ecml, int n) 
{
	ECategoriesMasterListArray *ecmla = E_CATEGORIES_MASTER_LIST_ARRAY (ecml);
	int i;

	g_return_if_fail (n < ecmla->priv->count);
	g_return_if_fail (n >= 0);

	if (ecmla->priv->states == NULL) {
		ecmla->priv->states = g_new(ECategoriesMasterListArrayState, ecmla->priv->count);
		for (i = 0; i < ecmla->priv->count; i++)
			ecmla->priv->states[i] = E_CATEGORIES_MASTER_LIST_ARRAY_NORMAL;
	}
	ecmla->priv->states[n] = E_CATEGORIES_MASTER_LIST_ARRAY_DELETED;
}

/**
 * ecmla_commit:
 * @ecml: the master list to remove from.
 */
static void
ecmla_commit (ECategoriesMasterList *ecml)
{
	ECategoriesMasterListArray *ecmla = E_CATEGORIES_MASTER_LIST_ARRAY (ecml);
	int count;
	int i;
	int j;
	char **new_list;

	if (ecmla->priv->states != NULL) {
		count = 0;
		for (i = 0; i < ecmla->priv->count; i++) {
			if (ecmla->priv->states[i] == E_CATEGORIES_MASTER_LIST_ARRAY_NORMAL)
				count ++;
		}
	} else {
		count = ecmla->priv->count;
	}

	count += ecmla->priv->appends_count;

	new_list = g_new (char *, count);

	j = 0;
	for (i = 0; i < ecmla->priv->count; i++) {
		if (ecmla->priv->states == NULL || ecmla->priv->states[i] == E_CATEGORIES_MASTER_LIST_ARRAY_NORMAL)
			new_list[j++] = ecmla->priv->categories[i];
		else
			g_free (ecmla->priv->categories[i]);
	}
	for (i = 0; i < ecmla->priv->appends_count; i++ )
		new_list[j++] = ecmla->priv->appends[i];

	g_free (ecmla->priv->categories);
	g_free (ecmla->priv->states);
	g_free (ecmla->priv->appends);
	ecmla->priv->states = NULL;
	ecmla->priv->appends = NULL;
	ecmla->priv->appends_count = 0;

	ecmla->priv->categories = new_list;
	ecmla->priv->count = j;

	e_categories_master_list_changed (ecml);
}


/**
 * ecmla_reset:
 * @ecml: the master list to reset.
 */
static void
ecmla_reset (ECategoriesMasterList *ecml)
{
	ECategoriesMasterListArray *ecmla = E_CATEGORIES_MASTER_LIST_ARRAY (ecml);
	int i;

	ecmla_clear (ecmla);

	ecmla->priv->count = BUILTIN_CATEGORY_COUNT;
	ecmla->priv->categories = g_new (char *, BUILTIN_CATEGORY_COUNT);

	for (i = 0; i < BUILTIN_CATEGORY_COUNT; i++) {
		ecmla->priv->categories[i] = e_utf8_from_locale_string(_(builtin_categories[i]));
	}
}

static void
ecmla_destroy (GtkObject *object)
{
	ECategoriesMasterListArray *ecmla = E_CATEGORIES_MASTER_LIST_ARRAY (object);

	ecmla_clear (ecmla);
	g_free (ecmla->priv);
	ecmla->priv = NULL;
}


static void
ecmla_class_init (GtkObjectClass *object_class)
{
	ECategoriesMasterListClass *ecml_class = E_CATEGORIES_MASTER_LIST_CLASS(object_class);

	parent_class          = gtk_type_class (PARENT_TYPE);

	ecml_class->count     = ecmla_count ;
	ecml_class->nth       = ecmla_nth   ;
	ecml_class->append    = ecmla_append;
	ecml_class->delete    = ecmla_delete;
	ecml_class->commit    = ecmla_commit;

	ecml_class->reset     = ecmla_reset ;

	object_class->destroy = ecmla_destroy;
}

static void
ecmla_init (ECategoriesMasterListArray *ecmla)
{
	int i;

	ecmla->priv = g_new (ECategoriesMasterListArrayPriv, 1);
	ecmla->priv->count = BUILTIN_CATEGORY_COUNT;
	ecmla->priv->categories = g_new (char *, BUILTIN_CATEGORY_COUNT);
	ecmla->priv->states = NULL;
	ecmla->priv->appends = NULL;
	ecmla->priv->appends_count = 0;

	for (i = 0; i < BUILTIN_CATEGORY_COUNT; i++) {
		ecmla->priv->categories[i] = e_utf8_from_locale_string(_(builtin_categories[i]));
	}
}

guint
e_categories_master_list_array_get_type (void)
{
	static guint type = 0;
	
	if (!type) {
		GtkTypeInfo info = {
			"ECategoriesMasterListArray",
			sizeof (ECategoriesMasterListArray),
			sizeof (ECategoriesMasterListArrayClass),
			(GtkClassInitFunc) ecmla_class_init,
			(GtkObjectInitFunc) ecmla_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		type = gtk_type_unique (PARENT_TYPE, &info);
	}

	return type;
}

ECategoriesMasterList *
e_categories_master_list_array_new       (void)
{
	return E_CATEGORIES_MASTER_LIST (gtk_type_new (e_categories_master_list_array_get_type ()));
}

void
e_categories_master_list_array_from_string (ECategoriesMasterListArray *ecmla,
					    const char *string)
{
	xmlDoc *doc;
	xmlNode *node;
	xmlNode *children;
	int count;
	int i;
	char **categories;

	char *string_copy = g_strdup (string);
	doc = xmlParseMemory(string_copy, strlen (string_copy));
	node = xmlDocGetRootElement (doc);
	g_free (string_copy);

	count = 0;

	for (children = node->xmlChildrenNode;
	     children;
	     children = children->next) {
		count ++;
	}

	categories = g_new (char *, count);
	i = 0;

	for (children = node->xmlChildrenNode;
	     children;
	     children = children->next) {
		categories[i++] = e_xml_get_string_prop_by_name (children,
								 "a");
	}

	ecmla_clear (ecmla);
	ecmla->priv->count = count;
	ecmla->priv->categories = categories;

	e_categories_master_list_changed (E_CATEGORIES_MASTER_LIST(ecmla));

	xmlFreeDoc (doc);
}

char *
e_categories_master_list_array_to_string (ECategoriesMasterListArray *ecmla)
{
	xmlDoc *doc;
	xmlNode *node;
	xmlNode *child;
	int i;
	char *string;
	xmlChar *temp;
	int length;

	doc = xmlNewDoc (XML_DEFAULT_VERSION);
	node = xmlNewNode (NULL, "catlist");
	xmlDocSetRootElement (doc, node);

	for (i = 0; i < ecmla->priv->count; i++) {
		child = xmlNewChild (node, NULL, "cat", NULL);
		e_xml_set_string_prop_by_name (child, "a", ecmla->priv->categories[i]);
	}
	xmlDocDumpMemory (doc, &temp, &length);
	string = g_strdup (temp);
	xmlFree (temp);
	return string;
}
