import gettext

import gtk
import gobject

from gazpacho.commandmanager import command_manager
from gazpacho.dialogs import BaseDialog
from gazpacho.environ import environ
from gazpacho.kiwiutils import gsignal
from gazpacho.sizegroup import GSizeGroup
from gazpacho.util import select_iter
from gazpacho.widget import Widget

_ = gettext.gettext


EDITOR_OBJECT_COLUMN, EDITOR_CALLBACK_IDS_COLUMN = range(2)

class SizeGroupView(gtk.ScrolledWindow):
    """
    The sizegroup view/editor that displays the sizegroup tree.
    """

    gsignal('selection-changed', object)

    def __init__(self, app):
        """
        Initialize the size group editor.

        @param app: the application
        @type app: gazpacho.application.Application
        """
        gtk.ScrolledWindow.__init__(self)
        self.set_shadow_type(gtk.SHADOW_IN)

        self._app = app
        self.project = None
        self._project_callback_ids = ()        
        self._mode_icons = self._load_mode_icons()

        self._treeview = self._create_treeview()
        self._model = gtk.TreeStore(object, object)
        self._treeview.set_model(self._model)

        # Connect the callbacks
        self._treeview.connect('button-press-event',
                               self._on_treeview_button_press_event)
        self._treeview.connect('key-press-event',
                               self._on_treeview_key_press_event)
        selection = self._treeview.get_selection()
        selection.connect('changed', self._on_treeview_selection_changed)

        self.add(self._treeview)

    #
    # Private methods
    #
    
    def _create_treeview(self):
        """
        Create the TreeView.
        """
        treeview = gtk.TreeView()
        treeview.set_headers_visible(False)

        column = gtk.TreeViewColumn()

        renderer1 = gtk.CellRendererPixbuf()
        column.pack_start(renderer1, False)
        column.set_cell_data_func(renderer1, self._draw_cell_icon)
        
        renderer2 = gtk.CellRendererText()
        column.pack_start(renderer2)
        column.set_cell_data_func(renderer2, self._draw_cell_data)

        treeview.append_column(column)
        return treeview

    def _load_mode_icons(self):
        """
        Load the pixbufs that are to be used as sizegroup mode icons.

        Mode constants:
         - gtk.SIZE_GROUP_HORIZONTAL
         - gtk.SIZE_GROUP_VERTICAL
         - gtk.SIZE_GROUP_BOTH
        
        @return: a mapping of sizegroup mode constants to pixbufs
        @rtype: map
        """
        icon_names = {gtk.SIZE_GROUP_HORIZONTAL: 'horizontal.png',
                      gtk.SIZE_GROUP_VERTICAL: 'vertical.png',
                      gtk.SIZE_GROUP_BOTH: 'both.png'}
        icon_map = {}
        for key, name in icon_names.iteritems():
            pixbuf = None
            filename = environ.find_pixmap(name)
            if filename:
                pixbuf = gtk.gdk.pixbuf_new_from_file(filename)
            icon_map[key] = pixbuf
            
        return icon_map

    def _draw_cell_icon(self, tree_column, cell, model, model_iter):
        """
        Draw the sizegroup or widget icon.
        """
        row = model[model_iter]
        item = row[EDITOR_OBJECT_COLUMN]

        if isinstance(item, GSizeGroup):
            pixbuf = self._mode_icons.get(item.mode)
        else:
            pixbuf = item.adaptor.pixbuf

        cell.set_property('pixbuf', pixbuf)


    def _draw_cell_data(self, column, cell, model, model_iter):
        """
        Display the name of the sizegroup or widget
        """
        sizegroup = model[model_iter][EDITOR_OBJECT_COLUMN]
        text = sizegroup.name
        cell.set_property('text', text)


    def _fill_model_from_project(self):
        """
        Fill the view with data from the current project. This will
        add all sizegroups that belong to the project and also connect
        the necessary callbacks.

        Note, it is important that the project has been set befor
        calling this method.
        """
        sizegroup_iter = None
        for sizegroup in self.project.sizegroups:
            sizegroup_iter = self._add_sizegroup(sizegroup)

        if sizegroup_iter:
            select_iter(self._treeview, sizegroup_iter)

    def _add_sizegroup(self, sizegroup):
        """
        Add a sizegroup and its widgets to the treeview. This will the
        necessary callbacks as well.

        @param sizegroup: the sizegroup that should be added
        @type sizegroup: gazpacho.sizegroup.GSizeGroup

        @return: the iter pointing to the sizegroup that was added
        @rtype: gtk.TreeIter
        """
        id1 = sizegroup.connect('widgets-added',
                                self._on_sizegroup_widgets_added)
        id2 = sizegroup.connect('widgets-removed',
                                self._on_sizegroup_widgets_removed)
        id3 = sizegroup.connect('name-changed',
                                self._on_sizegroup_name_changed)
        
        sizegroup_iter = self._model.append(None, (sizegroup, (id1, id2, id3)))
        self._add_widgets(sizegroup_iter, sizegroup.get_widgets())
        return sizegroup_iter

    def _add_widgets(self, sizegroup_iter, widgets):
        """
        Add a number of widgets to the treeview. This will the
        necessary callbacks as well.

        @param sizegroup_iter: the parent iter where the widgets should be added
        @type sizegroup_iter: gtk.TreeIter
        @param widgets: the widgets that should be added
        @type widgets: list (of gazpacho.widget.Widget)
 
        @return: the iter pointing to the last widget that was added
        @rtype: gtk.TreeIter
        """
        sizegroup = self._model[sizegroup_iter][EDITOR_OBJECT_COLUMN]
        
        widget_iter = None
        for widget in widgets:
            callback_id = widget.connect('notify::name',
                                         self._on_widget_notify_name, sizegroup)
            row = (widget, (callback_id,))
            widget_iter = self._model.append(sizegroup_iter, row)

        return widget_iter

    def _disconnect_item_callbacks(self, model_iter):
        """
        Disconnect all callbacks for the item (sizegroup or widget)
        that the iter is pointing to.

        @param model_iter: the iter pointing to the item who's
                           callbacks should be disconnected
        @type model_iter: gtk.TreeIter        
        """
        row = self._model[model_iter]
        item = row[EDITOR_OBJECT_COLUMN]
        ids = row[EDITOR_CALLBACK_IDS_COLUMN]
        for callback_id in ids:
            item.disconnect(callback_id)

    def _disconnect_all_callbacks(self, sizegroup_iter):
        """
        Disconnect all callbacks for a sizegroup and all its widgets.

        @param sizegroup_iter: the iter pointing to the sizegroup
        who's callbacks should be disconnected
        @type sizegroup_iter: gtk.TreeIter        
        """
        widget_iter = self._model.iter_children(sizegroup_iter)
        while widget_iter:
            self._disconnect_item_callbacks(widget_iter)
            widget_iter = self._model.iter_next(widget_iter)
            
        self._disconnect_item_callbacks(sizegroup_iter)
    
    def _find_sizegroup(self, sizegroup):
        """
        Find a certain sizegroup in the TreeStore.

        @param sizegroup: the sizegroup that the widget belong to
        @type sizegroup: gazpacho.sizegroup.GSizeGroup

        @return: the iter pointing to the sizegroup or None if it wasn't found
        @rtype: gtk.TreeIter
        """
        model = self._model
        for sizegroup_row in model:
            if sizegroup_row[EDITOR_OBJECT_COLUMN] == sizegroup:
                return sizegroup_row.iter
        return None

    def _find_widget(self, sizegroup, widget):
        """
        Find a certain widget in the TreeStore.

        @param sizegroup: the sizegroup that the widget belong to
        @type sizegroup: gazpacho.sizegroup.GSizeGroup
        @param widget: the widget that we're looking for
        @type widget: gazpacho.widget.Widget

        @return: the iter pointing to the widget or None if it wasn't found
        @rtype: gtk.TreeIter
        """
        sizegroup_iter = self._find_sizegroup(sizegroup)
        if not sizegroup_iter:
            return None

        model = self._model
        for widget_row in model[sizegroup_iter].iterchildren():
            if widget_row[EDITOR_OBJECT_COLUMN] == widget:
                return widget_row.iter
        return None

    #
    # Editor callbacks
    #
    
    def _on_treeview_selection_changed(self, selection):
        """
        Callback for selection changes in the tree view.

        @param selection: the selection that has changed
        @type selection: gtk.TreeSelection
        """
        model, model_iter = selection.get_selected()
        item = None
        if model_iter:
            item = model[model_iter][EDITOR_OBJECT_COLUMN]
        self.emit('selection-changed', item)

    def _on_treeview_key_press_event(self, view, event):
        """
        Callback for handling key press events. Right now it's only
        used for deleting sizegroups and widgets.

        @param view: the sizegroup treeview
        @type view: gtk.TreeView
        @param event: the event that was triggered
        @type event: gtk.gdk.Event
        """
        if event.keyval in [gtk.keysyms.Delete, gtk.keysyms.KP_Delete]:
            self.remove_selected_item()

    def _on_treeview_button_press_event(self, view, event):
        """
        Callback for handling mouse clicks. It is used to show a
        context menu.

        @param view: the sizegroup treeview
        @type view: gtk.TreeView
        @param event: the event that was triggered
        @type event: gtk.gdk.Event
        """
        if event.button != 3:
            return False
    
        # No need for a context menu if there is no project
        if not self.project:
            return False
         
        result = view.get_path_at_pos(int(event.x), int(event.y))
        if not result:
            return
        path = result[0]

        # Select the row
        view.set_cursor(path)
        view.grab_focus()

        # Show popup
        row = self._model[path]
        if row.parent:
            sizegroup = row.parent[EDITOR_OBJECT_COLUMN]
            widget = row[EDITOR_OBJECT_COLUMN]
            popup = WidgetPopup(widget, sizegroup, self.project)            
        else:
            sizegroup = row[EDITOR_OBJECT_COLUMN]
            popup = SizeGroupPopup(sizegroup, self.project)

        popup.show(event)

    #
    # General callbacks
    #
    
    def _on_project_add_sizegroup(self, project, sizegroup):
        """
        Callback that is executed when a sizegroup is added to a
        project.

        @param project: the project that the sizegroup was added to
        @type project: gazpacho.project.Project
        @param sizegroup: the sizegroup that have been added
        @type sizegroup: gazpacho.sizegroup.GSizeGroup
        """
        sizegroup_iter = self._add_sizegroup(sizegroup)
        if sizegroup_iter:
            select_iter(self._treeview, sizegroup_iter)

    def _on_project_remove_sizegroup(self, project, sizegroup):
        """
        Callback that is executed when a sizegroup is removed from a
        project.

        @param project: the project that the sizegroup was removed from
        @type project: gazpacho.project.Project
        @param sizegroup: the sizegroup that have been removed
        @type sizegroup: gazpacho.sizegroup.GSizeGroup
        """
        sizegroup_iter = self._find_sizegroup(sizegroup)
        if not sizegroup_iter:
            return

        self._disconnect_all_callbacks(sizegroup_iter)
        del self._model[sizegroup_iter]

    def _on_sizegroup_name_changed(self, sizegroup):
        """
        Callback that is executed when the name of a sizegroup has
        changed. This will make sure that the editor is updated.

        @param sizegroup: the sizegroup who's name has changed
        @type sizegroup: gazpacho.sizegroup.GSizeGroup
        """
        sizegroup_iter = self._find_sizegroup(sizegroup)            
        if sizegroup_iter:
            path = self._model[sizegroup_iter].path
            self._model.row_changed(path, sizegroup_iter)            


    def _on_sizegroup_widgets_added(self, sizegroup, widgets):
        """
        Callback that is executed when widgets are added to a
        sizegroup. This will add the widgets to the editor.

        @param sizegroup: the sizegroup that the widgets belong to
        @type sizegroup: gazpacho.sizegroup.GSizeGroup
        @param widgets: the widgets that have been added
        @type widgets: list (of gazpacho.widget.Widget)
        """
        sizegroup_iter = self._find_sizegroup(sizegroup)
        if not sizegroup_iter:
            return

        widget_iter = self._add_widgets(sizegroup_iter, widgets)
        if widget_iter:
            select_iter(self._treeview, widget_iter)

    def _on_sizegroup_widgets_removed(self, sizegroup, widgets):
        """
        Callback that is executed when widgets are removed from a
        sizegroup. This will remove the widgets from the editor.

        @param sizegroup: the sizegroup that the widgets belong to
        @type sizegroup: gazpacho.sizegroup.GSizeGroup
        @param widgets: the widgets that have been removed
        @type widgets: list (of gazpacho.widget.Widget)
        """
        for widget in widgets:
            widget_iter = self._find_widget(sizegroup, widget)
            if widget_iter:
                self._disconnect_item_callbacks(widget_iter)                
                del self._model[widget_iter]

    def _on_widget_notify_name(self, widget, pspec, sizegroup):
        """
        Callback that is executed when the name property of a widget
        has changed. This will make sure that the editor is updated.

        @param widget: the widget who's name has changed
        @type widget: gazpacho.widget.Widget
        @param pspec: the gobject.GParamSpec of the property that was changed
        @type pspec: gobject.GParamSpec
        @param sizegroup: the sizegroup that the widgets belong to
        @type sizegroup: gazpacho.sizegroup.GSizeGroup
        """
        widget_iter = self._find_widget(sizegroup, widget)            
        if widget_iter:
            path = self._model[widget_iter].path
            self._model.row_changed(path, widget_iter)
            
    #
    # Public methods
    #
    def has_selected(self):
        """
        Check if a sizegroup or widget is selected in the tree.
        """
        model, model_iter = self._treeview.get_selection().get_selected()
        return model_iter is not None
    
    def set_project(self, project):
        """
        Set the current project.

        @param project: the current project
        @type project: gazpacho.project.project
        """
        if self.project:
            for sizegroup_row in self._model:
                self._disconnect_all_callbacks(sizegroup_row.iter)

            for callback_id in self._project_callback_ids:
                self.project.disconnect(callback_id)
                
            self._model.clear()

        self.project = project
        self._project_callback_ids = ()
            
        if project:
            self._fill_model_from_project()
            id1 = project.connect('add-sizegroup',
                                  self._on_project_add_sizegroup)
            id2 = project.connect('remove-sizegroup',
                                   self._on_project_remove_sizegroup)
            self._project_callback_ids = (id1, id2)

    def remove_selected_item(self):
        """
        Convenient method for removing the selecte sizegroup or widget

        This method will not remove the action group directly but
        delegate to the command manager.
        """
        model, model_iter = self._treeview.get_selection().get_selected()
        if not model_iter:
                return
        
        row = self._model[model_iter]
        if row.parent:
            sizegroup = row.parent[EDITOR_OBJECT_COLUMN]
            widgets = [row[EDITOR_OBJECT_COLUMN]]
            command_manager.remove_sizegroup_widgets(sizegroup, widgets,
                                                     self.project)
        else:
            sizegroup = row[EDITOR_OBJECT_COLUMN]
            command_manager.remove_sizegroup(sizegroup, self.project)


gobject.type_register(SizeGroupView)

def add_sizegroup_widgets(project):
    """
    Convenient method for adding widgets to a sizegroup. It will add
    the selected widgets to a user specifed sizegroup.

    @param project: the current project
    @type project: gazpacho.project.Project
    """
    from gazpacho.app.app import gazpacho
    
    widgets = project.selection
    if not widgets:
        return

    dialog = SizeGroupDialog(gazpacho.window, project.sizegroups)
    if dialog.run() != gtk.RESPONSE_OK:
        dialog.destroy()
        return

    sizegroup = dialog.get_selected_sizegroup()
    dialog.destroy()

    # We only support single selection right now
    gwidget = Widget.from_widget(widgets[0])
    command_manager.add_sizegroup_widget(sizegroup, gwidget, project)


class SizeGroupPopup(object):
    """
    Popup menu for the sizegroup items.
    """

    def __init__(self, sizegroup, project):
        """
        Initialize the popup.

        @param sizegroup: the sizegroup that the popup refers to
        @type sizegroup: gazpacho.sizegroup.GSizeGroup
        @param project: the current project
        @type project: gazpacho.project.project
        """
        self._project = project
        self._popup_menu = self._create_menu(sizegroup)
        
    def show(self, event):
        """
        Show the popup menu.

        @param event: the event that triggered the popup
        @type event: gtk.gdk.Event
        """        
        button = event.button
        event_time = event.time

        self._popup_menu.popup(None, None, None, button, event_time)
        return self._popup_menu

    def _create_menu(self, sizegroup):
        """
        Create the menu.

        @param sizegroup: the sizegroup that the popup refers to
        @type sizegroup: gazpacho.sizegroup.GSizeGroup
        """
        menu = gtk.Menu()
    
        # Delete sizegroup item
        delete_item = gtk.ImageMenuItem(gtk.STOCK_DELETE, None)
        delete_item.connect_object('activate',
                                   self._on_delete_item_activate, sizegroup)
        menu.append(delete_item)

        menu.show_all()
        
        return menu

    def _on_delete_item_activate(self, sizegroup):
        """
        Callback for the delete menu item.

        @param sizegroup: the sizegroup that the popup refers to
        @type sizegroup: gazpacho.sizegroup.GSizeGroup
        """
        command_manager.remove_sizegroup(sizegroup, self._project)

class WidgetPopup(object):
    """
    Popup menu for the widget items.
    """

    def __init__(self, widget, sizegroup, project):
        """
        Initialize the popup.

        @param widget: the widget that the popup refers to
        @type widget: gazpacho.widget.Widget
        @param sizegroup: the sizegroup the widget belongs to
        @type sizegroup: gazpacho.sizegroup.GSizeGroup
        @param project: the current project
        @type project: gazpacho.project.project
        """
        self._project = project
        self._popup_menu = self._create_menu(widget, sizegroup)
        
    def show(self, event):
        """
        Show the popup menu.

        @param event: the event that triggered the popup
        @type event: gtk.gdk.Event
        """
        button = event.button
        event_time = event.time

        self._popup_menu.popup(None, None, None, button, event_time)
        return self._popup_menu

    def _create_menu(self, widget, sizegroup):
        """
        Create the menu.

        @param widget: the widget that the popup refers to
        @type widget: gazpacho.widget.Widget
        @param sizegroup: the sizegroup the widget belongs to
        @type sizegroup: gazpacho.sizegroup.GSizeGroup
        """
        menu = gtk.Menu()
    
        # Delete widget item
        delete_item = gtk.ImageMenuItem(gtk.STOCK_DELETE, None)
        delete_item.connect_object('activate', self._on_delete_item_activate,
                                   widget, sizegroup)
        menu.append(delete_item)

        menu.show_all()
        
        return menu

    def _on_delete_item_activate(self, widget, sizegroup):
        """
        Callback for the delete menu item.

        @param widget: the widget that the popup refers to
        @type widget: gazpacho.widget.Widget
        @param sizegroup: the sizegroup the widget belongs to
        @type sizegroup: gazpacho.sizegroup.GSizeGroup
        """
        widgets = [widget]
        command_manager.remove_sizegroup_widgets(sizegroup, widgets,
                                                 self._project)


DIALOG_MODE_NAME_COLUMN, DIALOG_MODE_VALUE_COLUMN = range(2)

DIALOG_SIZEGROUP_NAME_COLUMN, DIALOG_SIZEGROUP_COLUMN = range(2)

class SizeGroupDialog(BaseDialog):
    """
    A dialog for choosing which sizegroup the widgets should be added
    to. It also offers the possibility to create a new sizegroup.
    """

    def __init__(self, parent, sizegroups):
        """
        Initialize the dialog.

        @param sizegroups: all available sizegroups
        @type sizegroups: list (of gazpacho.sizegroup.GSizeGroup)
        """
        BaseDialog.__init__(self, title=_('Add Size Group Widgets'),
                            parent=parent)

        # Dict of available sizegroups
        self._sizegroups = self._get_sizegroup_dict(sizegroups)
        
        # Existing sizegroups combo
        self._sizegroup_combo = None

        # New sizegroup name
        self._name_entry = None

        # New sizegroup modes combo
        self._mode_combo = None

        # Radio buttons
        self._selection_radio = None
        self._new_radio = None

        self.add_button(gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
        self._add_button = self.add_button(gtk.STOCK_ADD, gtk.RESPONSE_OK)
        self.set_border_width(6)

        self.set_has_separator(False)
        self.vbox.set_spacing(6)

        self._create_content()

        # Set default name
        name = self._suggest_sizegroup_name()
        self._name_entry.set_text(name)
        
        self._name_entry.connect('changed', self._on_name_entry_changed)

        self.set_default_response(gtk.RESPONSE_OK)        
        self.vbox.show_all()

    def _get_sizegroup_dict(self, sizegroups):
        """
        Get a dictionary that maps sizegroup names to sizegroups.

        @return: dict mapping name to sizegroup
        @rtype: dict (str: gazpacho.sizegroup.GSizeGroup)
        """
        group_dict = {}
        for sizegroup in sizegroups:
            group_dict[sizegroup.name] = sizegroup
        return group_dict

    #
    # Public methods
    #

    def get_selected_sizegroup(self):
        """
        Get the selected sizegroup. This might return an existing
        sizegroup or a new one.

        @return: the sizegroup
        @rtype: gazpacho.sizegroup.GSizeGroup
        """
        if self._selection_radio.get_active():
            model = self._sizegroup_combo.get_model()
            active = self._sizegroup_combo.get_active()
            sizegroup = model[active][DIALOG_SIZEGROUP_COLUMN]
        else:
            model = self._mode_combo.get_model()
            active = self._mode_combo.get_active()
            mode = model[active][DIALOG_MODE_VALUE_COLUMN]
            name =  self._name_entry.get_text()
            sizegroup = GSizeGroup(name, gtk.SizeGroup(mode))

        return sizegroup

    #
    # Private methods
    #
    
    def _create_content(self):
        """
        Create the dialog content.
        """
        size_group = gtk.SizeGroup(gtk.SIZE_GROUP_HORIZONTAL)

        # Existing sizegroup area
        self._selection_radio = gtk.RadioButton(None,
                                                _('Existing size group:'))
        self.vbox.pack_start(self._selection_radio, False, False)

        selection_area = self._create_sizegroup_selection_area(size_group)
        self.vbox.pack_start(selection_area, False, False)

        # New sizegroup area
        self._new_radio = gtk.RadioButton(self._selection_radio,
                                          _('New size group:'))
        self.vbox.pack_start(self._new_radio, False, False)

        new_sizegroup_area = self._create_new_sizegroup_area(size_group)
        self.vbox.pack_start(new_sizegroup_area, False, False)

        # Select which should be active
        if self._sizegroups:
            new_sizegroup_area.set_sensitive(False)
        else:
            self._selection_radio.set_sensitive(False)
            self._new_radio.set_active(True)
            selection_area.set_sensitive(False)

        # Add some handlers
        self._selection_radio.connect('toggled',
                                      self._on_radio_toggled, selection_area)
        self._new_radio.connect('toggled',
                                self._on_radio_toggled, new_sizegroup_area)


    def _create_sizegroup_selection_area(self, size_group):
        """
        Create the GUI for selecting an existing sizegroup.

        @param size_group: a SizeGroup for the labels
        @type size_group: gtk.SizeGroup
        """
        alignment = gtk.Alignment(1.0, 0.0, 1.0, 1.0)
        alignment.set_padding(0, 12, 48, 0)

        hbox = gtk.HBox()
        hbox.set_spacing(6)
        
        label = gtk.Label(('Size group:'))
        label.set_alignment(0.0, 0.5)
        size_group.add_widget(label)
        hbox.pack_start(label, False, False)
        
        liststore = gtk.ListStore(str, object)
        sizegroups = self._sizegroups.values()
        sizegroups.sort(lambda x,y: cmp(x.name, y.name))
        for group in sizegroups:            
            liststore.append((group.name, group))
        self._sizegroup_combo = gtk.ComboBox(liststore)
        renderer = gtk.CellRendererText()
        self._sizegroup_combo.pack_start(renderer)
        self._sizegroup_combo.add_attribute(renderer, 'text', 0)

        self._sizegroup_combo.set_active(0)
        hbox.pack_start(self._sizegroup_combo, True, True)

        alignment.add(hbox)
        return alignment

    def _create_name_entry(self, size_group):
        """
        Create the sizegroup name entry area.
        
        @param size_group: a SizeGroup for the labels
        @type size_group: gtk.SizeGroup
        """
        hbox = gtk.HBox()
        hbox.set_spacing(6)
        
        label = gtk.Label(('Name:'))
        label.set_alignment(0.0, 0.5)
        size_group.add_widget(label)
        
        hbox.pack_start(label, False, False)        
        
        self._name_entry = gtk.Entry()
        hbox.pack_start(self._name_entry, True, True)

        return hbox

    def _create_mode_entry(self, size_group):
        """
        Create the sizegroup name mode area.
        
        @param size_group: a SizeGroup for the labels
        @type size_group: gtk.SizeGroup
        """
        hbox = gtk.HBox()
        hbox.set_spacing(6)

        label = gtk.Label(('Mode:'))
        label.set_alignment(0.0, 0.5)
        size_group.add_widget(label)
        hbox.pack_start(label, False, False)

        liststore = gtk.ListStore(str, int)
        liststore.append((_('Horizontal'), gtk.SIZE_GROUP_HORIZONTAL))
        liststore.append((_('Vertical'), gtk.SIZE_GROUP_VERTICAL))
        liststore.append((_('Both'), gtk.SIZE_GROUP_BOTH))
        self._mode_combo = gtk.ComboBox(liststore)
        renderer = gtk.CellRendererText()
        self._mode_combo.pack_start(renderer)
        self._mode_combo.add_attribute(renderer, 'text', 0)

        self._mode_combo.set_active(0)
        hbox.pack_start(self._mode_combo, True, True)

        return hbox
        
    def _create_new_sizegroup_area(self, size_group):
        """
        Create the GUI for creating a new sizegroup.

        @param size_group: a SizeGroup for the labels
        @type size_group: gtk.SizeGroup
        """
        vbox = gtk.VBox()
        vbox.set_spacing(6)
        vbox.pack_start(self._create_name_entry(size_group), False, False)
        vbox.pack_start(self._create_mode_entry(size_group), False, False)

        alignment = gtk.Alignment(1.0, 0.0, 1.0, 1.0)
        alignment.set_padding(0, 12, 48, 0)
        alignment.add(vbox)

        return alignment

    def _suggest_sizegroup_name(self):
        """
        Get a unique name suggestion for the sizegroup. It will be
        something like 'sizegroup3'.

        @return: a unique name
        @rtype: str
        """
        default_name = "sizegroup%d"
        i = 1
        while True:
            name = default_name % i
            if name not in self._sizegroups:
                return name
            i += 1

    def _on_radio_toggled(self, button, widget):
        """
        Callback that is executed when the radio buttons are
        toggled. This will enable or disable parts of the gui.

        @param button: the radio button that was toggled
        @type button: gtk.RadioButton
        @param widget: the gui element that should be enabled or disabled
        @type widget: gtk.Widget
        """
        value = button.get_active()
        widget.set_sensitive(value)

    def _on_name_entry_changed(self, entry):
        """
        Callback that is executed when the text in the name entry is
        modified.

        @param entry: the name entry
        @type entry: gtk.Entry
        """
        name = entry.get_text()
        if name in self._sizegroups:
            self._add_button.set_sensitive(False)
        else:
            self._add_button.set_sensitive(True)
