/* GDM - The Gnome Display Manager
 * Copyright (C) 1999, 2000 Martin K. Petersen <mkp@mkp.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/* This file contains functions for controlling local X servers */

#include <config.h>
#include <gnome.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <strings.h>
#include <signal.h>
#include <errno.h>
#include <X11/Xlib.h>

#include "gdm.h"
#include "server.h"
#include "misc.h"
#include "xdmcp.h"
#include "display.h"
#include "auth.h"

static const gchar RCSid[]="$Id: server.c,v 1.11 2001/01/10 18:55:11 jirka Exp $";


/* Local prototypes */
static void gdm_server_spawn (GdmDisplay *d);
static void gdm_server_usr1_handler (gint);
static void gdm_server_alarm_handler (gint);
static void gdm_server_child_handler (gint);

/* Configuration options */
extern gchar *argdelim;
extern gchar *GdmDisplayInit;
extern gchar *GdmServAuthDir;
extern gchar *GdmLogDir;
extern gint GdmXdmcp;
extern sigset_t sysmask;

/* Global vars */
static GdmDisplay *d = NULL;

static gboolean
gdm_server_check_loop (GdmDisplay *disp)
{
  time_t now;
  time_t since_last;
  
  now = time (NULL);

  if (disp->disabled)
    return FALSE;
  
  if (disp->last_start_time > now || disp->last_start_time == 0)
    {
      /* Reset everything if this is the first time in this
       * function, or if the system clock got reset backward.
       */
      disp->last_start_time = now;
      disp->retry_count = 1;

      gdm_debug ("Resetting counts for loop of death detection");
      
      return TRUE;
    }

  since_last = now - disp->last_start_time;

  /* If it's been at least 1.5 minutes since the last startup
   * attempt, then we reset everything.
   */

  if (since_last >= 90)
    {
      disp->last_start_time = now;
      disp->retry_count = 1;

      gdm_debug ("Resetting counts for loop of death detection, 90 seconds elapsed.");
      
      return TRUE;
    }

  /* If we've tried too many times we bail out. i.e. this means we
   * tried too many times in the 90-second period.
   */
  if (disp->retry_count > 4)
    {
      gchar *msg;
      msg = g_strdup_printf (_("Failed to start X server several times in a short time period; disabling display %s"), disp->name);
      gdm_error (msg);
      g_free (msg);
      disp->disabled = TRUE;

      gdm_debug ("Failed to start X server after several retries; aborting.");
      
      exit (SERVER_ABORT);
    }
  
  /* At least 8 seconds between start attempts,
   * so you can try to kill gdm from the console
   * in these gaps.
   */
  if (since_last < 8)
    {
      gdm_debug ("Sleeping %d seconds before next X server restart attempt",
                 8 - since_last);
      sleep (8 - since_last);
      now = time (NULL);
    }

  disp->retry_count += 1;
  disp->last_start_time = now;

  return TRUE;
}

/**
 * gdm_server_stop:
 * @disp: Pointer to a GdmDisplay structure
 *
 * Stops a local X server, but only if it exists
 */

void
gdm_server_kill (GdmDisplay *disp)
{
    if (disp == NULL ||
	disp->servpid <= 0)
	return;

    gdm_server_stop (disp);
}

/**
 * gdm_server_start:
 * @disp: Pointer to a GdmDisplay structure
 *
 * Starts a local X server. Handles retries and fatal errors properly.
 */

gboolean
gdm_server_start (GdmDisplay *disp)
{
    struct sigaction usr1, chld, alrm;
    struct sigaction old_usr1, old_chld, old_alrm;
    sigset_t mask, oldmask;
    gboolean retvalue;
    
    if (!disp)
	return FALSE;
    
    d = disp;
    d->servtries = 0;

    /* if an X server exists, wipe it */
    gdm_server_kill (d);

    gdm_debug ("gdm_server_start: %s", d->name);

    if (!gdm_server_check_loop (disp))
      return FALSE;

    gdm_debug ("Attempting to start X server");
    
    /* Create new cookie */
    gdm_auth_secure_display (d);
    gdm_setenv ("DISPLAY", d->name);
    
    /* Catch USR1 from X server */
    usr1.sa_handler = gdm_server_usr1_handler;
    usr1.sa_flags = SA_RESTART|SA_RESETHAND;
    sigemptyset (&usr1.sa_mask);
    
    if (sigaction (SIGUSR1, &usr1, &old_usr1) < 0) {
	gdm_error (_("gdm_server_start: Error setting up USR1 signal handler"));
	return FALSE;
    }

    /* Catch CHLD from X server */
    chld.sa_handler = gdm_server_child_handler;
    chld.sa_flags = SA_RESTART|SA_RESETHAND;
    sigemptyset (&chld.sa_mask);
    
    if (sigaction (SIGCHLD, &chld, &old_chld) < 0) {
	gdm_error (_("gdm_server_start: Error setting up CHLD signal handler"));
	sigaction (SIGUSR1, &old_usr1, NULL);
	return FALSE;
    }

    /* Catch ALRM from X server */
    alrm.sa_handler = gdm_server_alarm_handler;
    alrm.sa_flags = SA_RESTART|SA_RESETHAND;
    sigemptyset (&alrm.sa_mask);
    
    if (sigaction (SIGALRM, &alrm, &old_alrm) < 0) {
	gdm_error (_("gdm_server_start: Error setting up ALRM signal handler"));
	sigaction (SIGUSR1, &old_usr1, NULL);
	sigaction (SIGCHLD, &old_chld, NULL);
	return FALSE;
    }
    
    /* Set signal mask */
    sigemptyset (&mask);
    sigaddset (&mask, SIGUSR1);
    sigaddset (&mask, SIGCHLD);
    sigaddset (&mask, SIGALRM);
    sigprocmask (SIG_UNBLOCK, &mask, &oldmask);
    
    /* fork X server process */
    gdm_server_spawn (d);

    retvalue = FALSE;

    /* Wait for X server to send ready signal */
    while (d->servtries < 5) {
	gdm_run ();

	switch (d->servstat) {

	case SERVER_TIMEOUT:
	    /* Unset alarm and try again */
	    alarm (0);
	    waitpid (d->servpid, 0, WNOHANG);

	    gdm_debug ("gdm_server_start: Temporary server failure (%d)", d->name);
	    sleep (1 + d->servtries*2);

	    gdm_server_spawn (d);
	    
	    break;

	case SERVER_RUNNING:
	    /* Unset alarm */
	    alarm (0);
	    
	    gdm_debug ("gdm_server_start: Completed %s!", d->name);

	    retvalue = TRUE;
	    goto spawn_done;

	case SERVER_ABORT:
	    alarm (0);
	    gdm_debug ("gdm_server_start: Server %s died during startup!", d->name);
	    sleep (1 + d->servtries*2);

	    gdm_server_spawn (d);

	    break;

	default:
	    break;
	}

	d->servtries ++;
    }


spawn_done:

    sigprocmask (SIG_SETMASK, &oldmask, NULL);

    /* restore default handlers */
    sigaction (SIGUSR1, &old_usr1, NULL);
    sigaction (SIGCHLD, &old_chld, NULL);
    sigaction (SIGALRM, &old_alrm, NULL);

    return retvalue;
}


/**
 * gdm_server_spawn:
 * @disp: Pointer to a GdmDisplay structure
 *
 * forks an actual X server process
 */

static void 
gdm_server_spawn (GdmDisplay *d)
{
    struct sigaction usr1;
    gchar *srvcmd = NULL;
    gchar **argv = NULL;
    int logfd;

    if (!d)
	return;

    d->servstat = SERVER_STARTED;

    /* Log all output from spawned programs to a file */
    logfd = open (g_strconcat (GdmLogDir, "/", d->name, ".log", NULL),
		  O_CREAT|O_TRUNC|O_APPEND|O_WRONLY, 0666);
    
    if (logfd != -1) {
	dup2 (logfd, 1);
	dup2 (logfd, 2);
    }
    else
	gdm_error (_("gdm_server_spawn: Could not open logfile for display %s!"), d->name);
    
    /* We add a timeout in case the X server fails to start. This
     * might happen because X servers take a while to die, close their
     * sockets etc. If the old X server isn't completely dead, the new
     * one will fail and we'll hang here forever */

    alarm (10);

    /* Fork into two processes. Parent remains the gdm process. Child
     * becomes the X server. */
    
    switch (d->servpid = fork()) {
	
    case 0:
	/* Close the XDMCP fd inherited by the daemon process */
	if (GdmXdmcp)
	    gdm_xdmcp_close();
	
	/* The X server expects USR1 to be SIG_IGN */
	usr1.sa_handler = SIG_IGN;
	usr1.sa_flags = SA_RESTART;
	sigemptyset (&usr1.sa_mask);
	
	if (sigaction (SIGUSR1, &usr1, NULL) < 0) {
	    gdm_error (_("gdm_server_spawn: Error setting USR1 to SIG_IGN"));
	    exit (SERVER_ABORT);
	}
	
	srvcmd = g_strconcat (d->command, " -auth ", GdmServAuthDir, \
			      "/", d->name, ".Xauth ", 
			      d->name, NULL);
	
	gdm_debug ("gdm_server_spawn: '%s'", srvcmd);
	
	argv = g_strsplit (srvcmd, argdelim, MAX_ARGS);
	g_free (srvcmd);
	
	setpgid (0, 0);
	execv (argv[0], argv);
	
	gdm_error (_("gdm_server_spawn: Xserver not found: %s"), d->command);
	
	exit (SERVER_ABORT);
	
    case -1:
	gdm_error (_("gdm_server_spawn: Can't fork Xserver process!"));
	d->servpid = 0;
	d->servstat = SERVER_DEAD;
	break;
	
    default:
	break;
    }
}


/**
 * gdm_server_stop:
 * @disp: Pointer to a GdmDisplay structure
 *
 * Stops a local X server
 */

void
gdm_server_stop (GdmDisplay *d)
{
    gdm_debug ("gdm_server_stop: Server for %s going down!", d->name);
    
    d->servstat = SERVER_DEAD;

    if (d->servpid != 0) {
	    if (kill (d->servpid, SIGTERM) == 0)
		    waitpid (d->servpid, 0, 0);
	    d->servpid = 0;
    }

    unlink (d->authfile);
}


/**
 * gdm_server_usr1_handler:
 * @sig: Signal value
 *
 * Received when the server is ready to accept connections
 */

static void
gdm_server_usr1_handler (gint sig)
{
    d->servstat = SERVER_RUNNING; /* Server ready to accept connections */

    gdm_quit ();
}


/**
 * gdm_server_alarm_handler:
 * @sig: Signal value
 *
 * Server start timeout handler
 */

static void 
gdm_server_alarm_handler (gint signal)
{
    d->servstat = SERVER_TIMEOUT; /* Server didn't start */

    gdm_quit ();
}


/**
 * gdm_server_child_handler:
 * @sig: Signal value
 *
 * Received when server died during startup
 */

static void 
gdm_server_child_handler (gint signal)
{
    gdm_debug ("gdm_server_child_handler: Got SIGCHLD, server abort");

    d->servstat = SERVER_ABORT;	/* Server died unexpectedly */
    d->servpid = 0;

    gdm_quit ();
}


/**
 * gdm_server_alloc:
 * @id: Local display number
 * @command: Command line for starting the X server
 *
 * Allocate display structure for a local X server
 */

GdmDisplay * 
gdm_server_alloc (gint id, gchar *command)
{
    gchar *dname = g_new0 (gchar, 1024);
    gchar *hostname = g_new0 (gchar, 1024);
    GdmDisplay *d = g_new0 (GdmDisplay, 1);
    
    if (gethostname (hostname, 1023) == -1)
	return NULL;

    sprintf (dname, ":%d", id);  
    d->authfile = NULL;
    d->auths = NULL;
    d->userauth = NULL;
    d->command = g_strdup (command);
    d->cookie = NULL;
    d->dispstat = DISPLAY_DEAD;
    d->greetpid = 0;
    d->name = g_strdup (dname);
    d->hostname = g_strdup (hostname);
    d->dispnum = id;
    d->servpid = 0;
    d->servstat = SERVER_DEAD;
    d->sessionid = 0;
    d->sesspid = 0;
    d->slavepid = 0;
    d->type = TYPE_LOCAL;
    d->sessionid = 0;
    d->acctime = 0;
    d->dsp = NULL;

    d->last_start_time = 0;
    d->retry_count = 0;
    d->disabled = FALSE;
    
    g_free (dname);
    g_free (hostname);

    return d;
}


/* EOF */
