/* $Header: /cvs/gnome/gIDE/src/gI_window.c,v 1.19 2000/04/16 04:59:57 jpr Exp $ */
/* gIDE
 * Copyright (C) 1998-2000 Steffen Kern
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <config.h>
#include "gide.h"
#include "gI_menus.h"
#include "gI_file.h"
#include "gI_edit.h"
#include "gI_tools.h"
#include "gI_hilite.h"
#include "gI_project.h"
#include "gI_toolbar.h"
#include "gI_commands.h"
#include "gide-tools.h"

/* Prototypes */
static void gI_window_class_init (GideWindowClass *class);
static void gI_window_init (GideWindow *tool);
static void gI_window_switch_to_next_doc(GtkWidget* widget, GdkEventKey* kevent,
					 gpointer data);
static void gI_window_switch_notebookpage(GtkWidget* widget,
					  GtkNotebookPage* page,
					  gint page_num, gpointer data);
static void gI_window_drag_recv(GtkWidget* widget, GdkDragContext* context,
				gint x, gint y, GtkSelectionData* seldata,
				guint info, guint time, gpointer data);
static void gI_window_doc_modified( GtkWidget *widget, gpointer data );
static void gI_window_doc_unmodified( GtkWidget *widget, gpointer data );
static void gI_window_doc_readonly( GtkWidget *widget, gpointer data );
static void gI_window_doc_unreadonly( GtkWidget *widget, gpointer data );
static void gI_window_doc_source( GtkWidget *widget, gchar *filename,
				  gpointer data );
static void gI_window_doc_destroy( GtkWidget *widget, gpointer data );
static void gI_window_destroy( GtkWidget *widget, gpointer data );
			       	
guint
gI_window_get_type()
{
	static guint window_type = 0;

	if (!window_type) {
           GtkTypeInfo window_info = {
               "GideWindow",
               sizeof (GideWindow),
               sizeof (GideWindowClass),
               (GtkClassInitFunc) gI_window_class_init,
               (GtkObjectInitFunc) gI_window_init,
               (GtkArgSetFunc) NULL,
               (GtkArgGetFunc) NULL
             };
             window_type = gtk_type_unique (gnome_app_get_type (),
					    &window_info);
           }

         return window_type;
}

static void
gI_window_class_init( GideWindowClass *class )
{
	GtkObjectClass *object_class;
	
	object_class = (GtkObjectClass*) class;
}

static void
gI_window_init( GideWindow *window )
{
	GideDocument *document;
	GtkWidget *vbox;
	GtkTargetEntry dragtypes[] = {{"text/uri-list", 0, 0}};

	/* Initialize Gnome Application */
	gnome_app_construct( GNOME_APP(window), "gide", "gIDE"  );

	/* Set some defaults */
	window->popup = NULL;
	window->notebook = NULL;
	window->switch_page_id = FALSE;
	window->toolbar = NULL;

	/* Setup the window */
	gtk_window_set_policy(GTK_WINDOW(window), TRUE, TRUE, TRUE);

	vbox = gtk_vbox_new(FALSE, 0);
	gnome_app_set_contents( GNOME_APP(window), vbox );
	gtk_widget_show(vbox);
		
	gtk_widget_set_usize( GTK_WIDGET(window), 800, 550 );
	gtk_widget_realize( GTK_WIDGET(window) );

	/* Add the status bar */
	window->statusbar = gnome_appbar_new( FALSE, TRUE,
					      GNOME_PREFERENCES_ALWAYS );
	gnome_app_set_statusbar( GNOME_APP(window), window->statusbar );
	gtk_widget_show(window->statusbar);

	/* Add the menubar */
	gI_menus_create_menubar(window);
	
	/* Add toolbar */
	if(gdk_screen_width() < 600) {
		gnome_preferences_set_toolbar_labels(FALSE);
	}
	
	gI_toolbar_create(GTK_WIDGET(window));			
	/* Pop-up menu */
	gI_window_create_popupmenu( window );
	
	/* drag'n'drop */
	gtk_drag_dest_set( GTK_WIDGET(window),
			   GTK_DEST_DEFAULT_ALL, dragtypes,
			   sizeof(dragtypes) / sizeof(dragtypes[0]),
			   GDK_ACTION_COPY );
	gtk_signal_connect(GTK_OBJECT(window), "drag_data_received",
			   GTK_SIGNAL_FUNC(gI_window_drag_recv),
			   (gpointer)window);


	/* Create the Layout of the window */

	/* Create the project/document separator */
	window->hpane = gtk_hpaned_new();
	gtk_paned_gutter_size(GTK_PANED(window->hpane), 10);
	gtk_paned_set_position(GTK_PANED(window->hpane), 0);
	gtk_box_pack_start(GTK_BOX(vbox), window->hpane, TRUE, TRUE, 0);
	gtk_widget_show(window->hpane);

	/* Project tree is on the left */
	window->tree_box = gtk_hbox_new(TRUE, 0);
	gtk_paned_add1( GTK_PANED(window->hpane),
			window->tree_box );

	gI_window_create_prjftree(window);

	gtk_widget_hide( window->tree_box );
	gtk_widget_show( window->tree_frame );

	/* Notebook is on the right */
	window->notebook = gtk_notebook_new();
	gtk_notebook_set_scrollable( GTK_NOTEBOOK( window->notebook ),
				     TRUE );
	
	gtk_paned_add2(GTK_PANED(window->hpane), window->notebook);
	gtk_widget_show(window->notebook);

	/* Add an empty document */
/* added to be able to add popup menu */
	main_window = window;

	document = GIDE_DOCUMENT(gI_document_new());
	gI_window_add_doc( window, document );
	
	/* autosave */
	/* belongs in preferences code (is actually duplicated there) */
	if ( cfg->autosave ) {
		window->timeout_id =
			gtk_timeout_add(cfg->autosave_freq * 1000,
					(GtkFunction)file_autosave, NULL);
	} else {
		window->timeout_id = 0;
	}

	/* Connect signals */
	gtk_signal_connect( GTK_OBJECT(window), "key_press_event",
			    GTK_SIGNAL_FUNC(gI_window_switch_to_next_doc),
			    window );

	window->switch_page_id =
		gtk_signal_connect_after(
			GTK_OBJECT( window->notebook ), "switch_page",
			GTK_SIGNAL_FUNC(gI_window_switch_notebookpage),
			window );

	gtk_signal_connect( GTK_OBJECT(window), "destroy",
			    GTK_SIGNAL_FUNC(gI_window_destroy), NULL );

	gtk_widget_queue_draw( GTK_WIDGET(window) );
	gtk_widget_queue_resize( GTK_WIDGET(window) );
}

GtkWidget *
gI_window_new( void )
{
	GideWindow *window;
	
	window = GIDE_WINDOW( gtk_type_new( gI_window_get_type() ));

	return GTK_WIDGET(window);
}

void
gI_window_add_doc ( GideWindow *window, GideDocument *document )
{
	GtkWidget *hbox, *vscrollbar;
	GtkWidget *label;
	gchar label_str[25];
	static glong doc_count = 1;

	/* Add the document to the list */
	window->documents = g_list_append( window->documents,
					   (gpointer)document );

	/* Add the document to the notebook */
	g_snprintf( label_str, sizeof(label_str),
		    "Untitled%ld", doc_count++ );
	label = gtk_label_new( label_str );

	/* Build an hbox to hold the document */
	hbox = gtk_hbox_new( FALSE, 0 );
        gtk_box_pack_start( GTK_BOX( hbox ), GTK_WIDGET(document),
			    TRUE, TRUE, 5 );

	/* Build the doc a scrollbar */
	vscrollbar =
		gtk_vscrollbar_new( GTK_ADJUSTMENT(gI_text_vadj( document )) );
	gtk_box_pack_end( GTK_BOX( hbox ), vscrollbar, FALSE, TRUE, 5 );

	gtk_widget_show( vscrollbar );
        gtk_widget_show( hbox );

	/* We'll remember the hbox for tab manipulation */
	gtk_object_set_data(GTK_OBJECT(document),
			    "GideWindow::hbox",
			    (gpointer)hbox);
			    
	/* Add the doc to the notebook */
	gtk_notebook_append_page( GTK_NOTEBOOK( window->notebook ),
				  hbox, label );

	/* Listen for changes */
	gtk_signal_connect( GTK_OBJECT( document ), "doc_modified",
			    GTK_SIGNAL_FUNC( gI_window_doc_modified ),
			    (gpointer) window );
	gtk_signal_connect( GTK_OBJECT( document ), "doc_unmodified",
			    GTK_SIGNAL_FUNC( gI_window_doc_unmodified ),
			    (gpointer) window );
	gtk_signal_connect( GTK_OBJECT( document ), "doc_readonly",
			    GTK_SIGNAL_FUNC( gI_window_doc_readonly ),
			    (gpointer) window );
	gtk_signal_connect( GTK_OBJECT( document ), "doc_unreadonly",
			    GTK_SIGNAL_FUNC( gI_window_doc_unreadonly ),
			    (gpointer) window );
	gtk_signal_connect( GTK_OBJECT( document ), "doc_source",
			    GTK_SIGNAL_FUNC( gI_window_doc_source ),
			    (gpointer) window );
	gtk_signal_connect( GTK_OBJECT( document ), "destroy",
			    GTK_SIGNAL_FUNC( gI_window_doc_destroy ),
			    (gpointer) window );

	gtk_widget_show( GTK_WIDGET(document) );
	gtk_widget_grab_focus( GTK_WIDGET(document) );

	command_undo_redo_menu_labels(document);

	/* Flip to the new page, will set off the signal */
	gI_window_goto_doc_by_index( window, -1 );
}

gint
gI_window_num_docs( GideWindow *window )
{
	return g_list_length( window->documents );
}

glong
gI_window_num_changed_docs( GideWindow *window )
{
	GideDocument *document;
	GList *docs;
	glong cnt = 0;
	
	docs = window->documents;

	while( docs ) {
		document = (GideDocument *) docs->data;
		if( !document ) {
			docs = docs->next;
			continue;
		}

		if( document->changed ) {
			cnt++;
		}

		docs = docs->next;
	}

	return cnt;
}

GideDocument *
gI_window_get_nth_doc( GideWindow *window, guint index )
{
	GideDocument *document;

	document = NULL;

	document = g_list_nth_data( window->documents, index );

	return document;
}

GideDocument *
gI_window_get_current_doc ( GideWindow *window )
{
	gint pagenum;
        GideDocument *document;
 
        document = NULL;
     
        pagenum =
		gtk_notebook_get_current_page(GTK_NOTEBOOK(window->notebook));

	return gI_window_get_nth_doc( window, pagenum );
}

gchar *
gI_window_get_doc_label( GideWindow *window, GideDocument *document )
{
	GtkWidget *label;
	gpointer hbox;
	gchar *str;
	
	hbox = gtk_object_get_data( GTK_OBJECT(document),
				    "GideWindow::hbox");
	label = gtk_notebook_get_tab_label( GTK_NOTEBOOK(window->notebook),
					    GTK_WIDGET(hbox) );

	gtk_label_get( GTK_LABEL(label), &str );

	return str;
}

void
gI_window_set_doc_label( GideWindow *window,
			 GideDocument *document, gchar *str )
{
	GtkWidget *label;
	gpointer hbox;
	
	hbox = gtk_object_get_data( GTK_OBJECT(document),
				    "GideWindow::hbox");
	label = gtk_notebook_get_tab_label( GTK_NOTEBOOK(window->notebook),
					    GTK_WIDGET(hbox) );

	gtk_label_set( GTK_LABEL(label), str );
}

void
gI_window_set_doc_label_style( GideWindow *window, GideDocument *document,
			       GtkStyle *style )
{
	GtkWidget *label;
	gpointer hbox;

	hbox = gtk_object_get_data( GTK_OBJECT(document),
				    "GideWindow::hbox");
	label = gtk_notebook_get_tab_label( GTK_NOTEBOOK(window->notebook),
					    GTK_WIDGET(hbox) );

	gtk_widget_set_style( label, style );
}

gboolean
gI_window_is_open_doc( GideWindow *window, const gchar *filename )
{
	GideDocument *document;
	gint len, i;

	len = gI_window_num_docs( window );
	for (i=0; i<len; i++) {
		document = gI_window_get_nth_doc( window, i );
		if (document->filename &&
		    !strcmp(document->filename, filename))
			return TRUE;
	}
	return FALSE;
}

void
gI_window_goto_doc_by_index( GideWindow *window, gint index )
{
	g_return_if_fail( index >= -1 );
	g_return_if_fail( index < gI_window_num_docs(window) );
	
	gtk_notebook_set_page( GTK_NOTEBOOK( window->notebook ), index);
}

void
gI_window_goto_doc_by_file( GideWindow *window, const gchar *filename )
{
	GideDocument *document;
	gint len, i;

	len = gI_window_num_docs( window );
	for (i=0; i<len; i++) {
		document = gI_window_get_nth_doc( window, i );
		if (!strcmp(document->filename, filename))
			gI_window_goto_doc_by_index( window, i );
	}
}

void gI_window_update_title( GideWindow *window )
{
	gchar buftxt[200];
	GideDocument *document;

	document = gI_window_get_current_doc( window );

	g_snprintf(buftxt, sizeof(buftxt), "gIDE %s", VERSION);

	if ( document && document->filename ) {
		strncat(buftxt, " <", sizeof(buftxt)-strlen(buftxt));
		strncat(buftxt, document->filename,
			sizeof(buftxt)-strlen(buftxt));
		strncat(buftxt, ">", sizeof(buftxt)-strlen(buftxt));
	}

	gtk_window_set_title(GTK_WINDOW(window), buftxt);
}

/*
 * Function: gI_window_set_statusbar()
 * Desc: Sets the content of the Statusbar of a Window
 * (and the main_window title)
 */
void
gI_window_set_statusbar( GideWindow *window )
{
	gchar sb_msg[200] = "";
	GideDocument *document;

	document = gI_window_get_current_doc( window );

	if (document) {
		g_snprintf(sb_msg, sizeof(sb_msg),
			   _("%s, Size: %d bytes %s / [row:%d, col:%d] %s"),
			   document->filename ? document->filename : _("(no file)"),
			   gI_text_get_length(document),
			   document->changed ? _("[modified]") : "",
			   document->x, document->y,
			   document->read_only ? _(" / [read-only]") : "");
//		   document->window->debug_window ? _(" / [debug]") : "");
	}
	
	gnome_appbar_pop( GNOME_APPBAR(window->statusbar) );
	gnome_appbar_push(GNOME_APPBAR(window->statusbar), sb_msg);

	gide_sensitivity_tools( window );
}

/*
 * Function: gI_window_clear_statusbar()
 * Desc: Clears the Statusbar of a Window
 */
void
gI_window_clear_statusbar( GideWindow *window )
{
	gnome_appbar_pop( GNOME_APPBAR(window->statusbar) );
}

/*
 * Function: gI_window_set_visible_read_only()
 * Desc: This function sets the Read-Only entry sensitive true or false
 */
/* does this actually work? */
void
gI_window_set_visible_read_only(gint sensitive)
{
	gI_menus_set_sensitive(main_window, "/Edit/Read Only", sensitive);
	if ( sensitive == FALSE ) {
		gtk_check_menu_item_set_state(
			GTK_CHECK_MENU_ITEM(gI_menus_get_item_widget(main_window, "/Edit/Read Only")), FALSE);
	}
}

/*
 * Function: gI_window_create_prjftree()
 * Desc: Creates Project-File-Tree
 */
/* somehow this functions doesn't seem complete... */
void
gI_window_create_prjftree(GideWindow *window)
{
	if(cfg->prjftree)
	{
		if(window->tree_frame != NULL)
		{
			return;
		}

		window->tree_frame = gtk_frame_new(_("Project Files"));
		gtk_box_pack_start(GTK_BOX(window->tree_box), window->tree_frame, FALSE, TRUE, 5);
		window->tree_scrolled_window = gtk_scrolled_window_new(NULL, NULL);
		gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(window->tree_scrolled_window),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
		gtk_container_add(GTK_CONTAINER(window->tree_frame), window->tree_scrolled_window);
		window->tree_viewport = gtk_viewport_new(NULL, NULL);
		gtk_container_add(GTK_CONTAINER(window->tree_scrolled_window), window->tree_viewport);
		gtk_widget_show(window->tree_viewport);
		gtk_widget_show(window->tree_scrolled_window);
		gtk_widget_show(window->tree_frame);
	}
	else
	{
		if(window->tree_frame != NULL)
		{
			gtk_widget_destroy(window->tree_frame);
			window->tree_frame = NULL;
		}
	}
}

/*
 * Function: gI_window_create_popupmenu()
 * Desc: Creates PopUp-Menu
 */
void
gI_window_create_popupmenu(GideWindow *window)
{
	GnomeUIInfo specialmenu[] = {
		GNOMEUIINFO_ITEM_NONE(_("Delete active file"),
			_("Delete active file"), special_remove_current_file),
		GNOMEUIINFO_ITEM_NONE(_("Insert file"),
			_("Insert file"), special_insert_file),
		GNOMEUIINFO_ITEM_NONE(_("Mail active document"),
			_("Mail active document"), special_mail_doc),
		GNOMEUIINFO_END
	};

	GnomeUIInfo popupmenu[] = {
		GNOMEUIINFO_ITEM_NONE(_("Open (swap) .c/.h file"),
			_("Open (swap) .c/.h file"), gI_document_cmd_swaphc),
		GNOMEUIINFO_ITEM_NONE(_("Open Selected"),
			_("Open Selected"), gI_document_cmd_open_file_at_line),
		GNOMEUIINFO_MENU_CUT_ITEM(edit_cut, NULL),
		GNOMEUIINFO_MENU_COPY_ITEM(edit_copy, NULL),
		GNOMEUIINFO_MENU_PASTE_ITEM(edit_paste, NULL),
		GNOMEUIINFO_SEPARATOR,
		GNOMEUIINFO_MENU_SAVE_ITEM(file_save, NULL),
		GNOMEUIINFO_MENU_PRINT_ITEM(file_print, NULL),
		GNOMEUIINFO_MENU_CLOSE_ITEM(file_close, NULL),
#ifdef HAVE_GTKTEXT_PATCH
		GNOMEUIINFO_SEPARATOR,
		GNOMEUIINFO_ITEM_NONE(_("Re-Highlight"), _("Re-Highlight"),
			gI_hilite_rehilite),
#endif
		GNOMEUIINFO_SEPARATOR,
		GNOMEUIINFO_SUBTREE(_("Special"), specialmenu),
		GNOMEUIINFO_SEPARATOR,
		GNOMEUIINFO_ITEM_NONE(_("Show Manual Page"), _("Show Manual Page"),
			show_manpage),
		GNOMEUIINFO_SEPARATOR,
		GNOMEUIINFO_END
	};

	/* create menus */
	window->popup = gnome_popup_menu_new(popupmenu);
}

/*
 * Function: gI_window_switch_notebookpage()
 * Desc: This function is called, when the Notebook-Page is switched
 */
static void
gI_window_switch_notebookpage(GtkWidget* widget, GtkNotebookPage* page,
			      gint page_num, gpointer data)
{
	GideWindow *window;
	GideDocument *document;

	g_assert(IS_GIDE_WINDOW(data));

	window = GIDE_WINDOW( data );
	
	g_return_if_fail( window );
	g_return_if_fail( page_num < gI_window_num_docs(window) );

	document = gI_window_get_nth_doc( window, page_num );

	g_return_if_fail (document);

	gI_window_set_visible_read_only(TRUE);

	/* set check menu item state & correct editable flag in widget */

	gI_menus_set_sensitive_readonly(window, document->read_only);

	command_undo_redo_menu_labels (document);

	gI_window_clear_statusbar(window);
	gI_window_set_statusbar(window);

	gtk_widget_grab_focus( GTK_WIDGET(document) );

	check_current_doc();
}

/*
 * Make shift-tab jump to next document
 */
static void
gI_window_switch_to_next_doc(GtkWidget* widget, GdkEventKey* kevent,
			     gpointer data)
{
	GideWindow *window;
	gint curpag;
	GtkNotebookPage* page;
	static guint KeyValSave = 0;

	g_assert(IS_GIDE_WINDOW(widget));

	window = GIDE_WINDOW( widget );
	
	g_assert(window != NULL);
	g_assert(window->notebook != NULL);

	if(kevent->type == GDK_KEY_PRESS &&
		kevent->keyval == GDK_Shift_L)
	{
		KeyValSave = kevent->keyval;
		return;
	}

	if(kevent->type != GDK_KEY_PRESS ||
		kevent->keyval != GDK_ISO_Left_Tab ||
		KeyValSave != GDK_Shift_L)
	{
		return;
	}

	curpag = gtk_notebook_get_current_page(GTK_NOTEBOOK(window->notebook));
	if(curpag == gI_window_num_docs(window) - 1)
	{
		gtk_notebook_set_page(GTK_NOTEBOOK(window->notebook), 0);
	}
	else
	{
		gtk_notebook_next_page(GTK_NOTEBOOK(window->notebook));
	}

	page = (GtkNotebookPage*)GTK_NOTEBOOK(window->notebook)->cur_page;

	g_assert(page != NULL);

	gtk_widget_grab_focus(page->tab_label);
	gtk_widget_draw_focus(page->tab_label);

	gtk_signal_emit_stop_by_name(GTK_OBJECT(window), "key_press_event");
}

static void
gI_window_drag_recv(GtkWidget* widget, GdkDragContext* context,
		    gint x, gint y, GtkSelectionData* seldata,
		    guint info, guint time, gpointer data)
{
	GList* files;
	GList* fnp;
	gint count;
	gchar* fname;

	g_assert(IS_GIDE_WINDOW(widget));

	/* get filenames */
	files = gnome_uri_list_extract_filenames((gchar*)seldata->data);
	count = g_list_length(files);

	/* open files */
	if(count > 0)
	{
		fnp = g_list_first(files);
		while(fnp)
		{
			fname = (gchar*)fnp->data;

			file_open_by_name(GIDE_WINDOW(widget), fname);

			fnp = g_list_next(fnp);
		}
	}

	gnome_uri_list_free_strings(files);
}

static void
gI_window_doc_modified( GtkWidget *widget, gpointer data )
{
	GideWindow *window;
	GideDocument *document;
	GtkStyle *style;
	GdkColor color = {0, 0xffff, 0, 0};
	
	g_return_if_fail(data);
		
	g_assert(IS_GIDE_WINDOW(data));
	g_assert(IS_GIDE_DOCUMENT(widget));

	window = GIDE_WINDOW(data);
	document = GIDE_DOCUMENT(widget);

	/* Set red text color style */
	style = gtk_style_new();
	style->fg[0] = color;
	
	gI_window_set_doc_label_style( window, document, style );
	
	gI_window_update_title( window );
	gI_window_set_statusbar( window );
}

static void
gI_window_doc_unmodified( GtkWidget *widget, gpointer data )
{
	GideWindow *window;
	GideDocument *document;
	GtkStyle *style;
	GdkColor color = {0, 0, 0, 0};
	
	g_return_if_fail(data);
		
	g_assert(IS_GIDE_WINDOW(data));
	g_assert(IS_GIDE_DOCUMENT(widget));

	window = GIDE_WINDOW(data);
	document = GIDE_DOCUMENT(widget);

	/* Set black text color style */
	style = gtk_style_new();
	style->fg[0] = color;
	
	gI_window_set_doc_label_style( window, document, style );

	gI_window_update_title( window );
	gI_window_set_statusbar( window );
}

static void
gI_window_doc_readonly( GtkWidget *widget, gpointer data )
{
	GideWindow *window;
	GideDocument *document;
	
	g_assert(IS_GIDE_WINDOW(data));
	g_assert(IS_GIDE_DOCUMENT(widget));

	window = GIDE_WINDOW(data);
	document = GIDE_DOCUMENT(widget);
	
	g_return_if_fail( window );

	command_list_release (document->undo_commands);
	command_list_release (document->redo_commands);
	document->undo_commands = document->redo_commands = NULL;
	command_undo_redo_menu_labels( document );

	gI_menus_set_sensitive_readonly(window, TRUE);			
	gI_window_update_title( window );
	gI_window_set_statusbar( window );
}

static void
gI_window_doc_unreadonly( GtkWidget *widget, gpointer data )
{
	GideWindow *window;
	
	g_assert(IS_GIDE_WINDOW(data));

	window = GIDE_WINDOW(data);

	g_return_if_fail( window );

	gI_menus_set_sensitive_readonly(window, FALSE);		
	gI_window_update_title( window );
	gI_window_set_statusbar( window );
}

static void
gI_window_doc_source( GtkWidget *widget, gchar *filename, gpointer data )
{
	GideWindow *window;
	GideDocument *document;
	gpointer hbox;
	gchar *basename;

	g_return_if_fail(data);
	
	g_assert(IS_GIDE_WINDOW(data));
	g_assert(IS_GIDE_DOCUMENT(widget));

	window = GIDE_WINDOW(data);
	document = GIDE_DOCUMENT(widget);
	hbox = gtk_object_get_data( GTK_OBJECT(widget), "GideWindow::hbox");

	if ( filename ) {
		basename = file_strip_name(filename);

		gI_window_set_doc_label( window, document, basename );
	}

	gI_window_update_title( window );
	gI_window_set_statusbar( window );
}

static void
gI_window_doc_destroy( GtkWidget *widget, gpointer data )
{
	GideWindow *window;
	GideDocument *document;
	gint length, i;
	
	g_return_if_fail(data);
	
	g_assert(IS_GIDE_WINDOW(data));
	g_assert(IS_GIDE_DOCUMENT(widget));

	window = GIDE_WINDOW(data);
	document = GIDE_DOCUMENT(widget);

	length = gI_window_num_docs( window );
	for (i=0; i<length; i++) {
		if ( (gpointer)document
		     == g_list_nth_data(window->documents, i)) {
			gtk_notebook_remove_page(
				GTK_NOTEBOOK( window->notebook ), i );
			window->documents =
				g_list_remove( window->documents, document );
			break;
		}
	}
	
	gI_window_update_title( window );
	gI_window_set_statusbar( window );
}

static void
gI_window_destroy( GtkWidget *widget, gpointer data )
{
	GideWindow *window;
	GideDocument *document;
	gI_project* project;
	gint nod;
	gint i;

	g_assert(IS_GIDE_WINDOW(widget));

	window = GIDE_WINDOW( widget );

	/* close opened project */
	if((project = gI_project_get_current()))
	{
		gI_project_close(NULL, NULL);
	}

	nod = gI_window_num_docs( window );
	for(i = 1; i <= nod; i++) {
		gtk_notebook_set_page(GTK_NOTEBOOK(window->notebook), i);
		document = gI_window_get_current_doc(window);
		gtk_object_destroy ( GTK_OBJECT(document) );
	}

	gtk_widget_destroy(window->popup);
}





