#include <config.h>

#include <gdl/gdl.h>
#include <liboaf/liboaf.h>
#include <libgide/libgide.h>

#include "tools.h"
#include "shell.h"
#include "e-splash.h"

typedef struct 
{
	char *name;
	Bonobo_PropertyBag tool;
} InstalledTool;

static char *
get_icon_path_for_component_info (const OAF_ServerInfo *info)
{
        OAF_Property *property;
        const char *shell_component_icon_value;

        /* FIXME: liboaf is not const-safe.  */
        property = oaf_server_info_prop_find ((OAF_ServerInfo *) info,
                                              "gide:shell_component_icon");
        if (property == NULL || property->v._d != OAF_P_STRING)
		return NULL;
	
        shell_component_icon_value = property->v._u.value_string;
	
        if (g_path_is_absolute (shell_component_icon_value))
                return g_strdup (shell_component_icon_value);
	
        else
                return g_concat_dir_and_file (GIDE_IMAGES, shell_component_icon_value);
}

static InstalledTool *
initialize_tool (GideShell *shell, OAF_ServerInfo *s)
{
	Bonobo_PropertyBag tool = CORBA_OBJECT_NIL;
	
	CORBA_Environment ev;
	InstalledTool *ret;

	CORBA_exception_init (&ev);

	g_print ("Installing Tool: %s\n", 
		 oaf_server_info_prop_lookup (s, "name", NULL));

	tool = oaf_activate_from_id (s->iid, 0, NULL, &ev);
	if (ev._major == CORBA_NO_EXCEPTION && !CORBA_Object_is_nil (tool,
								     &ev)) {
		bonobo_property_bag_client_set_value_string (tool, 
							     "gide-shell",
							     shell->id,
							     &ev);
	} else {
		if (ev._major != CORBA_NO_EXCEPTION) {
			g_print ("Exception activating %s: %s",  
				 oaf_server_info_prop_lookup (s, "name", NULL),
				 ev._repo_id);
		} else if (!CORBA_Object_is_nil (tool, &ev)) {
			g_print ("Null object while activating %s\n", 
				 oaf_server_info_prop_lookup (s, "name", NULL));
		}
		
		
	}
	
	if (!CORBA_Object_is_nil (tool, &ev)) {
		ret = g_new0 (InstalledTool, 1);
		ret->name = g_strdup (oaf_server_info_prop_lookup (s, "name", NULL));
		ret->tool = tool;
	} else {
		ret = NULL;
	}

	CORBA_exception_free (&ev);


	return ret;
}

void
tools_init (GideWindow *win, ESplash *splash)
{
	GideShell *shell;
	CORBA_Environment ev;
	OAF_ServerInfoList *oaf_result;
	CORBA_char *query;
	GSList *icon_tools = NULL;
	GSList *no_icon_tools = NULL;
	GSList *l;
	
	GList *tools = NULL;

	shell = gide_shell_new (win);

	gtk_object_set_data (GTK_OBJECT (win), "GideShell", shell);
	
	CORBA_exception_init (&ev);

	query = "repo_ids.has ('IDL:Bonobo/PropertyBag:1.0') AND gide:shell_component == 'true'";

	oaf_result = oaf_query (query, NULL, &ev);

	GDL_TRACE ();

	if (ev._major == CORBA_NO_EXCEPTION && oaf_result != NULL) {
		int i;
		
		GDL_TRACE ();

		if (splash) {
			for (i = 0; i < oaf_result->_length; i++) {
				OAF_ServerInfo *s = &oaf_result->_buffer[i];
				GdkPixbuf *icon_pixbuf;
				char *icon_path;
				
				GDL_TRACE ();
		
				icon_path = get_icon_path_for_component_info (s);
				if (icon_path) {
					icon_pixbuf = gdk_pixbuf_new_from_file (icon_path);
					e_splash_add_icon (splash, icon_pixbuf);
					gdk_pixbuf_unref (icon_pixbuf);
					g_free (icon_path);
					icon_tools = 
						g_slist_prepend (icon_tools,
								 s);
				} else {
					no_icon_tools = 
						g_slist_prepend (no_icon_tools,
								 s);
				}
				
			}
		}
		if (no_icon_tools && splash) {
			GdkPixbuf *icon_pixbuf = gdk_pixbuf_new_from_file (GIDE_IMAGES "/unknown-components.png");
			e_splash_add_icon (splash, icon_pixbuf);
			gdk_pixbuf_unref (icon_pixbuf);
		}
			
		while (gtk_events_pending ())
			gtk_main_iteration ();
		
		i = 0;
		for (l = icon_tools, i = 0; l != NULL; 
		     l = l->next, i++) {			
			OAF_ServerInfo *s = l->data;
			InstalledTool *t;

			t = initialize_tool (shell, s);
			if (t) 
				tools = g_list_prepend (tools, t);
			
			if (splash) {
				e_splash_set_icon_highlight (splash, i, TRUE);
			}
			
			while (gtk_events_pending ())
				gtk_main_iteration ();
		}
		/* i now points at the "all the other tools" icon if one
		 * exists. */
		if (no_icon_tools) {
			for (l = no_icon_tools; l != NULL; l = l->next) {
				OAF_ServerInfo *s = l->data;
				InstalledTool *t;
				
				t = initialize_tool (shell, s);
				if (t)
					tools = g_list_prepend (tools, t);
				
				while (gtk_events_pending ())
					gtk_main_iteration ();
			}
			if (splash) {
				e_splash_set_icon_highlight (splash, i, TRUE);
			}
			
			while (gtk_events_pending ())
				gtk_main_iteration ();
		}
		
		CORBA_free (oaf_result);
	} else if (ev._major == CORBA_USER_EXCEPTION) {
		g_warning ("Error querying oaf.");
	} else {
		g_warning ("No plugins found.");
	}
	
	gtk_object_set_data (GTK_OBJECT (win), "InstalledTools", tools);
}

void 
tools_cleanup (GideWindow *win)
{
	CORBA_Environment ev;
	GideShell *shell = gtk_object_get_data (GTK_OBJECT (win), "GideShell");
	GList *tools = gtk_object_get_data (GTK_OBJECT (win), "InstalledTools");
	GList *i;
	
	CORBA_exception_init (&ev);
	for (i = tools; i != NULL; i = i->next) {
		InstalledTool *tool = i->data;
		g_print ("Uninstalling %s\n", tool->name);

		bonobo_property_bag_client_set_value_string (tool->tool, 
							     "gide-shell",
							     "", &ev);

		g_free (tool->name);
		CORBA_Object_release (tool->tool, &ev);
		g_free (tool);
	}	
	CORBA_exception_free (&ev);

	g_list_free (tools);
	
	bonobo_object_unref (BONOBO_OBJECT (shell));
}
