/*
 *  GMF: The GNOME Media Framework
 *
 *  Copyright (C) 1999 Elliot Lee
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of the
 *  License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this library; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Author: Elliot Lee <sopwith@redhat.com>
 *
 */

#include <gmf.h>
#include <libgmf/gnome-genericfactory.h>
#include <fcntl.h>
#include <sys/types.h>
#include <unistd.h>

#include "Gnumeric.h"

typedef struct {
  CORBA_ORB orb;
  CORBA_Environment *ev;
  GMFTimeReference *timeref;
  GnomeGenericFactory *factory;
  CORBA_Object gnumeric_handle;
} StatsInfo;

typedef struct {
  StatsInfo *ai;
  GMFFilter *filter;
  CORBA_Object sheet_handle;
  GList *value_queue;
  GList *velocity_queue;
} FiltInfo;

static void stats_create_filter(GnomeGenericFactory *factory,
				    const char *goad_id, GNOME_stringlist *params,
				    CORBA_Object *new_object, StatsInfo *ai);

static void stats_in_get_pipe(GMFFilter *filter,
				  GMF_Direction pDirection,
				  GMFPipe **pipe,
				  FiltInfo *fi);

int main(int argc, char *argv[])
{
  CORBA_Environment ev;
  poptContext ctx;
  StatsInfo appinfo;
  static volatile int barrier = -1;

  if(barrier == -1) barrier = GPOINTER_TO_INT(getenv("STATISTICS_MAIN_BARRIER"));
  while(barrier);

  CORBA_exception_init(&ev);
  appinfo.ev = &ev;
  appinfo.orb = gnome_CORBA_init_with_popt_table("statistics", VERSION,
						 &argc, argv, NULL, 0,
						 &ctx, GNORBA_INIT_SERVER_FUNC,
						 &ev);

  appinfo.gnumeric_handle = goad_server_activate_with_id(NULL, "IDL:GNOME:Gnumeric:Workbook:1.0", 0, NULL);
  g_assert(!CORBA_Object_is_nil(appinfo.gnumeric_handle, &ev));

  appinfo.timeref = GMF_TIME_REFERENCE(gmf_time_reference_new());
  appinfo.factory = GNOME_GENERIC_FACTORY(gnome_generic_factory_new("gmf-filter-statistics-factory"));

  gtk_signal_connect(GTK_OBJECT(appinfo.factory), "create_object",
		     GTK_SIGNAL_FUNC(stats_create_filter), &appinfo);

  gtk_main();

  return 0;
}

static void
stats_create_filter(GnomeGenericFactory *factory,
		    const char *goad_id, GNOME_stringlist *params,
		    CORBA_Object *new_object, StatsInfo *ai)
{
  GMFFilter *new_filter;
  GMF_Filter_Type ftype;
  FiltInfo *fi;
  int i;

  if(!strcmp(goad_id, "gmf-filter-statistics")) {
    ftype = GMF_Filter_RENDERER;
  } else {
    return; /* Can't activate anything else */
  }

  new_filter = GMF_FILTER(gmf_filter_new(ftype, goad_id));
  g_assert(new_filter);

  fi = g_new0(FiltInfo, 1);
  fi->ai = ai;
  if(!strcmp(goad_id, "gmf-filter-statistics")) {
    gtk_signal_connect(GTK_OBJECT(new_filter), "get_pipe",
		       GTK_SIGNAL_FUNC(stats_in_get_pipe), fi);
  } else
    g_assert(!"Not reached!");

  fi->filter = new_filter;

  {
    char buf[32];
    g_snprintf(buf, sizeof(buf), "GMF stats %d", rand());
    fi->sheet_handle = GNOME_Gnumeric_Workbook_sheet_new(ai->gnumeric_handle, buf, ai->ev);
  }

  g_assert(ai->ev->_major == CORBA_NO_EXCEPTION);
  g_assert(!CORBA_Object_is_nil(fi->sheet_handle, ai->ev));

  for(i = 0; i < 20; i++) {
    char buf[32];

    g_snprintf(buf, sizeof(buf), "-0.%.2d", i);

    GNOME_Gnumeric_Sheet_cell_set_text(fi->sheet_handle, 0, i, buf, ai->ev);
  }

  *new_object = CORBA_Object_duplicate(new_filter->corba_object, ai->ev);
}

static void stats_pipe_can_process(GMFPipe *pipe, GMF_MediaTypeInfo *type_info,
				 gboolean *retval)
{
  /* We can handle anything */

  *retval = TRUE;
}

static void stats_pipe_processable_types(GMFPipe *pipe,
				       GMF_MediaTypeInfoList **out_typelist,
				       FiltInfo *fi)
{
  GMF_MediaTypeInfoList *list;
  GMF_MediaTypeInfo *ti;

  list = GMF_MediaTypeInfoList__alloc();
  list->_length = 1;
  list->_buffer = CORBA_sequence_GMF_MediaTypeInfo_allocbuf(list->_length);

  ti = &list->_buffer[0];
  {
    memset(ti, 0, sizeof(*ti));
    ti->majorType = GMF_MEDIA_DATA;
    ti->minorType = CORBA_string_dup("*");
    ti->typeData._type = (CORBA_TypeCode)CORBA_Object_duplicate((CORBA_Object)TC_null, fi->ai->ev);
    ti->formatData._type = (CORBA_TypeCode)CORBA_Object_duplicate((CORBA_Object)TC_null, fi->ai->ev);
  }

  *out_typelist = list;
}

static void stats_pipe_in_handle_sample(GMFPipe *pipe,
					GMF_Sample *sample,
					gboolean must_copy,
					FiltInfo *fi)
{
  GMF_TimeVal thislag, curtime;
  guint lagval, velval;
  int i;
  GList *ltmp;
  char buf[32];

#if 0
  curtime = gmf_time_reference_current_time(fi->ai->timeref);
  thislag = gmf_time_subtract(curtime, sample->sInfo.mediaStartTime);

  lagval = (thislag.tvSec * 1000000) + thislag.tvUsec;
#endif
  lagval = (guint)sample->sData._buffer[1];
  velval = (guint)sample->sData._buffer[2];

  fi->value_queue = g_list_append(fi->value_queue, GUINT_TO_POINTER(lagval));
  fi->velocity_queue = g_list_append(fi->velocity_queue, GUINT_TO_POINTER(velval));

  if(g_list_length(fi->value_queue) > 20) {

    ltmp = fi->value_queue;
    fi->value_queue = g_list_remove_link(fi->value_queue, ltmp);
    g_list_free_1(ltmp);
  }

  if(g_list_length(fi->velocity_queue) > 20) {

    ltmp = fi->velocity_queue;
    fi->velocity_queue = g_list_remove_link(fi->velocity_queue, ltmp);
    g_list_free_1(ltmp);
  }

  for(ltmp = g_list_last(fi->value_queue), i = 0; ltmp && (i < 20); i++, ltmp = g_list_previous(ltmp)) {
    lagval = GPOINTER_TO_UINT(ltmp->data);
    g_snprintf(buf, sizeof(buf), "%u", lagval);
    GNOME_Gnumeric_Sheet_cell_set_text(fi->sheet_handle, 1, i, buf, fi->ai->ev);
  }

  for(ltmp = g_list_last(fi->velocity_queue), i = 0; ltmp && (i < 20); i++, ltmp = g_list_previous(ltmp)) {
    velval = GPOINTER_TO_UINT(ltmp->data);
    g_snprintf(buf, sizeof(buf), "%u", velval);
    GNOME_Gnumeric_Sheet_cell_set_text(fi->sheet_handle, 2, i, buf,
				       fi->ai->ev);
  }
}

static void stats_in_get_pipe(GMFFilter *filter,
			      GMF_Direction pDirection,
			      GMFPipe **pipe,
			      FiltInfo *fi)
{
  GMFPipe *newpipe;

  g_return_if_fail(pDirection == GMF_IN);

  newpipe = GMF_PIPE(gmf_pipe_new(filter, pDirection,
				  GMF_Transport_UNIX_SOCKETS|GMF_Transport_CORBA));

  g_assert(newpipe);

  gtk_signal_connect(GTK_OBJECT(newpipe), "can_process_type",
		     GTK_SIGNAL_FUNC(stats_pipe_can_process), fi);
  gtk_signal_connect(GTK_OBJECT(newpipe), "get_processable_types",
		     GTK_SIGNAL_FUNC(stats_pipe_processable_types), fi);

  gtk_signal_connect(GTK_OBJECT(newpipe), "receive_sample",
		     GTK_SIGNAL_FUNC(stats_pipe_in_handle_sample), fi);

  *pipe = newpipe;
}
