/*
 *  GMF: The GNOME Media Framework
 *
 *  Copyright (C) 1999 Elliot Lee
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Author: Elliot Lee <sopwith@redhat.com>
 *
 */

#include "gnome-genericfactory.h"

enum { SUPPORTS, CREATE_OBJECT, LAST_SIGNAL };
static guint factory_signals[LAST_SIGNAL] = { 0 };

/*** App-specific servant structures ***/

typedef struct {
  POA_GNOME_GenericFactory servant;
  GnomeGenericFactory *gtk_obj;
} impl_POA_GNOME_GenericFactory;

/*** Implementation stub prototypes ***/

static CORBA_boolean
impl_GNOME_GenericFactory_supports(impl_POA_GNOME_GenericFactory *servant,
				   CORBA_char * obj_goad_id,
				   CORBA_Environment *ev);

static CORBA_Object
impl_GNOME_GenericFactory_create_object(impl_POA_GNOME_GenericFactory *servant,
					CORBA_char * goad_id,
					GNOME_stringlist* params,
					CORBA_Environment *ev);

/*** epv structures ***/

static PortableServer_ServantBase__epv impl_GNOME_GenericFactory_base_epv = {
  NULL, /* _private data */
  NULL, /* finalize routine */
  NULL, /* default_POA routine */
};

static POA_GNOME_GenericFactory__epv impl_GNOME_GenericFactory_epv = {
  NULL, /* _private */
  (gpointer)&impl_GNOME_GenericFactory_supports,
  (gpointer)&impl_GNOME_GenericFactory_create_object,
};

/*** vepv structures ***/

static POA_GNOME_GenericFactory__vepv impl_GNOME_GenericFactory_vepv = {
  &impl_GNOME_GenericFactory_base_epv,
  &impl_GNOME_GenericFactory_epv,
};

/*** Stub implementations ***/

static
GNOME_GenericFactory
impl_GNOME_GenericFactory__create(impl_POA_GNOME_GenericFactory *newservant,
				  PortableServer_POA poa,
				  CORBA_Environment *ev)
{
  GNOME_GenericFactory retval;
  PortableServer_ObjectId *objid;

  newservant->servant.vepv = &impl_GNOME_GenericFactory_vepv;
  POA_GNOME_GenericFactory__init((PortableServer_Servant)newservant, ev);
  objid = PortableServer_POA_activate_object(poa, newservant, ev);
  CORBA_free(objid);
  retval = PortableServer_POA_servant_to_reference(poa, newservant, ev);
  
  return retval;
}

static CORBA_boolean
impl_GNOME_GenericFactory_supports(impl_POA_GNOME_GenericFactory *servant,
				   CORBA_char * obj_goad_id,
				   CORBA_Environment *ev)
{
  CORBA_boolean retval = CORBA_FALSE;
  gboolean setme = FALSE;

  gtk_signal_emit(GTK_OBJECT(servant->gtk_obj), factory_signals[SUPPORTS],
		  obj_goad_id, &setme);

  retval = (CORBA_boolean)setme;
  
  return retval;
}


static CORBA_Object
impl_GNOME_GenericFactory_create_object(impl_POA_GNOME_GenericFactory *servant,
					CORBA_char * goad_id,
					GNOME_stringlist* params,
					CORBA_Environment *ev)
{
  CORBA_Object retval = CORBA_OBJECT_NIL;

  gtk_signal_emit(GTK_OBJECT(servant->gtk_obj), factory_signals[CREATE_OBJECT],
		  goad_id, params, &retval);

  if(retval == CORBA_OBJECT_NIL)
    CORBA_exception_set(ev, CORBA_USER_EXCEPTION,
			ex_GNOME_GenericFactory_CannotActivate, NULL);

  return retval;
}

static void gnome_generic_factory_class_init (GnomeGenericFactoryClass * klass);
static void gnome_generic_factory_init (GnomeGenericFactory *factory);
static void gnome_generic_factory_destroy(GnomeGenericFactory *factory);

GtkType
gnome_generic_factory_get_type (void)
{
  static GtkType gmf_filter_type = 0;
  if (!gmf_filter_type)
    {
      static const GtkTypeInfo gmf_filter_info =
      {
	"GnomeGenericFactory",
	sizeof (GnomeGenericFactory),
	sizeof (GnomeGenericFactoryClass),
	(GtkClassInitFunc) gnome_generic_factory_class_init,
	(GtkObjectInitFunc) gnome_generic_factory_init,
	NULL, NULL,		/* reserved 1 & 2 */
	NULL
      };

      gmf_filter_type = gtk_type_unique (gtk_object_get_type(), &gmf_filter_info);
    }

  return gmf_filter_type;
}

static void
gnome_generic_factory_init(GnomeGenericFactory *factory)
{
  PortableServer_POA poa;
  PortableServer_POAManager pm;
  CORBA_Environment ev;

  /* Bad hack to avoid allocating a separate block for the servant */
  factory->self = factory;

  CORBA_exception_init(&ev);
  poa = (PortableServer_POA)
    CORBA_ORB_resolve_initial_references(gnome_CORBA_ORB(), "RootPOA", &ev);
  pm = PortableServer_POA__get_the_POAManager(poa, &ev);
  PortableServer_POAManager_activate(pm, &ev);
  factory->corba_object = impl_GNOME_GenericFactory__create((impl_POA_GNOME_GenericFactory *)&(factory->servant),
							    poa, &ev);
  factory->poa = poa;
  CORBA_exception_free(&ev);
}

typedef void (*GtkSignal_NONE__STRING_POINTER) (GtkObject *object, 
						const char * arg1,
						gpointer arg2,
						gpointer user_data);
static void
gtk_marshal_NONE__STRING_POINTER (GtkObject    *object, 
				  GtkSignalFunc func, 
				  gpointer      func_data, 
				  GtkArg       *args)
{
  GtkSignal_NONE__STRING_POINTER rfunc;

  rfunc = (GtkSignal_NONE__STRING_POINTER) func;
  (* rfunc) (object,
	     GTK_VALUE_STRING(args[0]),
	     GTK_VALUE_POINTER(args[1]),
	     func_data);
}

typedef void (*GtkSignal_NONE__STRING_POINTER_POINTER) (GtkObject *object, 
							const char * arg1,
							gpointer arg2,
							gpointer arg3,
							gpointer user_data);
static void
gtk_marshal_NONE__STRING_POINTER_POINTER (GtkObject    *object, 
					  GtkSignalFunc func, 
					  gpointer      func_data, 
					  GtkArg       *args)
{
  GtkSignal_NONE__STRING_POINTER_POINTER rfunc;

  rfunc = (GtkSignal_NONE__STRING_POINTER_POINTER) func;
  (* rfunc) (object,
	     GTK_VALUE_STRING(args[0]),
	     GTK_VALUE_POINTER(args[1]),
	     GTK_VALUE_POINTER(args[2]),
	     func_data);
}

static void 
gnome_generic_factory_class_init (GnomeGenericFactoryClass * klass)
{
  GtkObjectClass *object_class = GTK_OBJECT_CLASS (klass);

  object_class->destroy = (void (*)(GtkObject *))gnome_generic_factory_destroy;

  factory_signals[SUPPORTS] =
    gtk_signal_new("supports", GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GnomeGenericFactoryClass, supports),
		   gtk_marshal_NONE__STRING_POINTER,
		   GTK_TYPE_NONE, 2,
		   GTK_TYPE_STRING, GTK_TYPE_POINTER);

  factory_signals[CREATE_OBJECT] =
    gtk_signal_new("create_object", GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GnomeGenericFactoryClass, create_object),
		   gtk_marshal_NONE__STRING_POINTER_POINTER,
		   GTK_TYPE_NONE, 3,
		   GTK_TYPE_STRING, GTK_TYPE_POINTER, GTK_TYPE_POINTER);

  gtk_object_class_add_signals(object_class, factory_signals, LAST_SIGNAL);
}

static void
gnome_generic_factory_destroy(GnomeGenericFactory *factory)
{
  PortableServer_ObjectId *objid;
  CORBA_Environment ev;

  CORBA_exception_init(&ev);

  if(factory->goad_id) {
    goad_server_unregister(CORBA_OBJECT_NIL, factory->goad_id, "object", &ev);
    g_free(factory->goad_id);
  }

  objid = PortableServer_POA_servant_to_id(factory->poa,
					   &(factory->servant), &ev);
  PortableServer_POA_deactivate_object(factory->poa, objid, &ev);
  CORBA_free(objid);

  POA_GNOME_GenericFactory__fini((PortableServer_Servant)&(factory->servant),
				 &ev);
  CORBA_Object_release((CORBA_Object)factory->poa, &ev);

  CORBA_exception_free(&ev);
}

GtkObject *
gnome_generic_factory_new(const char *goad_id)
{
  GtkObject *retval;
  GnomeGenericFactory *factory;
  CORBA_Environment ev;
  static volatile int barrier = -1;

  if(barrier == -1)
    barrier = GPOINTER_TO_INT(getenv("GNOME_GENERICFACTORY_BARRIER"));

  retval = gtk_type_new(gnome_generic_factory_get_type());
  factory = GNOME_GENERIC_FACTORY(retval);
  CORBA_exception_init(&ev);

  while(barrier);

  if(goad_server_register(CORBA_OBJECT_NIL, factory->corba_object, goad_id,
			  "object", &ev)) {
    gtk_object_sink(retval);
    return NULL;
  }
  factory->goad_id = g_strdup(goad_id);

  return retval;
}
