/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/* File: keygrab.c
 * Purpose: GNOME Keyboard switcher keyboard grabbing functions
 *
 * Copyright (C) 1998-2000 Free Software Foundation
 * Authors: Szabolcs Ban  <shooby@gnome.hu>
 *          Chema Celorio <chema@celorio.com>
 *
 * Thanks for aid of George Lebl <jirka@5z.com> and solidarity
 * Balazs Nagy <js@lsc.hu>, Charles Levert <charles@comm.polymtl.ca>
 * and Emese Kovacs <emese@gnome.hu> for her docs and ideas.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <X11/keysym.h>
#include <X11/Xmd.h>
#include <X11/Xlib.h>
#include <gdk/gdkx.h>
#include <sys/stat.h>
#include <X11/Xlib.h>
#include "gkb.h"

int NumLockMask, CapsLockMask, ScrollLockMask;

/* --- keygrab --- */

static gboolean
string_empty (const char *string)
{
  if (string == NULL || string[0] == '\0')
    return TRUE;
  else
    return FALSE;
}

gboolean
convert_string_to_keysym_state (const char *string,
				guint * keysym, guint * state)
{
  char *s, *p;

  g_return_val_if_fail (keysym != NULL, FALSE);
  g_return_val_if_fail (state != NULL, FALSE);

  *state = 0;
  *keysym = 0;

  if (string_empty (string) ||
      strcmp (string, "Disabled") == 0 || strcmp (string, _("Disabled")) == 0)
    return FALSE;

  s = g_strdup (string);

  gdk_error_trap_push ();

  p = strtok (s, "-");
  while (p != NULL)
    {
      if (strcmp (p, "Control") == 0)
	{
	  *state |= GDK_CONTROL_MASK;
	}
      else if (strcmp (p, "Lock") == 0)
	{
	  *state |= GDK_LOCK_MASK;
	}
      else if (strcmp (p, "Shift") == 0)
	{
	  *state |= GDK_SHIFT_MASK;
	}
      else if (strcmp (p, "Mod1") == 0)
	{
	  *state |= GDK_MOD1_MASK;
	}
      else if (strcmp (p, "Mod2") == 0)
	{
	  *state |= GDK_MOD2_MASK;
	}
      else if (strcmp (p, "Mod3") == 0)
	{
	  *state |= GDK_MOD3_MASK;
	}
      else if (strcmp (p, "Mod4") == 0)
	{
	  *state |= GDK_MOD4_MASK;
	}
      else if (strcmp (p, "Mod5") == 0)
	{
	  *state |= GDK_MOD5_MASK;
	}
      else
	{
	  *keysym = gdk_keyval_from_name (p);
	  if (*keysym == 0)
	    {
	      gdk_flush ();
	      gdk_error_trap_pop ();
	      g_free (s);
	      return FALSE;
	    }
	}
      p = strtok (NULL, "-");
    }

  gdk_flush ();
  gdk_error_trap_pop ();

  g_free (s);

  if (*keysym == 0)
    return FALSE;

  return TRUE;
}

char *
convert_keysym_state_to_string (guint keysym, guint state)
{
  GString *gs;
  char *sep = "";
  char *key;

  if (keysym == 0)
    return g_strdup (_("Disabled"));

  gdk_error_trap_push ();
  key = gdk_keyval_name (keysym);
  gdk_flush ();
  gdk_error_trap_pop ();
  if (!key)
    return NULL;

  gs = g_string_new (NULL);

  if (state & GDK_CONTROL_MASK)
    {
      /*g_string_append(gs, sep); */
      g_string_append (gs, "Control");
      sep = "-";
    }
  if (state & GDK_LOCK_MASK)
    {
      g_string_append (gs, sep);
      g_string_append (gs, "Lock");
      sep = "-";
    }
  if (state & GDK_SHIFT_MASK)
    {
      g_string_append (gs, sep);
      g_string_append (gs, "Shift");
      sep = "-";
    }
  if (state & GDK_MOD1_MASK)
    {
      g_string_append (gs, sep);
      g_string_append (gs, "Mod1");
      sep = "-";
    }
  if (state & GDK_MOD2_MASK)
    {
      g_string_append (gs, sep);
      g_string_append (gs, "Mod2");
      sep = "-";
    }
  if (state & GDK_MOD3_MASK)
    {
      g_string_append (gs, sep);
      g_string_append (gs, "Mod3");
      sep = "-";
    }
  if (state & GDK_MOD4_MASK)
    {
      g_string_append (gs, sep);
      g_string_append (gs, "Mod4");
      sep = "-";
    }
  if (state & GDK_MOD5_MASK)
    {
      g_string_append (gs, sep);
      g_string_append (gs, "Mod5");
      sep = "-";
    }

  g_string_append (gs, sep);
  g_string_append (gs, key);

  {
    char *ret = gs->str;
    g_string_free (gs, FALSE);
    return ret;
  }
}

static GtkWidget *grab_dialog;

static GdkFilterReturn
grab_key_filter (GdkXEvent * gdk_xevent, GdkEvent * event, gpointer data)
{
  XEvent *xevent = (XEvent *) gdk_xevent;
  GtkEntry *entry;
  KeySym keysym;
  guint  state;
  char buf[10];
  char *key;
  guint newkey;

  if (xevent->type != KeyRelease)
    return GDK_FILTER_CONTINUE;

  entry = GTK_ENTRY (data);

  state = xevent->xkey.state & (GDK_SHIFT_MASK | GDK_CONTROL_MASK | GDK_MOD1_MASK);

  XLookupString (&xevent->xkey, buf, 0, &keysym, NULL);
  
  gdk_keyboard_ungrab (GDK_CURRENT_TIME);
  gtk_widget_destroy (grab_dialog);
  gdk_window_remove_filter (gdk_get_default_root_window(), grab_key_filter, data);
  
  /* Esc cancels */
  if (event->key.keyval == GDK_Escape) 
    return GDK_FILTER_REMOVE;
  
  key = convert_keysym_state_to_string (event->key.keyval, event->key.state);
  gtk_entry_set_text (GTK_ENTRY (entry), key ? key : "");
  
  newkey = XKeysymToKeycode (GDK_DISPLAY (), gkb->keysym);

  gkb_xungrab (key, gkb->state);

  gkb->key = g_strdup (key);
  g_free (key);
  
  convert_string_to_keysym_state (gkb->key, &gkb->keysym, &gkb->state);

  newkey = XKeysymToKeycode (GDK_DISPLAY (), gkb->keysym);

  gkb_xgrab ( newkey, gkb->state);
  
  panel_applet_gconf_set_string (PANEL_APPLET (gkb->applet), "key", gkb->key, NULL);
  
  return GDK_FILTER_REMOVE;
}

void
grab_button_pressed (GtkButton * button, gpointer data)
{
  GtkWidget *frame;
  GtkWidget *box;
  GtkWidget *label;
  grab_dialog = gtk_window_new (GTK_WINDOW_POPUP);

  gdk_keyboard_grab (gdk_get_default_root_window(), FALSE, GDK_CURRENT_TIME);
  gdk_window_add_filter( gdk_get_default_root_window (), grab_key_filter, data);

  g_object_set (G_OBJECT (grab_dialog),
    		"allow_grow", FALSE,
    	        "allow_shrink", FALSE,
                "resizable", FALSE,
                NULL);

  gtk_window_set_position (GTK_WINDOW (grab_dialog), GTK_WIN_POS_CENTER);
  gtk_window_set_modal (GTK_WINDOW (grab_dialog), TRUE);

  frame = gtk_frame_new (NULL);
  gtk_container_add (GTK_CONTAINER (grab_dialog), frame);

  box = gtk_hbox_new (0, 0);
  gtk_container_set_border_width (GTK_CONTAINER (box), 20);
  gtk_container_add (GTK_CONTAINER (frame), box);

  label = gtk_label_new (_("Press a key or press Esc to cancel"));
  gtk_container_add (GTK_CONTAINER (box), label);

  gtk_widget_show_all (grab_dialog);
  return;
}

init_xmaps () {
  XModifierKeymap *xmk=NULL;
  KeyCode *map;
  int m, k;

  xmk=XGetModifierMapping(GDK_DISPLAY());
  if(xmk)
  {
    map=xmk->modifiermap;
    for(m=0;m<8;m++)
	for(k=0;k<xmk->max_keypermod; k++, map++)
	{
	if(*map==XKeysymToKeycode(GDK_DISPLAY(), XK_Num_Lock))
		NumLockMask=(1<<m);
	if(*map==XKeysymToKeycode(GDK_DISPLAY(), XK_Caps_Lock))
	  CapsLockMask=(1<<m);
        if(*map==XKeysymToKeycode(GDK_DISPLAY(), XK_Scroll_Lock))
          ScrollLockMask=(1<<m);
      }
    XFreeModifiermap(xmk);
  }
}


void
gkb_xgrab (gint keycode, gint modifiers) {

  init_xmaps();

  XGrabKey (GDK_DISPLAY(), keycode, modifiers,
            GDK_ROOT_WINDOW(), True, GrabModeAsync, GrabModeAsync);
  XGrabKey (GDK_DISPLAY(), keycode, modifiers|NumLockMask,
            GDK_ROOT_WINDOW(), True, GrabModeAsync, GrabModeAsync);
  XGrabKey (GDK_DISPLAY(), keycode, modifiers|CapsLockMask,
            GDK_ROOT_WINDOW(), True, GrabModeAsync, GrabModeAsync);
  XGrabKey (GDK_DISPLAY(), keycode, modifiers|ScrollLockMask,
            GDK_ROOT_WINDOW(), True, GrabModeAsync, GrabModeAsync);
  XGrabKey (GDK_DISPLAY(), keycode, modifiers|NumLockMask|CapsLockMask,
            GDK_ROOT_WINDOW(), True, GrabModeAsync, GrabModeAsync);
  XGrabKey (GDK_DISPLAY(), keycode, modifiers|NumLockMask|ScrollLockMask,
            GDK_ROOT_WINDOW(), True, GrabModeAsync, GrabModeAsync);
  XGrabKey (GDK_DISPLAY(), keycode, modifiers|CapsLockMask|ScrollLockMask,
            GDK_ROOT_WINDOW(), True, GrabModeAsync, GrabModeAsync);
  XGrabKey (GDK_DISPLAY(), keycode, modifiers|NumLockMask|CapsLockMask|ScrollLockMask,
            GDK_ROOT_WINDOW(), True, GrabModeAsync, GrabModeAsync);
}

void
gkb_xungrab (gint keycode, gint modifiers) {

  init_xmaps();

  XUngrabKey(GDK_DISPLAY(), keycode, modifiers, GDK_ROOT_WINDOW());
  XUngrabKey(GDK_DISPLAY(), keycode, modifiers|NumLockMask, GDK_ROOT_WINDOW());
  XUngrabKey(GDK_DISPLAY(), keycode, modifiers|CapsLockMask, GDK_ROOT_WINDOW());
  XUngrabKey(GDK_DISPLAY(), keycode, modifiers|ScrollLockMask, GDK_ROOT_WINDOW());
  XUngrabKey(GDK_DISPLAY(), keycode, modifiers|NumLockMask|CapsLockMask, GDK_ROOT_WINDOW());
  XUngrabKey(GDK_DISPLAY(), keycode, modifiers|NumLockMask|ScrollLockMask, GDK_ROOT_WINDOW());
  XUngrabKey(GDK_DISPLAY(), keycode, modifiers|CapsLockMask|ScrollLockMask, GDK_ROOT_WINDOW());
  XUngrabKey(GDK_DISPLAY(), keycode, modifiers|NumLockMask|CapsLockMask|ScrollLockMask, GDK_ROOT_WINDOW());

}
