/* GNOME DB library
 * Copyright (C) 1999-2000 Rodrigo Moya
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "config.h"
#include "gnome-db-config.h"
#include "gnome-db-util.h"
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-stock.h>

void
gnome_db_clear_clist (GtkCList *clist)
{
	g_return_if_fail(GTK_IS_CLIST(clist));
	gtk_clist_clear(clist);
}

GnomePixmap *
gnome_db_get_pixmap (const gchar *stock_pixmap)
{
	struct loaded_pixmap_info {
		gchar*       name;
		GnomePixmap* pixmap;
	};

	GList*                     node;
	struct loaded_pixmap_info* info;
	static GList*              loaded_pixmaps = NULL;
	
	g_return_val_if_fail(stock_pixmap != NULL, NULL);
	
	/* first, try to find if the pixmap is already loaded */
	node = g_list_first(loaded_pixmaps);
	while (node) {
		info = (struct loaded_pixmap_info *) node->data;
		if (info && !strcmp(info->name, stock_pixmap))
			return info->pixmap;
		node = g_list_next(node);
	}
	
	/* it's not loaded, so get it */
	info = g_new0(struct loaded_pixmap_info, 1);
	info->name = g_strdup(stock_pixmap);
	info->pixmap = GNOME_PIXMAP(gnome_stock_pixmap_widget(NULL, info->name));
	loaded_pixmaps = g_list_append(loaded_pixmaps, (gpointer) info);
	
	return info->pixmap;
}

void
gnome_db_show_about_dialog (const gchar *title)
{
	const gchar *authors[] = { "Michael Lausch <michael@lausch.at>",
	                           "Rodrigo Moya <rodrigo@gnome-db.org>",
	                           "Stephan Heinze <stephan.heinze@xcom.de>",
	                           "Vivien Malerba <malerba@gnome-db.org>",
	                           "Nick Gorham <nick@lurcher.org>",
	                           "Chris Wiegand <cwiegand@urgentmail.com>",
	                           "Akira TAGOH <tagoh@gnome-db.org>",
	                           "Carlos Perell Marn <carlos@gnome-db.org>",
	                           "Holger Thon <holger.thon@gnome-db.org>",
	                           "Reinhard Mller <reinhard@gnue.org>",
	                           "Gerhard Dieringer <gdieringer@compuserve.com>",
	                           NULL };
	GtkWidget *about = gnome_about_new (
		title, VERSION,
		_("Copyright The Free Software Foundation (C) 1998-2000"),
		authors,
		_("This program is part of the GNOME project for LINUX. "
		  "GNOME Data Access comes with ABSOLUTELY NO WARRANTY. "
		  "This is free software, and you are welcome to redistribute "
		  "it under the conditions of the GNU General Public Licence."),
		"gnome-db.png");
	gtk_widget_show(about);
}

void
gnome_db_show_error (const gchar *format, ...)
{
	va_list args;
	gchar sz[256];
	
	/* build the message string */
	va_start (args, format);
	vsprintf (sz, format, args);
	va_end (args);
	
	gnome_error_dialog (sz);
}

void
gnome_db_show_message (const gchar *format, ...)
{
	va_list args;
	gchar sz[256];
	
	/* build the message string */
	va_start(args, format);
	vsprintf(sz, format, args);
	va_end(args);
	
	gnome_ok_dialog(sz);
}

void
gnome_db_show_file (const gchar *filename)
{
	GtkWidget* dialog;
	GtkWidget* viewer;
	
	g_return_if_fail(filename != 0);
	
	dialog = gnome_dialog_new(filename, GNOME_STOCK_BUTTON_CLOSE, 0);
	viewer = gnome_less_new();
	gtk_widget_show(viewer);
	gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), viewer, 1, 1, 0);
	if (gnome_less_show_file(GNOME_LESS(viewer), filename)) {
		gnome_dialog_run_and_close(GNOME_DIALOG(dialog));
	}
	else gnome_db_show_error(_("Could not open file\n%s"), filename);
}

void
gnome_db_set_bg_color (GtkWidget *w, gint red, gint green, gint blue)
{
	GtkRcStyle* rc_style;

	g_return_if_fail (GTK_IS_WIDGET (w));

	rc_style = gtk_rc_style_new ();
	rc_style->bg[GTK_STATE_NORMAL].red = red;
	rc_style->bg[GTK_STATE_NORMAL].green = green;
	rc_style->bg[GTK_STATE_NORMAL].blue = blue;
	rc_style->color_flags[GTK_STATE_NORMAL] = GTK_RC_BG;
	gtk_widget_modify_style (w, rc_style);
	gtk_rc_style_unref (rc_style);
}

void
gnome_db_set_widget_tooltip (GtkWidget *w, const gchar *text)
{
	static GtkTooltips *tooltips = NULL;

	g_return_if_fail(GTK_IS_WIDGET(w));
	g_return_if_fail(text != NULL);

	if (gda_config_get_boolean(GNOME_DB_CONFIG_KEY_SHOW_TOOLTIPS)) {
		if (!tooltips)
			tooltips = gtk_tooltips_new();
		gtk_tooltips_set_tip(tooltips, w, text, NULL);
	}
}

static gint
get_config_int (const gchar *format, ...)
{
	gchar   buffer[2048];
	va_list args;
	
	g_return_val_if_fail(format, 0);
	
	va_start(args, format);
	vsprintf(buffer, format, args);
	va_end(args);
	
	return gda_config_get_int(buffer);
}

static void
set_config_int (gint value, const gchar *format, ...)
{
	gchar   buffer[2048];
	va_list args;
	
	g_return_if_fail(format);
	
	va_start(args, format);
	vsprintf(buffer, format, args);
	va_end(args);
	
	gda_config_set_int(buffer, value);
}

void
gnome_db_load_window_config (const gchar *name, GtkWidget *window)
{
	g_return_if_fail(name != NULL);
	g_return_if_fail(GTK_IS_WIDGET(window));
	
	gdk_window_move(window->window,
	                get_config_int("/apps/%s/Placement/%s_X",
	                               g_get_prgname(),
	                               name) + 20,
	                get_config_int("/apps/%s/Placement/%s_Y",
                                 g_get_prgname(),
                                 name) + 20);
	gtk_widget_set_usize(window,
                       get_config_int("/apps/%s/Placement/%s_Width",
                                      g_get_prgname(),
                                      name),
                       get_config_int("/apps/%s/Placement/%s_Height",
                                      g_get_prgname(),
                                      name));
}

void
gnome_db_save_window_config (const gchar *name, GtkWidget *window)
{
	gint x, y, width, height;
	
	g_return_if_fail(name != NULL);
	g_return_if_fail(GTK_IS_WIDGET(window));
	
	gdk_window_get_geometry(window->window, &x, &y, &width, &height, NULL);
	set_config_int(x, "/apps/%s/Placement/%s_X", g_get_prgname(), name);
	set_config_int(y, "/apps/%s/Placement/%s_Y", g_get_prgname(), name);
	set_config_int(width, "/apps/%s/Placement/%s_Width", g_get_prgname(), name);
	set_config_int(height, "/apps/%s/Placement/%s_Height", g_get_prgname(), name);
}

/*
 * File selection dialog
 */
static void
filesel_button_clicked (GtkWidget *w, glong *selected)
{
	*selected = (glong) gtk_object_get_data(GTK_OBJECT(w), "button");
}

gchar *
gnome_db_select_file (const gchar *title)
{
	GtkWidget* filesel;
	glong*     selected;
	gchar*     res = NULL;
	
	selected = (glong *) g_malloc(sizeof(glong));
	*selected = 0;
	
	/* create dialog */
	filesel = gtk_file_selection_new(title);
	gtk_object_set_data(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button),
	                    "button",
	                    (gpointer) 1);
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button),
	                   "clicked",
	                   GTK_SIGNAL_FUNC(filesel_button_clicked),
	                   (gpointer) selected);
	gtk_object_set_data(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button),
	                    "button",
	                    (gpointer) 2);
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button),
	                   "clicked",
	                   GTK_SIGNAL_FUNC(filesel_button_clicked),
	                   (gpointer) selected);
	gtk_widget_show(filesel);
	gtk_file_selection_show_fileop_buttons(GTK_FILE_SELECTION(filesel));
	
	/* wait for selection from user */
	while (!*selected)
		gtk_main_iteration_do(TRUE);
	if (*selected == 1) /* OK button */ {
		res = g_strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(filesel)));
	}
	else res = NULL;
	
	/* clean up */
	gtk_widget_destroy(filesel);
	g_free((gpointer) selected);
	return res;
}

void
gnome_db_submit_bug_cb (GtkWidget *w, gpointer data)
{
	int pid;
	gchar* args[] = {
		"bug-buddy",
		"--sm-disable",
		"--package=gnome-db",
		"--package-ver="VERSION,
		NULL
	};
	
	args[0] = gnome_is_program_in_path ("bug-buddy");
	if (!args[0]) {
		gnome_db_show_error (_("Bug-buddy was not found in your $PATH"));
		return;
	}
	pid = gnome_execute_async(NULL, 4, args);
	g_free (args[0]);
	if (pid == -1)
		gnome_db_show_error(_("Unable to execute Bug Report Tool"));
}

gboolean
gnome_db_yes_no_dialog (const gchar *msg, ...)
{
	va_list args;
	gchar buffer[1024];
	GtkWidget *dialog;
	GtkWidget *label;

	g_return_val_if_fail (msg != NULL, FALSE);

	/* build the message string */
	va_start (args, msg);
	vsprintf (buffer, msg, args);
	va_end (args);

	/* build the dialog */
	dialog = gnome_dialog_new (_("Question"),
				   GNOME_STOCK_BUTTON_YES,
				   GNOME_STOCK_BUTTON_NO,
				   NULL);
	gnome_dialog_set_default (GNOME_DIALOG (dialog), 0);
	label = gnome_db_new_label_widget (buffer);
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), label, 1, 1, 0);

	return gnome_dialog_run_and_close (GNOME_DIALOG (dialog)) ? FALSE : TRUE;
}
