/*  -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * 
 * This file is part of the GNOME Debugging Framework.
 * 
 * Copyright (C) 1999-2000 Dave Camp <campd@oit.edu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.  
 */

#include "control-factories.h"
#include "gdf-breakpoint-manager.h"
#include "gdf-source-viewer-manager.h"
#include "gdf-register-viewer.h"
#include "gdf-locals-viewer.h"
#include "../lib/gdf.h"

#include <bonobo.h>

enum {
    PROP_DEBUGGER_IOR = 1,
    PROP_SHOW_ENTRY
};

static void init_bpm_factory (void);
static BonoboObject *bpm_factory (BonoboGenericFactory *fact,
								 void *closure);
static void init_source_viewer_factory (void);
static BonoboObject *source_viewer_factory (BonoboGenericFactory *fact,
                                           void *closure);
static void init_stack_browser_factory (void);
static BonoboObject *sb_factory (BonoboGenericFactory *fact,
                                 void *closure);
static void init_register_viewer_factory (void);
static BonoboObject *rv_factory (BonoboGenericFactory *fact,
                                void *closure);
static void init_locals_viewer_factory (void);
static BonoboObject *lv_factory (BonoboGenericFactory *fact, void *closure);
static void init_variable_viewer_factory (void);
static BonoboObject *vv_factory (BonoboGenericFactory *fact, void *closure);

extern void control_server_quit ();

static void set_prop (BonoboPropertyBag *bag,
                      const BonoboArg *arg,
                      guint arg_id,
                      gpointer user_data);
static void get_prop (BonoboPropertyBag *bag,
                      BonoboArg *arg,
                      guint arg_id,
                      gpointer user_data);
static GdfDebuggerClient *get_debugger (gchar *ior);


void
init_control_factories (void)
{	
	gdf_server_manager_init (control_server_quit);
	init_bpm_factory ();
	init_source_viewer_factory ();
    init_stack_browser_factory ();
    init_register_viewer_factory ();
    init_locals_viewer_factory ();
    init_variable_viewer_factory ();
}

static void 
init_bpm_factory (void)
{
	static BonoboGenericFactory *bpm_control_factory = NULL;

	if (bpm_control_factory != NULL)
		return;

	bpm_control_factory = 
		bonobo_generic_factory_new ("OAFIID:control-factory:gdf-breakpoint-manager:8aaa6895-a3dc-4afd-b6fe-03abcfed2fc9",
                                    bpm_factory, NULL);
	
	if (bpm_control_factory == NULL) 
		g_error (_("Could not initialize Breakpoint Manager factory."));
	
}

static BonoboObject *
bpm_factory (BonoboGenericFactory *fact,
			 void *closure)
{
	BonoboPropertyBag *pb;
	BonoboControl *control;
	GdfBreakpointManager *bpm;

	/* Create the control */
	bpm = gdf_breakpoint_manager_new ();
	gtk_widget_show (GTK_WIDGET (bpm));
	gdf_server_manager_register_object (GTK_OBJECT (bpm));
	
	control = bonobo_control_new (GTK_WIDGET (bpm));
	
	/* Create the properties */
	pb = bonobo_property_bag_new (get_prop, set_prop, bpm);
	bonobo_control_set_property_bag (control, pb);
	bonobo_property_bag_add (pb, "debugger-ior", PROP_DEBUGGER_IOR,
                             BONOBO_ARG_STRING, NULL, 
                             "Object reference of the debugging backend", 0);

	return BONOBO_OBJECT (control);
}

static void 
init_source_viewer_factory (void)
{
	static BonoboGenericFactory *sv_control_factory = NULL;

	if (sv_control_factory != NULL)
		return;

	sv_control_factory = 
		bonobo_generic_factory_new ("OAFIID:control-factory:gdf-default-source-viewer:a8d7b557-e629-4365-9faf-9c30556bdeff",
								   source_viewer_factory, NULL);
	
	if (sv_control_factory == NULL) 
		g_error (_("Could not initialize Source Viewer factory."));
	
}

static BonoboObject *
source_viewer_factory (BonoboGenericFactory *fact,
					   void *closure)
{
	BonoboPropertyBag *pb;
	BonoboControl *control;
	GdfSourceViewerManager *svm;

	/* Create the control */
	svm = gdf_source_viewer_manager_new ();
	gtk_widget_show (GTK_WIDGET (svm));
	gdf_server_manager_register_object (GTK_OBJECT (svm));
	
	control = bonobo_control_new (GTK_WIDGET (svm));

	/* Create the properties */
    pb = bonobo_property_bag_new (get_prop, set_prop, svm);
	bonobo_control_set_property_bag (control, pb);
	bonobo_property_bag_add (pb, "debugger-ior", PROP_DEBUGGER_IOR,
                             BONOBO_ARG_STRING, NULL, 
                             "Object reference of the debugging backend", 0);

	
	return BONOBO_OBJECT (control);
}

static void 
init_stack_browser_factory (void)
{
	static BonoboGenericFactory *sb_control_factory = NULL;

	if (sb_control_factory != NULL)
		return;

    sb_control_factory = 
		bonobo_generic_factory_new ("OAFIID:control-factory:gdf-stack-browser:726853bd-d335-4b01-a2d4-b7fb52562525",
                                    sb_factory, NULL);
	
	if (sb_control_factory == NULL) 
		g_error (_("Could not initialize Stack Browser factory."));
	
}

static BonoboObject *
sb_factory (BonoboGenericFactory *fact,
            void *closure)
{
	BonoboPropertyBag *pb;
	BonoboControl *control;
	GtkWidget *sb;

	/* Create the control */
	sb = gdf_stack_browser_new ();
	gtk_widget_show (GTK_WIDGET (sb));
	gdf_server_manager_register_object (GTK_OBJECT (sb));
	
	control = bonobo_control_new (GTK_WIDGET (sb));
	
	/* Create the properties */
	pb = bonobo_property_bag_new (get_prop, set_prop, sb);
	bonobo_control_set_property_bag (control, pb);
	bonobo_property_bag_add (pb, "debugger-ior", PROP_DEBUGGER_IOR,
                             BONOBO_ARG_STRING, NULL, 
                             "Object reference of the debugging backend", 0);
	
	return BONOBO_OBJECT (control);
}

static void 
init_register_viewer_factory (void)
{
	static BonoboGenericFactory *rv_control_factory = NULL;

	if (rv_control_factory != NULL)
		return;

    rv_control_factory = 
		bonobo_generic_factory_new ("OAFIID:control-factory:gdf-register-viewer:12a0a3e5-8972-4ea1-9308-01789bc2962d",
                                    rv_factory, NULL);
	
	if (rv_control_factory == NULL) 
		g_error (_("Could not initialize Register Viewer factory."));
	
}

static BonoboObject *
rv_factory (BonoboGenericFactory *fact,
            void *closure)
{
	BonoboPropertyBag *pb;
	BonoboControl *control;
	GtkWidget *rv;

	/* Create the control */
	rv = gdf_register_viewer_new ();
	gtk_widget_show (GTK_WIDGET (rv));
	gdf_server_manager_register_object (GTK_OBJECT (rv));
	
	control = bonobo_control_new (GTK_WIDGET (rv));
	
	/* Create the properties */
	pb = bonobo_property_bag_new (get_prop, set_prop, rv);
	bonobo_control_set_property_bag (control, pb);
	bonobo_property_bag_add (pb, "debugger-ior", PROP_DEBUGGER_IOR,
                             BONOBO_ARG_STRING, NULL, 
                             "Object reference of the debugging backend", 0);
	
	return BONOBO_OBJECT (control);
}

static void 
init_locals_viewer_factory (void)
{
	static BonoboGenericFactory *lv_control_factory = NULL;

	if (lv_control_factory != NULL)
		return;

    lv_control_factory = 
		bonobo_generic_factory_new ("OAFIID:control-factory:gdf-locals-viewer:412ac1ce-4cf1-4b24-96c1-4efae3d5a1a3",
                                    lv_factory, NULL);
	
	if (lv_control_factory == NULL) 
		g_error (_("Could not initialize locals viewer factory."));
	
}

static BonoboObject *
lv_factory (BonoboGenericFactory *fact,
            void *closure)
{
	BonoboPropertyBag *pb;
	BonoboControl *control;
	GtkWidget *lv;

	/* Create the control */
	lv = GTK_WIDGET (gdf_locals_viewer_new ());
	gtk_widget_show (GTK_WIDGET (lv));
	gdf_server_manager_register_object (GTK_OBJECT (lv));
	
	control = bonobo_control_new (GTK_WIDGET (lv));
	
	/* Create the properties */
	pb = bonobo_property_bag_new (get_prop, set_prop, lv);
	bonobo_control_set_property_bag (control, pb);
	bonobo_property_bag_add (pb, "debugger-ior", PROP_DEBUGGER_IOR,
                             BONOBO_ARG_STRING, NULL, 
                             "Object reference of the debugging backend", 0);
	
	return BONOBO_OBJECT (control);
}

static void 
init_variable_viewer_factory (void)
{
	static BonoboGenericFactory *vv_control_factory = NULL;

	if (vv_control_factory != NULL)
		return;

    vv_control_factory = 
		bonobo_generic_factory_new ("OAFIID:control-factory:gdf-variable-viewer:cf41eceb-b5f5-447a-a0e6-77c48103a1ae",
                                    vv_factory, NULL);
	
	if (vv_control_factory == NULL) 
		g_error (_("Could not initialize variable viewer factory."));
	
}

static BonoboObject *
vv_factory (BonoboGenericFactory *fact,
            void *closure)
{
	BonoboPropertyBag *pb;
	BonoboControl *control;
	GtkWidget *vv;
    BonoboArg *def;

	/* Create the control */
	vv = GTK_WIDGET (gdf_variable_viewer_new ());
	gtk_widget_show (GTK_WIDGET (vv));
	gdf_server_manager_register_object (GTK_OBJECT (vv));
	
	control = bonobo_control_new (GTK_WIDGET (vv));
	
	/* Create the properties */
	pb = bonobo_property_bag_new (get_prop, set_prop, vv);
	bonobo_control_set_property_bag (control, pb);
	bonobo_property_bag_add (pb, "debugger-ior", PROP_DEBUGGER_IOR,
                             BONOBO_ARG_STRING, NULL, 
                             "Object reference of the debugging backend", 
                             BONOBO_PROPERTY_READABLE 
                             | BONOBO_PROPERTY_WRITEABLE);
    

    def = bonobo_arg_new (BONOBO_ARG_BOOLEAN);
    BONOBO_ARG_SET_BOOLEAN (def, TRUE);
	bonobo_property_bag_add (pb, "show_entry", PROP_SHOW_ENTRY,
                             BONOBO_ARG_BOOLEAN, def,
                             "Show the \"Add symbol\" entry", 
                             BONOBO_PROPERTY_READABLE 
                             | BONOBO_PROPERTY_WRITEABLE);
    bonobo_arg_release (def);

	return BONOBO_OBJECT (control);
}

static void
set_prop (BonoboPropertyBag *bag,
          const BonoboArg *arg,
          guint arg_id,
          gpointer user_data)
{
    GtkObject *obj = user_data;
    
    GDF_TRACE ();

    switch (arg_id) {
    case PROP_DEBUGGER_IOR :
    {
        GdfDebuggerClient *dbg;
        char *s;

        s = BONOBO_ARG_GET_STRING (arg);
        
        if (strcmp (s, ""))
            dbg = get_debugger (s);
        else
            dbg = NULL;
        
        gtk_object_set (GTK_OBJECT (obj), "debugger", 
                        (GtkObject *)dbg, NULL);
        break;
    }
    case PROP_SHOW_ENTRY :
    {
        gtk_object_set (GTK_OBJECT (obj), "show_entry",
                        BONOBO_ARG_GET_BOOLEAN (arg), NULL);
        break;
    }
    
    }
}

static void
get_prop (BonoboPropertyBag *bag,
          BonoboArg *arg,
          guint arg_id,
          gpointer user_data)
{
    GtkObject *obj = user_data;
    
    switch (arg_id) {
    case PROP_DEBUGGER_IOR :
    {
        GdfDebuggerClient *dbg;
        char *ior;
        GtkArg args[1];
        args[0].name = "debugger";
        gtk_object_getv (obj, 1, args);
        g_assert (args[0].type == GTK_TYPE_OBJECT);
        if (GTK_VALUE_OBJECT (args[0]))
            dbg = GDF_DEBUGGER_CLIENT (GTK_VALUE_OBJECT (args[0]));
        else 
            dbg = NULL;
        
        if (dbg) {
            CORBA_Environment ev;
            CORBA_exception_init (&ev);
            ior = CORBA_ORB_object_to_string (oaf_orb_get (), 
                                              dbg->objref, &ev);
            CORBA_exception_free (&ev);
        } else {
            ior = CORBA_string_dup ("");
        }
        
        BONOBO_ARG_SET_STRING (arg, ior);
        CORBA_free (ior);
        break;
    }
    case PROP_SHOW_ENTRY :
    {
        GtkArg args[1];
        args[0].name = "show_entry";
        gtk_object_getv (obj, 1, args);
        g_assert (args[0].type == GTK_TYPE_BOOL);
        BONOBO_ARG_SET_BOOLEAN (arg, GTK_VALUE_BOOL (args[0]));
        break;
    }
    }
}

static GdfDebuggerClient *
get_debugger (gchar *ior)
{
	CORBA_Object objref;
	GdfDebuggerClient *dbg;
	CORBA_Environment ev;

    static GHashTable *active_debuggers = NULL;
	
    if (ior[0] == '\0')
        return NULL;

	if (!active_debuggers) 
		active_debuggers = g_hash_table_new (g_str_hash, g_str_equal);

	dbg = (GdfDebuggerClient *)g_hash_table_lookup (active_debuggers, ior);
	
	if (!dbg) {
		CORBA_exception_init (&ev);
		objref = CORBA_ORB_string_to_object (oaf_orb_get (), ior, &ev);
		/* FIXME: Check for exceptions */
		CORBA_exception_free (&ev);
	
		dbg = gdf_debugger_client_new_from_corba (objref);
		g_hash_table_insert (active_debuggers, (gpointer)ior, (gpointer)dbg);
	}
	
	return dbg;
}
