/*  -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * 
 * This file is part of the GNOME Debugging Framework.
 * 
 * Copyright (C) 1999-2000 Dave Camp <campd@oit.edu>
 *                         Martin Baulig <martin@home-of-linux.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.  
 */

#include <config.h>
#include "gdf-debugger-client.h"

#include <liboaf/liboaf.h>
#include "gdf-event-marshallers.h"

static void debugger_client_class_init (GdfDebuggerClientClass *class);
static void debugger_client_init (GdfDebuggerClient *prog);
static void debugger_client_destroy (GdfDebuggerClient *prog);

static void event_pushed_cb (GdfEventChannelClient *event_channel,
                             CORBA_any *val,
                             GdfDebuggerClient *client);
static void program_loaded_impl (GdfDebuggerClient *dbg);
static void program_unloaded_impl (GdfDebuggerClient *dbg);


static GtkObjectClass *parent_class;

struct _GdfDebuggerClientPrivate {
    GdfEventChannelClient *event_channel;

    int cur_frame;
};

enum {
    UNKNOWN_EVENT_PUSHED,
    PROGRAM_LOADED,
    PROGRAM_UNLOADED,
    BREAKPOINT_SET,
    BREAKPOINT_ENABLED,
    BREAKPOINT_DISABLED,
    BREAKPOINT_DELETED,
    EXECUTION_STARTED,
    EXECUTION_RUNNING,
    EXECUTION_STOPPED,
    EXECUTION_KILLED,
    EXECUTION_EXITED,
    EXECUTION_SOURCE_LINE,
    STACK_FRAME_CHANGED,
    SIGNAL_RECEIVED,
    SIGNAL_TERMINATION,
    LAST_SIGNAL
};

static gint debugger_client_signals[LAST_SIGNAL];

static void 
gdf_debugger_client_construct (GdfDebuggerClient *dbg, 
                                CORBA_Object corba_object)
{
    CORBA_Object channel;
    CORBA_Environment ev;

    dbg->program_loaded = FALSE;
    dbg->objref = corba_object;
    dbg->priv = g_new0 (GdfDebuggerClientPrivate, 1);
    dbg->priv->cur_frame = -1;
    
    CORBA_exception_init (&ev);

    channel = GDF_Debugger_get_event_channel (corba_object,
                                              &ev);
    
    /* FIXME: Check env */

    CORBA_exception_free (&ev);

    dbg->priv->event_channel = 
        gdf_event_channel_client_new_from_corba (channel);

    gdf_event_channel_client_listen (dbg->priv->event_channel);

    gtk_signal_connect (GTK_OBJECT (dbg->priv->event_channel),
                        "event_pushed", 
                        GTK_SIGNAL_FUNC (event_pushed_cb),
                        (gpointer)dbg);  
}


/**
 * gdf_debugger_client_new:
 * @iid: Backend to load.
 *
 * Starts a new GDF::Debugger server and returns a GdfDebuggerClient
 * with a reference to that object.
 *
 * Returns: the initialized GdfDebuggerClient object.
 */
GdfDebuggerClient *
gdf_debugger_client_new (const char *iid)
{
    CORBA_Object corba_object;
    GdfDebuggerClient *ret = NULL;
    CORBA_Environment ev;
    
#if 0
    corba_object = 
        goad_server_activate_with_id (NULL,
                                      iid,
                                      GOAD_ACTIVATE_NEW_ONLY, NULL);
#endif
    CORBA_exception_init (&ev);

    corba_object = oaf_activate_from_id (iid, OAF_FLAG_IGNORE_EXISTING, 
                                         NULL, &ev);    
    
    if (CORBA_Object_is_nil (corba_object, &ev)) {
        CORBA_exception_free (&ev);
        return NULL;
    }
    CORBA_exception_free (&ev);
    
    ret = gtk_type_new (gdf_debugger_client_get_type ());
 
    gdf_debugger_client_construct (ret, corba_object);
    
    return ret;
}


/**
 * gdf_debugger_client_new_from_objref:
 * @program: GDF::Program CORBA object reference.
 *
 * Creates a GdfDebuggerClient GTK Object which wraps around the
 * GDF::Program CORBA object.
 *
 * Returns: the initialized GdfDebuggerClient object.
 */
GdfDebuggerClient *
gdf_debugger_client_new_from_corba (GDF_Debugger debugger) 
{
    GdfDebuggerClient *ret;
    CORBA_Environment ev;

    g_return_val_if_fail (debugger != CORBA_OBJECT_NIL, NULL);

    ret = gtk_type_new (gdf_debugger_client_get_type ());
    gdf_debugger_client_construct (ret, debugger);
   
    CORBA_exception_init (&ev);

    Bonobo_Unknown_ref (debugger, &ev);

    ret->program_loaded = GDF_Debugger__get_binary_loaded (debugger, &ev);
    
    /* FIXME: Check env */
    
    CORBA_exception_free (&ev);

    return ret;
}

GtkType
gdf_debugger_client_get_type (void)
{
    static GtkType type = 0;
    
    if (!type) {
        GtkTypeInfo info = {
            "Handle to a GDF::Debugger object",
            sizeof (GdfDebuggerClient),
            sizeof (GdfDebuggerClientClass),
            (GtkClassInitFunc) debugger_client_class_init,
            (GtkObjectInitFunc) debugger_client_init,
            NULL,
            NULL,
            (GtkClassInitFunc) NULL
        };
	
        type = gtk_type_unique (gtk_object_get_type (), &info);
    }
    
    return type;
}

GDF_Debugger_State
gdf_debugger_client_get_state (GdfDebuggerClient *dbg)
{
    GDF_Debugger_State ret;
    CORBA_Environment ev;

    g_return_val_if_fail (dbg != NULL, GDF_Debugger_UNKNOWN);
    g_return_val_if_fail (GDF_IS_DEBUGGER_CLIENT (dbg),
                          GDF_Debugger_UNKNOWN);

    CORBA_exception_init (&ev);
    
    ret = GDF_Debugger_get_state (dbg->objref, &ev);
    
    /* FIXME: Check env */

    CORBA_exception_free (&ev);   
    
    return ret;
}

gboolean
gdf_debugger_client_load_binary (GdfDebuggerClient *dbg, 
                                  gchar *file_name)
{
    CORBA_Environment ev;

    g_return_val_if_fail (dbg != NULL, FALSE);
    g_return_val_if_fail (file_name != NULL, FALSE);
    g_return_val_if_fail (GDF_IS_DEBUGGER_CLIENT (dbg), FALSE);

    CORBA_exception_init (&ev);
    
    GDF_Debugger_load_binary (dbg->objref,
                               file_name,
                               &ev);

    /* FIXME: Check env for system error */
    
    if (ev._major == CORBA_USER_EXCEPTION) {
        if (strcmp (ev._repo_id, ex_GDF_Debugger_InvalidBinary) == 0) {
            CORBA_exception_free (&ev);   
            return FALSE;
        } else {
            g_assert_not_reached ();
        }
    }

    CORBA_exception_free (&ev);   

    return TRUE;
}

void 
gdf_debugger_client_unload_binary (GdfDebuggerClient *dbg)
{
    CORBA_Environment ev;
    
    g_return_if_fail (dbg != NULL);
    g_return_if_fail (GDF_IS_DEBUGGER_CLIENT (dbg));
    
    CORBA_exception_init (&ev);
    
    GDF_Debugger_unload_binary (dbg->objref,
                                 &ev);

    /* FIXME: Check env */
    
    CORBA_exception_free (&ev);
}

GDF_source_file_list*
gdf_debugger_client_get_sources (GdfDebuggerClient *dbg)
{
    GDF_source_file_list *retval;
    CORBA_Environment ev;

    g_return_val_if_fail (dbg != NULL, NULL);
    g_return_val_if_fail (GDF_IS_DEBUGGER_CLIENT (dbg), NULL);

    CORBA_exception_init (&ev);

    retval = GDF_Debugger_get_sources (dbg->objref,
                                        &ev);

    /* FIXME: Check env */
    
    CORBA_exception_free (&ev);
    
    return retval;
}

gchar *
gdf_debugger_client_get_absolute_source_path (GdfDebuggerClient *dbg,
                                               const gchar *file_name)
{
    gchar *retval;
    CORBA_char *corba_ret;
    CORBA_Environment ev;

    g_return_val_if_fail (dbg != NULL, NULL);
    g_return_val_if_fail (GDF_IS_DEBUGGER_CLIENT (dbg), NULL);

    CORBA_exception_init (&ev);

    corba_ret = GDF_Debugger_get_absolute_source_path (dbg->objref,
                                                        file_name,
                                                        &ev);

    /* FIXME: Check env */

    retval = g_strdup (corba_ret);   /* let clients use g_free rather than */
    CORBA_free (corba_ret);          /* CORBA_free */
    
    CORBA_exception_free (&ev);
    
    return retval;
}

void
gdf_debugger_client_execute (GdfDebuggerClient *dbg, 
                              int argc,
                              char *argv[])
{
    GDF_arg_list *arg_list;
    CORBA_Environment ev;
    
    g_return_if_fail (dbg != NULL);
    g_return_if_fail (argv != NULL);
    g_return_if_fail (GDF_IS_DEBUGGER_CLIENT (dbg));

    CORBA_exception_init (&ev);
    
    /* FIXME: This has not been tested */
    arg_list = GDF_arg_list__alloc();
    arg_list->_length = argc;
    arg_list->_buffer = (CORBA_char**)argv;
    
    GDF_Debugger_execute (dbg->objref,
                           arg_list,
                           &ev);

    /* FIXME: Check env */

    CORBA_exception_free (&ev);
}

void
gdf_debugger_client_attach (GdfDebuggerClient *dbg, int pid)
{
    CORBA_Environment ev;
    
    g_return_if_fail (dbg != NULL);
    g_return_if_fail (GDF_IS_DEBUGGER_CLIENT (dbg));
    
    CORBA_exception_init (&ev);
    
    GDF_Debugger_attach (dbg->objref,
                          pid,
                          &ev);

    /* FIXME: Check env */

    CORBA_exception_free (&ev);
}

void
gdf_debugger_client_load_corefile (GdfDebuggerClient *dbg,
                                    const char *corefile_name)
{
    CORBA_Environment ev;
    
    g_return_if_fail (dbg != NULL);
    g_return_if_fail (corefile_name != NULL);
    g_return_if_fail (GDF_IS_DEBUGGER_CLIENT (dbg));
    
    CORBA_exception_init (&ev);
    
    GDF_Debugger_load_corefile (dbg->objref,
                                 (CORBA_char*)corefile_name,
                                 &ev);
    
    /* FIXME: Check env */
    
    CORBA_exception_free (&ev);
}

long
gdf_debugger_client_set_breakpoint (GdfDebuggerClient *dbg,
                                     const char *file_name,
                                     long line_num,
                                     const char *condition)
{
    CORBA_Environment ev;
    long bp_num;
    
    g_return_val_if_fail (dbg != NULL, -1);
    g_return_val_if_fail (file_name != NULL, -1);
    g_return_val_if_fail (GDF_IS_DEBUGGER_CLIENT (dbg), -1);
    
    CORBA_exception_init (&ev);
    
    bp_num = 
        GDF_Debugger_set_breakpoint (dbg->objref,
                                      (CORBA_char*)(file_name ? file_name :""),
                                      line_num,
                                      (CORBA_char*)(condition ? condition :""),
                                      &ev);

    /* FIXME: Check env */

    CORBA_exception_free (&ev);
    
    return bp_num;
}

long
gdf_debugger_client_set_breakpoint_function (GdfDebuggerClient *dbg,
                                              const char *file_name,
                                              const char *function_name,
                                              const char *condition)
{
    CORBA_Environment ev;
    long bp_num;
    
    g_return_val_if_fail (dbg != NULL, -1);
    g_return_val_if_fail (GDF_IS_DEBUGGER_CLIENT (dbg), -1);
    g_return_val_if_fail (function_name != NULL, -1);
    
    CORBA_exception_init (&ev);
    
    bp_num = 
        GDF_Debugger_set_breakpoint_function 
        (dbg->objref,
         (CORBA_char*)(file_name ? file_name : ""),
         (CORBA_char*)(function_name ? function_name : ""),
         (CORBA_char*)(condition ? condition : ""),
         &ev);
    
    /* FIXME: Check env */
    
    CORBA_exception_free (&ev);
    
    return bp_num;
}

void 
gdf_debugger_client_enable_breakpoint (GdfDebuggerClient *dbg,
                                        long bp_num)
{
    CORBA_Environment ev;
    
    g_return_if_fail (dbg != NULL);
    g_return_if_fail (GDF_IS_DEBUGGER_CLIENT (dbg));

    CORBA_exception_init (&ev);
    
    GDF_Debugger_enable_breakpoint (dbg->objref,
                                     bp_num,
                                     &ev);

    /* FIXMR: Check env */
    
    CORBA_exception_free (&ev);
}

void 
gdf_debugger_client_disable_breakpoint (GdfDebuggerClient *dbg,
                                         long bp_num)
{
    CORBA_Environment ev;
    
    g_return_if_fail (dbg != NULL);
    g_return_if_fail (GDF_IS_DEBUGGER_CLIENT (dbg));
    
    CORBA_exception_init (&ev);
    
    GDF_Debugger_disable_breakpoint (dbg->objref,
                                      bp_num,
                                      &ev);
    /* FIXME: Check env */
    
    CORBA_exception_free (&ev);
}

void
gdf_debugger_client_delete_breakpoint (GdfDebuggerClient *dbg,
                                        long bp_num)
{
    CORBA_Environment ev;
    
    g_return_if_fail (dbg != NULL);
    g_return_if_fail (GDF_IS_DEBUGGER_CLIENT (dbg));
    
    CORBA_exception_init (&ev);
    GDF_Debugger_delete_breakpoint (dbg->objref,
                                     bp_num,
                                     &ev);

    /* FIXME: Check env */
    
    CORBA_exception_free (&ev);
}

GDF_Breakpoint*
gdf_debugger_client_get_breakpoint_info (GdfDebuggerClient *dbg,
                                          long bp_num)
{
    GDF_Breakpoint *retval;
    CORBA_Environment ev;
    
    g_return_val_if_fail (dbg != NULL, NULL);
    g_return_val_if_fail (GDF_IS_DEBUGGER_CLIENT (dbg), NULL);
    
    CORBA_exception_init (&ev);
    retval = GDF_Debugger_get_breakpoint_info (dbg->objref,
                                                (CORBA_long)bp_num,
                                                &ev);
 
    /* FIXME: Check env */
    
    if (ev._major != CORBA_NO_EXCEPTION) {
        CORBA_exception_free (&ev);
        retval = NULL;
    }

    CORBA_exception_free (&ev);
    return retval;
}

GDF_RegisterList *
gdf_debugger_client_get_registers (GdfDebuggerClient *dbg)
{
    GDF_RegisterList *retval;
    CORBA_Environment ev;
    
    g_return_val_if_fail (dbg != NULL, NULL);
    g_return_val_if_fail (GDF_IS_DEBUGGER_CLIENT (dbg), NULL);

    CORBA_exception_init (&ev);
    retval = GDF_Debugger_get_registers (dbg->objref,
                                          &ev);
    
    /* FIXME: Check env */
    
    if (ev._major != CORBA_NO_EXCEPTION) {
        CORBA_exception_free (&ev);
        retval = NULL;
    }

    CORBA_exception_free (&ev);
    return retval;    
    
}

long
gdf_debugger_client_current_frame (GdfDebuggerClient *dbg)
{
    return dbg->priv->cur_frame;
}
 
void 
gdf_debugger_client_up_frame (GdfDebuggerClient *dbg)
{
    gdf_debugger_client_change_frame (dbg, dbg->priv->cur_frame + 1);
}

void 
gdf_debugger_client_down_frame (GdfDebuggerClient *dbg)
{
    gdf_debugger_client_change_frame (dbg, dbg->priv->cur_frame - 1);
}

void 
gdf_debugger_client_change_frame (GdfDebuggerClient *dbg, 
                                   long new_id)
{
    CORBA_Environment ev;
    
    g_return_if_fail (dbg != NULL);
    g_return_if_fail (GDF_IS_DEBUGGER_CLIENT (dbg));

    CORBA_exception_init (&ev);
    GDF_Debugger_change_frame (dbg->objref, new_id, &ev);
    
    /* FIXME: Check env */

    CORBA_exception_free (&ev);
    return;
}

GDF_StackFrame *
gdf_debugger_client_get_frame (GdfDebuggerClient *dbg,
                                long id)
{
    GDF_StackFrame *retval;
    CORBA_Environment ev;
    
    g_return_val_if_fail (dbg != NULL, NULL);
    g_return_val_if_fail (GDF_IS_DEBUGGER_CLIENT (dbg), NULL);

    CORBA_exception_init (&ev);
    retval = GDF_Debugger_get_frame (dbg->objref,
                                      id,
                                      &ev);

    /* FIXME: Check env */
    
    if (ev._major != CORBA_NO_EXCEPTION) {
        CORBA_exception_free (&ev);
        CORBA_free (retval);
        return NULL;
    }

    CORBA_exception_free (&ev);
    return retval;
}

GDF_Stack *
gdf_debugger_client_get_backtrace (GdfDebuggerClient *dbg)
{
    GDF_Stack *retval;
    CORBA_Environment ev;
    
    g_return_val_if_fail (dbg != NULL, NULL);
    g_return_val_if_fail (GDF_IS_DEBUGGER_CLIENT (dbg), NULL);
    
    CORBA_exception_init (&ev);
    
    retval = GDF_Debugger_get_backtrace (dbg->objref, &ev);

    /* FIXME: Check env */
    
    if (ev._major != CORBA_NO_EXCEPTION) {
        CORBA_exception_free (&ev);
        retval = NULL;
    }
    
    CORBA_exception_free (&ev);
    
    return retval;
}

void 
gdf_debugger_client_cont (GdfDebuggerClient *dbg)
{
    CORBA_Environment ev;
    
    g_return_if_fail (dbg != NULL);
    g_return_if_fail (GDF_IS_DEBUGGER_CLIENT (dbg));
    
    CORBA_exception_init (&ev);
    GDF_Debugger_cont (dbg->objref,
                       &ev);

    /* FIXME: Check env */

    CORBA_exception_free (&ev);    
}

void 
gdf_debugger_client_stop (GdfDebuggerClient *dbg)
{
    CORBA_Environment ev;
    
    g_return_if_fail (dbg != NULL);
    g_return_if_fail (GDF_IS_DEBUGGER_CLIENT (dbg));
    
    CORBA_exception_init (&ev);
    GDF_Debugger_stop (dbg->objref,
                        &ev);
    /* FIXME: Check env */
    
    CORBA_exception_free (&ev);    
}

void 
gdf_debugger_client_restart (GdfDebuggerClient *dbg)
{
    CORBA_Environment ev;
    
    g_return_if_fail (dbg != NULL);
    g_return_if_fail (GDF_IS_DEBUGGER_CLIENT (dbg));
    
    CORBA_exception_init (&ev);
    GDF_Debugger_restart (dbg->objref,
                           &ev);

    /* FIXME: Check env */
    
    CORBA_exception_free (&ev);    
}

void 
gdf_debugger_client_step_over (GdfDebuggerClient *dbg)
{
    CORBA_Environment ev;
    
    g_return_if_fail (dbg != NULL);
    g_return_if_fail (GDF_IS_DEBUGGER_CLIENT (dbg));
    
    CORBA_exception_init (&ev);
    GDF_Debugger_step_over (dbg->objref,
                             &ev);
 
    /* FIXME: Check env */
    
    CORBA_exception_free (&ev);    
}

void 
gdf_debugger_client_step_into (GdfDebuggerClient *dbg)
{
    CORBA_Environment ev;
    
    g_return_if_fail (dbg != NULL);
    g_return_if_fail (GDF_IS_DEBUGGER_CLIENT (dbg));
    
    CORBA_exception_init (&ev);
    GDF_Debugger_step_into (dbg->objref,
                             &ev);

    /* FIXME: Check env */
    
    CORBA_exception_free (&ev);    
}

void 
gdf_debugger_client_step_out (GdfDebuggerClient *dbg)
{
    CORBA_Environment ev;
    
    g_return_if_fail (dbg != NULL);
    g_return_if_fail (GDF_IS_DEBUGGER_CLIENT (dbg));
   
    CORBA_exception_init (&ev);
    GDF_Debugger_step_out (dbg->objref,
                            &ev);

    /* FIXME: Check env */
    
    CORBA_exception_free (&ev);
}

GdfSymbolSetClient *
gdf_debugger_client_get_locals (GdfDebuggerClient *dbg)
{
    CORBA_Environment ev;
    GdfSymbolSetClient *retval;
    GDF_SymbolSet set;
    
    g_return_val_if_fail (dbg != NULL, NULL);
    g_return_val_if_fail (GDF_IS_DEBUGGER_CLIENT (dbg), NULL);
   
    CORBA_exception_init (&ev);
    set = GDF_Debugger_get_locals (dbg->objref, &ev);

    /* FIXME: Check env */
    
    retval = gdf_symbol_set_client_new_from_corba (set);

    CORBA_exception_free (&ev);

    return retval;
}   

GdfSymbolSetClient *
gdf_debugger_client_allocate_symbol_set (GdfDebuggerClient *dbg)
{    
    CORBA_Environment ev;
    GdfSymbolSetClient *retval;
    GDF_SymbolSet set;
    
    g_return_val_if_fail (dbg != NULL, NULL);
    g_return_val_if_fail (GDF_IS_DEBUGGER_CLIENT (dbg), NULL);
   
    CORBA_exception_init (&ev);
    set = GDF_Debugger_allocate_symbol_set (dbg->objref, &ev);

    /* FIXME: Check env */

    retval = gdf_symbol_set_client_new_from_corba (set);
    
    CORBA_exception_free (&ev);

    return retval;
}

void 
gdf_debugger_client_set_output_tty (GdfDebuggerClient *dbg,
                                     const char *tty_name)
{
    CORBA_Environment ev;
    
    g_return_if_fail (dbg != NULL);
    g_return_if_fail (tty_name != NULL);
    g_return_if_fail (GDF_IS_DEBUGGER_CLIENT (dbg));
    
    CORBA_exception_init (&ev);

    GDF_Debugger_set_output_tty (dbg->objref, (CORBA_char*)tty_name, &ev);
    
    /* FIXME: Check env */
    CORBA_exception_free (&ev);
}

/* private routines */
static void
debugger_client_class_init (GdfDebuggerClientClass *klass)
{
    GtkObjectClass *object_class = (GtkObjectClass *)klass;
    
    g_return_if_fail (klass != NULL);
    g_return_if_fail (GDF_IS_DEBUGGER_CLIENT_CLASS (klass));
    
    parent_class = gtk_type_class (gtk_object_get_type ());
    
    debugger_client_signals [UNKNOWN_EVENT_PUSHED] = 
        gtk_signal_new ("unknown_event_pushed",
                        GTK_RUN_FIRST,
                        object_class->type,
                        GTK_SIGNAL_OFFSET (GdfDebuggerClientClass,
                                           unknown_event_pushed),
                        gtk_marshal_NONE__POINTER_POINTER,
                        GTK_TYPE_NONE, 2,
                        GTK_TYPE_STRING, GTK_TYPE_POINTER);
    debugger_client_signals [PROGRAM_LOADED] = 
        gtk_signal_new ("program_loaded",
                        GTK_RUN_FIRST,
                        object_class->type,
                        GTK_SIGNAL_OFFSET (GdfDebuggerClientClass,
                                           program_loaded),
                        gtk_marshal_NONE__NONE,
                        GTK_TYPE_NONE, 0);
    debugger_client_signals [PROGRAM_UNLOADED] = 
        gtk_signal_new ("program_unloaded",
                        GTK_RUN_FIRST,
                        object_class->type,
                        GTK_SIGNAL_OFFSET (GdfDebuggerClientClass,
                                           program_unloaded),
                        gtk_marshal_NONE__NONE,
                        GTK_TYPE_NONE, 0);
    debugger_client_signals [BREAKPOINT_SET] = 
        gtk_signal_new ("breakpoint_set",
                        GTK_RUN_FIRST,
                        object_class->type,
                        GTK_SIGNAL_OFFSET (GdfDebuggerClientClass,
                                           breakpoint_set),
                        gtk_marshal_NONE__INT,
                        GTK_TYPE_NONE, 1,
                        GTK_TYPE_INT);
    debugger_client_signals [BREAKPOINT_ENABLED] = 
        gtk_signal_new ("breakpoint_enabled",
                        GTK_RUN_FIRST,
                        object_class->type,
                        GTK_SIGNAL_OFFSET (GdfDebuggerClientClass,
                                           breakpoint_enabled),
                        gtk_marshal_NONE__INT,
                        GTK_TYPE_NONE, 1,
                        GTK_TYPE_INT);
    debugger_client_signals [BREAKPOINT_DISABLED] = 
        gtk_signal_new ("breakpoint_disabled",
                        GTK_RUN_FIRST,
                        object_class->type,
                        GTK_SIGNAL_OFFSET (GdfDebuggerClientClass,
                                           breakpoint_disabled),
                        gtk_marshal_NONE__INT,
                        GTK_TYPE_NONE, 1,
                        GTK_TYPE_INT);
    debugger_client_signals [BREAKPOINT_DELETED] = 
        gtk_signal_new ("breakpoint_deleted",
                        GTK_RUN_FIRST,
                        object_class->type,
                        GTK_SIGNAL_OFFSET (GdfDebuggerClientClass,
                                           breakpoint_deleted),
                        gtk_marshal_NONE__INT,
                        GTK_TYPE_NONE, 1,
                        GTK_TYPE_INT);
    debugger_client_signals [EXECUTION_STARTED] = 
        gtk_signal_new ("execution_started",
                        GTK_RUN_FIRST,
                        object_class->type,
                        GTK_SIGNAL_OFFSET (GdfDebuggerClientClass,
                                           execution_started),
                        gtk_marshal_NONE__NONE,
                        GTK_TYPE_NONE, 0);
    debugger_client_signals [EXECUTION_RUNNING] = 
        gtk_signal_new ("execution_running",
                        GTK_RUN_FIRST,
                        object_class->type,
                        GTK_SIGNAL_OFFSET (GdfDebuggerClientClass,
                                           execution_running),
                        gtk_marshal_NONE__NONE,
                        GTK_TYPE_NONE, 0);
    debugger_client_signals [EXECUTION_STOPPED] = 
        gtk_signal_new ("execution_stopped",
                        GTK_RUN_FIRST,
                        object_class->type,
                        GTK_SIGNAL_OFFSET (GdfDebuggerClientClass,
                                           execution_stopped),
                        gtk_marshal_NONE__NONE,
                        GTK_TYPE_NONE, 0);
    debugger_client_signals [EXECUTION_KILLED] = 
        gtk_signal_new ("execution_killed",
                        GTK_RUN_FIRST,
                        object_class->type,
                        GTK_SIGNAL_OFFSET (GdfDebuggerClientClass,
                                           execution_killed),
                        gtk_marshal_NONE__INT,
                        GTK_TYPE_NONE, 1,
                        GTK_TYPE_INT);
    debugger_client_signals [EXECUTION_EXITED] = 
        gtk_signal_new ("execution_exited",
                        GTK_RUN_FIRST,
                        object_class->type,
                        GTK_SIGNAL_OFFSET (GdfDebuggerClientClass,
                                           execution_exited),
                        gtk_marshal_NONE__INT,
                        GTK_TYPE_NONE, 1,
                        GTK_TYPE_INT);
    debugger_client_signals [EXECUTION_SOURCE_LINE] = 
        gtk_signal_new ("execution_source_line",
                        GTK_RUN_FIRST,
                        object_class->type,
                        GTK_SIGNAL_OFFSET (GdfDebuggerClientClass,
                                           execution_source_line),
                        gtk_marshal_NONE__POINTER_INT,
                        GTK_TYPE_NONE, 2,
                        GTK_TYPE_STRING,
                        GTK_TYPE_INT);
    debugger_client_signals [STACK_FRAME_CHANGED] = 
        gtk_signal_new ("stack_frame_changed",
                        GTK_RUN_FIRST,
                        object_class->type,
                        GTK_SIGNAL_OFFSET (GdfDebuggerClientClass,
                                           stack_frame_changed),
                        gtk_marshal_NONE__INT,
                        GTK_TYPE_NONE, 1,
                        GTK_TYPE_INT);
    debugger_client_signals [SIGNAL_RECEIVED] = 
        gtk_signal_new ("signal_received",
                        GTK_RUN_FIRST,
                        object_class->type,
                        GTK_SIGNAL_OFFSET (GdfDebuggerClientClass,
                                           signal_received),
                        gtk_marshal_NONE__STRING,
                        GTK_TYPE_NONE, 1,
                        GTK_TYPE_STRING);
    debugger_client_signals [SIGNAL_TERMINATION] = 
        gtk_signal_new ("signal_termination",
                        GTK_RUN_FIRST,
                        object_class->type,
                        GTK_SIGNAL_OFFSET (GdfDebuggerClientClass,
                                           signal_termination),
                        gtk_marshal_NONE__STRING,
                        GTK_TYPE_NONE, 1,
                        GTK_TYPE_STRING);

    gtk_object_class_add_signals (object_class,
                                  debugger_client_signals,
                                  LAST_SIGNAL);

    klass->program_loaded = program_loaded_impl;
    klass->program_unloaded = program_unloaded_impl;
    klass->breakpoint_set = NULL;
    object_class->destroy = (GtkSignalFunc) debugger_client_destroy;
}

static void
debugger_client_init (GdfDebuggerClient *client)
{
    client->priv = NULL;
}

static void
debugger_client_destroy (GdfDebuggerClient *client)
{
    CORBA_Environment ev;

    if (client->priv->event_channel) {
        bonobo_object_unref (BONOBO_OBJECT (client->priv->event_channel));
        client->priv->event_channel = NULL;
    }

    if (client->priv) {
        g_free (client->priv);
        client->priv = NULL;
    }

    CORBA_exception_init (&ev);
    
    Bonobo_Unknown_unref (client->objref, &ev);
    
    /* FIXME: Check env */
    
    CORBA_exception_free (&ev);

    if (parent_class->destroy)
        parent_class->destroy (GTK_OBJECT (client));
    
}

static void
emit_event_signal (GdfDebuggerClient *dbg,
                   GdfEvent *event)
{
    /* FIXME: Some table logic would be much better here */

    if (strcmp (event->event_name, "program_loaded") == 0) {
        gtk_signal_emit (GTK_OBJECT (dbg),
                         debugger_client_signals[PROGRAM_LOADED]);
    } else if (strcmp (event->event_name, "program_unloaded") == 0) {
        gtk_signal_emit (GTK_OBJECT (dbg),
                         debugger_client_signals[PROGRAM_UNLOADED]);
    } else if (strcmp (event->event_name, "breakpoint_set") == 0) {
        g_assert (event->arg_type == GDF_EVENT_ARG_LONG);
        gtk_signal_emit (GTK_OBJECT (dbg),
                         debugger_client_signals[BREAKPOINT_SET],
                         (gint)event->argument.long_arg);
    } else if (strcmp (event->event_name, "breakpoint_enabled") == 0) {
        g_assert (event->arg_type == GDF_EVENT_ARG_LONG);
        gtk_signal_emit (GTK_OBJECT (dbg),
                         debugger_client_signals[BREAKPOINT_ENABLED],
                         (gint)event->argument.long_arg);
    } else if (strcmp (event->event_name, "breakpoint_disabled") == 0) {
        g_assert (event->arg_type == GDF_EVENT_ARG_LONG);
        gtk_signal_emit (GTK_OBJECT (dbg),
                         debugger_client_signals[BREAKPOINT_DISABLED],
                         (gint)event->argument.long_arg);
    } else if (strcmp (event->event_name, "breakpoint_deleted") == 0) {
        g_assert (event->arg_type == GDF_EVENT_ARG_LONG);
        gtk_signal_emit (GTK_OBJECT (dbg),
                         debugger_client_signals[BREAKPOINT_DELETED],
                         (gint)event->argument.long_arg);
    } else if (strcmp (event->event_name, "started") == 0) {
        gtk_signal_emit (GTK_OBJECT (dbg),
                         debugger_client_signals[EXECUTION_STARTED]);
    } else if (strcmp (event->event_name, "running") == 0) {
        gtk_signal_emit (GTK_OBJECT (dbg),
                         debugger_client_signals[EXECUTION_RUNNING]);
    } else if (strcmp (event->event_name, "stopped") == 0) {
        gtk_signal_emit (GTK_OBJECT (dbg),
                         debugger_client_signals[EXECUTION_STOPPED]);
    } else if (strcmp (event->event_name, "killed") == 0) {
        g_assert (event->arg_type == GDF_EVENT_ARG_LONG);
        gtk_signal_emit (GTK_OBJECT (dbg),
                         debugger_client_signals[EXECUTION_KILLED],
                         (gint)event->argument.long_arg);
    } else if (strcmp (event->event_name, "exited") == 0) {
        g_assert (event->arg_type == GDF_EVENT_ARG_LONG);
        dbg->priv->cur_frame = -1;
        gtk_signal_emit (GTK_OBJECT (dbg),
                         debugger_client_signals[EXECUTION_EXITED],
                         (gint)event->argument.long_arg);
    } else if (strcmp (event->event_name, "source_line") == 0) {
        g_assert (event->arg_type == GDF_EVENT_ARG_SOURCE_LOCATION);
        gtk_signal_emit (GTK_OBJECT (dbg),
                         debugger_client_signals[EXECUTION_SOURCE_LINE],
                         (gchar*)event->argument.loc_arg->file,
                         (gint)event->argument.loc_arg->line);
    } else if (strcmp (event->event_name, "frame_change") == 0) {
        g_assert (event->arg_type = GDF_EVENT_ARG_LONG);
        dbg->priv->cur_frame = event->argument.long_arg;
        gtk_signal_emit (GTK_OBJECT (dbg), 
                         debugger_client_signals[STACK_FRAME_CHANGED],
                         GPOINTER_TO_INT (event->argument.long_arg));
    } else if (strcmp (event->event_name, "signal") == 0) {
        g_assert (event->arg_type = GDF_EVENT_ARG_STRING);
        gtk_signal_emit (GTK_OBJECT (dbg),
                         debugger_client_signals[SIGNAL_RECEIVED],
                         event->argument.string_arg);
    } else if (strcmp (event->event_name, "signal_termination") == 0) {
        g_assert (event->arg_type = GDF_EVENT_ARG_STRING);
        gtk_signal_emit (GTK_OBJECT (dbg),
                         debugger_client_signals[SIGNAL_TERMINATION],
                         event->argument.string_arg);
    }
}

void 
event_pushed_cb (GdfEventChannelClient *event_channel,
                 CORBA_any *val,
                 GdfDebuggerClient *dbg)
{
    GdfEvent *event;
    event = gdf_demarshal_event (val);
    
    emit_event_signal (dbg, event);

    gdf_event_destroy (event);
}

void
program_loaded_impl (GdfDebuggerClient *dbg) 
{
    dbg->program_loaded = TRUE;
}

void
program_unloaded_impl (GdfDebuggerClient *dbg)
{
    dbg->program_loaded = FALSE;
}



