/*  -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * 
 * This file is part of the GNOME Debugging Framework.
 * 
 * Copyright (C) 1999-2000 Dave Camp <campd@oit.edu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.  
 */

#include <gnome.h>
#include "gdf-breakpoint-manager.h"

struct _GdfBreakpointManagerPriv {
	GtkWidget *clist;
	GtkWidget *scrolled;
	
	/* signal identifiers */
	guint program_unloaded_sig;
	guint breakpoint_set_sig;
	guint breakpoint_disabled_sig;
	guint breakpoint_enabled_sig;
	guint breakpoint_deleted_sig;

	GdfDebuggerClient *dbg;
};

enum {
    ARG_0,
    ARG_DEBUGGER
};

static void breakpoint_manager_class_init (GdfBreakpointManagerClass *klass);
static void breakpoint_manager_init (GdfBreakpointManager *bpm);
static void breakpoint_manager_destroy (GtkObject *object);
static void get_arg (GtkObject *object, 
                     GtkArg *arg, 
                     guint arg_id);
static void set_arg (GtkObject *object, 
                     GtkArg *arg, 
                     guint arg_id);
static void create_children (GdfBreakpointManager *bpm);
static void clear_breakpoint_manager (GdfBreakpointManager *bpm);
static void connect_debugger_signals (GdfBreakpointManager *bpm);
static void disconnect_debugger_signals (GdfBreakpointManager *bpm);
static gint get_row_by_bpnum (GdfBreakpointManager *bpm, gint bp_num);
static gboolean bp_is_enabled (GdfBreakpointManager *bpm, gint bp_num);
static void add_breakpoint (GdfBreakpointManager *bpm, gint bp_num,
							gboolean enabled, GDF_MemoryAddress address, 
							gchar *filename, gint line_num);
static void enable_breakpoint (GdfBreakpointManager *bpm, gint bp_num);
static void disable_breakpoint (GdfBreakpointManager *bpm, gint bp_num);
static void delete_breakpoint (GdfBreakpointManager *bpm, gint bp_num);
static gint compare_func (GtkCList *clist, 
						  gconstpointer ptr1, gconstpointer ptr2);
static void enable_breakpoint_selected_cb (GtkWidget *widget,
										   GdfBreakpointManager *bpm);
static void disable_breakpoint_selected_cb (GtkWidget *widget, 
											GdfBreakpointManager *bpm);
static void delete_breakpoint_selected_cb (GtkWidget *widget, 
										   GdfBreakpointManager *bpm);
static void do_popup (GdfBreakpointManager *bpm, gint bp_num);
static gint button_press_cb (GtkWidget *clist, GdkEventButton *event, 
							 GdfBreakpointManager *bpm);
static void breakpoint_set_cb (GdfDebuggerClient *dbg, gint bp_num, 
							   GdfBreakpointManager *bpm);
static void breakpoint_enabled_cb (GdfDebuggerClient *dbg, gint bp_num, 
								   GdfBreakpointManager *bpm);
static void breakpoint_disabled_cb (GdfDebuggerClient *dbg, gint bp_num, 
									GdfBreakpointManager *bpm);
static void breakpoint_deleted_cb (GdfDebuggerClient *dbg, gint bp_num, 
								   GdfBreakpointManager *bpm);


#define NUM_COLUMNS 5
static const gint bpm_column_widths[NUM_COLUMNS] = { 10, 30, 55, 35 };

static GnomeUIInfo popup_menu[] =
{
    {                       /* Will be enable/disable breakpoint */
        GNOME_APP_UI_ITEM,
        NULL,
        NULL,
        NULL,
        NULL,
        NULL,
        GNOME_APP_PIXMAP_STOCK,
        NULL,
        0, 0, NULL
    },
    {
        GNOME_APP_UI_ITEM,
        N_("Delete breakpoint"),
        N_("Remove the current breakpoint"),
        delete_breakpoint_selected_cb,
        NULL,
        NULL,
        GNOME_APP_PIXMAP_NONE,
        NULL,
        0, 0, NULL
    }, 
    GNOMEUIINFO_END
};

static GtkFrameClass *parent_class;

/* 
 * Public interface 
 */

GtkType
gdf_breakpoint_manager_get_type (void)
{
	static GtkType breakpoint_manager_type = 0;
	
	if (!breakpoint_manager_type) {
		static const GtkTypeInfo breakpoint_manager_info = {
			"GdfBreakpointManager",
			sizeof (GdfBreakpointManager),
			sizeof (GdfBreakpointManagerClass),
			(GtkClassInitFunc) breakpoint_manager_class_init,
			(GtkObjectInitFunc) breakpoint_manager_init,
			NULL,
			NULL,
			(GtkClassInitFunc)NULL
		};
		
		breakpoint_manager_type = gtk_type_unique (gtk_frame_get_type (),
												   &breakpoint_manager_info);
	}
	
	return breakpoint_manager_type;
}

/**
 * gdf_breakpoint_manager_new:
 * 
 * Creates a new breakpoint manager.
 * 
 * Return value: An initialized breakpoint manager.
 **/
GdfBreakpointManager *
gdf_breakpoint_manager_new (void)
{
	GdfBreakpointManager *bpm;
	
	bpm = gtk_type_new (gdf_breakpoint_manager_get_type ());
	
	return bpm;
}

/**
 * gdf_breakpoint_manager_set_debugger:
 * @bpm: The breakpoint manager.
 * @dbg: The debugger to associate.
 * 
 * Sets the breakpoint manager's associated debugger.  The new debugger will
 * be ref'ed by the breakpoint manager.  Any previous debugger will be 
 * unref'ed.
 **/
void
gdf_breakpoint_manager_set_debugger (GdfBreakpointManager *bpm,
									  GdfDebuggerClient *dbg)
{
	g_return_if_fail (bpm != NULL);
	g_return_if_fail (GDF_IS_BREAKPOINT_MANAGER (bpm));

	if (bpm->priv->dbg) {
		disconnect_debugger_signals (bpm);
		clear_breakpoint_manager (bpm);
		gtk_object_unref (GTK_OBJECT (bpm->priv->dbg));
	}
	
	bpm->priv->dbg = dbg;

    if (dbg) {
        gtk_object_ref (GTK_OBJECT (dbg));
        if (GTK_OBJECT_FLOATING (GTK_OBJECT (dbg))) 
            gtk_object_sink (GTK_OBJECT (dbg));
        
        connect_debugger_signals (bpm);
    }
}

/*
 * Class/object functions.
 */

void
breakpoint_manager_class_init (GdfBreakpointManagerClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass *)klass;
	
    gtk_object_add_arg_type ("GdfBreakpointManager::debugger",
                             GTK_TYPE_OBJECT,
                             GTK_ARG_READWRITE | GTK_ARG_CONSTRUCT, 
                             ARG_DEBUGGER);

	parent_class = gtk_type_class (gtk_frame_get_type ());

	object_class->destroy = breakpoint_manager_destroy;
    object_class->get_arg = get_arg;
    object_class->set_arg = set_arg;
}

void
breakpoint_manager_init (GdfBreakpointManager *bpm)
{
	bpm->priv = g_new0 (GdfBreakpointManagerPriv, 1);

	bpm->priv->dbg = NULL;

	create_children (bpm);
}

void 
breakpoint_manager_destroy (GtkObject *obj)
{
	GdfBreakpointManager *bpm = GDF_BREAKPOINT_MANAGER (obj);
	
	g_return_if_fail (obj != NULL);
	
	if (bpm->priv->dbg)
		gtk_object_unref (GTK_OBJECT (bpm->priv->dbg));

	g_free (bpm->priv);
	
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (obj);
}

void 
get_arg (GtkObject *object, 
         GtkArg *arg, 
         guint arg_id)
{
    GdfBreakpointManager *bpm = GDF_BREAKPOINT_MANAGER (object);
    
    switch (arg_id) {
    case ARG_DEBUGGER :
        GTK_VALUE_OBJECT (*arg) = GTK_OBJECT (bpm->priv->dbg);
        break;
    default :
        arg->type = GTK_TYPE_INVALID;
    }   
}

void 
set_arg (GtkObject *object, 
         GtkArg *arg, 
         guint arg_id)
{
    GdfBreakpointManager *bpm = GDF_BREAKPOINT_MANAGER (object);
    
    switch (arg_id) {
    case ARG_DEBUGGER :
        gdf_breakpoint_manager_set_debugger (bpm,
                                             GTK_VALUE_OBJECT (*arg) ? GDF_DEBUGGER_CLIENT (GTK_VALUE_OBJECT (*arg)) : NULL);
        break;
    default :
        break;
    }   
}

/* 
 * Helper functions
 */
void
create_children (GdfBreakpointManager *bpm)
{
	gint i;
	static gchar *titles[] = {
		N_("#"),
		N_("Enbl"),
		N_("Address"),
		N_("Line #"),
		N_("Filename")
	};

	bpm->priv->clist = gtk_clist_new_with_titles (NUM_COLUMNS, titles);
	gtk_clist_set_compare_func (GTK_CLIST (bpm->priv->clist), compare_func);
	gtk_clist_set_auto_sort (GTK_CLIST (bpm->priv->clist), TRUE);
	gtk_signal_connect (GTK_OBJECT (bpm->priv->clist), "button_press_event",
						GTK_SIGNAL_FUNC (button_press_cb), (gpointer) bpm);
	
	/* Set the width on all but the last column */
	for (i = 0; i < NUM_COLUMNS - 1; i++) {
		gtk_clist_set_column_width (GTK_CLIST (bpm->priv->clist), i, 
									bpm_column_widths[i]);	
	}

	bpm->priv->scrolled = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (bpm->priv->scrolled),
									GTK_POLICY_AUTOMATIC,
									GTK_POLICY_AUTOMATIC);	

	gtk_container_add (GTK_CONTAINER (bpm->priv->scrolled), bpm->priv->clist);
	gtk_container_add (GTK_CONTAINER (bpm), bpm->priv->scrolled);

	gtk_widget_show (bpm->priv->clist);
	gtk_widget_show (bpm->priv->scrolled);
}

void 
clear_breakpoint_manager (GdfBreakpointManager *bpm)
{
	gtk_clist_clear (GTK_CLIST (bpm->priv->clist));
}

void
connect_debugger_signals (GdfBreakpointManager *bpm)
{
	bpm->priv->program_unloaded_sig = 
		gtk_signal_connect_object (GTK_OBJECT (bpm->priv->dbg), 
								   "program_unloaded",
								   GTK_SIGNAL_FUNC (clear_breakpoint_manager),
								   (gpointer)bpm);
	
	bpm->priv->breakpoint_set_sig = 
		gtk_signal_connect (GTK_OBJECT (bpm->priv->dbg), 
							"breakpoint_set",
							GTK_SIGNAL_FUNC (breakpoint_set_cb),
							(gpointer)bpm);

	bpm->priv->breakpoint_enabled_sig = 
		gtk_signal_connect (GTK_OBJECT (bpm->priv->dbg), 
							"breakpoint_enabled",
							GTK_SIGNAL_FUNC (breakpoint_enabled_cb),
							(gpointer)bpm);

	bpm->priv->breakpoint_disabled_sig = 
		gtk_signal_connect (GTK_OBJECT (bpm->priv->dbg), 
							"breakpoint_disabled",
							GTK_SIGNAL_FUNC (breakpoint_disabled_cb),
							(gpointer)bpm);

	bpm->priv->breakpoint_deleted_sig = 
		gtk_signal_connect (GTK_OBJECT (bpm->priv->dbg), 
							"breakpoint_deleted",
							GTK_SIGNAL_FUNC (breakpoint_deleted_cb),
							(gpointer)bpm);
	
}

void
disconnect_debugger_signals (GdfBreakpointManager *bpm)
{
	gtk_signal_disconnect (GTK_OBJECT (bpm->priv->dbg), 
						   bpm->priv->program_unloaded_sig);
	gtk_signal_disconnect (GTK_OBJECT (bpm->priv->dbg),
						   bpm->priv->breakpoint_set_sig);
	gtk_signal_disconnect (GTK_OBJECT (bpm->priv->dbg),
						   bpm->priv->breakpoint_enabled_sig);
	gtk_signal_disconnect (GTK_OBJECT (bpm->priv->dbg),
						   bpm->priv->breakpoint_disabled_sig);
	gtk_signal_disconnect (GTK_OBJECT (bpm->priv->dbg),
						   bpm->priv->breakpoint_deleted_sig);
}


gint
get_row_by_bpnum (GdfBreakpointManager *bpm, gint bp_num)
{
	gchar *row_text;
	gint num;
	gint row;
	gint top, bottom;
	gboolean success = FALSE;
	
	/* Do a binary search of the breakpoints */

	top = GTK_CLIST (bpm->priv->clist)->rows - 1;
	bottom = 0;
	
	do {
		row = ((top + bottom) / 2);
		gtk_clist_get_text (GTK_CLIST (bpm->priv->clist), row, 0, &row_text);
		num = atoi (row_text);
		
		if (num == bp_num) 
			success = TRUE;
		else if (num < bp_num) 
			bottom = row + 1;
		else 
			top = row - 1;
	} while (!success && bottom <= top);
	
	return success ? row : -1;
}	

gboolean 
bp_is_enabled (GdfBreakpointManager *bpm, gint bp_num)
{
	gchar *text;
	gint row;
	
	row = get_row_by_bpnum (bpm, bp_num);
	gtk_clist_get_text (GTK_CLIST (bpm->priv->clist), row, 1, &text);
	
	return text[0] == 'Y' ? TRUE : FALSE;
}

void 
add_breakpoint (GdfBreakpointManager *bpm, gint bp_num,
				gboolean enabled, GDF_MemoryAddress address, gchar *filename,
				gint line_num)
{
	gchar *text[6];
	
	text[0] = g_strdup_printf ("%d", bp_num);
	text[1] = enabled ? _("Y") : _("N");
	text[2] = g_strdup_printf ("%p", (gpointer)address);
	text[3] = g_strdup_printf ("%d", line_num);
	text[4] = filename;
	
	gtk_clist_append (GTK_CLIST (bpm->priv->clist), text);

	g_free (text[0]);
	g_free (text[2]);
	g_free (text[3]);
}

void 
enable_breakpoint (GdfBreakpointManager *bpm, gint bp_num)
{
	gint row = get_row_by_bpnum (bpm, bp_num);
	
	g_return_if_fail (row != -1);

	gtk_clist_set_text (GTK_CLIST (bpm->priv->clist), row, 1, _("Y"));
}

void 
disable_breakpoint (GdfBreakpointManager *bpm, gint bp_num)
{
	gint row = get_row_by_bpnum (bpm, bp_num);
	
	g_return_if_fail (row != -1);

	gtk_clist_set_text (GTK_CLIST (bpm->priv->clist), row, 1, _("N"));
}

void
delete_breakpoint (GdfBreakpointManager *bpm, gint bp_num)
{
	gint row = get_row_by_bpnum (bpm, bp_num);
	
	gtk_clist_remove (GTK_CLIST (bpm->priv->clist), row);
}

gint 
compare_func (GtkCList *clist, 
			  gconstpointer ptr1, gconstpointer ptr2)
{
	gchar *txt1 = NULL;
	gchar *txt2 = NULL;
	gint line_num1 = 0;
	gint line_num2 = 0;
	GtkCListRow *row1 = (GtkCListRow *) ptr1;
	GtkCListRow *row2 = (GtkCListRow *) ptr2;
	
	txt1 = GTK_CELL_TEXT (row1->cell[0])->text;
	txt2 = GTK_CELL_TEXT (row2->cell[0])->text;
	line_num1 = atoi (txt1);
	line_num2 = atoi (txt2);
	
	if (line_num1 == line_num2) {
		return 0;
	} else if (line_num1 < line_num2) {
		return -1;
	} else {
		return 1;
	}
}

void 
enable_breakpoint_selected_cb (GtkWidget *widget, GdfBreakpointManager *bpm)
{
	g_return_if_fail (bpm != NULL);

	if (bpm->priv->dbg) {
		gint bp_num;
		bp_num = 
			GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (bpm), "bp_num"));
		
		gdf_debugger_client_enable_breakpoint (bpm->priv->dbg, bp_num);
	}
}

void 
disable_breakpoint_selected_cb (GtkWidget *widget, GdfBreakpointManager *bpm)
{
	g_return_if_fail (bpm != NULL);

	if (bpm->priv->dbg) {
		gint bp_num;
		bp_num = 
			GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (bpm), "bp_num"));
		
		gdf_debugger_client_disable_breakpoint (bpm->priv->dbg, bp_num);
	}
}

void 
delete_breakpoint_selected_cb (GtkWidget *widget, GdfBreakpointManager *bpm)
{
	g_return_if_fail (bpm != NULL);

	if (bpm->priv->dbg) {
		gint bp_num;
		bp_num = 
			GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (bpm), "bp_num"));
		
		gdf_debugger_client_delete_breakpoint (bpm->priv->dbg, bp_num);
	}
}

void
do_popup (GdfBreakpointManager *bpm, gint bp_num)
{
	GtkWidget *menu = NULL;

	gtk_object_set_data (GTK_OBJECT (bpm), "bp_num", GINT_TO_POINTER (bp_num));

	if (bp_is_enabled (bpm, bp_num)) {
		popup_menu[0].label = _("Disable breakpoint");
		popup_menu[0].hint = _("Disable the selected breakpoint");
		popup_menu[0].moreinfo = (gpointer)disable_breakpoint_selected_cb;
		popup_menu[0].user_data = bpm;
 		popup_menu[0].pixmap_info = GNOME_STOCK_MENU_CLOSE;
	} else {
		popup_menu[0].label = _("Enable breakpoint");
		popup_menu[0].hint = _("Enable the selected breakpoint");
		popup_menu[0].moreinfo = (gpointer)enable_breakpoint_selected_cb;
		popup_menu[0].user_data = bpm;
		popup_menu[0].pixmap_info = GNOME_STOCK_MENU_STOP;
	}

	menu = gnome_popup_menu_new (popup_menu);
	gnome_popup_menu_do_popup (menu, NULL, NULL, NULL, (gpointer)bpm);
}


/* 
 * Widget Callbacks
 */

gint
button_press_cb (GtkWidget *widget, GdkEventButton *event, 
				 GdfBreakpointManager *bpm)
{
	gint row;
	gint bp_num;
	gchar *text;
	
	if (!gtk_clist_get_selection_info (GTK_CLIST (widget), 
									   (gint)event->x, (gint)event->y,
									   &row, NULL))
		return TRUE;

	gtk_clist_get_text (GTK_CLIST (widget), row, 0, &text);
	bp_num = atoi (text);
	
	if (event->button == 3)
		do_popup (bpm, bp_num);

	return FALSE;
}

/* 
 * Debugger Callbacks 
 */					
					  
void
breakpoint_set_cb (GdfDebuggerClient *dbg, gint bp_num, 
				   GdfBreakpointManager *bpm)
{
	GDF_Breakpoint *bp;
	gchar *filename;

	gdf_debugger_client_get_breakpoint_info (dbg, bp_num, &bp);
	
	filename = strrchr (bp->file_name, G_DIR_SEPARATOR);
	if (!filename++)
		filename = bp->file_name;
	
	add_breakpoint (bpm, bp_num, (gboolean) bp->enabled,
					bp->address, (gchar*)filename, bp->line_num);

	CORBA_free (bp);
}

void
breakpoint_enabled_cb (GdfDebuggerClient *dbg, gint bp_num, 
					   GdfBreakpointManager *bpm)
{
	enable_breakpoint (bpm, bp_num);
}

void
breakpoint_disabled_cb (GdfDebuggerClient *dbg, gint bp_num, 
						GdfBreakpointManager *bpm)
{
	disable_breakpoint (bpm, bp_num);
}

void
breakpoint_deleted_cb (GdfDebuggerClient *dbg, gint bp_num, 
					   GdfBreakpointManager *bpm)
{
	delete_breakpoint (bpm, bp_num);
}

