/*  -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * 
 * This file is part of the GNOME Debugging Framework.
 * 
 * Copyright (C) 1999-2000 Dave Camp <campd@oit.edu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.  
 */

#include <gnome.h>
#include "gdf-stack-browser.h"

struct _GdfStackBrowserPriv {
    GtkWidget *clist;
    GtkWidget *scrolled;

    GdfDebuggerClient *dbg;

    int frame_changed_sig;
    int corefile_loaded_sig;
    int corefile_unloaded_sig;
    int execution_stopped_sig;
    int execution_exited_sig;
    int execution_killed_sig;
    int program_unloaded_sig;
};

enum {
    ARG_0,
    ARG_DEBUGGER
};

static void stack_browser_init (GdfStackBrowser *sb);
static void stack_browser_class_init (GdfStackBrowserClass *klass);
static void stack_browser_destroy (GtkObject *object);
static void get_arg (GtkObject *object, 
                     GtkArg *arg, 
                     guint arg_id);
static void set_arg (GtkObject *object, 
                     GtkArg *arg, 
                     guint arg_id);
static void connect_debugger_signals (GdfStackBrowser *sb);
static void disconnect_debugger_signals (GdfStackBrowser *sb);
static void frame_changed_cb (GdfDebuggerClient *dbg, int frame_id, 
                              GdfStackBrowser *sb);
static void execution_stopped_cb (GdfDebuggerClient *dbg,
                                  GdfStackBrowser *sb);
static void select_row_cb (GtkWidget *clist, gint row, gint column, 
                           GdkEvent *event, GdfStackBrowser *sb);
static void create_children (GdfStackBrowser *sb);

static GtkFrameClass *parent_class;

/*
 * Public Interface 
 */

GtkType 
gdf_stack_browser_get_type ()
{
    static GtkType type = 0;
	
    if (!type) {
        static const GtkTypeInfo info = {
            "GdfStackBrowser",
            sizeof (GdfStackBrowser),
            sizeof (GdfStackBrowserClass),
            (GtkClassInitFunc) stack_browser_class_init,
            (GtkObjectInitFunc) stack_browser_init,
            NULL,
            NULL,
            (GtkClassInitFunc)NULL
        };
		
        type = gtk_type_unique (gtk_frame_get_type (), &info);
    }
	
    return type;
}

GtkWidget *
gdf_stack_browser_new ()
{
    GdfStackBrowser *sb;
    
    sb = gtk_type_new (gdf_stack_browser_get_type ());
	
    return GTK_WIDGET (sb);
}

void
gdf_stack_browser_set_debugger (GdfStackBrowser *sb,
                                   GdfDebuggerClient *dbg)
{
    g_return_if_fail (sb != NULL);
    g_return_if_fail (GDF_IS_STACK_BROWSER (sb));
	
    if (sb->priv->dbg) {
        disconnect_debugger_signals (sb);
        gtk_object_unref (GTK_OBJECT (sb->priv->dbg));
    }
	
    sb->priv->dbg = dbg;

    if (dbg) {
        gtk_object_ref (GTK_OBJECT (dbg));
        if (GTK_OBJECT_FLOATING (GTK_OBJECT (dbg)))
            gtk_object_sink (GTK_OBJECT (dbg));
        
        connect_debugger_signals (sb);
    }
}


/*
 * Class / Object functions
 */

void
stack_browser_class_init (GdfStackBrowserClass *klass)
{
    GtkObjectClass *object_class = (GtkObjectClass *)klass;

    gtk_object_add_arg_type ("GdfStackBrowser::debugger",
                             GTK_TYPE_OBJECT,
                             GTK_ARG_READWRITE | GTK_ARG_CONSTRUCT, 
                             ARG_DEBUGGER);


    parent_class = gtk_type_class (gtk_frame_get_type ());
	
    object_class->destroy = stack_browser_destroy;
    object_class->get_arg = get_arg;
    object_class->set_arg = set_arg;
}

void
stack_browser_init (GdfStackBrowser *sb)
{
    sb->priv = g_new0 (GdfStackBrowserPriv, 1);
    
    sb->priv->dbg = NULL;

    create_children (sb);
}

void
stack_browser_destroy (GtkObject *obj)
{
    GdfStackBrowser *sb = GDF_STACK_BROWSER (obj);
    
    if (sb->priv->dbg)
        gtk_object_unref (GTK_OBJECT (sb->priv->dbg));
    
    g_free (sb->priv);
    
    if (GTK_OBJECT_CLASS (parent_class)->destroy)
        (*GTK_OBJECT_CLASS (parent_class)->destroy) (obj);
}

void 
get_arg (GtkObject *object, 
         GtkArg *arg, 
         guint arg_id)
{
    GdfStackBrowser *sb = GDF_STACK_BROWSER (object);
    
    switch (arg_id) {
    case ARG_DEBUGGER :
        GTK_VALUE_OBJECT (*arg) = GTK_OBJECT (sb->priv->dbg);
        break;
    default :
        arg->type = GTK_TYPE_INVALID;
    }   
}

void 
set_arg (GtkObject *object, 
         GtkArg *arg, 
         guint arg_id)
{
    GdfStackBrowser *sb = GDF_STACK_BROWSER (object);
    
    switch (arg_id) {
    case ARG_DEBUGGER :
        gdf_stack_browser_set_debugger (sb,
                                        GTK_VALUE_OBJECT (*arg) ? GDF_DEBUGGER_CLIENT (GTK_VALUE_OBJECT (*arg)) : NULL);
        break;
    default :
        break;
    }   
}

/* 
 * Helper functions 
 */

void
connect_debugger_signals (GdfStackBrowser *sb)
{
    sb->priv->frame_changed_sig = 
        gtk_signal_connect (GTK_OBJECT (sb->priv->dbg), 
                            "stack_frame_changed",
                            GTK_SIGNAL_FUNC (frame_changed_cb),
                            (gpointer)sb);
    sb->priv->corefile_loaded_sig = 
        gtk_signal_connect (GTK_OBJECT (sb->priv->dbg), 
                            "corefile_loaded",
                            GTK_SIGNAL_FUNC (execution_stopped_cb),
                            (gpointer)sb);
    sb->priv->corefile_unloaded_sig = 
        gtk_signal_connect_object (GTK_OBJECT (sb->priv->dbg),
                                   "corefile_unloaded",
                                   GTK_SIGNAL_FUNC (gtk_clist_clear),
                                   (gpointer)sb->priv->clist);
    sb->priv->execution_stopped_sig = 
        gtk_signal_connect (GTK_OBJECT (sb->priv->dbg), 
                            "execution_stopped",
                            GTK_SIGNAL_FUNC (execution_stopped_cb),
                            (gpointer)sb);
    sb->priv->execution_exited_sig = 
        gtk_signal_connect_object (GTK_OBJECT (sb->priv->dbg),
                                   "execution_exited",
                                   GTK_SIGNAL_FUNC (gtk_clist_clear),
                                   (gpointer)sb->priv->clist);
    sb->priv->execution_killed_sig = 
        gtk_signal_connect_object (GTK_OBJECT (sb->priv->dbg),
                                   "execution_killed",
                                   GTK_SIGNAL_FUNC (gtk_clist_clear),
                                   (gpointer)sb->priv->clist);
    sb->priv->program_unloaded_sig = 
        gtk_signal_connect_object (GTK_OBJECT (sb->priv->dbg),
                                   "program_unloaded",
                                   GTK_SIGNAL_FUNC (gtk_clist_clear),
                                   (gpointer)sb->priv->clist);
}

void
disconnect_debugger_signals (GdfStackBrowser *sb)
{
    gtk_signal_disconnect (GTK_OBJECT (sb->priv->dbg),
                           sb->priv->frame_changed_sig);
    gtk_signal_disconnect (GTK_OBJECT (sb->priv->dbg),
                           sb->priv->corefile_loaded_sig);
    gtk_signal_disconnect (GTK_OBJECT (sb->priv->dbg),
                           sb->priv->corefile_unloaded_sig);
    gtk_signal_disconnect (GTK_OBJECT (sb->priv->dbg),
                           sb->priv->execution_stopped_sig);
    gtk_signal_disconnect (GTK_OBJECT (sb->priv->dbg), 
                           sb->priv->execution_exited_sig);
    gtk_signal_disconnect (GTK_OBJECT (sb->priv->dbg), 
                           sb->priv->execution_killed_sig);
    gtk_signal_disconnect (GTK_OBJECT (sb->priv->dbg), 
                           sb->priv->program_unloaded_sig);
    
}

void
frame_changed_cb (GdfDebuggerClient *dbg,
                  int frame_id,
                  GdfStackBrowser *sb)
{
    gtk_clist_select_row (GTK_CLIST (sb->priv->clist), frame_id, 0);
}

void
execution_stopped_cb (GdfDebuggerClient *dbg, 
                      GdfStackBrowser *sb)
{
    GdfDebuggerClientResult res;
    GDF_Stack *stack;
    int i;
    char *text[5];
    char buf1[20];
    char buf2[20];

    res = gdf_debugger_client_get_backtrace (dbg, &stack);
    if (res != GDF_DEBUGGER_CLIENT_OK) {
        gtk_clist_clear (GTK_CLIST (sb->priv->clist));
        return;
    }
    
    gtk_clist_freeze (GTK_CLIST (sb->priv->clist));
    gtk_clist_clear (GTK_CLIST (sb->priv->clist));

    for (i = 0; i < stack->_length; i++) {
        sprintf (buf1, "%d", stack->_buffer[i].id);
        text[0] = buf1;
        text[1] = stack->_buffer[i].function;
        text[2] = stack->_buffer[i].location.file;
        sprintf (buf2, "%d", stack->_buffer[i].location.line);
        text[3] = buf2;
        text[4] = "";
        gtk_clist_append (GTK_CLIST (sb->priv->clist), text);
    }
    gtk_clist_thaw (GTK_CLIST (sb->priv->clist));
    CORBA_free (stack);
}

void 
select_row_cb (GtkWidget *clist, gint row, gint column, GdkEvent *event,
               GdfStackBrowser *sb)
{
    int cur_frame;

    cur_frame = gdf_debugger_client_current_frame (sb->priv->dbg);
    
    if (cur_frame != -1 && cur_frame != row) {
        gdf_debugger_client_change_frame (sb->priv->dbg, row);
    }
}

void
create_children (GdfStackBrowser *sb)
{
    gchar *titles[] = { N_("ID"), N_("Function"), 
                        N_("File"), N_("Line"), N_("Address") };
    
    sb->priv->clist = gtk_clist_new_with_titles (5, titles);
	
    sb->priv->scrolled = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sb->priv->scrolled),
                                    GTK_POLICY_AUTOMATIC,
                                    GTK_POLICY_AUTOMATIC);
	
    gtk_signal_connect (GTK_OBJECT (sb->priv->clist), "select_row",
                        GTK_SIGNAL_FUNC (select_row_cb), sb);

    gtk_container_add (GTK_CONTAINER (sb->priv->scrolled), sb->priv->clist);
    gtk_container_add (GTK_CONTAINER (sb), sb->priv->scrolled);
    gtk_widget_show (sb->priv->clist);
    gtk_widget_show (sb->priv->scrolled);
}

