/*
 * Copyright (C) 2008,2009 Sebastian Pölsterl
 *
 * This file is part of GNOME DVB Daemon.
 *
 * GNOME DVB Daemon is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GNOME DVB Daemon is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNOME DVB Daemon.  If not, see <http://www.gnu.org/licenses/>.
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))

#include <src/RecordingsStore.h>
#include <gee.h>
#include <time.h>
#include "src/ChangeType.h"
#include "src/Utils.h"

typedef struct _DBusObjectVTable _DBusObjectVTable;

struct _DBusObjectVTable {
	void (*register_object) (DBusConnection*, const char*, void*);
};



struct _DVBRecordingsStorePrivate {
	GStaticRecMutex __lock_recordings;
	GeeHashMap* recordings;
	GStaticRecMutex __lock_last_id;
	guint32 last_id;
};

#define DVB_RECORDINGS_STORE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), DVB_TYPE_RECORDINGS_STORE, DVBRecordingsStorePrivate))
enum  {
	DVB_RECORDINGS_STORE_DUMMY_PROPERTY
};
static DVBRecordingsStore* dvb_recordings_store_instance = NULL;
static GStaticRecMutex dvb_recordings_store_instance_mutex = {0};
static void _dvb_recordings_store_on_recording_file_changed_g_file_monitor_changed (GFileMonitor* _sender, GFile* file, GFile* other_file, GFileMonitorEvent event_type, gpointer self);
static guint32* dvb_recordings_store_real_GetRecordings (DVBIDBusRecordingsStore* base, int* result_length1);
static char* dvb_recordings_store_real_GetLocation (DVBIDBusRecordingsStore* base, guint32 rec_id);
static char* dvb_recordings_store_real_GetName (DVBIDBusRecordingsStore* base, guint32 rec_id);
static char* dvb_recordings_store_real_GetDescription (DVBIDBusRecordingsStore* base, guint32 rec_id);
static guint* dvb_recordings_store_real_GetStartTime (DVBIDBusRecordingsStore* base, guint32 rec_id, int* result_length1);
static gint64 dvb_recordings_store_real_GetStartTimestamp (DVBIDBusRecordingsStore* base, guint32 rec_id);
static gint64 dvb_recordings_store_real_GetLength (DVBIDBusRecordingsStore* base, guint32 rec_id);
static gboolean dvb_recordings_store_real_Delete (DVBIDBusRecordingsStore* base, guint32 rec_id);
static char* dvb_recordings_store_real_GetChannelName (DVBIDBusRecordingsStore* base, guint32 rec_id);
static gboolean dvb_recordings_store_delete_recording_by_location (DVBRecordingsStore* self, const char* location);
static void dvb_recordings_store_on_recording_file_changed (DVBRecordingsStore* self, GFileMonitor* monitor, GFile* file, GFile* other_file, GFileMonitorEvent event);
static GObject * dvb_recordings_store_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static gpointer dvb_recordings_store_parent_class = NULL;
void _dvb_recordings_store_dbus_unregister (DBusConnection* connection, void* user_data);
static DBusMessage* _dbus_dvb_recordings_store_introspect (DVBRecordingsStore* self, DBusConnection* connection, DBusMessage* message);
static DVBIDBusRecordingsStoreIface* dvb_recordings_store_dvb_id_bus_recordings_store_parent_iface = NULL;
static void dvb_recordings_store_finalize (GObject* obj);
static int _vala_strcmp0 (const char * str1, const char * str2);
static void _vala_dbus_register_object (DBusConnection* connection, const char* path, void* object);
static void _vala_dbus_unregister_object (gpointer connection, GObject* object);

static const DBusObjectPathVTable _dvb_recordings_store_dbus_path_vtable = {_dvb_recordings_store_dbus_unregister, dvb_recordings_store_dbus_message};
static const _DBusObjectVTable _dvb_recordings_store_dbus_vtable = {dvb_recordings_store_dbus_register_object};


DVBRecordingsStore* dvb_recordings_store_get_instance (void) {
	g_static_rec_mutex_lock (&dvb_recordings_store_instance_mutex);
	if (dvb_recordings_store_instance == NULL) {
		DVBRecordingsStore* _tmp0;
		_tmp0 = NULL;
		dvb_recordings_store_instance = (_tmp0 = dvb_recordings_store_new (), (dvb_recordings_store_instance == NULL) ? NULL : (dvb_recordings_store_instance = (g_object_unref (dvb_recordings_store_instance), NULL)), _tmp0);
	}
	g_static_rec_mutex_unlock (&dvb_recordings_store_instance_mutex);
	return dvb_recordings_store_instance;
}


void dvb_recordings_store_shutdown (void) {
	DVBRecordingsStore* _tmp0;
	DVBRecordingsStore* rs;
	g_static_rec_mutex_lock (&dvb_recordings_store_instance_mutex);
	_tmp0 = NULL;
	rs = (_tmp0 = dvb_recordings_store_instance, (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0));
	if (rs != NULL) {
		DVBRecordingsStore* _tmp1;
		gee_map_clear ((GeeMap*) rs->priv->recordings);
		_tmp1 = NULL;
		dvb_recordings_store_instance = (_tmp1 = NULL, (dvb_recordings_store_instance == NULL) ? NULL : (dvb_recordings_store_instance = (g_object_unref (dvb_recordings_store_instance), NULL)), _tmp1);
	}
	g_static_rec_mutex_unlock (&dvb_recordings_store_instance_mutex);
	(rs == NULL) ? NULL : (rs = (g_object_unref (rs), NULL));
}


void dvb_recordings_store_update_last_id (DVBRecordingsStore* self, guint32 new_last_id) {
	g_return_if_fail (self != NULL);
	g_static_rec_mutex_lock (&self->priv->__lock_last_id);
	{
		if (new_last_id > self->priv->last_id) {
			self->priv->last_id = new_last_id;
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_last_id);
}


static void _dvb_recordings_store_on_recording_file_changed_g_file_monitor_changed (GFileMonitor* _sender, GFile* file, GFile* other_file, GFileMonitorEvent event_type, gpointer self) {
	dvb_recordings_store_on_recording_file_changed (self, _sender, file, other_file, event_type);
}


gboolean dvb_recordings_store_add (DVBRecordingsStore* self, DVBRecording* rec) {
	GError * inner_error;
	guint32 id;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (rec != NULL, FALSE);
	inner_error = NULL;
	id = dvb_recording_get_Id (rec);
	g_static_rec_mutex_lock (&self->priv->__lock_recordings);
	{
		if (gee_map_contains ((GeeMap*) self->priv->recordings, GUINT_TO_POINTER (id))) {
			g_critical ("RecordingsStore.vala:70: Recording with id %u already available", (guint) id);
			return FALSE;
		}
		{
			GFileMonitor* _tmp1;
			GFileMonitor* monitor;
			_tmp1 = NULL;
			monitor = (_tmp1 = g_file_monitor_file (dvb_recording_get_Location (rec), 0, NULL, &inner_error), (_tmp1 == NULL) ? NULL : g_object_ref (_tmp1));
			if (inner_error != NULL) {
				goto __catch14_g_error;
				goto __finally14;
			}
			g_signal_connect_object (monitor, "changed", (GCallback) _dvb_recordings_store_on_recording_file_changed_g_file_monitor_changed, self, 0);
			(monitor == NULL) ? NULL : (monitor = (g_object_unref (monitor), NULL));
		}
		goto __finally14;
		__catch14_g_error:
		{
			GError * e;
			e = inner_error;
			inner_error = NULL;
			{
				g_warning ("RecordingsStore.vala:79: Could not create FileMonitor: %s", e->message);
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			}
		}
		__finally14:
		if (inner_error != NULL) {
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return FALSE;
		}
		gee_map_set ((GeeMap*) self->priv->recordings, GUINT_TO_POINTER (id), rec);
		g_signal_emit_by_name ((DVBIDBusRecordingsStore*) self, "changed", id, (guint) DVB_CHANGE_TYPE_ADDED);
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_recordings);
	return TRUE;
}


guint32 dvb_recordings_store_get_next_id (DVBRecordingsStore* self) {
	guint32 val;
	g_return_val_if_fail (self != NULL, 0U);
	val = 0U;
	g_static_rec_mutex_lock (&self->priv->__lock_last_id);
	{
		val = self->priv->last_id = self->priv->last_id + 1;
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_last_id);
	return val;
}


/**
         * @returns: A list of ids for all recordings
         */
static guint32* dvb_recordings_store_real_GetRecordings (DVBIDBusRecordingsStore* base, int* result_length1) {
	DVBRecordingsStore * self;
	gint ids_size;
	gint ids_length1;
	guint32* ids;
	guint32* _tmp4;
	self = (DVBRecordingsStore*) base;
	ids = (ids_length1 = 0, NULL);
	g_static_rec_mutex_lock (&self->priv->__lock_recordings);
	{
		guint32* _tmp1;
		gint _tmp0;
		gint i;
		_tmp1 = NULL;
		ids = (_tmp1 = g_new0 (guint32, _tmp0 = gee_map_get_size ((GeeMap*) self->priv->recordings)), ids = (g_free (ids), NULL), ids_length1 = _tmp0, ids_size = ids_length1, _tmp1);
		i = 0;
		{
			GeeSet* _tmp2;
			GeeIterator* _tmp3;
			GeeIterator* _key_it;
			_tmp2 = NULL;
			_tmp3 = NULL;
			_key_it = (_tmp3 = gee_iterable_iterator ((GeeIterable*) (_tmp2 = gee_map_get_keys ((GeeMap*) self->priv->recordings))), (_tmp2 == NULL) ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL)), _tmp3);
			while (gee_iterator_next (_key_it)) {
				guint32 key;
				key = GPOINTER_TO_UINT (gee_iterator_get (_key_it));
				ids[i] = key;
				i++;
			}
			(_key_it == NULL) ? NULL : (_key_it = (g_object_unref (_key_it), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_recordings);
	_tmp4 = NULL;
	return (_tmp4 = ids, *result_length1 = ids_length1, _tmp4);
}


/**
         * @rec_id: The id of the recording
         * @returns: The location of the recording on the filesystem
         */
static char* dvb_recordings_store_real_GetLocation (DVBIDBusRecordingsStore* base, guint32 rec_id) {
	DVBRecordingsStore * self;
	char* val;
	self = (DVBRecordingsStore*) base;
	val = g_strdup ("");
	g_static_rec_mutex_lock (&self->priv->__lock_recordings);
	{
		if (gee_map_contains ((GeeMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id))) {
			char* _tmp1;
			DVBRecording* _tmp0;
			_tmp1 = NULL;
			_tmp0 = NULL;
			val = (_tmp1 = g_file_get_uri (dvb_recording_get_Location (_tmp0 = (DVBRecording*) gee_map_get ((GeeMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id)))), val = (g_free (val), NULL), _tmp1);
			(_tmp0 == NULL) ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_recordings);
	return val;
}


/**
         * @rec_id: The id of the recording
         * @returns: The name of the recording (e.g. the name of
         * a TV show)
         */
static char* dvb_recordings_store_real_GetName (DVBIDBusRecordingsStore* base, guint32 rec_id) {
	DVBRecordingsStore * self;
	char* val;
	self = (DVBRecordingsStore*) base;
	val = g_strdup ("");
	g_static_rec_mutex_lock (&self->priv->__lock_recordings);
	{
		if (gee_map_contains ((GeeMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id))) {
			char* _tmp2;
			const char* _tmp1;
			DVBRecording* _tmp0;
			_tmp2 = NULL;
			_tmp1 = NULL;
			_tmp0 = NULL;
			val = (_tmp2 = (_tmp1 = dvb_recording_get_Name (_tmp0 = (DVBRecording*) gee_map_get ((GeeMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id))), (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), val = (g_free (val), NULL), _tmp2);
			(_tmp0 == NULL) ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL));
			if (val == NULL) {
				char* _tmp3;
				_tmp3 = NULL;
				val = (_tmp3 = g_strdup (""), val = (g_free (val), NULL), _tmp3);
			}
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_recordings);
	return val;
}


/**
         * @rec_id: The id of the recording
         * @returns: A short text describing the recorded item
         * (e.g. the description from EPG)
         */
static char* dvb_recordings_store_real_GetDescription (DVBIDBusRecordingsStore* base, guint32 rec_id) {
	DVBRecordingsStore * self;
	char* val;
	self = (DVBRecordingsStore*) base;
	val = g_strdup ("");
	g_static_rec_mutex_lock (&self->priv->__lock_recordings);
	{
		if (gee_map_contains ((GeeMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id))) {
			char* _tmp2;
			const char* _tmp1;
			DVBRecording* _tmp0;
			_tmp2 = NULL;
			_tmp1 = NULL;
			_tmp0 = NULL;
			val = (_tmp2 = (_tmp1 = dvb_recording_get_Description (_tmp0 = (DVBRecording*) gee_map_get ((GeeMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id))), (_tmp1 == NULL) ? NULL : g_strdup (_tmp1)), val = (g_free (val), NULL), _tmp2);
			(_tmp0 == NULL) ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL));
			if (val == NULL) {
				char* _tmp3;
				_tmp3 = NULL;
				val = (_tmp3 = g_strdup (""), val = (g_free (val), NULL), _tmp3);
			}
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_recordings);
	return val;
}


/**
         * @rec_id: The id of the recording
         * @returns: The starting time of the recording
         */
static guint* dvb_recordings_store_real_GetStartTime (DVBIDBusRecordingsStore* base, guint32 rec_id, int* result_length1) {
	DVBRecordingsStore * self;
	gint val_size;
	gint val_length1;
	guint* val;
	guint* _tmp5;
	self = (DVBRecordingsStore*) base;
	val = (val_length1 = 0, NULL);
	g_static_rec_mutex_lock (&self->priv->__lock_recordings);
	{
		if (gee_map_contains ((GeeMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id))) {
			guint* _tmp2;
			gint _tmp1;
			DVBRecording* _tmp0;
			_tmp2 = NULL;
			_tmp0 = NULL;
			val = (_tmp2 = dvb_recording_get_start (_tmp0 = (DVBRecording*) gee_map_get ((GeeMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id)), &_tmp1), val = (g_free (val), NULL), val_length1 = _tmp1, val_size = val_length1, _tmp2);
			(_tmp0 == NULL) ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL));
		} else {
			guint* _tmp4;
			guint* _tmp3;
			_tmp4 = NULL;
			_tmp3 = NULL;
			val = (_tmp4 = (_tmp3 = g_new0 (guint, 0), _tmp3), val = (g_free (val), NULL), val_length1 = 0, val_size = val_length1, _tmp4);
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_recordings);
	_tmp5 = NULL;
	return (_tmp5 = val, *result_length1 = val_length1, _tmp5);
}


/**
         * @rec_id: The id of the recording
         * @returns: Start time as UNIX timestamp
         */
static gint64 dvb_recordings_store_real_GetStartTimestamp (DVBIDBusRecordingsStore* base, guint32 rec_id) {
	DVBRecordingsStore * self;
	gint64 val;
	self = (DVBRecordingsStore*) base;
	val = (gint64) (-1);
	g_static_rec_mutex_lock (&self->priv->__lock_recordings);
	{
		if (gee_map_contains ((GeeMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id))) {
			struct tm _tmp2 = {0};
			DVBRecording* _tmp0;
			struct tm _tmp1 = {0};
			_tmp0 = NULL;
			val = (gint64) mktime ((_tmp2 = (dvb_recording_get_StartTime (_tmp0 = (DVBRecording*) gee_map_get ((GeeMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id)), &_tmp1), _tmp1), &_tmp2));
			(_tmp0 == NULL) ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_recordings);
	return val;
}


/**
         * @rec_id: The id of the recording
         * @returns: The length of the recording in seconds
         * or -1 if no recording with the given id exists
         */
static gint64 dvb_recordings_store_real_GetLength (DVBIDBusRecordingsStore* base, guint32 rec_id) {
	DVBRecordingsStore * self;
	gint64 val;
	self = (DVBRecordingsStore*) base;
	val = (gint64) (-1);
	g_static_rec_mutex_lock (&self->priv->__lock_recordings);
	{
		if (gee_map_contains ((GeeMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id))) {
			DVBRecording* _tmp0;
			_tmp0 = NULL;
			val = dvb_recording_get_Length (_tmp0 = (DVBRecording*) gee_map_get ((GeeMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id)));
			(_tmp0 == NULL) ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_recordings);
	return val;
}


/**
         * @rec_id: The id of the recording
         * @returns: TRUE on success, FALSE otherwises
         *
         * Delete the recording. This deletes all files in the directory
         * created by the Recorder
         */
static gboolean dvb_recordings_store_real_Delete (DVBIDBusRecordingsStore* base, guint32 rec_id) {
	DVBRecordingsStore * self;
	GError * inner_error;
	gboolean val;
	self = (DVBRecordingsStore*) base;
	inner_error = NULL;
	val = FALSE;
	g_static_rec_mutex_lock (&self->priv->__lock_recordings);
	{
		if (!gee_map_contains ((GeeMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id))) {
			val = FALSE;
		} else {
			DVBRecording* rec;
			g_debug ("RecordingsStore.vala:224: Deleting recording %u", (guint) rec_id);
			rec = (DVBRecording*) gee_map_get ((GeeMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id));
			{
				GFile* _tmp0;
				_tmp0 = NULL;
				dvb_utils_delete_dir_recursively (_tmp0 = g_file_get_parent (dvb_recording_get_Location (rec)), &inner_error);
				if (inner_error != NULL) {
					goto __catch15_g_error;
					goto __finally15;
				}
				(_tmp0 == NULL) ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL));
				gee_map_remove ((GeeMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id));
				val = TRUE;
			}
			goto __finally15;
			__catch15_g_error:
			{
				GError * e;
				e = inner_error;
				inner_error = NULL;
				{
					g_critical ("RecordingsStore.vala:231: Could not delete recording: %s", e->message);
					val = FALSE;
					(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
				}
			}
			__finally15:
			if (inner_error != NULL) {
				(rec == NULL) ? NULL : (rec = (g_object_unref (rec), NULL));
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
				return FALSE;
			}
			g_signal_emit_by_name ((DVBIDBusRecordingsStore*) self, "changed", rec_id, (guint) DVB_CHANGE_TYPE_DELETED);
			(rec == NULL) ? NULL : (rec = (g_object_unref (rec), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_recordings);
	return val;
}


/**
         * @rec_id: The id of the recording
         * @returns: The channel's name or an empty string if
         * rec_id doesn't exist
         */
static char* dvb_recordings_store_real_GetChannelName (DVBIDBusRecordingsStore* base, guint32 rec_id) {
	DVBRecordingsStore * self;
	char* ret;
	self = (DVBRecordingsStore*) base;
	ret = NULL;
	g_static_rec_mutex_lock (&self->priv->__lock_recordings);
	{
		if (gee_map_contains ((GeeMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id))) {
			DVBRecording* rec;
			char* _tmp1;
			const char* _tmp0;
			rec = (DVBRecording*) gee_map_get ((GeeMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id));
			_tmp1 = NULL;
			_tmp0 = NULL;
			ret = (_tmp1 = (_tmp0 = dvb_recording_get_ChannelName (rec), (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), ret = (g_free (ret), NULL), _tmp1);
			(rec == NULL) ? NULL : (rec = (g_object_unref (rec), NULL));
		} else {
			char* _tmp2;
			_tmp2 = NULL;
			ret = (_tmp2 = g_strdup (""), ret = (g_free (ret), NULL), _tmp2);
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_recordings);
	return ret;
}


/**
         * @recordingsbasedir: The directory to search
         *
         * Searches recursively in the given directory
         * for "info.rec" files, restores a new Recording
         * from that file and adds it to itsself.
         */
void dvb_recordings_store_restore_from_dir (DVBRecordingsStore* self, GFile* recordingsbasedir) {
	GError * inner_error;
	char* attrs;
	GFileInfo* info;
	GFileEnumerator* files;
	g_return_if_fail (self != NULL);
	g_return_if_fail (recordingsbasedir != NULL);
	inner_error = NULL;
	if (!g_file_query_exists (recordingsbasedir, NULL)) {
		char* _tmp0;
		_tmp0 = NULL;
		g_debug ("RecordingsStore.vala:269: Directory %s does not exist", _tmp0 = g_file_get_path (recordingsbasedir));
		_tmp0 = (g_free (_tmp0), NULL);
		return;
	}
	attrs = g_strdup_printf ("%s,%s,%s", G_FILE_ATTRIBUTE_STANDARD_TYPE, G_FILE_ATTRIBUTE_ACCESS_CAN_READ, G_FILE_ATTRIBUTE_STANDARD_NAME);
	info = NULL;
	{
		GFileInfo* _tmp1;
		GFileInfo* _tmp2;
		_tmp1 = g_file_query_info (recordingsbasedir, attrs, 0, NULL, &inner_error);
		if (inner_error != NULL) {
			goto __catch16_g_error;
			goto __finally16;
		}
		_tmp2 = NULL;
		info = (_tmp2 = _tmp1, (info == NULL) ? NULL : (info = (g_object_unref (info), NULL)), _tmp2);
	}
	goto __finally16;
	__catch16_g_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			g_critical ("RecordingsStore.vala:279: Could not retrieve attributes: %s", e->message);
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			attrs = (g_free (attrs), NULL);
			(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
			return;
		}
	}
	__finally16:
	if (inner_error != NULL) {
		attrs = (g_free (attrs), NULL);
		(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return;
	}
	if (g_file_info_get_file_type (info) != G_FILE_TYPE_DIRECTORY) {
		char* _tmp3;
		_tmp3 = NULL;
		g_critical ("RecordingsStore.vala:284: %s is not a directory", _tmp3 = g_file_get_path (recordingsbasedir));
		_tmp3 = (g_free (_tmp3), NULL);
		attrs = (g_free (attrs), NULL);
		(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
		return;
	}
	if (!g_file_info_get_attribute_boolean (info, G_FILE_ATTRIBUTE_ACCESS_CAN_READ)) {
		char* _tmp4;
		_tmp4 = NULL;
		g_critical ("RecordingsStore.vala:289: Cannot read %s", _tmp4 = g_file_get_path (recordingsbasedir));
		_tmp4 = (g_free (_tmp4), NULL);
		attrs = (g_free (attrs), NULL);
		(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
		return;
	}
	files = NULL;
	{
		GFileEnumerator* _tmp5;
		GFileEnumerator* _tmp6;
		_tmp5 = g_file_enumerate_children (recordingsbasedir, attrs, 0, NULL, &inner_error);
		if (inner_error != NULL) {
			goto __catch17_g_error;
			goto __finally17;
		}
		_tmp6 = NULL;
		files = (_tmp6 = _tmp5, (files == NULL) ? NULL : (files = (g_object_unref (files), NULL)), _tmp6);
	}
	goto __finally17;
	__catch17_g_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			g_critical ("RecordingsStore.vala:298: Could not read directory: %s", e->message);
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			attrs = (g_free (attrs), NULL);
			(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
			(files == NULL) ? NULL : (files = (g_object_unref (files), NULL));
			return;
		}
	}
	__finally17:
	if (inner_error != NULL) {
		attrs = (g_free (attrs), NULL);
		(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
		(files == NULL) ? NULL : (files = (g_object_unref (files), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return;
	}
	{
		GFileInfo* childinfo;
		childinfo = NULL;
		while (TRUE) {
			GFileInfo* _tmp7;
			GFileInfo* _tmp8;
			guint32 type;
			GFile* child;
			_tmp7 = g_file_enumerator_next_file (files, NULL, &inner_error);
			if (inner_error != NULL) {
				(childinfo == NULL) ? NULL : (childinfo = (g_object_unref (childinfo), NULL));
				goto __catch18_g_error;
				goto __finally18;
			}
			_tmp8 = NULL;
			if (!((childinfo = (_tmp8 = _tmp7, (childinfo == NULL) ? NULL : (childinfo = (g_object_unref (childinfo), NULL)), _tmp8)) != NULL)) {
				break;
			}
			type = g_file_info_get_attribute_uint32 (childinfo, G_FILE_ATTRIBUTE_STANDARD_TYPE);
			child = g_file_get_child (recordingsbasedir, g_file_info_get_name (childinfo));
			switch (type) {
				case G_FILE_TYPE_DIRECTORY:
				{
					dvb_recordings_store_restore_from_dir (self, child);
					break;
				}
				case G_FILE_TYPE_REGULAR:
				{
					if (_vala_strcmp0 (g_file_info_get_name (childinfo), "info.rec") == 0) {
						DVBRecording* rec;
						rec = NULL;
						{
							DVBRecording* _tmp9;
							DVBRecording* _tmp10;
							_tmp9 = dvb_recording_deserialize (child, &inner_error);
							if (inner_error != NULL) {
								goto __catch20_g_error;
								goto __finally20;
							}
							_tmp10 = NULL;
							rec = (_tmp10 = _tmp9, (rec == NULL) ? NULL : (rec = (g_object_unref (rec), NULL)), _tmp10);
						}
						goto __finally20;
						__catch20_g_error:
						{
							GError * e;
							e = inner_error;
							inner_error = NULL;
							{
								g_critical ("RecordingsStore.vala:322: Could not deserialize recording: %s", e->message);
								(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
							}
						}
						__finally20:
						if (inner_error != NULL) {
							(rec == NULL) ? NULL : (rec = (g_object_unref (rec), NULL));
							(child == NULL) ? NULL : (child = (g_object_unref (child), NULL));
							(childinfo == NULL) ? NULL : (childinfo = (g_object_unref (childinfo), NULL));
							goto __catch18_g_error;
							goto __finally18;
						}
						if (rec != NULL) {
							char* _tmp11;
							_tmp11 = NULL;
							g_debug ("RecordingsStore.vala:327: Restored recording from %s", _tmp11 = g_file_get_path (child));
							_tmp11 = (g_free (_tmp11), NULL);
							g_static_rec_mutex_lock (&self->priv->__lock_recordings);
							{
								dvb_recordings_store_add (self, rec);
							}
							g_static_rec_mutex_unlock (&self->priv->__lock_recordings);
							g_static_rec_mutex_lock (&self->priv->__lock_last_id);
							{
								if (dvb_recording_get_Id (rec) > self->priv->last_id) {
									self->priv->last_id = dvb_recording_get_Id (rec);
								}
							}
							g_static_rec_mutex_unlock (&self->priv->__lock_last_id);
						}
						(rec == NULL) ? NULL : (rec = (g_object_unref (rec), NULL));
					}
					break;
				}
			}
			(child == NULL) ? NULL : (child = (g_object_unref (child), NULL));
		}
		(childinfo == NULL) ? NULL : (childinfo = (g_object_unref (childinfo), NULL));
	}
	goto __finally18;
	__catch18_g_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			g_critical ("RecordingsStore.vala:344: %s", e->message);
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally18:
	{
		{
			g_file_enumerator_close (files, NULL, &inner_error);
			if (inner_error != NULL) {
				goto __catch21_g_error;
				goto __finally21;
			}
		}
		goto __finally21;
		__catch21_g_error:
		{
			GError * e;
			e = inner_error;
			inner_error = NULL;
			{
				g_critical ("RecordingsStore.vala:349: Could not close file: %s", e->message);
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			}
		}
		__finally21:
		if (inner_error != NULL) {
			attrs = (g_free (attrs), NULL);
			(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
			(files == NULL) ? NULL : (files = (g_object_unref (files), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return;
		}
	}
	if (inner_error != NULL) {
		attrs = (g_free (attrs), NULL);
		(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
		(files == NULL) ? NULL : (files = (g_object_unref (files), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return;
	}
	attrs = (g_free (attrs), NULL);
	(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
	(files == NULL) ? NULL : (files = (g_object_unref (files), NULL));
}


/**
         * @location: Path to the .ts file of the recording
         * @returns: TRUE on success
         *
         * Delete a recording by the path of the recording
         */
static gboolean dvb_recordings_store_delete_recording_by_location (DVBRecordingsStore* self, const char* location) {
	guint32 rec_id;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (location != NULL, FALSE);
	rec_id = (guint32) 0;
	{
		GeeSet* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* _id_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		_id_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = gee_map_get_keys ((GeeMap*) self->priv->recordings))), (_tmp0 == NULL) ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (_id_it)) {
			guint32 id;
			DVBRecording* rec;
			char* _tmp2;
			gboolean _tmp3;
			id = GPOINTER_TO_UINT (gee_iterator_get (_id_it));
			rec = (DVBRecording*) gee_map_get ((GeeMap*) self->priv->recordings, GUINT_TO_POINTER (id));
			_tmp2 = NULL;
			if ((_tmp3 = _vala_strcmp0 (_tmp2 = g_file_get_path (dvb_recording_get_Location (rec)), location) == 0, _tmp2 = (g_free (_tmp2), NULL), _tmp3)) {
				rec_id = id;
				(rec == NULL) ? NULL : (rec = (g_object_unref (rec), NULL));
				break;
			}
			(rec == NULL) ? NULL : (rec = (g_object_unref (rec), NULL));
		}
		(_id_it == NULL) ? NULL : (_id_it = (g_object_unref (_id_it), NULL));
	}
	if (rec_id != 0) {
		g_debug ("RecordingsStore.vala:371: Deleting recording %u", (guint) rec_id);
		gee_map_remove ((GeeMap*) self->priv->recordings, GUINT_TO_POINTER (rec_id));
		g_signal_emit_by_name ((DVBIDBusRecordingsStore*) self, "changed", rec_id, (guint) DVB_CHANGE_TYPE_DELETED);
		return TRUE;
	}
	return FALSE;
}


static void dvb_recordings_store_on_recording_file_changed (DVBRecordingsStore* self, GFileMonitor* monitor, GFile* file, GFile* other_file, GFileMonitorEvent event) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (monitor != NULL);
	g_return_if_fail (file != NULL);
	if (event == G_FILE_MONITOR_EVENT_DELETED) {
		char* location;
		location = g_file_get_path (file);
		g_debug ("RecordingsStore.vala:384: %s has been deleted", location);
		dvb_recordings_store_delete_recording_by_location (self, location);
		g_file_monitor_cancel (monitor);
		location = (g_free (location), NULL);
	}
}


/**
     * This class manages the recordings off all devices
     */
DVBRecordingsStore* dvb_recordings_store_construct (GType object_type) {
	DVBRecordingsStore * self;
	self = g_object_newv (object_type, 0, NULL);
	return self;
}


DVBRecordingsStore* dvb_recordings_store_new (void) {
	return dvb_recordings_store_construct (DVB_TYPE_RECORDINGS_STORE);
}


static GObject * dvb_recordings_store_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	DVBRecordingsStoreClass * klass;
	GObjectClass * parent_class;
	DVBRecordingsStore * self;
	klass = DVB_RECORDINGS_STORE_CLASS (g_type_class_peek (DVB_TYPE_RECORDINGS_STORE));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = DVB_RECORDINGS_STORE (obj);
	{
		GeeHashMap* _tmp1;
		_tmp1 = NULL;
		self->priv->recordings = (_tmp1 = gee_hash_map_new (G_TYPE_UINT, NULL, NULL, DVB_TYPE_RECORDING, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_hash, g_direct_equal, g_direct_equal), (self->priv->recordings == NULL) ? NULL : (self->priv->recordings = (g_object_unref (self->priv->recordings), NULL)), _tmp1);
		self->priv->last_id = (guint32) 0;
	}
	return obj;
}


void _dvb_recordings_store_dbus_unregister (DBusConnection* connection, void* user_data) {
}


static DBusMessage* _dbus_dvb_recordings_store_introspect (DVBRecordingsStore* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessage* reply;
	DBusMessageIter iter;
	GString* xml_data;
	char** children;
	int i;
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &iter);
	xml_data = g_string_new ("<!DOCTYPE node PUBLIC \"-//freedesktop//DTD D-BUS Object Introspection 1.0//EN\" \"http://www.freedesktop.org/standards/dbus/1.0/introspect.dtd\">\n");
	g_string_append (xml_data, "<node>\n<interface name=\"org.freedesktop.DBus.Introspectable\">\n  <method name=\"Introspect\">\n    <arg name=\"data\" direction=\"out\" type=\"s\"/>\n  </method>\n</interface>\n<interface name=\"org.freedesktop.DBus.Properties\">\n  <method name=\"Get\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"propname\" direction=\"in\" type=\"s\"/>\n    <arg name=\"value\" direction=\"out\" type=\"v\"/>\n  </method>\n  <method name=\"Set\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"propname\" direction=\"in\" type=\"s\"/>\n    <arg name=\"value\" direction=\"in\" type=\"v\"/>\n  </method>\n  <method name=\"GetAll\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"props\" direction=\"out\" type=\"a{sv}\"/>\n  </method>\n</interface>\n<interface name=\"org.gnome.DVB.RecordingsStore\">\n  <method name=\"GetRecordings\">\n    <arg name=\"result\" type=\"au\" direction=\"out\"/>\n  </method>\n  <method name=\"GetLocation\">\n    <arg name=\"rec_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <method name=\"GetName\">\n    <arg name=\"rec_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <method name=\"GetDescription\">\n    <arg name=\"rec_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <method name=\"GetStartTime\">\n    <arg name=\"rec_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"au\" direction=\"out\"/>\n  </method>\n  <method name=\"GetStartTimestamp\">\n    <arg name=\"rec_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"x\" direction=\"out\"/>\n  </method>\n  <method name=\"GetLength\">\n    <arg name=\"rec_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"x\" direction=\"out\"/>\n  </method>\n  <method name=\"Delete\">\n    <arg name=\"rec_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"b\" direction=\"out\"/>\n  </method>\n  <method name=\"GetChannelName\">\n    <arg name=\"rec_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <signal name=\"Changed\">\n    <arg name=\"rec_id\" type=\"u\"/>\n    <arg name=\"type\" type=\"u\"/>\n  </signal>\n</interface>\n");
	dbus_connection_list_registered (connection, g_object_get_data ((GObject *) self, "dbus_object_path"), &children);
	for (i = 0; children[i]; i++) {
		g_string_append_printf (xml_data, "<node name=\"%s\"/>\n", children[i]);
	}
	g_string_append (xml_data, "</node>\n");
	dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &xml_data->str);
	g_string_free (xml_data, TRUE);
	return reply;
}


DBusHandlerResult dvb_recordings_store_dbus_message (DBusConnection* connection, DBusMessage* message, void* object) {
	DBusMessage* reply;
	reply = NULL;
	if (dbus_message_is_method_call (message, "org.freedesktop.DBus.Introspectable", "Introspect")) {
		reply = _dbus_dvb_recordings_store_introspect (object, connection, message);
	}
	if (reply) {
		dbus_connection_send (connection, reply, NULL);
		dbus_message_unref (reply);
		return DBUS_HANDLER_RESULT_HANDLED;
	} else if (dvb_id_bus_recordings_store_dbus_message (connection, message, object) == DBUS_HANDLER_RESULT_HANDLED) {
		return DBUS_HANDLER_RESULT_HANDLED;
	} else {
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}
}


void dvb_recordings_store_dbus_register_object (DBusConnection* connection, const char* path, void* object) {
	if (!g_object_get_data (object, "dbus_object_path")) {
		g_object_set_data (object, "dbus_object_path", g_strdup (path));
		dbus_connection_register_object_path (connection, path, &_dvb_recordings_store_dbus_path_vtable, object);
		g_object_weak_ref (object, _vala_dbus_unregister_object, connection);
	}
	dvb_id_bus_recordings_store_dbus_register_object (connection, path, object);
}


static void dvb_recordings_store_class_init (DVBRecordingsStoreClass * klass) {
	GStaticRecMutex _tmp0 = {0};
	dvb_recordings_store_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (DVBRecordingsStorePrivate));
	G_OBJECT_CLASS (klass)->constructor = dvb_recordings_store_constructor;
	G_OBJECT_CLASS (klass)->finalize = dvb_recordings_store_finalize;
	g_type_set_qdata (DVB_TYPE_RECORDINGS_STORE, g_quark_from_static_string ("DBusObjectVTable"), (void*) (&_dvb_recordings_store_dbus_vtable));
	dvb_recordings_store_instance_mutex = (g_static_rec_mutex_init (&_tmp0), _tmp0);
}


static void dvb_recordings_store_dvb_id_bus_recordings_store_interface_init (DVBIDBusRecordingsStoreIface * iface) {
	dvb_recordings_store_dvb_id_bus_recordings_store_parent_iface = g_type_interface_peek_parent (iface);
	iface->GetRecordings = dvb_recordings_store_real_GetRecordings;
	iface->GetLocation = dvb_recordings_store_real_GetLocation;
	iface->GetName = dvb_recordings_store_real_GetName;
	iface->GetDescription = dvb_recordings_store_real_GetDescription;
	iface->GetStartTime = dvb_recordings_store_real_GetStartTime;
	iface->GetStartTimestamp = dvb_recordings_store_real_GetStartTimestamp;
	iface->GetLength = dvb_recordings_store_real_GetLength;
	iface->Delete = dvb_recordings_store_real_Delete;
	iface->GetChannelName = dvb_recordings_store_real_GetChannelName;
}


static void dvb_recordings_store_instance_init (DVBRecordingsStore * self) {
	self->priv = DVB_RECORDINGS_STORE_GET_PRIVATE (self);
	g_static_rec_mutex_init (&self->priv->__lock_recordings);
	g_static_rec_mutex_init (&self->priv->__lock_last_id);
}


static void dvb_recordings_store_finalize (GObject* obj) {
	DVBRecordingsStore * self;
	self = DVB_RECORDINGS_STORE (obj);
	g_static_rec_mutex_free (&self->priv->__lock_recordings);
	(self->priv->recordings == NULL) ? NULL : (self->priv->recordings = (g_object_unref (self->priv->recordings), NULL));
	g_static_rec_mutex_free (&self->priv->__lock_last_id);
	G_OBJECT_CLASS (dvb_recordings_store_parent_class)->finalize (obj);
}


GType dvb_recordings_store_get_type (void) {
	static GType dvb_recordings_store_type_id = 0;
	if (dvb_recordings_store_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (DVBRecordingsStoreClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_recordings_store_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBRecordingsStore), 0, (GInstanceInitFunc) dvb_recordings_store_instance_init, NULL };
		static const GInterfaceInfo dvb_id_bus_recordings_store_info = { (GInterfaceInitFunc) dvb_recordings_store_dvb_id_bus_recordings_store_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		dvb_recordings_store_type_id = g_type_register_static (G_TYPE_OBJECT, "DVBRecordingsStore", &g_define_type_info, 0);
		g_type_add_interface_static (dvb_recordings_store_type_id, DVB_TYPE_ID_BUS_RECORDINGS_STORE, &dvb_id_bus_recordings_store_info);
	}
	return dvb_recordings_store_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}


static void _vala_dbus_register_object (DBusConnection* connection, const char* path, void* object) {
	const _DBusObjectVTable * vtable;
	vtable = g_type_get_qdata (G_TYPE_FROM_INSTANCE (object), g_quark_from_static_string ("DBusObjectVTable"));
	if (vtable) {
		vtable->register_object (connection, path, object);
	} else {
		g_warning ("Object does not implement any D-Bus interface");
	}
}


static void _vala_dbus_unregister_object (gpointer connection, GObject* object) {
	char* path;
	path = g_object_steal_data ((GObject*) object, "dbus_object_path");
	dbus_connection_unregister_object_path (connection, path);
	g_free (path);
}




