/*
 * Copyright (C) 2008,2009 Sebastian Pölsterl
 *
 * This file is part of GNOME DVB Daemon.
 *
 * GNOME DVB Daemon is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GNOME DVB Daemon is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNOME DVB Daemon.  If not, see <http://www.gnu.org/licenses/>.
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))

#include <src/EPGScanner.h>
#include <stdlib.h>
#include <string.h>
#include <gee/hashset.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <gee/collection.h>
#include "src/Factory.h"
#include "src/Settings.h"
#include "src/Channel.h"
#include "src/Device.h"
#include "src/ChannelList.h"
#include "src/Event.h"
#include "src/Schedule.h"




struct _DVBEPGScannerPrivate {
	DVBDeviceGroup* _DeviceGroup;
	GStaticRecMutex __lock_pipeline;
	GstElement* pipeline;
	GQueue* channels;
	guint scan_event_id;
	guint queue_scan_event_id;
	gboolean do_stop;
};

#define DVB_EPG_SCANNER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), DVB_TYPE_EPG_SCANNER, DVBEPGScannerPrivate))
enum  {
	DVB_EPG_SCANNER_DUMMY_PROPERTY,
	DVB_EPG_SCANNER_DEVICE_GROUP
};
static gint dvb_epg_scanner_CHECK_EIT_INTERVAL = 0;
#define DVB_EPG_SCANNER_WAIT_FOR_EIT_DURATION 10
#define DVB_EPG_SCANNER_PIPELINE_TEMPLATE "dvbsrc name=dvbsrc adapter=%u frontend=%u pids=0:16:17:18 stats-reporting-interval=0 ! mpegtsparse ! fakesink silent=true"
static void dvb_epg_scanner_remove_timeouts (DVBEPGScanner* self);
static void dvb_epg_scanner_reset (DVBEPGScanner* self);
static void _dvb_epg_scanner_bus_watch_func_gst_bus_message (GstBus* _sender, GstMessage* message, gpointer self);
static gboolean _dvb_epg_scanner_scan_new_frequency_gsource_func (gpointer self);
static gboolean _dvb_epg_scanner_start_gsource_func (gpointer self);
static gboolean dvb_epg_scanner_scan_new_frequency (DVBEPGScanner* self);
static void dvb_epg_scanner_bus_watch_func (DVBEPGScanner* self, GstBus* bus, GstMessage* message);
static guint dvb_epg_scanner_get_uint_val (const GstStructure* structure, const char* name);
static GObject * dvb_epg_scanner_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static gpointer dvb_epg_scanner_parent_class = NULL;
static void dvb_epg_scanner_finalize (GObject* obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
         * @device: The device where EPG should be collected from
         */
DVBEPGScanner* dvb_epg_scanner_construct (GType object_type, DVBDeviceGroup* device) {
	GError * inner_error;
	DVBEPGScanner * self;
	g_return_val_if_fail (device != NULL, NULL);
	inner_error = NULL;
	self = g_object_newv (object_type, 0, NULL);
	dvb_epg_scanner_set_DeviceGroup (self, device);
	if (dvb_epg_scanner_CHECK_EIT_INTERVAL == (-1)) {
		DVBSettings* _tmp0;
		DVBSettings* settings;
		_tmp0 = NULL;
		settings = (_tmp0 = dvb_factory_get_settings (), (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0));
		{
			gint _tmp1;
			_tmp1 = dvb_settings_get_integer (settings, DVB_SETTINGS_EPG_SECTION, DVB_SETTINGS_SCAN_INTERVAL, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == G_KEY_FILE_ERROR) {
					goto __catch3_g_key_file_error;
				}
				goto __finally3;
			}
			dvb_epg_scanner_CHECK_EIT_INTERVAL = _tmp1 * 60;
		}
		goto __finally3;
		__catch3_g_key_file_error:
		{
			GError * e;
			e = inner_error;
			inner_error = NULL;
			{
				g_critical ("EPGScanner.vala:62: %s", e->message);
				dvb_epg_scanner_CHECK_EIT_INTERVAL = 15 * 60;
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			}
		}
		__finally3:
		if (inner_error != NULL) {
			(settings == NULL) ? NULL : (settings = (g_object_unref (settings), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
		(settings == NULL) ? NULL : (settings = (g_object_unref (settings), NULL));
	}
	return self;
}


DVBEPGScanner* dvb_epg_scanner_new (DVBDeviceGroup* device) {
	return dvb_epg_scanner_construct (DVB_TYPE_EPG_SCANNER, device);
}


/**
         * Stop collecting EPG data
         */
void dvb_epg_scanner_stop (DVBEPGScanner* self) {
	g_return_if_fail (self != NULL);
	g_debug ("EPGScanner.vala:72: Stopping EPG scan for group %u", dvb_device_group_get_Id (self->priv->_DeviceGroup));
	dvb_epg_scanner_remove_timeouts (self);
	dvb_epg_scanner_reset (self);
}


static void dvb_epg_scanner_remove_timeouts (DVBEPGScanner* self) {
	g_return_if_fail (self != NULL);
	/* Remove timed scans */
	if (self->priv->scan_event_id != 0) {
		g_source_remove (self->priv->scan_event_id);
		self->priv->scan_event_id = (guint) 0;
	}
	if (self->priv->queue_scan_event_id != 0) {
		g_source_remove (self->priv->queue_scan_event_id);
		self->priv->queue_scan_event_id = (guint) 0;
	}
}


void dvb_epg_scanner_destroy (DVBEPGScanner* self) {
	g_return_if_fail (self != NULL);
	dvb_epg_scanner_remove_timeouts (self);
	/* Don't call reset directly here
	             or we get in a in-consistent state */
	self->priv->do_stop = TRUE;
}


static void dvb_epg_scanner_reset (DVBEPGScanner* self) {
	DVBChannel* c;
	DVBChannel* _tmp1;
	g_return_if_fail (self != NULL);
	g_static_rec_mutex_lock (&self->priv->__lock_pipeline);
	{
		if (self->priv->pipeline != NULL) {
			GstBus* bus;
			GstElement* _tmp0;
			bus = gst_element_get_bus (self->priv->pipeline);
			gst_bus_remove_signal_watch (bus);
			gst_element_set_state (self->priv->pipeline, GST_STATE_NULL);
			_tmp0 = NULL;
			self->priv->pipeline = (_tmp0 = NULL, (self->priv->pipeline == NULL) ? NULL : (self->priv->pipeline = (gst_object_unref (self->priv->pipeline), NULL)), _tmp0);
			(bus == NULL) ? NULL : (bus = (gst_object_unref (bus), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_pipeline);
	/* clear doesn't unref for us so we do this instead*/
	c = NULL;
	_tmp1 = NULL;
	while ((c = (_tmp1 = (DVBChannel*) g_queue_pop_head (self->priv->channels), (c == NULL) ? NULL : (c = (g_object_unref (c), NULL)), _tmp1)) != NULL) {
	}
	/* Vala unref's Channel instances for us*/
	g_queue_clear (self->priv->channels);
	(c == NULL) ? NULL : (c = (g_object_unref (c), NULL));
}


static void _dvb_epg_scanner_bus_watch_func_gst_bus_message (GstBus* _sender, GstMessage* message, gpointer self) {
	dvb_epg_scanner_bus_watch_func (self, _sender, message);
}


static gboolean _dvb_epg_scanner_scan_new_frequency_gsource_func (gpointer self) {
	return dvb_epg_scanner_scan_new_frequency (self);
}


/**
         * Start collection EPG data for all channels
         */
gboolean dvb_epg_scanner_start (DVBEPGScanner* self) {
	GError * inner_error;
	GeeHashSet* unique_frequencies;
	DVBDevice* device;
	gboolean _tmp7;
	g_return_val_if_fail (self != NULL, FALSE);
	inner_error = NULL;
	g_debug ("EPGScanner.vala:120: Starting EPG scan for group %u", dvb_device_group_get_Id (self->priv->_DeviceGroup));
	/* TODO scan all channels?*/
	unique_frequencies = gee_hash_set_new (G_TYPE_UINT, NULL, NULL, g_direct_hash, g_direct_equal);
	{
		GeeIterator* _c_it;
		_c_it = gee_iterable_iterator ((GeeIterable*) dvb_device_group_get_Channels (self->priv->_DeviceGroup));
		while (gee_iterator_next (_c_it)) {
			DVBChannel* c;
			guint freq;
			c = (DVBChannel*) gee_iterator_get (_c_it);
			freq = dvb_channel_get_Frequency (c);
			if (!gee_collection_contains ((GeeCollection*) unique_frequencies, GUINT_TO_POINTER (freq))) {
				DVBChannel* _tmp0;
				gee_collection_contains ((GeeCollection*) unique_frequencies, GUINT_TO_POINTER (freq));
				_tmp0 = NULL;
				g_queue_push_tail (self->priv->channels, (_tmp0 = c, (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0)));
			}
			(c == NULL) ? NULL : (c = (g_object_unref (c), NULL));
		}
		(_c_it == NULL) ? NULL : (_c_it = (g_object_unref (_c_it), NULL));
	}
	device = dvb_device_group_get_next_free_device (self->priv->_DeviceGroup);
	if (device == NULL) {
		gboolean _tmp1;
		return (_tmp1 = FALSE, (unique_frequencies == NULL) ? NULL : (unique_frequencies = (g_object_unref (unique_frequencies), NULL)), (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), _tmp1);
	}
	g_static_rec_mutex_lock (&self->priv->__lock_pipeline);
	{
		GstBus* bus;
		{
			char* _tmp2;
			GstElement* _tmp3;
			GstElement* _tmp4;
			GstElement* _tmp5;
			_tmp2 = NULL;
			_tmp3 = NULL;
			_tmp4 = (_tmp3 = gst_parse_launch (_tmp2 = g_strdup_printf (DVB_EPG_SCANNER_PIPELINE_TEMPLATE, dvb_device_get_Adapter (device), dvb_device_get_Frontend (device)), &inner_error), _tmp2 = (g_free (_tmp2), NULL), _tmp3);
			if (inner_error != NULL) {
				goto __catch4_g_error;
				goto __finally4;
			}
			_tmp5 = NULL;
			self->priv->pipeline = (_tmp5 = _tmp4, (self->priv->pipeline == NULL) ? NULL : (self->priv->pipeline = (gst_object_unref (self->priv->pipeline), NULL)), _tmp5);
		}
		goto __finally4;
		__catch4_g_error:
		{
			GError * e;
			e = inner_error;
			inner_error = NULL;
			{
				gboolean _tmp6;
				g_error ("EPGScanner.vala:140: Could not create pipeline: %s", e->message);
				return (_tmp6 = FALSE, (e == NULL) ? NULL : (e = (g_error_free (e), NULL)), (unique_frequencies == NULL) ? NULL : (unique_frequencies = (g_object_unref (unique_frequencies), NULL)), (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), _tmp6);
			}
		}
		__finally4:
		if (inner_error != NULL) {
			(unique_frequencies == NULL) ? NULL : (unique_frequencies = (g_object_unref (unique_frequencies), NULL));
			(device == NULL) ? NULL : (device = (g_object_unref (device), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return FALSE;
		}
		bus = gst_element_get_bus (self->priv->pipeline);
		gst_bus_add_signal_watch (bus);
		g_signal_connect_object (bus, "message", (GCallback) _dvb_epg_scanner_bus_watch_func_gst_bus_message, self, 0);
		(bus == NULL) ? NULL : (bus = (gst_object_unref (bus), NULL));
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_pipeline);
	self->priv->scan_event_id = g_timeout_add_seconds ((guint) DVB_EPG_SCANNER_WAIT_FOR_EIT_DURATION, _dvb_epg_scanner_scan_new_frequency_gsource_func, self);
	return (_tmp7 = FALSE, (unique_frequencies == NULL) ? NULL : (unique_frequencies = (g_object_unref (unique_frequencies), NULL)), (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), _tmp7);
}


static gboolean _dvb_epg_scanner_start_gsource_func (gpointer self) {
	return dvb_epg_scanner_start (self);
}


/**
         * Scan the next frequency for EPG data
         */
static gboolean dvb_epg_scanner_scan_new_frequency (DVBEPGScanner* self) {
	gboolean _tmp0;
	DVBChannel* channel;
	gboolean _tmp2;
	g_return_val_if_fail (self != NULL, FALSE);
	_tmp0 = FALSE;
	if (g_queue_is_empty (self->priv->channels)) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = self->priv->do_stop;
	}
	if (_tmp0) {
		g_debug ("EPGScanner.vala:160: Finished EPG scan for group %u", dvb_device_group_get_Id (self->priv->_DeviceGroup));
		dvb_epg_scanner_reset (self);
		/* Time the next iteration*/
		self->priv->queue_scan_event_id = g_timeout_add_seconds ((guint) dvb_epg_scanner_CHECK_EIT_INTERVAL, _dvb_epg_scanner_start_gsource_func, self);
		return FALSE;
	}
	channel = (DVBChannel*) g_queue_pop_head (self->priv->channels);
	dvb_schedule_remove_expired_events (dvb_channel_get_Schedule (channel));
	g_static_rec_mutex_lock (&self->priv->__lock_pipeline);
	{
		GstElement* dvbsrc;
		gst_element_set_state (self->priv->pipeline, GST_STATE_READY);
		dvbsrc = gst_bin_get_by_name (GST_BIN (self->priv->pipeline), "dvbsrc");
		dvb_channel_setup_dvb_source (channel, dvbsrc);
		gst_element_set_state (self->priv->pipeline, GST_STATE_PLAYING);
		(dvbsrc == NULL) ? NULL : (dvbsrc = (gst_object_unref (dvbsrc), NULL));
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_pipeline);
	return (_tmp2 = TRUE, (channel == NULL) ? NULL : (channel = (g_object_unref (channel), NULL)), _tmp2);
}


static void dvb_epg_scanner_bus_watch_func (DVBEPGScanner* self, GstBus* bus, GstMessage* message) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (bus != NULL);
	g_return_if_fail (message != NULL);
	switch (message->type) {
		case GST_MESSAGE_ELEMENT:
		{
			if (_vala_strcmp0 (gst_structure_get_name (message->structure), "dvb-read-failure") == 0) {
				g_critical ("EPGScanner.vala:189: Could not read from DVB device");
				dvb_epg_scanner_stop (self);
			} else {
				if (_vala_strcmp0 (gst_structure_get_name (message->structure), "eit") == 0) {
					dvb_epg_scanner_on_eit_structure (self, message->structure);
				}
			}
			break;
		}
		case GST_MESSAGE_ERROR:
		{
			GError* gerror;
			char* debug;
			char* _tmp3;
			char* _tmp2;
			GError* _tmp1;
			GError* _tmp0;
			gerror = NULL;
			debug = NULL;
			_tmp3 = NULL;
			_tmp2 = NULL;
			_tmp1 = NULL;
			_tmp0 = NULL;
			(gst_message_parse_error (message, &_tmp0, &_tmp2), gerror = (_tmp1 = _tmp0, (gerror == NULL) ? NULL : (gerror = (g_error_free (gerror), NULL)), _tmp1));
			debug = (_tmp3 = _tmp2, debug = (g_free (debug), NULL), _tmp3);
			g_critical ("EPGScanner.vala:200: %s %s", gerror->message, debug);
			dvb_epg_scanner_stop (self);
			(gerror == NULL) ? NULL : (gerror = (g_error_free (gerror), NULL));
			debug = (g_free (debug), NULL);
			break;
		}
		default:
		{
			break;
		}
	}
}


void dvb_epg_scanner_on_eit_structure (DVBEPGScanner* self, const GstStructure* structure) {
	GValue events;
	guint size;
	GValue val = {0};
	const GstStructure* event;
	g_return_if_fail (self != NULL);
	g_return_if_fail (structure != NULL);
	events = *gst_structure_get_value (structure, "events");
	if (!G_VALUE_HOLDS (&events, gst_value_list_get_type ())) {
		return;
	}
	size = gst_value_list_get_size (&events);
	event = NULL;
	{
		guint i;
		/* Iterate over events*/
		i = (guint) 0;
		for (; i < size; i++) {
			guint sid;
			DVBChannel* channel;
			guint event_id;
			DVBEvent* event_class;
			const char* _tmp0;
			char* name;
			gboolean _tmp1;
			const char* _tmp4;
			char* desc;
			gboolean _tmp5;
			const char* _tmp8;
			char* ext_desc;
			gboolean _tmp9;
			gboolean free_ca;
			GValue components;
			guint components_len;
			GValue comp_val = {0};
			const GstStructure* component;
			val = *gst_value_list_get_value (&events, i);
			event = gst_value_get_structure (&val);
			sid = dvb_epg_scanner_get_uint_val (structure, "service-id");
			channel = dvb_channel_list_get_channel (dvb_device_group_get_Channels (self->priv->_DeviceGroup), sid);
			if (channel == NULL) {
				g_warning ("EPGScanner.vala:226: Could not find channel %u for this device", sid);
				(channel == NULL) ? NULL : (channel = (g_object_unref (channel), NULL));
				return;
			}
			event_id = dvb_epg_scanner_get_uint_val (event, "event-id");
			event_class = dvb_event_new ();
			event_class->id = event_id;
			event_class->year = dvb_epg_scanner_get_uint_val (event, "year");
			event_class->month = dvb_epg_scanner_get_uint_val (event, "month");
			event_class->day = dvb_epg_scanner_get_uint_val (event, "day");
			event_class->hour = dvb_epg_scanner_get_uint_val (event, "hour");
			event_class->minute = dvb_epg_scanner_get_uint_val (event, "minute");
			event_class->second = dvb_epg_scanner_get_uint_val (event, "second");
			event_class->duration = dvb_epg_scanner_get_uint_val (event, "duration");
			event_class->running_status = dvb_epg_scanner_get_uint_val (event, "running-status");
			_tmp0 = NULL;
			name = (_tmp0 = gst_structure_get_string (event, "name"), (_tmp0 == NULL) ? NULL : g_strdup (_tmp0));
			_tmp1 = FALSE;
			if (name != NULL) {
				_tmp1 = g_utf8_validate (name, -1, NULL);
			} else {
				_tmp1 = FALSE;
			}
			if (_tmp1) {
				char* _tmp3;
				const char* _tmp2;
				_tmp3 = NULL;
				_tmp2 = NULL;
				event_class->name = (_tmp3 = (_tmp2 = name, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2)), event_class->name = (g_free (event_class->name), NULL), _tmp3);
			}
			_tmp4 = NULL;
			desc = (_tmp4 = gst_structure_get_string (event, "description"), (_tmp4 == NULL) ? NULL : g_strdup (_tmp4));
			_tmp5 = FALSE;
			if (desc != NULL) {
				_tmp5 = g_utf8_validate (desc, -1, NULL);
			} else {
				_tmp5 = FALSE;
			}
			if (_tmp5) {
				char* _tmp7;
				const char* _tmp6;
				_tmp7 = NULL;
				_tmp6 = NULL;
				event_class->description = (_tmp7 = (_tmp6 = desc, (_tmp6 == NULL) ? NULL : g_strdup (_tmp6)), event_class->description = (g_free (event_class->description), NULL), _tmp7);
			}
			_tmp8 = NULL;
			ext_desc = (_tmp8 = gst_structure_get_string (event, "extended-text"), (_tmp8 == NULL) ? NULL : g_strdup (_tmp8));
			_tmp9 = FALSE;
			if (ext_desc != NULL) {
				_tmp9 = g_utf8_validate (ext_desc, -1, NULL);
			} else {
				_tmp9 = FALSE;
			}
			if (_tmp9) {
				char* _tmp11;
				const char* _tmp10;
				_tmp11 = NULL;
				_tmp10 = NULL;
				event_class->extended_description = (_tmp11 = (_tmp10 = ext_desc, (_tmp10 == NULL) ? NULL : g_strdup (_tmp10)), event_class->extended_description = (g_free (event_class->extended_description), NULL), _tmp11);
			}
			free_ca = FALSE;
			gst_structure_get_boolean (event, "free-ca-mode", &free_ca);
			event_class->free_ca_mode = free_ca;
			components = *gst_structure_get_value (event, "components");
			components_len = gst_value_list_get_size (&components);
			component = NULL;
			{
				guint j;
				j = (guint) 0;
				for (; j < components_len; j++) {
					comp_val = *gst_value_list_get_value (&components, j);
					component = gst_value_get_structure (&comp_val);
					if (_vala_strcmp0 (gst_structure_get_name (component), "audio") == 0) {
						DVBEventAudioComponent* audio;
						char* _tmp13;
						const char* _tmp12;
						DVBEventAudioComponent* _tmp14;
						audio = dvb_event_audio_component_new ();
						_tmp13 = NULL;
						_tmp12 = NULL;
						audio->type = (_tmp13 = (_tmp12 = gst_structure_get_string (component, "type"), (_tmp12 == NULL) ? NULL : g_strdup (_tmp12)), audio->type = (g_free (audio->type), NULL), _tmp13);
						_tmp14 = NULL;
						event_class->audio_components = g_slist_append (event_class->audio_components, (_tmp14 = audio, (_tmp14 == NULL) ? NULL : dvb_event_audio_component_ref (_tmp14)));
						(audio == NULL) ? NULL : (audio = (dvb_event_audio_component_unref (audio), NULL));
					} else {
						if (_vala_strcmp0 (gst_structure_get_name (component), "video") == 0) {
							DVBEventVideoComponent* video;
							gboolean highdef;
							char* _tmp16;
							const char* _tmp15;
							gint freq;
							DVBEventVideoComponent* _tmp17;
							video = dvb_event_video_component_new ();
							highdef = FALSE;
							gst_structure_get_boolean (component, "high-definition", &highdef);
							video->high_definition = highdef;
							_tmp16 = NULL;
							_tmp15 = NULL;
							video->aspect_ratio = (_tmp16 = (_tmp15 = gst_structure_get_string (component, "high-definition"), (_tmp15 == NULL) ? NULL : g_strdup (_tmp15)), video->aspect_ratio = (g_free (video->aspect_ratio), NULL), _tmp16);
							freq = 0;
							gst_structure_get_int (component, "frequency", &freq);
							video->frequency = freq;
							_tmp17 = NULL;
							event_class->video_components = g_slist_append (event_class->video_components, (_tmp17 = video, (_tmp17 == NULL) ? NULL : dvb_event_video_component_ref (_tmp17)));
							(video == NULL) ? NULL : (video = (dvb_event_video_component_unref (video), NULL));
						} else {
							if (_vala_strcmp0 (gst_structure_get_name (component), "teletext") == 0) {
								DVBEventTeletextComponent* teletext;
								char* _tmp19;
								const char* _tmp18;
								DVBEventTeletextComponent* _tmp20;
								teletext = dvb_event_teletext_component_new ();
								_tmp19 = NULL;
								_tmp18 = NULL;
								teletext->type = (_tmp19 = (_tmp18 = gst_structure_get_string (component, "type"), (_tmp18 == NULL) ? NULL : g_strdup (_tmp18)), teletext->type = (g_free (teletext->type), NULL), _tmp19);
								_tmp20 = NULL;
								event_class->teletext_components = g_slist_append (event_class->teletext_components, (_tmp20 = teletext, (_tmp20 == NULL) ? NULL : dvb_event_teletext_component_ref (_tmp20)));
								(teletext == NULL) ? NULL : (teletext = (dvb_event_teletext_component_unref (teletext), NULL));
							}
						}
					}
				}
			}
			/*debug ("Adding new event: %s", event_class.to_string ());*/
			dvb_schedule_add (dvb_channel_get_Schedule (channel), event_class);
			(channel == NULL) ? NULL : (channel = (g_object_unref (channel), NULL));
			(event_class == NULL) ? NULL : (event_class = (dvb_event_unref (event_class), NULL));
			name = (g_free (name), NULL);
			desc = (g_free (desc), NULL);
			ext_desc = (g_free (ext_desc), NULL);
		}
	}
}


static guint dvb_epg_scanner_get_uint_val (const GstStructure* structure, const char* name) {
	guint val;
	g_return_val_if_fail (structure != NULL, 0U);
	g_return_val_if_fail (name != NULL, 0U);
	val = 0U;
	gst_structure_get_uint (structure, name, &val);
	return val;
}


DVBDeviceGroup* dvb_epg_scanner_get_DeviceGroup (DVBEPGScanner* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_DeviceGroup;
}


void dvb_epg_scanner_set_DeviceGroup (DVBEPGScanner* self, DVBDeviceGroup* value) {
	DVBDeviceGroup* _tmp2;
	DVBDeviceGroup* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_DeviceGroup = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : g_object_ref (_tmp1)), (self->priv->_DeviceGroup == NULL) ? NULL : (self->priv->_DeviceGroup = (g_object_unref (self->priv->_DeviceGroup), NULL)), _tmp2);
	g_object_notify ((GObject *) self, "DeviceGroup");
}


static GObject * dvb_epg_scanner_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	DVBEPGScannerClass * klass;
	GObjectClass * parent_class;
	DVBEPGScanner * self;
	klass = DVB_EPG_SCANNER_CLASS (g_type_class_peek (DVB_TYPE_EPG_SCANNER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = DVB_EPG_SCANNER (obj);
	{
		GQueue* _tmp0;
		_tmp0 = NULL;
		self->priv->channels = (_tmp0 = g_queue_new (), (self->priv->channels == NULL) ? NULL : (self->priv->channels = (g_queue_free (self->priv->channels), NULL)), _tmp0);
		self->priv->scan_event_id = (guint) 0;
		self->priv->do_stop = FALSE;
	}
	return obj;
}


static void dvb_epg_scanner_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	DVBEPGScanner * self;
	gpointer boxed;
	self = DVB_EPG_SCANNER (object);
	switch (property_id) {
		case DVB_EPG_SCANNER_DEVICE_GROUP:
		g_value_set_object (value, dvb_epg_scanner_get_DeviceGroup (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void dvb_epg_scanner_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	DVBEPGScanner * self;
	self = DVB_EPG_SCANNER (object);
	switch (property_id) {
		case DVB_EPG_SCANNER_DEVICE_GROUP:
		dvb_epg_scanner_set_DeviceGroup (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void dvb_epg_scanner_class_init (DVBEPGScannerClass * klass) {
	dvb_epg_scanner_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (DVBEPGScannerPrivate));
	G_OBJECT_CLASS (klass)->get_property = dvb_epg_scanner_get_property;
	G_OBJECT_CLASS (klass)->set_property = dvb_epg_scanner_set_property;
	G_OBJECT_CLASS (klass)->constructor = dvb_epg_scanner_constructor;
	G_OBJECT_CLASS (klass)->finalize = dvb_epg_scanner_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_EPG_SCANNER_DEVICE_GROUP, g_param_spec_object ("DeviceGroup", "DeviceGroup", "DeviceGroup", DVB_TYPE_DEVICE_GROUP, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	dvb_epg_scanner_CHECK_EIT_INTERVAL = -1;
}


static void dvb_epg_scanner_instance_init (DVBEPGScanner * self) {
	self->priv = DVB_EPG_SCANNER_GET_PRIVATE (self);
	g_static_rec_mutex_init (&self->priv->__lock_pipeline);
}


static void dvb_epg_scanner_finalize (GObject* obj) {
	DVBEPGScanner * self;
	self = DVB_EPG_SCANNER (obj);
	(self->priv->_DeviceGroup == NULL) ? NULL : (self->priv->_DeviceGroup = (g_object_unref (self->priv->_DeviceGroup), NULL));
	g_static_rec_mutex_free (&self->priv->__lock_pipeline);
	(self->priv->pipeline == NULL) ? NULL : (self->priv->pipeline = (gst_object_unref (self->priv->pipeline), NULL));
	(self->priv->channels == NULL) ? NULL : (self->priv->channels = (g_queue_free (self->priv->channels), NULL));
	G_OBJECT_CLASS (dvb_epg_scanner_parent_class)->finalize (obj);
}


GType dvb_epg_scanner_get_type (void) {
	static GType dvb_epg_scanner_type_id = 0;
	if (dvb_epg_scanner_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (DVBEPGScannerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_epg_scanner_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBEPGScanner), 0, (GInstanceInitFunc) dvb_epg_scanner_instance_init, NULL };
		dvb_epg_scanner_type_id = g_type_register_static (G_TYPE_OBJECT, "DVBEPGScanner", &g_define_type_info, 0);
	}
	return dvb_epg_scanner_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




