/*
 * Copyright (C) 2008,2009 Sebastian Pölsterl
 *
 * This file is part of GNOME DVB Daemon.
 *
 * GNOME DVB Daemon is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GNOME DVB Daemon is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNOME DVB Daemon.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <glib-object.h>
#include <gio/gio.h>
#include <stdlib.h>
#include <string.h>
#include <gst/gst.h>


#define DVB_TYPE_ADAPTER_TYPE (dvb_adapter_type_get_type ())

#define DVB_TYPE_DEVICE (dvb_device_get_type ())
#define DVB_DEVICE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_DEVICE, DVBDevice))
#define DVB_DEVICE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_DEVICE, DVBDeviceClass))
#define DVB_IS_DEVICE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_DEVICE))
#define DVB_IS_DEVICE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_DEVICE))
#define DVB_DEVICE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_DEVICE, DVBDeviceClass))

typedef struct _DVBDevice DVBDevice;
typedef struct _DVBDeviceClass DVBDeviceClass;
typedef struct _DVBDevicePrivate DVBDevicePrivate;

#define DVB_TYPE_CHANNEL_LIST (dvb_channel_list_get_type ())
#define DVB_CHANNEL_LIST(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CHANNEL_LIST, DVBChannelList))
#define DVB_CHANNEL_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CHANNEL_LIST, DVBChannelListClass))
#define DVB_IS_CHANNEL_LIST(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CHANNEL_LIST))
#define DVB_IS_CHANNEL_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CHANNEL_LIST))
#define DVB_CHANNEL_LIST_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CHANNEL_LIST, DVBChannelListClass))

typedef struct _DVBChannelList DVBChannelList;
typedef struct _DVBChannelListClass DVBChannelListClass;

typedef enum  {
	DVB_ADAPTER_TYPE_UNKNOWN,
	DVB_ADAPTER_TYPE_DVB_T,
	DVB_ADAPTER_TYPE_DVB_S,
	DVB_ADAPTER_TYPE_DVB_C
} DVBAdapterType;

struct _DVBDevice {
	GObject parent_instance;
	DVBDevicePrivate * priv;
};

struct _DVBDeviceClass {
	GObjectClass parent_class;
};

struct _DVBDevicePrivate {
	guint _Adapter;
	guint _Frontend;
	DVBChannelList* _Channels;
	GFile* _RecordingsDirectory;
	char* adapter_name;
	DVBAdapterType adapter_type;
};


static gpointer dvb_device_parent_class = NULL;

GType dvb_adapter_type_get_type (void);
GType dvb_device_get_type (void);
GType dvb_channel_list_get_type (void);
#define DVB_DEVICE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), DVB_TYPE_DEVICE, DVBDevicePrivate))
enum  {
	DVB_DEVICE_DUMMY_PROPERTY,
	DVB_DEVICE_ADAPTER,
	DVB_DEVICE_FRONTEND,
	DVB_DEVICE_TYPE,
	DVB_DEVICE_NAME,
	DVB_DEVICE_CHANNELS,
	DVB_DEVICE_RECORDINGS_DIRECTORY
};
#define DVB_DEVICE_PRIME 31
static void dvb_device_set_Adapter (DVBDevice* self, guint value);
static void dvb_device_set_Frontend (DVBDevice* self, guint value);
static gboolean dvb_device_setAdapterTypeAndName (DVBDevice* self, guint adapter, gboolean get_type);
DVBDevice* dvb_device_new (guint adapter, guint frontend, gboolean get_type_and_name);
DVBDevice* dvb_device_construct (GType object_type, guint adapter, guint frontend, gboolean get_type_and_name);
void dvb_device_set_Channels (DVBDevice* self, DVBChannelList* value);
void dvb_device_set_RecordingsDirectory (DVBDevice* self, GFile* value);
DVBDevice* dvb_device_new_full (guint adapter, guint frontend, DVBChannelList* channels, GFile* recordings_dir);
guint dvb_device_get_Adapter (DVBDevice* self);
guint dvb_device_get_Frontend (DVBDevice* self);
gboolean dvb_device_equal (DVBDevice* dev1, DVBDevice* dev2);
guint dvb_device_hash_without_device (guint adapter, guint frontend);
guint dvb_device_hash (DVBDevice* device);
gboolean dvb_device_is_busy (DVBDevice* self);
DVBAdapterType dvb_device_get_Type (DVBDevice* self);
const char* dvb_device_get_Name (DVBDevice* self);
DVBChannelList* dvb_device_get_Channels (DVBDevice* self);
GFile* dvb_device_get_RecordingsDirectory (DVBDevice* self);
static void dvb_device_finalize (GObject* obj);
static void dvb_device_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void dvb_device_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);
static int _vala_strcmp0 (const char * str1, const char * str2);




GType dvb_adapter_type_get_type (void) {
	static GType dvb_adapter_type_type_id = 0;
	if (G_UNLIKELY (dvb_adapter_type_type_id == 0)) {
		static const GEnumValue values[] = {{DVB_ADAPTER_TYPE_UNKNOWN, "DVB_ADAPTER_TYPE_UNKNOWN", "unknown"}, {DVB_ADAPTER_TYPE_DVB_T, "DVB_ADAPTER_TYPE_DVB_T", "dvb-t"}, {DVB_ADAPTER_TYPE_DVB_S, "DVB_ADAPTER_TYPE_DVB_S", "dvb-s"}, {DVB_ADAPTER_TYPE_DVB_C, "DVB_ADAPTER_TYPE_DVB_C", "dvb-c"}, {0, NULL, NULL}};
		dvb_adapter_type_type_id = g_enum_register_static ("DVBAdapterType", values);
	}
	return dvb_adapter_type_type_id;
}


DVBDevice* dvb_device_construct (GType object_type, guint adapter, guint frontend, gboolean get_type_and_name) {
	GParameter * __params;
	GParameter * __params_it;
	DVBDevice * self;
	__params = g_new0 (GParameter, 2);
	__params_it = __params;
	__params_it->name = "Adapter";
	g_value_init (&__params_it->value, G_TYPE_UINT);
	g_value_set_uint (&__params_it->value, adapter);
	__params_it++;
	__params_it->name = "Frontend";
	g_value_init (&__params_it->value, G_TYPE_UINT);
	g_value_set_uint (&__params_it->value, frontend);
	__params_it++;
	self = g_object_newv (object_type, __params_it - __params, __params);
	dvb_device_setAdapterTypeAndName (self, adapter, get_type_and_name);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


DVBDevice* dvb_device_new (guint adapter, guint frontend, gboolean get_type_and_name) {
	return dvb_device_construct (DVB_TYPE_DEVICE, adapter, frontend, get_type_and_name);
}


DVBDevice* dvb_device_new_full (guint adapter, guint frontend, DVBChannelList* channels, GFile* recordings_dir) {
	DVBDevice* result;
	DVBDevice* dev;
	g_return_val_if_fail (channels != NULL, NULL);
	g_return_val_if_fail (recordings_dir != NULL, NULL);
	dev = dvb_device_new (adapter, frontend, TRUE);
	dvb_device_set_Channels (dev, channels);
	dvb_device_set_RecordingsDirectory (dev, recordings_dir);
	result = dev;
	return result;
}


gboolean dvb_device_equal (DVBDevice* dev1, DVBDevice* dev2) {
	gboolean result;
	gboolean _tmp0_;
	gboolean _tmp1_;
	_tmp0_ = FALSE;
	if (dev1 == NULL) {
		_tmp0_ = TRUE;
	} else {
		_tmp0_ = dev2 == NULL;
	}
	if (_tmp0_) {
		result = FALSE;
		return result;
	}
	_tmp1_ = FALSE;
	if (dev1->priv->_Adapter == dev2->priv->_Adapter) {
		_tmp1_ = dev2->priv->_Frontend == dev2->priv->_Frontend;
	} else {
		_tmp1_ = FALSE;
	}
	result = _tmp1_;
	return result;
}


guint dvb_device_hash (DVBDevice* device) {
	guint result;
	if (device == NULL) {
		result = (guint) 0;
		return result;
	}
	result = dvb_device_hash_without_device (device->priv->_Adapter, device->priv->_Frontend);
	return result;
}


guint dvb_device_hash_without_device (guint adapter, guint frontend) {
	guint result;
	result = ((2 * DVB_DEVICE_PRIME) + (DVB_DEVICE_PRIME * adapter)) + frontend;
	return result;
}


gboolean dvb_device_is_busy (DVBDevice* self) {
	gboolean result;
	GstElement* dvbsrc;
	GstElement* pipeline;
	GstElement* _tmp0_;
	GstBus* bus;
	gboolean busy_val;
	g_return_val_if_fail (self != NULL, FALSE);
	dvbsrc = gst_element_factory_make ("dvbsrc", "text_dvbsrc");
	if (dvbsrc == NULL) {
		g_critical ("Device.vala:84: Could not create dvbsrc element");
		result = TRUE;
		(dvbsrc == NULL) ? NULL : (dvbsrc = (gst_object_unref (dvbsrc), NULL));
		return result;
	}
	g_object_set ((GObject*) dvbsrc, "adapter", self->priv->_Adapter, NULL);
	g_object_set ((GObject*) dvbsrc, "frontend", self->priv->_Frontend, NULL);
	pipeline = (GstElement*) ((GstPipeline*) gst_pipeline_new (""));
	_tmp0_ = NULL;
	gst_bin_add (GST_BIN (pipeline), (_tmp0_ = dvbsrc, (_tmp0_ == NULL) ? NULL : gst_object_ref (_tmp0_)));
	gst_element_set_state (pipeline, GST_STATE_READY);
	bus = gst_element_get_bus (pipeline);
	busy_val = FALSE;
	while (TRUE) {
		GstMessage* msg;
		gboolean _tmp1_;
		if (!gst_bus_have_pending (bus)) {
			break;
		}
		msg = gst_bus_pop (bus);
		_tmp1_ = FALSE;
		if (msg->type == GST_MESSAGE_ERROR) {
			_tmp1_ = msg->src == GST_OBJECT (dvbsrc);
		} else {
			_tmp1_ = FALSE;
		}
		if (_tmp1_) {
			GError* gerror;
			char* debug_text;
			char* _tmp5_;
			char* _tmp4_;
			GError* _tmp3_;
			GError* _tmp2_;
			gerror = NULL;
			debug_text = NULL;
			_tmp5_ = NULL;
			_tmp4_ = NULL;
			_tmp3_ = NULL;
			_tmp2_ = NULL;
			(gst_message_parse_error (msg, &_tmp2_, &_tmp4_), gerror = (_tmp3_ = _tmp2_, (gerror == NULL) ? NULL : (gerror = (g_error_free (gerror), NULL)), _tmp3_));
			debug_text = (_tmp5_ = _tmp4_, debug_text = (g_free (debug_text), NULL), _tmp5_);
			g_debug ("Device.vala:106: Error tuning: %s; %s", gerror->message, debug_text);
			busy_val = TRUE;
			(gerror == NULL) ? NULL : (gerror = (g_error_free (gerror), NULL));
			debug_text = (g_free (debug_text), NULL);
		}
		(msg == NULL) ? NULL : (msg = (gst_message_unref (msg), NULL));
	}
	gst_element_set_state (pipeline, GST_STATE_NULL);
	result = busy_val;
	(dvbsrc == NULL) ? NULL : (dvbsrc = (gst_object_unref (dvbsrc), NULL));
	(pipeline == NULL) ? NULL : (pipeline = (gst_object_unref (pipeline), NULL));
	(bus == NULL) ? NULL : (bus = (gst_object_unref (bus), NULL));
	return result;
}


static gboolean dvb_device_setAdapterTypeAndName (DVBDevice* self, guint adapter, gboolean get_type) {
	gboolean result;
	GstElement* dvbsrc;
	GstElement* pipeline;
	GstElement* _tmp0_;
	GstBus* bus;
	gboolean success;
	char* adapter_type;
	g_return_val_if_fail (self != NULL, FALSE);
	if (!get_type) {
		result = TRUE;
		return result;
	}
	dvbsrc = gst_element_factory_make ("dvbsrc", "test_dvbsrc");
	if (dvbsrc == NULL) {
		g_critical ("Device.vala:122: Could not create dvbsrc element");
		result = FALSE;
		(dvbsrc == NULL) ? NULL : (dvbsrc = (gst_object_unref (dvbsrc), NULL));
		return result;
	}
	g_object_set ((GObject*) dvbsrc, "adapter", adapter, NULL);
	pipeline = (GstElement*) ((GstPipeline*) gst_pipeline_new ("type_name"));
	_tmp0_ = NULL;
	gst_bin_add (GST_BIN (pipeline), (_tmp0_ = dvbsrc, (_tmp0_ == NULL) ? NULL : gst_object_ref (_tmp0_)));
	gst_element_set_state (pipeline, GST_STATE_READY);
	bus = gst_element_get_bus (pipeline);
	success = FALSE;
	adapter_type = NULL;
	while (TRUE) {
		GstMessage* msg;
		gboolean _tmp1_;
		if (!gst_bus_have_pending (bus)) {
			break;
		}
		msg = gst_bus_pop (bus);
		_tmp1_ = FALSE;
		if (msg->type == GST_MESSAGE_ELEMENT) {
			_tmp1_ = msg->src == GST_OBJECT (dvbsrc);
		} else {
			_tmp1_ = FALSE;
		}
		if (_tmp1_) {
			const GstStructure* structure;
			structure = msg->structure;
			if (_vala_strcmp0 (gst_structure_get_name (structure), "dvb-adapter") == 0) {
				char* _tmp2_;
				char* _tmp3_;
				_tmp2_ = NULL;
				adapter_type = (_tmp2_ = g_strdup_printf ("%s", gst_structure_get_string (structure, "type")), adapter_type = (g_free (adapter_type), NULL), _tmp2_);
				_tmp3_ = NULL;
				self->priv->adapter_name = (_tmp3_ = g_strdup_printf ("%s", gst_structure_get_string (structure, "name")), self->priv->adapter_name = (g_free (self->priv->adapter_name), NULL), _tmp3_);
				success = TRUE;
				(msg == NULL) ? NULL : (msg = (gst_message_unref (msg), NULL));
				break;
			}
		} else {
			if (msg->type == GST_MESSAGE_ERROR) {
				GError* gerror;
				char* debug;
				char* _tmp7_;
				char* _tmp6_;
				GError* _tmp5_;
				GError* _tmp4_;
				gerror = NULL;
				debug = NULL;
				_tmp7_ = NULL;
				_tmp6_ = NULL;
				_tmp5_ = NULL;
				_tmp4_ = NULL;
				(gst_message_parse_error (msg, &_tmp4_, &_tmp6_), gerror = (_tmp5_ = _tmp4_, (gerror == NULL) ? NULL : (gerror = (g_error_free (gerror), NULL)), _tmp5_));
				debug = (_tmp7_ = _tmp6_, debug = (g_free (debug), NULL), _tmp7_);
				g_critical ("Device.vala:152: %s %s", gerror->message, debug);
				(gerror == NULL) ? NULL : (gerror = (g_error_free (gerror), NULL));
				debug = (g_free (debug), NULL);
			}
		}
		(msg == NULL) ? NULL : (msg = (gst_message_unref (msg), NULL));
	}
	gst_element_set_state (pipeline, GST_STATE_NULL);
	if (_vala_strcmp0 (adapter_type, "DVB-T") == 0) {
		self->priv->adapter_type = DVB_ADAPTER_TYPE_DVB_T;
	} else {
		if (_vala_strcmp0 (adapter_type, "DVB-S") == 0) {
			self->priv->adapter_type = DVB_ADAPTER_TYPE_DVB_S;
		} else {
			if (_vala_strcmp0 (adapter_type, "DVB-C") == 0) {
				self->priv->adapter_type = DVB_ADAPTER_TYPE_DVB_C;
			} else {
				self->priv->adapter_type = DVB_ADAPTER_TYPE_UNKNOWN;
			}
		}
	}
	result = success;
	(dvbsrc == NULL) ? NULL : (dvbsrc = (gst_object_unref (dvbsrc), NULL));
	(pipeline == NULL) ? NULL : (pipeline = (gst_object_unref (pipeline), NULL));
	(bus == NULL) ? NULL : (bus = (gst_object_unref (bus), NULL));
	adapter_type = (g_free (adapter_type), NULL);
	return result;
}


guint dvb_device_get_Adapter (DVBDevice* self) {
	guint result;
	g_return_val_if_fail (self != NULL, 0U);
	result = self->priv->_Adapter;
	return result;
}


static void dvb_device_set_Adapter (DVBDevice* self, guint value) {
	g_return_if_fail (self != NULL);
	self->priv->_Adapter = value;
	g_object_notify ((GObject *) self, "Adapter");
}


guint dvb_device_get_Frontend (DVBDevice* self) {
	guint result;
	g_return_val_if_fail (self != NULL, 0U);
	result = self->priv->_Frontend;
	return result;
}


static void dvb_device_set_Frontend (DVBDevice* self, guint value) {
	g_return_if_fail (self != NULL);
	self->priv->_Frontend = value;
	g_object_notify ((GObject *) self, "Frontend");
}


DVBAdapterType dvb_device_get_Type (DVBDevice* self) {
	DVBAdapterType result;
	g_return_val_if_fail (self != NULL, 0);
	result = self->priv->adapter_type;
	return result;
}


const char* dvb_device_get_Name (DVBDevice* self) {
	const char* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->adapter_name;
	return result;
}


DVBChannelList* dvb_device_get_Channels (DVBDevice* self) {
	DVBChannelList* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_Channels;
	return result;
}


void dvb_device_set_Channels (DVBDevice* self, DVBChannelList* value) {
	DVBChannelList* _tmp1_;
	DVBChannelList* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_Channels = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_)), (self->priv->_Channels == NULL) ? NULL : (self->priv->_Channels = (g_object_unref (self->priv->_Channels), NULL)), _tmp1_);
	g_object_notify ((GObject *) self, "Channels");
}


GFile* dvb_device_get_RecordingsDirectory (DVBDevice* self) {
	GFile* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_RecordingsDirectory;
	return result;
}


void dvb_device_set_RecordingsDirectory (DVBDevice* self, GFile* value) {
	GFile* _tmp1_;
	GFile* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_RecordingsDirectory = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_)), (self->priv->_RecordingsDirectory == NULL) ? NULL : (self->priv->_RecordingsDirectory = (g_object_unref (self->priv->_RecordingsDirectory), NULL)), _tmp1_);
	g_object_notify ((GObject *) self, "RecordingsDirectory");
}


static void dvb_device_class_init (DVBDeviceClass * klass) {
	dvb_device_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (DVBDevicePrivate));
	G_OBJECT_CLASS (klass)->get_property = dvb_device_get_property;
	G_OBJECT_CLASS (klass)->set_property = dvb_device_set_property;
	G_OBJECT_CLASS (klass)->finalize = dvb_device_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_DEVICE_ADAPTER, g_param_spec_uint ("Adapter", "Adapter", "Adapter", 0, G_MAXUINT, 0U, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_DEVICE_FRONTEND, g_param_spec_uint ("Frontend", "Frontend", "Frontend", 0, G_MAXUINT, 0U, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_DEVICE_TYPE, g_param_spec_enum ("Type", "Type", "Type", DVB_TYPE_ADAPTER_TYPE, 0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_DEVICE_NAME, g_param_spec_string ("Name", "Name", "Name", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_DEVICE_CHANNELS, g_param_spec_object ("Channels", "Channels", "Channels", DVB_TYPE_CHANNEL_LIST, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_DEVICE_RECORDINGS_DIRECTORY, g_param_spec_object ("RecordingsDirectory", "RecordingsDirectory", "RecordingsDirectory", G_TYPE_FILE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void dvb_device_instance_init (DVBDevice * self) {
	self->priv = DVB_DEVICE_GET_PRIVATE (self);
}


static void dvb_device_finalize (GObject* obj) {
	DVBDevice * self;
	self = DVB_DEVICE (obj);
	(self->priv->_Channels == NULL) ? NULL : (self->priv->_Channels = (g_object_unref (self->priv->_Channels), NULL));
	(self->priv->_RecordingsDirectory == NULL) ? NULL : (self->priv->_RecordingsDirectory = (g_object_unref (self->priv->_RecordingsDirectory), NULL));
	self->priv->adapter_name = (g_free (self->priv->adapter_name), NULL);
	G_OBJECT_CLASS (dvb_device_parent_class)->finalize (obj);
}


GType dvb_device_get_type (void) {
	static GType dvb_device_type_id = 0;
	if (dvb_device_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (DVBDeviceClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_device_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBDevice), 0, (GInstanceInitFunc) dvb_device_instance_init, NULL };
		dvb_device_type_id = g_type_register_static (G_TYPE_OBJECT, "DVBDevice", &g_define_type_info, 0);
	}
	return dvb_device_type_id;
}


static void dvb_device_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	DVBDevice * self;
	gpointer boxed;
	self = DVB_DEVICE (object);
	switch (property_id) {
		case DVB_DEVICE_ADAPTER:
		g_value_set_uint (value, dvb_device_get_Adapter (self));
		break;
		case DVB_DEVICE_FRONTEND:
		g_value_set_uint (value, dvb_device_get_Frontend (self));
		break;
		case DVB_DEVICE_TYPE:
		g_value_set_enum (value, dvb_device_get_Type (self));
		break;
		case DVB_DEVICE_NAME:
		g_value_set_string (value, dvb_device_get_Name (self));
		break;
		case DVB_DEVICE_CHANNELS:
		g_value_set_object (value, dvb_device_get_Channels (self));
		break;
		case DVB_DEVICE_RECORDINGS_DIRECTORY:
		g_value_set_object (value, dvb_device_get_RecordingsDirectory (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void dvb_device_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	DVBDevice * self;
	self = DVB_DEVICE (object);
	switch (property_id) {
		case DVB_DEVICE_ADAPTER:
		dvb_device_set_Adapter (self, g_value_get_uint (value));
		break;
		case DVB_DEVICE_FRONTEND:
		dvb_device_set_Frontend (self, g_value_get_uint (value));
		break;
		case DVB_DEVICE_CHANNELS:
		dvb_device_set_Channels (self, g_value_get_object (value));
		break;
		case DVB_DEVICE_RECORDINGS_DIRECTORY:
		dvb_device_set_RecordingsDirectory (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




