/*
 * Copyright (C) 2008,2009 Sebastian Pölsterl
 *
 * This file is part of GNOME DVB Daemon.
 *
 * GNOME DVB Daemon is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GNOME DVB Daemon is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNOME DVB Daemon.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <glib-object.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <dbus/dbus-glib.h>
#include <stdlib.h>
#include <string.h>
#include <gst/gst.h>
#include <signal.h>
#include <stdio.h>
#include <gee.h>
#include <dbus/dbus.h>


#define DVB_TYPE_MANAGER (dvb_manager_get_type ())
#define DVB_MANAGER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_MANAGER, DVBManager))
#define DVB_MANAGER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_MANAGER, DVBManagerClass))
#define DVB_IS_MANAGER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_MANAGER))
#define DVB_IS_MANAGER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_MANAGER))
#define DVB_MANAGER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_MANAGER, DVBManagerClass))

typedef struct _DVBManager DVBManager;
typedef struct _DVBManagerClass DVBManagerClass;

#define DVB_TYPE_RECORDINGS_STORE (dvb_recordings_store_get_type ())
#define DVB_RECORDINGS_STORE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_RECORDINGS_STORE, DVBRecordingsStore))
#define DVB_RECORDINGS_STORE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_RECORDINGS_STORE, DVBRecordingsStoreClass))
#define DVB_IS_RECORDINGS_STORE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_RECORDINGS_STORE))
#define DVB_IS_RECORDINGS_STORE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_RECORDINGS_STORE))
#define DVB_RECORDINGS_STORE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_RECORDINGS_STORE, DVBRecordingsStoreClass))

typedef struct _DVBRecordingsStore DVBRecordingsStore;
typedef struct _DVBRecordingsStoreClass DVBRecordingsStoreClass;

#define DVB_TYPE_TIMERS_STORE (dvb_timers_store_get_type ())
#define DVB_TIMERS_STORE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_TIMERS_STORE, DVBTimersStore))
#define DVB_IS_TIMERS_STORE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_TIMERS_STORE))
#define DVB_TIMERS_STORE_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), DVB_TYPE_TIMERS_STORE, DVBTimersStoreIface))

typedef struct _DVBTimersStore DVBTimersStore;
typedef struct _DVBTimersStoreIface DVBTimersStoreIface;

#define DVB_TYPE_DEVICE_GROUP (dvb_device_group_get_type ())
#define DVB_DEVICE_GROUP(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_DEVICE_GROUP, DVBDeviceGroup))
#define DVB_DEVICE_GROUP_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_DEVICE_GROUP, DVBDeviceGroupClass))
#define DVB_IS_DEVICE_GROUP(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_DEVICE_GROUP))
#define DVB_IS_DEVICE_GROUP_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_DEVICE_GROUP))
#define DVB_DEVICE_GROUP_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_DEVICE_GROUP, DVBDeviceGroupClass))

typedef struct _DVBDeviceGroup DVBDeviceGroup;
typedef struct _DVBDeviceGroupClass DVBDeviceGroupClass;

#define DVB_TYPE_TIMER (dvb_timer_get_type ())
#define DVB_TIMER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_TIMER, DVBTimer))
#define DVB_TIMER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_TIMER, DVBTimerClass))
#define DVB_IS_TIMER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_TIMER))
#define DVB_IS_TIMER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_TIMER))
#define DVB_TIMER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_TIMER, DVBTimerClass))

typedef struct _DVBTimer DVBTimer;
typedef struct _DVBTimerClass DVBTimerClass;

#define DVB_TYPE_CONFIG_STORE (dvb_config_store_get_type ())
#define DVB_CONFIG_STORE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CONFIG_STORE, DVBConfigStore))
#define DVB_IS_CONFIG_STORE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CONFIG_STORE))
#define DVB_CONFIG_STORE_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), DVB_TYPE_CONFIG_STORE, DVBConfigStoreIface))

typedef struct _DVBConfigStore DVBConfigStore;
typedef struct _DVBConfigStoreIface DVBConfigStoreIface;

#define DVB_TYPE_DEVICE (dvb_device_get_type ())
#define DVB_DEVICE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_DEVICE, DVBDevice))
#define DVB_DEVICE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_DEVICE, DVBDeviceClass))
#define DVB_IS_DEVICE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_DEVICE))
#define DVB_IS_DEVICE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_DEVICE))
#define DVB_DEVICE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_DEVICE, DVBDeviceClass))

typedef struct _DVBDevice DVBDevice;
typedef struct _DVBDeviceClass DVBDeviceClass;

#define DVB_TYPE_RECORDER (dvb_recorder_get_type ())
#define DVB_RECORDER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_RECORDER, DVBRecorder))
#define DVB_RECORDER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_RECORDER, DVBRecorderClass))
#define DVB_IS_RECORDER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_RECORDER))
#define DVB_IS_RECORDER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_RECORDER))
#define DVB_RECORDER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_RECORDER, DVBRecorderClass))

typedef struct _DVBRecorder DVBRecorder;
typedef struct _DVBRecorderClass DVBRecorderClass;
typedef struct _DBusObjectVTable _DBusObjectVTable;

struct _DVBTimersStoreIface {
	GTypeInterface parent_iface;
	GeeList* (*get_all_timers_of_device_group) (DVBTimersStore* self, DVBDeviceGroup* dev);
	gboolean (*add_timer_to_device_group) (DVBTimersStore* self, DVBTimer* timer, DVBDeviceGroup* dev);
	gboolean (*remove_timer_from_device_group) (DVBTimersStore* self, guint timer_id, DVBDeviceGroup* dev);
};

struct _DVBConfigStoreIface {
	GTypeInterface parent_iface;
	GeeList* (*get_all_device_groups) (DVBConfigStore* self);
	gboolean (*add_device_group) (DVBConfigStore* self, DVBDeviceGroup* dev_group);
	gboolean (*remove_device_group) (DVBConfigStore* self, DVBDeviceGroup* devgroup);
	gboolean (*add_device_to_group) (DVBConfigStore* self, DVBDevice* dev, DVBDeviceGroup* devgroup);
	gboolean (*remove_device_from_group) (DVBConfigStore* self, DVBDevice* dev, DVBDeviceGroup* devgroup);
	gboolean (*update_from_group) (DVBConfigStore* self, DVBDeviceGroup* devgroup);
};

struct _DBusObjectVTable {
	void (*register_object) (DBusConnection*, const char*, void*);
};


extern DVBManager* main_manager;
DVBManager* main_manager = NULL;
extern DVBRecordingsStore* main_recstore;
DVBRecordingsStore* main_recstore = NULL;
extern gboolean main_has_debug;
gboolean main_has_debug = FALSE;
extern gboolean main_has_version;
gboolean main_has_version = FALSE;
extern gboolean main_disable_epg_scanner;
gboolean main_disable_epg_scanner = FALSE;
extern gboolean main_disable_rygel;
gboolean main_disable_rygel = FALSE;
extern GMainLoop* main_mainloop;
GMainLoop* main_mainloop = NULL;

GType dvb_manager_get_type (void);
GType dvb_recordings_store_get_type (void);
#define DVB_CONSTANTS_DBUS_SERVICE "org.gnome.DVB"
guint _dynamic_RequestName0 (DBusGProxy* self, const char* param1, guint param2, GError** error);
DVBManager* dvb_manager_get_instance (void);
#define DVB_CONSTANTS_DBUS_MANAGER_PATH "/org/gnome/DVB/Manager"
gboolean main_start_manager (void);
DVBRecordingsStore* dvb_recordings_store_get_instance (void);
void dvb_recordings_store_update_last_id (DVBRecordingsStore* self, guint32 new_last_id);
#define DVB_CONSTANTS_DBUS_RECORDINGS_STORE_PATH "/org/gnome/DVB/RecordingsStore"
gboolean main_start_recordings_store (guint32 minimum_id);
void dvb_rtsp_server_shutdown (void);
void dvb_manager_shutdown (void);
void dvb_factory_shutdown (void);
void dvb_recordings_store_shutdown (void);
void main_on_exit (gint signum);
void main_log_func (const char* log_domain, GLogLevelFlags log_levels, const char* message);
gboolean main_get_disable_epg_scanner (void);
gboolean main_check_feature_version (const char* name, guint major, guint minor, guint micro);
gboolean main_check_requirements (void);
static void _main_on_exit_cutils_signal_signal_handler (gint signum);
static void _main_log_func_glog_func (const char* log_domain, GLogLevelFlags log_levels, const char* message, gpointer self);
GType dvb_device_group_get_type (void);
GType dvb_timer_get_type (void);
GType dvb_timers_store_get_type (void);
DVBTimersStore* dvb_factory_get_timers_store (void);
GType dvb_device_get_type (void);
GType dvb_config_store_get_type (void);
DVBConfigStore* dvb_factory_get_config_store (void);
GeeList* dvb_config_store_get_all_device_groups (DVBConfigStore* self);
gboolean dvb_manager_add_device_group (DVBManager* self, DVBDeviceGroup* devgroup);
GType dvb_recorder_get_type (void);
DVBRecorder* dvb_device_group_get_recorder (DVBDeviceGroup* self);
guint dvb_device_group_get_Id (DVBDeviceGroup* self);
GeeList* dvb_timers_store_get_all_timers_of_device_group (DVBTimersStore* self, DVBDeviceGroup* dev);
guint32 dvb_timer_get_Id (DVBTimer* self);
guint32 dvb_recorder_add_timer (DVBRecorder* self, DVBTimer* new_timer);
gboolean dvb_timers_store_remove_timer_from_device_group (DVBTimersStore* self, guint timer_id, DVBDeviceGroup* dev);
gboolean dvb_rtsp_server_start (void);
static gboolean _dvb_rtsp_server_start_gsource_func (gpointer self);
gboolean dvb_rygel_service_start_rygel_services (void);
static gboolean _dvb_rygel_service_start_rygel_services_gsource_func (gpointer self);
gint main_main (char** args, int args_length1);
static void _vala_dbus_register_object (DBusConnection* connection, const char* path, void* object);
static void _vala_dbus_unregister_object (gpointer connection, GObject* object);

static const GOptionEntry MAIN_options[] = {{"debug", 'd', 0, G_OPTION_ARG_NONE, &main_has_debug, "Display debug statements on stdout", NULL}, {"version", (gchar) 0, 0, G_OPTION_ARG_NONE, &main_has_version, "Display version number", NULL}, {"disable-epg-scanner", (gchar) 0, 0, G_OPTION_ARG_NONE, &main_disable_epg_scanner, "Disable scanning for EPG data", NULL}, {"disable-rygel", (gchar) 0, 0, G_OPTION_ARG_NONE, &main_disable_rygel, "Disable exporting devices and channels for Rygel", NULL}, {NULL}};


guint _dynamic_RequestName0 (DBusGProxy* self, const char* param1, guint param2, GError** error) {
	guint result;
	dbus_g_proxy_call (self, "RequestName", error, G_TYPE_STRING, param1, G_TYPE_UINT, param2, G_TYPE_INVALID, G_TYPE_UINT, &result, G_TYPE_INVALID);
	if (*error) {
		return 0U;
	}
	return result;
}


gboolean main_start_manager (void) {
	gboolean result;
	GError * _inner_error_;
	_inner_error_ = NULL;
	{
		DBusGConnection* conn;
		DBusGProxy* bus;
		guint request_name_result;
		conn = dbus_g_bus_get (DBUS_BUS_SESSION, &_inner_error_);
		if (_inner_error_ != NULL) {
			goto __catch5_g_error;
			goto __finally5;
		}
		bus = dbus_g_proxy_new_for_name (conn, "org.freedesktop.DBus", "/org/freedesktop/DBus", "org.freedesktop.DBus");
		/* try to register service in session bus*/
		request_name_result = _dynamic_RequestName0 (bus, DVB_CONSTANTS_DBUS_SERVICE, (guint) 0, &_inner_error_);
		if (_inner_error_ != NULL) {
			(conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL));
			(bus == NULL) ? NULL : (bus = (g_object_unref (bus), NULL));
			goto __catch5_g_error;
			goto __finally5;
		}
		if (request_name_result == DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER) {
			g_message ("Main.vala:55: Creating new Manager D-Bus service");
			main_manager = dvb_manager_get_instance ();
			_vala_dbus_register_object (dbus_g_connection_get_connection (conn), DVB_CONSTANTS_DBUS_MANAGER_PATH, (GObject*) main_manager);
		} else {
			g_warning ("Main.vala:63: Manager D-Bus service is already running");
			result = FALSE;
			(conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL));
			(bus == NULL) ? NULL : (bus = (g_object_unref (bus), NULL));
			return result;
		}
		(conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL));
		(bus == NULL) ? NULL : (bus = (g_object_unref (bus), NULL));
	}
	goto __finally5;
	__catch5_g_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			g_error ("Main.vala:68: Oops %s", e->message);
			result = FALSE;
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			return result;
		}
	}
	__finally5:
	if (_inner_error_ != NULL) {
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return FALSE;
	}
	result = TRUE;
	return result;
}


gboolean main_start_recordings_store (guint32 minimum_id) {
	gboolean result;
	GError * _inner_error_;
	_inner_error_ = NULL;
	g_message ("Main.vala:76: Creating new RecordingsStore D-Bus service");
	{
		DBusGConnection* conn;
		DVBRecordingsStore* _tmp1_;
		DVBRecordingsStore* _tmp0_;
		conn = dbus_g_bus_get (DBUS_BUS_SESSION, &_inner_error_);
		if (_inner_error_ != NULL) {
			goto __catch6_g_error;
			goto __finally6;
		}
		_tmp1_ = NULL;
		_tmp0_ = NULL;
		main_recstore = (_tmp1_ = (_tmp0_ = dvb_recordings_store_get_instance (), (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_)), (main_recstore == NULL) ? NULL : (main_recstore = (g_object_unref (main_recstore), NULL)), _tmp1_);
		dvb_recordings_store_update_last_id (main_recstore, minimum_id);
		_vala_dbus_register_object (dbus_g_connection_get_connection (conn), DVB_CONSTANTS_DBUS_RECORDINGS_STORE_PATH, (GObject*) main_recstore);
		(conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL));
	}
	goto __finally6;
	__catch6_g_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			g_error ("Main.vala:88: Oops %s", e->message);
			result = FALSE;
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			return result;
		}
	}
	__finally6:
	if (_inner_error_ != NULL) {
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return FALSE;
	}
	result = TRUE;
	return result;
}


void main_on_exit (gint signum) {
	DVBRecordingsStore* _tmp0_;
	g_message ("Main.vala:96: Exiting");
	dvb_rtsp_server_shutdown ();
	dvb_manager_shutdown ();
	dvb_factory_shutdown ();
	dvb_recordings_store_shutdown ();
	_tmp0_ = NULL;
	main_recstore = (_tmp0_ = NULL, (main_recstore == NULL) ? NULL : (main_recstore = (g_object_unref (main_recstore), NULL)), _tmp0_);
	main_manager = NULL;
	g_main_loop_quit (main_mainloop);
}


void main_log_func (const char* log_domain, GLogLevelFlags log_levels, const char* message) {
	g_return_if_fail (message != NULL);
	if (main_has_debug) {
		g_log_default_handler (log_domain, log_levels, message, NULL);
	}
}


gboolean main_get_disable_epg_scanner (void) {
	gboolean result;
	result = main_disable_epg_scanner;
	return result;
}


static char* bool_to_string (gboolean self) {
	char* result;
	if (self) {
		result = g_strdup ("true");
		return result;
	} else {
		result = g_strdup ("false");
		return result;
	}
}


gboolean main_check_feature_version (const char* name, guint major, guint minor, guint micro) {
	gboolean result;
	GstRegistry* _tmp0_;
	GstRegistry* reg;
	GstPluginFeature* feature;
	gboolean ret;
	char* _tmp1_;
	g_return_val_if_fail (name != NULL, FALSE);
	_tmp0_ = NULL;
	reg = (_tmp0_ = gst_registry_get_default (), (_tmp0_ == NULL) ? NULL : gst_object_ref (_tmp0_));
	feature = gst_registry_lookup_feature (reg, name);
	ret = FALSE;
	if (feature == NULL) {
		ret = FALSE;
	} else {
		ret = gst_plugin_feature_check_version (feature, major, minor, micro);
	}
	_tmp1_ = NULL;
	g_debug ("Main.vala:128: Has %s >= %u.%u.%u: %s", name, major, minor, micro, _tmp1_ = bool_to_string (ret));
	_tmp1_ = (g_free (_tmp1_), NULL);
	result = ret;
	(reg == NULL) ? NULL : (reg = (gst_object_unref (reg), NULL));
	(feature == NULL) ? NULL : (feature = (gst_object_unref (feature), NULL));
	return result;
}


gboolean main_check_requirements (void) {
	gboolean result;
	gboolean val;
	val = FALSE;
	val = main_check_feature_version ("dvbsrc", (guint) 0, (guint) 10, (guint) 13);
	if (!val) {
		result = FALSE;
		return result;
	}
	val = main_check_feature_version ("dvbbasebin", (guint) 0, (guint) 10, (guint) 13);
	if (!val) {
		result = FALSE;
		return result;
	}
	val = main_check_feature_version ("mpegtsparse", (guint) 0, (guint) 10, (guint) 13);
	if (!val) {
		result = FALSE;
		return result;
	}
	val = main_check_feature_version ("rtpmp2tpay", (guint) 0, (guint) 10, (guint) 14);
	result = val;
	return result;
}


static void _main_on_exit_cutils_signal_signal_handler (gint signum) {
	main_on_exit (signum);
}


static void _main_log_func_glog_func (const char* log_domain, GLogLevelFlags log_levels, const char* message, gpointer self) {
	main_log_func (log_domain, log_levels, message);
}


static gboolean _dvb_rtsp_server_start_gsource_func (gpointer self) {
	return dvb_rtsp_server_start ();
}


static gboolean _dvb_rygel_service_start_rygel_services_gsource_func (gpointer self) {
	return dvb_rygel_service_start_rygel_services ();
}


gint main_main (char** args, int args_length1) {
	gint result;
	GError * _inner_error_;
	GOptionContext* context;
	GMainLoop* _tmp0_;
	guint32 max_id;
	DVBTimersStore* timers_store;
	DVBConfigStore* config_store;
	GeeList* device_groups;
	_inner_error_ = NULL;
	signal (SIGINT, _main_on_exit_cutils_signal_signal_handler);
	signal (SIGTERM, _main_on_exit_cutils_signal_signal_handler);
	context = g_option_context_new ("- record and watch TV shows using one or more DVB adapters");
	g_option_context_add_main_entries (context, MAIN_options, NULL);
	g_option_context_add_group (context, gst_init_get_option_group ());
	{
		g_option_context_parse (context, &args_length1, &args, &_inner_error_);
		if (_inner_error_ != NULL) {
			if (_inner_error_->domain == G_OPTION_ERROR) {
				goto __catch7_g_option_error;
			}
			goto __finally7;
		}
	}
	goto __finally7;
	__catch7_g_option_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			fprintf (stderr, "%s\n", e->message);
			fprintf (stderr, "Run '%s --help' to see a full list of available command line options.\n", args[0]);
			result = 1;
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			(context == NULL) ? NULL : (context = (g_option_context_free (context), NULL));
			return result;
		}
	}
	__finally7:
	if (_inner_error_ != NULL) {
		(context == NULL) ? NULL : (context = (g_option_context_free (context), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return 0;
	}
	if (main_has_version) {
		fprintf (stdout, PACKAGE_NAME);
		fprintf (stdout, " %s\n", PACKAGE_VERSION);
		result = 0;
		(context == NULL) ? NULL : (context = (g_option_context_free (context), NULL));
		return result;
	}
	g_log_set_handler (NULL, (G_LOG_LEVEL_DEBUG | G_LOG_FLAG_FATAL) | G_LOG_FLAG_RECURSION, _main_log_func_glog_func, NULL);
	/* Creating a GLib main loop with a default context*/
	_tmp0_ = NULL;
	main_mainloop = (_tmp0_ = g_main_loop_new (NULL, FALSE), (main_mainloop == NULL) ? NULL : (main_mainloop = (g_main_loop_unref (main_mainloop), NULL)), _tmp0_);
	/* Initializing GStreamer*/
	gst_init (&args_length1, &args);
	if (!main_check_requirements ()) {
		fprintf (stderr, "You don't have all of the necessary requirements to run %s.\n", PACKAGE_NAME);
		fprintf (stderr, "Start the daemon with the --debug flag for more details.\n");
		result = -1;
		(context == NULL) ? NULL : (context = (g_option_context_free (context), NULL));
		return result;
	}
	if (!main_start_manager ()) {
		result = -1;
		(context == NULL) ? NULL : (context = (g_option_context_free (context), NULL));
		return result;
	}
	max_id = (guint32) 0;
	timers_store = dvb_factory_get_timers_store ();
	config_store = dvb_factory_get_config_store ();
	g_message ("Main.vala:193: Restoring device groups");
	device_groups = dvb_config_store_get_all_device_groups (config_store);
	{
		GeeIterator* _device_group_it;
		_device_group_it = gee_iterable_iterator ((GeeIterable*) device_groups);
		while (TRUE) {
			DVBDeviceGroup* device_group;
			if (!gee_iterator_next (_device_group_it)) {
				break;
			}
			device_group = (DVBDeviceGroup*) gee_iterator_get (_device_group_it);
			if (dvb_manager_add_device_group (main_manager, device_group)) {
				DVBRecorder* _tmp1_;
				DVBRecorder* rec;
				GeeList* timers;
				_tmp1_ = NULL;
				rec = (_tmp1_ = dvb_device_group_get_recorder (device_group), (_tmp1_ == NULL) ? NULL : g_object_ref (_tmp1_));
				/* Restore timers*/
				g_message ("Main.vala:201: Restoring timers of device group %u", dvb_device_group_get_Id (device_group));
				timers = dvb_timers_store_get_all_timers_of_device_group (timers_store, device_group);
				{
					GeeIterator* _t_it;
					_t_it = gee_iterable_iterator ((GeeIterable*) timers);
					while (TRUE) {
						DVBTimer* t;
						if (!gee_iterator_next (_t_it)) {
							break;
						}
						t = (DVBTimer*) gee_iterator_get (_t_it);
						if (dvb_timer_get_Id (t) > max_id) {
							max_id = dvb_timer_get_Id (t);
						}
						if (dvb_recorder_add_timer (rec, t) == 0) {
							dvb_timers_store_remove_timer_from_device_group (timers_store, (guint) dvb_timer_get_Id (t), device_group);
						}
						(t == NULL) ? NULL : (t = (g_object_unref (t), NULL));
					}
					(_t_it == NULL) ? NULL : (_t_it = (g_object_unref (_t_it), NULL));
				}
				(rec == NULL) ? NULL : (rec = (g_object_unref (rec), NULL));
				(timers == NULL) ? NULL : (timers = (g_object_unref (timers), NULL));
			}
			(device_group == NULL) ? NULL : (device_group = (g_object_unref (device_group), NULL));
		}
		(_device_group_it == NULL) ? NULL : (_device_group_it = (g_object_unref (_device_group_it), NULL));
	}
	timers_store = NULL;
	config_store = NULL;
	if (!main_start_recordings_store (max_id)) {
		result = -1;
		(context == NULL) ? NULL : (context = (g_option_context_free (context), NULL));
		(device_groups == NULL) ? NULL : (device_groups = (g_object_unref (device_groups), NULL));
		return result;
	}
	g_idle_add (_dvb_rtsp_server_start_gsource_func, NULL);
	if (!main_disable_rygel) {
		g_idle_add (_dvb_rygel_service_start_rygel_services_gsource_func, NULL);
	}
	/* Start GLib mainloop*/
	g_main_loop_run (main_mainloop);
	result = 0;
	(context == NULL) ? NULL : (context = (g_option_context_free (context), NULL));
	(device_groups == NULL) ? NULL : (device_groups = (g_object_unref (device_groups), NULL));
	return result;
}


int main (int argc, char ** argv) {
	g_thread_init (NULL);
	g_type_init ();
	return main_main (argv, argc);
}


static void _vala_dbus_register_object (DBusConnection* connection, const char* path, void* object) {
	const _DBusObjectVTable * vtable;
	vtable = g_type_get_qdata (G_TYPE_FROM_INSTANCE (object), g_quark_from_static_string ("DBusObjectVTable"));
	if (vtable) {
		vtable->register_object (connection, path, object);
	} else {
		g_warning ("Object does not implement any D-Bus interface");
	}
}


static void _vala_dbus_unregister_object (gpointer connection, GObject* object) {
	char* path;
	path = g_object_steal_data ((GObject*) object, "dbus_object_path");
	dbus_connection_unregister_object_path (connection, path);
	g_free (path);
}




