/*
 * Copyright (C) 2008,2009 Sebastian Pölsterl
 *
 * This file is part of GNOME DVB Daemon.
 *
 * GNOME DVB Daemon is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GNOME DVB Daemon is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNOME DVB Daemon.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <glib-object.h>
#include <gio/gio.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>


#define DVB_TYPE_CHANNEL_LIST_READER (dvb_channel_list_reader_get_type ())
#define DVB_CHANNEL_LIST_READER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CHANNEL_LIST_READER, DVBChannelListReader))
#define DVB_CHANNEL_LIST_READER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CHANNEL_LIST_READER, DVBChannelListReaderClass))
#define DVB_IS_CHANNEL_LIST_READER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CHANNEL_LIST_READER))
#define DVB_IS_CHANNEL_LIST_READER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CHANNEL_LIST_READER))
#define DVB_CHANNEL_LIST_READER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CHANNEL_LIST_READER, DVBChannelListReaderClass))

typedef struct _DVBChannelListReader DVBChannelListReader;
typedef struct _DVBChannelListReaderClass DVBChannelListReaderClass;
typedef struct _DVBChannelListReaderPrivate DVBChannelListReaderPrivate;

#define DVB_TYPE_ADAPTER_TYPE (dvb_adapter_type_get_type ())

#define DVB_TYPE_CHANNEL_LIST (dvb_channel_list_get_type ())
#define DVB_CHANNEL_LIST(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CHANNEL_LIST, DVBChannelList))
#define DVB_CHANNEL_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CHANNEL_LIST, DVBChannelListClass))
#define DVB_IS_CHANNEL_LIST(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CHANNEL_LIST))
#define DVB_IS_CHANNEL_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CHANNEL_LIST))
#define DVB_CHANNEL_LIST_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CHANNEL_LIST, DVBChannelListClass))

typedef struct _DVBChannelList DVBChannelList;
typedef struct _DVBChannelListClass DVBChannelListClass;

#define DVB_TYPE_CHANNEL (dvb_channel_get_type ())
#define DVB_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CHANNEL, DVBChannel))
#define DVB_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CHANNEL, DVBChannelClass))
#define DVB_IS_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CHANNEL))
#define DVB_IS_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CHANNEL))
#define DVB_CHANNEL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CHANNEL, DVBChannelClass))

typedef struct _DVBChannel DVBChannel;
typedef struct _DVBChannelClass DVBChannelClass;

#define DVB_TYPE_TERRESTRIAL_CHANNEL (dvb_terrestrial_channel_get_type ())
#define DVB_TERRESTRIAL_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_TERRESTRIAL_CHANNEL, DVBTerrestrialChannel))
#define DVB_TERRESTRIAL_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_TERRESTRIAL_CHANNEL, DVBTerrestrialChannelClass))
#define DVB_IS_TERRESTRIAL_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_TERRESTRIAL_CHANNEL))
#define DVB_IS_TERRESTRIAL_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_TERRESTRIAL_CHANNEL))
#define DVB_TERRESTRIAL_CHANNEL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_TERRESTRIAL_CHANNEL, DVBTerrestrialChannelClass))

typedef struct _DVBTerrestrialChannel DVBTerrestrialChannel;
typedef struct _DVBTerrestrialChannelClass DVBTerrestrialChannelClass;

#define DVB_TYPE_SATELLITE_CHANNEL (dvb_satellite_channel_get_type ())
#define DVB_SATELLITE_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_SATELLITE_CHANNEL, DVBSatelliteChannel))
#define DVB_SATELLITE_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_SATELLITE_CHANNEL, DVBSatelliteChannelClass))
#define DVB_IS_SATELLITE_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_SATELLITE_CHANNEL))
#define DVB_IS_SATELLITE_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_SATELLITE_CHANNEL))
#define DVB_SATELLITE_CHANNEL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_SATELLITE_CHANNEL, DVBSatelliteChannelClass))

typedef struct _DVBSatelliteChannel DVBSatelliteChannel;
typedef struct _DVBSatelliteChannelClass DVBSatelliteChannelClass;

#define DVB_TYPE_CABLE_CHANNEL (dvb_cable_channel_get_type ())
#define DVB_CABLE_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CABLE_CHANNEL, DVBCableChannel))
#define DVB_CABLE_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CABLE_CHANNEL, DVBCableChannelClass))
#define DVB_IS_CABLE_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CABLE_CHANNEL))
#define DVB_IS_CABLE_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CABLE_CHANNEL))
#define DVB_CABLE_CHANNEL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CABLE_CHANNEL, DVBCableChannelClass))

typedef struct _DVBCableChannel DVBCableChannel;
typedef struct _DVBCableChannelClass DVBCableChannelClass;

#define DVB_TYPE_DVB_SRC_INVERSION (dvb_dvb_src_inversion_get_type ())

#define DVB_TYPE_DVB_SRC_BANDWIDTH (dvb_dvb_src_bandwidth_get_type ())

#define DVB_TYPE_DVB_SRC_CODE_RATE (dvb_dvb_src_code_rate_get_type ())

#define DVB_TYPE_DVB_SRC_MODULATION (dvb_dvb_src_modulation_get_type ())

#define DVB_TYPE_DVB_SRC_TRANSMISSION_MODE (dvb_dvb_src_transmission_mode_get_type ())

#define DVB_TYPE_DVB_SRC_GUARD (dvb_dvb_src_guard_get_type ())

#define DVB_TYPE_DVB_SRC_HIERARCHY (dvb_dvb_src_hierarchy_get_type ())

struct _DVBChannelListReader {
	GObject parent_instance;
	DVBChannelListReaderPrivate * priv;
};

struct _DVBChannelListReaderClass {
	GObjectClass parent_class;
};

typedef enum  {
	DVB_ADAPTER_TYPE_UNKNOWN,
	DVB_ADAPTER_TYPE_DVB_T,
	DVB_ADAPTER_TYPE_DVB_S,
	DVB_ADAPTER_TYPE_DVB_C
} DVBAdapterType;

struct _DVBChannelListReaderPrivate {
	GFile* _ChannelFile;
	DVBAdapterType _Type;
	guint _GroupId;
};

typedef enum  {
	DVB_DVB_SRC_INVERSION_INVERSION_OFF,
	DVB_DVB_SRC_INVERSION_INVERSION_ON,
	DVB_DVB_SRC_INVERSION_INVERSION_AUTO
} DVBDvbSrcInversion;

typedef enum  {
	DVB_DVB_SRC_BANDWIDTH_BANDWIDTH_8_MHZ,
	DVB_DVB_SRC_BANDWIDTH_BANDWIDTH_7_MHZ,
	DVB_DVB_SRC_BANDWIDTH_BANDWIDTH_6_MHZ,
	DVB_DVB_SRC_BANDWIDTH_BANDWIDTH_AUTO
} DVBDvbSrcBandwidth;

typedef enum  {
	DVB_DVB_SRC_CODE_RATE_FEC_NONE,
	DVB_DVB_SRC_CODE_RATE_FEC_1_2,
	DVB_DVB_SRC_CODE_RATE_FEC_2_3,
	DVB_DVB_SRC_CODE_RATE_FEC_3_4,
	DVB_DVB_SRC_CODE_RATE_FEC_4_5,
	DVB_DVB_SRC_CODE_RATE_FEC_5_6,
	DVB_DVB_SRC_CODE_RATE_FEC_6_7,
	DVB_DVB_SRC_CODE_RATE_FEC_7_8,
	DVB_DVB_SRC_CODE_RATE_FEC_8_9,
	DVB_DVB_SRC_CODE_RATE_FEC_AUTO
} DVBDvbSrcCodeRate;

typedef enum  {
	DVB_DVB_SRC_MODULATION_QPSK,
	DVB_DVB_SRC_MODULATION_QAM_16,
	DVB_DVB_SRC_MODULATION_QAM_32,
	DVB_DVB_SRC_MODULATION_QAM_64,
	DVB_DVB_SRC_MODULATION_QAM_128,
	DVB_DVB_SRC_MODULATION_QAM_256,
	DVB_DVB_SRC_MODULATION_QAM_AUTO
} DVBDvbSrcModulation;

typedef enum  {
	DVB_DVB_SRC_TRANSMISSION_MODE_TRANSMISSION_MODE_2K,
	DVB_DVB_SRC_TRANSMISSION_MODE_TRANSMISSION_MODE_8K,
	DVB_DVB_SRC_TRANSMISSION_MODE_TRANSMISSION_MODE_AUTO
} DVBDvbSrcTransmissionMode;

typedef enum  {
	DVB_DVB_SRC_GUARD_GUARD_INTERVAL_1_32,
	DVB_DVB_SRC_GUARD_GUARD_INTERVAL_1_16,
	DVB_DVB_SRC_GUARD_GUARD_INTERVAL_1_8,
	DVB_DVB_SRC_GUARD_GUARD_INTERVAL_1_4,
	DVB_DVB_SRC_GUARD_GUARD_INTERVAL_AUTO
} DVBDvbSrcGuard;

typedef enum  {
	DVB_DVB_SRC_HIERARCHY_HIERARCHY_NONE,
	DVB_DVB_SRC_HIERARCHY_HIERARCHY_1,
	DVB_DVB_SRC_HIERARCHY_HIERARCHY_2,
	DVB_DVB_SRC_HIERARCHY_HIERARCHY_4,
	DVB_DVB_SRC_HIERARCHY_HIERARCHY_AUTO
} DVBDvbSrcHierarchy;


static gpointer dvb_channel_list_reader_parent_class = NULL;

GType dvb_channel_list_reader_get_type (void);
GType dvb_adapter_type_get_type (void);
#define DVB_CHANNEL_LIST_READER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), DVB_TYPE_CHANNEL_LIST_READER, DVBChannelListReaderPrivate))
enum  {
	DVB_CHANNEL_LIST_READER_DUMMY_PROPERTY,
	DVB_CHANNEL_LIST_READER_CHANNEL_FILE,
	DVB_CHANNEL_LIST_READER_TYPE,
	DVB_CHANNEL_LIST_READER_GROUP_ID
};
static void dvb_channel_list_reader_set_ChannelFile (DVBChannelListReader* self, GFile* value);
static void dvb_channel_list_reader_set_Type (DVBChannelListReader* self, DVBAdapterType value);
static void dvb_channel_list_reader_set_GroupId (DVBChannelListReader* self, guint value);
DVBChannelListReader* dvb_channel_list_reader_new (GFile* file, DVBAdapterType type, guint group_id);
DVBChannelListReader* dvb_channel_list_reader_construct (GType object_type, GFile* file, DVBAdapterType type, guint group_id);
char* dvb_utils_read_file_contents (GFile* file, GError** error);
GFile* dvb_channel_list_reader_get_ChannelFile (DVBChannelListReader* self);
DVBChannelList* dvb_channel_list_new (GFile* channels);
DVBChannelList* dvb_channel_list_construct (GType object_type, GFile* channels);
GType dvb_channel_list_get_type (void);
GType dvb_channel_get_type (void);
static DVBChannel* dvb_channel_list_reader_parse_line (DVBChannelListReader* self, const char* line);
void dvb_channel_list_add (DVBChannelList* self, DVBChannel* channel);
DVBChannelList* dvb_channel_list_reader_read (DVBChannelListReader* self, GError** error);
DVBAdapterType dvb_channel_list_reader_get_Type (DVBChannelListReader* self);
GType dvb_terrestrial_channel_get_type (void);
static DVBTerrestrialChannel* dvb_channel_list_reader_parse_terrestrial_channel (DVBChannelListReader* self, const char* line);
GType dvb_satellite_channel_get_type (void);
static DVBSatelliteChannel* dvb_channel_list_reader_parse_satellite_channel (DVBChannelListReader* self, const char* line);
GType dvb_cable_channel_get_type (void);
static DVBCableChannel* dvb_channel_list_reader_parse_cable_channel (DVBChannelListReader* self, const char* line);
gboolean dvb_channel_is_valid (DVBChannel* self);
char* dvb_channel_to_string (DVBChannel* self);
DVBTerrestrialChannel* dvb_terrestrial_channel_new (void);
DVBTerrestrialChannel* dvb_terrestrial_channel_construct (GType object_type);
guint dvb_channel_list_reader_get_GroupId (DVBChannelListReader* self);
void dvb_channel_set_GroupId (DVBChannel* self, guint value);
void dvb_channel_set_Name (DVBChannel* self, const char* value);
void dvb_channel_set_Frequency (DVBChannel* self, guint value);
static gboolean dvb_channel_list_reader_get_value_with_prefix (GType enumtype, const char* name, const char* prefix, gint* val);
GType dvb_dvb_src_inversion_get_type (void);
void dvb_terrestrial_channel_set_Inversion (DVBTerrestrialChannel* self, DVBDvbSrcInversion value);
GType dvb_dvb_src_bandwidth_get_type (void);
void dvb_terrestrial_channel_set_Bandwidth (DVBTerrestrialChannel* self, DVBDvbSrcBandwidth value);
GType dvb_dvb_src_code_rate_get_type (void);
void dvb_terrestrial_channel_set_CodeRateHP (DVBTerrestrialChannel* self, DVBDvbSrcCodeRate value);
void dvb_terrestrial_channel_set_CodeRateLP (DVBTerrestrialChannel* self, DVBDvbSrcCodeRate value);
GType dvb_dvb_src_modulation_get_type (void);
void dvb_terrestrial_channel_set_Constellation (DVBTerrestrialChannel* self, DVBDvbSrcModulation value);
GType dvb_dvb_src_transmission_mode_get_type (void);
void dvb_terrestrial_channel_set_TransmissionMode (DVBTerrestrialChannel* self, DVBDvbSrcTransmissionMode value);
GType dvb_dvb_src_guard_get_type (void);
void dvb_terrestrial_channel_set_GuardInterval (DVBTerrestrialChannel* self, DVBDvbSrcGuard value);
GType dvb_dvb_src_hierarchy_get_type (void);
void dvb_terrestrial_channel_set_Hierarchy (DVBTerrestrialChannel* self, DVBDvbSrcHierarchy value);
void dvb_channel_set_VideoPID (DVBChannel* self, guint value);
GeeList* dvb_channel_get_AudioPIDs (DVBChannel* self);
void dvb_channel_set_Sid (DVBChannel* self, guint value);
DVBSatelliteChannel* dvb_satellite_channel_new (void);
DVBSatelliteChannel* dvb_satellite_channel_construct (GType object_type);
void dvb_satellite_channel_set_Polarization (DVBSatelliteChannel* self, const char* value);
void dvb_satellite_channel_set_DiseqcSource (DVBSatelliteChannel* self, gint value);
void dvb_satellite_channel_set_SymbolRate (DVBSatelliteChannel* self, guint value);
DVBCableChannel* dvb_cable_channel_new (void);
DVBCableChannel* dvb_cable_channel_construct (GType object_type);
void dvb_cable_channel_set_Inversion (DVBCableChannel* self, DVBDvbSrcInversion value);
void dvb_cable_channel_set_SymbolRate (DVBCableChannel* self, guint value);
void dvb_cable_channel_set_CodeRate (DVBCableChannel* self, DVBDvbSrcCodeRate value);
void dvb_cable_channel_set_Modulation (DVBCableChannel* self, DVBDvbSrcModulation value);
gboolean dvb_utils_get_value_by_name_from_enum (GType enumtype, const char* name, gint* evalue);
static void dvb_channel_list_reader_finalize (GObject* obj);
static void dvb_channel_list_reader_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void dvb_channel_list_reader_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);
static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static gint _vala_array_length (gpointer array);



DVBChannelListReader* dvb_channel_list_reader_construct (GType object_type, GFile* file, DVBAdapterType type, guint group_id) {
	GParameter * __params;
	GParameter * __params_it;
	DVBChannelListReader * self;
	g_return_val_if_fail (file != NULL, NULL);
	__params = g_new0 (GParameter, 3);
	__params_it = __params;
	__params_it->name = "ChannelFile";
	g_value_init (&__params_it->value, G_TYPE_FILE);
	g_value_set_object (&__params_it->value, file);
	__params_it++;
	__params_it->name = "Type";
	g_value_init (&__params_it->value, DVB_TYPE_ADAPTER_TYPE);
	g_value_set_enum (&__params_it->value, type);
	__params_it++;
	__params_it->name = "GroupId";
	g_value_init (&__params_it->value, G_TYPE_UINT);
	g_value_set_uint (&__params_it->value, group_id);
	__params_it++;
	self = g_object_newv (object_type, __params_it - __params, __params);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


DVBChannelListReader* dvb_channel_list_reader_new (GFile* file, DVBAdapterType type, guint group_id) {
	return dvb_channel_list_reader_construct (DVB_TYPE_CHANNEL_LIST_READER, file, type, group_id);
}


DVBChannelList* dvb_channel_list_reader_read (DVBChannelListReader* self, GError** error) {
	DVBChannelList* result;
	GError * _inner_error_;
	char* contents;
	DVBChannelList* channels;
	g_return_val_if_fail (self != NULL, NULL);
	_inner_error_ = NULL;
	contents = dvb_utils_read_file_contents (self->priv->_ChannelFile, &_inner_error_);
	if (_inner_error_ != NULL) {
		g_propagate_error (error, _inner_error_);
		return NULL;
	}
	if (contents == NULL) {
		result = NULL;
		contents = (g_free (contents), NULL);
		return result;
	}
	channels = dvb_channel_list_new (self->priv->_ChannelFile);
	{
		char** _tmp0_;
		char** line_collection;
		int line_collection_length1;
		int line_it;
		_tmp0_ = NULL;
		line_collection = _tmp0_ = g_strsplit (contents, "\n", 0);
		line_collection_length1 = _vala_array_length (_tmp0_);
		for (line_it = 0; line_it < _vala_array_length (_tmp0_); line_it = line_it + 1) {
			const char* _tmp1_;
			char* line;
			_tmp1_ = NULL;
			line = (_tmp1_ = line_collection[line_it], (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_));
			{
				if (strlen (line) > 0) {
					DVBChannel* c;
					c = dvb_channel_list_reader_parse_line (self, line);
					if (c != NULL) {
						dvb_channel_list_add (channels, c);
					} else {
						g_warning ("ChannelListReader.vala:48: Could not parse channel");
					}
					(c == NULL) ? NULL : (c = (g_object_unref (c), NULL));
				}
				line = (g_free (line), NULL);
			}
		}
		line_collection = (_vala_array_free (line_collection, line_collection_length1, (GDestroyNotify) g_free), NULL);
	}
	result = channels;
	contents = (g_free (contents), NULL);
	return result;
}


static DVBChannel* dvb_channel_list_reader_parse_line (DVBChannelListReader* self, const char* line) {
	DVBChannel* result;
	DVBChannel* c;
	gboolean _tmp3_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (line != NULL, NULL);
	c = NULL;
	switch (self->priv->_Type) {
		case DVB_ADAPTER_TYPE_DVB_T:
		{
			DVBChannel* _tmp0_;
			_tmp0_ = NULL;
			c = (_tmp0_ = (DVBChannel*) dvb_channel_list_reader_parse_terrestrial_channel (self, line), (c == NULL) ? NULL : (c = (g_object_unref (c), NULL)), _tmp0_);
			break;
		}
		case DVB_ADAPTER_TYPE_DVB_S:
		{
			DVBChannel* _tmp1_;
			_tmp1_ = NULL;
			c = (_tmp1_ = (DVBChannel*) dvb_channel_list_reader_parse_satellite_channel (self, line), (c == NULL) ? NULL : (c = (g_object_unref (c), NULL)), _tmp1_);
			break;
		}
		case DVB_ADAPTER_TYPE_DVB_C:
		{
			DVBChannel* _tmp2_;
			_tmp2_ = NULL;
			c = (_tmp2_ = (DVBChannel*) dvb_channel_list_reader_parse_cable_channel (self, line), (c == NULL) ? NULL : (c = (g_object_unref (c), NULL)), _tmp2_);
			break;
		}
		default:
		{
			g_critical ("ChannelListReader.vala:71: Unknown adapter type");
			break;
		}
	}
	_tmp3_ = FALSE;
	if (c != NULL) {
		_tmp3_ = dvb_channel_is_valid (c);
	} else {
		_tmp3_ = FALSE;
	}
	if (_tmp3_) {
		result = c;
		return result;
	} else {
		char* _tmp4_;
		const char* _tmp7_;
		char* val;
		_tmp4_ = NULL;
		if (c == NULL) {
			char* _tmp5_;
			_tmp5_ = NULL;
			_tmp4_ = (_tmp5_ = g_strdup ("(null)"), _tmp4_ = (g_free (_tmp4_), NULL), _tmp5_);
		} else {
			char* _tmp6_;
			_tmp6_ = NULL;
			_tmp4_ = (_tmp6_ = dvb_channel_to_string (c), _tmp4_ = (g_free (_tmp4_), NULL), _tmp6_);
		}
		_tmp7_ = NULL;
		val = (_tmp7_ = _tmp4_, (_tmp7_ == NULL) ? NULL : g_strdup (_tmp7_));
		g_warning ("ChannelListReader.vala:79: Channel is not valid: %s", val);
		result = NULL;
		_tmp4_ = (g_free (_tmp4_), NULL);
		val = (g_free (val), NULL);
		(c == NULL) ? NULL : (c = (g_object_unref (c), NULL));
		return result;
	}
	(c == NULL) ? NULL : (c = (g_object_unref (c), NULL));
}


/**
         * @line: The line to parse
         * @returns: #TerrestrialChannel representing that line
         * 
         * A line looks like
         * Das Erste:212500000:INVERSION_AUTO:BANDWIDTH_7_MHZ:FEC_3_4:FEC_1_2:QAM_16:TRANSMISSION_MODE_8K:GUARD_INTERVAL_1_4:HIERARCHY_NONE:513:514:32
         */
static DVBTerrestrialChannel* dvb_channel_list_reader_parse_terrestrial_channel (DVBChannelListReader* self, const char* line) {
	DVBTerrestrialChannel* result;
	DVBTerrestrialChannel* channel;
	char** _tmp1_;
	gint fields_size;
	gint fields_length1;
	char** _tmp0_;
	char** fields;
	gint i;
	char* val;
	gboolean failed;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (line != NULL, NULL);
	channel = dvb_terrestrial_channel_new ();
	dvb_channel_set_GroupId ((DVBChannel*) channel, self->priv->_GroupId);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	fields = (_tmp1_ = _tmp0_ = g_strsplit (line, ":", 0), fields_length1 = _vala_array_length (_tmp0_), fields_size = fields_length1, _tmp1_);
	i = 0;
	val = NULL;
	failed = FALSE;
	while (TRUE) {
		char* _tmp3_;
		const char* _tmp2_;
		_tmp3_ = NULL;
		_tmp2_ = NULL;
		if (!((val = (_tmp3_ = (_tmp2_ = fields[i], (_tmp2_ == NULL) ? NULL : g_strdup (_tmp2_)), val = (g_free (val), NULL), _tmp3_)) != NULL)) {
			break;
		}
		if (i == 0) {
			if (g_utf8_validate (val, -1, NULL)) {
				dvb_channel_set_Name ((DVBChannel*) channel, val);
			} else {
				g_warning ("ChannelListReader.vala:105: Bad UTF-8 encoded channel name");
				dvb_channel_set_Name ((DVBChannel*) channel, "Bad encoding");
			}
		} else {
			if (i == 1) {
				dvb_channel_set_Frequency ((DVBChannel*) channel, (guint) atoi (val));
			} else {
				if (i == 2) {
					gint eval;
					eval = 0;
					if (dvb_channel_list_reader_get_value_with_prefix (DVB_TYPE_DVB_SRC_INVERSION, val, "DVB_DVB_SRC_INVERSION_", &eval)) {
						dvb_terrestrial_channel_set_Inversion (channel, (DVBDvbSrcInversion) eval);
					} else {
						failed = TRUE;
						break;
					}
				} else {
					if (i == 3) {
						gint eval;
						eval = 0;
						if (dvb_channel_list_reader_get_value_with_prefix (DVB_TYPE_DVB_SRC_BANDWIDTH, val, "DVB_DVB_SRC_BANDWIDTH_", &eval)) {
							dvb_terrestrial_channel_set_Bandwidth (channel, (DVBDvbSrcBandwidth) eval);
						} else {
							failed = TRUE;
							break;
						}
					} else {
						if (i == 4) {
							gint eval;
							eval = 0;
							if (dvb_channel_list_reader_get_value_with_prefix (DVB_TYPE_DVB_SRC_CODE_RATE, val, "DVB_DVB_SRC_CODE_RATE_", &eval)) {
								dvb_terrestrial_channel_set_CodeRateHP (channel, (DVBDvbSrcCodeRate) eval);
							} else {
								failed = TRUE;
								break;
							}
						} else {
							if (i == 5) {
								gint eval;
								eval = 0;
								if (dvb_channel_list_reader_get_value_with_prefix (DVB_TYPE_DVB_SRC_CODE_RATE, val, "DVB_DVB_SRC_CODE_RATE_", &eval)) {
									dvb_terrestrial_channel_set_CodeRateLP (channel, (DVBDvbSrcCodeRate) eval);
								} else {
									failed = TRUE;
									break;
								}
							} else {
								if (i == 6) {
									gint eval;
									eval = 0;
									if (dvb_channel_list_reader_get_value_with_prefix (DVB_TYPE_DVB_SRC_MODULATION, val, "DVB_DVB_SRC_MODULATION_", &eval)) {
										dvb_terrestrial_channel_set_Constellation (channel, (DVBDvbSrcModulation) eval);
									} else {
										failed = TRUE;
										break;
									}
								} else {
									if (i == 7) {
										gint eval;
										eval = 0;
										if (dvb_channel_list_reader_get_value_with_prefix (DVB_TYPE_DVB_SRC_TRANSMISSION_MODE, val, "DVB_DVB_SRC_TRANSMISSION_MODE_", &eval)) {
											dvb_terrestrial_channel_set_TransmissionMode (channel, (DVBDvbSrcTransmissionMode) eval);
										} else {
											failed = TRUE;
											break;
										}
									} else {
										if (i == 8) {
											gint eval;
											eval = 0;
											if (dvb_channel_list_reader_get_value_with_prefix (DVB_TYPE_DVB_SRC_GUARD, val, "DVB_DVB_SRC_GUARD_", &eval)) {
												dvb_terrestrial_channel_set_GuardInterval (channel, (DVBDvbSrcGuard) eval);
											} else {
												failed = TRUE;
												break;
											}
										} else {
											if (i == 9) {
												gint eval;
												eval = 0;
												if (dvb_channel_list_reader_get_value_with_prefix (DVB_TYPE_DVB_SRC_HIERARCHY, val, "DVB_DVB_SRC_HIERARCHY_", &eval)) {
													dvb_terrestrial_channel_set_Hierarchy (channel, (DVBDvbSrcHierarchy) eval);
												} else {
													failed = TRUE;
													break;
												}
											} else {
												if (i == 10) {
													dvb_channel_set_VideoPID ((DVBChannel*) channel, (guint) atoi (val));
												} else {
													if (i == 11) {
														gee_collection_add ((GeeCollection*) dvb_channel_get_AudioPIDs ((DVBChannel*) channel), GUINT_TO_POINTER ((guint) atoi (val)));
													} else {
														if (i == 12) {
															dvb_channel_set_Sid ((DVBChannel*) channel, (guint) atoi (val));
														}
													}
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
		i++;
	}
	if (failed) {
		result = NULL;
		(channel == NULL) ? NULL : (channel = (g_object_unref (channel), NULL));
		fields = (_vala_array_free (fields, fields_length1, (GDestroyNotify) g_free), NULL);
		val = (g_free (val), NULL);
		return result;
	} else {
		result = channel;
		fields = (_vala_array_free (fields, fields_length1, (GDestroyNotify) g_free), NULL);
		val = (g_free (val), NULL);
		return result;
	}
	(channel == NULL) ? NULL : (channel = (g_object_unref (channel), NULL));
	fields = (_vala_array_free (fields, fields_length1, (GDestroyNotify) g_free), NULL);
	val = (g_free (val), NULL);
}


/**
         *
         * A line looks like
         * Das Erste:11836:h:0:27500:101:102:28106
         */
static DVBSatelliteChannel* dvb_channel_list_reader_parse_satellite_channel (DVBChannelListReader* self, const char* line) {
	DVBSatelliteChannel* result;
	DVBSatelliteChannel* channel;
	char** _tmp1_;
	gint fields_size;
	gint fields_length1;
	char** _tmp0_;
	char** fields;
	gint i;
	char* val;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (line != NULL, NULL);
	channel = dvb_satellite_channel_new ();
	dvb_channel_set_GroupId ((DVBChannel*) channel, self->priv->_GroupId);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	fields = (_tmp1_ = _tmp0_ = g_strsplit (line, ":", 0), fields_length1 = _vala_array_length (_tmp0_), fields_size = fields_length1, _tmp1_);
	i = 0;
	val = NULL;
	while (TRUE) {
		char* _tmp3_;
		const char* _tmp2_;
		_tmp3_ = NULL;
		_tmp2_ = NULL;
		if (!((val = (_tmp3_ = (_tmp2_ = fields[i], (_tmp2_ == NULL) ? NULL : g_strdup (_tmp2_)), val = (g_free (val), NULL), _tmp3_)) != NULL)) {
			break;
		}
		if (i == 0) {
			if (g_utf8_validate (val, -1, NULL)) {
				dvb_channel_set_Name ((DVBChannel*) channel, val);
			} else {
				g_warning ("ChannelListReader.vala:215: Bad UTF-8 encoded channel name");
				dvb_channel_set_Name ((DVBChannel*) channel, "Bad encoding");
			}
		} else {
			if (i == 1) {
				/* frequency is stored in MHz*/
				dvb_channel_set_Frequency ((DVBChannel*) channel, (guint) (atoi (val) * 1000));
			} else {
				if (i == 2) {
					dvb_satellite_channel_set_Polarization (channel, val);
				} else {
					if (i == 3) {
						/* Sat number*/
						dvb_satellite_channel_set_DiseqcSource (channel, atoi (val));
					} else {
						if (i == 4) {
							/* symbol rate is stored in kBaud*/
							dvb_satellite_channel_set_SymbolRate (channel, (guint) atoi (val));
						} else {
							if (i == 5) {
								dvb_channel_set_VideoPID ((DVBChannel*) channel, (guint) atoi (val));
							} else {
								if (i == 6) {
									gee_collection_add ((GeeCollection*) dvb_channel_get_AudioPIDs ((DVBChannel*) channel), GUINT_TO_POINTER ((guint) atoi (val)));
								} else {
									if (i == 7) {
										dvb_channel_set_Sid ((DVBChannel*) channel, (guint) atoi (val));
									}
								}
							}
						}
					}
				}
			}
		}
		i++;
	}
	result = channel;
	fields = (_vala_array_free (fields, fields_length1, (GDestroyNotify) g_free), NULL);
	val = (g_free (val), NULL);
	return result;
}


/**
         *
         * line looks like
         * ProSieben:330000000:INVERSION_AUTO:6900000:FEC_NONE:QAM_64:255:256:898
         */
static DVBCableChannel* dvb_channel_list_reader_parse_cable_channel (DVBChannelListReader* self, const char* line) {
	DVBCableChannel* result;
	DVBCableChannel* channel;
	char** _tmp1_;
	gint fields_size;
	gint fields_length1;
	char** _tmp0_;
	char** fields;
	gint i;
	char* val;
	gboolean failed;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (line != NULL, NULL);
	channel = dvb_cable_channel_new ();
	dvb_channel_set_GroupId ((DVBChannel*) channel, self->priv->_GroupId);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	fields = (_tmp1_ = _tmp0_ = g_strsplit (line, ":", 0), fields_length1 = _vala_array_length (_tmp0_), fields_size = fields_length1, _tmp1_);
	i = 0;
	val = NULL;
	failed = FALSE;
	while (TRUE) {
		char* _tmp3_;
		const char* _tmp2_;
		_tmp3_ = NULL;
		_tmp2_ = NULL;
		if (!((val = (_tmp3_ = (_tmp2_ = fields[i], (_tmp2_ == NULL) ? NULL : g_strdup (_tmp2_)), val = (g_free (val), NULL), _tmp3_)) != NULL)) {
			break;
		}
		if (i == 0) {
			if (g_utf8_validate (val, -1, NULL)) {
				dvb_channel_set_Name ((DVBChannel*) channel, val);
			} else {
				g_warning ("ChannelListReader.vala:262: Bad UTF-8 encoded channel name");
				dvb_channel_set_Name ((DVBChannel*) channel, "Bad encoding");
			}
		} else {
			if (i == 1) {
				dvb_channel_set_Frequency ((DVBChannel*) channel, (guint) atoi (val));
			} else {
				if (i == 2) {
					gint eval;
					eval = 0;
					if (dvb_channel_list_reader_get_value_with_prefix (DVB_TYPE_DVB_SRC_INVERSION, val, "DVB_DVB_SRC_INVERSION_", &eval)) {
						dvb_cable_channel_set_Inversion (channel, (DVBDvbSrcInversion) eval);
					} else {
						failed = TRUE;
						break;
					}
				} else {
					if (i == 3) {
						dvb_cable_channel_set_SymbolRate (channel, (guint) (atoi (val) / 1000));
					} else {
						if (i == 4) {
							gint eval;
							eval = 0;
							if (dvb_channel_list_reader_get_value_with_prefix (DVB_TYPE_DVB_SRC_CODE_RATE, val, "DVB_DVB_SRC_CODE_RATE_", &eval)) {
								dvb_cable_channel_set_CodeRate (channel, (DVBDvbSrcCodeRate) eval);
							} else {
								failed = TRUE;
								break;
							}
						} else {
							if (i == 5) {
								gint eval;
								eval = 0;
								if (dvb_channel_list_reader_get_value_with_prefix (DVB_TYPE_DVB_SRC_MODULATION, val, "DVB_DVB_SRC_MODULATION_", &eval)) {
									dvb_cable_channel_set_Modulation (channel, (DVBDvbSrcModulation) eval);
								} else {
									failed = TRUE;
									break;
								}
							} else {
								if (i == 6) {
									dvb_channel_set_VideoPID ((DVBChannel*) channel, (guint) atoi (val));
								} else {
									if (i == 7) {
										gee_collection_add ((GeeCollection*) dvb_channel_get_AudioPIDs ((DVBChannel*) channel), GUINT_TO_POINTER ((guint) atoi (val)));
									} else {
										if (i == 8) {
											dvb_channel_set_Sid ((DVBChannel*) channel, (guint) atoi (val));
										}
									}
								}
							}
						}
					}
				}
			}
		}
		i++;
	}
	if (failed) {
		result = NULL;
		(channel == NULL) ? NULL : (channel = (g_object_unref (channel), NULL));
		fields = (_vala_array_free (fields, fields_length1, (GDestroyNotify) g_free), NULL);
		val = (g_free (val), NULL);
		return result;
	} else {
		result = channel;
		fields = (_vala_array_free (fields, fields_length1, (GDestroyNotify) g_free), NULL);
		val = (g_free (val), NULL);
		return result;
	}
	(channel == NULL) ? NULL : (channel = (g_object_unref (channel), NULL));
	fields = (_vala_array_free (fields, fields_length1, (GDestroyNotify) g_free), NULL);
	val = (g_free (val), NULL);
}


static gboolean dvb_channel_list_reader_get_value_with_prefix (GType enumtype, const char* name, const char* prefix, gint* val) {
	gboolean result;
	char* _tmp0_;
	gboolean _tmp1_;
	g_return_val_if_fail (name != NULL, FALSE);
	g_return_val_if_fail (prefix != NULL, FALSE);
	_tmp0_ = NULL;
	result = (_tmp1_ = dvb_utils_get_value_by_name_from_enum (enumtype, _tmp0_ = g_strconcat (prefix, name, NULL), &(*val)), _tmp0_ = (g_free (_tmp0_), NULL), _tmp1_);
	return result;
}


GFile* dvb_channel_list_reader_get_ChannelFile (DVBChannelListReader* self) {
	GFile* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_ChannelFile;
	return result;
}


static void dvb_channel_list_reader_set_ChannelFile (DVBChannelListReader* self, GFile* value) {
	GFile* _tmp1_;
	GFile* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_ChannelFile = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_)), (self->priv->_ChannelFile == NULL) ? NULL : (self->priv->_ChannelFile = (g_object_unref (self->priv->_ChannelFile), NULL)), _tmp1_);
	g_object_notify ((GObject *) self, "ChannelFile");
}


DVBAdapterType dvb_channel_list_reader_get_Type (DVBChannelListReader* self) {
	DVBAdapterType result;
	g_return_val_if_fail (self != NULL, 0);
	result = self->priv->_Type;
	return result;
}


static void dvb_channel_list_reader_set_Type (DVBChannelListReader* self, DVBAdapterType value) {
	g_return_if_fail (self != NULL);
	self->priv->_Type = value;
	g_object_notify ((GObject *) self, "Type");
}


guint dvb_channel_list_reader_get_GroupId (DVBChannelListReader* self) {
	guint result;
	g_return_val_if_fail (self != NULL, 0U);
	result = self->priv->_GroupId;
	return result;
}


static void dvb_channel_list_reader_set_GroupId (DVBChannelListReader* self, guint value) {
	g_return_if_fail (self != NULL);
	self->priv->_GroupId = value;
	g_object_notify ((GObject *) self, "GroupId");
}


static void dvb_channel_list_reader_class_init (DVBChannelListReaderClass * klass) {
	dvb_channel_list_reader_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (DVBChannelListReaderPrivate));
	G_OBJECT_CLASS (klass)->get_property = dvb_channel_list_reader_get_property;
	G_OBJECT_CLASS (klass)->set_property = dvb_channel_list_reader_set_property;
	G_OBJECT_CLASS (klass)->finalize = dvb_channel_list_reader_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_CHANNEL_LIST_READER_CHANNEL_FILE, g_param_spec_object ("ChannelFile", "ChannelFile", "ChannelFile", G_TYPE_FILE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_CHANNEL_LIST_READER_TYPE, g_param_spec_enum ("Type", "Type", "Type", DVB_TYPE_ADAPTER_TYPE, 0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_CHANNEL_LIST_READER_GROUP_ID, g_param_spec_uint ("GroupId", "GroupId", "GroupId", 0, G_MAXUINT, 0U, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
}


static void dvb_channel_list_reader_instance_init (DVBChannelListReader * self) {
	self->priv = DVB_CHANNEL_LIST_READER_GET_PRIVATE (self);
}


static void dvb_channel_list_reader_finalize (GObject* obj) {
	DVBChannelListReader * self;
	self = DVB_CHANNEL_LIST_READER (obj);
	(self->priv->_ChannelFile == NULL) ? NULL : (self->priv->_ChannelFile = (g_object_unref (self->priv->_ChannelFile), NULL));
	G_OBJECT_CLASS (dvb_channel_list_reader_parent_class)->finalize (obj);
}


GType dvb_channel_list_reader_get_type (void) {
	static GType dvb_channel_list_reader_type_id = 0;
	if (dvb_channel_list_reader_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (DVBChannelListReaderClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_channel_list_reader_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBChannelListReader), 0, (GInstanceInitFunc) dvb_channel_list_reader_instance_init, NULL };
		dvb_channel_list_reader_type_id = g_type_register_static (G_TYPE_OBJECT, "DVBChannelListReader", &g_define_type_info, 0);
	}
	return dvb_channel_list_reader_type_id;
}


static void dvb_channel_list_reader_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	DVBChannelListReader * self;
	gpointer boxed;
	self = DVB_CHANNEL_LIST_READER (object);
	switch (property_id) {
		case DVB_CHANNEL_LIST_READER_CHANNEL_FILE:
		g_value_set_object (value, dvb_channel_list_reader_get_ChannelFile (self));
		break;
		case DVB_CHANNEL_LIST_READER_TYPE:
		g_value_set_enum (value, dvb_channel_list_reader_get_Type (self));
		break;
		case DVB_CHANNEL_LIST_READER_GROUP_ID:
		g_value_set_uint (value, dvb_channel_list_reader_get_GroupId (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void dvb_channel_list_reader_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	DVBChannelListReader * self;
	self = DVB_CHANNEL_LIST_READER (object);
	switch (property_id) {
		case DVB_CHANNEL_LIST_READER_CHANNEL_FILE:
		dvb_channel_list_reader_set_ChannelFile (self, g_value_get_object (value));
		break;
		case DVB_CHANNEL_LIST_READER_TYPE:
		dvb_channel_list_reader_set_Type (self, g_value_get_enum (value));
		break;
		case DVB_CHANNEL_LIST_READER_GROUP_ID:
		dvb_channel_list_reader_set_GroupId (self, g_value_get_uint (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	_vala_array_destroy (array, array_length, destroy_func);
	g_free (array);
}


static gint _vala_array_length (gpointer array) {
	int length;
	length = 0;
	if (array) {
		while (((gpointer*) array)[length]) {
			length++;
		}
	}
	return length;
}




