/*
 * Copyright (C) 2008,2009 Sebastian Pölsterl
 *
 * This file is part of GNOME DVB Daemon.
 *
 * GNOME DVB Daemon is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GNOME DVB Daemon is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNOME DVB Daemon.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <dbus/dbus-glib.h>
#include <gio/gio.h>
#include <dbus/dbus.h>


#define DVB_TYPE_ID_BUS_MANAGER (dvb_id_bus_manager_get_type ())
#define DVB_ID_BUS_MANAGER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_ID_BUS_MANAGER, DVBIDBusManager))
#define DVB_IS_ID_BUS_MANAGER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_ID_BUS_MANAGER))
#define DVB_ID_BUS_MANAGER_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), DVB_TYPE_ID_BUS_MANAGER, DVBIDBusManagerIface))

typedef struct _DVBIDBusManager DVBIDBusManager;
typedef struct _DVBIDBusManagerIface DVBIDBusManagerIface;

#define DVB_TYPE_MANAGER (dvb_manager_get_type ())
#define DVB_MANAGER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_MANAGER, DVBManager))
#define DVB_MANAGER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_MANAGER, DVBManagerClass))
#define DVB_IS_MANAGER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_MANAGER))
#define DVB_IS_MANAGER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_MANAGER))
#define DVB_MANAGER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_MANAGER, DVBManagerClass))

typedef struct _DVBManager DVBManager;
typedef struct _DVBManagerClass DVBManagerClass;
typedef struct _DVBManagerPrivate DVBManagerPrivate;

#define DVB_TYPE_SCANNER (dvb_scanner_get_type ())
#define DVB_SCANNER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_SCANNER, DVBScanner))
#define DVB_SCANNER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_SCANNER, DVBScannerClass))
#define DVB_IS_SCANNER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_SCANNER))
#define DVB_IS_SCANNER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_SCANNER))
#define DVB_SCANNER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_SCANNER, DVBScannerClass))

typedef struct _DVBScanner DVBScanner;
typedef struct _DVBScannerClass DVBScannerClass;

#define DVB_TYPE_DEVICE_GROUP (dvb_device_group_get_type ())
#define DVB_DEVICE_GROUP(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_DEVICE_GROUP, DVBDeviceGroup))
#define DVB_DEVICE_GROUP_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_DEVICE_GROUP, DVBDeviceGroupClass))
#define DVB_IS_DEVICE_GROUP(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_DEVICE_GROUP))
#define DVB_IS_DEVICE_GROUP_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_DEVICE_GROUP))
#define DVB_DEVICE_GROUP_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_DEVICE_GROUP, DVBDeviceGroupClass))

typedef struct _DVBDeviceGroup DVBDeviceGroup;
typedef struct _DVBDeviceGroupClass DVBDeviceGroupClass;

#define DVB_TYPE_DEVICE (dvb_device_get_type ())
#define DVB_DEVICE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_DEVICE, DVBDevice))
#define DVB_DEVICE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_DEVICE, DVBDeviceClass))
#define DVB_IS_DEVICE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_DEVICE))
#define DVB_IS_DEVICE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_DEVICE))
#define DVB_DEVICE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_DEVICE, DVBDeviceClass))

typedef struct _DVBDevice DVBDevice;
typedef struct _DVBDeviceClass DVBDeviceClass;

#define DVB_TYPE_EPG_SCANNER (dvb_epg_scanner_get_type ())
#define DVB_EPG_SCANNER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_EPG_SCANNER, DVBEPGScanner))
#define DVB_EPG_SCANNER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_EPG_SCANNER, DVBEPGScannerClass))
#define DVB_IS_EPG_SCANNER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_EPG_SCANNER))
#define DVB_IS_EPG_SCANNER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_EPG_SCANNER))
#define DVB_EPG_SCANNER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_EPG_SCANNER, DVBEPGScannerClass))

typedef struct _DVBEPGScanner DVBEPGScanner;
typedef struct _DVBEPGScannerClass DVBEPGScannerClass;

#define DVB_TYPE_ADAPTER_TYPE (dvb_adapter_type_get_type ())

#define DVB_TYPE_TERRESTRIAL_SCANNER (dvb_terrestrial_scanner_get_type ())
#define DVB_TERRESTRIAL_SCANNER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_TERRESTRIAL_SCANNER, DVBTerrestrialScanner))
#define DVB_TERRESTRIAL_SCANNER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_TERRESTRIAL_SCANNER, DVBTerrestrialScannerClass))
#define DVB_IS_TERRESTRIAL_SCANNER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_TERRESTRIAL_SCANNER))
#define DVB_IS_TERRESTRIAL_SCANNER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_TERRESTRIAL_SCANNER))
#define DVB_TERRESTRIAL_SCANNER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_TERRESTRIAL_SCANNER, DVBTerrestrialScannerClass))

typedef struct _DVBTerrestrialScanner DVBTerrestrialScanner;
typedef struct _DVBTerrestrialScannerClass DVBTerrestrialScannerClass;

#define DVB_TYPE_SATELLITE_SCANNER (dvb_satellite_scanner_get_type ())
#define DVB_SATELLITE_SCANNER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_SATELLITE_SCANNER, DVBSatelliteScanner))
#define DVB_SATELLITE_SCANNER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_SATELLITE_SCANNER, DVBSatelliteScannerClass))
#define DVB_IS_SATELLITE_SCANNER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_SATELLITE_SCANNER))
#define DVB_IS_SATELLITE_SCANNER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_SATELLITE_SCANNER))
#define DVB_SATELLITE_SCANNER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_SATELLITE_SCANNER, DVBSatelliteScannerClass))

typedef struct _DVBSatelliteScanner DVBSatelliteScanner;
typedef struct _DVBSatelliteScannerClass DVBSatelliteScannerClass;

#define DVB_TYPE_CABLE_SCANNER (dvb_cable_scanner_get_type ())
#define DVB_CABLE_SCANNER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CABLE_SCANNER, DVBCableScanner))
#define DVB_CABLE_SCANNER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CABLE_SCANNER, DVBCableScannerClass))
#define DVB_IS_CABLE_SCANNER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CABLE_SCANNER))
#define DVB_IS_CABLE_SCANNER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CABLE_SCANNER))
#define DVB_CABLE_SCANNER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CABLE_SCANNER, DVBCableScannerClass))

typedef struct _DVBCableScanner DVBCableScanner;
typedef struct _DVBCableScannerClass DVBCableScannerClass;

#define DVB_TYPE_CONFIG_STORE (dvb_config_store_get_type ())
#define DVB_CONFIG_STORE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CONFIG_STORE, DVBConfigStore))
#define DVB_IS_CONFIG_STORE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CONFIG_STORE))
#define DVB_CONFIG_STORE_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), DVB_TYPE_CONFIG_STORE, DVBConfigStoreIface))

typedef struct _DVBConfigStore DVBConfigStore;
typedef struct _DVBConfigStoreIface DVBConfigStoreIface;

#define DVB_TYPE_ID_BUS_DEVICE_GROUP (dvb_id_bus_device_group_get_type ())
#define DVB_ID_BUS_DEVICE_GROUP(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_ID_BUS_DEVICE_GROUP, DVBIDBusDeviceGroup))
#define DVB_IS_ID_BUS_DEVICE_GROUP(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_ID_BUS_DEVICE_GROUP))
#define DVB_ID_BUS_DEVICE_GROUP_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), DVB_TYPE_ID_BUS_DEVICE_GROUP, DVBIDBusDeviceGroupIface))

typedef struct _DVBIDBusDeviceGroup DVBIDBusDeviceGroup;
typedef struct _DVBIDBusDeviceGroupIface DVBIDBusDeviceGroupIface;

#define DVB_TYPE_CHANNEL_LIST (dvb_channel_list_get_type ())
#define DVB_CHANNEL_LIST(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CHANNEL_LIST, DVBChannelList))
#define DVB_CHANNEL_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CHANNEL_LIST, DVBChannelListClass))
#define DVB_IS_CHANNEL_LIST(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CHANNEL_LIST))
#define DVB_IS_CHANNEL_LIST_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CHANNEL_LIST))
#define DVB_CHANNEL_LIST_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CHANNEL_LIST, DVBChannelListClass))

typedef struct _DVBChannelList DVBChannelList;
typedef struct _DVBChannelListClass DVBChannelListClass;

#define DVB_TYPE_EPG_STORE (dvb_epg_store_get_type ())
#define DVB_EPG_STORE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_EPG_STORE, DVBEPGStore))
#define DVB_IS_EPG_STORE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_EPG_STORE))
#define DVB_EPG_STORE_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), DVB_TYPE_EPG_STORE, DVBEPGStoreIface))

typedef struct _DVBEPGStore DVBEPGStore;
typedef struct _DVBEPGStoreIface DVBEPGStoreIface;

#define DVB_TYPE_EVENT (dvb_event_get_type ())
#define DVB_EVENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_EVENT, DVBEvent))
#define DVB_EVENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_EVENT, DVBEventClass))
#define DVB_IS_EVENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_EVENT))
#define DVB_IS_EVENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_EVENT))
#define DVB_EVENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_EVENT, DVBEventClass))

typedef struct _DVBEvent DVBEvent;
typedef struct _DVBEventClass DVBEventClass;
typedef struct _DBusObjectVTable _DBusObjectVTable;

struct _DVBIDBusManagerIface {
	GTypeInterface parent_iface;
	char** (*GetScannerForDevice) (DVBIDBusManager* self, guint adapter, guint frontend, int* result_length1);
	char** (*GetRegisteredDeviceGroups) (DVBIDBusManager* self, int* result_length1);
	char* (*GetDeviceGroup) (DVBIDBusManager* self, guint group_id);
	gboolean (*AddDeviceToNewGroup) (DVBIDBusManager* self, guint adapter, guint frontend, const char* channels_conf, const char* recordings_dir, const char* name);
	char* (*GetNameOfRegisteredDevice) (DVBIDBusManager* self, guint adapter, guint frontend);
	gint (*GetDeviceGroupSize) (DVBIDBusManager* self);
};

struct _DVBManager {
	GObject parent_instance;
	DVBManagerPrivate * priv;
};

struct _DVBManagerClass {
	GObjectClass parent_class;
};

struct _DVBManagerPrivate {
	GeeHashMap* scanners;
	GStaticRecMutex __lock_scanners;
	GeeHashMap* devices;
	GStaticRecMutex __lock_devices;
	guint device_group_counter;
};

typedef enum  {
	DVB_ADAPTER_TYPE_UNKNOWN,
	DVB_ADAPTER_TYPE_DVB_T,
	DVB_ADAPTER_TYPE_DVB_S,
	DVB_ADAPTER_TYPE_DVB_C
} DVBAdapterType;

struct _DVBConfigStoreIface {
	GTypeInterface parent_iface;
	GeeList* (*get_all_device_groups) (DVBConfigStore* self);
	gboolean (*add_device_group) (DVBConfigStore* self, DVBDeviceGroup* dev_group);
	gboolean (*remove_device_group) (DVBConfigStore* self, DVBDeviceGroup* devgroup);
	gboolean (*add_device_to_group) (DVBConfigStore* self, DVBDevice* dev, DVBDeviceGroup* devgroup);
	gboolean (*remove_device_from_group) (DVBConfigStore* self, DVBDevice* dev, DVBDeviceGroup* devgroup);
	gboolean (*update_from_group) (DVBConfigStore* self, DVBDeviceGroup* devgroup);
};

struct _DVBIDBusDeviceGroupIface {
	GTypeInterface parent_iface;
	char* (*GetType) (DVBIDBusDeviceGroup* self);
	char* (*GetRecorder) (DVBIDBusDeviceGroup* self);
	gboolean (*AddDevice) (DVBIDBusDeviceGroup* self, guint adapter, guint frontend);
	gboolean (*RemoveDevice) (DVBIDBusDeviceGroup* self, guint adapter, guint frontend);
	char* (*GetChannelList) (DVBIDBusDeviceGroup* self);
	char* (*GetName) (DVBIDBusDeviceGroup* self);
	gboolean (*SetName) (DVBIDBusDeviceGroup* self, const char* name);
	char** (*GetMembers) (DVBIDBusDeviceGroup* self, int* result_length1);
	char* (*GetSchedule) (DVBIDBusDeviceGroup* self, guint channel_sid);
	char* (*GetRecordingsDirectory) (DVBIDBusDeviceGroup* self);
	gboolean (*SetRecordingsDirectory) (DVBIDBusDeviceGroup* self, const char* location);
};

struct _DVBEPGStoreIface {
	GTypeInterface parent_iface;
	gboolean (*add_or_update_event) (DVBEPGStore* self, DVBEvent* event, guint channel_sid, guint group_id);
	DVBEvent* (*get_event) (DVBEPGStore* self, guint event_id, guint channel_sid, guint group_id);
	gboolean (*remove_event) (DVBEPGStore* self, guint event_id, guint channel_sid, guint group_id);
	gboolean (*contains_event) (DVBEPGStore* self, DVBEvent* event, guint channel_sid, guint group_id);
	GeeList* (*get_events) (DVBEPGStore* self, guint channel_sid, guint group_id);
	gboolean (*remove_events_of_group) (DVBEPGStore* self, guint group_id);
};

struct _DBusObjectVTable {
	void (*register_object) (DBusConnection*, const char*, void*);
};


static DVBManager* dvb_manager_instance;
static DVBManager* dvb_manager_instance = NULL;
static GStaticRecMutex dvb_manager_instance_mutex;
static GStaticRecMutex dvb_manager_instance_mutex = {0};
static gpointer dvb_manager_parent_class = NULL;
static DVBIDBusManagerIface* dvb_manager_dvb_id_bus_manager_parent_iface = NULL;

GType dvb_id_bus_manager_get_type (void);
GType dvb_manager_get_type (void);
GType dvb_scanner_get_type (void);
GType dvb_device_group_get_type (void);
#define DVB_MANAGER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), DVB_TYPE_MANAGER, DVBManagerPrivate))
enum  {
	DVB_MANAGER_DUMMY_PROPERTY,
	DVB_MANAGER_DEVICE_GROUPS
};
DVBManager* dvb_manager_new (void);
DVBManager* dvb_manager_construct (GType object_type);
DVBManager* dvb_manager_get_instance (void);
void dvb_scanner_Destroy (DVBScanner* self);
void dvb_device_group_destroy (DVBDeviceGroup* self);
void dvb_manager_shutdown (void);
#define DVB_CONSTANTS_DBUS_SCANNER_PATH "/org/gnome/DVB/Scanner/%d/%d"
GType dvb_device_get_type (void);
static DVBDevice* dvb_manager_get_registered_device (DVBManager* self, guint adapter, guint frontend);
DVBDevice* dvb_device_new (guint adapter, guint frontend, gboolean get_type_and_name);
DVBDevice* dvb_device_construct (GType object_type, guint adapter, guint frontend, gboolean get_type_and_name);
static DVBDeviceGroup* dvb_manager_get_device_group_of_device (DVBManager* self, DVBDevice* device);
GType dvb_epg_scanner_get_type (void);
DVBEPGScanner* dvb_device_group_get_epgscanner (DVBDeviceGroup* self);
void dvb_epg_scanner_stop (DVBEPGScanner* self);
GType dvb_adapter_type_get_type (void);
DVBAdapterType dvb_device_get_Type (DVBDevice* self);
DVBTerrestrialScanner* dvb_terrestrial_scanner_new (DVBDevice* device);
DVBTerrestrialScanner* dvb_terrestrial_scanner_construct (GType object_type, DVBDevice* device);
GType dvb_terrestrial_scanner_get_type (void);
DVBSatelliteScanner* dvb_satellite_scanner_new (DVBDevice* device);
DVBSatelliteScanner* dvb_satellite_scanner_construct (GType object_type, DVBDevice* device);
GType dvb_satellite_scanner_get_type (void);
DVBCableScanner* dvb_cable_scanner_new (DVBDevice* device);
DVBCableScanner* dvb_cable_scanner_construct (GType object_type, DVBDevice* device);
GType dvb_cable_scanner_get_type (void);
static void dvb_manager_on_scanner_destroyed (DVBManager* self, DVBScanner* scanner);
static void _dvb_manager_on_scanner_destroyed_dvb_scanner_destroyed (DVBScanner* _sender, gpointer self);
DBusGConnection* dvb_utils_get_dbus_connection (void);
static char** dvb_manager_real_GetScannerForDevice (DVBIDBusManager* base, guint adapter, guint frontend, int* result_length1);
#define DVB_CONSTANTS_DBUS_DEVICE_GROUP_PATH "/org/gnome/DVB/DeviceGroup/%u"
static char* dvb_manager_real_GetDeviceGroup (DVBIDBusManager* base, guint group_id);
static char** dvb_manager_real_GetRegisteredDeviceGroups (DVBIDBusManager* base, int* result_length1);
static DVBDevice* dvb_manager_create_device (guint adapter, guint frontend, const char* channels_conf, const char* recordings_dir, guint group_id);
gboolean dvb_manager_device_is_in_any_group (DVBManager* self, DVBDevice* device);
DVBDeviceGroup* dvb_device_group_new (guint id, DVBDevice* reference_device, gboolean with_epg_scanner);
DVBDeviceGroup* dvb_device_group_construct (GType object_type, guint id, DVBDevice* reference_device, gboolean with_epg_scanner);
void dvb_device_group_set_Name (DVBDeviceGroup* self, const char* value);
gboolean dvb_manager_add_device_group (DVBManager* self, DVBDeviceGroup* devgroup);
static gboolean dvb_manager_real_AddDeviceToNewGroup (DVBIDBusManager* base, guint adapter, guint frontend, const char* channels_conf, const char* recordings_dir, const char* name);
const char* dvb_device_get_Name (DVBDevice* self);
static char* dvb_manager_real_GetNameOfRegisteredDevice (DVBIDBusManager* base, guint adapter, guint frontend);
static gint dvb_manager_real_GetDeviceGroupSize (DVBIDBusManager* base);
guint dvb_device_group_get_Id (DVBDeviceGroup* self);
gint dvb_device_group_get_size (DVBDeviceGroup* self);
DVBAdapterType dvb_device_group_get_Type (DVBDeviceGroup* self);
GType dvb_config_store_get_type (void);
DVBConfigStore* dvb_factory_get_config_store (void);
gboolean dvb_config_store_add_device_group (DVBConfigStore* self, DVBDeviceGroup* dev_group);
static void dvb_manager_on_device_removed_from_group (DVBManager* self, DVBDeviceGroup* devgroup, guint adapter, guint frontend);
static void _dvb_manager_on_device_removed_from_group_dvb_id_bus_device_group_device_removed (DVBDeviceGroup* _sender, guint adapter, guint frontend, gpointer self);
GType dvb_id_bus_device_group_get_type (void);
gboolean main_get_disable_epg_scanner (void);
gboolean dvb_epg_scanner_start (DVBEPGScanner* self);
void dvb_device_set_RecordingsDirectory (DVBDevice* self, GFile* value);
GType dvb_channel_list_get_type (void);
DVBChannelList* dvb_channel_list_restore_from_file (GFile* channelsfile, DVBAdapterType type, guint group_id, GError** error);
void dvb_device_set_Channels (DVBDevice* self, DVBChannelList* value);
DVBDeviceGroup* dvb_manager_get_device_group_if_exists (DVBManager* self, guint group_id);
gboolean dvb_device_group_contains (DVBDeviceGroup* self, DVBDevice* device);
DVBDevice* dvb_scanner_get_Device (DVBScanner* self);
guint dvb_device_get_Adapter (DVBDevice* self);
guint dvb_device_get_Frontend (DVBDevice* self);
gboolean dvb_device_equal (DVBDevice* dev1, DVBDevice* dev2);
gboolean dvb_config_store_remove_device_group (DVBConfigStore* self, DVBDeviceGroup* devgroup);
gpointer dvb_event_ref (gpointer instance);
void dvb_event_unref (gpointer instance);
GParamSpec* dvb_param_spec_event (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void dvb_value_set_event (GValue* value, gpointer v_object);
gpointer dvb_value_get_event (const GValue* value);
GType dvb_event_get_type (void);
GType dvb_epg_store_get_type (void);
DVBEPGStore* dvb_factory_get_epg_store (void);
gboolean dvb_epg_store_remove_events_of_group (DVBEPGStore* self, guint group_id);
GeeCollection* dvb_manager_get_device_groups (DVBManager* self);
static GObject * dvb_manager_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
void dvb_manager_dbus_register_object (DBusConnection* connection, const char* path, void* object);
void _dvb_manager_dbus_unregister (DBusConnection* connection, void* user_data);
DBusHandlerResult dvb_manager_dbus_message (DBusConnection* connection, DBusMessage* message, void* object);
static DBusMessage* _dbus_dvb_manager_introspect (DVBManager* self, DBusConnection* connection, DBusMessage* message);
static void dvb_manager_finalize (GObject* obj);
static void dvb_manager_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_dbus_register_object (DBusConnection* connection, const char* path, void* object);
static void _vala_dbus_unregister_object (gpointer connection, GObject* object);

static const DBusObjectPathVTable _dvb_manager_dbus_path_vtable = {_dvb_manager_dbus_unregister, dvb_manager_dbus_message};
static const _DBusObjectVTable _dvb_manager_dbus_vtable = {dvb_manager_dbus_register_object};


DVBManager* dvb_manager_get_instance (void) {
	DVBManager* result;
	g_static_rec_mutex_lock (&dvb_manager_instance_mutex);
	if (dvb_manager_instance == NULL) {
		DVBManager* _tmp0_;
		_tmp0_ = NULL;
		dvb_manager_instance = (_tmp0_ = dvb_manager_new (), (dvb_manager_instance == NULL) ? NULL : (dvb_manager_instance = (g_object_unref (dvb_manager_instance), NULL)), _tmp0_);
	}
	g_static_rec_mutex_unlock (&dvb_manager_instance_mutex);
	result = dvb_manager_instance;
	return result;
}


void dvb_manager_shutdown (void) {
	DVBManager* _tmp0_;
	DVBManager* m;
	g_static_rec_mutex_lock (&dvb_manager_instance_mutex);
	_tmp0_ = NULL;
	m = (_tmp0_ = dvb_manager_instance, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_));
	if (dvb_manager_instance != NULL) {
		DVBManager* _tmp5_;
		g_static_rec_mutex_lock (&m->priv->__lock_scanners);
		{
			{
				GeeCollection* _tmp1_;
				GeeIterator* _tmp2_;
				GeeIterator* _scanner_it;
				_tmp1_ = NULL;
				_tmp2_ = NULL;
				_scanner_it = (_tmp2_ = gee_iterable_iterator ((GeeIterable*) (_tmp1_ = gee_abstract_map_get_values ((GeeAbstractMap*) m->priv->scanners))), (_tmp1_ == NULL) ? NULL : (_tmp1_ = (g_object_unref (_tmp1_), NULL)), _tmp2_);
				while (TRUE) {
					DVBScanner* scanner;
					if (!gee_iterator_next (_scanner_it)) {
						break;
					}
					scanner = (DVBScanner*) gee_iterator_get (_scanner_it);
					g_debug ("Manager.vala:67: Stopping scanner");
					dvb_scanner_Destroy (scanner);
					(scanner == NULL) ? NULL : (scanner = (g_object_unref (scanner), NULL));
				}
				(_scanner_it == NULL) ? NULL : (_scanner_it = (g_object_unref (_scanner_it), NULL));
			}
			gee_abstract_map_clear ((GeeAbstractMap*) m->priv->scanners);
		}
		g_static_rec_mutex_unlock (&m->priv->__lock_scanners);
		g_static_rec_mutex_lock (&m->priv->__lock_devices);
		{
			{
				GeeCollection* _tmp3_;
				GeeIterator* _tmp4_;
				GeeIterator* _devgrp_it;
				_tmp3_ = NULL;
				_tmp4_ = NULL;
				_devgrp_it = (_tmp4_ = gee_iterable_iterator ((GeeIterable*) (_tmp3_ = gee_abstract_map_get_values ((GeeAbstractMap*) m->priv->devices))), (_tmp3_ == NULL) ? NULL : (_tmp3_ = (g_object_unref (_tmp3_), NULL)), _tmp4_);
				while (TRUE) {
					DVBDeviceGroup* devgrp;
					if (!gee_iterator_next (_devgrp_it)) {
						break;
					}
					devgrp = (DVBDeviceGroup*) gee_iterator_get (_devgrp_it);
					dvb_device_group_destroy (devgrp);
					(devgrp == NULL) ? NULL : (devgrp = (g_object_unref (devgrp), NULL));
				}
				(_devgrp_it == NULL) ? NULL : (_devgrp_it = (g_object_unref (_devgrp_it), NULL));
			}
			gee_abstract_map_clear ((GeeAbstractMap*) m->priv->devices);
		}
		g_static_rec_mutex_unlock (&m->priv->__lock_devices);
		_tmp5_ = NULL;
		dvb_manager_instance = (_tmp5_ = NULL, (dvb_manager_instance == NULL) ? NULL : (dvb_manager_instance = (g_object_unref (dvb_manager_instance), NULL)), _tmp5_);
	}
	g_static_rec_mutex_unlock (&dvb_manager_instance_mutex);
	(m == NULL) ? NULL : (m = (g_object_unref (m), NULL));
}


static void _dvb_manager_on_scanner_destroyed_dvb_scanner_destroyed (DVBScanner* _sender, gpointer self) {
	dvb_manager_on_scanner_destroyed (self, _sender);
}


/**
         * @adapter: Number of the device's adapter
         * @frontend: Number of the device's frontend
         * @returns: Object path and interface to the scanner service
         *
         * Get the object path of the channel scanner for this device.
         */
static char** dvb_manager_real_GetScannerForDevice (DVBIDBusManager* base, guint adapter, guint frontend, int* result_length1) {
	DVBManager * self;
	char** result;
	char* path;
	DVBDevice* device;
	DVBDevice* reg_dev;
	char* dbusiface;
	char** _tmp20_;
	const char* _tmp19_;
	const char* _tmp18_;
	char** _tmp21_;
	self = (DVBManager*) base;
	path = g_strdup_printf (DVB_CONSTANTS_DBUS_SCANNER_PATH, adapter, frontend);
	device = NULL;
	reg_dev = dvb_manager_get_registered_device (self, adapter, frontend);
	if (reg_dev == NULL) {
		DVBDevice* _tmp0_;
		/* Create new device*/
		_tmp0_ = NULL;
		device = (_tmp0_ = dvb_device_new (adapter, frontend, TRUE), (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), _tmp0_);
	} else {
		DVBEPGScanner* _tmp2_;
		DVBDeviceGroup* _tmp1_;
		DVBEPGScanner* _tmp3_;
		DVBEPGScanner* epgscanner;
		DVBDevice* _tmp5_;
		DVBDevice* _tmp4_;
		/* Stop epgscanner for device if there's any*/
		_tmp2_ = NULL;
		_tmp1_ = NULL;
		_tmp3_ = NULL;
		epgscanner = (_tmp3_ = (_tmp2_ = dvb_device_group_get_epgscanner (_tmp1_ = dvb_manager_get_device_group_of_device (self, reg_dev)), (_tmp2_ == NULL) ? NULL : g_object_ref (_tmp2_)), (_tmp1_ == NULL) ? NULL : (_tmp1_ = (g_object_unref (_tmp1_), NULL)), _tmp3_);
		if (epgscanner != NULL) {
			dvb_epg_scanner_stop (epgscanner);
		}
		/* Assign existing device*/
		_tmp5_ = NULL;
		_tmp4_ = NULL;
		device = (_tmp5_ = (_tmp4_ = reg_dev, (_tmp4_ == NULL) ? NULL : g_object_ref (_tmp4_)), (device == NULL) ? NULL : (device = (g_object_unref (device), NULL)), _tmp5_);
		(epgscanner == NULL) ? NULL : (epgscanner = (g_object_unref (epgscanner), NULL));
	}
	dbusiface = NULL;
	switch (dvb_device_get_Type (device)) {
		case DVB_ADAPTER_TYPE_DVB_T:
		{
			char* _tmp6_;
			_tmp6_ = NULL;
			dbusiface = (_tmp6_ = g_strdup ("org.gnome.DVB.Scanner.Terrestrial"), dbusiface = (g_free (dbusiface), NULL), _tmp6_);
			break;
		}
		case DVB_ADAPTER_TYPE_DVB_S:
		{
			char* _tmp7_;
			_tmp7_ = NULL;
			dbusiface = (_tmp7_ = g_strdup ("org.gnome.DVB.Scanner.Satellite"), dbusiface = (g_free (dbusiface), NULL), _tmp7_);
			break;
		}
		case DVB_ADAPTER_TYPE_DVB_C:
		{
			char* _tmp8_;
			_tmp8_ = NULL;
			dbusiface = (_tmp8_ = g_strdup ("org.gnome.DVB.Scanner.Cable"), dbusiface = (g_free (dbusiface), NULL), _tmp8_);
			break;
		}
	}
	if (dbusiface == NULL) {
		char** _tmp9_;
		char** _tmp10_;
		g_critical ("Manager.vala:128: Unknown adapter type");
		_tmp9_ = NULL;
		_tmp10_ = NULL;
		result = (_tmp10_ = (_tmp9_ = g_new0 (char*, 2 + 1), _tmp9_[0] = g_strdup (""), _tmp9_[1] = g_strdup (""), _tmp9_), *result_length1 = 2, _tmp10_);
		path = (g_free (path), NULL);
		(device == NULL) ? NULL : (device = (g_object_unref (device), NULL));
		(reg_dev == NULL) ? NULL : (reg_dev = (g_object_unref (reg_dev), NULL));
		dbusiface = (g_free (dbusiface), NULL);
		return result;
	}
	g_static_rec_mutex_lock (&self->priv->__lock_scanners);
	{
		if (!gee_abstract_map_contains ((GeeAbstractMap*) self->priv->scanners, path)) {
			DVBScanner* scanner;
			DBusGConnection* conn;
			scanner = NULL;
			switch (dvb_device_get_Type (device)) {
				case DVB_ADAPTER_TYPE_DVB_T:
				{
					DVBScanner* _tmp11_;
					_tmp11_ = NULL;
					scanner = (_tmp11_ = (DVBScanner*) dvb_terrestrial_scanner_new (device), (scanner == NULL) ? NULL : (scanner = (g_object_unref (scanner), NULL)), _tmp11_);
					break;
				}
				case DVB_ADAPTER_TYPE_DVB_S:
				{
					DVBScanner* _tmp12_;
					_tmp12_ = NULL;
					scanner = (_tmp12_ = (DVBScanner*) dvb_satellite_scanner_new (device), (scanner == NULL) ? NULL : (scanner = (g_object_unref (scanner), NULL)), _tmp12_);
					break;
				}
				case DVB_ADAPTER_TYPE_DVB_C:
				{
					DVBScanner* _tmp13_;
					_tmp13_ = NULL;
					scanner = (_tmp13_ = (DVBScanner*) dvb_cable_scanner_new (device), (scanner == NULL) ? NULL : (scanner = (g_object_unref (scanner), NULL)), _tmp13_);
					break;
				}
			}
			if (scanner == NULL) {
				char** _tmp14_;
				char** _tmp15_;
				g_critical ("Manager.vala:150: Unknown adapter type");
				_tmp14_ = NULL;
				_tmp15_ = NULL;
				result = (_tmp15_ = (_tmp14_ = g_new0 (char*, 2 + 1), _tmp14_[0] = g_strdup (""), _tmp14_[1] = g_strdup (""), _tmp14_), *result_length1 = 2, _tmp15_);
				(scanner == NULL) ? NULL : (scanner = (g_object_unref (scanner), NULL));
				path = (g_free (path), NULL);
				(device == NULL) ? NULL : (device = (g_object_unref (device), NULL));
				(reg_dev == NULL) ? NULL : (reg_dev = (g_object_unref (reg_dev), NULL));
				dbusiface = (g_free (dbusiface), NULL);
				return result;
			}
			g_signal_connect_object (scanner, "destroyed", (GCallback) _dvb_manager_on_scanner_destroyed_dvb_scanner_destroyed, self, 0);
			gee_abstract_map_set ((GeeAbstractMap*) self->priv->scanners, path, scanner);
			conn = dvb_utils_get_dbus_connection ();
			if (conn == NULL) {
				char** _tmp16_;
				char** _tmp17_;
				_tmp16_ = NULL;
				_tmp17_ = NULL;
				result = (_tmp17_ = (_tmp16_ = g_new0 (char*, 0 + 1), _tmp16_), *result_length1 = 0, _tmp17_);
				(scanner == NULL) ? NULL : (scanner = (g_object_unref (scanner), NULL));
				(conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL));
				path = (g_free (path), NULL);
				(device == NULL) ? NULL : (device = (g_object_unref (device), NULL));
				(reg_dev == NULL) ? NULL : (reg_dev = (g_object_unref (reg_dev), NULL));
				dbusiface = (g_free (dbusiface), NULL);
				return result;
			}
			_vala_dbus_register_object (dbus_g_connection_get_connection (conn), path, (GObject*) scanner);
			g_debug ("Manager.vala:165: Created new Scanner D-Bus service for adapter %u, frontend %u (%s)", adapter, frontend, dbusiface);
			(scanner == NULL) ? NULL : (scanner = (g_object_unref (scanner), NULL));
			(conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_scanners);
	_tmp20_ = NULL;
	_tmp19_ = NULL;
	_tmp18_ = NULL;
	_tmp21_ = NULL;
	result = (_tmp21_ = (_tmp20_ = g_new0 (char*, 2 + 1), _tmp20_[0] = (_tmp18_ = path, (_tmp18_ == NULL) ? NULL : g_strdup (_tmp18_)), _tmp20_[1] = (_tmp19_ = dbusiface, (_tmp19_ == NULL) ? NULL : g_strdup (_tmp19_)), _tmp20_), *result_length1 = 2, _tmp21_);
	path = (g_free (path), NULL);
	(device == NULL) ? NULL : (device = (g_object_unref (device), NULL));
	(reg_dev == NULL) ? NULL : (reg_dev = (g_object_unref (reg_dev), NULL));
	dbusiface = (g_free (dbusiface), NULL);
	return result;
	path = (g_free (path), NULL);
	(device == NULL) ? NULL : (device = (g_object_unref (device), NULL));
	(reg_dev == NULL) ? NULL : (reg_dev = (g_object_unref (reg_dev), NULL));
	dbusiface = (g_free (dbusiface), NULL);
}


/**
         * @group_id: A group ID
         * @returns: Device group's DBus path
         */
static char* dvb_manager_real_GetDeviceGroup (DVBIDBusManager* base, guint group_id) {
	DVBManager * self;
	char* result;
	char* val;
	self = (DVBManager*) base;
	val = NULL;
	g_static_rec_mutex_lock (&self->priv->__lock_devices);
	{
		if (gee_abstract_map_contains ((GeeAbstractMap*) self->priv->devices, GUINT_TO_POINTER (group_id))) {
			char* _tmp1_;
			char* _tmp0_;
			_tmp1_ = NULL;
			_tmp0_ = NULL;
			val = (_tmp1_ = g_strdup (_tmp0_ = g_strdup_printf (DVB_CONSTANTS_DBUS_DEVICE_GROUP_PATH, group_id)), val = (g_free (val), NULL), _tmp1_);
			_tmp0_ = (g_free (_tmp0_), NULL);
		} else {
			char* _tmp2_;
			_tmp2_ = NULL;
			val = (_tmp2_ = g_strdup (""), val = (g_free (val), NULL), _tmp2_);
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_devices);
	result = val;
	return result;
}


/**
         * @returns: Device groups' DBus path
         */
static char** dvb_manager_real_GetRegisteredDeviceGroups (DVBIDBusManager* base, int* result_length1) {
	DVBManager * self;
	char** result;
	char** _tmp1_;
	gint devs_size;
	gint devs_length1;
	gint _tmp0_;
	char** devs;
	gint i;
	char** _tmp6_;
	self = (DVBManager*) base;
	_tmp1_ = NULL;
	devs = (_tmp1_ = g_new0 (char*, (_tmp0_ = gee_map_get_size ((GeeMap*) self->priv->devices)) + 1), devs_length1 = _tmp0_, devs_size = devs_length1, _tmp1_);
	i = 0;
	g_static_rec_mutex_lock (&self->priv->__lock_devices);
	{
		{
			GeeSet* _tmp2_;
			GeeIterator* _tmp3_;
			GeeIterator* _key_it;
			_tmp2_ = NULL;
			_tmp3_ = NULL;
			_key_it = (_tmp3_ = gee_iterable_iterator ((GeeIterable*) (_tmp2_ = gee_abstract_map_get_keys ((GeeAbstractMap*) self->priv->devices))), (_tmp2_ == NULL) ? NULL : (_tmp2_ = (g_object_unref (_tmp2_), NULL)), _tmp3_);
			while (TRUE) {
				guint key;
				char* _tmp5_;
				char* _tmp4_;
				if (!gee_iterator_next (_key_it)) {
					break;
				}
				key = GPOINTER_TO_UINT (gee_iterator_get (_key_it));
				_tmp5_ = NULL;
				_tmp4_ = NULL;
				devs[i] = (_tmp5_ = g_strdup (_tmp4_ = g_strdup_printf (DVB_CONSTANTS_DBUS_DEVICE_GROUP_PATH, key)), devs[i] = (g_free (devs[i]), NULL), _tmp5_);
				_tmp4_ = (g_free (_tmp4_), NULL);
				i++;
			}
			(_key_it == NULL) ? NULL : (_key_it = (g_object_unref (_key_it), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_devices);
	_tmp6_ = NULL;
	result = (_tmp6_ = devs, *result_length1 = devs_length1, _tmp6_);
	return result;
	devs = (_vala_array_free (devs, devs_length1, (GDestroyNotify) g_free), NULL);
}


/**
         * @adapter: Number of the device's adapter
         * @frontend: Number of the device's frontend
         * @channels_conf: Path to channels.conf for this device
         * @recordings_dir: Path where the recordings should be stored
         * @name: Name of group
         * @returns: TRUE when the device has been registered successfully
         *
         * Creates a new DeviceGroup and new DVB device whereas the
         * DVB device is the reference device of this group (i.e.
         * all other devices of this group will inherit the settings
         * of the reference device).
         */
static gboolean dvb_manager_real_AddDeviceToNewGroup (DVBIDBusManager* base, guint adapter, guint frontend, const char* channels_conf, const char* recordings_dir, const char* name) {
	DVBManager * self;
	gboolean result;
	DVBDevice* device;
	DVBDeviceGroup* devgroup;
	self = (DVBManager*) base;
	g_return_val_if_fail (channels_conf != NULL, FALSE);
	g_return_val_if_fail (recordings_dir != NULL, FALSE);
	g_return_val_if_fail (name != NULL, FALSE);
	device = dvb_manager_create_device (adapter, frontend, channels_conf, recordings_dir, self->priv->device_group_counter + 1);
	if (device == NULL) {
		result = FALSE;
		(device == NULL) ? NULL : (device = (g_object_unref (device), NULL));
		return result;
	}
	/* Check if device is already assigned to other group*/
	if (dvb_manager_device_is_in_any_group (self, device)) {
		result = FALSE;
		(device == NULL) ? NULL : (device = (g_object_unref (device), NULL));
		return result;
	}
	self->priv->device_group_counter++;
	devgroup = dvb_device_group_new (self->priv->device_group_counter, device, TRUE);
	dvb_device_group_set_Name (devgroup, name);
	dvb_manager_add_device_group (self, devgroup);
	g_signal_emit_by_name ((DVBIDBusManager*) self, "group-added", self->priv->device_group_counter);
	result = TRUE;
	(device == NULL) ? NULL : (device = (g_object_unref (device), NULL));
	(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
	return result;
}


/**
         * @adapter: Adapter of device
         * @frontend: Frontend of device
         * @returns: The name of the device or "Unknown"
         *
         * The device must be part of group, otherwise "Unknown"
         * is returned.
         */
static char* dvb_manager_real_GetNameOfRegisteredDevice (DVBIDBusManager* base, guint adapter, guint frontend) {
	DVBManager * self;
	char* result;
	DVBDevice* dev;
	self = (DVBManager*) base;
	dev = dvb_manager_get_registered_device (self, adapter, frontend);
	if (dev == NULL) {
		result = g_strdup ("Unknown");
		(dev == NULL) ? NULL : (dev = (g_object_unref (dev), NULL));
		return result;
	} else {
		const char* _tmp0_;
		_tmp0_ = NULL;
		result = (_tmp0_ = dvb_device_get_Name (dev), (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_));
		(dev == NULL) ? NULL : (dev = (g_object_unref (dev), NULL));
		return result;
	}
	(dev == NULL) ? NULL : (dev = (g_object_unref (dev), NULL));
}


/**
         * @returns: the numner of configured device groups
         */
static gint dvb_manager_real_GetDeviceGroupSize (DVBIDBusManager* base) {
	DVBManager * self;
	gint result;
	self = (DVBManager*) base;
	result = gee_map_get_size ((GeeMap*) self->priv->devices);
	return result;
}


static void _dvb_manager_on_device_removed_from_group_dvb_id_bus_device_group_device_removed (DVBDeviceGroup* _sender, guint adapter, guint frontend, gpointer self) {
	dvb_manager_on_device_removed_from_group (self, _sender, adapter, frontend);
}


/**
         * @returns: Whether the device has been added successfully
         *
         * Register device, create Recorder and ChannelList D-Bus service
         */
gboolean dvb_manager_add_device_group (DVBManager* self, DVBDeviceGroup* devgroup) {
	gboolean result;
	guint group_id;
	DBusGConnection* conn;
	char* path;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (devgroup != NULL, FALSE);
	group_id = dvb_device_group_get_Id (devgroup);
	g_debug ("Manager.vala:272: Adding device group %u with %d devices", group_id, dvb_device_group_get_size (devgroup));
	if (dvb_device_group_get_Type (devgroup) == DVB_ADAPTER_TYPE_UNKNOWN) {
		g_warning ("Manager.vala:276: Not adding device group %u of unknown type", dvb_device_group_get_Id (devgroup));
		result = FALSE;
		return result;
	}
	g_static_rec_mutex_lock (&self->priv->__lock_devices);
	{
		gee_abstract_map_set ((GeeAbstractMap*) self->priv->devices, GUINT_TO_POINTER (group_id), devgroup);
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_devices);
	dvb_config_store_add_device_group (dvb_factory_get_config_store (), devgroup);
	g_signal_connect_object ((DVBIDBusDeviceGroup*) devgroup, "device-removed", (GCallback) _dvb_manager_on_device_removed_from_group_dvb_id_bus_device_group_device_removed, self, 0);
	/* Register D-Bus object*/
	conn = dvb_utils_get_dbus_connection ();
	if (conn == NULL) {
		result = FALSE;
		(conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL));
		return result;
	}
	path = g_strdup_printf (DVB_CONSTANTS_DBUS_DEVICE_GROUP_PATH, group_id);
	_vala_dbus_register_object (dbus_g_connection_get_connection (conn), path, (GObject*) devgroup);
	if (group_id > self->priv->device_group_counter) {
		self->priv->device_group_counter = group_id;
	}
	if (!main_get_disable_epg_scanner ()) {
		dvb_epg_scanner_start (dvb_device_group_get_epgscanner (devgroup));
	}
	result = TRUE;
	(conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL));
	path = (g_free (path), NULL);
	return result;
}


static DVBDevice* dvb_manager_create_device (guint adapter, guint frontend, const char* channels_conf, const char* recordings_dir, guint group_id) {
	DVBDevice* result;
	GError * _inner_error_;
	GFile* channelsfile;
	GFile* recdir;
	DVBDevice* device;
	DVBChannelList* channels;
	g_return_val_if_fail (channels_conf != NULL, NULL);
	g_return_val_if_fail (recordings_dir != NULL, NULL);
	_inner_error_ = NULL;
	/* TODO Check if adapter and frontend exists*/
	channelsfile = g_file_new_for_path (channels_conf);
	recdir = g_file_new_for_path (recordings_dir);
	device = dvb_device_new (adapter, frontend, TRUE);
	dvb_device_set_RecordingsDirectory (device, recdir);
	channels = NULL;
	{
		DVBChannelList* _tmp0_;
		DVBChannelList* _tmp1_;
		_tmp0_ = dvb_channel_list_restore_from_file (channelsfile, dvb_device_get_Type (device), group_id, &_inner_error_);
		if (_inner_error_ != NULL) {
			goto __catch8_g_error;
			goto __finally8;
		}
		_tmp1_ = NULL;
		channels = (_tmp1_ = _tmp0_, (channels == NULL) ? NULL : (channels = (g_object_unref (channels), NULL)), _tmp1_);
	}
	goto __finally8;
	__catch8_g_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			g_critical ("Manager.vala:320: Could not create channels list from %s: %s", channels_conf, e->message);
			result = NULL;
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			(channelsfile == NULL) ? NULL : (channelsfile = (g_object_unref (channelsfile), NULL));
			(recdir == NULL) ? NULL : (recdir = (g_object_unref (recdir), NULL));
			(device == NULL) ? NULL : (device = (g_object_unref (device), NULL));
			(channels == NULL) ? NULL : (channels = (g_object_unref (channels), NULL));
			return result;
		}
	}
	__finally8:
	if (_inner_error_ != NULL) {
		(channelsfile == NULL) ? NULL : (channelsfile = (g_object_unref (channelsfile), NULL));
		(recdir == NULL) ? NULL : (recdir = (g_object_unref (recdir), NULL));
		(device == NULL) ? NULL : (device = (g_object_unref (device), NULL));
		(channels == NULL) ? NULL : (channels = (g_object_unref (channels), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return NULL;
	}
	dvb_device_set_Channels (device, channels);
	result = device;
	(channelsfile == NULL) ? NULL : (channelsfile = (g_object_unref (channelsfile), NULL));
	(recdir == NULL) ? NULL : (recdir = (g_object_unref (recdir), NULL));
	(channels == NULL) ? NULL : (channels = (g_object_unref (channels), NULL));
	return result;
}


DVBDeviceGroup* dvb_manager_get_device_group_if_exists (DVBManager* self, guint group_id) {
	DVBDeviceGroup* result;
	DVBDeviceGroup* _result_;
	g_return_val_if_fail (self != NULL, NULL);
	_result_ = NULL;
	g_static_rec_mutex_lock (&self->priv->__lock_devices);
	{
		if (gee_abstract_map_contains ((GeeAbstractMap*) self->priv->devices, GUINT_TO_POINTER (group_id))) {
			DVBDeviceGroup* _tmp0_;
			_tmp0_ = NULL;
			_result_ = (_tmp0_ = (DVBDeviceGroup*) gee_abstract_map_get ((GeeAbstractMap*) self->priv->devices, GUINT_TO_POINTER (group_id)), (_result_ == NULL) ? NULL : (_result_ = (g_object_unref (_result_), NULL)), _tmp0_);
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_devices);
	result = _result_;
	return result;
}


gboolean dvb_manager_device_is_in_any_group (DVBManager* self, DVBDevice* device) {
	gboolean result;
	gboolean _result_;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (device != NULL, FALSE);
	_result_ = FALSE;
	g_static_rec_mutex_lock (&self->priv->__lock_devices);
	{
		{
			GeeSet* _tmp0_;
			GeeIterator* _tmp1_;
			GeeIterator* _group_id_it;
			_tmp0_ = NULL;
			_tmp1_ = NULL;
			_group_id_it = (_tmp1_ = gee_iterable_iterator ((GeeIterable*) (_tmp0_ = gee_abstract_map_get_keys ((GeeAbstractMap*) self->priv->devices))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL)), _tmp1_);
			while (TRUE) {
				guint group_id;
				DVBDeviceGroup* devgroup;
				if (!gee_iterator_next (_group_id_it)) {
					break;
				}
				group_id = GPOINTER_TO_UINT (gee_iterator_get (_group_id_it));
				devgroup = (DVBDeviceGroup*) gee_abstract_map_get ((GeeAbstractMap*) self->priv->devices, GUINT_TO_POINTER (group_id));
				if (dvb_device_group_contains (devgroup, device)) {
					_result_ = TRUE;
					(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
					break;
				}
				(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
			}
			(_group_id_it == NULL) ? NULL : (_group_id_it = (g_object_unref (_group_id_it), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_devices);
	result = _result_;
	return result;
}


static void dvb_manager_on_scanner_destroyed (DVBManager* self, DVBScanner* scanner) {
	guint adapter;
	guint frontend;
	char* path;
	DVBDeviceGroup* devgroup;
	g_return_if_fail (self != NULL);
	g_return_if_fail (scanner != NULL);
	adapter = dvb_device_get_Adapter (dvb_scanner_get_Device (scanner));
	frontend = dvb_device_get_Frontend (dvb_scanner_get_Device (scanner));
	path = g_strdup_printf (DVB_CONSTANTS_DBUS_SCANNER_PATH, adapter, frontend);
	g_static_rec_mutex_lock (&self->priv->__lock_scanners);
	{
		gee_abstract_map_remove ((GeeAbstractMap*) self->priv->scanners, path, NULL);
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_scanners);
	g_debug ("Manager.vala:362: Destroying scanner for adapter %u, frontend %u (%s)", adapter, frontend, path);
	/* Start epgscanner for device again if there was one*/
	devgroup = dvb_manager_get_device_group_of_device (self, dvb_scanner_get_Device (scanner));
	if (devgroup != NULL) {
		DVBEPGScanner* _tmp0_;
		DVBEPGScanner* epgscanner;
		_tmp0_ = NULL;
		epgscanner = (_tmp0_ = dvb_device_group_get_epgscanner (devgroup), (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_));
		if (epgscanner != NULL) {
			dvb_epg_scanner_start (epgscanner);
		}
		(epgscanner == NULL) ? NULL : (epgscanner = (g_object_unref (epgscanner), NULL));
	}
	path = (g_free (path), NULL);
	(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
}


static DVBDevice* dvb_manager_get_registered_device (DVBManager* self, guint adapter, guint frontend) {
	DVBDevice* result;
	DVBDevice* _result_;
	DVBDevice* fake_device;
	g_return_val_if_fail (self != NULL, NULL);
	_result_ = NULL;
	fake_device = dvb_device_new (adapter, frontend, FALSE);
	g_static_rec_mutex_lock (&self->priv->__lock_devices);
	{
		{
			GeeSet* _tmp0_;
			GeeIterator* _tmp1_;
			GeeIterator* _group_id_it;
			_tmp0_ = NULL;
			_tmp1_ = NULL;
			_group_id_it = (_tmp1_ = gee_iterable_iterator ((GeeIterable*) (_tmp0_ = gee_abstract_map_get_keys ((GeeAbstractMap*) self->priv->devices))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL)), _tmp1_);
			while (TRUE) {
				guint group_id;
				DVBDeviceGroup* devgroup;
				if (!gee_iterator_next (_group_id_it)) {
					break;
				}
				group_id = GPOINTER_TO_UINT (gee_iterator_get (_group_id_it));
				devgroup = (DVBDeviceGroup*) gee_abstract_map_get ((GeeAbstractMap*) self->priv->devices, GUINT_TO_POINTER (group_id));
				if (dvb_device_group_contains (devgroup, fake_device)) {
					{
						GeeIterator* _device_it;
						_device_it = gee_iterable_iterator ((GeeIterable*) devgroup);
						while (TRUE) {
							DVBDevice* device;
							if (!gee_iterator_next (_device_it)) {
								break;
							}
							device = (DVBDevice*) gee_iterator_get (_device_it);
							if (dvb_device_equal (fake_device, device)) {
								DVBDevice* _tmp3_;
								DVBDevice* _tmp2_;
								_tmp3_ = NULL;
								_tmp2_ = NULL;
								_result_ = (_tmp3_ = (_tmp2_ = device, (_tmp2_ == NULL) ? NULL : g_object_ref (_tmp2_)), (_result_ == NULL) ? NULL : (_result_ = (g_object_unref (_result_), NULL)), _tmp3_);
								(device == NULL) ? NULL : (device = (g_object_unref (device), NULL));
								break;
							}
							(device == NULL) ? NULL : (device = (g_object_unref (device), NULL));
						}
						(_device_it == NULL) ? NULL : (_device_it = (g_object_unref (_device_it), NULL));
					}
				}
				(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
			}
			(_group_id_it == NULL) ? NULL : (_group_id_it = (g_object_unref (_group_id_it), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_devices);
	result = _result_;
	(fake_device == NULL) ? NULL : (fake_device = (g_object_unref (fake_device), NULL));
	return result;
}


static DVBDeviceGroup* dvb_manager_get_device_group_of_device (DVBManager* self, DVBDevice* device) {
	DVBDeviceGroup* result;
	DVBDeviceGroup* _result_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (device != NULL, NULL);
	_result_ = NULL;
	g_static_rec_mutex_lock (&self->priv->__lock_devices);
	{
		{
			GeeSet* _tmp0_;
			GeeIterator* _tmp1_;
			GeeIterator* _group_id_it;
			_tmp0_ = NULL;
			_tmp1_ = NULL;
			_group_id_it = (_tmp1_ = gee_iterable_iterator ((GeeIterable*) (_tmp0_ = gee_abstract_map_get_keys ((GeeAbstractMap*) self->priv->devices))), (_tmp0_ == NULL) ? NULL : (_tmp0_ = (g_object_unref (_tmp0_), NULL)), _tmp1_);
			while (TRUE) {
				guint group_id;
				DVBDeviceGroup* devgroup;
				if (!gee_iterator_next (_group_id_it)) {
					break;
				}
				group_id = GPOINTER_TO_UINT (gee_iterator_get (_group_id_it));
				devgroup = (DVBDeviceGroup*) gee_abstract_map_get ((GeeAbstractMap*) self->priv->devices, GUINT_TO_POINTER (group_id));
				if (dvb_device_group_contains (devgroup, device)) {
					{
						GeeIterator* _grp_device_it;
						_grp_device_it = gee_iterable_iterator ((GeeIterable*) devgroup);
						while (TRUE) {
							DVBDevice* grp_device;
							if (!gee_iterator_next (_grp_device_it)) {
								break;
							}
							grp_device = (DVBDevice*) gee_iterator_get (_grp_device_it);
							if (dvb_device_equal (grp_device, device)) {
								DVBDeviceGroup* _tmp3_;
								DVBDeviceGroup* _tmp2_;
								_tmp3_ = NULL;
								_tmp2_ = NULL;
								_result_ = (_tmp3_ = (_tmp2_ = devgroup, (_tmp2_ == NULL) ? NULL : g_object_ref (_tmp2_)), (_result_ == NULL) ? NULL : (_result_ = (g_object_unref (_result_), NULL)), _tmp3_);
								(grp_device == NULL) ? NULL : (grp_device = (g_object_unref (grp_device), NULL));
								break;
							}
							(grp_device == NULL) ? NULL : (grp_device = (g_object_unref (grp_device), NULL));
						}
						(_grp_device_it == NULL) ? NULL : (_grp_device_it = (g_object_unref (_grp_device_it), NULL));
					}
				}
				(devgroup == NULL) ? NULL : (devgroup = (g_object_unref (devgroup), NULL));
			}
			(_group_id_it == NULL) ? NULL : (_group_id_it = (g_object_unref (_group_id_it), NULL));
		}
	}
	g_static_rec_mutex_unlock (&self->priv->__lock_devices);
	result = _result_;
	return result;
}


static void dvb_manager_on_device_removed_from_group (DVBManager* self, DVBDeviceGroup* devgroup, guint adapter, guint frontend) {
	guint group_id;
	g_return_if_fail (self != NULL);
	g_return_if_fail (devgroup != NULL);
	group_id = dvb_device_group_get_Id (devgroup);
	if (dvb_device_group_get_size (devgroup) == 0) {
		gboolean success;
		success = FALSE;
		g_static_rec_mutex_lock (&self->priv->__lock_devices);
		{
			success = gee_abstract_map_remove ((GeeAbstractMap*) self->priv->devices, GUINT_TO_POINTER (group_id), NULL);
		}
		g_static_rec_mutex_unlock (&self->priv->__lock_devices);
		if (success) {
			dvb_device_group_destroy (devgroup);
			dvb_config_store_remove_device_group (dvb_factory_get_config_store (), devgroup);
			dvb_epg_store_remove_events_of_group (dvb_factory_get_epg_store (), dvb_device_group_get_Id (devgroup));
			g_signal_emit_by_name ((DVBIDBusManager*) self, "group-removed", group_id);
		}
	}
}


DVBManager* dvb_manager_construct (GType object_type) {
	DVBManager * self;
	self = g_object_newv (object_type, 0, NULL);
	return self;
}


DVBManager* dvb_manager_new (void) {
	return dvb_manager_construct (DVB_TYPE_MANAGER);
}


GeeCollection* dvb_manager_get_device_groups (DVBManager* self) {
	GeeCollection* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = gee_abstract_map_get_values ((GeeAbstractMap*) self->priv->devices);
	return result;
}


static GObject * dvb_manager_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	DVBManagerClass * klass;
	GObjectClass * parent_class;
	DVBManager * self;
	klass = DVB_MANAGER_CLASS (g_type_class_peek (DVB_TYPE_MANAGER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = DVB_MANAGER (obj);
	{
		GeeHashMap* _tmp1_;
		GeeHashMap* _tmp2_;
		_tmp1_ = NULL;
		self->priv->scanners = (_tmp1_ = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, DVB_TYPE_SCANNER, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_str_hash, g_str_equal, g_direct_equal), (self->priv->scanners == NULL) ? NULL : (self->priv->scanners = (g_object_unref (self->priv->scanners), NULL)), _tmp1_);
		_tmp2_ = NULL;
		self->priv->devices = (_tmp2_ = gee_hash_map_new (G_TYPE_UINT, NULL, NULL, DVB_TYPE_DEVICE_GROUP, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_hash, g_direct_equal, g_direct_equal), (self->priv->devices == NULL) ? NULL : (self->priv->devices = (g_object_unref (self->priv->devices), NULL)), _tmp2_);
		self->priv->device_group_counter = (guint) 0;
	}
	return obj;
}


void _dvb_manager_dbus_unregister (DBusConnection* connection, void* user_data) {
}


static DBusMessage* _dbus_dvb_manager_introspect (DVBManager* self, DBusConnection* connection, DBusMessage* message) {
	DBusMessage* reply;
	DBusMessageIter iter;
	GString* xml_data;
	char** children;
	int i;
	reply = dbus_message_new_method_return (message);
	dbus_message_iter_init_append (reply, &iter);
	xml_data = g_string_new ("<!DOCTYPE node PUBLIC \"-//freedesktop//DTD D-BUS Object Introspection 1.0//EN\" \"http://www.freedesktop.org/standards/dbus/1.0/introspect.dtd\">\n");
	g_string_append (xml_data, "<node>\n<interface name=\"org.freedesktop.DBus.Introspectable\">\n  <method name=\"Introspect\">\n    <arg name=\"data\" direction=\"out\" type=\"s\"/>\n  </method>\n</interface>\n<interface name=\"org.freedesktop.DBus.Properties\">\n  <method name=\"Get\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"propname\" direction=\"in\" type=\"s\"/>\n    <arg name=\"value\" direction=\"out\" type=\"v\"/>\n  </method>\n  <method name=\"Set\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"propname\" direction=\"in\" type=\"s\"/>\n    <arg name=\"value\" direction=\"in\" type=\"v\"/>\n  </method>\n  <method name=\"GetAll\">\n    <arg name=\"interface\" direction=\"in\" type=\"s\"/>\n    <arg name=\"props\" direction=\"out\" type=\"a{sv}\"/>\n  </method>\n</interface>\n<interface name=\"org.gnome.DVB.Manager\">\n  <method name=\"GetScannerForDevice\">\n    <arg name=\"adapter\" type=\"u\" direction=\"in\"/>\n    <arg name=\"frontend\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"as\" direction=\"out\"/>\n  </method>\n  <method name=\"GetRegisteredDeviceGroups\">\n    <arg name=\"result\" type=\"ao\" direction=\"out\"/>\n  </method>\n  <method name=\"GetDeviceGroup\">\n    <arg name=\"group_id\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"o\" direction=\"out\"/>\n  </method>\n  <method name=\"AddDeviceToNewGroup\">\n    <arg name=\"adapter\" type=\"u\" direction=\"in\"/>\n    <arg name=\"frontend\" type=\"u\" direction=\"in\"/>\n    <arg name=\"channels_conf\" type=\"s\" direction=\"in\"/>\n    <arg name=\"recordings_dir\" type=\"s\" direction=\"in\"/>\n    <arg name=\"name\" type=\"s\" direction=\"in\"/>\n    <arg name=\"result\" type=\"b\" direction=\"out\"/>\n  </method>\n  <method name=\"GetNameOfRegisteredDevice\">\n    <arg name=\"adapter\" type=\"u\" direction=\"in\"/>\n    <arg name=\"frontend\" type=\"u\" direction=\"in\"/>\n    <arg name=\"result\" type=\"s\" direction=\"out\"/>\n  </method>\n  <method name=\"GetDeviceGroupSize\">\n    <arg name=\"result\" type=\"i\" direction=\"out\"/>\n  </method>\n  <signal name=\"GroupAdded\">\n    <arg name=\"group_id\" type=\"u\"/>\n  </signal>\n  <signal name=\"GroupRemoved\">\n    <arg name=\"group_id\" type=\"u\"/>\n  </signal>\n</interface>\n");
	dbus_connection_list_registered (connection, g_object_get_data ((GObject *) self, "dbus_object_path"), &children);
	for (i = 0; children[i]; i++) {
		g_string_append_printf (xml_data, "<node name=\"%s\"/>\n", children[i]);
	}
	dbus_free_string_array (children);
	g_string_append (xml_data, "</node>\n");
	dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &xml_data->str);
	g_string_free (xml_data, TRUE);
	return reply;
}


DBusHandlerResult dvb_manager_dbus_message (DBusConnection* connection, DBusMessage* message, void* object) {
	DBusMessage* reply;
	reply = NULL;
	if (dbus_message_is_method_call (message, "org.freedesktop.DBus.Introspectable", "Introspect")) {
		reply = _dbus_dvb_manager_introspect (object, connection, message);
	}
	if (reply) {
		dbus_connection_send (connection, reply, NULL);
		dbus_message_unref (reply);
		return DBUS_HANDLER_RESULT_HANDLED;
	} else if (dvb_id_bus_manager_dbus_message (connection, message, object) == DBUS_HANDLER_RESULT_HANDLED) {
		return DBUS_HANDLER_RESULT_HANDLED;
	} else {
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}
}


void dvb_manager_dbus_register_object (DBusConnection* connection, const char* path, void* object) {
	if (!g_object_get_data (object, "dbus_object_path")) {
		g_object_set_data (object, "dbus_object_path", g_strdup (path));
		dbus_connection_register_object_path (connection, path, &_dvb_manager_dbus_path_vtable, object);
		g_object_weak_ref (object, _vala_dbus_unregister_object, connection);
	}
	dvb_id_bus_manager_dbus_register_object (connection, path, object);
}


static void dvb_manager_class_init (DVBManagerClass * klass) {
	GStaticRecMutex _tmp0_ = {0};
	dvb_manager_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (DVBManagerPrivate));
	G_OBJECT_CLASS (klass)->get_property = dvb_manager_get_property;
	G_OBJECT_CLASS (klass)->constructor = dvb_manager_constructor;
	G_OBJECT_CLASS (klass)->finalize = dvb_manager_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_MANAGER_DEVICE_GROUPS, g_param_spec_object ("device-groups", "device-groups", "device-groups", GEE_TYPE_COLLECTION, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE));
	g_type_set_qdata (DVB_TYPE_MANAGER, g_quark_from_static_string ("DBusObjectVTable"), (void*) (&_dvb_manager_dbus_vtable));
	dvb_manager_instance_mutex = (g_static_rec_mutex_init (&_tmp0_), _tmp0_);
}


static void dvb_manager_dvb_id_bus_manager_interface_init (DVBIDBusManagerIface * iface) {
	dvb_manager_dvb_id_bus_manager_parent_iface = g_type_interface_peek_parent (iface);
	iface->GetScannerForDevice = dvb_manager_real_GetScannerForDevice;
	iface->GetDeviceGroup = dvb_manager_real_GetDeviceGroup;
	iface->GetRegisteredDeviceGroups = dvb_manager_real_GetRegisteredDeviceGroups;
	iface->AddDeviceToNewGroup = dvb_manager_real_AddDeviceToNewGroup;
	iface->GetNameOfRegisteredDevice = dvb_manager_real_GetNameOfRegisteredDevice;
	iface->GetDeviceGroupSize = dvb_manager_real_GetDeviceGroupSize;
}


static void dvb_manager_instance_init (DVBManager * self) {
	self->priv = DVB_MANAGER_GET_PRIVATE (self);
	g_static_rec_mutex_init (&self->priv->__lock_scanners);
	g_static_rec_mutex_init (&self->priv->__lock_devices);
}


static void dvb_manager_finalize (GObject* obj) {
	DVBManager * self;
	self = DVB_MANAGER (obj);
	g_static_rec_mutex_free (&self->priv->__lock_scanners);
	(self->priv->scanners == NULL) ? NULL : (self->priv->scanners = (g_object_unref (self->priv->scanners), NULL));
	g_static_rec_mutex_free (&self->priv->__lock_devices);
	(self->priv->devices == NULL) ? NULL : (self->priv->devices = (g_object_unref (self->priv->devices), NULL));
	G_OBJECT_CLASS (dvb_manager_parent_class)->finalize (obj);
}


GType dvb_manager_get_type (void) {
	static GType dvb_manager_type_id = 0;
	if (dvb_manager_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (DVBManagerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_manager_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBManager), 0, (GInstanceInitFunc) dvb_manager_instance_init, NULL };
		static const GInterfaceInfo dvb_id_bus_manager_info = { (GInterfaceInitFunc) dvb_manager_dvb_id_bus_manager_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		dvb_manager_type_id = g_type_register_static (G_TYPE_OBJECT, "DVBManager", &g_define_type_info, 0);
		g_type_add_interface_static (dvb_manager_type_id, DVB_TYPE_ID_BUS_MANAGER, &dvb_id_bus_manager_info);
	}
	return dvb_manager_type_id;
}


static void dvb_manager_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	DVBManager * self;
	gpointer boxed;
	self = DVB_MANAGER (object);
	switch (property_id) {
		case DVB_MANAGER_DEVICE_GROUPS:
		g_value_set_object (value, dvb_manager_get_device_groups (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	_vala_array_destroy (array, array_length, destroy_func);
	g_free (array);
}


static void _vala_dbus_register_object (DBusConnection* connection, const char* path, void* object) {
	const _DBusObjectVTable * vtable;
	vtable = g_type_get_qdata (G_TYPE_FROM_INSTANCE (object), g_quark_from_static_string ("DBusObjectVTable"));
	if (vtable) {
		vtable->register_object (connection, path, object);
	} else {
		g_warning ("Object does not implement any D-Bus interface");
	}
}


static void _vala_dbus_unregister_object (gpointer connection, GObject* object) {
	char* path;
	path = g_object_steal_data ((GObject*) object, "dbus_object_path");
	dbus_connection_unregister_object_path (connection, path);
	g_free (path);
}




