/*
 * Copyright (C) 2008,2009 Sebastian Pölsterl
 *
 * This file is part of GNOME DVB Daemon.
 *
 * GNOME DVB Daemon is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GNOME DVB Daemon is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNOME DVB Daemon.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <glib-object.h>
#include <gee.h>
#include <sqlite3.h>
#include <stdlib.h>
#include <string.h>
#include <float.h>
#include <math.h>
#include <stdio.h>
#include <gio/gio.h>


#define DVB_TYPE_EPG_STORE (dvb_epg_store_get_type ())
#define DVB_EPG_STORE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_EPG_STORE, DVBEPGStore))
#define DVB_IS_EPG_STORE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_EPG_STORE))
#define DVB_EPG_STORE_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), DVB_TYPE_EPG_STORE, DVBEPGStoreIface))

typedef struct _DVBEPGStore DVBEPGStore;
typedef struct _DVBEPGStoreIface DVBEPGStoreIface;

#define DVB_TYPE_EVENT (dvb_event_get_type ())
#define DVB_EVENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_EVENT, DVBEvent))
#define DVB_EVENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_EVENT, DVBEventClass))
#define DVB_IS_EVENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_EVENT))
#define DVB_IS_EVENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_EVENT))
#define DVB_EVENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_EVENT, DVBEventClass))

typedef struct _DVBEvent DVBEvent;
typedef struct _DVBEventClass DVBEventClass;

#define DVB_TYPE_SQLITE_EPG_STORE (dvb_sqlite_epg_store_get_type ())
#define DVB_SQLITE_EPG_STORE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_SQLITE_EPG_STORE, DVBSqliteEPGStore))
#define DVB_SQLITE_EPG_STORE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_SQLITE_EPG_STORE, DVBSqliteEPGStoreClass))
#define DVB_IS_SQLITE_EPG_STORE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_SQLITE_EPG_STORE))
#define DVB_IS_SQLITE_EPG_STORE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_SQLITE_EPG_STORE))
#define DVB_SQLITE_EPG_STORE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_SQLITE_EPG_STORE, DVBSqliteEPGStoreClass))

typedef struct _DVBSqliteEPGStore DVBSqliteEPGStore;
typedef struct _DVBSqliteEPGStoreClass DVBSqliteEPGStoreClass;
typedef struct _DVBSqliteEPGStorePrivate DVBSqliteEPGStorePrivate;
typedef struct _DVBEventPrivate DVBEventPrivate;

#define DVB_EVENT_TYPE_AUDIO_COMPONENT (dvb_event_audio_component_get_type ())
#define DVB_EVENT_AUDIO_COMPONENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_EVENT_TYPE_AUDIO_COMPONENT, DVBEventAudioComponent))
#define DVB_EVENT_AUDIO_COMPONENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_EVENT_TYPE_AUDIO_COMPONENT, DVBEventAudioComponentClass))
#define DVB_EVENT_IS_AUDIO_COMPONENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_EVENT_TYPE_AUDIO_COMPONENT))
#define DVB_EVENT_IS_AUDIO_COMPONENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_EVENT_TYPE_AUDIO_COMPONENT))
#define DVB_EVENT_AUDIO_COMPONENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_EVENT_TYPE_AUDIO_COMPONENT, DVBEventAudioComponentClass))

typedef struct _DVBEventAudioComponent DVBEventAudioComponent;
typedef struct _DVBEventAudioComponentClass DVBEventAudioComponentClass;

#define DVB_EVENT_TYPE_VIDEO_COMPONENT (dvb_event_video_component_get_type ())
#define DVB_EVENT_VIDEO_COMPONENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_EVENT_TYPE_VIDEO_COMPONENT, DVBEventVideoComponent))
#define DVB_EVENT_VIDEO_COMPONENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_EVENT_TYPE_VIDEO_COMPONENT, DVBEventVideoComponentClass))
#define DVB_EVENT_IS_VIDEO_COMPONENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_EVENT_TYPE_VIDEO_COMPONENT))
#define DVB_EVENT_IS_VIDEO_COMPONENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_EVENT_TYPE_VIDEO_COMPONENT))
#define DVB_EVENT_VIDEO_COMPONENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_EVENT_TYPE_VIDEO_COMPONENT, DVBEventVideoComponentClass))

typedef struct _DVBEventVideoComponent DVBEventVideoComponent;
typedef struct _DVBEventVideoComponentClass DVBEventVideoComponentClass;

#define DVB_EVENT_TYPE_TELETEXT_COMPONENT (dvb_event_teletext_component_get_type ())
#define DVB_EVENT_TELETEXT_COMPONENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_EVENT_TYPE_TELETEXT_COMPONENT, DVBEventTeletextComponent))
#define DVB_EVENT_TELETEXT_COMPONENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_EVENT_TYPE_TELETEXT_COMPONENT, DVBEventTeletextComponentClass))
#define DVB_EVENT_IS_TELETEXT_COMPONENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_EVENT_TYPE_TELETEXT_COMPONENT))
#define DVB_EVENT_IS_TELETEXT_COMPONENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_EVENT_TYPE_TELETEXT_COMPONENT))
#define DVB_EVENT_TELETEXT_COMPONENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_EVENT_TYPE_TELETEXT_COMPONENT, DVBEventTeletextComponentClass))

typedef struct _DVBEventTeletextComponent DVBEventTeletextComponent;
typedef struct _DVBEventTeletextComponentClass DVBEventTeletextComponentClass;

struct _DVBEPGStoreIface {
	GTypeInterface parent_iface;
	gboolean (*add_or_update_event) (DVBEPGStore* self, DVBEvent* event, guint channel_sid, guint group_id);
	DVBEvent* (*get_event) (DVBEPGStore* self, guint event_id, guint channel_sid, guint group_id);
	gboolean (*remove_event) (DVBEPGStore* self, guint event_id, guint channel_sid, guint group_id);
	gboolean (*contains_event) (DVBEPGStore* self, DVBEvent* event, guint channel_sid, guint group_id);
	GeeList* (*get_events) (DVBEPGStore* self, guint channel_sid, guint group_id);
	gboolean (*remove_events_of_group) (DVBEPGStore* self, guint group_id);
};

struct _DVBSqliteEPGStore {
	GObject parent_instance;
	DVBSqliteEPGStorePrivate * priv;
};

struct _DVBSqliteEPGStoreClass {
	GObjectClass parent_class;
};

struct _DVBSqliteEPGStorePrivate {
	sqlite3_stmt* to_julian_statement;
	sqlite3_stmt* insert_event_statement;
	sqlite3_stmt* update_event_statement;
	sqlite3_stmt* delete_event_statement;
	sqlite3_stmt* has_event_statement;
	sqlite3_stmt* select_event_statement;
	sqlite3_stmt* delete_events_group;
	sqlite3* db;
};

/**
     * Represents an EPG event (i.e. a show with all its information)
     */
struct _DVBEvent {
	GTypeInstance parent_instance;
	volatile int ref_count;
	DVBEventPrivate * priv;
	guint id;
	guint year;
	guint month;
	guint hour;
	guint day;
	guint minute;
	guint second;
	guint duration;
	guint running_status;
	gboolean free_ca_mode;
	char* name;
	char* description;
	char* extended_description;
	GSList* audio_components;
	GSList* video_components;
	GSList* teletext_components;
};

struct _DVBEventClass {
	GTypeClass parent_class;
	void (*finalize) (DVBEvent *self);
};


static gpointer dvb_sqlite_epg_store_parent_class = NULL;
static DVBEPGStoreIface* dvb_sqlite_epg_store_dvb_epg_store_parent_iface = NULL;

gpointer dvb_event_ref (gpointer instance);
void dvb_event_unref (gpointer instance);
GParamSpec* dvb_param_spec_event (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void dvb_value_set_event (GValue* value, gpointer v_object);
gpointer dvb_value_get_event (const GValue* value);
GType dvb_event_get_type (void);
GType dvb_epg_store_get_type (void);
GType dvb_sqlite_epg_store_get_type (void);
#define DVB_SQLITE_EPG_STORE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), DVB_TYPE_SQLITE_EPG_STORE, DVBSqliteEPGStorePrivate))
enum  {
	DVB_SQLITE_EPG_STORE_DUMMY_PROPERTY
};
#define DVB_SQLITE_EPG_STORE_CREATE_EVENTS_TABLE_STATEMENT "CREATE TABLE events (group_id INTEGER,\n            sid INTEGER,\n            event_id INTEGER,\n            starttime JULIAN,\n            duration INTEGER,\n            running_status INTEGER(2),\n            free_ca_mode INTEGER(1),\n            name VARCHAR(255),\n            description VARCHAR(255),\n            extended_description TEXT,\n            PRIMARY KEY (group_id, sid, event_id))"
#define DVB_SQLITE_EPG_STORE_INSERT_EVENT_SQL "INSERT INTO events VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)"
#define DVB_SQLITE_EPG_STORE_DELETE_EVENT_STATEMENT "DELETE FROM events WHERE group_id=? AND sid=? AND event_id=?"
#define DVB_SQLITE_EPG_STORE_SELECT_ALL_EVENTS_STATEMENT "SELECT event_id, datetime(starttime),\n            duration, running_status, free_ca_mode, name,\n            description, extended_description\n            FROM events WHERE group_id='%u' AND sid='%u'"
#define DVB_SQLITE_EPG_STORE_HAS_EVENT_STATEMENT "SELECT COUNT(*) FROM events WHERE group_id=? AND sid=? AND event_id=?"
#define DVB_SQLITE_EPG_STORE_UPDATE_EVENT_SQL "UPDATE events SET starttime=?, duration=?, running_status=?,\n            free_ca_mode=?, name=?, description=?,\n            extended_description=? WHERE group_id=? AND sid=? AND event_id=?"
#define DVB_SQLITE_EPG_STORE_TO_JULIAN_SQL "SELECT julianday(?)"
#define DVB_SQLITE_EPG_STORE_SELECT_EVENT_SQL "SELECT event_id, datetime(starttime),\n            duration, running_status, free_ca_mode, name,\n            description, extended_description\n            FROM events WHERE group_id=? AND sid=? AND event_id=?"
#define DVB_SQLITE_EPG_STORE_DELETE_EVENTS_GROUP "DELETE FROM events WHERE group_id=?"
gpointer dvb_event_audio_component_ref (gpointer instance);
void dvb_event_audio_component_unref (gpointer instance);
GParamSpec* dvb_event_param_spec_audio_component (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void dvb_event_value_set_audio_component (GValue* value, gpointer v_object);
gpointer dvb_event_value_get_audio_component (const GValue* value);
GType dvb_event_audio_component_get_type (void);
gpointer dvb_event_video_component_ref (gpointer instance);
void dvb_event_video_component_unref (gpointer instance);
GParamSpec* dvb_event_param_spec_video_component (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void dvb_event_value_set_video_component (GValue* value, gpointer v_object);
gpointer dvb_event_value_get_video_component (const GValue* value);
GType dvb_event_video_component_get_type (void);
gpointer dvb_event_teletext_component_ref (gpointer instance);
void dvb_event_teletext_component_unref (gpointer instance);
GParamSpec* dvb_event_param_spec_teletext_component (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void dvb_event_value_set_teletext_component (GValue* value, gpointer v_object);
gpointer dvb_event_value_get_teletext_component (const GValue* value);
GType dvb_event_teletext_component_get_type (void);
static char* dvb_sqlite_epg_store_escape (const char* text);
static double dvb_sqlite_epg_store_to_julian (DVBSqliteEPGStore* self, guint year, guint month, guint day, guint hour, guint minute, guint second);
gboolean dvb_epg_store_contains_event (DVBEPGStore* self, DVBEvent* event, guint channel_sid, guint group_id);
static inline void dvb_sqlite_epg_store_print_last_error (DVBSqliteEPGStore* self);
static gboolean dvb_sqlite_epg_store_real_add_or_update_event (DVBEPGStore* base, DVBEvent* event, guint channel_sid, guint group_id);
static DVBEvent* dvb_sqlite_epg_store_create_event_from_statement (DVBSqliteEPGStore* self, sqlite3_stmt* statement);
static DVBEvent* dvb_sqlite_epg_store_real_get_event (DVBEPGStore* base, guint event_id, guint channel_sid, guint group_id);
static gboolean dvb_sqlite_epg_store_real_remove_event (DVBEPGStore* base, guint event_id, guint channel_sid, guint group_id);
static gboolean dvb_sqlite_epg_store_real_contains_event (DVBEPGStore* base, DVBEvent* event, guint channel_sid, guint group_id);
static GeeList* dvb_sqlite_epg_store_real_get_events (DVBEPGStore* base, guint channel_sid, guint group_id);
static gboolean dvb_sqlite_epg_store_real_remove_events_of_group (DVBEPGStore* base, guint group_id);
DVBEvent* dvb_event_new (void);
DVBEvent* dvb_event_construct (GType object_type);
static void _g_slist_free_dvb_event_audio_component_unref (GSList* self);
static void _g_slist_free_dvb_event_video_component_unref (GSList* self);
static void _g_slist_free_dvb_event_teletext_component_unref (GSList* self);
void dvb_utils_mkdirs (GFile* directory, GError** error);
static sqlite3* dvb_sqlite_epg_store_get_db_handler (void);
DVBSqliteEPGStore* dvb_sqlite_epg_store_new (void);
DVBSqliteEPGStore* dvb_sqlite_epg_store_construct (GType object_type);
static GObject * dvb_sqlite_epg_store_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static void dvb_sqlite_epg_store_finalize (GObject* obj);



static gboolean dvb_sqlite_epg_store_real_add_or_update_event (DVBEPGStore* base, DVBEvent* event, guint channel_sid, guint group_id) {
	DVBSqliteEPGStore * self;
	gboolean result;
	gint _tmp0_;
	gint free_ca_mode;
	char* name;
	char* desc;
	char* ext_desc;
	double julian_start;
	self = (DVBSqliteEPGStore*) base;
	g_return_val_if_fail (event != NULL, FALSE);
	if (self->priv->db == NULL) {
		g_critical ("SqliteEPGStore.vala:106: SQLite error: No database connection");
		result = FALSE;
		return result;
	}
	_tmp0_ = 0;
	if (event->free_ca_mode) {
		_tmp0_ = 1;
	} else {
		_tmp0_ = 0;
	}
	free_ca_mode = _tmp0_;
	name = dvb_sqlite_epg_store_escape (event->name);
	desc = dvb_sqlite_epg_store_escape (event->description);
	ext_desc = dvb_sqlite_epg_store_escape (event->extended_description);
	julian_start = dvb_sqlite_epg_store_to_julian (self, event->year, event->month, event->day, event->hour, event->minute, event->second);
	/* Check if start time got converted correctly*/
	if (julian_start <= 0) {
		result = FALSE;
		name = (g_free (name), NULL);
		desc = (g_free (desc), NULL);
		ext_desc = (g_free (ext_desc), NULL);
		return result;
	}
	if (dvb_epg_store_contains_event ((DVBEPGStore*) self, event, channel_sid, group_id)) {
		gboolean _tmp1_;
		gboolean _tmp2_;
		gboolean _tmp3_;
		gboolean _tmp4_;
		gboolean _tmp5_;
		gboolean _tmp6_;
		gboolean _tmp7_;
		gboolean _tmp8_;
		gboolean _tmp9_;
		sqlite3_reset (self->priv->update_event_statement);
		_tmp1_ = FALSE;
		_tmp2_ = FALSE;
		_tmp3_ = FALSE;
		_tmp4_ = FALSE;
		_tmp5_ = FALSE;
		_tmp6_ = FALSE;
		_tmp7_ = FALSE;
		_tmp8_ = FALSE;
		_tmp9_ = FALSE;
		if (sqlite3_bind_double (self->priv->update_event_statement, 1, julian_start) != SQLITE_OK) {
			_tmp9_ = TRUE;
		} else {
			_tmp9_ = sqlite3_bind_int (self->priv->update_event_statement, 2, (gint) event->duration) != SQLITE_OK;
		}
		if (_tmp9_) {
			_tmp8_ = TRUE;
		} else {
			_tmp8_ = sqlite3_bind_int (self->priv->update_event_statement, 3, (gint) event->running_status) != SQLITE_OK;
		}
		if (_tmp8_) {
			_tmp7_ = TRUE;
		} else {
			_tmp7_ = sqlite3_bind_int (self->priv->update_event_statement, 4, free_ca_mode) != SQLITE_OK;
		}
		if (_tmp7_) {
			_tmp6_ = TRUE;
		} else {
			const char* _tmp10_;
			_tmp10_ = NULL;
			_tmp6_ = sqlite3_bind_text (self->priv->update_event_statement, 5, (_tmp10_ = name, (_tmp10_ == NULL) ? NULL : g_strdup (_tmp10_)), -1, g_free) != SQLITE_OK;
		}
		if (_tmp6_) {
			_tmp5_ = TRUE;
		} else {
			const char* _tmp11_;
			_tmp11_ = NULL;
			_tmp5_ = sqlite3_bind_text (self->priv->update_event_statement, 6, (_tmp11_ = desc, (_tmp11_ == NULL) ? NULL : g_strdup (_tmp11_)), -1, g_free) != SQLITE_OK;
		}
		if (_tmp5_) {
			_tmp4_ = TRUE;
		} else {
			const char* _tmp12_;
			_tmp12_ = NULL;
			_tmp4_ = sqlite3_bind_text (self->priv->update_event_statement, 7, (_tmp12_ = ext_desc, (_tmp12_ == NULL) ? NULL : g_strdup (_tmp12_)), -1, g_free) != SQLITE_OK;
		}
		if (_tmp4_) {
			_tmp3_ = TRUE;
		} else {
			_tmp3_ = sqlite3_bind_int (self->priv->update_event_statement, 8, (gint) group_id) != SQLITE_OK;
		}
		if (_tmp3_) {
			_tmp2_ = TRUE;
		} else {
			_tmp2_ = sqlite3_bind_int (self->priv->update_event_statement, 9, (gint) channel_sid) != SQLITE_OK;
		}
		if (_tmp2_) {
			_tmp1_ = TRUE;
		} else {
			_tmp1_ = sqlite3_bind_int (self->priv->update_event_statement, 10, (gint) event->id) != SQLITE_OK;
		}
		if (_tmp1_) {
			dvb_sqlite_epg_store_print_last_error (self);
			result = FALSE;
			name = (g_free (name), NULL);
			desc = (g_free (desc), NULL);
			ext_desc = (g_free (ext_desc), NULL);
			return result;
		}
		if (sqlite3_step (self->priv->update_event_statement) != SQLITE_DONE) {
			dvb_sqlite_epg_store_print_last_error (self);
			result = FALSE;
			name = (g_free (name), NULL);
			desc = (g_free (desc), NULL);
			ext_desc = (g_free (ext_desc), NULL);
			return result;
		}
	} else {
		gboolean _tmp13_;
		gboolean _tmp14_;
		gboolean _tmp15_;
		gboolean _tmp16_;
		gboolean _tmp17_;
		gboolean _tmp18_;
		gboolean _tmp19_;
		gboolean _tmp20_;
		gboolean _tmp21_;
		sqlite3_reset (self->priv->insert_event_statement);
		_tmp13_ = FALSE;
		_tmp14_ = FALSE;
		_tmp15_ = FALSE;
		_tmp16_ = FALSE;
		_tmp17_ = FALSE;
		_tmp18_ = FALSE;
		_tmp19_ = FALSE;
		_tmp20_ = FALSE;
		_tmp21_ = FALSE;
		if (sqlite3_bind_int (self->priv->insert_event_statement, 1, (gint) group_id) != SQLITE_OK) {
			_tmp21_ = TRUE;
		} else {
			_tmp21_ = sqlite3_bind_int (self->priv->insert_event_statement, 2, (gint) channel_sid) != SQLITE_OK;
		}
		if (_tmp21_) {
			_tmp20_ = TRUE;
		} else {
			_tmp20_ = sqlite3_bind_int (self->priv->insert_event_statement, 3, (gint) event->id) != SQLITE_OK;
		}
		if (_tmp20_) {
			_tmp19_ = TRUE;
		} else {
			_tmp19_ = sqlite3_bind_double (self->priv->insert_event_statement, 4, julian_start) != SQLITE_OK;
		}
		if (_tmp19_) {
			_tmp18_ = TRUE;
		} else {
			_tmp18_ = sqlite3_bind_int (self->priv->insert_event_statement, 5, (gint) event->duration) != SQLITE_OK;
		}
		if (_tmp18_) {
			_tmp17_ = TRUE;
		} else {
			_tmp17_ = sqlite3_bind_int (self->priv->insert_event_statement, 6, (gint) event->running_status) != SQLITE_OK;
		}
		if (_tmp17_) {
			_tmp16_ = TRUE;
		} else {
			_tmp16_ = sqlite3_bind_int (self->priv->insert_event_statement, 7, free_ca_mode) != SQLITE_OK;
		}
		if (_tmp16_) {
			_tmp15_ = TRUE;
		} else {
			const char* _tmp22_;
			_tmp22_ = NULL;
			_tmp15_ = sqlite3_bind_text (self->priv->insert_event_statement, 8, (_tmp22_ = name, (_tmp22_ == NULL) ? NULL : g_strdup (_tmp22_)), -1, g_free) != SQLITE_OK;
		}
		if (_tmp15_) {
			_tmp14_ = TRUE;
		} else {
			const char* _tmp23_;
			_tmp23_ = NULL;
			_tmp14_ = sqlite3_bind_text (self->priv->insert_event_statement, 9, (_tmp23_ = desc, (_tmp23_ == NULL) ? NULL : g_strdup (_tmp23_)), -1, g_free) != SQLITE_OK;
		}
		if (_tmp14_) {
			_tmp13_ = TRUE;
		} else {
			const char* _tmp24_;
			_tmp24_ = NULL;
			_tmp13_ = sqlite3_bind_text (self->priv->insert_event_statement, 10, (_tmp24_ = ext_desc, (_tmp24_ == NULL) ? NULL : g_strdup (_tmp24_)), -1, g_free) != SQLITE_OK;
		}
		if (_tmp13_) {
			dvb_sqlite_epg_store_print_last_error (self);
			result = FALSE;
			name = (g_free (name), NULL);
			desc = (g_free (desc), NULL);
			ext_desc = (g_free (ext_desc), NULL);
			return result;
		}
		if (sqlite3_step (self->priv->insert_event_statement) != SQLITE_DONE) {
			dvb_sqlite_epg_store_print_last_error (self);
			result = FALSE;
			name = (g_free (name), NULL);
			desc = (g_free (desc), NULL);
			ext_desc = (g_free (ext_desc), NULL);
			return result;
		}
	}
	result = TRUE;
	name = (g_free (name), NULL);
	desc = (g_free (desc), NULL);
	ext_desc = (g_free (ext_desc), NULL);
	return result;
}


static DVBEvent* dvb_sqlite_epg_store_real_get_event (DVBEPGStore* base, guint event_id, guint channel_sid, guint group_id) {
	DVBSqliteEPGStore * self;
	DVBEvent* result;
	gboolean _tmp0_;
	gboolean _tmp1_;
	gint rc;
	gboolean _tmp2_;
	self = (DVBSqliteEPGStore*) base;
	sqlite3_reset (self->priv->select_event_statement);
	_tmp0_ = FALSE;
	_tmp1_ = FALSE;
	if (sqlite3_bind_int (self->priv->select_event_statement, 1, (gint) group_id) != SQLITE_OK) {
		_tmp1_ = TRUE;
	} else {
		_tmp1_ = sqlite3_bind_int (self->priv->select_event_statement, 2, (gint) channel_sid) != SQLITE_OK;
	}
	if (_tmp1_) {
		_tmp0_ = TRUE;
	} else {
		_tmp0_ = sqlite3_bind_int (self->priv->select_event_statement, 3, (gint) event_id) != SQLITE_OK;
	}
	if (_tmp0_) {
		dvb_sqlite_epg_store_print_last_error (self);
		result = NULL;
		return result;
	}
	rc = sqlite3_step (self->priv->select_event_statement);
	_tmp2_ = FALSE;
	if (rc != SQLITE_ROW) {
		_tmp2_ = rc != SQLITE_DONE;
	} else {
		_tmp2_ = FALSE;
	}
	if (_tmp2_) {
		dvb_sqlite_epg_store_print_last_error (self);
		result = NULL;
		return result;
	}
	/* ROW means there's data, DONE means there's none*/
	if (rc == SQLITE_DONE) {
		result = NULL;
		return result;
	} else {
		result = dvb_sqlite_epg_store_create_event_from_statement (self, self->priv->select_event_statement);
		return result;
	}
}


static gboolean dvb_sqlite_epg_store_real_remove_event (DVBEPGStore* base, guint event_id, guint channel_sid, guint group_id) {
	DVBSqliteEPGStore * self;
	gboolean result;
	gboolean _tmp0_;
	gboolean _tmp1_;
	self = (DVBSqliteEPGStore*) base;
	if (self->priv->db == NULL) {
		g_critical ("SqliteEPGStore.vala:191: SQLite error: No database connection");
		result = FALSE;
		return result;
	}
	sqlite3_reset (self->priv->delete_event_statement);
	_tmp0_ = FALSE;
	_tmp1_ = FALSE;
	if (sqlite3_bind_int (self->priv->delete_event_statement, 1, (gint) group_id) != SQLITE_OK) {
		_tmp1_ = TRUE;
	} else {
		_tmp1_ = sqlite3_bind_int (self->priv->delete_event_statement, 2, (gint) channel_sid) != SQLITE_OK;
	}
	if (_tmp1_) {
		_tmp0_ = TRUE;
	} else {
		_tmp0_ = sqlite3_bind_int (self->priv->delete_event_statement, 3, (gint) event_id) != SQLITE_OK;
	}
	if (_tmp0_) {
		dvb_sqlite_epg_store_print_last_error (self);
		result = FALSE;
		return result;
	}
	if (sqlite3_step (self->priv->delete_event_statement) != SQLITE_DONE) {
		dvb_sqlite_epg_store_print_last_error (self);
		result = FALSE;
		return result;
	}
	result = TRUE;
	return result;
}


static gboolean dvb_sqlite_epg_store_real_contains_event (DVBEPGStore* base, DVBEvent* event, guint channel_sid, guint group_id) {
	DVBSqliteEPGStore * self;
	gboolean result;
	gboolean _tmp0_;
	gboolean _tmp1_;
	gint c;
	self = (DVBSqliteEPGStore*) base;
	g_return_val_if_fail (event != NULL, FALSE);
	sqlite3_reset (self->priv->has_event_statement);
	_tmp0_ = FALSE;
	_tmp1_ = FALSE;
	if (sqlite3_bind_int (self->priv->has_event_statement, 1, (gint) group_id) != SQLITE_OK) {
		_tmp1_ = TRUE;
	} else {
		_tmp1_ = sqlite3_bind_int (self->priv->has_event_statement, 2, (gint) channel_sid) != SQLITE_OK;
	}
	if (_tmp1_) {
		_tmp0_ = TRUE;
	} else {
		_tmp0_ = sqlite3_bind_int (self->priv->has_event_statement, 3, (gint) event->id) != SQLITE_OK;
	}
	if (_tmp0_) {
		dvb_sqlite_epg_store_print_last_error (self);
		result = FALSE;
		return result;
	}
	c = 0;
	while (TRUE) {
		if (!(sqlite3_step (self->priv->has_event_statement) == SQLITE_ROW)) {
			break;
		}
		c = sqlite3_column_int (self->priv->has_event_statement, 0);
	}
	result = c > 0;
	return result;
}


static GeeList* dvb_sqlite_epg_store_real_get_events (DVBEPGStore* base, guint channel_sid, guint group_id) {
	DVBSqliteEPGStore * self;
	GeeList* result;
	GeeList* events;
	char* statement_str;
	sqlite3_stmt* statement;
	sqlite3_stmt* _tmp2_;
	gint _tmp1_;
	sqlite3_stmt* _tmp0_;
	self = (DVBSqliteEPGStore*) base;
	events = (GeeList*) gee_array_list_new (DVB_TYPE_EVENT, (GBoxedCopyFunc) dvb_event_ref, dvb_event_unref, g_direct_equal);
	if (self->priv->db == NULL) {
		result = events;
		return result;
	}
	statement_str = g_strdup_printf (DVB_SQLITE_EPG_STORE_SELECT_ALL_EVENTS_STATEMENT, group_id, channel_sid);
	statement = NULL;
	_tmp2_ = NULL;
	_tmp0_ = NULL;
	if ((_tmp1_ = sqlite3_prepare (self->priv->db, statement_str, -1, &_tmp0_, NULL), statement = (_tmp2_ = _tmp0_, (statement == NULL) ? NULL : (statement = (sqlite3_finalize (statement), NULL)), _tmp2_), _tmp1_) != SQLITE_OK) {
		dvb_sqlite_epg_store_print_last_error (self);
		result = events;
		statement_str = (g_free (statement_str), NULL);
		(statement == NULL) ? NULL : (statement = (sqlite3_finalize (statement), NULL));
		return result;
	}
	while (TRUE) {
		DVBEvent* event;
		if (!(sqlite3_step (statement) == SQLITE_ROW)) {
			break;
		}
		event = dvb_sqlite_epg_store_create_event_from_statement (self, statement);
		gee_collection_add ((GeeCollection*) events, event);
		(event == NULL) ? NULL : (event = (dvb_event_unref (event), NULL));
	}
	result = events;
	statement_str = (g_free (statement_str), NULL);
	(statement == NULL) ? NULL : (statement = (sqlite3_finalize (statement), NULL));
	return result;
}


static gboolean dvb_sqlite_epg_store_real_remove_events_of_group (DVBEPGStore* base, guint group_id) {
	DVBSqliteEPGStore * self;
	gboolean result;
	self = (DVBSqliteEPGStore*) base;
	sqlite3_reset (self->priv->delete_events_group);
	if (sqlite3_bind_int (self->priv->delete_events_group, 1, (gint) group_id) != SQLITE_OK) {
		dvb_sqlite_epg_store_print_last_error (self);
		result = FALSE;
		return result;
	}
	if (sqlite3_step (self->priv->delete_events_group) != SQLITE_DONE) {
		dvb_sqlite_epg_store_print_last_error (self);
		result = FALSE;
		return result;
	}
	result = TRUE;
	return result;
}


static void _g_slist_free_dvb_event_audio_component_unref (GSList* self) {
	g_slist_foreach (self, (GFunc) dvb_event_audio_component_unref, NULL);
	g_slist_free (self);
}


static void _g_slist_free_dvb_event_video_component_unref (GSList* self) {
	g_slist_foreach (self, (GFunc) dvb_event_video_component_unref, NULL);
	g_slist_free (self);
}


static void _g_slist_free_dvb_event_teletext_component_unref (GSList* self) {
	g_slist_foreach (self, (GFunc) dvb_event_teletext_component_unref, NULL);
	g_slist_free (self);
}


static DVBEvent* dvb_sqlite_epg_store_create_event_from_statement (DVBSqliteEPGStore* self, sqlite3_stmt* statement) {
	DVBEvent* result;
	DVBEvent* event;
	const char* starttime;
	char* _tmp0_;
	char* _tmp1_;
	char* _tmp2_;
	GSList* _tmp3_;
	GSList* _tmp4_;
	GSList* _tmp5_;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (statement != NULL, NULL);
	event = dvb_event_new ();
	event->id = (guint) sqlite3_column_int (statement, 0);
	starttime = sqlite3_column_text (statement, 1);
	sscanf (starttime, "%04u-%02u-%02u %02u:%02u:%02u", &event->year, &event->month, &event->day, &event->hour, &event->minute, &event->second);
	event->duration = (guint) sqlite3_column_int (statement, 2);
	event->running_status = (guint) sqlite3_column_int (statement, 3);
	event->free_ca_mode = sqlite3_column_int (statement, 4) == 1;
	/* Duplicate strings*/
	_tmp0_ = NULL;
	event->name = (_tmp0_ = g_strdup_printf ("%s", sqlite3_column_text (statement, 5)), event->name = (g_free (event->name), NULL), _tmp0_);
	_tmp1_ = NULL;
	event->description = (_tmp1_ = g_strdup_printf ("%s", sqlite3_column_text (statement, 6)), event->description = (g_free (event->description), NULL), _tmp1_);
	_tmp2_ = NULL;
	event->extended_description = (_tmp2_ = g_strdup_printf ("%s", sqlite3_column_text (statement, 7)), event->extended_description = (g_free (event->extended_description), NULL), _tmp2_);
	/* We don't save those*/
	_tmp3_ = NULL;
	event->audio_components = (_tmp3_ = NULL, (event->audio_components == NULL) ? NULL : (event->audio_components = (_g_slist_free_dvb_event_audio_component_unref (event->audio_components), NULL)), _tmp3_);
	_tmp4_ = NULL;
	event->video_components = (_tmp4_ = NULL, (event->video_components == NULL) ? NULL : (event->video_components = (_g_slist_free_dvb_event_video_component_unref (event->video_components), NULL)), _tmp4_);
	_tmp5_ = NULL;
	event->teletext_components = (_tmp5_ = NULL, (event->teletext_components == NULL) ? NULL : (event->teletext_components = (_g_slist_free_dvb_event_teletext_component_unref (event->teletext_components), NULL)), _tmp5_);
	result = event;
	return result;
}


static inline void dvb_sqlite_epg_store_print_last_error (DVBSqliteEPGStore* self) {
	g_return_if_fail (self != NULL);
	g_critical ("SqliteEPGStore.vala:297: SQLite error: %d, %s", sqlite3_errcode (self->priv->db), sqlite3_errmsg (self->priv->db));
}


static double dvb_sqlite_epg_store_to_julian (DVBSqliteEPGStore* self, guint year, guint month, guint day, guint hour, guint minute, guint second) {
	double result;
	char* datetime_str;
	const char* _tmp0_;
	g_return_val_if_fail (self != NULL, 0.0);
	sqlite3_reset (self->priv->to_julian_statement);
	datetime_str = g_strdup_printf ("%04u-%02u-%02u %02u:%02u:%02u", year, month, day, hour, minute, second);
	_tmp0_ = NULL;
	if (sqlite3_bind_text (self->priv->to_julian_statement, 1, (_tmp0_ = datetime_str, (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_)), -1, g_free) != SQLITE_OK) {
		dvb_sqlite_epg_store_print_last_error (self);
		result = (double) 0;
		datetime_str = (g_free (datetime_str), NULL);
		return result;
	}
	if (sqlite3_step (self->priv->to_julian_statement) != SQLITE_ROW) {
		dvb_sqlite_epg_store_print_last_error (self);
		result = (double) 0;
		datetime_str = (g_free (datetime_str), NULL);
		return result;
	}
	result = sqlite3_column_double (self->priv->to_julian_statement, 0);
	datetime_str = (g_free (datetime_str), NULL);
	return result;
}


/**
         * Replace "'" with "''"
         */
static char* dvb_sqlite_epg_store_escape (const char* text) {
	char* result;
	GError * _inner_error_;
	GRegex* regex;
	char* escaped_str;
	_inner_error_ = NULL;
	if (text == NULL) {
		result = g_strdup ("");
		return result;
	}
	regex = NULL;
	{
		GRegex* _tmp0_;
		GRegex* _tmp1_;
		_tmp0_ = g_regex_new ("'", G_REGEX_MULTILINE, 0, &_inner_error_);
		if (_inner_error_ != NULL) {
			if (_inner_error_->domain == G_REGEX_ERROR) {
				goto __catch35_g_regex_error;
			}
			goto __finally35;
		}
		_tmp1_ = NULL;
		regex = (_tmp1_ = _tmp0_, (regex == NULL) ? NULL : (regex = (g_regex_unref (regex), NULL)), _tmp1_);
	}
	goto __finally35;
	__catch35_g_regex_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			const char* _tmp2_;
			g_warning ("SqliteEPGStore.vala:335: RegexError: %s", e->message);
			_tmp2_ = NULL;
			result = (_tmp2_ = text, (_tmp2_ == NULL) ? NULL : g_strdup (_tmp2_));
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			(regex == NULL) ? NULL : (regex = (g_regex_unref (regex), NULL));
			return result;
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally35:
	if (_inner_error_ != NULL) {
		(regex == NULL) ? NULL : (regex = (g_regex_unref (regex), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return NULL;
	}
	escaped_str = NULL;
	{
		char* _tmp3_;
		char* _tmp4_;
		_tmp3_ = g_regex_replace_literal (regex, text, (glong) (-1), 0, "''", 0, &_inner_error_);
		if (_inner_error_ != NULL) {
			if (_inner_error_->domain == G_REGEX_ERROR) {
				goto __catch36_g_regex_error;
			}
			goto __finally36;
		}
		_tmp4_ = NULL;
		escaped_str = (_tmp4_ = _tmp3_, escaped_str = (g_free (escaped_str), NULL), _tmp4_);
	}
	goto __finally36;
	__catch36_g_regex_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			const char* _tmp5_;
			g_warning ("SqliteEPGStore.vala:344: RegexError: %s", e->message);
			_tmp5_ = NULL;
			result = (_tmp5_ = text, (_tmp5_ == NULL) ? NULL : g_strdup (_tmp5_));
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			(regex == NULL) ? NULL : (regex = (g_regex_unref (regex), NULL));
			escaped_str = (g_free (escaped_str), NULL);
			return result;
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally36:
	if (_inner_error_ != NULL) {
		(regex == NULL) ? NULL : (regex = (g_regex_unref (regex), NULL));
		escaped_str = (g_free (escaped_str), NULL);
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return NULL;
	}
	result = escaped_str;
	(regex == NULL) ? NULL : (regex = (g_regex_unref (regex), NULL));
	return result;
}


static sqlite3* dvb_sqlite_epg_store_get_db_handler (void) {
	sqlite3* result;
	GError * _inner_error_;
	GFile* cache_dir;
	GFile* our_cache;
	GFile* eventsdb;
	gboolean create_tables;
	sqlite3* db;
	sqlite3* _tmp3_;
	gint _tmp2_;
	sqlite3* _tmp1_;
	char* _tmp0_;
	_inner_error_ = NULL;
	cache_dir = g_file_new_for_path (g_get_user_cache_dir ());
	our_cache = g_file_get_child (cache_dir, "gnome-dvb-daemon");
	eventsdb = g_file_get_child (our_cache, "eventsdb.sqlite3");
	if (!g_file_query_exists (our_cache, NULL)) {
		{
			dvb_utils_mkdirs (our_cache, &_inner_error_);
			if (_inner_error_ != NULL) {
				goto __catch37_g_error;
				goto __finally37;
			}
		}
		goto __finally37;
		__catch37_g_error:
		{
			GError * e;
			e = _inner_error_;
			_inner_error_ = NULL;
			{
				g_critical ("SqliteEPGStore.vala:361: Could not create directory: %s", e->message);
				result = NULL;
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
				(cache_dir == NULL) ? NULL : (cache_dir = (g_object_unref (cache_dir), NULL));
				(our_cache == NULL) ? NULL : (our_cache = (g_object_unref (our_cache), NULL));
				(eventsdb == NULL) ? NULL : (eventsdb = (g_object_unref (eventsdb), NULL));
				return result;
			}
		}
		__finally37:
		if (_inner_error_ != NULL) {
			(cache_dir == NULL) ? NULL : (cache_dir = (g_object_unref (cache_dir), NULL));
			(our_cache == NULL) ? NULL : (our_cache = (g_object_unref (our_cache), NULL));
			(eventsdb == NULL) ? NULL : (eventsdb = (g_object_unref (eventsdb), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
			g_clear_error (&_inner_error_);
			return NULL;
		}
	}
	create_tables = !g_file_query_exists (eventsdb, NULL);
	db = NULL;
	_tmp3_ = NULL;
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	_tmp2_ = sqlite3_open (_tmp0_ = g_file_get_path (eventsdb), &_tmp1_);
	db = (_tmp3_ = _tmp1_, (db == NULL) ? NULL : (db = (sqlite3_close (db), NULL)), _tmp3_);
	_tmp2_;
	_tmp0_ = (g_free (_tmp0_), NULL);
	if (create_tables) {
		char* errormsg;
		char* _tmp6_;
		gint _tmp5_;
		char* _tmp4_;
		gint val;
		errormsg = NULL;
		_tmp6_ = NULL;
		_tmp4_ = NULL;
		val = (_tmp5_ = sqlite3_exec (db, DVB_SQLITE_EPG_STORE_CREATE_EVENTS_TABLE_STATEMENT, NULL, NULL, &_tmp4_), errormsg = (_tmp6_ = _tmp4_, errormsg = (g_free (errormsg), NULL), _tmp6_), _tmp5_);
		if (val != SQLITE_OK) {
			g_critical ("SqliteEPGStore.vala:375: SQLite error: %s", errormsg);
			result = NULL;
			errormsg = (g_free (errormsg), NULL);
			(cache_dir == NULL) ? NULL : (cache_dir = (g_object_unref (cache_dir), NULL));
			(our_cache == NULL) ? NULL : (our_cache = (g_object_unref (our_cache), NULL));
			(eventsdb == NULL) ? NULL : (eventsdb = (g_object_unref (eventsdb), NULL));
			(db == NULL) ? NULL : (db = (sqlite3_close (db), NULL));
			return result;
		}
		errormsg = (g_free (errormsg), NULL);
	}
	result = db;
	(cache_dir == NULL) ? NULL : (cache_dir = (g_object_unref (cache_dir), NULL));
	(our_cache == NULL) ? NULL : (our_cache = (g_object_unref (our_cache), NULL));
	(eventsdb == NULL) ? NULL : (eventsdb = (g_object_unref (eventsdb), NULL));
	return result;
}


DVBSqliteEPGStore* dvb_sqlite_epg_store_construct (GType object_type) {
	DVBSqliteEPGStore * self;
	self = g_object_newv (object_type, 0, NULL);
	return self;
}


DVBSqliteEPGStore* dvb_sqlite_epg_store_new (void) {
	return dvb_sqlite_epg_store_construct (DVB_TYPE_SQLITE_EPG_STORE);
}


static GObject * dvb_sqlite_epg_store_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	DVBSqliteEPGStoreClass * klass;
	GObjectClass * parent_class;
	DVBSqliteEPGStore * self;
	klass = DVB_SQLITE_EPG_STORE_CLASS (g_type_class_peek (DVB_TYPE_SQLITE_EPG_STORE));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = DVB_SQLITE_EPG_STORE (obj);
	{
		sqlite3* _tmp0_;
		sqlite3_stmt* _tmp3_;
		gint _tmp2_;
		sqlite3_stmt* _tmp1_;
		sqlite3_stmt* _tmp6_;
		gint _tmp5_;
		sqlite3_stmt* _tmp4_;
		sqlite3_stmt* _tmp9_;
		gint _tmp8_;
		sqlite3_stmt* _tmp7_;
		sqlite3_stmt* _tmp12_;
		gint _tmp11_;
		sqlite3_stmt* _tmp10_;
		sqlite3_stmt* _tmp15_;
		gint _tmp14_;
		sqlite3_stmt* _tmp13_;
		sqlite3_stmt* _tmp18_;
		gint _tmp17_;
		sqlite3_stmt* _tmp16_;
		sqlite3_stmt* _tmp21_;
		gint _tmp20_;
		sqlite3_stmt* _tmp19_;
		_tmp0_ = NULL;
		self->priv->db = (_tmp0_ = dvb_sqlite_epg_store_get_db_handler (), (self->priv->db == NULL) ? NULL : (self->priv->db = (sqlite3_close (self->priv->db), NULL)), _tmp0_);
		_tmp3_ = NULL;
		_tmp1_ = NULL;
		_tmp2_ = sqlite3_prepare (self->priv->db, DVB_SQLITE_EPG_STORE_TO_JULIAN_SQL, -1, &_tmp1_, NULL);
		self->priv->to_julian_statement = (_tmp3_ = _tmp1_, (self->priv->to_julian_statement == NULL) ? NULL : (self->priv->to_julian_statement = (sqlite3_finalize (self->priv->to_julian_statement), NULL)), _tmp3_);
		_tmp2_;
		_tmp6_ = NULL;
		_tmp4_ = NULL;
		_tmp5_ = sqlite3_prepare (self->priv->db, DVB_SQLITE_EPG_STORE_INSERT_EVENT_SQL, -1, &_tmp4_, NULL);
		self->priv->insert_event_statement = (_tmp6_ = _tmp4_, (self->priv->insert_event_statement == NULL) ? NULL : (self->priv->insert_event_statement = (sqlite3_finalize (self->priv->insert_event_statement), NULL)), _tmp6_);
		_tmp5_;
		_tmp9_ = NULL;
		_tmp7_ = NULL;
		_tmp8_ = sqlite3_prepare (self->priv->db, DVB_SQLITE_EPG_STORE_UPDATE_EVENT_SQL, -1, &_tmp7_, NULL);
		self->priv->update_event_statement = (_tmp9_ = _tmp7_, (self->priv->update_event_statement == NULL) ? NULL : (self->priv->update_event_statement = (sqlite3_finalize (self->priv->update_event_statement), NULL)), _tmp9_);
		_tmp8_;
		_tmp12_ = NULL;
		_tmp10_ = NULL;
		_tmp11_ = sqlite3_prepare (self->priv->db, DVB_SQLITE_EPG_STORE_DELETE_EVENT_STATEMENT, -1, &_tmp10_, NULL);
		self->priv->delete_event_statement = (_tmp12_ = _tmp10_, (self->priv->delete_event_statement == NULL) ? NULL : (self->priv->delete_event_statement = (sqlite3_finalize (self->priv->delete_event_statement), NULL)), _tmp12_);
		_tmp11_;
		_tmp15_ = NULL;
		_tmp13_ = NULL;
		_tmp14_ = sqlite3_prepare (self->priv->db, DVB_SQLITE_EPG_STORE_HAS_EVENT_STATEMENT, -1, &_tmp13_, NULL);
		self->priv->has_event_statement = (_tmp15_ = _tmp13_, (self->priv->has_event_statement == NULL) ? NULL : (self->priv->has_event_statement = (sqlite3_finalize (self->priv->has_event_statement), NULL)), _tmp15_);
		_tmp14_;
		_tmp18_ = NULL;
		_tmp16_ = NULL;
		_tmp17_ = sqlite3_prepare (self->priv->db, DVB_SQLITE_EPG_STORE_SELECT_EVENT_SQL, -1, &_tmp16_, NULL);
		self->priv->select_event_statement = (_tmp18_ = _tmp16_, (self->priv->select_event_statement == NULL) ? NULL : (self->priv->select_event_statement = (sqlite3_finalize (self->priv->select_event_statement), NULL)), _tmp18_);
		_tmp17_;
		_tmp21_ = NULL;
		_tmp19_ = NULL;
		_tmp20_ = sqlite3_prepare (self->priv->db, DVB_SQLITE_EPG_STORE_DELETE_EVENTS_GROUP, -1, &_tmp19_, NULL);
		self->priv->delete_events_group = (_tmp21_ = _tmp19_, (self->priv->delete_events_group == NULL) ? NULL : (self->priv->delete_events_group = (sqlite3_finalize (self->priv->delete_events_group), NULL)), _tmp21_);
		_tmp20_;
	}
	return obj;
}


static void dvb_sqlite_epg_store_class_init (DVBSqliteEPGStoreClass * klass) {
	dvb_sqlite_epg_store_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (DVBSqliteEPGStorePrivate));
	G_OBJECT_CLASS (klass)->constructor = dvb_sqlite_epg_store_constructor;
	G_OBJECT_CLASS (klass)->finalize = dvb_sqlite_epg_store_finalize;
}


static void dvb_sqlite_epg_store_dvb_epg_store_interface_init (DVBEPGStoreIface * iface) {
	dvb_sqlite_epg_store_dvb_epg_store_parent_iface = g_type_interface_peek_parent (iface);
	iface->add_or_update_event = dvb_sqlite_epg_store_real_add_or_update_event;
	iface->get_event = dvb_sqlite_epg_store_real_get_event;
	iface->remove_event = dvb_sqlite_epg_store_real_remove_event;
	iface->contains_event = dvb_sqlite_epg_store_real_contains_event;
	iface->get_events = dvb_sqlite_epg_store_real_get_events;
	iface->remove_events_of_group = dvb_sqlite_epg_store_real_remove_events_of_group;
}


static void dvb_sqlite_epg_store_instance_init (DVBSqliteEPGStore * self) {
	self->priv = DVB_SQLITE_EPG_STORE_GET_PRIVATE (self);
}


static void dvb_sqlite_epg_store_finalize (GObject* obj) {
	DVBSqliteEPGStore * self;
	self = DVB_SQLITE_EPG_STORE (obj);
	(self->priv->to_julian_statement == NULL) ? NULL : (self->priv->to_julian_statement = (sqlite3_finalize (self->priv->to_julian_statement), NULL));
	(self->priv->insert_event_statement == NULL) ? NULL : (self->priv->insert_event_statement = (sqlite3_finalize (self->priv->insert_event_statement), NULL));
	(self->priv->update_event_statement == NULL) ? NULL : (self->priv->update_event_statement = (sqlite3_finalize (self->priv->update_event_statement), NULL));
	(self->priv->delete_event_statement == NULL) ? NULL : (self->priv->delete_event_statement = (sqlite3_finalize (self->priv->delete_event_statement), NULL));
	(self->priv->has_event_statement == NULL) ? NULL : (self->priv->has_event_statement = (sqlite3_finalize (self->priv->has_event_statement), NULL));
	(self->priv->select_event_statement == NULL) ? NULL : (self->priv->select_event_statement = (sqlite3_finalize (self->priv->select_event_statement), NULL));
	(self->priv->delete_events_group == NULL) ? NULL : (self->priv->delete_events_group = (sqlite3_finalize (self->priv->delete_events_group), NULL));
	(self->priv->db == NULL) ? NULL : (self->priv->db = (sqlite3_close (self->priv->db), NULL));
	G_OBJECT_CLASS (dvb_sqlite_epg_store_parent_class)->finalize (obj);
}


GType dvb_sqlite_epg_store_get_type (void) {
	static GType dvb_sqlite_epg_store_type_id = 0;
	if (dvb_sqlite_epg_store_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (DVBSqliteEPGStoreClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_sqlite_epg_store_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBSqliteEPGStore), 0, (GInstanceInitFunc) dvb_sqlite_epg_store_instance_init, NULL };
		static const GInterfaceInfo dvb_epg_store_info = { (GInterfaceInitFunc) dvb_sqlite_epg_store_dvb_epg_store_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		dvb_sqlite_epg_store_type_id = g_type_register_static (G_TYPE_OBJECT, "DVBSqliteEPGStore", &g_define_type_info, 0);
		g_type_add_interface_static (dvb_sqlite_epg_store_type_id, DVB_TYPE_EPG_STORE, &dvb_epg_store_info);
	}
	return dvb_sqlite_epg_store_type_id;
}




