/*
 * Copyright (C) 2008,2009 Sebastian Pölsterl
 *
 * This file is part of GNOME DVB Daemon.
 *
 * GNOME DVB Daemon is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GNOME DVB Daemon is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNOME DVB Daemon.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <glib-object.h>
#include <gio/gio.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>


#define DVB_TYPE_CHANNEL_LIST_WRITER (dvb_channel_list_writer_get_type ())
#define DVB_CHANNEL_LIST_WRITER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CHANNEL_LIST_WRITER, DVBChannelListWriter))
#define DVB_CHANNEL_LIST_WRITER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CHANNEL_LIST_WRITER, DVBChannelListWriterClass))
#define DVB_IS_CHANNEL_LIST_WRITER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CHANNEL_LIST_WRITER))
#define DVB_IS_CHANNEL_LIST_WRITER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CHANNEL_LIST_WRITER))
#define DVB_CHANNEL_LIST_WRITER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CHANNEL_LIST_WRITER, DVBChannelListWriterClass))

typedef struct _DVBChannelListWriter DVBChannelListWriter;
typedef struct _DVBChannelListWriterClass DVBChannelListWriterClass;
typedef struct _DVBChannelListWriterPrivate DVBChannelListWriterPrivate;

#define DVB_TYPE_CHANNEL (dvb_channel_get_type ())
#define DVB_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CHANNEL, DVBChannel))
#define DVB_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CHANNEL, DVBChannelClass))
#define DVB_IS_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CHANNEL))
#define DVB_IS_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CHANNEL))
#define DVB_CHANNEL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CHANNEL, DVBChannelClass))

typedef struct _DVBChannel DVBChannel;
typedef struct _DVBChannelClass DVBChannelClass;

#define DVB_TYPE_TERRESTRIAL_CHANNEL (dvb_terrestrial_channel_get_type ())
#define DVB_TERRESTRIAL_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_TERRESTRIAL_CHANNEL, DVBTerrestrialChannel))
#define DVB_TERRESTRIAL_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_TERRESTRIAL_CHANNEL, DVBTerrestrialChannelClass))
#define DVB_IS_TERRESTRIAL_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_TERRESTRIAL_CHANNEL))
#define DVB_IS_TERRESTRIAL_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_TERRESTRIAL_CHANNEL))
#define DVB_TERRESTRIAL_CHANNEL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_TERRESTRIAL_CHANNEL, DVBTerrestrialChannelClass))

typedef struct _DVBTerrestrialChannel DVBTerrestrialChannel;
typedef struct _DVBTerrestrialChannelClass DVBTerrestrialChannelClass;

#define DVB_TYPE_SATELLITE_CHANNEL (dvb_satellite_channel_get_type ())
#define DVB_SATELLITE_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_SATELLITE_CHANNEL, DVBSatelliteChannel))
#define DVB_SATELLITE_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_SATELLITE_CHANNEL, DVBSatelliteChannelClass))
#define DVB_IS_SATELLITE_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_SATELLITE_CHANNEL))
#define DVB_IS_SATELLITE_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_SATELLITE_CHANNEL))
#define DVB_SATELLITE_CHANNEL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_SATELLITE_CHANNEL, DVBSatelliteChannelClass))

typedef struct _DVBSatelliteChannel DVBSatelliteChannel;
typedef struct _DVBSatelliteChannelClass DVBSatelliteChannelClass;

#define DVB_TYPE_CABLE_CHANNEL (dvb_cable_channel_get_type ())
#define DVB_CABLE_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), DVB_TYPE_CABLE_CHANNEL, DVBCableChannel))
#define DVB_CABLE_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), DVB_TYPE_CABLE_CHANNEL, DVBCableChannelClass))
#define DVB_IS_CABLE_CHANNEL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), DVB_TYPE_CABLE_CHANNEL))
#define DVB_IS_CABLE_CHANNEL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), DVB_TYPE_CABLE_CHANNEL))
#define DVB_CABLE_CHANNEL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), DVB_TYPE_CABLE_CHANNEL, DVBCableChannelClass))

typedef struct _DVBCableChannel DVBCableChannel;
typedef struct _DVBCableChannelClass DVBCableChannelClass;

#define DVB_TYPE_DVB_SRC_INVERSION (dvb_dvb_src_inversion_get_type ())

#define DVB_TYPE_DVB_SRC_BANDWIDTH (dvb_dvb_src_bandwidth_get_type ())

#define DVB_TYPE_DVB_SRC_CODE_RATE (dvb_dvb_src_code_rate_get_type ())

#define DVB_TYPE_DVB_SRC_MODULATION (dvb_dvb_src_modulation_get_type ())

#define DVB_TYPE_DVB_SRC_TRANSMISSION_MODE (dvb_dvb_src_transmission_mode_get_type ())

#define DVB_TYPE_DVB_SRC_GUARD (dvb_dvb_src_guard_get_type ())

#define DVB_TYPE_DVB_SRC_HIERARCHY (dvb_dvb_src_hierarchy_get_type ())

/**
     * Example:
     * try {
     *     var writer = new DVB.ChannelListWriter (File.new_for_path ("/path/to/channels.conf"));
     *     foreach (DVB.Channel c in reader.Channels) {
     *         writer.write (c);
     *     }
     *     writer.close ();
     * } catch (IOError e) {
     *     error (e.message);
     * }
     */
struct _DVBChannelListWriter {
	GObject parent_instance;
	DVBChannelListWriterPrivate * priv;
};

struct _DVBChannelListWriterClass {
	GObjectClass parent_class;
};

struct _DVBChannelListWriterPrivate {
	GFile* _file;
	GOutputStream* stream;
};

typedef enum  {
	DVB_DVB_SRC_INVERSION_INVERSION_OFF,
	DVB_DVB_SRC_INVERSION_INVERSION_ON,
	DVB_DVB_SRC_INVERSION_INVERSION_AUTO
} DVBDvbSrcInversion;

typedef enum  {
	DVB_DVB_SRC_BANDWIDTH_BANDWIDTH_8_MHZ,
	DVB_DVB_SRC_BANDWIDTH_BANDWIDTH_7_MHZ,
	DVB_DVB_SRC_BANDWIDTH_BANDWIDTH_6_MHZ,
	DVB_DVB_SRC_BANDWIDTH_BANDWIDTH_AUTO
} DVBDvbSrcBandwidth;

typedef enum  {
	DVB_DVB_SRC_CODE_RATE_FEC_NONE,
	DVB_DVB_SRC_CODE_RATE_FEC_1_2,
	DVB_DVB_SRC_CODE_RATE_FEC_2_3,
	DVB_DVB_SRC_CODE_RATE_FEC_3_4,
	DVB_DVB_SRC_CODE_RATE_FEC_4_5,
	DVB_DVB_SRC_CODE_RATE_FEC_5_6,
	DVB_DVB_SRC_CODE_RATE_FEC_6_7,
	DVB_DVB_SRC_CODE_RATE_FEC_7_8,
	DVB_DVB_SRC_CODE_RATE_FEC_8_9,
	DVB_DVB_SRC_CODE_RATE_FEC_AUTO
} DVBDvbSrcCodeRate;

typedef enum  {
	DVB_DVB_SRC_MODULATION_QPSK,
	DVB_DVB_SRC_MODULATION_QAM_16,
	DVB_DVB_SRC_MODULATION_QAM_32,
	DVB_DVB_SRC_MODULATION_QAM_64,
	DVB_DVB_SRC_MODULATION_QAM_128,
	DVB_DVB_SRC_MODULATION_QAM_256,
	DVB_DVB_SRC_MODULATION_QAM_AUTO
} DVBDvbSrcModulation;

typedef enum  {
	DVB_DVB_SRC_TRANSMISSION_MODE_TRANSMISSION_MODE_2K,
	DVB_DVB_SRC_TRANSMISSION_MODE_TRANSMISSION_MODE_8K,
	DVB_DVB_SRC_TRANSMISSION_MODE_TRANSMISSION_MODE_AUTO
} DVBDvbSrcTransmissionMode;

typedef enum  {
	DVB_DVB_SRC_GUARD_GUARD_INTERVAL_1_32,
	DVB_DVB_SRC_GUARD_GUARD_INTERVAL_1_16,
	DVB_DVB_SRC_GUARD_GUARD_INTERVAL_1_8,
	DVB_DVB_SRC_GUARD_GUARD_INTERVAL_1_4,
	DVB_DVB_SRC_GUARD_GUARD_INTERVAL_AUTO
} DVBDvbSrcGuard;

typedef enum  {
	DVB_DVB_SRC_HIERARCHY_HIERARCHY_NONE,
	DVB_DVB_SRC_HIERARCHY_HIERARCHY_1,
	DVB_DVB_SRC_HIERARCHY_HIERARCHY_2,
	DVB_DVB_SRC_HIERARCHY_HIERARCHY_4,
	DVB_DVB_SRC_HIERARCHY_HIERARCHY_AUTO
} DVBDvbSrcHierarchy;


static gpointer dvb_channel_list_writer_parent_class = NULL;

GType dvb_channel_list_writer_get_type (void);
#define DVB_CHANNEL_LIST_WRITER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), DVB_TYPE_CHANNEL_LIST_WRITER, DVBChannelListWriterPrivate))
enum  {
	DVB_CHANNEL_LIST_WRITER_DUMMY_PROPERTY,
	DVB_CHANNEL_LIST_WRITER_FILE
};
GFile* dvb_channel_list_writer_get_file (DVBChannelListWriter* self);
static void dvb_channel_list_writer_open_stream (DVBChannelListWriter* self, GError** error);
static void dvb_channel_list_writer_set_file (DVBChannelListWriter* self, GFile* value);
DVBChannelListWriter* dvb_channel_list_writer_new (GFile* file);
DVBChannelListWriter* dvb_channel_list_writer_construct (GType object_type, GFile* file);
GType dvb_channel_get_type (void);
const char* dvb_channel_get_Name (DVBChannel* self);
GType dvb_terrestrial_channel_get_type (void);
static void dvb_channel_list_writer_write_terrestrial_channel (DVBChannelListWriter* self, DVBTerrestrialChannel* channel, GError** error);
GType dvb_satellite_channel_get_type (void);
static void dvb_channel_list_writer_write_satellite_channel (DVBChannelListWriter* self, DVBSatelliteChannel* channel, GError** error);
GType dvb_cable_channel_get_type (void);
static void dvb_channel_list_writer_write_cable_channel (DVBChannelListWriter* self, DVBCableChannel* channel, GError** error);
GeeList* dvb_channel_get_AudioPIDs (DVBChannel* self);
guint dvb_channel_get_VideoPID (DVBChannel* self);
guint dvb_channel_get_Sid (DVBChannel* self);
void dvb_channel_list_writer_write (DVBChannelListWriter* self, DVBChannel* channel, GError** error);
gboolean dvb_channel_list_writer_close (DVBChannelListWriter* self, GError** error);
guint dvb_channel_get_Frequency (DVBChannel* self);
static char* dvb_channel_list_writer_get_name_without_prefix (GType enumtype, gint val, const char* prefix);
GType dvb_dvb_src_inversion_get_type (void);
DVBDvbSrcInversion dvb_terrestrial_channel_get_Inversion (DVBTerrestrialChannel* self);
GType dvb_dvb_src_bandwidth_get_type (void);
DVBDvbSrcBandwidth dvb_terrestrial_channel_get_Bandwidth (DVBTerrestrialChannel* self);
GType dvb_dvb_src_code_rate_get_type (void);
DVBDvbSrcCodeRate dvb_terrestrial_channel_get_CodeRateHP (DVBTerrestrialChannel* self);
DVBDvbSrcCodeRate dvb_terrestrial_channel_get_CodeRateLP (DVBTerrestrialChannel* self);
GType dvb_dvb_src_modulation_get_type (void);
DVBDvbSrcModulation dvb_terrestrial_channel_get_Constellation (DVBTerrestrialChannel* self);
GType dvb_dvb_src_transmission_mode_get_type (void);
DVBDvbSrcTransmissionMode dvb_terrestrial_channel_get_TransmissionMode (DVBTerrestrialChannel* self);
GType dvb_dvb_src_guard_get_type (void);
DVBDvbSrcGuard dvb_terrestrial_channel_get_GuardInterval (DVBTerrestrialChannel* self);
GType dvb_dvb_src_hierarchy_get_type (void);
DVBDvbSrcHierarchy dvb_terrestrial_channel_get_Hierarchy (DVBTerrestrialChannel* self);
const char* dvb_satellite_channel_get_Polarization (DVBSatelliteChannel* self);
gint dvb_satellite_channel_get_DiseqcSource (DVBSatelliteChannel* self);
guint dvb_satellite_channel_get_SymbolRate (DVBSatelliteChannel* self);
DVBDvbSrcInversion dvb_cable_channel_get_Inversion (DVBCableChannel* self);
guint dvb_cable_channel_get_SymbolRate (DVBCableChannel* self);
DVBDvbSrcCodeRate dvb_cable_channel_get_CodeRate (DVBCableChannel* self);
DVBDvbSrcModulation dvb_cable_channel_get_Modulation (DVBCableChannel* self);
const char* dvb_utils_get_name_by_value_from_enum (GType enumtype, gint val);
static void dvb_channel_list_writer_finalize (GObject* obj);
static void dvb_channel_list_writer_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void dvb_channel_list_writer_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);
static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);



static void dvb_channel_list_writer_open_stream (DVBChannelListWriter* self, GError** error) {
	GError * _inner_error_;
	GFileOutputStream* fostream;
	GOutputStream* _tmp4_;
	g_return_if_fail (self != NULL);
	_inner_error_ = NULL;
	fostream = NULL;
	if (g_file_query_exists (self->priv->_file, NULL)) {
		GFileOutputStream* _tmp0_;
		GFileOutputStream* _tmp1_;
		_tmp0_ = g_file_replace (self->priv->_file, NULL, TRUE, 0, NULL, &_inner_error_);
		if (_inner_error_ != NULL) {
			g_propagate_error (error, _inner_error_);
			(fostream == NULL) ? NULL : (fostream = (g_object_unref (fostream), NULL));
			return;
		}
		_tmp1_ = NULL;
		fostream = (_tmp1_ = _tmp0_, (fostream == NULL) ? NULL : (fostream = (g_object_unref (fostream), NULL)), _tmp1_);
	} else {
		GFileOutputStream* _tmp2_;
		GFileOutputStream* _tmp3_;
		_tmp2_ = g_file_create (self->priv->_file, 0, NULL, &_inner_error_);
		if (_inner_error_ != NULL) {
			g_propagate_error (error, _inner_error_);
			(fostream == NULL) ? NULL : (fostream = (g_object_unref (fostream), NULL));
			return;
		}
		_tmp3_ = NULL;
		fostream = (_tmp3_ = _tmp2_, (fostream == NULL) ? NULL : (fostream = (g_object_unref (fostream), NULL)), _tmp3_);
	}
	_tmp4_ = NULL;
	self->priv->stream = (_tmp4_ = (GOutputStream*) ((GBufferedOutputStream*) g_buffered_output_stream_new ((GOutputStream*) fostream)), (self->priv->stream == NULL) ? NULL : (self->priv->stream = (g_object_unref (self->priv->stream), NULL)), _tmp4_);
	(fostream == NULL) ? NULL : (fostream = (g_object_unref (fostream), NULL));
}


DVBChannelListWriter* dvb_channel_list_writer_construct (GType object_type, GFile* file) {
	GParameter * __params;
	GParameter * __params_it;
	DVBChannelListWriter * self;
	g_return_val_if_fail (file != NULL, NULL);
	__params = g_new0 (GParameter, 1);
	__params_it = __params;
	__params_it->name = "file";
	g_value_init (&__params_it->value, G_TYPE_FILE);
	g_value_set_object (&__params_it->value, file);
	__params_it++;
	self = g_object_newv (object_type, __params_it - __params, __params);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


DVBChannelListWriter* dvb_channel_list_writer_new (GFile* file) {
	return dvb_channel_list_writer_construct (DVB_TYPE_CHANNEL_LIST_WRITER, file);
}


void dvb_channel_list_writer_write (DVBChannelListWriter* self, DVBChannel* channel, GError** error) {
	GError * _inner_error_;
	char* buffer;
	char* _tmp0_;
	guint apid;
	char* _tmp1_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (channel != NULL);
	_inner_error_ = NULL;
	if (self->priv->stream == NULL) {
		dvb_channel_list_writer_open_stream (self, &_inner_error_);
		if (_inner_error_ != NULL) {
			g_propagate_error (error, _inner_error_);
			return;
		}
	}
	if (self->priv->stream == NULL) {
		return;
	}
	buffer = NULL;
	/* Write channel name*/
	_tmp0_ = NULL;
	buffer = (_tmp0_ = g_strdup_printf ("%s:", dvb_channel_get_Name (channel)), buffer = (g_free (buffer), NULL), _tmp0_);
	g_output_stream_write_all (self->priv->stream, buffer, strlen (buffer), NULL, NULL, &_inner_error_);
	if (_inner_error_ != NULL) {
		g_propagate_error (error, _inner_error_);
		buffer = (g_free (buffer), NULL);
		return;
	}
	/* Write special data*/
	if (DVB_IS_TERRESTRIAL_CHANNEL (channel)) {
		dvb_channel_list_writer_write_terrestrial_channel (self, DVB_TERRESTRIAL_CHANNEL (channel), &_inner_error_);
		if (_inner_error_ != NULL) {
			g_propagate_error (error, _inner_error_);
			buffer = (g_free (buffer), NULL);
			return;
		}
	} else {
		if (DVB_IS_SATELLITE_CHANNEL (channel)) {
			dvb_channel_list_writer_write_satellite_channel (self, DVB_SATELLITE_CHANNEL (channel), &_inner_error_);
			if (_inner_error_ != NULL) {
				g_propagate_error (error, _inner_error_);
				buffer = (g_free (buffer), NULL);
				return;
			}
		} else {
			if (DVB_IS_CABLE_CHANNEL (channel)) {
				dvb_channel_list_writer_write_cable_channel (self, DVB_CABLE_CHANNEL (channel), &_inner_error_);
				if (_inner_error_ != NULL) {
					g_propagate_error (error, _inner_error_);
					buffer = (g_free (buffer), NULL);
					return;
				}
			} else {
				g_warning ("ChannelListWriter.vala:76: Unknown channel type");
			}
		}
	}
	apid = 0U;
	if (gee_collection_get_size ((GeeCollection*) dvb_channel_get_AudioPIDs (channel)) == 0) {
		apid = (guint) 0;
	} else {
		apid = GPOINTER_TO_UINT (gee_list_get (dvb_channel_get_AudioPIDs (channel), 0));
	}
	/* Write common data*/
	_tmp1_ = NULL;
	buffer = (_tmp1_ = g_strdup_printf (":%u:%u:%u\n", dvb_channel_get_VideoPID (channel), apid, dvb_channel_get_Sid (channel)), buffer = (g_free (buffer), NULL), _tmp1_);
	g_output_stream_write_all (self->priv->stream, buffer, strlen (buffer), NULL, NULL, &_inner_error_);
	if (_inner_error_ != NULL) {
		g_propagate_error (error, _inner_error_);
		buffer = (g_free (buffer), NULL);
		return;
	}
	buffer = (g_free (buffer), NULL);
}


gboolean dvb_channel_list_writer_close (DVBChannelListWriter* self, GError** error) {
	gboolean result;
	GError * _inner_error_;
	gboolean _tmp0_;
	g_return_val_if_fail (self != NULL, FALSE);
	_inner_error_ = NULL;
	_tmp0_ = g_output_stream_close (self->priv->stream, NULL, &_inner_error_);
	if (_inner_error_ != NULL) {
		g_propagate_error (error, _inner_error_);
		return FALSE;
	}
	result = _tmp0_;
	return result;
}


static void dvb_channel_list_writer_write_terrestrial_channel (DVBChannelListWriter* self, DVBTerrestrialChannel* channel, GError** error) {
	GError * _inner_error_;
	char** _tmp0_;
	gint elements_size;
	gint elements_length1;
	char** elements;
	char* _tmp1_;
	char* _tmp2_;
	char* _tmp3_;
	char* _tmp4_;
	char* _tmp5_;
	char* _tmp6_;
	char* _tmp7_;
	char* _tmp8_;
	char* _tmp9_;
	char* buffer;
	g_return_if_fail (self != NULL);
	g_return_if_fail (channel != NULL);
	_inner_error_ = NULL;
	_tmp0_ = NULL;
	elements = (_tmp0_ = g_new0 (char*, 9 + 1), elements_length1 = 9, elements_size = elements_length1, _tmp0_);
	_tmp1_ = NULL;
	elements[0] = (_tmp1_ = g_strdup_printf ("%u", dvb_channel_get_Frequency ((DVBChannel*) channel)), elements[0] = (g_free (elements[0]), NULL), _tmp1_);
	_tmp2_ = NULL;
	elements[1] = (_tmp2_ = dvb_channel_list_writer_get_name_without_prefix (DVB_TYPE_DVB_SRC_INVERSION, (gint) dvb_terrestrial_channel_get_Inversion (channel), "DVB_DVB_SRC_INVERSION_"), elements[1] = (g_free (elements[1]), NULL), _tmp2_);
	_tmp3_ = NULL;
	elements[2] = (_tmp3_ = dvb_channel_list_writer_get_name_without_prefix (DVB_TYPE_DVB_SRC_BANDWIDTH, (gint) dvb_terrestrial_channel_get_Bandwidth (channel), "DVB_DVB_SRC_BANDWIDTH_"), elements[2] = (g_free (elements[2]), NULL), _tmp3_);
	_tmp4_ = NULL;
	elements[3] = (_tmp4_ = dvb_channel_list_writer_get_name_without_prefix (DVB_TYPE_DVB_SRC_CODE_RATE, (gint) dvb_terrestrial_channel_get_CodeRateHP (channel), "DVB_DVB_SRC_CODE_RATE_"), elements[3] = (g_free (elements[3]), NULL), _tmp4_);
	_tmp5_ = NULL;
	elements[4] = (_tmp5_ = dvb_channel_list_writer_get_name_without_prefix (DVB_TYPE_DVB_SRC_CODE_RATE, (gint) dvb_terrestrial_channel_get_CodeRateLP (channel), "DVB_DVB_SRC_CODE_RATE_"), elements[4] = (g_free (elements[4]), NULL), _tmp5_);
	_tmp6_ = NULL;
	elements[5] = (_tmp6_ = dvb_channel_list_writer_get_name_without_prefix (DVB_TYPE_DVB_SRC_MODULATION, (gint) dvb_terrestrial_channel_get_Constellation (channel), "DVB_DVB_SRC_MODULATION_"), elements[5] = (g_free (elements[5]), NULL), _tmp6_);
	_tmp7_ = NULL;
	elements[6] = (_tmp7_ = dvb_channel_list_writer_get_name_without_prefix (DVB_TYPE_DVB_SRC_TRANSMISSION_MODE, (gint) dvb_terrestrial_channel_get_TransmissionMode (channel), "DVB_DVB_SRC_TRANSMISSION_MODE_"), elements[6] = (g_free (elements[6]), NULL), _tmp7_);
	_tmp8_ = NULL;
	elements[7] = (_tmp8_ = dvb_channel_list_writer_get_name_without_prefix (DVB_TYPE_DVB_SRC_GUARD, (gint) dvb_terrestrial_channel_get_GuardInterval (channel), "DVB_DVB_SRC_GUARD_"), elements[7] = (g_free (elements[7]), NULL), _tmp8_);
	_tmp9_ = NULL;
	elements[8] = (_tmp9_ = dvb_channel_list_writer_get_name_without_prefix (DVB_TYPE_DVB_SRC_HIERARCHY, (gint) dvb_terrestrial_channel_get_Hierarchy (channel), "DVB_DVB_SRC_HIERARCHY_"), elements[8] = (g_free (elements[8]), NULL), _tmp9_);
	buffer = g_strjoinv (":", elements);
	g_output_stream_write_all (self->priv->stream, buffer, strlen (buffer), NULL, NULL, &_inner_error_);
	if (_inner_error_ != NULL) {
		g_propagate_error (error, _inner_error_);
		elements = (_vala_array_free (elements, elements_length1, (GDestroyNotify) g_free), NULL);
		buffer = (g_free (buffer), NULL);
		return;
	}
	elements = (_vala_array_free (elements, elements_length1, (GDestroyNotify) g_free), NULL);
	buffer = (g_free (buffer), NULL);
}


static void dvb_channel_list_writer_write_satellite_channel (DVBChannelListWriter* self, DVBSatelliteChannel* channel, GError** error) {
	GError * _inner_error_;
	char* buffer;
	g_return_if_fail (self != NULL);
	g_return_if_fail (channel != NULL);
	_inner_error_ = NULL;
	buffer = g_strdup_printf ("%u:%s:%d:%u", dvb_channel_get_Frequency ((DVBChannel*) channel) / 1000, dvb_satellite_channel_get_Polarization (channel), dvb_satellite_channel_get_DiseqcSource (channel), dvb_satellite_channel_get_SymbolRate (channel));
	g_output_stream_write_all (self->priv->stream, buffer, strlen (buffer), NULL, NULL, &_inner_error_);
	if (_inner_error_ != NULL) {
		g_propagate_error (error, _inner_error_);
		buffer = (g_free (buffer), NULL);
		return;
	}
	buffer = (g_free (buffer), NULL);
}


static void dvb_channel_list_writer_write_cable_channel (DVBChannelListWriter* self, DVBCableChannel* channel, GError** error) {
	GError * _inner_error_;
	char** _tmp0_;
	gint elements_size;
	gint elements_length1;
	char** elements;
	char* _tmp1_;
	char* _tmp2_;
	char* _tmp3_;
	char* _tmp4_;
	char* _tmp5_;
	char* buffer;
	g_return_if_fail (self != NULL);
	g_return_if_fail (channel != NULL);
	_inner_error_ = NULL;
	_tmp0_ = NULL;
	elements = (_tmp0_ = g_new0 (char*, 5 + 1), elements_length1 = 5, elements_size = elements_length1, _tmp0_);
	_tmp1_ = NULL;
	elements[0] = (_tmp1_ = g_strdup_printf ("%u", dvb_channel_get_Frequency ((DVBChannel*) channel)), elements[0] = (g_free (elements[0]), NULL), _tmp1_);
	_tmp2_ = NULL;
	elements[1] = (_tmp2_ = dvb_channel_list_writer_get_name_without_prefix (DVB_TYPE_DVB_SRC_INVERSION, (gint) dvb_cable_channel_get_Inversion (channel), "DVB_DVB_SRC_INVERSION_"), elements[1] = (g_free (elements[1]), NULL), _tmp2_);
	_tmp3_ = NULL;
	elements[2] = (_tmp3_ = g_strdup_printf ("%u", dvb_cable_channel_get_SymbolRate (channel) * 1000), elements[2] = (g_free (elements[2]), NULL), _tmp3_);
	_tmp4_ = NULL;
	elements[3] = (_tmp4_ = dvb_channel_list_writer_get_name_without_prefix (DVB_TYPE_DVB_SRC_CODE_RATE, (gint) dvb_cable_channel_get_CodeRate (channel), "DVB_DVB_SRC_CODE_RATE_"), elements[3] = (g_free (elements[3]), NULL), _tmp4_);
	_tmp5_ = NULL;
	elements[4] = (_tmp5_ = dvb_channel_list_writer_get_name_without_prefix (DVB_TYPE_DVB_SRC_MODULATION, (gint) dvb_cable_channel_get_Modulation (channel), "DVB_DVB_SRC_MODULATION_"), elements[4] = (g_free (elements[4]), NULL), _tmp5_);
	buffer = g_strjoinv (":", elements);
	g_output_stream_write_all (self->priv->stream, buffer, strlen (buffer), NULL, NULL, &_inner_error_);
	if (_inner_error_ != NULL) {
		g_propagate_error (error, _inner_error_);
		elements = (_vala_array_free (elements, elements_length1, (GDestroyNotify) g_free), NULL);
		buffer = (g_free (buffer), NULL);
		return;
	}
	elements = (_vala_array_free (elements, elements_length1, (GDestroyNotify) g_free), NULL);
	buffer = (g_free (buffer), NULL);
}


static char* string_substring (const char* self, glong offset, glong len) {
	char* result;
	glong string_length;
	const char* start;
	g_return_val_if_fail (self != NULL, NULL);
	string_length = g_utf8_strlen (self, -1);
	if (offset < 0) {
		offset = string_length + offset;
		g_return_val_if_fail (offset >= 0, NULL);
	} else {
		g_return_val_if_fail (offset <= string_length, NULL);
	}
	if (len < 0) {
		len = string_length - offset;
	}
	g_return_val_if_fail ((offset + len) <= string_length, NULL);
	start = g_utf8_offset_to_pointer (self, offset);
	result = g_strndup (start, ((gchar*) g_utf8_offset_to_pointer (start, len)) - ((gchar*) start));
	return result;
}


static char* dvb_channel_list_writer_get_name_without_prefix (GType enumtype, gint val, const char* prefix) {
	char* result;
	const char* _tmp0_;
	char* name;
	g_return_val_if_fail (prefix != NULL, NULL);
	_tmp0_ = NULL;
	name = (_tmp0_ = dvb_utils_get_name_by_value_from_enum (enumtype, val), (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_));
	if (name == NULL) {
		result = NULL;
		name = (g_free (name), NULL);
		return result;
	} else {
		result = string_substring (name, g_utf8_strlen (prefix, -1), g_utf8_strlen (name, -1) - g_utf8_strlen (prefix, -1));
		name = (g_free (name), NULL);
		return result;
	}
	name = (g_free (name), NULL);
}


GFile* dvb_channel_list_writer_get_file (DVBChannelListWriter* self) {
	GFile* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_file;
	return result;
}


static void dvb_channel_list_writer_set_file (DVBChannelListWriter* self, GFile* value) {
	GFile* _tmp1_;
	GFile* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_file = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_)), (self->priv->_file == NULL) ? NULL : (self->priv->_file = (g_object_unref (self->priv->_file), NULL)), _tmp1_);
	g_object_notify ((GObject *) self, "file");
}


static void dvb_channel_list_writer_class_init (DVBChannelListWriterClass * klass) {
	dvb_channel_list_writer_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (DVBChannelListWriterPrivate));
	G_OBJECT_CLASS (klass)->get_property = dvb_channel_list_writer_get_property;
	G_OBJECT_CLASS (klass)->set_property = dvb_channel_list_writer_set_property;
	G_OBJECT_CLASS (klass)->finalize = dvb_channel_list_writer_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), DVB_CHANNEL_LIST_WRITER_FILE, g_param_spec_object ("file", "file", "file", G_TYPE_FILE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
}


static void dvb_channel_list_writer_instance_init (DVBChannelListWriter * self) {
	self->priv = DVB_CHANNEL_LIST_WRITER_GET_PRIVATE (self);
}


static void dvb_channel_list_writer_finalize (GObject* obj) {
	DVBChannelListWriter * self;
	self = DVB_CHANNEL_LIST_WRITER (obj);
	(self->priv->_file == NULL) ? NULL : (self->priv->_file = (g_object_unref (self->priv->_file), NULL));
	(self->priv->stream == NULL) ? NULL : (self->priv->stream = (g_object_unref (self->priv->stream), NULL));
	G_OBJECT_CLASS (dvb_channel_list_writer_parent_class)->finalize (obj);
}


GType dvb_channel_list_writer_get_type (void) {
	static GType dvb_channel_list_writer_type_id = 0;
	if (dvb_channel_list_writer_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (DVBChannelListWriterClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_channel_list_writer_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBChannelListWriter), 0, (GInstanceInitFunc) dvb_channel_list_writer_instance_init, NULL };
		dvb_channel_list_writer_type_id = g_type_register_static (G_TYPE_OBJECT, "DVBChannelListWriter", &g_define_type_info, 0);
	}
	return dvb_channel_list_writer_type_id;
}


static void dvb_channel_list_writer_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	DVBChannelListWriter * self;
	gpointer boxed;
	self = DVB_CHANNEL_LIST_WRITER (object);
	switch (property_id) {
		case DVB_CHANNEL_LIST_WRITER_FILE:
		g_value_set_object (value, dvb_channel_list_writer_get_file (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void dvb_channel_list_writer_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	DVBChannelListWriter * self;
	self = DVB_CHANNEL_LIST_WRITER (object);
	switch (property_id) {
		case DVB_CHANNEL_LIST_WRITER_FILE:
		dvb_channel_list_writer_set_file (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	_vala_array_destroy (array, array_length, destroy_func);
	g_free (array);
}




