/*
 * Copyright (C) 2003 Sun Microsystems, Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 *
 * Authors:
 *    Erwann Chenede  <erwann.chenede@sun.com>
 *    Mark McLoughlin  <mark@skynet.ie>  
 */

#include <config.h>

#include "netstatus-sysdeps.h"

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <glib.h>
#include <libgnome/gnome-i18n.h>

static inline char *
parse_iface_name (const char *buf)
{
  char *p1;

  if ((p1 = strchr (buf, ':')))
    {
      char *p2;

      p2 = strchr (p1, ':');
      if (p2)
	*p2++ = '\0';
      else
	*p1++ = '\0';

      return p2 ? p2 : p1;
    }
  else if ((p1 = strchr (buf, ' ')))
    {
      *p1++ = '\0';
      return p1;
    }

  return NULL;
}

static inline void
parse_header (char *buf,
	      int  *prx_idx,
	      int  *ptx_idx,
	      int  *brx_idx,
	      int  *btx_idx)
{
  char *p;
  int   i;

  *prx_idx = *ptx_idx = -1;
  *brx_idx = *btx_idx = -1;

  p = strtok (buf, "| \t\n");
  p = strtok (NULL, "| \t\n"); /* Skip the first one */
  for (i = 0; p; i++, p = strtok (NULL, "| \t\n"))
    {
      if (!strcmp (p, "packets"))
	{
	  if (*prx_idx == -1)
	    *prx_idx = i;
	  else
	    *ptx_idx = i;
	}
      else if (!strcmp (p, "bytes"))
	{
	  if (*brx_idx == -1)
	    *brx_idx = i;
	  else
	    *btx_idx = i;
	}
    }
}

static inline int
parse_stats (char  *buf,
	     int    prx_idx,
	     int    ptx_idx,
	     long  *in_packets,
	     long  *out_packets,
	     int    brx_idx,
	     int    btx_idx,
	     long  *in_bytes,
	     long  *out_bytes)
{
  char *p;
  int   i;

  p = strtok (buf, " \t\n");
  for (i = 0; p; i++, p = strtok (NULL, " \t\n"))
    {
      if (i == prx_idx)
	*in_packets = g_ascii_strtoull (p, NULL, 10);
      if (i == ptx_idx)
	*out_packets = g_ascii_strtoull (p, NULL, 10);
      if (i == brx_idx)
	*in_bytes = g_ascii_strtoull (p, NULL, 10);
      if (i == btx_idx)
	*out_bytes = g_ascii_strtoull (p, NULL, 10);
    }

  if (i <= prx_idx || i <= ptx_idx || i <= brx_idx || i <=btx_idx)
    return FALSE;

  return TRUE;
}

static inline FILE *
get_proc_net_dev_fh (void)
{
  static FILE *retval = NULL;

  if (retval != NULL)
    return retval;

  return retval = fopen ("/proc/net/dev", "r");
}

char *
netstatus_sysdeps_read_iface_statistics (const char  *iface,
					 long        *in_packets,
					 long        *out_packets,
					 long        *in_bytes,
					 long        *out_bytes)
{
  FILE *fh;
  char  buf [512];
  int   prx_idx, ptx_idx;
  int   brx_idx, btx_idx;
  char *error_message = NULL;
  
  *in_packets = -1;
  *out_packets = -1;
  *in_bytes = -1;
  *out_bytes = -1;

  fh = get_proc_net_dev_fh ();
  if (!fh)
    return g_strdup_printf (_("Cannot open /proc/net/dev: %s"),
			    g_strerror (errno));

  fgets (buf, sizeof (buf), fh);
  fgets (buf, sizeof (buf), fh);

  parse_header (buf, &prx_idx, &ptx_idx, &brx_idx, &btx_idx);
  if (prx_idx == -1 || ptx_idx == -1 ||
      brx_idx == -1 || btx_idx == -1)
    return g_strdup (_("Could not parse /proc/net/dev. Unknown format."));

  while (fgets (buf, sizeof (buf), fh))
    {
      char *stats;
      char *name;

      name = buf;
      while (g_ascii_isspace (name [0]))
	name++;

      stats = parse_iface_name (name);
      if (!stats)
	{
	  if (!error_message)
	    error_message = g_strdup_printf (_("Could not parse interface name from '%s'"), buf);
	  continue;
	}

      if (strcmp (name, iface) != 0)
	continue;

      if (!parse_stats (stats,
			prx_idx, ptx_idx, in_packets, out_packets,
			brx_idx, btx_idx, in_bytes, out_bytes))
	{
	  if (error_message)
	    g_free (error_message);
	  error_message = g_strdup_printf (_("Could not parse interface statistics from '%s'. "
					     "prx_idx = %d; ptx_idx = %d; brx_idx = %d; btx_idx = %d;"),
					   buf, prx_idx, ptx_idx, brx_idx, btx_idx);
	  continue;
	}
    }

  if ((*in_packets == -1 || *out_packets == -1 || *in_bytes == -1 || *out_bytes == -1) && !error_message)
    error_message = g_strdup_printf ("Could not find information on interface '%s' in /proc/net/dev", iface);

  rewind (fh);

  return error_message;
}
