/* gnome-ppp - The GNOME PPP Dialer
 * Copyright (C) 1997 Jay Painter
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 *(at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <gnome.h>
#include "gnome-ppp.h"
#include "account-window.h"
#include "dial-window.h"
#include "global.h"
#include "misc.h"


enum
{
  _DISCONNECTED,
  _IN_PROGRESS,
  _CONNECTED,
  _ERROR
};


/* dial application context structure */
typedef struct _Dial
{
  GtkWidget *app;
  GtkWidget *account_clist;
  GtkWidget *appbar;
  GtkWidget *connect_button_label;
  GtkWidget *connect;

  Account *account;
  gint state;
  gint timer_id;
} Dial;



static void refresh_dial_windows();
static void refresh_dial_app(Dial *dial);
static void refresh_dial_account_list(Dial *dial);
static gint close_dial_app(GtkWidget *widget, Dial *dial);


/* app callbacks */
static void destroy_dial_app_cb(GtkWidget *widget);
static void account_list_select_cb(GtkWidget *widget, gint row, gint col, 
				   GdkEventButton *bevent);
static void connect_button_cb(GtkWidget *widget);


/* dial engine callbacks */
static gint timeout_cb(Dial *dial);
static void connect_message_cb(Account *account, PPPMessageType message, gchar *text, gpointer data);


/* menu callbacks */
static void menu_file_new_dialer_cb(GtkWidget *widget, Dial *dial);
static void menu_file_exit_cb(GtkWidget *widget, Dial *dial);
static void menu_account_new_cb(GtkWidget *widget, Dial *dial);
static void menu_account_duplicate_cb(GtkWidget *widget, Dial *dial);
static void menu_account_edit_cb(GtkWidget *widget, Dial *dial);
static void menu_account_delete_cb(GtkWidget *widget, Dial *dial);
static void menu_help_about_cb(GtkWidget *widget, Dial *dial);


/* dial application context */
static Dial *malloc_dial();
static void free_dial(Dial *dial);
static Dial *get_dial_context(GtkObject *object);
static void set_dial_context(GtkObject *object, Dial *dial);


/* menus */
static GnomeUIInfo file_menu[] = 
{
  GNOMEUIINFO_MENU_NEW_ITEM("New Dialer", NULL, menu_file_new_dialer_cb, NULL),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_MENU_EXIT_ITEM(menu_file_exit_cb, NULL),
  GNOMEUIINFO_END
};


static GnomeUIInfo account_menu[] = 
{
  GNOMEUIINFO_ITEM_NONE("New...", NULL, menu_account_new_cb),
  GNOMEUIINFO_ITEM_NONE("Duplicate...", NULL, menu_account_duplicate_cb),
  GNOMEUIINFO_ITEM_NONE("Edit...", NULL, menu_account_edit_cb),
  GNOMEUIINFO_ITEM_NONE("Delete...", NULL, menu_account_delete_cb),
  GNOMEUIINFO_END
};


static GnomeUIInfo help_menu[] =
{
  GNOMEUIINFO_MENU_ABOUT_ITEM(menu_help_about_cb, NULL),
  GNOMEUIINFO_END
};


/* The menu definitions: File/Exit and Help/About are mandatory */
static GnomeUIInfo main_menu[] =
{
  GNOMEUIINFO_MENU_FILE_TREE(file_menu),

  { GNOME_APP_UI_SUBTREE, N_("_Account"), NULL, account_menu,
    NULL, NULL, (GnomeUIPixmapType) 0, NULL, 0, (GdkModifierType) 0,
    NULL },

  GNOMEUIINFO_MENU_HELP_TREE(help_menu),

  GNOMEUIINFO_END
};


/* keep track of the number of dial applications */
static GList *__dial_app_list = NULL;


void
new_dial_app()
{
  static gboolean watcher_init = FALSE;
  Dial *dial;
  GtkWidget *vbox, *hbox;

  /* initalize account watcher to update the clist of all dial
   * application windows
   */
  if (!watcher_init)
    {
      watcher_init = TRUE;
      account_add_watcher(refresh_dial_windows);
    }

  dial = malloc_dial();

  /* set up the main application */
  dial->app = gnome_app_new("gnome-ppp", "gnome-ppp");
  set_dial_context(GTK_OBJECT(dial->app), dial);

  /* menus & status bar are imbedded into the gnome-app */
  gnome_app_create_menus_with_data(GNOME_APP(dial->app), main_menu, dial);

  dial->appbar = gnome_appbar_new(FALSE, TRUE, GNOME_PREFERENCES_USER);
  gnome_app_set_statusbar(GNOME_APP(dial->app), GTK_WIDGET(dial->appbar));

  gtk_signal_connect(
      GTK_OBJECT(dial->app),
      "destroy",
     (GtkSignalFunc) destroy_dial_app_cb,
      NULL);

  gtk_signal_connect(
      GTK_OBJECT(dial->app),
      "delete_event",
     (GtkSignalFunc) close_dial_app,
      NULL);

  /* hbox */
  hbox =  gtk_hbox_new(FALSE, 0);
  gnome_app_set_contents(GNOME_APP(dial->app), hbox);
  gtk_container_border_width(GTK_CONTAINER(hbox), 3);
  gtk_widget_show(hbox);

  /* boxes for widget layout */
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);
  gtk_box_set_spacing(GTK_BOX(vbox), 5);
  gtk_widget_show(vbox);

  /* account clist */
  dial->account_clist = gtk_clist_new(1);
  set_dial_context(GTK_OBJECT(dial->account_clist), dial);
  gtk_box_pack_start(GTK_BOX(vbox), dial->account_clist, TRUE, TRUE, 0);

  gtk_clist_column_titles_passive(GTK_CLIST(dial->account_clist));
  gtk_clist_set_selection_mode(GTK_CLIST(dial->account_clist), GTK_SELECTION_BROWSE);

  gtk_signal_connect(
      GTK_OBJECT(dial->account_clist),
      "select_row",
      (GtkSignalFunc) account_list_select_cb,
      NULL);

  gtk_widget_show(dial->account_clist);

  /* connect button */
  dial->connect_button_label = gtk_label_new("");
  set_dial_context(GTK_OBJECT(dial->connect_button_label), dial);
  gtk_widget_show(dial->connect_button_label);

  dial->connect = gtk_button_new();
  set_dial_context(GTK_OBJECT(dial->connect), dial);
  gtk_box_pack_start(GTK_BOX(vbox), dial->connect, FALSE, FALSE, 0);

  gtk_container_add(GTK_CONTAINER(dial->connect), dial->connect_button_label);
  gtk_widget_set_usize(dial->connect, 100, GPPP_BUTTON_HEIGHT);

  gtk_signal_connect(
      GTK_OBJECT(dial->connect),
      "clicked",
     (GtkSignalFunc) connect_button_cb,
      NULL);

  gtk_widget_show(dial->connect);

  /* set data in the dial application */
  refresh_dial_account_list(dial);
  refresh_dial_app(dial);

  /* Show it */
  gtk_widget_show(dial->app);
}


void
refresh_dial_windows()
{
  GList *list;
  Dial *dial;

  /* refresh all the dial applications */
  list = __dial_app_list;
  while (list)
    {
      dial = (Dial *) list->data;
      list = list->next;
      refresh_dial_account_list(dial);
    } 
}


static void
refresh_dial_account_list(Dial *dial)
{
  GList *list;
  Account *account;

  /* handle setting the default account context correctly */
  if (g_list_length(account_list()) == 0)
    {
      dial->account = NULL;
    }
  else if (!dial->account || !g_list_find(account_list(), dial->account))
    {
      dial->account = (Account *) account_list()->data;
    }

  /* refresh the accounts clist */
  gtk_clist_freeze(GTK_CLIST(dial->account_clist));
  gtk_clist_clear(GTK_CLIST(dial->account_clist));

  list = account_list();
  while (list)
    {
      gint row;
      gchar *text[1];

      account = (Account *) list->data;
      list = list->next;

      text[0] = account->name->str;
      row = gtk_clist_append(GTK_CLIST(dial->account_clist), text);
      if (row < 0)
	{
	  g_error("refresh_dial_app(): gtk_clist_append failed.");
	}
      gtk_clist_set_row_data(GTK_CLIST(dial->account_clist), row, account);

      /* select this row if it is the default account */
      if (account == dial->account)
	{
	  gtk_clist_select_row(GTK_CLIST(dial->account_clist), row, 0);
	}
    }
  gtk_clist_thaw(GTK_CLIST(dial->account_clist));
}


static void
refresh_dial_app(Dial *dial)
{
  /* set the active state of the connection button and the
   * account-selection option menu based on the connection state
   */
  if (dial->account)
    {
      switch (dial->state)
	{
	case _DISCONNECTED:
	  gtk_label_set_text(GTK_LABEL(dial->connect_button_label), "Connect");
	  gtk_widget_set_sensitive(GTK_WIDGET(dial->connect), TRUE);
	  gtk_widget_set_sensitive(GTK_WIDGET(dial->account_clist), TRUE);
	  break;
	  
	case _ERROR:
	  gtk_label_set_text(GTK_LABEL(dial->connect_button_label), "---");
	  gtk_widget_set_sensitive(GTK_WIDGET(dial->connect), FALSE);
	  gtk_widget_set_sensitive(GTK_WIDGET(dial->account_clist), FALSE);
	  break;

	case _IN_PROGRESS:
	  gtk_label_set_text(GTK_LABEL(dial->connect_button_label), "Cancel");
	  gtk_widget_set_sensitive(GTK_WIDGET(dial->connect), TRUE);
	  gtk_widget_set_sensitive(GTK_WIDGET(dial->account_clist), FALSE);
	  break;

	case _CONNECTED:
	  gtk_label_set_text(GTK_LABEL(dial->connect_button_label), "Disconnect");
	  gtk_widget_set_sensitive(GTK_WIDGET(dial->connect), TRUE);
	  gtk_widget_set_sensitive(GTK_WIDGET(dial->account_clist), FALSE);
	  break;
	}
    }
  else
    {
      gtk_label_set_text(GTK_LABEL(dial->connect_button_label), "Connect");
      gtk_widget_set_sensitive(GTK_WIDGET(dial->connect), FALSE);
    }
}


static gint
close_dial_app(GtkWidget *widget, Dial *dial)
{
  dial = get_dial_context(GTK_OBJECT(widget));

  /* don't close if the connection is active */
  if (dial->state != _DISCONNECTED)
    {
      gnome_error_dialog(_("You must disconnect before closing."));
      return TRUE;
    }

  return FALSE;
}


static void
destroy_dial_app_cb(GtkWidget *widget)
{ 
  Dial *dial;

  dial = get_dial_context(GTK_OBJECT(widget));

  free_dial(dial);

  /* quit gnome-ppp when there are no moer dial applications */
  if (__dial_app_list == NULL)
    {
      account_remove_watcher(refresh_dial_windows);
      gtk_main_quit();
    }
}


/* callback for clist */
static void 
account_list_select_cb(GtkWidget *widget, gint row, gint col, GdkEventButton *bevent)
{
  Account *account;
  Dial *dial;

  dial = get_dial_context(GTK_OBJECT(widget));

  account = gtk_clist_get_row_data(GTK_CLIST(widget), row);
  if (account == NULL)
    {
      return;
    }

  dial->account = account;

  if (!g_list_find(account_list(), account))
    {
      g_error("account_list_select_cb(): account from clist_date not in account list");
    }
}


static void
connect_button_cb(GtkWidget *widget)
{
  Dial *dial;

  dial = get_dial_context(GTK_OBJECT(widget));

  /* we need an account selected to connect */
  if (!g_list_find(account_list(), dial->account))
    {
      g_error("internal error !g_list_find(account_list(), dial->account");
    }

  switch (dial->state)
    {
    case _DISCONNECTED:
      /* locked accounts are already in use */
      if (dial->account->locked)
	{
	  gnome_error_dialog(_("This account is already in use."));
	}
      else
	{
	  dial->account->locked = TRUE;
	}

      /* start the connection; if the start is successful, then
       * add the timeout callback to run the state engine
       */
      if (connect_start(dial->account, connect_message_cb, dial))
	{
	  dial->timer_id = 
	    gtk_timeout_add(500, (GtkFunction) timeout_cb, dial);
	}
      else
	{
	  dial->account->locked = FALSE;
	}
    break;

    case _ERROR:
      g_error("connect_button_cb(): called while in ERROR state");
      break;

    case _IN_PROGRESS:
    case _CONNECTED:
      connect_stop(dial->account);
    break;
    }
}


static gint
timeout_cb(Dial *dial)
{
  if (connect_engine_iteration(dial->account))
    {
      return 1;
    }
  else
    {
      /* blank out the timeout id and return 0 shutting
       * down the gtk_timeout
       */
      dial->timer_id = -1;
      return 0;
    }
}


/* message callback from connect */
static void
connect_message_cb(Account *account, PPPMessageType message, gchar *text, gpointer data)
{
  gint saved_state;
  Dial *dial;

  dial = (Dial *) data;

  /* sanity checks */
  g_assert(account != NULL);
  if (dial->account != account)
    {
      g_error("connect_message_cb(): account=%s but dial->account=%s",
          account->name->str,
	  dial->account->name->str);
    }

  saved_state = dial->state;

  /* map messages from the connect engine to internal
   * states of the dial window
   */
  switch (message)
    {
    case PPP_DISCONNECTED:
      dial->state = _DISCONNECTED;
      break;

    case PPP_ERROR:
      dial->state = _ERROR;
      gnome_error_dialog(text);
      break;

    case PPP_IN_PROGRESS:
      dial->state = _IN_PROGRESS;
      break;

    case PPP_CONNECTED:
      dial->state = _CONNECTED;
      break;
    }

  /* refresh the dial window if the state changed */
  if (dial->state != saved_state)
    {
      refresh_dial_app(dial);
    }

  /* display a message if there is one */
  if (text)
    {
      gnome_appbar_set_status(GNOME_APPBAR(dial->appbar), text);
    }
  else
    {
      gnome_appbar_set_status(GNOME_APPBAR(dial->appbar), "");
    }

  /* if the state is disconnected, then unlock the account */
  if (dial->state == _DISCONNECTED)
    {
      account->locked = FALSE;
    }
}


/* menu callbacks need to be different */
static void
menu_file_new_dialer_cb(GtkWidget *widget, Dial *dial)
{
  g_assert(dial != NULL);
  new_dial_app();
}


static void
menu_file_exit_cb(GtkWidget *widget, Dial *dial)
{
  g_assert(dial != NULL);

  if (close_dial_app(dial->app, dial) == FALSE)
    {
      gtk_widget_destroy(dial->app);
    }
}


static void
menu_account_new_cb(GtkWidget *widget, Dial *dial)
{
  g_assert(dial != NULL);
  open_account_window(NULL, FALSE);
}


static void
menu_account_duplicate_cb(GtkWidget *widget, Dial *dial)
{
  g_assert(dial != NULL);

  if (dial->account == NULL)
    {
      return;
    }

  open_account_window(dial->account, TRUE);
}


static void
menu_account_edit_cb(GtkWidget *widget, Dial *dial)
{
  g_assert(dial != NULL);

  if (dial->account == NULL)
    {
      return;
    }

  open_account_window(dial->account, FALSE);
}


static void
menu_account_delete_cb(GtkWidget *widget, Dial *dial)
{
  g_assert(dial != NULL);

  if (dial->account == NULL)
    {
      return;
    }

  /* can't delete locked (in-use) accounts */
  if (dial->account->locked)
    {
      GString *gstr = g_string_new("");

      g_string_sprintf(
          gstr,
	  "Account %s is currently in use and cannot be removed.", 
	  dial->account->name->str);

      gnome_error_dialog(gstr->str);

      g_string_free(gstr, TRUE);
      return;
    }

  /* try to remove the account */
  if (!account_delete(dial->account))
    {
      GString *gstr = g_string_new("");
      
      g_string_sprintf(
	   gstr,
	  "Account %s cannot be removed.",
	  dial->account->name->str);
      gnome_error_dialog(gstr->str);

      g_string_free(gstr, TRUE);
      return;
    }

  account_save();
}


static void
menu_help_about_cb(GtkWidget *widget, Dial *dial)
{
  GtkWidget *about;
  const gchar *authors[] = 
  {
    "Jay Painter",
    NULL
  };
	
  about = gnome_about_new(
      "gnome-ppp", 
      "0.3",
      "(C) 1998,1999 Jay Painter",
      authors,
      NULL,
      NULL);

  gtk_widget_show(about);
}


/* dial application context */
static Dial*
malloc_dial()
{
  Dial *dial = g_malloc(sizeof(Dial));
  dial->account = NULL;
  dial->state = _DISCONNECTED;
  dial->timer_id = -1;

  __dial_app_list = g_list_append(__dial_app_list, dial);
  
  return dial;
}


static void
free_dial(Dial *dial)
{
  __dial_app_list = g_list_remove(__dial_app_list, dial);
  g_free(dial);
}


static Dial* 
get_dial_context(GtkObject *object)
{
  return gtk_object_get_data(object, "dial_context");
}


static void
set_dial_context(GtkObject *object, Dial *dial)
{
  gtk_object_set_data(object, "dial_context", dial);
}

