/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*-
 *
 * Copyright (C) 2013 Richard Hughes <richard@hughsie.com>
 *
 * Licensed under the GNU General Public License Version 2
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <config.h>

#include <gs-plugin.h>

struct GsPluginPrivate {
	GHashTable		*cache;
	gsize    		 loaded;
};

/**
 * gs_plugin_get_name:
 */
const gchar *
gs_plugin_get_name (void)
{
	return "hardcoded-ratings";
}

/**
 * gs_plugin_initialize:
 */
void
gs_plugin_initialize (GsPlugin *plugin)
{
	/* create private area */
	plugin->priv = GS_PLUGIN_GET_PRIVATE (GsPluginPrivate);
	plugin->priv->loaded = FALSE;
	plugin->priv->cache = g_hash_table_new_full (g_str_hash,
						     g_str_equal,
						     NULL,
						     NULL);
}

/**
 * gs_plugin_get_priority:
 */
gdouble
gs_plugin_get_priority (GsPlugin *plugin)
{
	return -100.0f;
}

/**
 * gs_plugin_destroy:
 */
void
gs_plugin_destroy (GsPlugin *plugin)
{
	g_hash_table_unref (plugin->priv->cache);
}

/**
 * gs_plugin_hardcoded_ratings_setup:
 *
 * The data was generated by contrib/get-ratings-for-fas.py on 2013-03-07
 */
static gboolean
gs_plugin_hardcoded_ratings_setup (GsPlugin *plugin, GError **error)
{
	guint i;
	static struct {
		gint value;
		const gchar *id;
	} ratings[] = {
		{ 26,	"AcetoneISO2" },
		{ 67,	"Azureus" },
		{ 100,	"HP-hplip" },
		{ 44,	"MySQLWorkbench" },
		{ 18,	"PyMca" },
		{ 21,	"QtDMM" },
		{ 87,	"Terminal" },
		{ 100,	"Thunar" },
		{ 100,	"Thunar-bulk-rename" },
		{ 100,	"Thunar-folder-handler" },
		{ 28,	"YAGF" },
		{ 54,	"abiword" },
		{ 38,	"abrt" },
		{ 18,	"acftperf" },
		{ 21,	"airsnort" },
		{ 20,	"alacarte" },
		{ 51,	"alexandria" },
		{ 18,	"alienarena" },
		{ 20,	"alltray" },
		{ 20,	"amsn" },
		{ 33,	"anki" },
		{ 22,	"apvlv" },
		{ 25,	"aqsis" },
		{ 25,	"aqsl" },
		{ 25,	"aqsltell" },
		{ 42,	"ardour2" },
		{ 23,	"arduino" },
		{ 22,	"arora" },
		{ 20,	"asunder" },
		{ 100,	"at-properties" },
		{ 100,	"audacious" },
		{ 100,	"audacious-amidi" },
		{ 62,	"audacity" },
		{ 32,	"authconfig" },
		{ 22,	"autocal" },
		{ 26,	"avahi-discover" },
		{ 51,	"avogadro" },
		{ 17,	"ayttm" },
		{ 31,	"backintime-gnome" },
		{ 31,	"backintime-gnome-root" },
		{ 48,	"bacula-bat" },
		{ 53,	"bacula-traymonitor" },
		{ 27,	"balsa" },
		{ 27,	"balsa-mailto-handler" },
		{ 100,	"banshee-audiocd" },
		{ 100,	"banshee-media-player" },
		{ 42,	"baobab" },
		{ 19,	"berusky" },
		{ 65,	"bibletime" },
		{ 22,	"bisho" },
		{ 32,	"blender" },
		{ 68,	"bluefish" },
		{ 20,	"blueman-manager" },
		{ 100,	"bluetooth-properties" },
		{ 22,	"boinc-manager" },
		{ 55,	"brasero" },
		{ 42,	"brasero-nautilus" },
		{ 26,	"bssh" },
		{ 26,	"bvnc" },
		{ 34,	"bzflag" },
		{ 100,	"caja" },
		{ 100,	"caja-autorun-software" },
		{ 100,	"caja-browser" },
		{ 100,	"caja-computer" },
		{ 100,	"caja-file-management-properties" },
		{ 100,	"caja-folder-handler" },
		{ 100,	"caja-home" },
		{ 30,	"calf-dssi" },
		{ 86,	"calibre-ebook-viewer" },
		{ 86,	"calibre-gui" },
		{ 86,	"calibre-lrfviewer" },
		{ 46,	"caribou" },
		{ 20,	"cellwriter" },
		{ 18,	"centerim" },
		{ 27,	"chatzilla" },
		{ 47,	"cheese" },
		{ 28,	"cherrytree" },
		{ 24,	"chkrootkit" },
		{ 27,	"clamtk" },
		{ 48,	"claws-mail" },
		{ 39,	"clementine" },
		{ 28,	"clusterssh" },
		{ 17,	"colossus" },
		{ 20,	"coot" },
		{ 18,	"dconf-editor" },
		{ 100,	"default-applications" },
		{ 37,	"deja-dup" },
		{ 37,	"deja-dup-preferences" },
		{ 37,	"denemo" },
		{ 22,	"dhc" },
		{ 20,	"diffuse" },
		{ 24,	"dillo" },
		{ 100,	"display-properties" },
		{ 21,	"djvulibre-djview4" },
		{ 19,	"drgeo" },
		{ 27,	"dx" },
		{ 22,	"dxr3view" },
		{ 18,	"easytag" },
		{ 55,	"eclipse" },
		{ 21,	"edb" },
		{ 32,	"eekboard" },
		{ 25,	"eiciel" },
		{ 46,	"emacs" },
		{ 51,	"emacs-terminal" },
		{ 46,	"emacsclient" },
		{ 28,	"emelfm2" },
		{ 38,	"emerillon" },
		{ 100,	"empathy" },
		{ 92,	"engrampa" },
		{ 25,	"epdfview" },
		{ 25,	"eqsl" },
		{ 35,	"esc" },
		{ 20,	"etherape" },
		{ 19,	"etracer" },
		{ 19,	"ettercap" },
		{ 100,	"evince" },
		{ 100,	"evolution" },
		{ 53,	"evolution-calendar" },
		{ 37,	"exaile" },
		{ 22,	"exfalso" },
		{ 100,	"exo-file-manager" },
		{ 100,	"exo-mail-reader" },
		{ 100,	"exo-preferred-applications" },
		{ 100,	"exo-terminal-emulator" },
		{ 100,	"exo-web-browser" },
		{ 30,	"f-spot" },
		{ 30,	"f-spot-import" },
		{ 30,	"f-spot-view" },
		{ 28,	"fcitx" },
		{ 28,	"fcitx-configtool" },
		{ 28,	"fcitx-skin-installer" },
		{ 33,	"ffado.org-ffadomixer" },
		{ 40,	"ffbatterymon" },
		{ 40,	"ffconfiggui" },
		{ 40,	"ffnetloggui" },
		{ 40,	"ffplugingui" },
		{ 18,	"filezilla" },
		{ 60,	"firefox" },
		{ 40,	"firestation" },
		{ 25,	"firewall-config" },
		{ 19,	"fityk" },
		{ 26,	"flarq" },
		{ 26,	"fldigi" },
		{ 18,	"flightdeck" },
		{ 48,	"flumotion-admin" },
		{ 40,	"font-manager" },
		{ 40,	"font-sampler" },
		{ 39,	"fontforge" },
		{ 17,	"fotoprint" },
		{ 44,	"freecell" },
		{ 48,	"frescobaldi" },
		{ 19,	"fslint" },
		{ 40,	"fvfuseviewer" },
		{ 19,	"fwbackups" },
		{ 52,	"gabedit" },
		{ 30,	"gajim" },
		{ 46,	"gambas2" },
		{ 39,	"gambas3" },
		{ 21,	"ganyremote" },
		{ 48,	"gapcmon" },
		{ 20,	"gaupol" },
		{ 29,	"gausssum" },
		{ 28,	"gbrainy" },
		{ 57,	"gcalctool" },
		{ 100,	"gchem3d-0.14" },
		{ 100,	"gchemcalc-0.14" },
		{ 100,	"gchempaint-0.14" },
		{ 100,	"gchemtable-0.14" },
		{ 18,	"gcin-setup" },
		{ 93,	"gcm-calibrate" },
		{ 93,	"gcm-import" },
		{ 93,	"gcm-picker" },
		{ 93,	"gcm-viewer" },
		{ 100,	"gcrystal-0.14" },
		{ 45,	"gda-browser-5.0" },
		{ 45,	"gda-control-center-5.0" },
		{ 27,	"gdesklets" },
		{ 21,	"gdpc" },
		{ 36,	"geany" },
		{ 100,	"gedit" },
		{ 32,	"geeqie" },
		{ 22,	"gendarme-wizard" },
		{ 19,	"ggz-gtk" },
		{ 26,	"ghemical" },
		{ 79,	"gimp" },
		{ 18,	"git-cola" },
		{ 18,	"git-dag" },
		{ 32,	"git-gui" },
		{ 21,	"gitg" },
		{ 35,	"gkbd-keyboard-display" },
		{ 32,	"glade" },
		{ 32,	"glade-3" },
		{ 63,	"globaltime" },
		{ 28,	"glom" },
		{ 21,	"glyphtracer" },
		{ 62,	"gnash" },
		{ 22,	"gnomad2" },
		{ 53,	"gnome-about" },
		{ 38,	"gnome-abrt" },
		{ 100,	"gnome-banshee" },
		{ 71,	"gnome-commander" },
		{ 51,	"gnome-contacts" },
		{ 100,	"gnome-control-center" },
		{ 48,	"gnome-devhelp" },
		{ 59,	"gnome-dictionary" },
		{ 26,	"gnome-documents" },
		{ 31,	"gnome-ekiga" },
		{ 100,	"gnome-eog" },
		{ 92,	"gnome-file-roller" },
		{ 24,	"gnome-gconf-editor" },
		{ 22,	"gnome-genius" },
		{ 61,	"gnome-glchess" },
		{ 67,	"gnome-glines" },
		{ 36,	"gnome-gmail" },
		{ 67,	"gnome-gnect" },
		{ 67,	"gnome-gnibbles" },
		{ 67,	"gnome-gnobots2" },
		{ 74,	"gnome-gnomine" },
		{ 67,	"gnome-gnotravex" },
		{ 67,	"gnome-gnotski" },
		{ 67,	"gnome-gtali" },
		{ 57,	"gnome-gucharmap" },
		{ 66,	"gnome-iagno" },
		{ 25,	"gnome-istanbul" },
		{ 34,	"gnome-lightsoff" },
		{ 25,	"gnome-lirc-properties" },
		{ 67,	"gnome-mahjongg" },
		{ 26,	"gnome-phone-manager" },
		{ 52,	"gnome-power-statistics" },
		{ 64,	"gnome-quadrapassel" },
		{ 22,	"gnome-sabayon" },
		{ 28,	"gnome-screenshot" },
		{ 42,	"gnome-search-tool" },
		{ 70,	"gnome-session-properties" },
		{ 53,	"gnome-shell" },
		{ 53,	"gnome-shell-extension-prefs" },
		{ 55,	"gnome-sound-recorder" },
		{ 33,	"gnome-subtitles" },
		{ 51,	"gnome-sudoku" },
		{ 40,	"gnome-swell-foop" },
		{ 100,	"gnome-system-monitor" },
		{ 87,	"gnome-terminal" },
		{ 34,	"gnome-user-share-properties" },
		{ 36,	"gnome-vino-preferences" },
		{ 100,	"gnome-yelp" },
		{ 29,	"gnomeradio" },
		{ 28,	"gnote" },
		{ 27,	"gnuclient" },
		{ 100,	"gnumeric" },
		{ 20,	"goldendict" },
		{ 46,	"gourmet" },
		{ 64,	"gparted" },
		{ 94,	"gphotoframe" },
		{ 100,	"gpicview" },
		{ 59,	"gpk-application" },
		{ 59,	"gpk-dbus-service" },
		{ 59,	"gpk-distro-upgrade" },
		{ 59,	"gpk-install-catalog" },
		{ 59,	"gpk-install-local-file" },
		{ 59,	"gpk-log" },
		{ 59,	"gpk-prefs" },
		{ 59,	"gpk-service-pack" },
		{ 59,	"gpk-update-viewer" },
		{ 100,	"gpodder" },
		{ 27,	"gpredict" },
		{ 18,	"gpsdrive" },
		{ 52,	"gretl" },
		{ 22,	"griffith" },
		{ 26,	"groovy" },
		{ 23,	"gscan2pdf" },
		{ 22,	"gsharp" },
		{ 100,	"gspectrum-0.14" },
		{ 55,	"gstreamer-properties" },
		{ 18,	"gtg" },
		{ 87,	"gthumb" },
		{ 87,	"gthumb-import" },
		{ 32,	"gtk-gnutella" },
		{ 29,	"gtk-v4l" },
		{ 55,	"gtkwave" },
		{ 76,	"guake" },
		{ 76,	"guake-prefs" },
		{ 46,	"guitarix" },
		{ 27,	"gummi" },
		{ 33,	"gvim" },
		{ 24,	"gweled" },
		{ 71,	"gwibber" },
		{ 28,	"gwsmhg" },
		{ 21,	"hedgewars" },
		{ 19,	"hexter" },
		{ 23,	"hocr-gtk" },
		{ 19,	"homebank" },
		{ 18,	"hotot" },
		{ 18,	"hotot" },
		{ 78,	"hydrogen" },
		{ 41,	"ibus-setup" },
		{ 22,	"ilcontrast" },
		{ 99,	"im-chooser" },
		{ 23,	"immix" },
		{ 30,	"inkscape" },
		{ 37,	"iok" },
		{ 21,	"jack-rack" },
		{ 35,	"jack_capture" },
		{ 25,	"jamin" },
		{ 77,	"java-1.7.0-openjdk-jconsole" },
		{ 77,	"java-1.7.0-openjdk-policytool" },
		{ 52,	"javaws" },
		{ 29,	"jaxodraw" },
		{ 63,	"jbrout" },
		{ 32,	"jd" },
		{ 19,	"jigdo" },
		{ 59,	"jmol" },
		{ 72,	"josm" },
		{ 19,	"k3d" },
		{ 25,	"kanyremote" },
		{ 100,	"keyboard" },
		{ 19,	"kflickr" },
		{ 27,	"kid3" },
		{ 31,	"klash" },
		{ 20,	"klatexformula" },
		{ 17,	"klavaro" },
		{ 29,	"krb5-auth-dialog" },
		{ 100,	"kshutdown" },
		{ 21,	"ksmarttray" },
		{ 30,	"kst" },
		{ 40,	"lasergui" },
		{ 23,	"latexila" },
		{ 34,	"lazarus" },
		{ 26,	"leechcraft" },
		{ 39,	"leksah" },
		{ 39,	"leksah_loadsession" },
		{ 100,	"libfm-pref-apps" },
		{ 100,	"libreoffice-base" },
		{ 100,	"libreoffice-calc" },
		{ 100,	"libreoffice-draw" },
		{ 100,	"libreoffice-impress" },
		{ 100,	"libreoffice-math" },
		{ 100,	"libreoffice-startcenter" },
		{ 100,	"libreoffice-writer" },
		{ 65,	"libsvm-svm-toy-gtk" },
		{ 68,	"libsvm-svm-toy-qt" },
		{ 24,	"linphone" },
		{ 31,	"listen" },
		{ 100,	"liveinst" },
		{ 51,	"lmms" },
		{ 21,	"log4j-chainsaw" },
		{ 21,	"log4j-logfactor5" },
		{ 19,	"logjam" },
		{ 51,	"luckybackup" },
		{ 51,	"luckybackup-gnome-su" },
		{ 51,	"luckybackup-kde-su" },
		{ 31,	"lv2rack" },
		{ 19,	"lxappearance" },
		{ 19,	"lxinput" },
		{ 29,	"lxtask" },
		{ 43,	"lyx" },
		{ 100,	"mail-notification-properties" },
		{ 32,	"maniadrive" },
		{ 32,	"maniadrive-track-editor" },
		{ 100,	"matecc" },
		{ 24,	"mateconf-editor" },
		{ 37,	"mathomatic" },
		{ 19,	"media-explorer" },
		{ 17,	"meiga" },
		{ 28,	"merkaartor" },
		{ 30,	"midori" },
		{ 30,	"midori-private" },
		{ 19,	"minicomputer" },
		{ 22,	"mirage" },
		{ 21,	"mkvinfo" },
		{ 21,	"mkvmergeGUI" },
		{ 19,	"molrender" },
		{ 31,	"monodevelop" },
		{ 22,	"monodoc" },
		{ 25,	"moserial" },
		{ 100,	"mozilla-seamonkey" },
		{ 100,	"mozilla-seamonkey-mail" },
		{ 64,	"mozilla-thunderbird" },
		{ 82,	"mscore" },
		{ 18,	"mtpaint" },
		{ 62,	"muse" },
		{ 20,	"nact" },
		{ 51,	"nautilus" },
		{ 51,	"nautilus-autorun-software" },
		{ 44,	"nautilus-beesu-manager" },
		{ 51,	"nautilus-connect-server" },
		{ 20,	"nemiver" },
		{ 51,	"nemo" },
		{ 51,	"nemo-autorun-software" },
		{ 21,	"netactview" },
		{ 25,	"nfoview" },
		{ 100,	"nm-applet" },
		{ 100,	"nm-connection-editor" },
		{ 100,	"nm-vpnc-auth-dialog" },
		{ 40,	"nmap-zenmap" },
		{ 38,	"nut-monitor" },
		{ 17,	"obconf" },
		{ 30,	"ocp" },
		{ 100,	"octave" },
		{ 24,	"omegat" },
		{ 30,	"openttd" },
		{ 31,	"openvas-client" },
		{ 88,	"orca" },
		{ 19,	"osmo" },
		{ 21,	"pan" },
		{ 100,	"panel-desktop-handler" },
		{ 100,	"panel-preferences" },
		{ 39,	"paraview" },
		{ 30,	"parrot_html" },
		{ 30,	"parrot_pdf" },
		{ 100,	"pcmanfm" },
		{ 100,	"pcmanfm-desktop-pref" },
		{ 27,	"peppy" },
		{ 20,	"perl6_specs_link" },
		{ 22,	"pessulus" },
		{ 33,	"phasex" },
		{ 26,	"phatch" },
		{ 26,	"phatch-inspector" },
		{ 77,	"pidgin" },
		{ 24,	"piklab" },
		{ 24,	"pinta" },
		{ 25,	"piqsl" },
		{ 44,	"pitivi" },
		{ 69,	"planner" },
		{ 18,	"pokerth" },
		{ 26,	"polybori" },
		{ 19,	"portecle" },
		{ 21,	"pragha" },
		{ 45,	"publican" },
		{ 34,	"pulseaudio-equalizer" },
		{ 23,	"pulsecaster" },
		{ 38,	"pybliographic" },
		{ 22,	"pychess" },
		{ 22,	"pymol" },
		{ 26,	"pyvnc2swf" },
		{ 26,	"pyvnc2swf-play" },
		{ 22,	"q4wine" },
		{ 100,	"qBittorrent" },
		{ 30,	"qelectrotech" },
		{ 62,	"qgis" },
		{ 37,	"qjackctl" },
		{ 74,	"qlandkartegt" },
		{ 40,	"qmidiarp" },
		{ 45,	"qmmp" },
		{ 45,	"qmmp_cue" },
		{ 45,	"qmmp_dir" },
		{ 45,	"qmmp_enqueue" },
		{ 38,	"qmpdclient" },
		{ 38,	"qsynth" },
		{ 25,	"quitcount" },
		{ 29,	"quodlibet" },
		{ 41,	"radiotray" },
		{ 45,	"rakarrack" },
		{ 20,	"rakudo_guide_pdf" },
		{ 25,	"rapid-photo-downloader" },
		{ 38,	"rawstudio" },
		{ 48,	"rawtherapee" },
		{ 27,	"rcsslogplayer" },
		{ 18,	"rcssmonitor" },
		{ 25,	"rcssserver" },
		{ 31,	"rcssserver3d" },
		{ 39,	"recoll-searchgui" },
		{ 73,	"redhat-sylpheed" },
		{ 25,	"redhat-userinfo" },
		{ 25,	"redhat-usermount" },
		{ 25,	"redhat-userpasswd" },
		{ 19,	"redmode" },
		{ 27,	"rednotebook" },
		{ 19,	"rekall" },
		{ 24,	"release-notes" },
		{ 24,	"release-notesX" },
		{ 30,	"remmina" },
		{ 49,	"resapplet" },
		{ 29,	"revisor" },
		{ 100,	"rhythmbox" },
		{ 100,	"rhythmbox-device" },
		{ 29,	"root" },
		{ 34,	"rosegarden" },
		{ 23,	"roxterm" },
		{ 18,	"rxvt-unicode" },
		{ 19,	"rxvt-unicode-256color" },
		{ 18,	"rxvt-unicode-256color-ml" },
		{ 19,	"rxvt-unicode-ml" },
		{ 34,	"rygel" },
		{ 34,	"rygel-preferences" },
		{ 18,	"sakura" },
		{ 23,	"sane-pygtk" },
		{ 25,	"scantailor" },
		{ 32,	"scribus" },
		{ 18,	"scummvm" },
		{ 23,	"seahorse" },
		{ 27,	"seahorse-pgp-encrypted" },
		{ 27,	"seahorse-pgp-keys" },
		{ 27,	"seahorse-pgp-preferences" },
		{ 27,	"seahorse-pgp-signature" },
		{ 94,	"selinux-polgengui" },
		{ 25,	"seq24" },
		{ 30,	"setroubleshoot" },
		{ 25,	"shotwell" },
		{ 25,	"shotwell-viewer" },
		{ 21,	"shutter" },
		{ 21,	"sim" },
		{ 36,	"simple-scan" },
		{ 40,	"skillgui" },
		{ 44,	"sol" },
		{ 66,	"solfege" },
		{ 27,	"sooperlooper" },
		{ 95,	"soundconverter" },
		{ 28,	"spe" },
		{ 21,	"spring" },
		{ 26,	"springlobby" },
		{ 21,	"spyder" },
		{ 23,	"starcal2" },
		{ 18,	"streamtuner2" },
		{ 36,	"strigiclient" },
		{ 44,	"subdownloader" },
		{ 22,	"subscription-manager-gui" },
		{ 30,	"subtitleeditor" },
		{ 51,	"sugar-emulator" },
		{ 23,	"swami" },
		{ 22,	"swift" },
		{ 26,	"sync" },
		{ 51,	"synce-trayicon" },
		{ 51,	"synce-trayicon" },
		{ 21,	"synfigstudio" },
		{ 34,	"systemadm" },
		{ 21,	"taskjuggler" },
		{ 23,	"tecnoballz" },
		{ 57,	"terminator" },
		{ 17,	"termit" },
		{ 42,	"texmacs" },
		{ 24,	"texmaker" },
		{ 23,	"texstudio" },
		{ 19,	"texworks" },
		{ 19,	"the-board" },
		{ 23,	"tomboy" },
		{ 22,	"toped" },
		{ 20,	"tortoisehg" },
		{ 40,	"totem" },
		{ 82,	"tracker-needle" },
		{ 82,	"tracker-preferences" },
		{ 49,	"transmission-gtk" },
		{ 50,	"transmission-qt" },
		{ 30,	"tryton" },
		{ 25,	"twinkle" },
		{ 27,	"ufraw" },
		{ 34,	"uget-gtk" },
		{ 27,	"uqm" },
		{ 19,	"uzbl" },
		{ 18,	"vfrairporteditor" },
		{ 18,	"vfrairspaceeditor" },
		{ 18,	"vfrnav" },
		{ 18,	"vfrnavaideditor" },
		{ 18,	"vfrtrackeditor" },
		{ 18,	"vfrwaypointeditor" },
		{ 20,	"vidalia" },
		{ 21,	"vifir" },
		{ 47,	"viking" },
		{ 57,	"vinagre" },
		{ 57,	"vinagre-file" },
		{ 50,	"virtaal" },
		{ 24,	"visualvm" },
		{ 31,	"vncviewer" },
		{ 29,	"vym" },
		{ 21,	"wesnoth" },
		{ 26,	"whaawmp" },
		{ 29,	"whysynth" },
		{ 100,	"window-properties" },
		{ 100,	"wine" },
		{ 100,	"wireshark" },
		{ 18,	"worker" },
		{ 24,	"workrave" },
		{ 20,	"wxmacmolplt" },
		{ 38,	"wxmaxima" },
		{ 51,	"x-tile" },
		{ 26,	"xarchiver" },
		{ 19,	"xchat" },
		{ 30,	"xclips" },
		{ 27,	"xemacs" },
		{ 23,	"xfburn" },
		{ 63,	"xfcalendar" },
		{ 45,	"xgnokii" },
		{ 21,	"xine" },
		{ 21,	"xiphos" },
		{ 19,	"xmaxima" },
		{ 19,	"xmoto" },
		{ 49,	"xnoise" },
		{ 31,	"xournal" },
		{ 23,	"xpaint" },
		{ 68,	"xpbs" },
		{ 68,	"xpbsmon" },
		{ 19,	"xpdf" },
		{ 26,	"xsane" },
		{ 26,	"xsynth-dssi" },
		{ 30,	"xtide" },
		{ 54,	"yumex" },
		{ 54,	"yumex-local" },
		{ 37,	"zeroinstall-add" },
		{ 37,	"zeroinstall-manage" },
		{ 44,	"zim" },
		{ 27,	"zynaddsubfx" },
		{ 31,	"zynjacku" },
		{ 28,	"zyx-liveinstaller" },
		{ -1	,NULL}
	};

	/* add each one to a hash table */
	for (i = 0; ratings[i].id != NULL; i++) {
		g_hash_table_insert (plugin->priv->cache,
				     (gpointer) (ratings[i].id),
				     GINT_TO_POINTER (ratings[i].value));
	}

	return TRUE;
}

/**
 * gs_plugin_refine:
 */
gboolean
gs_plugin_refine (GsPlugin *plugin,
		  GList *list,
		  GCancellable *cancellable,
		  GError **error)
{
	const gchar *id;
	gboolean ret = TRUE;
	GList *l;
	gpointer value;
	GsApp *app;

	/* already loaded */
	if (g_once_init_enter (&plugin->priv->loaded)) {
		ret = gs_plugin_hardcoded_ratings_setup (plugin, error);
		g_once_init_leave (&plugin->priv->loaded, TRUE);

		if (!ret)
			goto out;
	}

	/* add any missing ratings data */
	for (l = list; l != NULL; l = l->next) {
		app = GS_APP (l->data);
		if (gs_app_get_rating (app) != -1)
			continue;
		id = gs_app_get_id (app);
		if (id == NULL)
			continue;
		value = g_hash_table_lookup (plugin->priv->cache, id);
		if (value != NULL) {
			gs_app_set_rating (app, GPOINTER_TO_INT (value));
		} else {
			/* this is the wrong thing to do if we want multiple
			 * plugins to be able to set ratings data */
			gs_app_set_rating (app, 0);
		}
	}
out:
	return ret;
}
