/* $Id: gdict-app.c,v 1.33 2002/02/02 21:32:31 kevinv Exp $ */
/* -*- mode: c; style: k&r; c-basic-offset: 4 -*- */

/*
 *  Mike Hughes <mfh@psilord.com>
 *  Papadimitriou Spiros <spapadim+@cs.cmu.edu>
 *  Bradford Hovinen <hovinen@udel.edu>
 *
 *  This code released under the GNU GPL.
 *  Read the file COPYING for more information.
 *
 *  GDict main window
 *
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <ctype.h>

#define GTK_ENABLE_BROKEN
#include <gnome.h>

#include "dict.h"
#include "gdict-about.h"
#include "gdict-pref.h"

#include "gdict-app.h"
#include "gdict-defbox.h"
#include "gdict-speller.h"

#ifdef HAVE_GNOME_PRINT
#  include <libgnomeprint/gnome-printer-dialog.h>
#  include <libgnomeprint/gnome-printer-profile.h>
#endif /* HAVE_GNOME_PRINT */

#define APPNAME "gdict"

GtkWidget *gdict_app;
GtkWidget *gdict_appbar;
GtkWidget *gnome_word_entry;
GtkWidget *word_entry;
GDictDefbox *defbox;
GDictSpeller *speller;
GtkWidget *pref_dialog;
GtkWidget *socket_error_dialog;

#ifdef HAVE_GNOME_PRINT
GnomePrinter *gdict_printer = NULL;
#endif /* HAVE_GNOME_PRINT */

dict_context_t *context;

static gint
socket_dialog_close_cb (GtkWidget *widget, gpointer data) 
{
    socket_error_dialog = NULL;
    return FALSE;
}

static void
socket_error_cb (GtkWidget *widget, gchar *message, gpointer data) 
{
    socket_error_dialog = gtk_message_dialog_new (NULL, GTK_DIALOG_DESTROY_WITH_PARENT,
                                  		  GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
                                  		  "%s", message, NULL); 
    gtk_dialog_run (GTK_DIALOG (socket_error_dialog));
    gtk_widget_destroy (socket_error_dialog);
    
}

/* gdict_init_context
 *
 * Initialises the context object with information on the current server
 *
 * Retrurns 0 on success, -1 if the server could not be found
 */

gint
gdict_init_context (void) 
{
    if (context) 
      dict_context_destroy (context);
    
    context = dict_context_new (gdict_pref.server, gdict_pref.port);
    context->command = dict_disconnect_command_new ();
    
    defbox->context = context;

    if (context->hostinfo)
	return 0;
    else
	return -1;
}

void
gdict_app_clear (void) 
{
    gchar *word;
    
    gdict_defbox_clear (defbox);
  
    /* Update entry */
    if ((word = gdict_defbox_get_word (defbox))) {
        gtk_entry_set_text (GTK_ENTRY(word_entry), word);
        gtk_editable_select_region (GTK_EDITABLE(word_entry), 0, strlen(word));
    }
}

static void
spell_lookup_start_cb (GtkWidget *widget, gpointer data) 
{
    gnome_appbar_clear_stack(GNOME_APPBAR(gdict_appbar));
    gnome_appbar_push(GNOME_APPBAR(gdict_appbar), _("Spell-checking..."));
    gnome_appbar_refresh(GNOME_APPBAR(gdict_appbar));
}

static void
spell_lookup_done_cb (GtkWidget *widget, gpointer data) 
{
    gnome_appbar_pop (GNOME_APPBAR (gdict_appbar));
    gnome_appbar_push (GNOME_APPBAR (gdict_appbar), _("Spell check done"));
    gnome_appbar_refresh (GNOME_APPBAR (gdict_appbar));
}

static void
spell_not_found_cb (GtkWidget *widget, gpointer data) 
{
    gnome_appbar_pop (GNOME_APPBAR (gdict_appbar));
    gnome_appbar_push (GNOME_APPBAR (gdict_appbar), _("No matches found"));
    gnome_appbar_refresh (GNOME_APPBAR (gdict_appbar));
}

static gint
spell_close_cb (GtkWidget *widget, gpointer data) 
{
    speller = NULL;
    return FALSE;
}

void
gdict_open_speller (void) 
{
    if (!speller) {
        speller = GDICT_SPELLER (gdict_speller_new (context));
        
        if (!speller) return;

        gtk_signal_connect (GTK_OBJECT (speller), "word_lookup_start",
                            GTK_SIGNAL_FUNC (spell_lookup_start_cb), NULL);
        gtk_signal_connect (GTK_OBJECT (speller), "word_lookup_done",
                            GTK_SIGNAL_FUNC (spell_lookup_done_cb), NULL);
        gtk_signal_connect (GTK_OBJECT (speller), "word_not_found",
                            GTK_SIGNAL_FUNC (spell_not_found_cb), NULL);
        gtk_signal_connect (GTK_OBJECT (speller), "socket_error",
                            GTK_SIGNAL_FUNC (socket_error_cb), NULL);
        gtk_signal_connect (GTK_OBJECT (speller), "close",
                            GTK_SIGNAL_FUNC (spell_close_cb), NULL);
        gtk_widget_show_all (GTK_WIDGET (speller));
    }
    else
    	gtk_window_present (GTK_WINDOW (speller));
}


gint
gdict_spell (gchar *text, gboolean pattern) 
{
    g_return_val_if_fail(text != NULL, 0);

    gdict_open_speller ();

    if (!speller) return -1;

    if (pattern) 
	speller->strat = "re";
    else
	speller->strat = gdict_pref.dfl_strat;

    if (gdict_speller_lookup (speller, text) == -1) return -1;

    return 0;
}

static gboolean
is_pattern (gchar *text) 
{
    if (strpbrk (text, "*|{}()[]"))
	return TRUE;
    else
	return FALSE;
}

void
gdict_app_do_lookup (gchar *text) 
{
    gint retval;

    if (gdict_pref.smart && is_pattern (text)) {
	retval = gdict_spell (text, TRUE);
    }
    else {
	retval = gdict_defbox_lookup (defbox, text);
	if (!retval) gtk_widget_show (gdict_app);
    }

    if (retval) {
	gdict_not_online ();
    }
}

static void
lookup_entry (void) 
{
    gchar *text = gtk_editable_get_chars (GTK_EDITABLE (word_entry), 0, -1);
    g_strdown(text);
    gdict_app_do_lookup (text);
    if (text)
        g_free (text);
}

static void
lookup_defbox (void) 
{
    gchar *text = NULL;
    gtk_signal_emit_by_name (GTK_OBJECT (defbox), "copy_clipboard");
    gtk_entry_set_text (GTK_ENTRY (word_entry), "");
    gtk_signal_emit_by_name (GTK_OBJECT (word_entry), "paste_clipboard");
    text = gtk_editable_get_chars (GTK_EDITABLE (word_entry), 0, -1);
    g_strdown (text);
    gnome_entry_append_history (GNOME_ENTRY (gnome_word_entry), 1, text);
    gdict_app_do_lookup (text);
    if (text)
        g_free (text);
}

static void
lookup_any (void) 
{
    /*if (GTK_WIDGET_HAS_FOCUS (word_entry))
	lookup_entry ();*/
    if (GTK_WIDGET_HAS_FOCUS (defbox))
	lookup_defbox ();
    else
        lookup_entry ();
}

static void
def_lookup_start_cb (GtkWidget *widget, gpointer data) 
{
    gnome_appbar_clear_stack(GNOME_APPBAR(gdict_appbar));
    gnome_appbar_push (GNOME_APPBAR (gdict_appbar), _("Looking up word..."));
    gnome_appbar_refresh (GNOME_APPBAR (gdict_appbar));
}

static void
def_lookup_done_cb (GtkWidget *widget, gpointer data) 
{
    gnome_appbar_pop (GNOME_APPBAR (gdict_appbar));
    gnome_appbar_push (GNOME_APPBAR (gdict_appbar), _("Lookup done"));
    gnome_appbar_refresh (GNOME_APPBAR (gdict_appbar));
}

static void
def_not_found_cb (GtkWidget *widget, gpointer data) 
{
    gnome_appbar_pop (GNOME_APPBAR (gdict_appbar));
    gnome_appbar_push (GNOME_APPBAR (gdict_appbar), _("No matches found"));
    gnome_appbar_refresh (GNOME_APPBAR (gdict_appbar));
    
    if (gdict_pref.smart) {
        gdict_spell (gdict_defbox_get_word (defbox), FALSE);
    }
}

static void
def_substr_not_found_cb (GtkWidget *widget, gpointer data) 
{
    gnome_appbar_pop (GNOME_APPBAR (gdict_appbar));
    gnome_appbar_push (GNOME_APPBAR (gdict_appbar), _("String not found"));
    gnome_appbar_refresh (GNOME_APPBAR (gdict_appbar));
}

static void
lookup_cb (GtkWidget *menuitem, gpointer user_data) 
{
    lookup_any();
}

static void
lookup_button_cb (GtkButton *button, gpointer data)
{
    lookup_any();
}

static void
lookup_button_drag_cb (GtkWidget *widget, GdkDragContext *context, gint x, gint y,
		       GtkSelectionData *sd, guint info, guint t, gpointer data)
{
	gchar *text;
	
	text = gtk_selection_data_get_text (sd);
	
	if (text) {
		g_strdown (text);
		gtk_entry_set_text (GTK_ENTRY (word_entry), text);
    		gnome_entry_append_history (GNOME_ENTRY (gnome_word_entry), 1, text);
    		gdict_app_do_lookup (text);
		g_free (text);
	}
}

static void
spell_cb (GtkWidget *menuitem, gpointer user_data) 
{
    gchar *text;

    text = gtk_editable_get_chars (GTK_EDITABLE (word_entry), 0, -1);
    if (gdict_spell (text, FALSE) < 0)
	gdict_not_online ();
    if (text)
        g_free (text);
}

#ifdef HAVE_GNOME_PRINT

static void
print_cb (GtkMenuItem *menuitem, gpointer user_data) 
{
    gdict_defbox_print (defbox);
}

static void
print_setup_cb (GtkMenuItem *menuitem, gpointer user_data) 
{
    gdict_printer = gnome_printer_dialog_new_modal();
}

#endif /* HAVE_GNOME_PRINT */

static void
close_cb (GtkMenuItem *menuitem, gpointer user_data) 
{
    gtk_widget_hide(gdict_app);
}

static void
exit_cb (GtkMenuItem *menuitem, gpointer user_data) 
{
    gtk_main_quit();
}

static void
cut_cb (GtkWidget *button, gpointer user_data) 
{
    gtk_signal_emit_by_name (GTK_OBJECT (word_entry), "cut_clipboard");
}

static void
copy_cb (GtkWidget *menuitem, gpointer user_data) 
{
    if (GTK_WIDGET_HAS_FOCUS (defbox))
        gtk_signal_emit_by_name (GTK_OBJECT (defbox), "copy_clipboard");
    else if (GTK_WIDGET_HAS_FOCUS (word_entry))
        gtk_signal_emit_by_name (GTK_OBJECT (word_entry), "copy_clipboard");
}

static void
paste_cb (GtkWidget *menuitem, gpointer user_data) 
{
    gtk_signal_emit_by_name (GTK_OBJECT (word_entry), "paste_clipboard");
}

static void
clear_cb (GtkWidget *menuitem, gpointer user_data) 
{
    gdict_defbox_clear (defbox);
}

static void
select_all_cb (GtkMenuItem *menuitem, gpointer user_data) 
{
    GtkTextBuffer* buffer = NULL;
    GtkTextIter start_iter, end_iter;
    
    buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (defbox));
    
    gtk_text_buffer_get_bounds (buffer, &start_iter, &end_iter);
    
    gtk_text_buffer_place_cursor (buffer, &end_iter);
    gtk_text_buffer_move_mark (buffer,
                               gtk_text_buffer_get_mark (buffer, "selection_bound"),
                               &start_iter);

}

static gchar *find_text;

static void
prompt_response_cb (GnomeAppBar *ab) 
{
    gchar *tmp;

    tmp = gnome_appbar_get_response(ab);
    /* fprintf(stderr, "Response\n"); */
    if (tmp) {
        g_free(find_text);
        find_text = gnome_appbar_get_response(ab);
        if (find_text)
            gdict_defbox_find (defbox, find_text, TRUE);
        gnome_appbar_clear_prompt(GNOME_APPBAR(gdict_appbar));
	g_free(tmp);
    }
}

static void
find_cb (GtkMenuItem *menuitem, gpointer user_data) 
{
    if (!defbox->def_cmd)
        return;

    gnome_appbar_set_prompt (GNOME_APPBAR (gdict_appbar), _("Find:"), TRUE);
}

static void
find_again_cb (GtkMenuItem *menuitem, gpointer user_data) 
{
    if (!defbox->def_cmd)
        return;
    if (find_text)
        gdict_defbox_find (defbox, find_text, FALSE);
}

static void
pref_dialog_apply_cb (GtkWidget *widget, gpointer user_data) 
{
    gdict_defbox_reset (defbox, context);
    if (speller) gdict_speller_reset (speller, context);
}

void
gdict_not_online () 
{
    GtkWidget *w;
    gchar *s;
    
    s = g_strdup_printf (_("Unable to perform requested operation. \n"
    		           "Either the server you are using is not available \n"
    		           "or you are not connected to the Internet."));
    w = gtk_message_dialog_new (NULL, GTK_DIALOG_DESTROY_WITH_PARENT,
                                     GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
                                     "%s", s, NULL); 
    gtk_dialog_run (GTK_DIALOG (w));
    gtk_widget_destroy (w);
    g_free (s);
}

void
gdict_app_show_preferences (void) 
{
	
    if (!pref_dialog) {
    g_print ("show prefs \n");
        pref_dialog = gdict_pref_dialog_new (context);
        
    }

    gtk_widget_show (pref_dialog);
}

static void
preferences_cb (GtkWidget *menuitem, gpointer user_data) 
{
    gdict_app_show_preferences ();
}

static void
about_cb (GtkMenuItem *menuitem, gpointer user_data) 
{
    gdict_about();
}

static GnomeUIInfo file_menu_uiinfo[] = {
    GNOMEUIINFO_ITEM_STOCK (N_("Lookup"),
			    N_("Lookup word in dictionary"),
			    lookup_cb, GNOME_STOCK_MENU_SEARCH),
    GNOMEUIINFO_ITEM_STOCK (N_("Spell"), 
			    N_("Check word spelling"), 
			    spell_cb, GNOME_STOCK_MENU_SPELLCHECK),
#ifdef HAVE_GNOME_PRINT
    GNOMEUIINFO_SEPARATOR,
    GNOMEUIINFO_MENU_PRINT_ITEM (print_cb, NULL),
    GNOMEUIINFO_MENU_PRINT_SETUP_ITEM (print_setup_cb, NULL),
#endif /* HAVE_GNOME_PRINT */
    GNOMEUIINFO_SEPARATOR,
    GNOMEUIINFO_MENU_EXIT_ITEM (exit_cb, NULL),
    GNOMEUIINFO_END
};

#ifdef HAVE_GNOME_PRINT
#define EXIT_FILE_MENU_ITEM 6
#else
#define EXIT_FILE_MENU_ITEM 3
#endif

static GnomeUIInfo applet_close_item = 
  GNOMEUIINFO_MENU_CLOSE_ITEM (close_cb, NULL);

static GnomeUIInfo edit_menu_uiinfo[] = {
    GNOMEUIINFO_MENU_CUT_ITEM (cut_cb, NULL),
    GNOMEUIINFO_MENU_COPY_ITEM (copy_cb, NULL),
    GNOMEUIINFO_MENU_PASTE_ITEM (paste_cb, NULL),
    GNOMEUIINFO_MENU_CLEAR_ITEM (clear_cb, NULL),
    GNOMEUIINFO_MENU_SELECT_ALL_ITEM (select_all_cb, NULL),
    GNOMEUIINFO_SEPARATOR,
    GNOMEUIINFO_MENU_FIND_ITEM (find_cb, NULL),
    GNOMEUIINFO_MENU_FIND_AGAIN_ITEM (find_again_cb, NULL),
    GNOMEUIINFO_END
};

static GnomeUIInfo settings_menu_uiinfo[] = {
    GNOMEUIINFO_MENU_PREFERENCES_ITEM (preferences_cb, NULL),
    GNOMEUIINFO_END
};

static GnomeUIInfo help_menu_uiinfo[] = {
    GNOMEUIINFO_HELP(APPNAME),
    GNOMEUIINFO_MENU_ABOUT_ITEM (about_cb, NULL),
    GNOMEUIINFO_END
};

static GnomeUIInfo menubar_uiinfo[] = {
    GNOMEUIINFO_MENU_FILE_TREE (file_menu_uiinfo),
    GNOMEUIINFO_MENU_EDIT_TREE (edit_menu_uiinfo),
    GNOMEUIINFO_MENU_SETTINGS_TREE (settings_menu_uiinfo),
    GNOMEUIINFO_MENU_HELP_TREE (help_menu_uiinfo),
    GNOMEUIINFO_END
};

enum
{
  TARGET_STRING,
  TARGET_TEXT,
  TARGET_COMPOUND_TEXT,
  TARGET_UTF8_STRING,
  TARGET_TEXT_BUFFER_CONTENTS
};

GtkWidget *gdict_app_create (gboolean applet) {
    GtkWidget *dock;
    GtkWidget *vbox;
    GtkWidget *hbox;
    GtkWidget *button;
    GtkWidget *label;
    GtkWidget *scrolled;
    GtkWidget *defbox_vscrollbar;
    GtkTooltips *tooltips;
    static GtkTargetEntry drop_targets [] = {
    	{ "UTF8_STRING", 0, 0 },
  	{ "COMPOUND_TEXT", 0, 0 },
  	{ "TEXT", 0, 0 },
  	{ "text/plain", 0, 0 },
  	{ "STRING",     0, 0 }
    };
  
    tooltips = gtk_tooltips_new ();
  
    gdict_app = gnome_app_new ("Dictionary", "Dictionary");
    gtk_window_set_default_size (GTK_WINDOW (gdict_app), 450, 330);
    
    dock = GNOME_APP (gdict_app)->dock;
    gtk_widget_show (dock);
    
    if (applet)  /* quick hack */
        memcpy (&file_menu_uiinfo[EXIT_FILE_MENU_ITEM],
		&applet_close_item, sizeof(GnomeUIInfo));

    gnome_app_create_menus (GNOME_APP (gdict_app), menubar_uiinfo);
    
    vbox = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (vbox);
    gnome_app_set_contents (GNOME_APP (gdict_app), vbox);
    
    hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
    gtk_container_set_border_width (GTK_CONTAINER (hbox), GNOME_PAD_SMALL);
    gtk_widget_show (hbox);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    
    label = gtk_label_new (_("Word : "));
    gtk_widget_show (label);
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    
    label = gtk_label_new_with_mnemonic (_("_Look Up"));
    gtk_widget_show (label);
    gtk_misc_set_padding (GTK_MISC (label), 2, 1);
    
    button = gtk_button_new ();
    gtk_container_add (GTK_CONTAINER (button), label);
    gtk_widget_show (button);
    gtk_box_pack_end (GTK_BOX (hbox), button, FALSE, FALSE, 0);
    gtk_drag_dest_set (button, GTK_DEST_DEFAULT_ALL, drop_targets, 
    		       G_N_ELEMENTS (drop_targets), GDK_ACTION_COPY | GDK_ACTION_MOVE);
    g_signal_connect (G_OBJECT (button), "clicked",
    		      G_CALLBACK (lookup_button_cb), defbox);
    g_signal_connect (G_OBJECT (button), "drag_data_received",
    		      G_CALLBACK (lookup_button_drag_cb), defbox);

    gnome_word_entry = gnome_entry_new ("Wordentry");
    gtk_widget_show (gnome_word_entry);
    gtk_box_pack_start (GTK_BOX (hbox), gnome_word_entry, TRUE, TRUE, 0);
    GTK_WIDGET_SET_FLAGS (gnome_word_entry, GTK_CAN_FOCUS);
    gtk_widget_grab_focus (gnome_word_entry);
    gnome_entry_set_max_saved (GNOME_ENTRY (gnome_word_entry), 50);
    word_entry = gnome_entry_gtk_entry(GNOME_ENTRY (gnome_word_entry));
        
    scrolled = gtk_scrolled_window_new (NULL, NULL);
    gtk_container_set_border_width (GTK_CONTAINER (scrolled), GNOME_PAD_SMALL);    
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
                                        GTK_POLICY_AUTOMATIC,
                                        GTK_POLICY_AUTOMATIC);
    gtk_widget_show (scrolled);
  
    defbox = GDICT_DEFBOX (gdict_defbox_new ());
    defbox_setup_tags (defbox);
    gtk_text_view_set_left_margin (GTK_TEXT_VIEW (defbox), GNOME_PAD_SMALL);
    gtk_signal_connect (GTK_OBJECT (defbox), "word_lookup_start",
                        GTK_SIGNAL_FUNC (def_lookup_start_cb), defbox);
    gtk_signal_connect (GTK_OBJECT (defbox), "word_lookup_done",
                        GTK_SIGNAL_FUNC (def_lookup_done_cb), defbox);
    gtk_signal_connect (GTK_OBJECT (defbox), "word_not_found",
                        GTK_SIGNAL_FUNC (def_not_found_cb), defbox);
    gtk_signal_connect (GTK_OBJECT (defbox), "substr_not_found",
                        GTK_SIGNAL_FUNC (def_substr_not_found_cb), defbox);
    gtk_signal_connect (GTK_OBJECT (defbox), "socket_error",
                        GTK_SIGNAL_FUNC (socket_error_cb), defbox);
  
    gtk_widget_show (GTK_WIDGET (defbox));
    gtk_text_view_set_editable (GTK_TEXT_VIEW (defbox), FALSE);
    gtk_container_add (GTK_CONTAINER (scrolled), GTK_WIDGET (defbox));

    gtk_box_pack_start (GTK_BOX (vbox), scrolled, TRUE, TRUE, 0);

    gdict_appbar = gnome_appbar_new (TRUE, TRUE, GNOME_PREFERENCES_ALWAYS);
    gtk_widget_show (gdict_appbar);
    gnome_app_set_statusbar (GNOME_APP (gdict_app), gdict_appbar);
   
    if (!applet)
      gtk_signal_connect (GTK_OBJECT (gdict_app), "delete_event",
                          GTK_SIGNAL_FUNC (gtk_main_quit), NULL);
    else
      gtk_signal_connect (GTK_OBJECT (gdict_app), "delete_event",
                          GTK_SIGNAL_FUNC (gtk_widget_hide), NULL);
    
    gtk_signal_connect (GTK_OBJECT (word_entry), "activate",
                        GTK_SIGNAL_FUNC(lookup_cb), NULL);
    gtk_signal_connect (GTK_OBJECT (gdict_appbar), "user_response",
                        GTK_SIGNAL_FUNC(prompt_response_cb), NULL);
    
    return gdict_app;
}
