/* vi:set ts=8 sts=0 sw=8:
 * $Id: print.c,v 1.15 1999/01/18 20:04:54 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn <kahn@zk3.dec.com>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include "main.h"
#include "print.h"
#include "file.h"
#include "doc.h"
#include "dialog.h"
#include "msgbar.h"
#include "msgbox.h"
#include "prefs.h"
#include "misc.h"


/*** local function prototypes ***/
static char *get_filename(win_t *w);
static char *generate_temp_file(doc_t *d);
static void print_cancel(GtkWidget *wgt, gpointer cbdata);
static void print_destroy(GtkWidget *widget, gpointer cbdata);
static void print_execute(GtkWidget *w, gpointer cbdata);


/*** global function definitions ***/
/*
 * PUBLIC: print_cb
 *
 * creates print dialog box.  this should be the only routine global to
 * the world.
 */
void
print_cb(GtkWidget *widget, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	doc_t *d;
	char *title;

	g_assert(w != NULL);
	d = doc_current(w);
	g_assert(d != NULL);

	if (d->printwin) {
		gdk_window_raise(d->printwin->window);
		return;
	}

	title = (char *)g_malloc(strlen(doc_basefname(d)) + 11);
	sprintf(title, "Print '%s' ?", doc_basefname(d));

	get_dialog_entry(title,
			 "Enter print command.\nUse '%s' for file name",
			 "Print Command:",
			 128, prefs.printcmd,
			 GTK_SIGNAL_FUNC(print_execute), (gpointer)w,
			 GTK_SIGNAL_FUNC(print_cancel), (gpointer)d,
			 GTK_SIGNAL_FUNC(print_destroy), (gpointer)d,
			 &(d->printwin), &(d->printentry));
	g_free(title);
} /* file_print_cb */


/*** local function definitions ***/
/*
 * PRIVATE: get_filename
 *
 * returns the filename to be printed in a newly allocated buffer.
 * the filename is determined by this algorithm:
 *
 * 1. if the file to be printed has not been updated and exists on disk,
 * then print the file on disk immediately.
 * 2. if the file has been updated, or if it is "Untitled", we should
 * prompt the user whether or not to save the file, and then print it.
 * 2a.  if the user does not want to save it, then this is the only time
 * we actually need to create a temporary file.
 */
#define PRINT_TITLE	"Save before printing?"
#define PRINT_MSG	"has not been saved!"
static char *
get_filename(win_t *w)
{
	doc_t *d;
	char *fname = NULL;
	char *buttons[] =
		{ "Print anyway", " Save, then print ", BUTTON_CANCEL };
	char *title, *msg;

	g_assert(w != NULL);
	d = doc_current(w);
	g_assert(d != NULL);

	if (!d->changed && d->fname) {
		if (d->sb == NULL)
			d->sb = (struct stat *)g_malloc(sizeof(struct stat));
		if (stat(d->fname, d->sb) == -1) {
			/* print warning */
			g_free(d->sb);
			d->sb = NULL;
			fname = generate_temp_file(d);
		} else
			fname = g_strdup(d->fname);
	} else { /* doc changed or no filename */
		int ret;

		title = g_strdup(PRINT_TITLE);
		if (d->fname)
			msg = (char *)g_malloc(strlen(PRINT_MSG) +
						strlen(d->fname) + 6);
		else
			msg = (char *)g_malloc(strlen(PRINT_MSG) +
						strlen(UNTITLED) + 6);
		sprintf(msg, " '%s' %s ", 
			(d->fname) ? d->fname : UNTITLED, PRINT_MSG);

		ret = do_dialog(title, msg, 3, buttons, 3, NULL, NULL, TRUE);

		g_free(msg);
		g_free(title);

		switch (ret) {
		case 1 :
			fname = generate_temp_file(d);
			break;
		case 2 :
			if (d->fname == NULL)
				(void)doc_saveas_common(w);
			else
				(void)doc_save_common(w);

			fname = g_strdup(d->fname);
			break;
		case 3 :
			fname = NULL;
			break;
		default:
			printf("get_filename: do_dialog returned %d\n", ret);
			exit(-1);
		} /* switch */
	} /* doc changed or no filename */

	return fname;
} /* get_filename */


/*
 * PRIVATE: generate_temp_file
 *
 * create and write to temp file.  returns name of temp file in malloc'd
 * buffer (needs to be freed afterwards).
 *
 */
static char *
generate_temp_file(doc_t *d)
{
	FILE *fp;
	int oldmask;
	char *fname, *buf;

	fname = my_tempname();

	oldmask = umask(0600);
	if ((fp = fopen(fname, "w")) == NULL) {
		g_error("generate_temp_file: unable to open '%s'\n", fname);
		g_free(fname);
		umask(oldmask);
		return NULL;
	}

	buf = gtk_editable_get_chars(GTK_EDITABLE(d->data), 0,
				     gtk_text_get_length(GTK_TEXT(d->data)));
	if (fputs(buf, fp) == EOF) {
		perror("generate_temp_file: can't write to tmp file");
		g_free(fname);
		g_free(buf);
		fclose(fp);
		umask(oldmask);
		return NULL;
	}
	g_free(buf);
	fflush(fp);
	fclose(fp);
	umask(oldmask);

	return fname;
} /* generate_temp_file */


/*
 * PRIVATE: print_cancel
 *
 * destroy the print dialog box
 */
static void
print_cancel(GtkWidget *wgt, gpointer cbdata)
{
	doc_t *d = (doc_t *)cbdata;

	g_assert(d != NULL);
	gtk_widget_destroy(d->printwin);
} /* print_cancel */


/*
 * PRIVATE: print_destroy
 *
 * destroy the print dialog box
 */
static void
print_destroy(GtkWidget *wgt, gpointer cbdata)
{
	doc_t *d = (doc_t *)cbdata;

	g_assert(d != NULL);
	d->printentry = NULL;
	d->printwin = NULL;
} /* print_destroy */


/*
 * PRIVATE: print_execute
 *
 * actually execute the print command
 */
static void
print_execute(GtkWidget *wgt, gpointer cbdata)
{
	char *scmd, *pcmd, *tmp, *fname;
	doc_t *d;
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);
	d = doc_current(w);
	g_assert(d != NULL);

	/* print using specified command */
	if ((pcmd = gtk_entry_get_text(GTK_ENTRY(d->printentry))) == NULL)
		return;

	/* look for "file variable" and place marker */
	if ((tmp = strstr(pcmd, "%s")) == NULL)
		return;
	*tmp = '\0';
	tmp += 2;

	if ((fname = get_filename(w)) == NULL) {
		print_destroy(NULL, w->curdoc);
		return;	/* canceled */
	}

	/* build command and execute; g_malloc handles memory alloc errors */
	scmd = (char *)g_malloc(strlen(pcmd) + strlen(fname) + 1);
	sprintf(scmd, "%s%s%s", pcmd, fname, tmp);
	GNPDBG_PREFS(("print_execute: print command = '%s'\n", scmd));
	if (system(scmd) == -1)
		perror("print_execute: system() error");
	g_free(scmd);

	if (strncmp(fname, prefs.tmpdir, strlen(prefs.tmpdir)) == 0)
		if (unlink(fname))
			perror("print_execute: unlink() error");
	g_free(fname);

	msgbar_printf(w, "Printed %s", d->fname);
	msgbox_printf("printed %s", d->fname);

	gtk_widget_destroy(d->printwin);
} /* print_execute */


/* the end */
