#ifndef GNM_CELL_H
#define GNM_CELL_H

#include "gnumeric.h"
#include "dependent.h"

typedef enum {
	/* MUST BE > 0xFFF,FFFF to avoid conflict with GnmDependent */
	/* GnmCell is linked into the sheet */
	CELL_IN_SHEET_LIST  = 0x10000000,
	/* Is the top left corner of a merged region */
	CELL_IS_MERGED	    = 0x20000000,
	/* Cells expression was changed, recalc before rendering */
	CELL_HAS_NEW_EXPR    = 0x40000000
} GnmCellFlags;

/* Definition of a GnmCell */
#define DEP_TO_CELL(dep)	((GnmCell *)(dep))
#define CELL_TO_DEP(cell)	(&(cell)->base)
struct _GnmCell {
	GnmDependent base;

	/* Mandatory state information */
	GnmCellPos     pos;
	ColRowInfo    *row_info;

	GnmValue      *value;	/* computed or entered (Must be non NULL) */
	RenderedValue *rendered_value;
};

/**
 * GnmCell state checking
 */
#define	    cell_needs_recalc(cell)	((cell)->base.flags & DEPENDENT_NEEDS_RECALC)
#define	    cell_expr_is_linked(cell)	((cell)->base.flags & DEPENDENT_IS_LINKED)
#define	    cell_has_expr(cell)		((cell)->base.texpr != NULL)
#define	    cell_is_merged(cell)	((cell)->base.flags & CELL_IS_MERGED)
gboolean    cell_is_empty	  (GnmCell const *cell);
gboolean    cell_is_blank	  (GnmCell const *cell); /* empty, or "" */
GnmValue   *cell_is_error	  (GnmCell const *cell);
gboolean    cell_is_number	  (GnmCell const *cell);
gboolean    cell_is_zero	  (GnmCell const *cell);

gboolean    cell_is_array	  (GnmCell const *cell);
gboolean    cell_is_nonsingleton_array (GnmCell const *cell);
GnmExprArrayCorner const *
	    cell_is_array_corner  (GnmCell const *cell);
gboolean    cell_array_bound	  (GnmCell const *cell, GnmRange *res);

#define cell_eval(cell)											\
	do {												\
		if (cell_needs_recalc (cell)) {								\
			cell_eval_content (cell);							\
			cell->base.flags &= ~(DEPENDENT_NEEDS_RECALC | (int)CELL_HAS_NEW_EXPR );	\
		}											\
	} while (0)

/**
 * Utilities to assign the contents of a cell
 */
void cell_set_text		(GnmCell *c, char const *text);
void cell_assign_value		(GnmCell *c, GnmValue *v);
void cell_set_value		(GnmCell *c, GnmValue *v);
void cell_set_expr_and_value	(GnmCell *c,
				 GnmExprTop const *texpr, GnmValue *v,
				 gboolean link_expr);
void cell_set_expr		(GnmCell *c, GnmExprTop const *texpr);
void cell_set_expr_unsafe 	(GnmCell *cell, GnmExprTop const *texpr);
void cell_set_array_formula	(Sheet *sheet,
				 int cola, int rowa, int colb, int rowb,
				 GnmExprTop const *texpr);
void cell_cleanout		(GnmCell *cell);
void cell_convert_expr_to_value	(GnmCell *cell);

/**
 * Manipulate GnmCell attributes
 */
GnmStyle const *cell_get_style	(GnmCell const *cell);
GOFormat *cell_get_format	(GnmCell const *cell);
void	cell_set_format		(GnmCell *cell, char const *format);

void	cell_render_value	(GnmCell *cell, gboolean allow_variable_width);
int	cell_rendered_height	(GnmCell const * cell);
int	cell_rendered_width	(GnmCell const * cell);	/* excludes offset */
int	cell_rendered_offset	(GnmCell const * cell);
GOColor cell_get_render_color	(GnmCell const * cell);
char *	cell_get_entered_text	(GnmCell const * cell);
char *  cell_get_rendered_text  (GnmCell *cell);

#endif /* GNM_CELL_H */
