/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/**
 * gpdf bonobo persistor
 *
 * Authors:
 *   Michael Meeks (michael@ximian.com)
 *   Martin Kretzschmar <Martin.Kretzschmar@inf.tu-dresden.de>
 *
 * Copyright 1999, 2000 Ximian, Inc.
 * Copyright 2002 Martin Kretzschmar
 */

#include "gpdf-persist-stream.h"
#include "gpdf-util.h"
#include "gpdf-marshal.h"
#include "gpdf-g-switch.h"
#  include <libbonobo.h>
#include "gpdf-g-switch.h"
#include "Object.h"
#include "PDFDoc.h"
#include "BonoboStream.h"

#define noPDF_DEBUG
#ifdef PDF_DEBUG
#  define DBG(x) x
#else
#  define DBG(x)
#endif

BEGIN_EXTERN_C

struct _GPdfPersistStreamPrivate {
	PDFDoc        *pdf_doc;

	Bonobo_Stream  bonobo_stream;
};

#define PARENT_TYPE BONOBO_TYPE_PERSIST
BONOBO_CLASS_BOILERPLATE_FULL (GPdfPersistStream, gpdf_persist_stream,
			       Bonobo_PersistStream,
			       BonoboPersist, PARENT_TYPE);

enum {
	SET_PDF_SIGNAL,
	LAST_SIGNAL
};

static guint gpdf_persist_stream_signals [LAST_SIGNAL];

PDFDoc *
gpdf_persist_stream_get_pdf_doc (GPdfPersistStream *persist_stream)
{
	g_return_val_if_fail (persist_stream != NULL, NULL);
	g_return_val_if_fail (GPDF_IS_PERSIST_STREAM (persist_stream), NULL);

	return persist_stream->priv->pdf_doc;
}

static void
gpdf_persist_stream_delete_doc_and_stream (GPdfPersistStream *persist_stream)
{
	GPdfPersistStreamPrivate *priv;

	g_return_if_fail (persist_stream != NULL);
	g_return_if_fail (GPDF_IS_PERSIST_STREAM (persist_stream));

	priv = persist_stream->priv;

	if (priv->pdf_doc) {
		delete priv->pdf_doc;
		priv->pdf_doc = NULL;
	}

	if (priv->bonobo_stream) {
		bonobo_object_release_unref (priv->bonobo_stream, NULL);
		priv->bonobo_stream = NULL;
	}
}

static void
impl_bonobo_persist_stream_load (PortableServer_Servant servant,
				 Bonobo_Stream stream, const CORBA_char *type,
				 CORBA_Environment *ev)
{
	BonoboObject *object = bonobo_object_from_servant (servant);
	GPdfPersistStream *ps = GPDF_PERSIST_STREAM (object);
	GPdfPersistStreamPrivate *priv;
	Object obj;
	BaseStream *pdf_stream;

	bonobo_return_if_fail (ps != NULL, ev);
	bonobo_return_if_fail (GPDF_IS_PERSIST_STREAM (ps), ev);

	priv = ps->priv;

	if (priv->pdf_doc || priv->bonobo_stream)
		gpdf_persist_stream_delete_doc_and_stream (ps);

	/* FIXME: is this against Bonobo::PersistStream's contract ? */

	/* We need to keep the alive stream for later */

	CORBA_Object_duplicate (stream, ev);
	Bonobo_Unknown_ref (stream, ev);
	priv->bonobo_stream = stream;
	BONOBO_RET_EX (ev);

	DBG (g_message ("Loading PDF from PersistStream"));
	
	obj.initNull ();
	pdf_stream = new bonoboStream (stream, 0, gFalse, 0, &obj);
	priv->pdf_doc = new PDFDoc (pdf_stream);

	DBG (g_message ("Done load"));

	if (!priv->pdf_doc->isOk ()) {
		g_warning ("Duff pdf data");
		goto exit_bad_file;
	}

	goto exit_clean;

exit_bad_file:
	gpdf_persist_stream_delete_doc_and_stream (ps);

exit_clean:
	g_signal_emit (G_OBJECT (ps),
		       gpdf_persist_stream_signals [SET_PDF_SIGNAL],
		       0);	
}

static void
impl_bonobo_persist_stream_save (PortableServer_Servant servant,
				 Bonobo_Stream stream, const CORBA_char *type,
				 CORBA_Environment *ev)
{
	CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
			     ex_Bonobo_NotSupported, NULL);
};

static Bonobo_Persist_ContentTypeList *
impl_get_content_types (BonoboPersist *persist, CORBA_Environment *ev)
{
	return bonobo_persist_generate_content_types (1, "application/pdf");
}


static void
gpdf_persist_stream_destroy (BonoboObject *object)
{
	GPdfPersistStream *persist_stream = GPDF_PERSIST_STREAM (object);

	g_return_if_fail (object != NULL);
	g_return_if_fail (GPDF_IS_PERSIST_STREAM (object));
	
	DBG (g_message ("Destroying GPdfPersistStream"));

	gpdf_persist_stream_delete_doc_and_stream (persist_stream);

	BONOBO_CALL_PARENT (BONOBO_OBJECT_CLASS, destroy, (object));
}

static void
gpdf_persist_stream_finalize (GObject *object)
{
	g_return_if_fail (object != NULL);
	g_return_if_fail (GPDF_IS_PERSIST_STREAM (object));

	g_free ((GPDF_PERSIST_STREAM (object))->priv);

	BONOBO_CALL_PARENT (G_OBJECT_CLASS, finalize, (object));
}

static void
gpdf_persist_stream_class_init (GPdfPersistStreamClass *klass)
{
 	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	BonoboObjectClass *bonobo_object_class = BONOBO_OBJECT_CLASS (klass);
	BonoboPersistClass *persist_class = BONOBO_PERSIST_CLASS (klass);
	POA_Bonobo_PersistStream__epv *epv = &klass->epv;

 	object_class->finalize = gpdf_persist_stream_finalize;
	bonobo_object_class->destroy = gpdf_persist_stream_destroy;
	persist_class->get_content_types = impl_get_content_types;

	epv->load = impl_bonobo_persist_stream_load;
	epv->save = impl_bonobo_persist_stream_save;

	gpdf_persist_stream_signals [SET_PDF_SIGNAL] =
		g_signal_new ("set_pdf",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GPdfPersistStreamClass,
					       set_pdf),
			      NULL, NULL,
			      gpdf_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);
}

static void
gpdf_persist_stream_instance_init (GPdfPersistStream *persist_stream)
{
 	persist_stream->priv = g_new0 (GPdfPersistStreamPrivate, 1);
}

GPdfPersistStream *
gpdf_persist_stream_construct (GPdfPersistStream *persist_stream, 
			       const gchar *iid)
{
	return GPDF_PERSIST_STREAM (
	     bonobo_persist_construct (BONOBO_PERSIST (persist_stream), iid));
}

GPdfPersistStream *
gpdf_persist_stream_new (const gchar *iid)
{
	GPdfPersistStream *persist_stream;

	DBG (g_message ("Creating GPdfPersistStream"));

	persist_stream = 
	     GPDF_PERSIST_STREAM (g_object_new (GPDF_TYPE_PERSIST_STREAM, NULL));

	return gpdf_persist_stream_construct (persist_stream, iid);
}

END_EXTERN_C
