/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* GPdf Bonobo PersistFile implementation
 *
 * Copyright (C) 2003 Martin Kretzschmar
 *
 * Author:
 *   Martin Kretzschmar <Martin.Kretzschmar@inf.tu-dresden.de>
 *
 * GPdf is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * GPdf is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */

#include "gpdf-persist-file.h"
#include "gpdf-util.h"
#include "gpdf-marshal.h"
#include "gpdf-g-switch.h"
#  include <libbonobo.h>
#  include <libgnomevfs/gnome-vfs.h>
#include "gpdf-g-switch.h"
#include "Object.h"
#include "ErrorCodes.h"
#include "PDFDoc.h"
#include "GnomeVFSStream.h"

BEGIN_EXTERN_C

#define GPDF_IS_NON_NULL_PERSIST_FILE(obj) \
	((obj) && GPDF_IS_PERSIST_FILE ((obj)))

struct _GPdfPersistFilePrivate {
	PDFDoc *pdf_doc;

	GnomeVFSHandle *vfs_handle;
	char *uri;
};

#define PARENT_TYPE BONOBO_TYPE_PERSIST
BONOBO_CLASS_BOILERPLATE_FULL (GPdfPersistFile, gpdf_persist_file,
			       Bonobo_PersistFile,
			       BonoboPersist, PARENT_TYPE);

enum {
	LOADING_FINISHED_SIGNAL,
	LOADING_FAILED_SIGNAL,
	LAST_SIGNAL
};

static guint gpdf_persist_file_signals [LAST_SIGNAL];

PDFDoc *
gpdf_persist_file_get_pdf_doc (GPdfPersistFile *persist_file)
{
	g_return_val_if_fail (GPDF_IS_NON_NULL_PERSIST_FILE (persist_file),
			      NULL);
	return persist_file->priv->pdf_doc;
}

static void
gpdf_persist_file_unload (GPdfPersistFile *persist_file)
{
	GPdfPersistFilePrivate *priv;

	g_return_if_fail (GPDF_IS_NON_NULL_PERSIST_FILE (persist_file));

	priv = persist_file->priv;

	if (priv->pdf_doc) {
		delete priv->pdf_doc;
		priv->pdf_doc = NULL;
	}

	if (priv->vfs_handle) {
		gnome_vfs_close (priv->vfs_handle);
		priv->vfs_handle = NULL;
	}

	if (priv->uri) {
		g_free (priv->uri);
		priv->uri;
	}
}

static void
impl_bonobo_persist_file_load (PortableServer_Servant servant,
			       const CORBA_char *uri,
			       CORBA_Environment *ev)
{
	BonoboObject *object = bonobo_object_from_servant (servant);
	GPdfPersistFile *pf = GPDF_PERSIST_FILE (object);
	GPdfPersistFilePrivate *priv;
	GnomeVFSResult vfs_result;
	Object obj;
	BaseStream *pdf_stream;
	const gchar *error_msg;

	bonobo_return_if_fail (GPDF_IS_NON_NULL_PERSIST_FILE (pf), ev);

	priv = pf->priv;

	if (priv->pdf_doc || priv->vfs_handle || priv->uri)
		gpdf_persist_file_unload (pf);

	priv->uri = g_strdup (uri);

	vfs_result = gnome_vfs_open (
		&priv->vfs_handle, uri,
		(GnomeVFSOpenMode) (GNOME_VFS_OPEN_READ |
				    GNOME_VFS_OPEN_RANDOM));
	if (vfs_result != GNOME_VFS_OK) {
		g_signal_emit (G_OBJECT (pf),
			       gpdf_persist_file_signals [LOADING_FAILED_SIGNAL],
			       0, gnome_vfs_result_to_string (vfs_result));
		return;
	}

	obj.initNull ();
	pdf_stream = new GnomeVFSStream (priv->vfs_handle, 0, gFalse, 0, &obj);
	/* FIXME if (pdf_stream == NULL) ? */
	priv->pdf_doc = new PDFDoc (pdf_stream);

	/* FIXME if (pdf_doc == NULL) ? */
	if (priv->pdf_doc->isOk ()) {
		g_signal_emit (G_OBJECT (pf),
			       gpdf_persist_file_signals [LOADING_FINISHED_SIGNAL],
			       0);
		return;
	}

	switch (priv->pdf_doc->getErrorCode ()) {
	case errBadCatalog:
		/* translators: page catalog is a part of the PDF file */
		/* The last period (.) is missing on purpose */
		error_msg = _("The PDF file is damaged. Its page catalog could not be read");
		break;
	case errDamaged:
		error_msg = _("The PDF file is damaged or it is not a PDF file. It could not be repaired");
		break;
	case errEncrypted:
		error_msg = _("The PDF file is encrypted and requires a password. This program does not support password-protected files");
		break;
	case errNone:
	case errOpenFile:
	case errHighlightFile:
	default:
		error_msg = NULL;
		g_assert_not_reached ();
		break;
	}			

	g_signal_emit (G_OBJECT (pf),
		       gpdf_persist_file_signals [LOADING_FAILED_SIGNAL],
		       0, error_msg);
	
	gpdf_persist_file_unload (pf);
}

static void
impl_bonobo_persist_file_save (PortableServer_Servant servant,
			       const CORBA_char *uri,
			       CORBA_Environment *ev)
{
	CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
			     ex_Bonobo_NotSupported, NULL);
};

static Bonobo_Persist_ContentTypeList *
impl_get_content_types (BonoboPersist *persist, CORBA_Environment *ev)
{
	return bonobo_persist_generate_content_types (1, "application/pdf");
}

static CORBA_char *
impl_bonobo_persist_file_getCurrentFile (PortableServer_Servant servant,
					 CORBA_Environment *ev)
{
	BonoboObject *object = bonobo_object_from_servant (servant);
	GPdfPersistFile *pfile = GPDF_PERSIST_FILE (object);

	/* if our persist_file has a filename with any length, return it */
	if (pfile->priv->uri && strlen (pfile->priv->uri))
		return CORBA_string_dup ((CORBA_char*)pfile->priv->uri);
	else {
		/* otherwise, raise a `NoCurrentName' exception */
		Bonobo_PersistFile_NoCurrentName *exception;
		exception = Bonobo_PersistFile_NoCurrentName__alloc ();

		exception->extension = CORBA_string_dup ("");
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_Bonobo_PersistFile_NoCurrentName,
				     exception);
		return NULL;
	}
}

const char *
gpdf_persist_file_get_current_uri (GPdfPersistFile *pfile)
{
	g_return_val_if_fail (GPDF_IS_NON_NULL_PERSIST_FILE (pfile), NULL);

	return pfile->priv->uri;
}

static void
gpdf_persist_file_destroy (BonoboObject *object)
{
	GPdfPersistFile *persist_file = GPDF_PERSIST_FILE (object);

	g_return_if_fail (object != NULL);
	g_return_if_fail (GPDF_IS_PERSIST_FILE (object));

	BONOBO_CALL_PARENT (BONOBO_OBJECT_CLASS, destroy, (object));
}

static void
gpdf_persist_file_finalize (GObject *object)
{
	g_return_if_fail (object != NULL);
	g_return_if_fail (GPDF_IS_PERSIST_FILE (object));

	g_free ((GPDF_PERSIST_FILE (object))->priv);

	BONOBO_CALL_PARENT (G_OBJECT_CLASS, finalize, (object));
}

static void
gpdf_persist_file_class_init (GPdfPersistFileClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	BonoboObjectClass *bonobo_object_class = BONOBO_OBJECT_CLASS (klass);
	BonoboPersistClass *persist_class = BONOBO_PERSIST_CLASS (klass);
	POA_Bonobo_PersistFile__epv *epv = &klass->epv;

	object_class->finalize = gpdf_persist_file_finalize;
	bonobo_object_class->destroy = gpdf_persist_file_destroy;
	persist_class->get_content_types = impl_get_content_types;

	epv->load = impl_bonobo_persist_file_load;
	epv->save = impl_bonobo_persist_file_save;
	epv->getCurrentFile = impl_bonobo_persist_file_getCurrentFile;

	gpdf_persist_file_signals [LOADING_FINISHED_SIGNAL] =
		g_signal_new ("loading_finished",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GPdfPersistFileClass,
					       loading_finished),
			      NULL, NULL,
			      gpdf_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);

	gpdf_persist_file_signals [LOADING_FAILED_SIGNAL] =
		g_signal_new ("loading_failed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GPdfPersistFileClass,
					       loading_failed),
			      NULL, NULL,
			      gpdf_marshal_VOID__STRING,
			      G_TYPE_NONE, 1,
			      G_TYPE_STRING);
}

static void
gpdf_persist_file_instance_init (GPdfPersistFile *persist_file)
{
	persist_file->priv = g_new0 (GPdfPersistFilePrivate, 1);
}

GPdfPersistFile *
gpdf_persist_file_construct (GPdfPersistFile *persist_file,
			     const gchar *iid)
{
	return GPDF_PERSIST_FILE (
	     bonobo_persist_construct (BONOBO_PERSIST (persist_file), iid));
}

GPdfPersistFile *
gpdf_persist_file_new (const gchar *iid)
{
	GPdfPersistFile *persist_file;

	persist_file =
	     GPDF_PERSIST_FILE (g_object_new (GPDF_TYPE_PERSIST_FILE, NULL));

	return gpdf_persist_file_construct (persist_file, iid);
}

END_EXTERN_C
