/* GStreamer - Matroska Demuxer
 * Copyright (C) <2003> Ronald Bultje <rbultje@ronald.bitfreak.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <math.h>
#include <string.h>
#include <gst/gst.h>
#include <gst/riff/riff.h>

#include <typeinfo>

#include "EbmlHead.h"
#include "EbmlSubHead.h"
#include "EbmlStream.h"
#include "EbmlString.h"
#include "EbmlVersion.h"
#include "EbmlVoid.h"

#include "FileKax.h"
#include "KaxAttachments.h"
#include "KaxBlock.h"
#include "KaxBlockData.h"
#include "KaxChapters.h"
#include "KaxCluster.h"
#include "KaxClusterData.h"
#include "KaxCues.h"
#include "KaxCuesData.h"
#include "KaxInfo.h"
#include "KaxInfoData.h"
#include "KaxSeekHead.h"
#include "KaxSegment.h"
#include "KaxTags.h"
#include "KaxTracks.h"
#include "KaxTrackEntryData.h"
#include "KaxTrackAudio.h"
#include "KaxTrackVideo.h"
#include "KaxVersion.h"

#include "gstmatroskademux.h"

#define TIME_SCALE 1000000LLU /* this is the default scale if the file
			       * doesn't contain a scale itself */

using namespace std;
using namespace LIBMATROSKA_NAMESPACE;

/* elementfactory information */
static GstElementDetails gst_matroska_demux_details = {
  "Matroska demuxer",
  "Codec/Demuxer",
  "LGPL",
  "Demuxes a Matroska Stream into video/audio",
  VERSION,
  "Ronald Bultje <rbultje@ronald.bitfreak.net>",
  "(C) 2003",
};

static GstCaps * gst_matroska_type_find (GstByteStream *bs,
					 gpointer      priv);

/* typefactory for 'matroska' */
static GstTypeDefinition matroskadefinition = {
  "matroskademux_type",
  "video/x-matroska",
  ".mkv .mka",
  gst_matroska_type_find,
};

/* MatroskaDemux signals and args */
enum {
  /* FILL ME */
  LAST_SIGNAL
};

enum {
  ARG_0,
  ARG_METADATA,
  ARG_STREAMINFO,
  /* FILL ME */
};

/* input (bytestreami) template */
GST_PAD_TEMPLATE_FACTORY (sink_templ,
  "sink",
  GST_PAD_SINK,
  GST_PAD_ALWAYS,
  GST_CAPS_NEW (
    "matroskademux_sink",
     "video/x-matroska",
       NULL
  )
)

/* gobject magic foo */
static void 	gst_matroska_demux_class_init (GstMatroskaDemuxClass *klass);
static void 	gst_matroska_demux_init       (GstMatroskaDemux *avi_demux);

/* element functions */
static void 	gst_matroska_demux_loop 	      (GstElement  *element);
static gboolean	gst_matroska_demux_send_event         (GstElement  *element,
						       GstEvent    *event);

/* pad functions */
static const GstEventMask *
		gst_matroska_demux_get_event_mask     (GstPad      *pad);
static gboolean gst_matroska_demux_handle_src_event   (GstPad      *pad,
						       GstEvent    *event);
static const GstFormat *
		gst_matroska_demux_get_src_formats    (GstPad      *pad); 
static const GstQueryType*
		gst_matroska_demux_get_src_query_types(GstPad      *pad);
static gboolean gst_matroska_demux_handle_src_query   (GstPad      *pad,
						       GstQueryType type, 
						       GstFormat   *format,
						       gint64      *value);

/* gst internal change state handler */
static GstElementStateReturn
		gst_matroska_demux_change_state       (GstElement  *element);

/* gobject bla bla */
static void     gst_matroska_demux_get_property       (GObject     *object,
						       guint        prop_id, 	
						       GValue      *value,
						       GParamSpec  *pspec);

/* caps functions */
static GstCaps *gst_matroska_demux_video_caps         (GstMatroskaVideoSrcContext
								   *context,
						       const gchar *codec_id,
						       gpointer     data,
						       guint        size);
static GstCaps *gst_matroska_demux_audio_caps         (GstMatroskaAudioSrcContext
								   *context,
						       const gchar *codec_id,
						       gpointer     data,
						       guint        size);
static GstCaps *gst_matroska_demux_iavs_caps          (GstMatroskaComplexSrcContext
								   *context,
						       const gchar *codec_id,
						       gpointer     data,
						       guint        size);
static GstCaps *gst_matroska_demux_subtitle_caps      (GstMatroskaSubtitleSrcContext
								   *context,
						       const gchar *codec_id,
						       gpointer     data,
						       guint        size);

/* stream methods */
static void     gst_matroska_demux_reset	      (GstElement  *element);

static GstElementClass *parent_class = NULL;
static GstPadTemplate *videosrctempl, *audiosrctempl, *subtitlesrctempl;
/*static guint gst_matroska_demux_signals[LAST_SIGNAL] = { 0 };*/

GType
gst_matroska_demux_get_type (void) 
{
  static GType matroska_demux_type = 0;

  if (!matroska_demux_type) {
    static const GTypeInfo matroska_demux_info = {
      sizeof (GstMatroskaDemuxClass),      
      NULL,
      NULL,
      (GClassInitFunc) gst_matroska_demux_class_init,
      NULL,
      NULL,
      sizeof (GstMatroskaDemux),
      0,
      (GInstanceInitFunc) gst_matroska_demux_init,
    };
    matroska_demux_type = g_type_register_static (GST_TYPE_ELEMENT,
						  "GstMatroskaDemux",
						  &matroska_demux_info,
						  (GTypeFlags) 0);
  }
  return matroska_demux_type;
}

static void
gst_matroska_demux_class_init (GstMatroskaDemuxClass *klass) 
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;

  g_object_class_install_property (gobject_class, ARG_METADATA,
    g_param_spec_boxed ("metadata", "Metadata", "Metadata",
                        GST_TYPE_CAPS, G_PARAM_READABLE));
  g_object_class_install_property (gobject_class, ARG_STREAMINFO,
    g_param_spec_boxed ("streaminfo", "Streaminfo", "Streaminfo",
                        GST_TYPE_CAPS, G_PARAM_READABLE));

  parent_class = (GstElementClass *) g_type_class_ref (GST_TYPE_ELEMENT);

  gobject_class->get_property = gst_matroska_demux_get_property;

  gstelement_class->change_state = gst_matroska_demux_change_state;
  gstelement_class->send_event = gst_matroska_demux_send_event;
}

static void 
gst_matroska_demux_init (GstMatroskaDemux *demux) 
{
  gint i;

  GST_FLAG_SET (GST_OBJECT (demux), GST_ELEMENT_EVENT_AWARE);
				
  demux->sinkpad = gst_pad_new_from_template (
		  GST_PAD_TEMPLATE_GET (sink_templ), "sink");
  gst_element_add_pad (GST_ELEMENT (demux), demux->sinkpad);

  gst_element_set_loop_function (GST_ELEMENT (demux),
				 gst_matroska_demux_loop);

  /* initial stream no. */
  for (i = 0; i < GST_MATROSKA_DEMUX_MAX_STREAMS; i++) {
    demux->src[i] = NULL;
  }
  demux->io = NULL;
  demux->str = NULL;
  demux->last_el = NULL;
  demux->streaminfo = demux->metadata = NULL;
  demux->writing_app = demux->muxing_app = NULL;
  demux->index = NULL;

  /* finish off */
  gst_matroska_demux_reset (GST_ELEMENT (demux));
}

static GstCaps *
gst_matroska_type_find (GstByteStream *bs,
			gpointer       priv)
{
  GstBuffer *buf = NULL;
  GstCaps *newc = NULL;

  if (gst_bytestream_peek (bs, &buf, 16) == 16) {
    guint8 header[16] = { 0x1A, 0x45, 0xDF, 0xA3, 0x93, 0x42, 0x82, 0x88,
			  'm', 'a', 't', 'r', 'o', 's', 'k', 'a' };

    if (!memcmp(GST_BUFFER_DATA (buf), header, 16)) {
      newc = GST_CAPS_NEW ("matroska_type",
			   "video/x-matroska",
			     NULL);
    }
  }

  if (buf != NULL) {
    gst_buffer_unref (buf);
  }

  return newc;
}

static void
gst_matroska_demux_reset (GstElement *element)
{
  GstMatroskaDemux *demux = GST_MATROSKA_DEMUX (element);
  guint i;

  /* reset input */
  demux->state = GST_MATROSKA_DEMUX_START;
  if (demux->str != NULL) {
    delete demux->str;
    demux->str = NULL;
  }
  if (demux->io != NULL) {
    delete demux->io;
    demux->io = NULL;
  }

  /* clean up existing streams */
  for (i = 0; i < GST_MATROSKA_DEMUX_MAX_STREAMS; i++) {
    if (demux->src[i] != NULL) {
      if (demux->src[i]->pad != NULL) {
        /* these currently trigger assertions.. hmm... */
        /*gst_element_remove_pad (GST_ELEMENT (demux), demux->src[i]->pad);*/
        /*g_object_unref (G_OBJECT (demux->src[i]->pad));*/
      }
      g_free(demux->src[i]);
      demux->src[i] = NULL;
    }
  }
  demux->num_streams = 0;
  demux->num_a_streams = 0;
  demux->num_t_streams = 0;
  demux->num_v_streams = 0;

  /* reset media info */
  gst_caps_replace (&demux->metadata, NULL);
  gst_caps_replace (&demux->streaminfo, NULL);

  if (demux->writing_app != NULL) {
    g_free (demux->writing_app);
    demux->writing_app = NULL;
  }
  if (demux->muxing_app != NULL) {
    g_free (demux->muxing_app);
    demux->muxing_app = NULL;
  }

  /* reset indexes */
  demux->num_indexes = 0;
  if (demux->index != NULL) {
    g_free (demux->index);
  }

  /* reset timers */
  demux->time_scale = TIME_SCALE;
  demux->duration = 0;
  demux->seek_pending = GST_CLOCK_TIME_NONE;
}

static EbmlElement *
gst_matroska_demux_next_element (GstMatroskaDemux *demux,
				 EbmlElement      *parent)
{
  EbmlElement *child = NULL;

  if (demux->last_level > 0) {
    //g_assert (demux->last_level > 0);
    child = demux->last_el;
    demux->last_el = NULL;
    demux->last_level--;
  } else {
    demux->last_level = 0;
    if (demux->io->eos != TRUE) {
      //g_assert (demux->last_el == NULL);
      child = demux->str->FindNextElement (parent->Generic ().Context,
					   demux->last_level, ~0, true, 5);
    }
  }

  return child;
}

static gint
gst_matroska_demux_stream_from_num (GstMatroskaDemux *demux,
				    guint             track_num)
{
  guint n;

  for (n = 0; n < demux->num_streams; n++) {
    if (demux->src[n] != NULL &&
        demux->src[n]->num == track_num) {
      return n;
    }
  }

  if (n == demux->num_streams) {
    GST_DEBUG ("Failed to find corresponding pad for tracknum %d",
	       track_num); 
  }

  return -1;
}

static gboolean
gst_matroska_demux_add_stream (GstElement    *element,
                               KaxTrackEntry *entry)
{
  GstMatroskaDemux *demux = GST_MATROSKA_DEMUX (element);
  GstMatroskaSrcContext *context;
  GstPadTemplate *templ = NULL;
  GstCaps *caps = NULL;
  gchar *codecname = NULL, *codecid = NULL, *padname = NULL;
  gpointer codecpriv = NULL;
  gboolean res = TRUE;
  guint codecprivsize = 0;

  if (demux->num_streams >= GST_MATROSKA_DEMUX_MAX_STREAMS) {
    g_warning ("Maximum number of streams (%d) exceeded, skipping",
	       GST_MATROSKA_DEMUX_MAX_STREAMS);
    return FALSE;
  }

  /* allocate generic... if we know the type, we'll realloc()
   * with the precise size */
  context = (GstMatroskaSrcContext *) g_malloc (sizeof (GstMatroskaSrcContext));
  memset (context, 0, sizeof (GstMatroskaSrcContext));
  demux->src[demux->num_streams] = context;
  context->index = demux->num_streams;
  context->type = (track_type) 0; /* no type yet */
  demux->num_streams++;

  /* try reading the trackentry headers */
  do {
    EbmlElement *entryval = gst_matroska_demux_next_element (demux, entry);
    if (entryval == NULL) {
      g_warning ("Failed to read next element in Matroska track header");
      res = FALSE;
      break;
    } else if (demux->last_level != 0) {
      demux->last_el = entryval;
      break;
    }

    /* well, cases don't work here since these aren't constants...
     * argh, the horrors of C. Ohwell, fortunately, these IDs are
     * actually fixed... */
    switch (EbmlId (*entryval).Value) {
      /* track number (unique identifier) */
      case 0xD7 /*KaxTrackNumber::ClassInfos.GlobalId*/: {
        KaxTrackNumber *num = (KaxTrackNumber *) entryval;
        num->ReadData (demux->str->I_O ());
        context->num = uint32 (*num);
        GST_DEBUG ("header: track number = %d", context->num);
        break;
      }

      /* track UID (unique identifier) */
      case 0x73C5 /*KaxTrackUID::ClassInfos.GlobalId*/: {
        KaxTrackUID *uid = (KaxTrackUID *) entryval;
        uid->ReadData (demux->str->I_O ());
        context->uid = uint32 (*uid);
        GST_DEBUG ("header: track UID = %d", context->uid);
        break;
      }

      /* track type (video, audio, combined, subtitle, etc.) */
      case 0x83 /*KaxTrackType::ClassInfos.GlobalId*/: {
        guint size = 0;
        gchar *tracktypename = NULL;
        KaxTrackType *type = (KaxTrackType *) entryval;
        type->ReadData (demux->str->I_O ());
        if (context->type != 0) {
          g_warning ("More than one KaxTrackType in one entry, skipping");
          break;
        }
        context->type = (track_type) uint8 (*type);

        /* ok, so we're actually going to reallocate this thing */
        switch (context->type) {
          case track_video: {
            /* video */
            size = sizeof (GstMatroskaVideoSrcContext);
            tracktypename = "track_video";
            break;
          }

          case track_audio: {
            /* audio */
            size = sizeof (GstMatroskaAudioSrcContext);
            tracktypename = "track_audio";
            break;
          }

          case track_complex: {
            /* video/audio interleaved, such as DV */
            size = sizeof (GstMatroskaComplexSrcContext);
            tracktypename = "track_complex";
            break;
          }

          case track_logo:
            /* overlay logo */
            g_warning ("We don't support overlay logo's yet");
            res = FALSE;
            break;

          case track_subtitle: {
            /* subtitle */
            size = sizeof (GstMatroskaSubtitleSrcContext);
            tracktypename = "track_subtitle";
            break;
          }

          case track_control:
            /* DVD like menus and such */
            g_warning ("We don't support menus and controls yet");
            res = FALSE;
            break;

          default:
            g_warning ("Unknown track type %d", context->type);
            res = FALSE;
            break;
        }
        demux->src[demux->num_streams-1] = context =
		(GstMatroskaSrcContext *) g_realloc (context, size);
        GST_DEBUG ("header: track type = %d (%s)", context->type, tracktypename);
        break;
      }

      /* tracktype specific stuff for video and audio */
      case 0xE1 /*KaxTrackAudio::ClassInfos.GlobalId*/:
      case 0xE0 /*KaxTrackVideo::ClassInfos.GlobalId*/: {
        if ((EbmlId (*entryval) == KaxTrackAudio::ClassInfos.GlobalId &&
             context->type != track_audio) ||
            (EbmlId (*entryval) == KaxTrackVideo::ClassInfos.GlobalId &&
             context->type != track_video)) {
          g_warning ("Subtrackentry information for different track type than current track type in track header: %u/%u",
		     EbmlId (*entryval).Value, context->type);
          break;
        }

        while (1) {
          EbmlElement *subentryval = gst_matroska_demux_next_element (demux, entryval);
          if (subentryval == NULL) {
            g_warning ("Failed to read next element in Matroska header");
            res = FALSE;
            break;
          } else if (demux->last_level != 0) {
            demux->last_el = subentryval;
            break;
          }

          switch (EbmlId (*entryval).Value) {
            case 0xE1 /*KaxTrackAudio::ClassInfos.GlobalId*/: {
              GstMatroskaAudioSrcContext *audiocontext =
			(GstMatroskaAudioSrcContext *) context;

              /* defaults */
              audiocontext->channels = 1;

              switch (EbmlId (*subentryval).Value) {
                /* samplerate */
                case 0xB5 /*KaxTrackAudioSamplingFreq::ClassInfos.GlobalId*/: {
                  KaxAudioSamplingFreq *freq = (KaxAudioSamplingFreq *) subentryval;
                  freq->ReadData (demux->str->I_O ());
                  audiocontext->samplerate = (gint) float (*freq);
                  GST_DEBUG ("header: audio sampling frequency = %d",
			     audiocontext->samplerate);
                  break;
                }

                /* bitdepth */
                case 0x6264 /*KaxTrackAudioBitDepth::ClassInfos.GlobalId*/: {
                  KaxAudioBitDepth *bps = (KaxAudioBitDepth *) subentryval;
                  bps->ReadData (demux->str->I_O ());
                  audiocontext->bitdepth = uint8 (*bps);
                  GST_DEBUG ("header: audio bitdepth = %d",
			     audiocontext->bitdepth);
                  break;
                }

                /* channels */
                case 0x9F /*KaxTrackAudioChannels::ClassInfos.GlobalId*/: {
                  KaxAudioChannels *chans = (KaxAudioChannels *) subentryval;
                  chans->ReadData (demux->str->I_O ());
                  audiocontext->channels = uint8 (*chans);
                  GST_DEBUG ("header: audio channels = %d",
			     audiocontext->channels);
                  break;
                }

                case 0xEC /*EbmlVoid::ClassInfos.GlobalID*/:
                  break;

                default:
                  GST_DEBUG ("Unknown matroska audio-subtrack header ID 0x%x (%s)",
			     EbmlId (*entryval).Value, typeid (*entryval).name ());
              }
              break;
            }

            case 0xE0 /*KaxTrackVideo::ClassInfos.GlobalId*/: {
              GstMatroskaVideoSrcContext *videocontext =
			(GstMatroskaVideoSrcContext *) context;

              /* framerate */
              switch (EbmlId (*subentryval).Value) {
                case 0x2383E3 /*KaxVideoFrameRate::ClassInfos.GlobalId*/: {
                  KaxVideoFrameRate *fps = (KaxVideoFrameRate *) subentryval;
                  fps->ReadData (demux->str->I_O ());
                  videocontext->framerate = float (*fps);
                  GST_DEBUG ("header: video framerate = %f",
			     videocontext->framerate);
                  break;
                }

                /* width of the size to display the video at */
                case 0x54B0 /*KaxVideoDisplayWidth::ClassInfos.GlobalId*/: {
                  KaxVideoDisplayWidth *width = (KaxVideoDisplayWidth *) subentryval;
                  width->ReadData (demux->str->I_O ());
                  videocontext->display_width = uint16 (*width);
                  GST_DEBUG ("header: video displaywidth = %d",
			     videocontext->display_width);
                  break;
                }

                /* height of the size to display the video at */
                case 0x54BA /*KaxVideoDisplayHeight::ClassInfos.GlobalId*/: {
                  KaxVideoDisplayHeight *height = (KaxVideoDisplayHeight *) subentryval;
                  height->ReadData (demux->str->I_O ());
                  videocontext->display_height = uint16 (*height);
                  GST_DEBUG ("header: video displayheight = %d",
			     videocontext->display_height);
                  break;
                }

                /* width of the video as stored in the file */
                case 0xB0 /*KaxVideoPixelWidth::ClassInfos.GlobalId*/: {
                  KaxVideoPixelWidth *width = (KaxVideoPixelWidth *) subentryval;
                  width->ReadData (demux->str->I_O ());
                  videocontext->pixel_width = uint16 (*width);
                  GST_DEBUG ("header: video pixelwidth = %d",
			     videocontext->pixel_width);
                  break;
                }

                /* height of the video as stored in the file */
                case 0xBA /*KaxVideoPixelHeight::ClassInfos.GlobalId*/: {
                  KaxVideoPixelHeight *height = (KaxVideoPixelHeight *) subentryval;
                  height->ReadData (demux->str->I_O ());
                  videocontext->pixel_height = uint16 (*height);
                  GST_DEBUG ("header: video pixelheight = %d",
			     videocontext->pixel_height);
                  break;
                }

                /* whether the video is interlaced */
                case 0x9A /*KaxVideoFlagInterlaced::ClassInfos.GlobalId*/: {
                  KaxVideoFlagInterlaced *interlaced =
				(KaxVideoFlagInterlaced *) subentryval;
                  interlaced->ReadData (demux->str->I_O ());
                  if (uint32 (*interlaced)) {
                    context->flags |= GST_MATROSKA_SRC_VIDEO_INTERLACED;
                  } else {
                    context->flags &= ~GST_MATROSKA_SRC_VIDEO_INTERLACED;
                  }
                  GST_DEBUG ("header: video interlaced = %s",
			     context->flags & GST_MATROSKA_SRC_VIDEO_INTERLACED ?
			     "true" : "false");
                  break;
                }

                /* stereo mode (whether the video has two streams, where
                 * one is for the left eye and the other for the right eye,
                 * which creates a 3D-like effect) */
                case 0x53B9 /*KaxVideoStereoMode::ClassInfos.GlobalId*/: {
                  KaxVideoStereoMode *stereomode =
			(KaxVideoStereoMode *) subentryval;
                  stereomode->ReadData (demux->str->I_O ());
                  videocontext->mode = (eye_mode) uint32 (*stereomode);
                  GST_DEBUG ("header: video stereo mode = %d",
			     videocontext->mode);
                  break;
                }

                /* aspect ratio behaviour */
                case 0x54B3 /*KaxVideoAspectRatio::ClassInfos.GlobalId*/: {
                  KaxVideoAspectRatio *asr = (KaxVideoAspectRatio *) subentryval;
                  asr->ReadData (demux->str->I_O ());
                  videocontext->asr = (aspect_ratio_type) uint32 (*asr);
                  GST_DEBUG ("header: video aspect ratio type = %u",
			     videocontext->asr);
                  break;
                }

                /* colourspace (only matters for raw video) fourcc */
                case 0x2EB524 /*KaxVideoColourSpace::ClassInfos.GlobalId*/: {
                  KaxVideoColourSpace *csp = (KaxVideoColourSpace *) subentryval;
                  csp->ReadData (demux->str->I_O ());
                  videocontext->fourcc = uint32 (csp);
                  GST_DEBUG ("header: video colourspace = " GST_FOURCC_FORMAT,
			     GST_FOURCC_ARGS (videocontext->fourcc));
                  break;
                }

                case 0xEC /*EbmlVoid::ClassInfos.GlobalID*/:
                  break;

                default:
                  GST_DEBUG ("Unknown matroska video-subtrack header ID 0x%x (%s)",
			     EbmlId (*entryval).Value, typeid (*entryval).name ());
              }
              break;
            }

            default:
              g_assert (0);
          }

          if (demux->last_level == 0) {
            subentryval->SkipData (*demux->str, subentryval->Generic ().Context);
          }
          delete subentryval;
        }
        break;
      }

      /* codec identifier */
      case 0x86 /*KaxCodecID::ClassInfos.GlobalId*/: {
        KaxCodecID *id = (KaxCodecID *) entryval;
        id->ReadData (demux->str->I_O ());
        codecid = g_strdup (string (*(EbmlString *) id).c_str ());
        GST_DEBUG ("header: codec ID = %s", codecid);
        break;
      }

      /* codec private data */
      case 0x63A2 /*KaxCodecPrivate::ClassInfos.GlobalId*/: {
        KaxCodecPrivate *priv = (KaxCodecPrivate *) entryval;
        priv->ReadData (demux->str->I_O ());
        codecpriv = (gpointer) g_memdup ((gconstpointer) &binary (*priv),
					 priv->GetSize ());
        codecprivsize = priv->GetSize ();
        GST_DEBUG ("header: codec private = %p", codecpriv);
        break;
      }

      /* name of the codec */
      case 0x258688 /*KaxCodecName::ClassInfos.GlobalId*/: {
        KaxCodecName *name = (KaxCodecName *) entryval;
        name->ReadData (demux->str->I_O ());
        codecname = g_strdup (string (*(EbmlString *) name).c_str ());
        GST_DEBUG ("header: codec name = %s", codecname);
        break;
      }

      /* information URL */
      case 0x3B4040 /*KaxCodecInfoUrl::ClassInfos.GlobalId*/: {
        KaxCodecInfoURL *url = (KaxCodecInfoURL *) entryval;
        url->ReadData (demux->str->I_O ());
        GST_DEBUG ("header: codec information URL = %s",
		   string (*url).c_str ());
        break;
      }

      /* where to get */
      case 0x26B240 /*KaxCodecDownloadUrl::ClassInfos.GlobalId*/: {
        KaxCodecDownloadURL *url = (KaxCodecDownloadURL *) entryval;
        url->ReadData (demux->str->I_O ());
        GST_DEBUG ("header: codec download URL = %s",
		   string (*url).c_str ());
        break;
      }

      /* name of this track */
      case 0x536E /*KaxTrackName::ClassInfos.GlobalId*/: {
        KaxTrackName *name = (KaxTrackName *) entryval;
        name->ReadData (demux->str->I_O ());
        context->name = g_strdup (string (*(EbmlString *) name).c_str ());
        GST_DEBUG ("header: track name = %s", context->name);
        break;
      }

      /* language (matters for audio/subtitles, mostly) */
      case 0x22B59C /*KaxTrackLanguage::ClassInfos.GlobalId*/: {
        KaxTrackLanguage *lang = (KaxTrackLanguage *) entryval;
        lang->ReadData (demux->str->I_O ());
        context->language = g_strdup (string (*(EbmlString *) lang).c_str ());
        GST_DEBUG ("header: track language = %s", context->language);
        break;
      }

      /* whether this is actually used */
      case 0xB9 /*KaxTrackFlagEnabled::ClassInfos.GlobalId*/: {
        KaxTrackFlagEnabled *enabled = (KaxTrackFlagEnabled *) entryval;
        enabled->ReadData (demux->str->I_O ());
        if (uint32 (*enabled)) {
          context->flags |= GST_MATROSKA_SRC_ENABLED;
        } else {
          context->flags &= ~GST_MATROSKA_SRC_ENABLED;
        }
        GST_DEBUG ("header: track enabled = %s",
		   context->flags & GST_MATROSKA_SRC_ENABLED ?
		   "true" : "false");
        break;
      }

      /* whether it's the default for this track type */
      case 0x88 /*KaxTrackFlagDefault::ClassInfos.GlobalId*/: {
        KaxTrackFlagDefault *def = (KaxTrackFlagDefault *) entryval;
        def->ReadData (demux->str->I_O ());
        if (uint32 (*def)) {
          context->flags |= GST_MATROSKA_SRC_DEFAULT;
        } else {
          context->flags &= ~GST_MATROSKA_SRC_DEFAULT;
        }
        GST_DEBUG ("header: track default = %s",
		   context->flags & GST_MATROSKA_SRC_DEFAULT ?
		   "true" : "false");
        break;
      }

      /* lacing (like MPEG, where blocks don't end/start on frame
       * boundaries) */
      case 0x9C /*KaxTrackFlagLacing::ClassInfos.GlobalId*/: {
        KaxTrackFlagLacing *lacing = (KaxTrackFlagLacing *) entryval;
        lacing->ReadData (demux->str->I_O ());
        if (uint32 (*lacing)) {
          context->flags |= GST_MATROSKA_SRC_LACING;
        } else {
          context->flags &= ~GST_MATROSKA_SRC_LACING;
        }
        GST_DEBUG ("header: track lacing = %s",
		   context->flags & GST_MATROSKA_SRC_LACING ?
		   "true" : "false");
        break;
      }

      /* min. number of frames to keep in cache */
      case 0x6DE7 /*KaxTrackFlagMinCache::ClassInfos.GlobalId*/: {
        KaxTrackMinCache *min_cache = (KaxTrackMinCache *) entryval;
        min_cache->ReadData (demux->str->I_O ());
        GST_DEBUG ("header: min. cache = %u", uint32 (min_cache));
        break;
      }

      /* max. number of frames to keep in cache */
      case 0x6DF8 /*KaxTrackFlagMaxCache::ClassInfos.GlobalId*/: {
        KaxTrackMaxCache *max_cache = (KaxTrackMaxCache *) entryval;
        max_cache->ReadData (demux->str->I_O ());
        GST_DEBUG ("header: max. cache = %u", uint32 (max_cache));
        break;
      }

      /* default length (in time) of one data block in this track */
      case 0x23E383 /*KaxTrackDefaultDuration::ClassInfos.GlobalId*/: {
        KaxTrackDefaultDuration *defdur = (KaxTrackDefaultDuration *) entryval;
        defdur->ReadData (demux->str->I_O ());
        context->default_duration = uint64 (*defdur);
        GST_DEBUG ("header: default duration = %" G_GUINT64_FORMAT,
		   context->default_duration);
        break;
      }

      case 0xEC /*EbmlVoid::ClassInfos.GlobalID*/:
        break;

      default:
        GST_DEBUG ("Unknown matroska track header ID 0x%x (%s)",
		   EbmlId (*entryval).Value, typeid (*entryval).name ());
    }

    if (demux->last_level == 0) {
      entryval->SkipData (*demux->str, entryval->Generic ().Context);
    }
    delete entryval;
  } while (res == TRUE);

  if (context == NULL ||
      context->type == 0 ||
      codecid == NULL ||
      !res) {
    GST_DEBUG ("Unknown stream or codec or general read error");
    return FALSE;
  }

  /* now create the GStreamer connectivity */
  switch (context->type) {
    case track_video:
      /* video */
      padname = g_strdup_printf ("video_%02d", demux->num_v_streams);
      templ = videosrctempl;
      caps = gst_matroska_demux_video_caps ((GstMatroskaVideoSrcContext *) context,
					    codecid, codecpriv, codecprivsize);
      break;

    case track_audio:
      /* audio */
      padname = g_strdup_printf ("audio_%02d", demux->num_a_streams);
      templ = audiosrctempl;
      caps = gst_matroska_demux_audio_caps ((GstMatroskaAudioSrcContext *) context,
					    codecid, codecpriv, codecprivsize);
      break;

    case track_complex:
      /* video/audio interleaved, such as DV */
      padname = g_strdup_printf ("video_%02d", demux->num_v_streams);
      templ = videosrctempl;
      caps = gst_matroska_demux_iavs_caps ((GstMatroskaComplexSrcContext *) context,
					   codecid, codecpriv, codecprivsize);
      break;

    case track_subtitle:
      /* subtitle */
      padname = g_strdup_printf ("subtitle_%02d", demux->num_t_streams);
      templ = subtitlesrctempl;
      caps = gst_matroska_demux_subtitle_caps ((GstMatroskaSubtitleSrcContext *)
							context,
					       codecid, codecpriv, codecprivsize);
      break;

    case track_logo:
    case track_control:
    default:
      /* we should already have quit by now */
      g_assert (0);
  }

  /* the pad in here */
  context->pad =  gst_pad_new_from_template (templ, padname);

  if (caps != NULL) {
    if (gst_pad_try_set_caps (context->pad, caps) <= 0) {
      GST_DEBUG ("Failed to set caps on next element for %s",
		 padname); 
    }
  }

  g_free (padname);

  /* set some functions */
  gst_pad_set_formats_function (context->pad,
				gst_matroska_demux_get_src_formats);
  gst_pad_set_event_mask_function (context->pad,
				   gst_matroska_demux_get_event_mask);
  gst_pad_set_event_function (context->pad,
			      gst_matroska_demux_handle_src_event);
  gst_pad_set_query_type_function (context->pad,
				   gst_matroska_demux_get_src_query_types);
  gst_pad_set_query_function (context->pad,
			      gst_matroska_demux_handle_src_query);

  gst_element_add_pad (element, context->pad);

  /* tadaah! */
  return TRUE;
}

static const GstFormat *
gst_matroska_demux_get_src_formats (GstPad *pad) 
{
  /*GstMatroskaDemux *demux = GST_MATROSKA_DEMUX (gst_pad_get_parent (pad));*/

  /* we could try to look for units (i.e. samples) in audio streams
   * or video streams, but both samplerate and framerate are not
   * always constant, and since we only have a time indication, we
   * cannot guarantee anything here based purely on index. So, we
   * only support time for now. */
  static const GstFormat src_formats[] = {
    GST_FORMAT_TIME,
    (GstFormat) 0
  };

  return src_formats;
}

static const GstQueryType *
gst_matroska_demux_get_src_query_types (GstPad *pad) 
{
  static const GstQueryType src_types[] = {
    GST_QUERY_TOTAL,
    GST_QUERY_POSITION,
    (GstQueryType) 0
  };

  return src_types;
}

static gboolean
gst_matroska_demux_handle_src_query (GstPad       *pad,
				     GstQueryType  type, 
				     GstFormat    *format,
				     gint64       *value)
{
  gboolean res = TRUE;
  GstMatroskaDemux *demux = GST_MATROSKA_DEMUX (gst_pad_get_parent (pad));
  guint n;

  switch (type) {
    case GST_QUERY_TOTAL:
      switch (*format) {
        case GST_FORMAT_DEFAULT:
          *format = GST_FORMAT_TIME;
          /* fall through */
        case GST_FORMAT_TIME:
          *value = demux->duration;
	  break;
	default:
          res = FALSE;
	  break;
      }
      break;

    case GST_QUERY_POSITION:
      switch (*format) {
        case GST_FORMAT_DEFAULT:
          *format = GST_FORMAT_TIME;
          /* fall through */
        case GST_FORMAT_TIME:
          for (n = 0; n < demux->num_streams; n++) {
            if (demux->src[n] != NULL &&
                demux->src[n]->pad == pad) {
              break;
            }
          }
          if (n == demux->num_streams) {
            g_warning ("Unknown pad %p", pad);
            res = FALSE;
          } else {
            *value = demux->src[n]->pos;
          }
	  break;
	default:
          res = FALSE;
	  break;
      }
      break;

    default:
      res = FALSE;
      break;
  }

  return res;
}

static GstMatroskaDemuxIndex *
gst_matroskademux_seek (GstMatroskaDemux *demux)
{
  guint entry = (guint) -1;
  guint64 offset = demux->seek_pending;
  guint n;

  /* make sure we don't seek twice */
  demux->seek_pending = GST_CLOCK_TIME_NONE;

  for (n = 0; n < demux->num_indexes; n++) {
    if (entry == (guint) -1) {
      entry = n;
    } else {
      gfloat diff_old = fabs (1. * (demux->index[entry].time - offset)),                     diff_new = fabs (1. * (demux->index[n].time - offset));

      if (diff_new < diff_old) {
        entry = n;
      }
    }
  }

  if (entry != (guint) -1) {
    return &demux->index[entry];
  }

  return NULL;
}

static gboolean
gst_matroska_demux_send_event (GstElement *element,
			       GstEvent   *event)
{
  GstMatroskaDemux *demux = GST_MATROSKA_DEMUX (element);
  gboolean res = TRUE;

  switch (GST_EVENT_TYPE (event)) {
    case GST_EVENT_SEEK:
      switch (GST_EVENT_SEEK_FORMAT (event)) {
        case GST_FORMAT_TIME:
          demux->seek_pending = GST_EVENT_SEEK_OFFSET (event);
          break;

        default:
          g_warning("Only time seek is supported");
          res = FALSE;
          break;
      }
      break;

    default:
      GST_DEBUG ("Unhandled event of type %d",
		 GST_EVENT_TYPE (event));
      res = FALSE;
      break;
  }

  gst_event_unref (event);

  return res;
}

static const GstEventMask *
gst_matroska_demux_get_event_mask (GstPad *pad)
{
  static const GstEventMask masks[] = {
    { GST_EVENT_SEEK, 	      (GstEventFlag) ((gint) GST_SEEK_METHOD_SET |
					      (gint) GST_SEEK_FLAG_KEY_UNIT) },
    { GST_EVENT_SEEK_SEGMENT, (GstEventFlag) ((gint) GST_SEEK_METHOD_SET |
					      (gint) GST_SEEK_FLAG_KEY_UNIT) },
    { (GstEventType) 0,	      (GstEventFlag) 0 }
  };

  return masks;
}
	
static gboolean
gst_matroska_demux_handle_src_event (GstPad   *pad,
				     GstEvent *event)
{
  GstMatroskaDemux *demux = GST_MATROSKA_DEMUX (gst_pad_get_parent (pad));
  gboolean res = TRUE;
  
  switch (GST_EVENT_TYPE (event)) {
    case GST_EVENT_SEEK_SEGMENT:
    case GST_EVENT_SEEK:
      return gst_matroska_demux_send_event (GST_ELEMENT (demux), event);

    default:
      GST_DEBUG ("Unhandled event of type %d",
		 GST_EVENT_TYPE (event));
      res = FALSE;
      break;
  }

  gst_event_unref (event);

  return res;
}

static gboolean
gst_matroska_demux_handle_sink_event (GstMatroskaDemux *demux,
				      GstEvent         *event)
{
  guint i;

  /* forward to all src pads */
  for (i = 0; i < demux->num_streams; i++) {
    if (GST_PAD_IS_USABLE (demux->src[i]->pad)) {
      gst_event_ref (event);
      gst_pad_push (demux->src[i]->pad, GST_DATA (event));
    }
  }
  gst_event_unref (event);

  if (GST_EVENT_TYPE (event) == GST_EVENT_EOS) {
    gst_element_set_eos (GST_ELEMENT (demux));
  }

  return TRUE;
}

static gboolean
gst_matroska_demux_init_stream (GstElement *element)
{
  GstMatroskaDemux *demux = GST_MATROSKA_DEMUX (element);
  EbmlElement *header;

  /* open "file" and read intro header */
  gchar *filename = g_strdup_printf ("gst-matroska-io://%p",
				     demux->sinkpad);
  demux->io = new GstMatroskaIO (filename, MODE_READ);
  g_free(filename);

  /* create stream object from this I/O wrapper */
  demux->str = new EbmlStream (*demux->io);

  /* header */
  header = demux->str->FindNextID (EbmlHead::ClassInfos, ~0);
  if (header == NULL) {
    gst_element_error (GST_ELEMENT (demux),
		       "Failed to find EbmlHead/Matroska header");
    goto fail;
  }

  /* typefinding already validates the header, skip it here */
  header->SkipData(*demux->str, header->Generic().Context);
  delete header;

  /* find segment, must be the next element */
  while (1) {
    EbmlElement *segment;

    segment = demux->str->FindNextID (KaxSegment::ClassInfos, ~0);
    if (segment == NULL) {
      gst_element_error (GST_ELEMENT (demux),
			 "Failed to find toplevel KaxSegment element in Matroska file");
      goto fail;
    }

    if (EbmlId (*segment) == KaxSegment::ClassInfos.GlobalId) {
      demux->segment = (KaxSegment *) segment;
      break;
    }

    GST_DEBUG ("Expected a KaxSegment element, but received a %s - retrying",
               typeid (*segment).name ());

    /* so, try again... */
    segment->SkipData (*demux->str, segment->Generic ().Context);
    delete segment;
  }

  /* let's start parsing this segment at the bottom level */
  demux->last_level = 0;

  return TRUE;

fail:
  if (demux->last_el != NULL) {
    delete demux->last_el;
    demux->last_el = NULL;
  }
  if (demux->str != NULL) {
    delete demux->str;
    demux->str = NULL;
  }
  if (demux->io != NULL) {
    delete demux->io;
    demux->io = NULL;
  }
  return FALSE;
}

static gboolean
gst_matroska_demux_parse_tracks (GstElement *element,
				 KaxTracks  *tracks)
{
  gboolean res = TRUE;
  GstMatroskaDemux *demux = GST_MATROSKA_DEMUX (element);
  EbmlElement *trackentry, *tracks_el;

  tracks_el = (EbmlElement *) tracks;
  do {
    trackentry = gst_matroska_demux_next_element (demux, tracks_el);
    if (trackentry == NULL) {
      g_warning ("Failed to read next element in Matroska file header");
      res = FALSE;
      break;
    } else if (demux->last_level != 0) {
      demux->last_el = trackentry;
      break;
    }

    switch (EbmlId (*trackentry).Value) {
      /* one track within the "all-tracks" header */
      case 0xAE /*KaxTrackEntry::ClassInfos.GlobalId*/: {
        KaxTrackEntry *entry = (KaxTrackEntry *) trackentry;
        if (!gst_matroska_demux_add_stream (GST_ELEMENT (demux), entry)) {
          GST_DEBUG ("Failed to add stream, continuing anyway");
        }
        break;
      }

      case 0xEC /*EbmlVoid::ClassInfos.GlobalID*/:
        break;

      default:
        GST_DEBUG ("Unknown entry 0x%x (%s) in KaxTracks Matroska header",
		   EbmlId (*trackentry).Value, typeid (*trackentry).name ());
    }

    if (demux->last_level == 0) {
      trackentry->SkipData (*demux->str, trackentry->Generic ().Context);
    }
    delete trackentry;
  } while (res == TRUE);

  return res;
}

static gboolean
gst_matroska_demux_parse_index (GstElement *element,
				KaxCues    *index)
{
  gboolean res = TRUE;
  GstMatroskaDemux *demux = GST_MATROSKA_DEMUX (element);
  EbmlElement *indexentry, *index_el;

  index_el = (EbmlElement *) index;
  do {
    indexentry = gst_matroska_demux_next_element (demux, index_el);
    if (indexentry == NULL) {
      GST_DEBUG ("Failed to read next element in Matroska cue index");
      res = FALSE;
      break;
    } else if (demux->last_level != 0) {
      demux->last_el = indexentry;
      break;
    }

    switch (EbmlId (*indexentry).Value) {
      /* one single index entry ('point') */
      case 0xBB /*KaxTrackEntry::ClassInfos.GlobalId*/: {
        EbmlElement *pointentry;
        /* in the end, we hope to fill one entry with a
	 * timestamp, a file position and a tracknum */
        GstMatroskaDemuxIndex idx;
        idx.pos   = (guint64) -1;
        idx.time  = (guint64) -1;
        idx.track = (guint16) -1;

        do {
          pointentry = gst_matroska_demux_next_element (demux, indexentry);
          if (pointentry == NULL) {
            GST_DEBUG ("Failed to read next element in Matroska cuepoint index entry");
            res = FALSE;
            break;
          } else if (demux->last_level != 0) {
            demux->last_el = pointentry;
            break;
          }

          switch (EbmlId (*pointentry).Value) {
            /* one single index entry ('point') */
            case 0xB3 /*KaxCueTime::ClassInfos.GlobalId*/: {
              KaxCueTime *time = (KaxCueTime *) pointentry;
              time->ReadData (demux->str->I_O ());
              idx.time = uint64 (*time) * demux->time_scale;
              break;
            }

            /* position in the file + track to which it belongs */
            case 0xB7 /*KaxCueTrackPositions::ClassInfos.GlobalId*/: {
              EbmlElement *trackposentry;

              do {
                trackposentry = gst_matroska_demux_next_element (demux, pointentry);
                if (trackposentry == NULL) {
                  GST_DEBUG ("Failed to read next element in Matroska trackpos index entry");
                  res = FALSE;
                  break;
                } else if (demux->last_level != 0) {
                  demux->last_el = trackposentry;
                  break;
                }

                switch (EbmlId (*trackposentry).Value) {
                  /* track number */
                  case 0xF7 /*KaxCueTrack::ClassInfos.GlobalId*/: {
                    KaxCueTrack *track = (KaxCueTrack *) trackposentry;
                    track->ReadData (demux->str->I_O ());
                    idx.track = uint16 (*track);
                    break;
                  }

                  /* position in file */
                  case 0xF1 /*KaxCueClusterPosition::ClassInfos.GlobalId*/: {
                    KaxCueClusterPosition *pos = (KaxCueClusterPosition *)
							trackposentry;
                    pos->ReadData (demux->str->I_O ());
                    idx.pos = uint64 (*pos);
                    break;
                  }

                  case 0xEC /*EbmlVoid::ClassInfos.GlobalID*/:
                    break;

                  default:
                    GST_DEBUG ("Unknown entry 0x%x (%s) in KaxCuesTrackPositions Matroska index",
                  		   EbmlId (*trackposentry).Value,
				   typeid (*trackposentry).name ());
                }

                if (demux->last_level == 0) {
                  trackposentry->SkipData (*demux->str,
					   trackposentry->Generic ().Context);
                }
                delete trackposentry;
              } while (res == TRUE);

              break;
            }

            case 0xEC /*EbmlVoid::ClassInfos.GlobalID*/:
              break;

            default:
              GST_DEBUG ("Unknown entry 0x%x (%s) in KaxCuesPoint Matroska index",
            		   EbmlId (*pointentry).Value, typeid (*pointentry).name ());
          }

          if (demux->last_level == 0) {
            pointentry->SkipData (*demux->str, pointentry->Generic ().Context);
          }
          delete pointentry;
        } while (res == TRUE);

        /* so let's see if we got what we wanted */
        if (idx.pos   != (guint64) -1 &&
	    idx.time  != (guint64) -1 &&
	    idx.track != (guint16) -1) {
          if (demux->num_indexes % 32 == 0) {
            /* re-allocate bigger index */
            demux->index = (GstMatroskaDemuxIndex *)
				g_realloc (demux->index,
				           sizeof (GstMatroskaDemuxIndex) *
				             (demux->num_indexes + 32));
          }
          demux->index[demux->num_indexes].pos   = idx.pos;
          demux->index[demux->num_indexes].time  = idx.time;
          demux->index[demux->num_indexes].track = idx.track;
          demux->num_indexes++;
        }

        break;
      }

      case 0xEC /*EbmlVoid::ClassInfos.GlobalID*/:
        break;

      default:
        GST_DEBUG ("Unknown entry 0x%x (%s) in KaxCues Matroska index",
		   EbmlId (*indexentry).Value, typeid (*indexentry).name ());
    }

    if (demux->last_level == 0) {
      indexentry->SkipData (*demux->str, indexentry->Generic ().Context);
    }
    delete indexentry;
  } while (res == TRUE);

  return res;
}

static gboolean
gst_matroska_demux_parse_info (GstElement *element,
			       KaxInfo    *info)
{
  gboolean res = TRUE;
  GstMatroskaDemux *demux = GST_MATROSKA_DEMUX (element);
  EbmlElement *info_el, *infoentry;

  info_el = (EbmlElement *) info;
  do {
    infoentry = gst_matroska_demux_next_element (demux, info_el);
    if (infoentry == NULL) {
      g_warning ("Failed to read next element in Matroska info header");
      res = FALSE;
      break;
    } else if (demux->last_level != 0) {
      demux->last_el = infoentry;
      break;
    }

    switch (EbmlId (*infoentry).Value) {
      /* cluster timecode */
      case 0x2AD7B1 /*KaxTimecodeScale::ClassInfos.GlobalId*/: {
        KaxTimecodeScale *timescale = (KaxTimecodeScale *) infoentry;
        timescale->ReadData (demux->str->I_O ());
        demux->time_scale = uint64 (*timescale);
        break;
      }

      case 0x4489 /*KaxDuration::ClassInfos.GlobalId*/: {
        KaxDuration *dur = (KaxDuration *) infoentry;
        dur->ReadData (demux->str->I_O ());
        demux->duration = guint64 (float (*dur) * demux->time_scale);
        break;
      }

      case 0x5741 /*KaxWritingApp::ClassInfos.GlobalId*/: {
        KaxWritingApp *wr_app = (KaxWritingApp *) infoentry;
        wr_app->ReadData (demux->str->I_O ());
        demux->writing_app = g_strdup ((gchar *)(UTFstring (*wr_app).c_str ()));
        GST_DEBUG ("Written by %s", demux->writing_app);
        break;
      }

      case 0x4D80 /*KaxMuxingApp::ClassInfos.GlobalId*/: {
        KaxMuxingApp *mux_app = (KaxMuxingApp *) infoentry;
        mux_app->ReadData (demux->str->I_O ());
        demux->muxing_app = g_strdup ((gchar *)(UTFstring (*mux_app).c_str ()));
        GST_DEBUG ("Muxed by %s", demux->muxing_app);
        break;
      }

      case 0x4461 /*KaxDateUTC::ClassInfos.GlobalId*/: {
        KaxDateUTC *date = (KaxDateUTC *) infoentry;
        time_t ep_time = date->GetEpochDate ();
        char str[256];
        if (ctime_r (&ep_time, str) != NULL) {
          GST_DEBUG ("File created at %s (UTC)",
		     str);
        } else {
          GST_DEBUG ("Invalid date specified (%u seconds since epoch)",
		     (guint32) ep_time);
        }
        break;
      }

      case 0xEC /*EbmlVoid::ClassInfos.GlobalID*/:
        break;

      default:
        GST_DEBUG ("Unknown entry 0x%x (%s) in KaxInfo Matroska header",
		   EbmlId (*infoentry).Value, typeid(*infoentry).name());
    }

    if (demux->last_level == 0) {
      infoentry->SkipData (*demux->str, infoentry->Generic ().Context);
    }
    delete infoentry;
  } while (res == TRUE);

  return res;
}

static gboolean
gst_matroska_demux_parse_metadata (GstElement *element,
				   KaxTags    *headentry)
{
  gboolean res = TRUE;
  /*GstMatroskaDemux *demux = GST_MATROSKA_DEMUX (element);*/

  //..

  return res;
}

static gboolean
gst_matroska_demux_parse_blockgroup (GstElement    *element,
				     KaxBlockGroup *blockgroup,
				     KaxCluster    *cluster)
{
  gboolean res = TRUE;
  GstMatroskaDemux *demux = GST_MATROSKA_DEMUX (element);
  EbmlElement *bg_el, *bgentry;

  bg_el = (EbmlElement *) blockgroup;
  do {
    bgentry = gst_matroska_demux_next_element (demux, bg_el);

    if (bgentry == NULL) {
      GST_DEBUG ("Failed to read next element in Matroska blockgroup - EOS?");
      res = FALSE;
      break;
    } else if (demux->last_level != 0) {
      demux->last_el = bgentry;
      break;
    }

    switch (EbmlId (*bgentry).Value) {
      /* one block inside this group */
      case 0xA1 /*KaxBlock::ClassInfos.GlobalId*/: {
        guint n;
        KaxBlock *block = (KaxBlock *) bgentry;
        block->ReadData (demux->str->I_O ());
        block->SetParent (*cluster);

        /* find pad that it belongs to */
        n = gst_matroska_demux_stream_from_num (demux, block->TrackNum ());

        if (demux->src[n]->pad != NULL &&
	    GST_PAD_IS_USABLE (demux->src[n]->pad)) {
          guint i;

          for (i = 0; i < block->NumberFrames (); i++) {
            DataBuffer *data = &block->GetBuffer (i);
            GstBuffer *buf = gst_buffer_new_and_alloc (data->Size ());

            /* copy into our own controlled buffers */
            memcpy (GST_BUFFER_DATA (buf),
		    (gpointer) data->Buffer (),
		    data->Size());
            demux->src[n]->pos = GST_BUFFER_TIMESTAMP (buf) =
			block->GlobalTimecode ();
            GST_BUFFER_DURATION (buf) = demux->src[n]->default_duration;

            gst_pad_push (demux->src[n]->pad, GST_DATA(buf));
          }
        }

        break;
      }

      case 0x9B /*KaxBlockDuration::ClassInfos.GlobalID*/: {
        KaxBlockDuration *dur = (KaxBlockDuration *) bgentry;
        g_warning ("Fixme: implement support for BlockDuration (%"
		   G_GUINT64_FORMAT ")", uint64 (*dur));
        break;
      }

      case 0xEC /*EbmlVoid::ClassInfos.GlobalID*/:
        break;

      default:
        GST_DEBUG ("Unknown entry 0x%x (%s) in KaxBlockGroup Matroska data",
		   EbmlId (*bgentry).Value, typeid (*bgentry).name ());
    }

    if (demux->last_level == 0) {
      bgentry->SkipData (*demux->str, bgentry->Generic ().Context);
    }
    delete bgentry;
  } while (res == TRUE);

  return res;
}

static gboolean
gst_matroska_demux_parse_cluster (GstElement *element,
				  KaxCluster *cluster)
{
  gboolean res = TRUE;
  GstMatroskaDemux *demux = GST_MATROSKA_DEMUX (element);
  EbmlElement *cluster_el, *clusterentry;

  /* Not intending to look like a moron, but we only seek when
   * we've parsed the headers (for indexes etc.) - so we do that
   * here... Yes, this is ugly. Thanks for noticing. */
  if (demux->seek_pending != GST_CLOCK_TIME_NONE) {
    GstMatroskaDemuxIndex *entry = gst_matroskademux_seek (demux);
    if (entry != NULL) {
      GstEvent *discont = gst_event_new_discontinuous (FALSE, GST_FORMAT_TIME,
                                                       entry->time);
      gst_bytestream_seek (demux->io->bs, entry->pos,
                           GST_SEEK_METHOD_SET);
      gst_matroska_demux_handle_sink_event (demux, discont);
    }
  }

  cluster_el = (EbmlElement *) cluster;
  do {
    clusterentry = gst_matroska_demux_next_element (demux, cluster_el);
    if (clusterentry == NULL) {
      GST_DEBUG ("Failed to read next element in Matroska cluster - EOS?");
      res = FALSE;
      break;
    } else if (demux->last_level != 0) {
      demux->last_el = clusterentry;
      break;
    }

    switch (EbmlId (*clusterentry).Value) {
      /* cluster timecode */
      case 0xE7 /*KaxClusterTimecode::ClassInfos.GlobalId*/: {
        KaxClusterTimecode *timecode = (KaxClusterTimecode *) clusterentry;
        timecode->ReadData (demux->str->I_O ());
        cluster->InitTimecode (uint64 (*timecode), demux->time_scale);
        break;
      }

      /* a group of blocks inside a cluster */
      case 0xA0 /*KaxBlockGroup::ClassInfos.GlobalId*/: {
        KaxBlockGroup *blockgroup = (KaxBlockGroup *) clusterentry;
        res = gst_matroska_demux_parse_blockgroup (GST_ELEMENT (demux),
						   blockgroup, cluster);
        break;
      }

      case 0xEC /*EbmlVoid::ClassInfos.GlobalID*/:
        break;

      default:
        GST_DEBUG ("Unknown entry 0x%x (%s) in KaxCluster Matroska data",
		   EbmlId (*clusterentry).Value, typeid (*clusterentry).name ());
    }

    if (demux->last_level == 0) {
      clusterentry->SkipData (*demux->str, clusterentry->Generic ().Context);
    }
    delete clusterentry;
  } while (res == TRUE);

  return res;
}

static gboolean
gst_matroska_demux_parse_contents (GstElement  *element,
				   KaxSeekHead *headentry)
{
  gboolean res = TRUE;
  /*GstMatroskaDemux *demux = GST_MATROSKA_DEMUX (element);*/

  //..

  return res;
}

static gboolean
gst_matroska_demux_loop_stream (GstElement *element)
{
  gboolean res = TRUE;
  GstMatroskaDemux *demux = GST_MATROSKA_DEMUX (element);

  /* we've found our segment, start reading the different contents in here */
  do {
    EbmlElement *headentry;
    headentry = gst_matroska_demux_next_element (demux, demux->segment);
    if (headentry == NULL) {
      GST_DEBUG ("Could not read next element from main level - EOS?");
      gst_matroska_demux_handle_sink_event (demux,
					    gst_event_new (GST_EVENT_EOS));
      res = FALSE;
      break;
    } else if (demux->last_level != 0) {
      //g_assert (0); /* cannot happen, this *is* the top level */
      demux->last_level = 0;
    }

    /* well, cases don't work here since these aren't constants...
     * argh, the horrors of C. Ohwell, fortunately, these IDs are
     * actually fixed... */
    switch (EbmlId (*headentry).Value) {
      /* stream info */
      case 0x1549A966 /*KaxInfo::ClassInfos.GlobalId*/: {
        KaxInfo *info = (KaxInfo *) headentry;
        res = gst_matroska_demux_parse_info (GST_ELEMENT (demux), info);
        break;
      }

      /* track info headers */
      case 0x1654AE6B /*KaxTracks::ClassInfos.GlobalId*/: {
        KaxTracks *tracks = (KaxTracks *) headentry;
        res = gst_matroska_demux_parse_tracks (GST_ELEMENT (demux), tracks);
        break;
      }

      /* stream index */
      case 0x1C53BB6B /*KaxCues::ClassInfos.GlobalId*/: {
        KaxCues *cues = (KaxCues *) headentry;
        res = gst_matroska_demux_parse_index (GST_ELEMENT (demux), cues);
        break;
      }

      /* metadata */
      case 0x1254C367 /*KaxTags::ClassInfos.GlobalId*/: {
        KaxTags *tags = (KaxTags *) headentry;
        res = gst_matroska_demux_parse_metadata (GST_ELEMENT (demux), tags);
        break;
      }

      /* file index (if seekable, seek to Cues/Tags to parse it) */
      case 0x114D9B74 /*KaxSeekHead::ClassInfos.GlobalId*/: {
        KaxSeekHead *contents = (KaxSeekHead *) headentry;
        res = gst_matroska_demux_parse_contents (GST_ELEMENT (demux), contents);
        break;
      }

      case 0x1F43B675 /*KaxCluster::ClassInfos.GlobalId*/: {
        KaxCluster *cluster = (KaxCluster *) headentry;
        /* The idea is that we parse one cluster per loop and
         * then break out of the loop here. In the next call
         * of the loopfunc, we will get back here with the
         * next cluster. If an error occurs, we didn't
         * actually push a buffer, but we still want to break
         * out of the loop to handle a possible error. We'll
         * get back here if it's recoverable. */
        gst_matroska_demux_parse_cluster (GST_ELEMENT (demux), cluster);
        res = FALSE;
        break;
      }

      case 0xEC /*EbmlVoid::ClassInfos.GlobalID*/:
        break;

      default:
        GST_DEBUG ("Unknown matroska file header ID 0x%x (%s)",
		   EbmlId (*headentry).Value, typeid (*headentry).name ());
    }

    if (demux->last_level == 0) {
      headentry->SkipData (*demux->str, headentry->Generic ().Context);
    }
    delete headentry;
  } while (res == TRUE);

  return res;
}

static void
gst_matroska_demux_loop (GstElement *element)
{
  GstMatroskaDemux *demux = GST_MATROSKA_DEMUX (element);

  /* first, if we're to start, let's actually get starting */
  if (demux->state == GST_MATROSKA_DEMUX_START) {
    if (!gst_matroska_demux_init_stream (element)) {
      gst_element_error (element,
			 "Input is not a matroska file");
      return;
    }

    demux->state = GST_MATROSKA_DEMUX_HEADER;
  }

  gst_matroska_demux_loop_stream (element);
}

static GstCaps *
gst_matroska_demux_vfw_caps (guint32             codec_fcc,
			     gst_riff_strf_vids *vids)
{
  GstCaps *caps = NULL;

  switch (codec_fcc) {
    case GST_MAKE_FOURCC('I','4','2','0'):
    case GST_MAKE_FOURCC('Y','U','Y','2'):
      caps = GST_CAPS_NEW (
                  "matroskademux_vfw_video_src_raw",
                  "video/x-raw-yuv",
                    "format",  GST_PROPS_FOURCC (codec_fcc)
                );
      break;

    case GST_MAKE_FOURCC('M','J','P','G'): /* YUY2 MJPEG */
    case GST_MAKE_FOURCC('J','P','E','G'): /* generic (mostly RGB) MJPEG */
    case GST_MAKE_FOURCC('P','I','X','L'): /* Miro/Pinnacle fourccs */
    case GST_MAKE_FOURCC('V','I','X','L'): /* Miro/Pinnacle fourccs */
      caps = GST_CAPS_NEW (
                  "matroskademux_vfw_video_src_jpeg",
                  "video/x-jpeg",
                    NULL
                );
      break;

    case GST_MAKE_FOURCC('H','F','Y','U'):
      caps = GST_CAPS_NEW (
                  "matroskademux_vfw_video_src_hfyu",
                  "video/x-huffyuv",
                    NULL
                );
      break;

    case GST_MAKE_FOURCC('M','P','E','G'):
    case GST_MAKE_FOURCC('M','P','G','I'):
      caps = GST_CAPS_NEW (
                  "matroskademux_vfw_video_src_mpeg",
                  "video/mpeg",
                    "systemstream", GST_PROPS_BOOLEAN (FALSE),
		    "mpegversion", GST_PROPS_BOOLEAN (1)
                );
      break;

    case GST_MAKE_FOURCC('H','2','6','3'):
    case GST_MAKE_FOURCC('i','2','6','3'):
    case GST_MAKE_FOURCC('L','2','6','3'):
    case GST_MAKE_FOURCC('M','2','6','3'):
    case GST_MAKE_FOURCC('V','D','O','W'):
    case GST_MAKE_FOURCC('V','I','V','O'):
    case GST_MAKE_FOURCC('x','2','6','3'):
      caps = GST_CAPS_NEW (
                  "matroskademux_vfw_video_src_263",
                  "video/x-h263",
                    NULL
                );
      break;

    case GST_MAKE_FOURCC('D','I','V','3'):
    case GST_MAKE_FOURCC('D','I','V','4'):
    case GST_MAKE_FOURCC('D','I','V','5'):
      caps = GST_CAPS_NEW (
                  "matroskademux_vfw_video_src_divx3",
                  "video/x-divx",
		    "divxversion", GST_PROPS_INT(3)
                );
      break;

    case GST_MAKE_FOURCC('d','i','v','x'):
    case GST_MAKE_FOURCC('D','I','V','X'):
    case GST_MAKE_FOURCC('D','X','5','0'):
      caps = GST_CAPS_NEW (
                  "matroskademux_vfw_video_src_divx5",
                  "video/x-divx",
		    "divxversion", GST_PROPS_INT(5)
                );
      break;

    case GST_MAKE_FOURCC('X','V','I','D'):
    case GST_MAKE_FOURCC('x','v','i','d'):
      caps = GST_CAPS_NEW (
                  "matroskademux_vfw_video_src",
                  "video/x-xvid",
                    NULL
                );
      break;

    case GST_MAKE_FOURCC('M','P','G','4'):
      caps = GST_CAPS_NEW (
                  "matroskademux_vfw_video_src",
                  "video/x-msmpeg",
		    "msmpegversion", GST_PROPS_INT (41)
                );
      break;

    case GST_MAKE_FOURCC('M','P','4','2'):
      caps = GST_CAPS_NEW (
                  "matroskademux_vfw_video_src",
                  "video/x-msmpeg",
		    "msmpegversion", GST_PROPS_INT (42)
                );
      break;

    case GST_MAKE_FOURCC('M','P','4','3'):
      caps = GST_CAPS_NEW (
                  "matroskademux_vfw_video_src",
                  "video/x-msmpeg",
		    "msmpegversion", GST_PROPS_INT (43)
                );
      break;

    case GST_MAKE_FOURCC('3','I','V','1'):
    case GST_MAKE_FOURCC('3','I','V','2'):
      caps = GST_CAPS_NEW (
		  "matroskademux_vfw_video_src_3ivx",
		  "video/x-3ivx",
		    NULL
		);
      break;

    case GST_MAKE_FOURCC('D','V','S','D'):
    case GST_MAKE_FOURCC('d','v','s','d'):
      caps = GST_CAPS_NEW (
                  "matroskademux_vfw_video_src",
                  "video/x-dv",
                    "systemstream", GST_PROPS_BOOLEAN (FALSE)
                );
      break;

    case GST_MAKE_FOURCC('W','M','V','1'):
      caps = GST_CAPS_NEW (
                  "matroskademux_vfw_video_src_wmv1",
                  "video/x-wmv",
                    "wmvversion", GST_PROPS_INT (1)
                );
      break;

    case GST_MAKE_FOURCC('W','M','V','2'):
      caps = GST_CAPS_NEW (
                  "matroskademux_vfw_video_src_wmv2",
                  "video/x-wmv",
                    "wmvversion", GST_PROPS_INT (2)
                );
      break;

    default:
      g_warning ("avidemux: unkown video format " GST_FOURCC_FORMAT,
		 GST_FOURCC_ARGS(codec_fcc));
      break;
  }

  return caps;
}

static GstCaps *
gst_matroska_demux_video_caps (GstMatroskaVideoSrcContext *context,
			       const gchar                *codec_id,
			       gpointer                    data,
			       guint                       size)
{
  GstCaps *caps = NULL;

  if (!strcmp (codec_id, "V_MS/VFW/FOURCC")) {
    gst_riff_strf_vids *vids = (gst_riff_strf_vids *) data;

    /* assure size is big enough */
    if (size < 24) {
      g_warning ("Too small BITMAPINFOHEADER (%d bytes)", size);
      return NULL;
    }
    if (size < sizeof (gst_riff_strf_vids)) {
      vids = (gst_riff_strf_vids *) g_realloc (vids, sizeof (gst_riff_strf_vids));
    }

    /* little-endian -> byte-order */
    vids->size        = GUINT32_FROM_LE (vids->size);
    vids->width       = GUINT32_FROM_LE (vids->width);
    vids->height      = GUINT32_FROM_LE (vids->height);
    vids->planes      = GUINT16_FROM_LE (vids->planes);
    vids->bit_cnt     = GUINT16_FROM_LE (vids->bit_cnt);
    vids->compression = GUINT32_FROM_LE (vids->compression);
    vids->image_size  = GUINT32_FROM_LE (vids->image_size);
    vids->xpels_meter = GUINT32_FROM_LE (vids->xpels_meter);
    vids->ypels_meter = GUINT32_FROM_LE (vids->ypels_meter);
    vids->num_colors  = GUINT32_FROM_LE (vids->num_colors);
    vids->imp_colors  = GUINT32_FROM_LE (vids->imp_colors);

    caps = gst_matroska_demux_vfw_caps (vids->compression, vids);
  } else if (!strcmp (codec_id, "V_UNCOMPRESSED")) {
    guint32 fourcc = 0;

    /* how nice, this is undocumented... */
    switch (context->fourcc) {
      case GST_MAKE_FOURCC ('I','4','2','0'):
      case GST_MAKE_FOURCC ('Y','U','Y','2'):
      case GST_MAKE_FOURCC ('Y','V','1','2'):
        fourcc = context->fourcc;
        break;
    }

    if (!fourcc) {
      GST_DEBUG ("Unknown fourcc " GST_FOURCC_FORMAT,
		 GST_FOURCC_ARGS (context->fourcc));
      return NULL;
    }

    caps = GST_CAPS_NEW ("matroskademux_src_uncompressed",
			 "video/x-raw-yuv",
			   "format", GST_PROPS_FOURCC (fourcc));
  } else if (!strcmp (codec_id, "V_MPEG4/ISO/SP")) {
    caps = GST_CAPS_NEW ("matroskademux_src_divx4",
			 "video/x-divx",
			   "divxversion", GST_PROPS_INT (4));
  } else if (!strcmp (codec_id, "V_MPEG4/ISO/ASP") ||
	     !strcmp (codec_id, "V_MPEG4/ISO/AP")) {
    caps = GST_CAPS_NEW ("matroskademux_src_divx5",
			 "video/x-divx",
			   "divxversion", GST_PROPS_INT (5));
    caps = gst_caps_append (caps,
	   GST_CAPS_NEW ("matroskademux_src_xvid",
			 "video/x-xvid",
			   NULL));
    caps = gst_caps_append (caps,
	   GST_CAPS_NEW ("matroskademux_src_mpeg4asp/ap",
			 "video/mpeg",
			   "mpegversion",  GST_PROPS_INT (4),
			   "systemstream", GST_PROPS_BOOLEAN (FALSE)));
  } else if (!strcmp (codec_id, "V_MPEG4/MS/V3")) {
    caps = GST_CAPS_NEW ("matroskademux_src_msmpeg4v3",
			 "video/x-divx",
			   "divxversion", GST_PROPS_INT (3));
    caps = gst_caps_append (caps,
	   GST_CAPS_NEW ("matroskademux_src_divx3",
			 "video/x-msmpeg",
			   "divxversion", GST_PROPS_INT (43)));
  } else if (!strcmp (codec_id, "V_MPEG1") ||
	     !strcmp (codec_id, "V_MPEG2")) {
    gint mpegversion = (gint) (codec_id[6] - '0');

    caps = GST_CAPS_NEW ("matroska_demux_mpeg1/2",
			 "video/mpeg",
			   "systemstream", GST_PROPS_BOOLEAN (FALSE),
			   "mpegversion",  GST_PROPS_INT (mpegversion));
  } else {
    GST_DEBUG ("Unknown codec '%s', cannot build Caps",
	       codec_id);
  }

  if (caps != NULL) {
    GstCaps *one;
    GstPropsEntry *fps = NULL;
    GstPropsEntry *width = NULL, *height = NULL;
    GstPropsEntry *pixel_width = NULL, *pixel_height = NULL;

    for (one = caps; one != NULL; one = one->next) { 
      if (context != NULL) {
        if (context->pixel_width > 0 &&
            context->pixel_height > 0) {
          gint w = context->pixel_width;
          gint h = context->pixel_height;

          width = gst_props_entry_new ("width",
				       GST_PROPS_INT (w));
          height = gst_props_entry_new ("height",
				        GST_PROPS_INT (h));
        }
        if (context->display_width > 0 &&
	    context->display_height > 0) {
          gint w = 100 * context->display_width / context->pixel_width;
          gint h = 100 * context->display_height / context->pixel_height;

          pixel_width = gst_props_entry_new ("pixel_width",
					     GST_PROPS_INT (w));
          pixel_height = gst_props_entry_new ("pixel_height",
					      GST_PROPS_INT (h));
        }
        if (((GstMatroskaSrcContext *) context)->default_duration > 0) {
          gfloat framerate = 1. * GST_SECOND /
		((GstMatroskaSrcContext *) context)->default_duration;

          fps = gst_props_entry_new ("framerate",
				     GST_PROPS_FLOAT (framerate));
        } else {
          /* sort of a hack to get most codecs to support,
	   * even if the default_duration is missing */
          fps = gst_props_entry_new ("framerate", GST_PROPS_FLOAT (0.));
        }
      } else {
        width = gst_props_entry_new ("width",
				     GST_PROPS_INT_RANGE (16, 4096));
        height = gst_props_entry_new ("height",
				      GST_PROPS_INT_RANGE (16, 4096));

        pixel_width = gst_props_entry_new ("pixel_width",
				           GST_PROPS_INT_RANGE (0, 255));
        pixel_height = gst_props_entry_new ("pixel_height",
					    GST_PROPS_INT_RANGE (0, 255));

        fps = gst_props_entry_new ("framerate",
			           GST_PROPS_FLOAT_RANGE (0, G_MAXFLOAT));
      }

      if (one->properties == NULL) {
        one->properties = gst_props_empty_new ();
      }

      if (width != NULL && height != NULL) {
        gst_props_add_entry (one->properties, width);
        gst_props_add_entry (one->properties, height);
      }

      if (pixel_width != NULL && pixel_height != NULL) {
        gst_props_add_entry (one->properties, pixel_width);
        gst_props_add_entry (one->properties, pixel_height);
      }

      if (fps != NULL) {
        gst_props_add_entry (one->properties, fps);
      }
    }
  }

  return caps;
}

static GstCaps *
gst_matroska_demux_acm_caps (guint16             codec_id,
			     gst_riff_strf_auds *auds)
{
  GstCaps *caps = NULL;

  switch (codec_id) {
    case GST_RIFF_WAVE_FORMAT_MPEGL3: /* mp3 */
      caps = GST_CAPS_NEW ("avi_demux_audio_src_mp3",
			   "audio/mpeg",
			     "layer", GST_PROPS_INT (3));
      break;

    case GST_RIFF_WAVE_FORMAT_MPEGL12: /* mp1 or mp2 */
      caps = GST_CAPS_NEW ("avi_demux_audio_src_mp12",
			   "audio/mpeg",
			     "layer", GST_PROPS_INT (2));
      break;

    case GST_RIFF_WAVE_FORMAT_PCM: /* PCM/wav */ {
      GstPropsEntry *width = NULL, *depth = NULL, *signedness = NULL;

      if (auds != NULL) {
        gint ba = GUINT16_FROM_LE (auds->blockalign);
        gint ch = GUINT16_FROM_LE (auds->channels);
        gint ws = GUINT16_FROM_LE (auds->size);

        width = gst_props_entry_new ("width",
				     GST_PROPS_INT (ba * 8 / ch));
        depth = gst_props_entry_new ("depth",
				     GST_PROPS_INT (ws));
        signedness = gst_props_entry_new ("signed",
					  GST_PROPS_BOOLEAN (ws != 8));
      } else {
        signedness = gst_props_entry_new ("signed",
					  GST_PROPS_LIST (
					    GST_PROPS_BOOLEAN (TRUE),
					    GST_PROPS_BOOLEAN (FALSE)));
        width = gst_props_entry_new ("width",
				     GST_PROPS_LIST (
				       GST_PROPS_INT (8),
				       GST_PROPS_INT (16)));
        depth = gst_props_entry_new ("depth",
				     GST_PROPS_LIST (
				       GST_PROPS_INT (8),
				       GST_PROPS_INT (16)));
      }

      caps = GST_CAPS_NEW ("avi_demux_audio_src_pcm",
			   "audio/x-raw-int",
			     "endianness",
				GST_PROPS_INT (G_LITTLE_ENDIAN));
      gst_props_add_entry (caps->properties, width);
      gst_props_add_entry (caps->properties, depth);
      gst_props_add_entry (caps->properties, signedness);
    }
      break;

    case GST_RIFF_WAVE_FORMAT_MULAW:
      if (auds != NULL && auds->size != 8) {
        g_warning ("invalid depth (%d) of mulaw audio, overwriting.",
		   auds->size);
      }
      caps = GST_CAPS_NEW ("avidemux_audio_src",
			   "audio/x-mulaw",
			     NULL);
      break;

    case GST_RIFF_WAVE_FORMAT_ALAW:
      if (auds != NULL && auds->size != 8) {
        g_warning ("invalid depth (%d) of alaw audio, overwriting.",
		   auds->size);
      }
      caps = GST_CAPS_NEW ("avidemux_audio_src",
			   "audio/x-alaw",
			     NULL);
      break;

    case GST_RIFF_WAVE_FORMAT_VORBIS1: /* ogg/vorbis mode 1 */
    case GST_RIFF_WAVE_FORMAT_VORBIS2: /* ogg/vorbis mode 2 */
    case GST_RIFF_WAVE_FORMAT_VORBIS3: /* ogg/vorbis mode 3 */
    case GST_RIFF_WAVE_FORMAT_VORBIS1PLUS: /* ogg/vorbis mode 1+ */
    case GST_RIFF_WAVE_FORMAT_VORBIS2PLUS: /* ogg/vorbis mode 2+ */
    case GST_RIFF_WAVE_FORMAT_VORBIS3PLUS: /* ogg/vorbis mode 3+ */
      caps = GST_CAPS_NEW ("asf_demux_audio_src_vorbis",
			   "audio/x-vorbis",
			     NULL);
      break;

    case GST_RIFF_WAVE_FORMAT_A52:
      caps = GST_CAPS_NEW ("asf_demux_audio_src_ac3",
			   "audio/x-ac3",
			     NULL);
      break;

    default:
      g_warning ("avidemux: unkown audio format 0x%04x",
		 codec_id);
      break;
  }

  return caps;
}

static GstCaps *
gst_matroska_demux_audio_caps (GstMatroskaAudioSrcContext *context,
			       const gchar                *codec_id,
			       gpointer                    data,
			       guint                       size)
{
  GstCaps *caps = NULL;

  if (!strcmp (codec_id, "A_MPEG/L3") ||
      !strcmp (codec_id, "A_MPEG/L2") ||
      !strcmp (codec_id, "A_MPEG/L1")) {
    gint layer = (gint) (codec_id[8] - '0');

    caps = GST_CAPS_NEW ("matroskademux_mpeg1-l1/2/3",
			 "audio/mpeg",
			   "systemstream", GST_PROPS_BOOLEAN (FALSE),
			   "layer",        GST_PROPS_INT (layer));
  } else if (!strcmp (codec_id, "A_PCM/INT/BIG") ||
	     !strcmp (codec_id, "A_PCM/INT/LIT")) {
    gint endianness = strcmp (codec_id+10, "BIG")==0 ?
		G_BIG_ENDIAN : G_LITTLE_ENDIAN;
    GstPropsEntry *depth, *width, *sign;

    if (context != NULL) {
      width = gst_props_entry_new ("width",
				   GST_PROPS_INT (context->bitdepth));
      depth = gst_props_entry_new ("depth",
				   GST_PROPS_INT (context->bitdepth));
      sign = gst_props_entry_new ("signed",
				  GST_PROPS_BOOLEAN (context->bitdepth == 8));
    } else {
      width = gst_props_entry_new ("width", GST_PROPS_LIST (
					GST_PROPS_INT (8),
					GST_PROPS_INT (16)));
      depth = gst_props_entry_new ("depth", GST_PROPS_LIST (
					GST_PROPS_INT (8),
					GST_PROPS_INT (16)));
      sign = gst_props_entry_new ("signed", GST_PROPS_LIST (
					GST_PROPS_BOOLEAN (TRUE),
					GST_PROPS_BOOLEAN (FALSE)));
    }

    caps = GST_CAPS_NEW ("matroskademux_audio_raw",
			 "audio/x-raw-int",
			   "endianness", GST_PROPS_INT (endianness));
    gst_props_add_entry (caps->properties, width);
    gst_props_add_entry (caps->properties, depth);
    gst_props_add_entry (caps->properties, sign);
  } else if (!strcmp (codec_id, "A_PCM/FLOAT/IEEE")) {
    GstPropsEntry *width;

    if (context != NULL) {
      width = gst_props_entry_new ("width",
				   GST_PROPS_INT (context->bitdepth));
    } else {
      width = gst_props_entry_new ("width", GST_PROPS_LIST (
					GST_PROPS_INT (32),
					GST_PROPS_INT (64)));
    }

    caps = GST_CAPS_NEW ("matroskademux_audio_float",
			 "audio/x-raw-float",
			   "endianness", GST_PROPS_INT (G_BYTE_ORDER),
                           "buffer-frames", GST_PROPS_INT_RANGE (1, G_MAXINT));

    gst_props_add_entry (caps->properties, width);
  } else if (!strcmp (codec_id, "A_AC3") ||
	     !strcmp (codec_id, "A_DTS")) {
    caps = GST_CAPS_NEW ("matroskademux_audio_ac3/dts",
			 "audio/x-ac3",
			   NULL);
  } else if (!strcmp (codec_id, "A_VORBIS")) {
    caps = GST_CAPS_NEW ("matroskademux_audio_vorbis",
			 "audio/x-vorbis",
			   NULL);
  } else if (!strcmp (codec_id, "A_MS/ACM")) {
    gst_riff_strf_auds *auds = (gst_riff_strf_auds *) data;

    /* little-endian -> byte-order */
    auds->format     = GUINT16_FROM_LE (auds->format);
    auds->channels   = GUINT16_FROM_LE (auds->channels);
    auds->rate       = GUINT32_FROM_LE (auds->rate);
    auds->av_bps     = GUINT32_FROM_LE (auds->av_bps);
    auds->blockalign = GUINT16_FROM_LE (auds->blockalign);
    auds->size       = GUINT16_FROM_LE (auds->size);

    caps = gst_matroska_demux_acm_caps (auds->format, auds);
  } else if (!strncmp (codec_id, "A_AAC/MPEG2/", 12)) {
    caps = GST_CAPS_NEW ("matroska_demux_aac_mpeg2",
			 "audio/mpeg",
			   "mpegversion",  GST_PROPS_INT (2),
			   "systemstream", GST_PROPS_BOOLEAN (FALSE));
  } else if (!strncmp (codec_id, "A_AAC/MPEG4/", 12)) {
    caps = GST_CAPS_NEW ("matroska_demux_aac_mpeg4",
			 "audio/mpeg",
			   "mpegversion",  GST_PROPS_INT (4),
			   "systemstream", GST_PROPS_BOOLEAN (FALSE));
  } else {
    GST_DEBUG ("Unknown codec '%s', cannot build Caps",
	       codec_id);
  }

  if (caps != NULL) {
    GstCaps *one;
    GstPropsEntry *chans = NULL, *rate = NULL;

    for (one = caps; one != NULL; one = one->next) { 
      if (context != NULL) {
        if (context->samplerate > 0 &&
            context->channels > 0) {
          chans = gst_props_entry_new ("channels",
				       GST_PROPS_INT (context->channels));
          rate = gst_props_entry_new ("rate",
				      GST_PROPS_INT (context->samplerate));
        }
      } else {
        chans = gst_props_entry_new ("channels",
				     GST_PROPS_INT_RANGE (1, 6));
        rate = gst_props_entry_new ("rate",
				    GST_PROPS_INT_RANGE (4000, 96000));
      }

      if (caps->properties == NULL) {
        caps->properties = gst_props_empty_new ();
      }

      if (chans != NULL && rate != NULL) {
        gst_props_add_entry (caps->properties, chans);
        gst_props_add_entry (caps->properties, rate);
      }
    }
  }

  return caps;
}

static GstCaps *
gst_matroska_demux_iavs_caps (GstMatroskaComplexSrcContext *context,
			      const gchar                  *codec_id,
			      gpointer                      data,
			      guint                         size)
{
  GstCaps *caps = NULL;

  //..

  return caps;
}

static GstCaps *
gst_matroska_demux_subtitle_caps (GstMatroskaSubtitleSrcContext *context,
			          const gchar                   *codec_id,
			          gpointer                       data,
			          guint                          size)
{
  GstCaps *caps = NULL;

  //..

  return caps;
}

static GstElementStateReturn
gst_matroska_demux_change_state (GstElement *element)
{
  GstMatroskaDemux *demux = GST_MATROSKA_DEMUX (element);

  switch (GST_STATE_TRANSITION (element)) {
    case GST_STATE_PAUSED_TO_READY:
      gst_matroska_demux_reset (GST_ELEMENT (demux));
      break;

    default:
      break;
  }

  if (parent_class->change_state)
    return parent_class->change_state (element);

  return GST_STATE_SUCCESS;
}

static void
gst_matroska_demux_get_property (GObject    *object,
				 guint       prop_id,
				 GValue     *value,
				 GParamSpec *pspec)
{
  GstMatroskaDemux *demux;

  g_return_if_fail (GST_IS_MATROSKA_DEMUX (object));
  demux = GST_MATROSKA_DEMUX (object);

  switch (prop_id) {
    case ARG_STREAMINFO:
      g_value_set_boxed (value, demux->streaminfo);
      break;
    case ARG_METADATA:
      g_value_set_boxed (value, demux->metadata);
      break;
    default:
      /*G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);*/
      break;
  }
}

static gboolean
plugin_init (GModule *module, GstPlugin *plugin)
{
  GstElementFactory *factory;
  GstTypeFactory *type;
  gint i;
  GstCaps *videosrccaps = NULL, *audiosrccaps = NULL,
	  *subtitlesrccaps = NULL, *temp;
  const gchar *video_id[] = {
    /* FILLME */
    NULL,
  }, *audio_id[] = {
    /* FILLME */
    NULL,
  }, *complex_id[] = {
    /* FILLME */
    NULL,
  }, *subtitle_id[] = {
    /* FILLME */
    NULL,
  };

  /* this filter needs the riff parser */
  if (!gst_library_load ("gstbytestream"))
    return FALSE;

  /* create an elementfactory for the matroska_demux element */
  factory = gst_element_factory_new ("matroskademux",
				     GST_TYPE_MATROSKA_DEMUX,
                                     &gst_matroska_demux_details);
  g_return_val_if_fail (factory != NULL, FALSE);

  /* magic */
  gst_element_factory_set_rank (factory, GST_ELEMENT_RANK_PRIMARY);

  /* video src template */
  for (i = 0; video_id[i] != NULL; i++) {
    temp = gst_matroska_demux_video_caps (NULL, video_id[i], NULL, 0);
    videosrccaps = gst_caps_append (videosrccaps, temp);
  }
  for (i = 0; complex_id[i] != NULL; i++) {
    temp = gst_matroska_demux_iavs_caps (NULL, video_id[i], NULL, 0);
    videosrccaps = gst_caps_append (videosrccaps, temp);
  }
  videosrctempl = gst_pad_template_new ("video_%02d",
					GST_PAD_SRC,
					GST_PAD_SOMETIMES,
					videosrccaps, NULL);
  gst_element_factory_add_pad_template (factory, videosrctempl);

  /* audio src template */
  for (i = 0; audio_id[i] != NULL; i++) {
    temp = gst_matroska_demux_audio_caps (NULL, audio_id[i], NULL, 0);
    audiosrccaps = gst_caps_append (audiosrccaps, temp);
  }
  audiosrctempl = gst_pad_template_new ("audio_%02d",
					GST_PAD_SRC,
					GST_PAD_SOMETIMES,
					audiosrccaps, NULL);
  gst_element_factory_add_pad_template (factory, audiosrctempl);

  /* subtitle src template */
  for (i = 0; video_id[i] != NULL; i++) {
    temp = gst_matroska_demux_video_caps (NULL, subtitle_id[i], NULL, 0);
    subtitlesrccaps = gst_caps_append (subtitlesrccaps, temp);
  }
  subtitlesrctempl = gst_pad_template_new ("subtitle_%02d",
					   GST_PAD_SRC,
					   GST_PAD_SOMETIMES,
					   subtitlesrccaps, NULL);
  gst_element_factory_add_pad_template (factory, subtitlesrctempl);

  /* input bytestream template */
  gst_element_factory_add_pad_template (factory,
					GST_PAD_TEMPLATE_GET (sink_templ));

  type = gst_type_factory_new (&matroskadefinition);
  gst_plugin_add_feature (plugin, GST_PLUGIN_FEATURE (type));

  gst_plugin_add_feature (plugin, GST_PLUGIN_FEATURE (factory));

  return TRUE;
}

GstPluginDesc plugin_desc = {
  GST_VERSION_MAJOR,
  GST_VERSION_MINOR,
  "matroskademux",
  plugin_init
};
