/* GStreamer
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 * Copyright (C) <2003> David Schleef <ds@schleef.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/* Based on xqcam.c by Paul Chinn <loomer@svpal.org>:
 *
 *   XWindow Support for QuickCam
 *   by Paul Chinn <loomer@svpal.org>
 *   Modified by Scott Laird <scott@laird.com>
 *
 * xqcam.c is under the X license.
 */
 
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gst/gst.h>
/* gcc -ansi -pedantic on GNU/Linux causes warnings and errors
 * unless this is defined:
 * warning: #warning "Files using this header must be compiled with _SVID_SOURCE or _XOPEN_SOURCE"
 */
#ifndef _XOPEN_SOURCE
#  define _XOPEN_SOURCE 1
#endif

#define HAVE_XSHM

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>

#if defined (HAVE_IPC_H) && defined (HAVE_XSHM_H) && defined (HAVE_XSHM_H)
#define HAVE_XSHM
#endif

#ifdef HAVE_XSHM
#include <sys/ipc.h>
#include <sys/shm.h>
#endif /* HAVE_XSHM */

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/Xproto.h>
#include <X11/Xmd.h>

#ifdef HAVE_XSHM
#include <X11/extensions/XShm.h>
#endif /* HAVE_XSHM */

#include "gstximage.h"
#include "gstxwindow.h"

static int XJ_caught_error;

/* FIXME: this could cause problems with multiple xvideosinks */
static int 
XJ_error_catcher (Display * d, XErrorEvent * xeev)
{
/* NOTE: This is just for debug, don't enable the printf normally.
 * When shm is unavailable this path gets taken.
 */
/*
  char buf[255];
  XGetErrorText(d, xeev->error_code, buf, 255);
  fprintf(stderr, "Caught X Error: %s\n", buf);
*/
  ++XJ_caught_error;
  return 0;
}

static void 	_gst_ximage_destroy 	(GstXImage *image);
static void 	_gst_ximage_put		(GstXWindow *window, GstXImage *image);

GstXImage *
_gst_ximage_new (GstXWindow *window, int width, int height)
{
  int (*old_handler)();
  GstXImage *new;

  new = g_new (GstXImage, 1);
  
  GST_IMAGE_TYPE (new) 		= GST_TYPE_XIMAGE;
  GST_IMAGE_DESTROYFUNC (new) 	= (GstImageDestroyFunc) _gst_ximage_destroy;
  GST_IMAGE_PUTFUNC (new) 	= (GstImagePutFunc) _gst_ximage_put;
  
  new->width = width;
  new->height = height;
  GST_IMAGE_WINDOW (new) = window;
  new->visual = DefaultVisual(window->disp, window->screen_num); 
  new->endianness = (ImageByteOrder (window->disp) == LSBFirst) ? G_LITTLE_ENDIAN:G_BIG_ENDIAN;

  XJ_caught_error = 0;

  old_handler = XSetErrorHandler(XJ_error_catcher);
  XSync(window->disp, 0);

  new->ximage = XShmCreateImage(window->disp, new->visual,
			   window->depth, ZPixmap, NULL, &new->SHMInfo,
			   new->width, new->height);
  if(!new->ximage) {
    fprintf(stderr, "CreateImage Failed\n");
    return NULL;
  }
 
  GST_IMAGE_SIZE (new) = new->ximage->bytes_per_line * new->ximage->height;

  new->SHMInfo.shmid = shmget(IPC_PRIVATE, 
		       GST_IMAGE_SIZE (new),
		       IPC_CREAT|0777);

  if (new->SHMInfo.shmid < 0) {
    perror("shmget failed:");
    return NULL;
  }
 
  GST_IMAGE_DATA (new) = new->ximage->data = new->SHMInfo.shmaddr = shmat(new->SHMInfo.shmid, 0, 0);

  if (new->SHMInfo.shmaddr < 0) {
    perror("shmat failed:");
    return NULL;
  }

  new->SHMInfo.readOnly = False;

  if (!XShmAttach(window->disp, &new->SHMInfo)) {
    fprintf(stderr, "XShmAttach failed\n");
    return NULL;;
  }

  XSync(window->disp, 0);
  XSetErrorHandler(old_handler);

  if (XJ_caught_error) {
    /* This path gets taken when shm is unavailable */
    /* fprintf(stderr, "Shared memory unavailable, using regular images\n"); */
    shmdt(new->SHMInfo.shmaddr);
    new->SHMInfo.shmaddr = 0;

    GST_IMAGE_DATA (new) = g_malloc (((window->depth + 7) / 8) * new->width * new->height);

    new->ximage = XCreateImage (window->disp, DefaultVisual (window->disp, window->screen_num), 
			  window->depth, ZPixmap, 0, 
			  GST_IMAGE_DATA (new), 
			  new->width, new->height, window->depth,
			  new->width * ((window->depth + 7) / 8));
    if(!new->ximage) {
      fprintf(stderr, "CreateImage Failed\n");
      return NULL;
    }
  }
  
  _gst_xwindow_ref (window);
  return new;
}


static void 
_gst_ximage_destroy (GstXImage *image)
{
  if (image->SHMInfo.shmaddr)
    XShmDetach(GST_IMAGE_WINDOW (image)->disp, &image->SHMInfo);
  if (image->ximage)
    XDestroyImage(image->ximage);
  if (image->SHMInfo.shmaddr)
    shmdt(image->SHMInfo.shmaddr);
  if (image->SHMInfo.shmid > 0)
    shmctl(image->SHMInfo.shmid, IPC_RMID, 0);
  _gst_xwindow_unref (GST_IMAGE_WINDOW (image));
  g_free (image);
}

   
static void
_gst_ximage_put (GstXWindow *window, GstXImage *image)
{
  XWindowAttributes attr; 
  int x, y;

  g_return_if_fail (window != NULL);
  g_return_if_fail (image != NULL);

  XGetWindowAttributes(window->disp, window->win, &attr); 

  x = MAX (0, (attr.width - image->width) / 2);
  y = MAX (0, (attr.height - image->height) / 2);
  if (image->SHMInfo.shmaddr) {
    XShmPutImage(window->disp, window->win, 
		 window->gc, image->ximage, 
		 0, 0, x, y, image->width, image->height, 
		 False);
  } else {
    XPutImage(window->disp, window->win, 
	      window->gc, image->ximage,  
	      0, 0, x, y, image->width, image->height);
  }
  XSync(window->disp, False);
}

void
_gst_ximage_add_formats (GstXWindow *window)
{
  GstImageFormat *image_format;
  GstXImage *ximage;

  g_return_if_fail(window);

  image_format = g_new0(GstImageFormat, 1);

  ximage = _gst_ximage_new (window, 100, 100);
  if(!ximage){
    g_warning("Could not create X image\n");
    return;
  }

  image_format->type = GST_TYPE_XIMAGE;
  image_format->caps = GST_CAPS_NEW("xvideosink_ximage_caps", "video/x-raw-rgb",
      "bpp", GST_PROPS_INT (GST_XIMAGE_BPP (ximage)),
      "depth",      GST_PROPS_INT (GST_XIMAGE_DEPTH (ximage)),
      "endianness", GST_PROPS_INT (GST_XIMAGE_ENDIANNESS (ximage)),
      "red_mask",   GST_PROPS_INT (GST_XIMAGE_RED_MASK (ximage)),
      "green_mask", GST_PROPS_INT (GST_XIMAGE_GREEN_MASK (ximage)),
      "blue_mask",  GST_PROPS_INT (GST_XIMAGE_BLUE_MASK (ximage)),
      "width",      GST_PROPS_INT_RANGE (0, G_MAXINT),
      "height",     GST_PROPS_INT_RANGE (0, G_MAXINT),
      "framerate",  GST_PROPS_FLOAT_RANGE (0, G_MAXFLOAT));

  _gst_image_destroy (GST_IMAGE (ximage));

  g_ptr_array_add(window->image_formats, image_format);
}

