/* GStreamer
 * Copyright (C) 2005 Tim-Philipp Müller <tim centricular net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "gstwavpackreader.h"

#include <gst/gst.h>

#include <string.h>

#define SAMPLES_PER_BUFFER  2048

#define GST_TYPE_WAVPACKDEC \
  (wavpackdec_get_type())
#define GST_WAVPACKDEC(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_WAVPACKDEC,GstWavpackDec))
#define GST_WAVPACKDEC_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_WAVPACKDEC,GstWavpackDec))
#define GST_IS_WAVPACKDEC(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_WAVPACKDEC))
#define GST_IS_WAVPACKDEC_CLASS(obj) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_WAVPACKDEC))


typedef struct _GstWavpackDec GstWavpackDec;
typedef struct _GstWavpackDecClass GstWavpackDecClass;

struct _GstWavpackDec
{
  GstElement element;

  /* pads */
  GstPad *sinkpad, *srcpad;

  /* state */
  stream_reader reader;
  WavpackContext *ctx;
  GstByteStream *bs;

  gboolean seek_pending;
  guint64 seek_offset;          /* in samples */

  gboolean need_discont;
  gboolean need_flush;
  guint64 offset;

  glong *decodebuf;

  gchar err_string[128];

  gint rate;
  gint channels;
  gint width;
  gint depth;
  gint total_samples;
};

struct _GstWavpackDecClass
{
  GstElementClass parent_class;
};

/* elementfactory information */
static GstElementDetails wavpackdec_details =
GST_ELEMENT_DETAILS ("wavpack decoder",
    "Codec/Decoder/Audio",
    "Decodes wavpack streams",
    "Tim-Philipp Müller <tim centricular net>");


GST_DEBUG_CATEGORY_STATIC (wavpackdec_debug);
#define GST_CAT_DEFAULT wavpackdec_debug

static GstStaticPadTemplate wavpackdec_src_template_factory =
GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("audio/x-raw-int, "
        "endianness = (int) " G_STRINGIFY (G_BYTE_ORDER) ", "
        "signed = (boolean) true, "
        "width = (int) { 8, 16, 32 }, "
        "depth = (int) { 8, 16, 24, 32 }, "
        "rate = (int) { 8000, 11025, 12000, 16000, 22050, 24000, 32000, 44100, 48000 }, "
        "channels = (int) [ 1, 2 ]")
    );

static GstStaticPadTemplate wavpackdec_sink_template_factory =
GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("audio/x-wavpack")
    );

static GType
wavpackdec_get_type (void)
    G_GNUC_CONST;
     static void wavpackdec_base_init (gpointer g_class);
     static void wavpackdec_class_init (GstWavpackDecClass * klass);
     static void wavpackdec_instance_init (GstWavpackDec * wavpackdec);

     static gboolean wavpackdec_src_event (GstPad * pad, GstEvent * event);
     static const GstEventMask *wavpackdec_get_event_masks (GstPad * pad);
     static const GstQueryType *wavpackdec_get_query_types (GstPad * pad);
     static gboolean wavpackdec_src_query (GstPad * pad, GstQueryType type,
    GstFormat * format, gint64 * value);
     static void wavpackdec_loop (GstElement * element);
     static GstElementStateReturn wavpackdec_change_state (GstElement *
    element);

     static GstElementClass *parent_class;      /* NULL */

/***************************************************************************
 *
 *   wavpackdec_get_type
 *
 ***************************************************************************/

     static GType wavpackdec_get_type (void)
{
  static GType type = 0;

  if (!type) {
    static const GTypeInfo info = {
      sizeof (GstWavpackDecClass),
      wavpackdec_base_init,
      NULL,
      (GClassInitFunc) wavpackdec_class_init,
      NULL,
      NULL,
      sizeof (GstWavpackDec),
      0,
      (GInstanceInitFunc) wavpackdec_instance_init,
    };

    type = g_type_register_static (GST_TYPE_ELEMENT, "GstWavpackDec", &info, 0);
  }
  GST_DEBUG_CATEGORY_INIT (wavpackdec_debug, "wavpackdec", 0,
      "wavpack decoding");
  return type;
}

/***************************************************************************
 *
 *   wavpackdec_base_init
 *
 ***************************************************************************/

static void
wavpackdec_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&wavpackdec_sink_template_factory));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&wavpackdec_src_template_factory));
  gst_element_class_set_details (element_class, &wavpackdec_details);
}

/***************************************************************************
 *
 *   wavpackdec_class_init
 *
 ***************************************************************************/

static void
wavpackdec_class_init (GstWavpackDecClass * klass)
{
  GstElementClass *gstelement_class;

  gstelement_class = (GstElementClass *) klass;

  parent_class = g_type_class_peek_parent (klass);

  gstelement_class->change_state = wavpackdec_change_state;
}

/***************************************************************************
 *
 *   wavpackdec_instance_init
 *
 ***************************************************************************/

static void
wavpackdec_instance_init (GstWavpackDec * wavpackdec)
{
  /* GST_FLAG_SET (wavpackdec, GST_ELEMENT_EVENT_AWARE); */

  /* create the sink and src pads */
  wavpackdec->sinkpad =
      gst_pad_new_from_template (gst_static_pad_template_get
      (&wavpackdec_sink_template_factory), "sink");
  gst_element_add_pad (GST_ELEMENT (wavpackdec), wavpackdec->sinkpad);

  wavpackdec->srcpad =
      gst_pad_new_from_template (gst_static_pad_template_get
      (&wavpackdec_src_template_factory), "src");
  gst_element_add_pad (GST_ELEMENT (wavpackdec), wavpackdec->srcpad);
  gst_pad_set_event_function (wavpackdec->srcpad,
      GST_DEBUG_FUNCPTR (wavpackdec_src_event));
  gst_pad_set_event_mask_function (wavpackdec->srcpad,
      GST_DEBUG_FUNCPTR (wavpackdec_get_event_masks));
  gst_pad_set_query_function (wavpackdec->srcpad,
      GST_DEBUG_FUNCPTR (wavpackdec_src_query));
  gst_pad_set_query_type_function (wavpackdec->srcpad,
      GST_DEBUG_FUNCPTR (wavpackdec_get_query_types));
  gst_pad_use_explicit_caps (wavpackdec->srcpad);

  gst_element_set_loop_function (GST_ELEMENT (wavpackdec),
      GST_DEBUG_FUNCPTR (wavpackdec_loop));

  wavpackdec->ctx = NULL;

  GST_FLAG_SET (wavpackdec, GST_ELEMENT_EVENT_AWARE);
}

/***************************************************************************
 *
 *   wavpackdec_get_query_types
 *
 ***************************************************************************/

static const GstQueryType *
wavpackdec_get_query_types (GstPad * pad)
{
  static const GstQueryType wavpackdec_src_query_types[] = {
    GST_QUERY_TOTAL,
    GST_QUERY_POSITION,
    0
  };

  return wavpackdec_src_query_types;
}

/***************************************************************************
 *
 *   wavpackdec_src_query
 *
 ***************************************************************************/

static gboolean
wavpackdec_src_query (GstPad * pad, GstQueryType type,
    GstFormat * format, gint64 * value)
{
  GstWavpackDec *wavpackdec = GST_WAVPACKDEC (gst_pad_get_parent (pad));
  gint64 val;

  if (wavpackdec->ctx == NULL)
    return FALSE;

  if (type == GST_QUERY_TOTAL)
    val = wavpackdec->total_samples;
  else if (type == GST_QUERY_POSITION)
    val = wavpackdec->offset;
  else
    return FALSE;

  switch (*format) {
    case GST_FORMAT_DEFAULT:
      *value = val;
      break;
    case GST_FORMAT_BYTES:
      /* *value = val * (wavpackdec->width/8); *//* or depth? */
      return FALSE;             /* can be lossy or lossless, just leave it for now */
    case GST_FORMAT_TIME:
      *value = val * GST_SECOND / wavpackdec->rate;
      break;
    default:
      return FALSE;
  }

  return TRUE;
}


/***************************************************************************
 *
 *   wavpackdec_src_event
 *
 ***************************************************************************/

static const GstEventMask *
wavpackdec_get_event_masks (GstPad * pad)
{
  static const GstEventMask wavpackdec_src_event_masks[] = {
    {GST_EVENT_SEEK, GST_SEEK_METHOD_SET | GST_SEEK_FLAG_FLUSH},
    {0,}
  };

  return wavpackdec_src_event_masks;
}

/***************************************************************************
 *
 *   wavpackdec_src_event
 *
 ***************************************************************************/

static gboolean
wavpackdec_src_event (GstPad * pad, GstEvent * event)
{
  GstWavpackDec *wavpackdec;
  GstSeekType method;
  GstFormat format;
  gboolean need_flush;
  gint64 offset, dest;

  wavpackdec = GST_WAVPACKDEC (gst_pad_get_parent (pad));

  if (GST_EVENT_TYPE (event) != GST_EVENT_SEEK) {
    GST_DEBUG ("forwarding event of type %d to sink pad",
        GST_EVENT_TYPE (event));
    return gst_pad_send_event (GST_PAD_PEER (wavpackdec->sinkpad), event);
  }

  format = GST_EVENT_SEEK_FORMAT (event);
  offset = GST_EVENT_SEEK_OFFSET (event);
  method = GST_EVENT_SEEK_METHOD (event);
  need_flush = ((GST_EVENT_SEEK_FLAGS (event) & GST_SEEK_FLAG_FLUSH));

  gst_data_unref (GST_DATA (event));
  event = NULL;

  if (offset < 0 || method != GST_SEEK_METHOD_SET)
    return FALSE;

  if (wavpackdec->ctx == NULL)
    return FALSE;

  /* we can only seek to samples, so we convert
   * the requested offsets to samples first */
  if (format == GST_FORMAT_TIME) {
    dest = offset * wavpackdec->rate / GST_SECOND;
  } else if (format == GST_FORMAT_BYTES) {
    /* dest = offset / (wavpackdec->width/8); *//* or depth? */
    return FALSE;               /* can be lossy or lossless, just leave it for now */
  } else if (format == GST_FORMAT_DEFAULT) {
    dest = offset;
  } else {
    return FALSE;
  }

  if (dest < 0 || dest >= wavpackdec->total_samples) {
    GST_DEBUG ("seek out of range (dest=%u, total_samples=%d)",
        (guint) dest, wavpackdec->total_samples);
    return FALSE;
  }

  wavpackdec->need_discont = TRUE;
  wavpackdec->need_flush = need_flush;

  wavpackdec->seek_pending = TRUE;
  wavpackdec->seek_offset = dest;

  return TRUE;
}

/***************************************************************************
 *
 *   wavpackdec_init_stream
 *
 ***************************************************************************/

static gboolean
wavpackdec_init_stream (GstWavpackDec * wavpackdec)
{
  GstCaps *caps;

  wavpackdec->err_string[0] = 0x00;

  wavpackdec->ctx = WavpackOpenFileInputEx (&wavpackdec->reader,
      wavpackdec->bs, NULL, wavpackdec->err_string,
      OPEN_2CH_MAX | OPEN_NORMALIZE, 0);

  if (wavpackdec->err_string[0]) {
    GST_ELEMENT_ERROR (wavpackdec, STREAM, WRONG_TYPE,
        ("%s", wavpackdec->err_string), (NULL));
    return FALSE;
  }

  wavpackdec->depth = WavpackGetBytesPerSample (wavpackdec->ctx) * 8;
  wavpackdec->width = (wavpackdec->depth == 24) ? 32 : wavpackdec->depth;
  wavpackdec->rate = WavpackGetSampleRate (wavpackdec->ctx);
  wavpackdec->channels = WavpackGetNumChannels (wavpackdec->ctx);
  wavpackdec->total_samples = WavpackGetNumSamples (wavpackdec->ctx);
  caps = gst_caps_new_simple ("audio/x-raw-int",
      "signed", G_TYPE_BOOLEAN, TRUE,
      "endianness", G_TYPE_INT, G_BYTE_ORDER,
      "channels", G_TYPE_INT, wavpackdec->channels,
      "rate", G_TYPE_INT, wavpackdec->rate,
      "width", G_TYPE_INT, wavpackdec->width,
      "depth", G_TYPE_INT, wavpackdec->depth, NULL);

  if (!gst_pad_set_explicit_caps (wavpackdec->srcpad, caps)) {
    GST_ELEMENT_ERROR (wavpackdec, CORE, NEGOTIATION, (NULL), (NULL));
    gst_caps_free (caps);
    return FALSE;
  }

  wavpackdec->decodebuf =
      g_new0 (long, SAMPLES_PER_BUFFER * wavpackdec->channels);

  wavpackdec->offset = 0;
  wavpackdec->need_discont = TRUE;
  wavpackdec->need_flush = FALSE;
  wavpackdec->seek_pending = FALSE;

  gst_caps_free (caps);
  return TRUE;
}

/***************************************************************************
 *
 *   wavpackdec_repack_samples
 *
 ***************************************************************************/

static inline GstBuffer *
wavpackdec_repack_samples (GstWavpackDec * wavpackdec, long *samples,
    guint num_samples)
{
  GstBuffer *buf;
  gint i;

  buf =
      gst_buffer_new_and_alloc (num_samples * wavpackdec->width / 8 *
      wavpackdec->channels);

  switch (wavpackdec->width) {
    case 8:
    {
      guint8 *d = (guint8 *) GST_BUFFER_DATA (buf);

      for (i = 0; i < num_samples * wavpackdec->channels; ++i)
        *(d++) = (guint8) * (samples++);
    }
      break;

    case 16:
    {
      guint16 *d = (guint16 *) GST_BUFFER_DATA (buf);

      for (i = 0; i < num_samples * wavpackdec->channels; ++i)
        *(d++) = (guint16) * (samples++);
    }
      break;

    case 32:
    {
      guint32 *d = (guint32 *) GST_BUFFER_DATA (buf);

      for (i = 0; i < num_samples * wavpackdec->channels; ++i)
        *(d++) = (guint32) * (samples++);
    }
      break;

    default:
      g_warning ("wavpackdec->width = %u", wavpackdec->width);
      g_return_val_if_reached (NULL);
  }

  return buf;
}

/***************************************************************************
 *
 *   wavpackdec_loop
 *
 ***************************************************************************/

static void
wavpackdec_loop (GstElement * element)
{
  GstWavpackDec *wavpackdec;
  GstBuffer *buf;
  ulong ret;

  wavpackdec = GST_WAVPACKDEC (element);

  if (wavpackdec->ctx == NULL) {
    if (!wavpackdec_init_stream (wavpackdec))
      return;
  }

  if (wavpackdec->seek_pending) {
    wavpackdec->seek_pending = FALSE;
    if (WavpackSeekSample (wavpackdec->ctx, (ulong) wavpackdec->seek_offset)) {
      if (wavpackdec->need_flush) {
        GstEvent *flush_event = gst_event_new (GST_EVENT_FLUSH);

        gst_pad_push (wavpackdec->srcpad, GST_DATA (flush_event));
        wavpackdec->need_flush = FALSE;
      }
      wavpackdec->need_discont = TRUE;
      wavpackdec->offset = wavpackdec->seek_offset;
    } else {
      GST_DEBUG ("seek to sample %" G_GUINT64_FORMAT "/%d failed",
          wavpackdec->seek_offset, wavpackdec->total_samples);
    }
  }

  if (wavpackdec->need_discont) {
    guint64 timestamp = wavpackdec->offset * GST_SECOND / wavpackdec->rate;

    gst_pad_push (wavpackdec->srcpad,
        GST_DATA (gst_event_new_discontinuous (FALSE,
                GST_FORMAT_TIME, timestamp,
                GST_FORMAT_DEFAULT, wavpackdec->offset, GST_FORMAT_UNDEFINED)));
    wavpackdec->need_discont = FALSE;
  }

  ret =
      WavpackUnpackSamples (wavpackdec->ctx, wavpackdec->decodebuf,
      SAMPLES_PER_BUFFER);

  if (ret <= 0) {
    if (ret == 0) {
      gst_element_set_eos (element);
      gst_pad_push (wavpackdec->srcpad,
          GST_DATA (gst_event_new (GST_EVENT_EOS)));
    } else {
      GST_ERROR_OBJECT (wavpackdec, "Failed to decode samples: %s",
          wavpackdec->err_string);
    }
    return;
  }

  buf =
      wavpackdec_repack_samples (wavpackdec, wavpackdec->decodebuf,
      (guint) ret);
  g_return_if_fail (buf != NULL);

  GST_BUFFER_OFFSET (buf) = wavpackdec->offset;
  GST_BUFFER_DURATION (buf) = ret * GST_SECOND / wavpackdec->rate;
  GST_BUFFER_TIMESTAMP (buf) =
      wavpackdec->offset * GST_SECOND / wavpackdec->rate;

  gst_pad_push (wavpackdec->srcpad, GST_DATA (buf));

  wavpackdec->offset += ret;
}

static GstElementStateReturn
wavpackdec_change_state (GstElement * element)
{
  GstWavpackDec *wavpackdec;

  wavpackdec = GST_WAVPACKDEC (element);

  switch (GST_STATE_TRANSITION (element)) {
    case GST_STATE_NULL_TO_READY:
      wavpackdec->bs = gst_bytestream_new (wavpackdec->sinkpad);
      gst_wavpack_reader_init (&wavpackdec->reader);
      break;
    case GST_STATE_READY_TO_PAUSED:
      wavpackdec->need_discont = TRUE;
      break;
    case GST_STATE_PAUSED_TO_PLAYING:
      break;
    case GST_STATE_PLAYING_TO_PAUSED:
      break;
    case GST_STATE_PAUSED_TO_READY:
      wavpackdec->seek_pending = FALSE;
      break;
    case GST_STATE_READY_TO_NULL:
      WavpackCloseFile (wavpackdec->ctx);
      gst_bytestream_destroy (wavpackdec->bs);
      g_free (wavpackdec->decodebuf);
      wavpackdec->decodebuf = NULL;
      wavpackdec->ctx = NULL;
      wavpackdec->bs = NULL;
      wavpackdec->depth = 0;
      wavpackdec->width = 0;
      wavpackdec->rate = 0;
      wavpackdec->channels = 0;
      wavpackdec->total_samples = 0;
      wavpackdec->offset = 0;
      break;
  }

  return parent_class->change_state (element);
}

/***************************************************************************
 *
 *   PLUGIN INITIALIZATION
 *
 ***************************************************************************/

static gboolean
plugin_init (GstPlugin * plugin)
{
  if (!gst_library_load ("gstbytestream")
      || !gst_element_register (plugin, "wavpackdec", GST_RANK_PRIMARY,
          wavpackdec_get_type ()))
    return FALSE;

  return TRUE;
}

GST_PLUGIN_DEFINE
    (GST_VERSION_MAJOR,
    GST_VERSION_MINOR,
    "wavpack",
    "wavpack decoding and encoding",
    plugin_init, VERSION, "LGPL", GST_PACKAGE, GST_ORIGIN)
