/* GStreamer
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "ximagesrc.h"

#include <string.h>
#include <stdlib.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>

GST_DEBUG_CATEGORY (ximagesrc_debug);
#define GST_CAT_DEFAULT ximagesrc_debug

/* elementfactory information */
static GstElementDetails ximagesrc_details =
GST_ELEMENT_DETAILS ("Ximage video source",
    "Source/Video",
    "Creates a screenshot video stream",
    "Lutz Mueller <lutz@users.sourceforge.net>");

static GstStaticPadTemplate gst_ximagesrc_src_template_factory =
GST_STATIC_PAD_TEMPLATE ("src", GST_PAD_SRC, GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("video/x-raw-rgb, "
        "framerate = (double) [ 1.0, 100.0 ], "
        "width = (int) [ 1, MAX ], " "height = (int) [ 1, MAX ]"));

#define DEFAULT_PROP_SYNC TRUE

enum
{
  PROP_0,
  PROP_DISPLAY_NAME,
  PROP_SCREEN_NUM,
  PROP_SYNC
};

static void gst_ximagesrc_set_clock (GstElement * element, GstClock * clock);

static GstElementClass *parent_class = NULL;

struct _GstXimagesrc
{
  GstElement element;

  /* Information on display */
  Display *display;
  guint screen_num;
  gint endianness;
  gint red_mask, green_mask, blue_mask;
  gint bpp;

  /* Information on window */
  Window window;
  gint depth;
  gint width, height;

  /* State */
  gulong frame;
  gdouble framerate;

  GstClock *clock;
  gboolean sync;
};

struct _GstXimagesrcClass
{
  GstElementClass parent_class;
};

static void
gst_ximagesrc_recalc (GstXimagesrc * src)
{
  XPixmapFormatValues *px_formats = NULL;
  gint nb_formats = 0, i;
  XWindowAttributes attr;

  g_return_if_fail (GST_IS_XIMAGESRC (src));

  if (!src->display)
    return;

  /* We could use XQueryPointer to get only the current window. */

  /* Width, height, depth */
  src->window = RootWindow (src->display, src->screen_num);
  XGetWindowAttributes (src->display, src->window, &attr);
  src->width = attr.width;
  src->height = attr.height;
  src->depth = attr.depth;

  /* Masks, bpp */
  src->red_mask = attr.visual->red_mask;
  src->green_mask = attr.visual->green_mask;
  src->blue_mask = attr.visual->blue_mask;
  px_formats = XListPixmapFormats (src->display, &nb_formats);
  if (!px_formats)
    return;
  for (i = 0; i < nb_formats; i++)
    if (px_formats[i].depth == src->depth)
      src->bpp = px_formats[i].bits_per_pixel;
  src->endianness = (ImageByteOrder (src->display) == LSBFirst) ?
      G_LITTLE_ENDIAN : G_BIG_ENDIAN;
  if ((src->bpp == 24 || src->bpp == 32) && src->endianness == G_LITTLE_ENDIAN) {
    src->endianness = G_BIG_ENDIAN;
    src->red_mask = GUINT32_TO_BE (src->red_mask);
    src->green_mask = GUINT32_TO_BE (src->green_mask);
    src->blue_mask = GUINT32_TO_BE (src->blue_mask);
    if (src->bpp == 24) {
      src->red_mask >>= 8;
      src->green_mask >>= 8;
      src->blue_mask >>= 8;
    }
  }
}

/*********************************
 * Everyting relating to the pad *
 *********************************/

static GstCaps *
gst_ximagesrc_getcaps (GstPad * pad)
{
  GstXimagesrc *s = GST_XIMAGESRC (gst_pad_get_parent (pad));

  gst_ximagesrc_recalc (s);

  if (s->display)
    return gst_caps_new_simple ("video/x-raw-rgb",
        "bpp", G_TYPE_INT, s->bpp,
        "endianess", G_TYPE_INT, s->endianness,
        "red_mask", G_TYPE_INT, s->red_mask,
        "green_mask", G_TYPE_INT, s->green_mask,
        "blue_mask", G_TYPE_INT, s->blue_mask,
        "width", G_TYPE_INT, s->width,
        "height", G_TYPE_INT, s->height,
        "framerate", GST_TYPE_DOUBLE_RANGE, 1., 100., NULL);
  else
    return gst_caps_new_empty ();
}

static GstPadLinkReturn
gst_ximagesrc_src_link (GstPad * pad, const GstCaps * caps)
{
  GstXimagesrc *src;
  const GstStructure *structure;
  GstPadLinkReturn ret;

  src = GST_XIMAGESRC (gst_pad_get_parent (pad));

  structure = gst_caps_get_structure (caps, 0);

  ret = gst_structure_get_double (structure, "framerate", &src->framerate);

  if ((!ret) || (src->framerate < 1))
    return GST_PAD_LINK_REFUSED;

  return GST_PAD_LINK_OK;
}

static void
gst_ximagesrc_free_data_func (GstBuffer * buf)
{
  XDestroyImage ((XImage *) GST_BUFFER_PRIVATE (buf));
}

static GstData *
gst_ximagesrc_get (GstPad * pad)
{
  GstXimagesrc *s = GST_XIMAGESRC (gst_pad_get_parent (pad));
  GstBuffer *buf;
  XImage *image;

  gst_ximagesrc_recalc (s);

  image = XGetImage (s->display, s->window, 0, 0, s->width, s->height,
      AllPlanes, ZPixmap);

  buf = gst_buffer_new ();
  GST_BUFFER_FLAG_SET (buf, GST_BUFFER_DONTFREE);
  GST_BUFFER_FREE_DATA_FUNC (buf) = gst_ximagesrc_free_data_func;
  GST_BUFFER_PRIVATE (buf) = image;
  GST_BUFFER_DURATION (buf) = GST_SECOND / s->framerate;
  GST_BUFFER_TIMESTAMP (buf) = (gdouble) GST_SECOND *s->frame / s->framerate;

  GST_BUFFER_DATA (buf) = (guint8 *) image->data;
  GST_BUFFER_SIZE (buf) = s->width * s->height * 3;

  if ((s->sync) && (s->clock))
    gst_element_wait (GST_ELEMENT (s), GST_BUFFER_TIMESTAMP (buf));

  s->frame++;

  return GST_DATA (buf);
}

/**************************************
 * Everything relating to the element *
 **************************************/

static GstElementStateReturn
gst_ximagesrc_change_state (GstElement * element)
{
  GstXimagesrc *s = GST_XIMAGESRC (element);

  switch (GST_STATE_TRANSITION (element)) {
    case GST_STATE_PAUSED_TO_READY:
      s->frame = 0;
      break;
    default:
      break;
  }

  if (GST_ELEMENT_CLASS (parent_class)->change_state)
    return GST_ELEMENT_CLASS (parent_class)->change_state (element);

  return GST_STATE_SUCCESS;
}

static void
gst_ximagesrc_init (GstXimagesrc * s)
{
  GstPad *pad;

  pad =
      gst_pad_new_from_template (gst_static_pad_template_get
      (&gst_ximagesrc_src_template_factory), "src");
  gst_pad_set_getcaps_function (pad, gst_ximagesrc_getcaps);
  gst_pad_set_link_function (pad, gst_ximagesrc_src_link);
  gst_pad_set_get_function (pad, gst_ximagesrc_get);
  gst_element_add_pad (GST_ELEMENT (s), pad);

  s->display = XOpenDisplay (NULL);
  s->framerate = 10.;
  s->clock = NULL;
  s->sync = DEFAULT_PROP_SYNC;
}

static void
gst_ximagesrc_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstXimagesrc *src = GST_XIMAGESRC (object);

  switch (prop_id) {
    case PROP_DISPLAY_NAME:
      if (src->display)
        XCloseDisplay (src->display);
      src->display = XOpenDisplay (g_value_get_string (value));
      if (!src->display)
        break;
      src->screen_num = MIN (src->screen_num, ScreenCount (src->display) - 1);
      break;
    case PROP_SCREEN_NUM:
      src->screen_num = g_value_get_uint (value);
      src->screen_num = MIN (src->screen_num, ScreenCount (src->display) - 1);
      break;
    case PROP_SYNC:
      src->sync = g_value_get_boolean (value);
      break;

    default:
      break;
  }
}

static void
gst_ximagesrc_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstXimagesrc *src = GST_XIMAGESRC (object);

  switch (prop_id) {
    case PROP_DISPLAY_NAME:
      g_value_set_string (value, DisplayString (src->display));
      break;
    case PROP_SCREEN_NUM:
      g_value_set_uint (value, src->screen_num);
      break;
    case PROP_SYNC:
      g_value_set_boolean (value, src->sync);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_ximagesrc_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  element_class->change_state = gst_ximagesrc_change_state;

  gst_element_class_set_details (element_class, &ximagesrc_details);

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_ximagesrc_src_template_factory));
}

static void
gst_ximagesrc_finalize (GObject * object)
{
  GstXimagesrc *src = GST_XIMAGESRC (object);

  if (src->display)
    XCloseDisplay (src->display);
  src->display = NULL;

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gst_ximagesrc_class_init (GstXimagesrcClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;

  g_object_class_install_property (G_OBJECT_CLASS (klass), PROP_DISPLAY_NAME,
      g_param_spec_string ("display_name", "Display", "X Display name", NULL,
          G_PARAM_READWRITE));
  g_object_class_install_property (G_OBJECT_CLASS (klass), PROP_SCREEN_NUM,
      g_param_spec_uint ("screen_num", "Screen number", "X Screen number",
          0, G_MAXINT, 0, G_PARAM_READWRITE));
  g_object_class_install_property (G_OBJECT_CLASS (klass), PROP_SYNC,
      g_param_spec_boolean ("sync", "Sync", "Synchronize to clock",
          DEFAULT_PROP_SYNC, G_PARAM_READWRITE));

  parent_class = g_type_class_ref (GST_TYPE_ELEMENT);

  gobject_class->set_property = gst_ximagesrc_set_property;
  gobject_class->get_property = gst_ximagesrc_get_property;
  gobject_class->finalize = gst_ximagesrc_finalize;

  gstelement_class->set_clock = gst_ximagesrc_set_clock;

  GST_DEBUG_CATEGORY_INIT (ximagesrc_debug, "ximagesrc", 0,
      "Video Test Source");
}

static void
gst_ximagesrc_set_clock (GstElement * element, GstClock * clock)
{
  GstXimagesrc *x;

  x = GST_XIMAGESRC (element);

  gst_object_replace ((GstObject **) & x->clock, (GstObject *) clock);
}


GType
gst_ximagesrc_get_type (void)
{
  static GType ximagesrc_type = 0;

  if (!ximagesrc_type) {
    static const GTypeInfo ximagesrc_info = {
      sizeof (GstXimagesrcClass),
      gst_ximagesrc_base_init,
      NULL,
      (GClassInitFunc) gst_ximagesrc_class_init,
      NULL,
      NULL,
      sizeof (GstXimagesrc),
      0,
      (GInstanceInitFunc) gst_ximagesrc_init,
    };

    ximagesrc_type =
        g_type_register_static (GST_TYPE_ELEMENT, "GstXimagesrc",
        &ximagesrc_info, 0);
  }
  return ximagesrc_type;
}
