/* The Cairo CSS Drawing Library.
 * Copyright (C) 2008 Robert Staudinger
 *
 * This  library is free  software; you can  redistribute it and/or
 * modify it  under  the terms  of the  GNU Lesser  General  Public
 * License  as published  by the Free  Software  Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed  in the hope that it will be useful,
 * but  WITHOUT ANY WARRANTY; without even  the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License  along  with  this library;  if not,  write to  the Free
 * Software Foundation, Inc., 51  Franklin St, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 */

#ifndef CCD_BORDER_H
#define CCD_BORDER_H

#include <stdbool.h>
#include <stdint.h>
#include <cairo.h>
#include <glib.h>
#include <ccd/ccd-features.h>
#include <ccd/ccd-color.h>
#include <ccd/ccd-property.h>

G_BEGIN_DECLS

typedef enum {
	CCD_BORDER_STYLE_HIDDEN,
	CCD_BORDER_STYLE_DOTTED,
	CCD_BORDER_STYLE_DASHED,
	CCD_BORDER_STYLE_SOLID,
	CCD_BORDER_STYLE_DOUBLE,
	CCD_BORDER_STYLE_GROOVE,
	CCD_BORDER_STYLE_RIDGE,
	CCD_BORDER_STYLE_INSET,
	CCD_BORDER_STYLE_OUTSET
} ccd_border_style_t;

/*
 * Remember which properties were set explicitely, e.g. using "border-top", 
 * so it's not overwritten by a subsequent "border" property.
 */
typedef enum {
	CCD_BORDER_FLAGS_COMMON_WIDTH	= 1 << 0,
	CCD_BORDER_FLAGS_SPECIFIC_WIDTH = 1 << 1,
	CCD_BORDER_FLAGS_WIDTH_MASK	= CCD_BORDER_FLAGS_COMMON_WIDTH | CCD_BORDER_FLAGS_SPECIFIC_WIDTH,

	CCD_BORDER_FLAGS_COMMON_STYLE	= 1 << 2,
	CCD_BORDER_FLAGS_SPECIFIC_STYLE = 1 << 3,
	CCD_BORDER_FLAGS_STYLE_MASK	= CCD_BORDER_FLAGS_COMMON_STYLE | CCD_BORDER_FLAGS_SPECIFIC_STYLE,

	CCD_BORDER_FLAGS_COMMON_COLOR	= 1 << 4,
	CCD_BORDER_FLAGS_SPECIFIC_COLOR = 1 << 5,
	CCD_BORDER_FLAGS_COLOR_MASK	= CCD_BORDER_FLAGS_COMMON_COLOR | CCD_BORDER_FLAGS_SPECIFIC_COLOR
} ccd_border_flags_t;

#define CCD_BORDER_STROKE_ASSIGN(lhs_, rhs_) {		\
	lhs_.width	= rhs_.width;			\
	lhs_.width_spec	= rhs_.width_spec;		\
	lhs_.style	= rhs_.style;			\
	lhs_.style_spec	= rhs_.style_spec;		\
	lhs_.color	= rhs_.color;			\
	lhs_.color_spec	= rhs_.color_spec;		\
	lhs_.flags	= rhs_.flags;			\
}

typedef struct {
	double			width;
	ccd_property_spec_t	width_spec;
	ccd_border_style_t	style;
	ccd_property_spec_t	style_spec;
	ccd_color_t		color;
	ccd_property_spec_t	color_spec;
	int			flags;
} ccd_border_stroke_t;

typedef struct {
	ccd_border_stroke_t	left;
	ccd_border_stroke_t	top;
	ccd_border_stroke_t	right;
	ccd_border_stroke_t	bottom;
} ccd_border_t;

ccd_border_t *	ccd_border_new	(void);
void		ccd_border_free	(ccd_border_t *self);

bool ccd_border_parse (ccd_border_t *self, char const *property, 
				CRTerm const *values);

void ccd_border_draw (ccd_border_stroke_t const *left, 
		      ccd_border_stroke_t const *top, 
		      ccd_border_stroke_t const *right,
		      ccd_border_stroke_t const *bottom, 
		      cairo_t *cr, double x, double y, double width, double height);

void ccd_border_dump		(ccd_border_t const *self);
void ccd_border_stroke_dump	(ccd_border_stroke_t const *self);

G_END_DECLS

#endif /* CCD_BORDER_H */

