/*
 *  Copyright (C) 2005 Robert Staudinger
 *
 *  This software is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Library General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <glib/gprintf.h>

#include <gtkmozembed.h>
#include <gtkmozembed_internal.h>
#include <gtkmozedit/gtk-moz-edit-private.h>
#include <gtkmozedit/gme-web-browser-private.h>

#include "nsIWidget.h"
#include "nsIWebNavigation.h"
#include "nsWidgetsCID.h"
#include "nsRect.h"
#include "mozilla-config.h"
#include "nsString.h"
#include "nsCOMPtr.h"
#include "nsIEventQueueService.h"
#include "nsIInputStream.h"
#include "nsIAtom.h"
#include "nsIPref.h"
#include "nsISupports.h"
#include "nsIBaseWindow.h"
#include "nsIWebBrowser.h"
#include "nsIWebBrowserPersist.h"
#include "nsIWebBrowserChrome.h"
#include "nsIEmbeddingSiteWindow.h"
#include "nsEmbedAPI.h"
#include "nsIWindowCreator.h"
#include "nsIWindowWatcher.h"
#include "nsCWebBrowser.h"
#include "nsIWebProgressListener.h"
#include "nsIContextMenuListener2.h"
#include "nsIInterfaceRequestor.h"
#include "nsIWebBrowserSetup.h"
#include "nsIDocShellTreeItem.h"
#include "nsIWebBrowserChromeFocus.h"
#include "nsIInterfaceRequestor.h"
#include "nsIInterfaceRequestorUtils.h"
#include "nsReadableUtils.h"
#include "nsIWeakReference.h"
#include "nsIWeakReferenceUtils.h"
#include "nsWeakReference.h"
#include "nsISelection.h"
#include "nsIURI.h"
#include "nsIClipboardCommands.h"
#include "nsIWebBrowserFind.h"
#include "nsIDOMWindow.h"
#include "nsIDOMEventListener.h"
#include "nsPIDOMWindow.h"
#include "nsIChromeEventHandler.h"
#include "nsIDOMWindowInternal.h"
#include "nsIDocShell.h"
#include "nsIDocument.h"
#include "nsIDOMDocument.h"
#include "nsIDocumentEncoder.h"
#include "nsIStringStream.h"
#include "nsIAppShell.h"
#include "nsIEditorSpellCheck.h"
#include "nsIURIContentListener.h"
#include "nsIEditingSession.h"
#include "nsICommandManager.h"

static GtkMozEmbedClass *gtk_moz_embed_parent_class = NULL;

void
realize_cb (GtkMozEdit *self,
	    gpointer data)
{
	nsCOMPtr<nsIDOMWindow> dom_window;
	nsCOMPtr<nsIEditingSession> edit_session;
	nsCOMPtr<nsICommandManager> cmd_mgr; 
	nsresult rv;

	g_printf ("%s() pending: '%s'\n", __FUNCTION__, self->pending_url);

	gtk_moz_embed_get_nsIWebBrowser (GTK_MOZ_EMBED (self), getter_AddRefs (self->browser));
	self->browser->GetContentDOMWindow (getter_AddRefs(dom_window));
	g_assert (dom_window);

	edit_session = do_GetInterface (self->browser, &rv);
	g_assert (edit_session);

	cmd_mgr = do_GetInterface (self->browser, &rv);
	/* NOTE: For Moz 1.1 and below, the middle parameter was not used */
	edit_session->MakeWindowEditable (dom_window, "html", PR_TRUE);

	gtk_moz_embed_load_url (GTK_MOZ_EMBED (self), self->pending_url);
}

static void
instance_init (GtkMozEdit *self)
{
	g_printf ("%s()\n", __FUNCTION__);

	self->pending_url = g_strdup ("about:blank");

	g_signal_connect (G_OBJECT (self), "realize", G_CALLBACK (realize_cb), NULL);
}

static void
instance_destroy (GtkObject *instance)
{
	GtkMozEdit *self = GTK_MOZ_EDIT (instance);

	g_printf ("%s()\n", __FUNCTION__);

	if (self->pending_url) {
		g_free (self->pending_url);
		self->pending_url = NULL;
	}

	if (self->browser) {		
		self->browser = NULL;
	}
}

static void
class_init (GtkMozEditClass *klass)
{
	GtkObjectClass *object_class = NULL;

	gtk_moz_embed_parent_class = (GtkMozEmbedClass*) gtk_type_class (GTK_TYPE_MOZ_EMBED);

	object_class = GTK_OBJECT_CLASS (klass);
	object_class->destroy = instance_destroy;
}

GType
gtk_moz_edit_get_gtype (void)
{
        static GType type = 0;
        if (!type) {
                static const GTypeInfo info = {
                        sizeof (GtkMozEditClass),
                        NULL,           /* base_init */
                        NULL,           /* base_finalize */
                        (GClassInitFunc) class_init,
                        NULL,           /* class_finalize */
                        NULL,           /* class_data */
                        sizeof (GtkMozEdit),
                        0,              /* n_preallocs */
                        (GInstanceInitFunc) instance_init,
                };
                type = g_type_register_static (GTK_TYPE_MOZ_EMBED, "GtkMozEdit", &info, (GTypeFlags)0);
        }
        return type;
}

GtkWidget *
gtk_moz_edit_new ()
{
	return GTK_WIDGET (g_object_new (GTK_TYPE_MOZ_EDIT, NULL));
}

gboolean 
gtk_moz_edit_command (GtkMozEdit *self,
		      const gchar *cmd, 
		      const gchar *val)
{
	nsCOMPtr<nsICommandParams> params;
	nsCOMPtr<nsIDOMWindow> dom_window;
	nsCOMPtr<nsICommandManager> cmd_mgr;
	nsresult rv;

	g_return_val_if_fail (self->browser, FALSE);

	params = do_CreateInstance (NS_COMMAND_PARAMS_CONTRACTID, &rv);
	g_return_val_if_fail (params, FALSE);

	//if the attribute is blank, or not necessary, this doesn't cause any harm
	if (strcmp (cmd, "cmd_insertHTML") == 0) {
		params->SetCStringValue ("state_data", val);
	}
	params->SetCStringValue ("state_attribute", val);

	self->browser->GetContentDOMWindow (getter_AddRefs (dom_window));
	g_return_val_if_fail (dom_window, FALSE);

	cmd_mgr = do_GetInterface (self->browser, &rv);
	g_return_val_if_fail (cmd_mgr, FALSE);

	rv = cmd_mgr->DoCommand (cmd, params, dom_window);
	if (NS_FAILED (rv)) {
		g_printf ("Unable to execute editing command.\n");
	}

	params = nsnull;
}

void
gtk_moz_edit_load (GtkMozEdit *self,
		   const gchar *url)
{
	g_return_if_fail (self && url);

	if (GTK_WIDGET_REALIZED (self)) {
		gtk_moz_embed_load_url (GTK_MOZ_EMBED (self), url);
	}
	else {
		self->pending_url = g_strdup (url);
	}
}

gboolean 
gtk_moz_edit_save (GtkMozEdit *self,
		   const gchar *filename, 
		   gboolean save_files)
{
	//Works, but has troubles with frames pages

	gchar *folder = g_strdup_printf ("%s-files", filename);
	nsCOMPtr<nsILocalFile> file;
	nsCOMPtr<nsILocalFile> data;
	PRUint32 flags;

	// Save the file
	nsCOMPtr<nsIWebBrowserPersist> persist (do_QueryInterface (self->browser));
	g_return_val_if_fail (persist, FALSE);

	PRUint32 currentState;
	persist->GetCurrentState (&currentState);
	if (currentState == nsIWebBrowserPersist::PERSIST_STATE_SAVING) {
		//still performing a previous save - can't save again
		return FALSE; 
	}
	NS_NewNativeLocalFile (nsDependentCString (filename), PR_TRUE, getter_AddRefs (file));

	NS_NewNativeLocalFile(nsDependentCString (folder), PR_TRUE, getter_AddRefs (data));
	persist->GetPersistFlags (&flags);
	if (!(flags & nsIWebBrowserPersist::PERSIST_FLAGS_REPLACE_EXISTING_FILES)) {
		persist->SetPersistFlags (nsIWebBrowserPersist::PERSIST_FLAGS_REPLACE_EXISTING_FILES);
	}
	if (save_files) {
		persist->SaveDocument (nsnull, file, data, nsnull, 0, 0);
	}
	else {
		//Hack for a weird Mozilla bug - if you pass nsnull for the data
		//property, (meaning just save the HTML) it will finish saving properly 
		//but internally the save operation is never marked as finished. We need 
		//to explicitly tell it to cancel the (finished) save before we can save again.
		//Make sense? Didn't think so. But it works! =)
		if (currentState == nsIWebBrowserPersist::PERSIST_STATE_READY) {
			persist->CancelSave ();
		}	
		persist->SaveDocument (nsnull, file, nsnull, nsnull, 0, 0);
	}
	//persist->SaveCurrentURI(file);

	return TRUE;
}

GmeWebBrowser* 
gtk_moz_edit_get_web_browser (GtkMozEdit *self)
{
	GmeWebBrowser *browser = NULL;
	g_assert (self);

	browser = gme_web_browser_new (self->browser);
	return browser;
}
