/* -*- mode: C; c-file-style: "gnu" -*- */
/*
 * Copyright (C) 2003 Richard Hult <richard@imendio.com>
 * Copyright (C) 2003 Johan Dahlin <johan@gnome.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <string.h>
#include <gconf/gconf.h>
#include <glib/gi18n.h>
#include <gtk/gtkcellrendererpixbuf.h>
#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtkimagemenuitem.h>
#include <gtk/gtkliststore.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtkstock.h>
#include <gtk/gtktreeselection.h>
#include <gtk/gtktreeview.h>
#include "play-list-view.h"
#include "playlist-xml.h"
#include "utils.h"

static const GtkTargetEntry playlist_target_types[] = {
  { "x-special/jamboree-song-list", GTK_TARGET_SAME_APP, 0 }
};

static void     play_list_view_class_init           (PlayListViewClass *klass);
static void     play_list_view_init                 (PlayListView      *view);
static void     play_list_view_finalize             (GObject           *object);
static void     playlist_name_func                  (GtkTreeViewColumn *tree_column,
						     GtkCellRenderer   *cell,
						     GtkTreeModel      *tree_model,
						     GtkTreeIter       *iter,
						     gpointer           data);
static void     playlist_pixbuf_func                (GtkTreeViewColumn *tree_column,
						     GtkCellRenderer   *cell,
						     GtkTreeModel      *tree_model,
						     GtkTreeIter       *iter,
						     gpointer           data);
static gboolean playlist_tree_drag_motion_cb        (GtkWidget         *widget,
						     GdkDragContext    *context,
						     int                x,
						     int                y,
						     guint              time,
						     gpointer           data);
static gboolean playlist_tree_button_press_event_cb (GtkWidget         *widget,
						     GdkEventButton    *event,
						     gpointer           data);


static GObjectClass *parent_class;

GType
play_list_view_get_type (void)
{
  static GType type = 0;
	
  if (!type)
    {
      static const GTypeInfo info =
	{
	  sizeof (PlayListViewClass),
	  NULL,           /* base_init */
	  NULL,           /* base_finalize */
	  (GClassInitFunc) play_list_view_class_init,
	  NULL,           /* class_finalize */
	  NULL,           /* class_data */
	  sizeof (PlayListView),
	  0,
	  (GInstanceInitFunc) play_list_view_init,
	};

      type = g_type_register_static (GTK_TYPE_TREE_VIEW, "PlayListView",
				     &info, 0);
    }

  return type;
}

static void
play_list_view_class_init (PlayListViewClass *klass)
{
  GObjectClass *object_class;

  parent_class = g_type_class_peek_parent (klass);
  object_class = (GObjectClass*) klass;

  object_class->finalize = play_list_view_finalize;
}

static void
play_list_view_init (PlayListView *view)
{
  Playlist *playlist;
  GtkTreeViewColumn *column;
  GtkCellRenderer *cell;
  GtkTreeIter iter;
  
  view->model = GTK_TREE_MODEL (gtk_list_store_new (1, G_TYPE_POINTER));
  gtk_tree_view_set_model (GTK_TREE_VIEW (view), view->model);

  view->selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
  gtk_tree_selection_set_mode (view->selection, GTK_SELECTION_BROWSE);

  playlist = playlist_new (PLAYLIST_TYPE_LIBRARY, _("All Songs"));
  gtk_list_store_append (GTK_LIST_STORE (view->model), &iter);
  gtk_list_store_set (GTK_LIST_STORE (view->model), &iter, 0, playlist, -1);

  /* FIXME: Is this really necessary? */
  gtk_tree_selection_select_iter (view->selection, &iter);

  /* Drag n drop. */
  g_signal_connect (view,
		    "drag_motion",
		    G_CALLBACK (playlist_tree_drag_motion_cb),
		    NULL);
  
  gtk_drag_dest_set (GTK_WIDGET (view),
		     GTK_DEST_DEFAULT_ALL ,
		     playlist_target_types,
		     G_N_ELEMENTS (playlist_target_types),
		     GDK_ACTION_COPY);

  g_signal_connect (view,
		    "button_press_event",
		    G_CALLBACK (playlist_tree_button_press_event_cb),
		    NULL);
  
  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Playlists"));

  cell = gtk_cell_renderer_pixbuf_new ();
  gtk_tree_view_column_pack_start (column, cell, FALSE);
  gtk_tree_view_column_set_cell_data_func (column, cell,
					   playlist_pixbuf_func,
					   NULL,
					   NULL);
  
  view->cell = gtk_cell_renderer_text_new ();
  gtk_tree_view_column_pack_start (column, view->cell, TRUE);
  gtk_tree_view_column_set_cell_data_func (column, view->cell,
					   playlist_name_func,
					   NULL,
					   NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (view), column);
}

static void
play_list_view_finalize (GObject *object)
{
  /*PlayListView *view = PLAY_LIST_VIEW (object);*/

  if (G_OBJECT_CLASS (parent_class)->finalize)
    G_OBJECT_CLASS (parent_class)->finalize (object);
}


static void
playlist_name_func (GtkTreeViewColumn *tree_column,
		    GtkCellRenderer   *cell,
		    GtkTreeModel      *tree_model,
		    GtkTreeIter       *iter,
		    gpointer           data)
{
  Playlist *playlist;
	
  gtk_tree_model_get (tree_model, iter, 0, &playlist,  -1);

  g_object_set (cell,
		"text", playlist_get_name (playlist),
		"editable", playlist_is_editable (playlist),
		NULL);
}

static void
playlist_pixbuf_func (GtkTreeViewColumn *tree_column,
		      GtkCellRenderer   *cell,
		      GtkTreeModel      *tree_model,
		      GtkTreeIter       *iter,
		      gpointer           data)
{
  Playlist *playlist;
  static GdkPixbuf *smart = NULL;
  static GdkPixbuf *regular = NULL;
  static GdkPixbuf *library = NULL;
  GdkPixbuf *pixbuf;

  if (!smart)
    {
      smart = gdk_pixbuf_new_from_file (DATADIR "/jamboree/jamboree-playlist-smart.png", NULL);
      regular = gdk_pixbuf_new_from_file (DATADIR "/jamboree/jamboree-playlist-regular.png", NULL);
      library = gdk_pixbuf_new_from_file (DATADIR "/jamboree/jamboree-playlist-library.png", NULL);
    }
  
  gtk_tree_model_get (tree_model, iter, 0, &playlist, -1);
  
  switch (playlist->type) {
  case PLAYLIST_TYPE_SMART:
    pixbuf = smart;
    break;
  case PLAYLIST_TYPE_LIBRARY:
    pixbuf = library;
    break;
  case PLAYLIST_TYPE_REGULAR:
    pixbuf = regular;
    break;

  default:
  case PLAYLIST_TYPE_IRADIO:
    pixbuf = NULL;
    g_assert_not_reached ();
  }
  
  g_object_set (cell, "pixbuf", pixbuf, NULL);
}

GtkWidget *
play_list_view_new (SongDB *db)
{
  PlayListView *view;
  
  view = PLAY_LIST_VIEW (g_object_new (TYPE_PLAY_LIST_VIEW, NULL));
  view->db = db;
  
  return GTK_WIDGET (view);
}

Playlist *
play_list_view_get_playlist (PlayListView *view)
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;
  Playlist *playlist;

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
  
  if (gtk_tree_selection_get_selected (selection, &model, &iter))
    {
      gtk_tree_model_get (model, &iter, 0, &playlist, -1);
      return playlist;
    }

  return NULL;
}

void
play_list_view_store_prefs (PlayListView *view)
{
  Playlist *playlist;
  const char *str = "";
  
  playlist = play_list_view_get_playlist (view);
  if (playlist)
    {
      if (playlist->type != PLAYLIST_TYPE_LIBRARY)
	str = playlist->name;
    }
  
  gconf_client_set_string (gconf_client, "/apps/jamboree/active_view", str, NULL);
}

void
play_list_view_restore_prefs (PlayListView *view)
{
  char *active_view = NULL;
  GtkTreeModel *model;
  GtkTreeIter iter;
  Playlist *playlist;
  GtkTreeSelection *selection;
  
  model = gtk_tree_view_get_model (GTK_TREE_VIEW (view));
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
  
  active_view = gconf_client_get_string (gconf_client, "/apps/jamboree/active_view", NULL);
  if (!active_view)
    return;
  
  if (!gtk_tree_model_get_iter_first (model, &iter))
    return;
  
  while (1)
    {
      gtk_tree_model_get (model, &iter, 0, &playlist, -1);
      
      if (strcmp (playlist->name, active_view) == 0)
	{
	  gtk_tree_selection_select_iter (selection, &iter);
	  break;
	}
      
      if (!gtk_tree_model_iter_next (model, &iter))
	break;
    }

  g_free (active_view);
}

static gboolean
playlist_tree_drag_motion_cb (GtkWidget      *widget,
			      GdkDragContext *context,
			      int             x,
			      int             y,
			      guint           time,
			      gpointer        data)
{
  GtkTreeModel *model;
  GtkTreePath *path;
  GtkTreeIter iter;
  gboolean found;
  Playlist *playlist;
  
  g_signal_stop_emission_by_name (widget, "drag_motion");
  
  found = gtk_tree_view_get_dest_row_at_pos (GTK_TREE_VIEW (widget), x, y, &path, NULL);
  if (found)
    {
      model = gtk_tree_view_get_model (GTK_TREE_VIEW (widget));
      
      if (gtk_tree_model_get_iter (model, &iter, path))
	{
	  gtk_tree_model_get (model, &iter,
			      0, &playlist,
			      -1);

	  if (playlist->type == PLAYLIST_TYPE_REGULAR)
	    {
	      gtk_tree_view_set_drag_dest_row (GTK_TREE_VIEW (widget), path, GTK_TREE_VIEW_DROP_INTO_OR_AFTER);
	      gdk_drag_status (context, context->suggested_action, time);
	      
	      gtk_tree_path_free (path);
	      return TRUE;
	    }
	}

      gtk_tree_path_free (path);
    }
  
  gdk_drag_status (context, 0, time);  

  return TRUE;
}
 
static void
context_remove_cb (GtkWidget    *menu_item,
		   PlayListView *view)
{
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeModel *model;
  GList *playlists;
  Playlist *playlist;
  GList *l, *m;
  Song* song;
  
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
  if (!gtk_tree_selection_get_selected (selection, &model, &iter))
    return;
  
  gtk_tree_model_get (model, &iter, 0, &playlist, -1);
  gtk_list_store_remove (GTK_LIST_STORE (model), &iter);

  if (playlist->type == PLAYLIST_TYPE_REGULAR)
    {
      for (l = view->db->songs; l; l = l->next)
	{
	  song = l->data;
	  
	  for (m = song->playlists; m; m = m->next)
	    if (GPOINTER_TO_INT (m->data) == playlist->id)
	      {
		song->playlists = g_list_delete_link (song->playlists, m);
		song_db_update_song (view->db, song);
	      }
	}
    }

  /* FIXME: Butt ugly. */
  playlists = playlist_xml_load_playlists ();
  playlists = g_list_remove (playlists, playlist);
  playlist_xml_save_playlists (playlists);
}

static GtkWidget *
append_context_item (GtkWidget *menu,
		     char      *stock_item,
		     gpointer   function,
		     gpointer   data)
{
  GtkWidget *menuitem;
    
  menuitem = gtk_image_menu_item_new_from_stock (stock_item, NULL);
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
  gtk_widget_show (menuitem);

  if (function)
    g_signal_connect (menuitem,
		      "activate",
		      G_CALLBACK (function),
		      data);
  return menuitem;
}

static gboolean
playlist_tree_button_press_event_cb (GtkWidget      *widget,
				     GdkEventButton *event,
				     gpointer        data)
{
  GtkWidget *menu;
  GtkTreePath *path;
  GtkTreeIter iter;
  gboolean has_cell;
  GtkTreeSelection *selection;
  Playlist *playlist;
  
  if (event->button != 3)
    return FALSE;

  has_cell = gtk_tree_view_get_path_at_pos (GTK_TREE_VIEW (widget),
					    event->x,
					    event->y,
					    &path,
					    NULL, NULL, NULL);
  if (!has_cell)
    return FALSE;

  gtk_tree_model_get_iter (PLAY_LIST_VIEW (widget)->model, &iter, path);
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (widget));
  if (!gtk_tree_selection_iter_is_selected (selection, &iter))
    {
      gtk_tree_selection_unselect_all (selection);
      gtk_tree_selection_select_iter (selection, &iter);
    }

  playlist = play_list_view_get_playlist (PLAY_LIST_VIEW (widget));

  menu = gtk_menu_new ();
  append_context_item (menu, GTK_STOCK_REMOVE, context_remove_cb, widget);
  /* FIXME: Show smart-playlist-dialog */
#if 0
  if (playlist->type == PLAYLIST_TYPE_SMART)
    append_context_item (menu, GTK_STOCK_PROPERTIES, pl_context_edit_cb, widget);
  append_context_item (menu, GTK_STOCK_RENAME, pl_context_rename_cb, widget);
#endif
  
  gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL, 3, event->time);
  
  return TRUE;
}

