package org.gnu.libebook;

import org.gnu.glib.Handle;
import org.gnu.glib.MemStruct;
import org.gnu.libedataserver.EDSMemStruct;

/**
 * This object is used to Query against an {@link EBook} ultimatly resulting in a list of EContacts that match the query
 */
public class EBookQuery extends EDSMemStruct
{
    /**
     * Construct a new EBookQuery using the handle returned by a call to the native layer.
     * @param handle
     *              The handle (which is retrieved from a native layer) which represents an EBookQuery
     */
    private EBookQuery(Handle handle)
    {
        super(handle);
    }

    protected static EBookQuery getEBookQuery(Handle handle)
    {
        if (handle == null)
            return null;

        EBookQuery obj = (EBookQuery) MemStruct.getMemStructFromHandle(handle);
        return obj == null ? new EBookQuery(handle) : obj;
    }

    /**
     * Get a String representation of this EBookQuery
     * @return A String representation of this EBookQuery
     */
    public String getAsString()
    {
        return e_book_query_to_string(getHandle());
    }

    /**
     * Create an EBookQuery from a String
     * @param query
     *              A string representing an EBookQuery
     * @return
     */
    public static EBookQuery createFromString(String query)
    {
        return getEBookQuery(e_book_query_from_string(query));
    }

    /**
     * Invert or 'not' the EBookQuery
     * @return An EBookQuery which is the inverse of the original
     */
    public EBookQuery invert()
    {
        return getEBookQuery(e_book_query_not(getHandle()));
    }

    /**
     * Create a new EBookQuery which is the logical OR of two queries.
     * @param query
     *              The EBookQuery to OR with
     * @return An EBookQuery which is equivient to original || query
     */
    public EBookQuery or(EBookQuery query)
    {
        return getEBookQuery(e_book_query_or(getHandle(), query.getHandle()));
    }

    /**
     * Create a new EBookQuery which is the logical AND of two queries.
     * @param query
     *              The EBookQuery to AND with
     * @return An EBookQuery which is equivient to original && query
     */
    public EBookQuery and(EBookQuery query)
    {
        return getEBookQuery(e_book_query_and(getHandle(), query.getHandle()));
    }

    /**
     * Creates a new EBookQuery which tests if an EContactField exists.
     * @param field
     *              The field to test for existance
     * @return An EBookQuery that tests if 'field' exists
     */
    public static EBookQuery createQueryFieldExists(EContactField field)
    {
        return getEBookQuery(e_book_query_field_exists(field.getFieldEnumId()));
    }

    /**
     * This will create an empty EBookQuery - All EContacts will be macthed if used.
     * @return An Empty EBookQuery
     */
    public static EBookQuery createEmptyQuery()
    {
        return getEBookQuery(createQueryHandleAnyFieldContains(""));
    }

    /**
     * Creates a new EBookQuery which tests if any field contains value.
     * @param value
     *              The String value to match a field against
     * @return A new EBookQuery that matchs any field that contains a String value
     */
    public static EBookQuery createQueryAnyFieldContains(String value)
    {
        return getEBookQuery(createQueryHandleAnyFieldContains(value));
    }

    private static Handle createQueryHandleAnyFieldContains(String value)
    {
        return e_book_query_any_field_contains(value);
    }

    /**
     * Creates a new EBookQuery which tests if the field equals value.
     * @param field
     *              The {@link EContactField} to match against
     * @param value
     *              The String value to match the field against
     * @return A new EBookQuery that matchs the field against a String value
     */
    public static EBookQuery createQueryFieldEquals(EContactField field, String value)
    {
        return getEBookQuery(e_book_query_field_equals(field.getFieldEnumId(), value));
    }

    /**
     * Creates a new EBookQuery which tests if the field contains value.
     * @param field
     *              The {@link EContactField} to match against
     * @param value
     *              The String value to match the field against
     * @return A new EBookQuery that matchs a field containing a String value
     */
    public static EBookQuery createQueryFieldContains(EContactField field, String value)
    {
        return getEBookQuery(e_book_query_field_contains(field.getFieldEnumId(), value));
    }

    /**
     * Creates a new EBookQuery which tests if the field begins with value.
     * @param field
     *              The {@link EContactField} to match against
     * @param value
     *              The String value to match the field against
     * @return A new EBookQuery that matchs a field beginning with a String value
     */
    public static EBookQuery createQueryFieldBeginsWith(EContactField field, String value)
    {
        return getEBookQuery(e_book_query_field_begins_with(field.getFieldEnumId(), value));
    }

    /**
     * Creates a new EBookQuery which tests if the field ends with value.
     * @param field
     *              The {@link EContactField} to match against
     * @param value
     *              The String value to match the field against
     * @return A new EBookQuery that matchs a field ending with a String value
     */
    public static EBookQuery createQueryFieldEndsWith(EContactField field, String value)
    {
        return getEBookQuery(e_book_query_field_ends_with(field.getFieldEnumId(), value));
    }

    native static private Handle e_book_query_not(Handle handle);

    native static private Handle e_book_query_and(Handle handle1, Handle handle2);

    native static private Handle e_book_query_or(Handle handle1, Handle handle2);

    native static private Handle e_book_query_field_equals(int fieldEnumId, String value);

    native static private Handle e_book_query_field_contains(int fieldEnumId, String value);

    native static private Handle e_book_query_field_begins_with(int fieldEnumId, String value);

    native static private Handle e_book_query_field_ends_with(int fieldEnumId, String value);

    native static private Handle e_book_query_field_exists(int fieldEnumId);

    native static private Handle e_book_query_any_field_contains(String value);

    native static private Handle e_book_query_from_string(String query);

    native static private String e_book_query_to_string(Handle handle);
}
