/* testhashmap.vala
 *
 * Copyright (C) 2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <gee.h>
#include <stdlib.h>
#include <string.h>




#define CODE_NOT_REACHABLE "*code should not be reached*"
void test_hashmap_get (void);
void test_hashmap_set (void);
void test_hashmap_remove (void);
void test_hashmap_contains (void);
void test_hashmap_size (void);
void test_hashmap_get_keys (void);
void test_hashmap_get_values (void);
void test_hashmap_clear (void);
static void _test_hashmap_get_gcallback (void);
static void _test_hashmap_set_gcallback (void);
static void _test_hashmap_remove_gcallback (void);
static void _test_hashmap_contains_gcallback (void);
static void _test_hashmap_size_gcallback (void);
static void _test_hashmap_get_keys_gcallback (void);
static void _test_hashmap_get_values_gcallback (void);
static void _test_hashmap_clear_gcallback (void);
void _main (char** args, int args_length1);
static int _vala_strcmp0 (const char * str1, const char * str2);



void test_hashmap_get (void) {
	GeeHashMap* hashmap;
	char* _tmp0_;
	char* _tmp1_;
	char* _tmp2_;
	char* _tmp3_;
	char* _tmp4_;
	char* _tmp5_;
	hashmap = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_str_equal);
	/* Check get from empty map*/
	_tmp0_ = NULL;
	g_assert ((_tmp0_ = (char*) gee_map_get ((GeeMap*) hashmap, "foo")) == NULL);
	_tmp0_ = (g_free (_tmp0_), NULL);
	/* Check get from map with one items*/
	gee_map_set ((GeeMap*) hashmap, "key", "value");
	_tmp1_ = NULL;
	g_assert (_vala_strcmp0 (_tmp1_ = (char*) gee_map_get ((GeeMap*) hashmap, "key"), "value") == 0);
	_tmp1_ = (g_free (_tmp1_), NULL);
	/* Check get from non-existing key*/
	_tmp2_ = NULL;
	g_assert ((_tmp2_ = (char*) gee_map_get ((GeeMap*) hashmap, "foo")) == NULL);
	_tmp2_ = (g_free (_tmp2_), NULL);
	/* Check get from map with multiple items*/
	gee_map_set ((GeeMap*) hashmap, "key2", "value2");
	gee_map_set ((GeeMap*) hashmap, "key3", "value3");
	_tmp3_ = NULL;
	g_assert (_vala_strcmp0 (_tmp3_ = (char*) gee_map_get ((GeeMap*) hashmap, "key"), "value") == 0);
	_tmp3_ = (g_free (_tmp3_), NULL);
	_tmp4_ = NULL;
	g_assert (_vala_strcmp0 (_tmp4_ = (char*) gee_map_get ((GeeMap*) hashmap, "key2"), "value2") == 0);
	_tmp4_ = (g_free (_tmp4_), NULL);
	_tmp5_ = NULL;
	g_assert (_vala_strcmp0 (_tmp5_ = (char*) gee_map_get ((GeeMap*) hashmap, "key3"), "value3") == 0);
	_tmp5_ = (g_free (_tmp5_), NULL);
	(hashmap == NULL) ? NULL : (hashmap = (g_object_unref (hashmap), NULL));
}


void test_hashmap_set (void) {
	GeeHashMap* hashmap;
	char* _tmp0_;
	char* _tmp1_;
	char* _tmp2_;
	char* _tmp3_;
	char* _tmp4_;
	hashmap = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_str_equal);
	/* check map is empty*/
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 0);
	/* check set an item to map*/
	gee_map_set ((GeeMap*) hashmap, "abc", "one");
	g_assert (gee_map_contains ((GeeMap*) hashmap, "abc"));
	_tmp0_ = NULL;
	g_assert (_vala_strcmp0 (_tmp0_ = (char*) gee_map_get ((GeeMap*) hashmap, "abc"), "one") == 0);
	_tmp0_ = (g_free (_tmp0_), NULL);
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 1);
	/* check set an item to map with same value*/
	gee_map_set ((GeeMap*) hashmap, "def", "one");
	g_assert (gee_map_contains ((GeeMap*) hashmap, "def"));
	_tmp1_ = NULL;
	g_assert (_vala_strcmp0 (_tmp1_ = (char*) gee_map_get ((GeeMap*) hashmap, "abc"), "one") == 0);
	_tmp1_ = (g_free (_tmp1_), NULL);
	_tmp2_ = NULL;
	g_assert (_vala_strcmp0 (_tmp2_ = (char*) gee_map_get ((GeeMap*) hashmap, "def"), "one") == 0);
	_tmp2_ = (g_free (_tmp2_), NULL);
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 2);
	/* check set with same key*/
	gee_map_set ((GeeMap*) hashmap, "def", "two");
	g_assert (gee_map_contains ((GeeMap*) hashmap, "def"));
	_tmp3_ = NULL;
	g_assert (_vala_strcmp0 (_tmp3_ = (char*) gee_map_get ((GeeMap*) hashmap, "abc"), "one") == 0);
	_tmp3_ = (g_free (_tmp3_), NULL);
	_tmp4_ = NULL;
	g_assert (_vala_strcmp0 (_tmp4_ = (char*) gee_map_get ((GeeMap*) hashmap, "def"), "two") == 0);
	_tmp4_ = (g_free (_tmp4_), NULL);
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 2);
	(hashmap == NULL) ? NULL : (hashmap = (g_object_unref (hashmap), NULL));
}


void test_hashmap_remove (void) {
	GeeHashMap* hashmap;
	hashmap = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_str_equal);
	/* check removing when map is empty*/
	gee_map_remove ((GeeMap*) hashmap, "foo");
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 0);
	/* add items*/
	gee_map_set ((GeeMap*) hashmap, "aaa", "111");
	gee_map_set ((GeeMap*) hashmap, "bbb", "222");
	gee_map_set ((GeeMap*) hashmap, "ccc", "333");
	gee_map_set ((GeeMap*) hashmap, "ddd", "444");
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 4);
	/* check remove on first place*/
	gee_map_remove ((GeeMap*) hashmap, "aaa");
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 3);
	/* check remove in between */
	gee_map_remove ((GeeMap*) hashmap, "ccc");
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 2);
	/* check remove in last place*/
	gee_map_remove ((GeeMap*) hashmap, "ddd");
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 1);
	/* check remove invalid key*/
	gee_map_remove ((GeeMap*) hashmap, "bar");
	/* check remove last in map*/
	gee_map_remove ((GeeMap*) hashmap, "bbb");
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 0);
	(hashmap == NULL) ? NULL : (hashmap = (g_object_unref (hashmap), NULL));
}


void test_hashmap_contains (void) {
	GeeHashMap* hashmap;
	char* _tmp0_;
	char* _tmp1_;
	char* _tmp2_;
	char* _tmp3_;
	char* _tmp4_;
	char* _tmp5_;
	GeeHashMap* hashmapOfInt;
	hashmap = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_str_equal);
	/* Check on empty map*/
	g_assert (!gee_map_contains ((GeeMap*) hashmap, "111"));
	/* Check items*/
	gee_map_set ((GeeMap*) hashmap, "10", "111");
	g_assert (gee_map_contains ((GeeMap*) hashmap, "10"));
	g_assert (!gee_map_contains ((GeeMap*) hashmap, "20"));
	g_assert (!gee_map_contains ((GeeMap*) hashmap, "30"));
	_tmp0_ = NULL;
	g_assert (_vala_strcmp0 (_tmp0_ = (char*) gee_map_get ((GeeMap*) hashmap, "10"), "111") == 0);
	_tmp0_ = (g_free (_tmp0_), NULL);
	gee_map_set ((GeeMap*) hashmap, "20", "222");
	g_assert (gee_map_contains ((GeeMap*) hashmap, "10"));
	g_assert (gee_map_contains ((GeeMap*) hashmap, "20"));
	g_assert (!gee_map_contains ((GeeMap*) hashmap, "30"));
	_tmp1_ = NULL;
	g_assert (_vala_strcmp0 (_tmp1_ = (char*) gee_map_get ((GeeMap*) hashmap, "10"), "111") == 0);
	_tmp1_ = (g_free (_tmp1_), NULL);
	_tmp2_ = NULL;
	g_assert (_vala_strcmp0 (_tmp2_ = (char*) gee_map_get ((GeeMap*) hashmap, "20"), "222") == 0);
	_tmp2_ = (g_free (_tmp2_), NULL);
	gee_map_set ((GeeMap*) hashmap, "30", "333");
	g_assert (gee_map_contains ((GeeMap*) hashmap, "10"));
	g_assert (gee_map_contains ((GeeMap*) hashmap, "20"));
	g_assert (gee_map_contains ((GeeMap*) hashmap, "30"));
	_tmp3_ = NULL;
	g_assert (_vala_strcmp0 (_tmp3_ = (char*) gee_map_get ((GeeMap*) hashmap, "10"), "111") == 0);
	_tmp3_ = (g_free (_tmp3_), NULL);
	_tmp4_ = NULL;
	g_assert (_vala_strcmp0 (_tmp4_ = (char*) gee_map_get ((GeeMap*) hashmap, "20"), "222") == 0);
	_tmp4_ = (g_free (_tmp4_), NULL);
	_tmp5_ = NULL;
	g_assert (_vala_strcmp0 (_tmp5_ = (char*) gee_map_get ((GeeMap*) hashmap, "30"), "333") == 0);
	_tmp5_ = (g_free (_tmp5_), NULL);
	/* Clear and recheck*/
	gee_map_clear ((GeeMap*) hashmap);
	g_assert (!gee_map_contains ((GeeMap*) hashmap, "10"));
	g_assert (!gee_map_contains ((GeeMap*) hashmap, "20"));
	g_assert (!gee_map_contains ((GeeMap*) hashmap, "30"));
	hashmapOfInt = gee_hash_map_new (G_TYPE_INT, NULL, NULL, G_TYPE_INT, NULL, NULL, g_direct_hash, g_direct_equal, g_direct_equal);
	/* Check items*/
	gee_map_set ((GeeMap*) hashmapOfInt, GINT_TO_POINTER (10), GINT_TO_POINTER (111));
	g_assert (gee_map_contains ((GeeMap*) hashmapOfInt, GINT_TO_POINTER (10)));
	g_assert (!gee_map_contains ((GeeMap*) hashmapOfInt, GINT_TO_POINTER (20)));
	g_assert (!gee_map_contains ((GeeMap*) hashmapOfInt, GINT_TO_POINTER (30)));
	g_assert (GPOINTER_TO_INT (gee_map_get ((GeeMap*) hashmapOfInt, GINT_TO_POINTER (10))) == 111);
	gee_map_set ((GeeMap*) hashmapOfInt, GINT_TO_POINTER (20), GINT_TO_POINTER (222));
	g_assert (gee_map_contains ((GeeMap*) hashmapOfInt, GINT_TO_POINTER (10)));
	g_assert (gee_map_contains ((GeeMap*) hashmapOfInt, GINT_TO_POINTER (20)));
	g_assert (!gee_map_contains ((GeeMap*) hashmapOfInt, GINT_TO_POINTER (30)));
	g_assert (GPOINTER_TO_INT (gee_map_get ((GeeMap*) hashmapOfInt, GINT_TO_POINTER (10))) == 111);
	g_assert (GPOINTER_TO_INT (gee_map_get ((GeeMap*) hashmapOfInt, GINT_TO_POINTER (20))) == 222);
	gee_map_set ((GeeMap*) hashmapOfInt, GINT_TO_POINTER (30), GINT_TO_POINTER (333));
	g_assert (gee_map_contains ((GeeMap*) hashmapOfInt, GINT_TO_POINTER (10)));
	g_assert (gee_map_contains ((GeeMap*) hashmapOfInt, GINT_TO_POINTER (20)));
	g_assert (gee_map_contains ((GeeMap*) hashmapOfInt, GINT_TO_POINTER (30)));
	g_assert (GPOINTER_TO_INT (gee_map_get ((GeeMap*) hashmapOfInt, GINT_TO_POINTER (10))) == 111);
	g_assert (GPOINTER_TO_INT (gee_map_get ((GeeMap*) hashmapOfInt, GINT_TO_POINTER (20))) == 222);
	g_assert (GPOINTER_TO_INT (gee_map_get ((GeeMap*) hashmapOfInt, GINT_TO_POINTER (30))) == 333);
	(hashmap == NULL) ? NULL : (hashmap = (g_object_unref (hashmap), NULL));
	(hashmapOfInt == NULL) ? NULL : (hashmapOfInt = (g_object_unref (hashmapOfInt), NULL));
}


void test_hashmap_size (void) {
	GeeHashMap* hashmap;
	hashmap = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_str_equal);
	/* Check empty map*/
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 0);
	/* Check when one item*/
	gee_map_set ((GeeMap*) hashmap, "1", "1");
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 1);
	/* Check when more items*/
	gee_map_set ((GeeMap*) hashmap, "2", "2");
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 2);
	/* Check when items cleared*/
	gee_map_clear ((GeeMap*) hashmap);
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 0);
	(hashmap == NULL) ? NULL : (hashmap = (g_object_unref (hashmap), NULL));
}


void test_hashmap_get_keys (void) {
	GeeHashMap* hashmap;
	GeeSet* keySet;
	GeeSet* _tmp0_;
	GeeSet* _tmp1_;
	GeeSet* _tmp2_;
	hashmap = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_str_equal);
	/* Check keys on empty map*/
	keySet = gee_map_get_keys ((GeeMap*) hashmap);
	g_assert (gee_collection_get_size ((GeeCollection*) keySet) == 0);
	/* Check keys on map with one item*/
	gee_map_set ((GeeMap*) hashmap, "aaa", "111");
	g_assert (gee_collection_get_size ((GeeCollection*) keySet) == 1);
	g_assert (gee_collection_contains ((GeeCollection*) keySet, "aaa"));
	_tmp0_ = NULL;
	keySet = (_tmp0_ = gee_map_get_keys ((GeeMap*) hashmap), (keySet == NULL) ? NULL : (keySet = (g_object_unref (keySet), NULL)), _tmp0_);
	g_assert (gee_collection_get_size ((GeeCollection*) keySet) == 1);
	g_assert (gee_collection_contains ((GeeCollection*) keySet, "aaa"));
	/* Check modify key set directly*/
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		gee_collection_add ((GeeCollection*) keySet, "ccc");
		(hashmap == NULL) ? NULL : (hashmap = (g_object_unref (hashmap), NULL));
		(keySet == NULL) ? NULL : (keySet = (g_object_unref (keySet), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	g_test_trap_assert_stderr (CODE_NOT_REACHABLE);
	/* Check keys on map with multiple items*/
	gee_map_set ((GeeMap*) hashmap, "bbb", "222");
	g_assert (gee_collection_get_size ((GeeCollection*) keySet) == 2);
	g_assert (gee_collection_contains ((GeeCollection*) keySet, "aaa"));
	g_assert (gee_collection_contains ((GeeCollection*) keySet, "bbb"));
	_tmp1_ = NULL;
	keySet = (_tmp1_ = gee_map_get_keys ((GeeMap*) hashmap), (keySet == NULL) ? NULL : (keySet = (g_object_unref (keySet), NULL)), _tmp1_);
	g_assert (gee_collection_get_size ((GeeCollection*) keySet) == 2);
	g_assert (gee_collection_contains ((GeeCollection*) keySet, "aaa"));
	g_assert (gee_collection_contains ((GeeCollection*) keySet, "bbb"));
	/* Check keys on map clear*/
	gee_map_clear ((GeeMap*) hashmap);
	g_assert (gee_collection_get_size ((GeeCollection*) keySet) == 0);
	_tmp2_ = NULL;
	keySet = (_tmp2_ = gee_map_get_keys ((GeeMap*) hashmap), (keySet == NULL) ? NULL : (keySet = (g_object_unref (keySet), NULL)), _tmp2_);
	g_assert (gee_collection_get_size ((GeeCollection*) keySet) == 0);
	(hashmap == NULL) ? NULL : (hashmap = (g_object_unref (hashmap), NULL));
	(keySet == NULL) ? NULL : (keySet = (g_object_unref (keySet), NULL));
}


void test_hashmap_get_values (void) {
	GeeHashMap* hashmap;
	GeeCollection* valueCollection;
	GeeCollection* _tmp0_;
	GeeCollection* _tmp1_;
	GeeCollection* _tmp2_;
	hashmap = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_str_equal);
	/* Check keys on empty map*/
	valueCollection = gee_map_get_values ((GeeMap*) hashmap);
	g_assert (gee_collection_get_size (valueCollection) == 0);
	/* Check keys on map with one item*/
	gee_map_set ((GeeMap*) hashmap, "aaa", "111");
	g_assert (gee_collection_get_size (valueCollection) == 1);
	g_assert (gee_collection_contains (valueCollection, "111"));
	_tmp0_ = NULL;
	valueCollection = (_tmp0_ = gee_map_get_values ((GeeMap*) hashmap), (valueCollection == NULL) ? NULL : (valueCollection = (g_object_unref (valueCollection), NULL)), _tmp0_);
	g_assert (gee_collection_get_size (valueCollection) == 1);
	g_assert (gee_collection_contains (valueCollection, "111"));
	/* Check modify key set directly*/
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		gee_collection_add (valueCollection, "ccc");
		(hashmap == NULL) ? NULL : (hashmap = (g_object_unref (hashmap), NULL));
		(valueCollection == NULL) ? NULL : (valueCollection = (g_object_unref (valueCollection), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	g_test_trap_assert_stderr (CODE_NOT_REACHABLE);
	/* Check keys on map with multiple items*/
	gee_map_set ((GeeMap*) hashmap, "bbb", "222");
	g_assert (gee_collection_get_size (valueCollection) == 2);
	g_assert (gee_collection_contains (valueCollection, "111"));
	g_assert (gee_collection_contains (valueCollection, "222"));
	_tmp1_ = NULL;
	valueCollection = (_tmp1_ = gee_map_get_values ((GeeMap*) hashmap), (valueCollection == NULL) ? NULL : (valueCollection = (g_object_unref (valueCollection), NULL)), _tmp1_);
	g_assert (gee_collection_get_size (valueCollection) == 2);
	g_assert (gee_collection_contains (valueCollection, "111"));
	g_assert (gee_collection_contains (valueCollection, "222"));
	/* Check keys on map clear*/
	gee_map_clear ((GeeMap*) hashmap);
	g_assert (gee_collection_get_size (valueCollection) == 0);
	_tmp2_ = NULL;
	valueCollection = (_tmp2_ = gee_map_get_values ((GeeMap*) hashmap), (valueCollection == NULL) ? NULL : (valueCollection = (g_object_unref (valueCollection), NULL)), _tmp2_);
	g_assert (gee_collection_get_size (valueCollection) == 0);
	(hashmap == NULL) ? NULL : (hashmap = (g_object_unref (hashmap), NULL));
	(valueCollection == NULL) ? NULL : (valueCollection = (g_object_unref (valueCollection), NULL));
}


void test_hashmap_clear (void) {
	GeeHashMap* hashmap;
	hashmap = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_str_equal);
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 0);
	/* Check clear on empty map*/
	gee_map_clear ((GeeMap*) hashmap);
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 0);
	/* Check clear one item*/
	gee_map_set ((GeeMap*) hashmap, "1", "1");
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 1);
	gee_map_clear ((GeeMap*) hashmap);
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 0);
	/* Check clear multiple items*/
	gee_map_set ((GeeMap*) hashmap, "1", "1");
	gee_map_set ((GeeMap*) hashmap, "2", "2");
	gee_map_set ((GeeMap*) hashmap, "3", "3");
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 3);
	gee_map_clear ((GeeMap*) hashmap);
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 0);
	(hashmap == NULL) ? NULL : (hashmap = (g_object_unref (hashmap), NULL));
}


static void _test_hashmap_get_gcallback (void) {
	test_hashmap_get ();
}


static void _test_hashmap_set_gcallback (void) {
	test_hashmap_set ();
}


static void _test_hashmap_remove_gcallback (void) {
	test_hashmap_remove ();
}


static void _test_hashmap_contains_gcallback (void) {
	test_hashmap_contains ();
}


static void _test_hashmap_size_gcallback (void) {
	test_hashmap_size ();
}


static void _test_hashmap_get_keys_gcallback (void) {
	test_hashmap_get_keys ();
}


static void _test_hashmap_get_values_gcallback (void) {
	test_hashmap_get_values ();
}


static void _test_hashmap_clear_gcallback (void) {
	test_hashmap_clear ();
}


void _main (char** args, int args_length1) {
	g_test_init (&args_length1, &args, NULL);
	g_test_add_func ("/HashMap/Map/get", _test_hashmap_get_gcallback);
	g_test_add_func ("/HashMap/Map/set", _test_hashmap_set_gcallback);
	g_test_add_func ("/HashMap/Map/remove", _test_hashmap_remove_gcallback);
	g_test_add_func ("/HashMap/Map/contains", _test_hashmap_contains_gcallback);
	g_test_add_func ("/HashMap/Map/size", _test_hashmap_size_gcallback);
	g_test_add_func ("/HashMap/Map/get_keys", _test_hashmap_get_keys_gcallback);
	g_test_add_func ("/HashMap/Map/get_values", _test_hashmap_get_values_gcallback);
	g_test_add_func ("/HashMap/Map/clear", _test_hashmap_clear_gcallback);
	g_test_run ();
}


int main (int argc, char ** argv) {
	g_type_init ();
	_main (argv, argc);
	return 0;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




