/* testhashmap.vala
 *
 * Copyright (C) 2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <gee.h>
#include <stdlib.h>
#include <string.h>




#define CODE_NOT_REACHABLE "*code should not be reached*"
void test_hashmap_get (void);
void test_hashmap_set (void);
void test_hashmap_remove (void);
void test_hashmap_contains (void);
void test_hashmap_size (void);
void test_hashmap_get_keys (void);
void test_hashmap_get_values (void);
void test_hashmap_clear (void);
void test_hashmap_empty (void);
void test_hashmap_set_all (void);
void test_hashmap_remove_all (void);
void test_hashmap_contains_all (void);
static void _test_hashmap_get_gcallback (void);
static void _test_hashmap_set_gcallback (void);
static void _test_hashmap_remove_gcallback (void);
static void _test_hashmap_contains_gcallback (void);
static void _test_hashmap_size_gcallback (void);
static void _test_hashmap_get_keys_gcallback (void);
static void _test_hashmap_get_values_gcallback (void);
static void _test_hashmap_clear_gcallback (void);
static void _test_hashmap_empty_gcallback (void);
static void _test_hashmap_set_all_gcallback (void);
static void _test_hashmap_remove_all_gcallback (void);
static void _test_hashmap_contains_all_gcallback (void);
void _main (char** args, int args_length1);
static int _vala_strcmp0 (const char * str1, const char * str2);



void test_hashmap_get (void) {
	GeeHashMap* hashmap;
	char* _tmp0_;
	char* _tmp1_;
	char* _tmp2_;
	char* _tmp3_;
	char* _tmp4_;
	char* _tmp5_;
	hashmap = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_str_equal);
	/* Check get from empty map*/
	_tmp0_ = NULL;
	g_assert ((_tmp0_ = (char*) gee_abstract_map_get ((GeeAbstractMap*) hashmap, "foo")) == NULL);
	_tmp0_ = (g_free (_tmp0_), NULL);
	/* Check get from map with one items*/
	gee_abstract_map_set ((GeeAbstractMap*) hashmap, "key", "value");
	_tmp1_ = NULL;
	g_assert (_vala_strcmp0 (_tmp1_ = (char*) gee_abstract_map_get ((GeeAbstractMap*) hashmap, "key"), "value") == 0);
	_tmp1_ = (g_free (_tmp1_), NULL);
	/* Check get from non-existing key*/
	_tmp2_ = NULL;
	g_assert ((_tmp2_ = (char*) gee_abstract_map_get ((GeeAbstractMap*) hashmap, "foo")) == NULL);
	_tmp2_ = (g_free (_tmp2_), NULL);
	/* Check get from map with multiple items*/
	gee_abstract_map_set ((GeeAbstractMap*) hashmap, "key2", "value2");
	gee_abstract_map_set ((GeeAbstractMap*) hashmap, "key3", "value3");
	_tmp3_ = NULL;
	g_assert (_vala_strcmp0 (_tmp3_ = (char*) gee_abstract_map_get ((GeeAbstractMap*) hashmap, "key"), "value") == 0);
	_tmp3_ = (g_free (_tmp3_), NULL);
	_tmp4_ = NULL;
	g_assert (_vala_strcmp0 (_tmp4_ = (char*) gee_abstract_map_get ((GeeAbstractMap*) hashmap, "key2"), "value2") == 0);
	_tmp4_ = (g_free (_tmp4_), NULL);
	_tmp5_ = NULL;
	g_assert (_vala_strcmp0 (_tmp5_ = (char*) gee_abstract_map_get ((GeeAbstractMap*) hashmap, "key3"), "value3") == 0);
	_tmp5_ = (g_free (_tmp5_), NULL);
	(hashmap == NULL) ? NULL : (hashmap = (g_object_unref (hashmap), NULL));
}


void test_hashmap_set (void) {
	GeeHashMap* hashmap;
	char* _tmp0_;
	char* _tmp1_;
	char* _tmp2_;
	char* _tmp3_;
	char* _tmp4_;
	hashmap = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_str_equal);
	/* check map is empty*/
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 0);
	/* check set an item to map*/
	gee_abstract_map_set ((GeeAbstractMap*) hashmap, "abc", "one");
	g_assert (gee_abstract_map_contains ((GeeAbstractMap*) hashmap, "abc"));
	_tmp0_ = NULL;
	g_assert (_vala_strcmp0 (_tmp0_ = (char*) gee_abstract_map_get ((GeeAbstractMap*) hashmap, "abc"), "one") == 0);
	_tmp0_ = (g_free (_tmp0_), NULL);
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 1);
	/* check set an item to map with same value*/
	gee_abstract_map_set ((GeeAbstractMap*) hashmap, "def", "one");
	g_assert (gee_abstract_map_contains ((GeeAbstractMap*) hashmap, "def"));
	_tmp1_ = NULL;
	g_assert (_vala_strcmp0 (_tmp1_ = (char*) gee_abstract_map_get ((GeeAbstractMap*) hashmap, "abc"), "one") == 0);
	_tmp1_ = (g_free (_tmp1_), NULL);
	_tmp2_ = NULL;
	g_assert (_vala_strcmp0 (_tmp2_ = (char*) gee_abstract_map_get ((GeeAbstractMap*) hashmap, "def"), "one") == 0);
	_tmp2_ = (g_free (_tmp2_), NULL);
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 2);
	/* check set with same key*/
	gee_abstract_map_set ((GeeAbstractMap*) hashmap, "def", "two");
	g_assert (gee_abstract_map_contains ((GeeAbstractMap*) hashmap, "def"));
	_tmp3_ = NULL;
	g_assert (_vala_strcmp0 (_tmp3_ = (char*) gee_abstract_map_get ((GeeAbstractMap*) hashmap, "abc"), "one") == 0);
	_tmp3_ = (g_free (_tmp3_), NULL);
	_tmp4_ = NULL;
	g_assert (_vala_strcmp0 (_tmp4_ = (char*) gee_abstract_map_get ((GeeAbstractMap*) hashmap, "def"), "two") == 0);
	_tmp4_ = (g_free (_tmp4_), NULL);
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 2);
	(hashmap == NULL) ? NULL : (hashmap = (g_object_unref (hashmap), NULL));
}


void test_hashmap_remove (void) {
	GeeHashMap* hashmap;
	char* value;
	char* _tmp2_;
	gboolean _tmp1_;
	gpointer _tmp0_;
	char* _tmp5_;
	gboolean _tmp4_;
	gpointer _tmp3_;
	hashmap = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_str_equal);
	value = NULL;
	/* check removing when map is empty*/
	gee_abstract_map_remove ((GeeAbstractMap*) hashmap, "foo", NULL);
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 0);
	/* add items*/
	gee_abstract_map_set ((GeeAbstractMap*) hashmap, "aaa", "111");
	gee_abstract_map_set ((GeeAbstractMap*) hashmap, "bbb", "222");
	gee_abstract_map_set ((GeeAbstractMap*) hashmap, "ccc", "333");
	gee_abstract_map_set ((GeeAbstractMap*) hashmap, "ddd", "444");
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 4);
	/* check remove on first place*/
	gee_abstract_map_remove ((GeeAbstractMap*) hashmap, "aaa", NULL);
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 3);
	/* check remove in between */
	_tmp2_ = NULL;
	_tmp0_ = NULL;
	_tmp1_ = gee_abstract_map_remove ((GeeAbstractMap*) hashmap, "ccc", &_tmp0_);
	value = (_tmp2_ = (char*) _tmp0_, value = (g_free (value), NULL), _tmp2_);
	_tmp1_;
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 2);
	g_assert (_vala_strcmp0 (value, "333") == 0);
	/* check remove in last place*/
	gee_abstract_map_remove ((GeeAbstractMap*) hashmap, "ddd", NULL);
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 1);
	/* check remove invalid key*/
	_tmp5_ = NULL;
	_tmp3_ = NULL;
	_tmp4_ = gee_abstract_map_remove ((GeeAbstractMap*) hashmap, "bar", &_tmp3_);
	value = (_tmp5_ = (char*) _tmp3_, value = (g_free (value), NULL), _tmp5_);
	_tmp4_;
	g_assert (value == NULL);
	/* check remove last in map*/
	gee_abstract_map_remove ((GeeAbstractMap*) hashmap, "bbb", NULL);
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 0);
	(hashmap == NULL) ? NULL : (hashmap = (g_object_unref (hashmap), NULL));
	value = (g_free (value), NULL);
}


void test_hashmap_contains (void) {
	GeeHashMap* hashmap;
	char* _tmp0_;
	char* _tmp1_;
	char* _tmp2_;
	char* _tmp3_;
	char* _tmp4_;
	char* _tmp5_;
	GeeHashMap* hashmapOfInt;
	hashmap = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_str_equal);
	/* Check on empty map*/
	g_assert (!gee_abstract_map_contains ((GeeAbstractMap*) hashmap, "111"));
	/* Check items*/
	gee_abstract_map_set ((GeeAbstractMap*) hashmap, "10", "111");
	g_assert (gee_abstract_map_contains ((GeeAbstractMap*) hashmap, "10"));
	g_assert (!gee_abstract_map_contains ((GeeAbstractMap*) hashmap, "20"));
	g_assert (!gee_abstract_map_contains ((GeeAbstractMap*) hashmap, "30"));
	_tmp0_ = NULL;
	g_assert (_vala_strcmp0 (_tmp0_ = (char*) gee_abstract_map_get ((GeeAbstractMap*) hashmap, "10"), "111") == 0);
	_tmp0_ = (g_free (_tmp0_), NULL);
	gee_abstract_map_set ((GeeAbstractMap*) hashmap, "20", "222");
	g_assert (gee_abstract_map_contains ((GeeAbstractMap*) hashmap, "10"));
	g_assert (gee_abstract_map_contains ((GeeAbstractMap*) hashmap, "20"));
	g_assert (!gee_abstract_map_contains ((GeeAbstractMap*) hashmap, "30"));
	_tmp1_ = NULL;
	g_assert (_vala_strcmp0 (_tmp1_ = (char*) gee_abstract_map_get ((GeeAbstractMap*) hashmap, "10"), "111") == 0);
	_tmp1_ = (g_free (_tmp1_), NULL);
	_tmp2_ = NULL;
	g_assert (_vala_strcmp0 (_tmp2_ = (char*) gee_abstract_map_get ((GeeAbstractMap*) hashmap, "20"), "222") == 0);
	_tmp2_ = (g_free (_tmp2_), NULL);
	gee_abstract_map_set ((GeeAbstractMap*) hashmap, "30", "333");
	g_assert (gee_abstract_map_contains ((GeeAbstractMap*) hashmap, "10"));
	g_assert (gee_abstract_map_contains ((GeeAbstractMap*) hashmap, "20"));
	g_assert (gee_abstract_map_contains ((GeeAbstractMap*) hashmap, "30"));
	_tmp3_ = NULL;
	g_assert (_vala_strcmp0 (_tmp3_ = (char*) gee_abstract_map_get ((GeeAbstractMap*) hashmap, "10"), "111") == 0);
	_tmp3_ = (g_free (_tmp3_), NULL);
	_tmp4_ = NULL;
	g_assert (_vala_strcmp0 (_tmp4_ = (char*) gee_abstract_map_get ((GeeAbstractMap*) hashmap, "20"), "222") == 0);
	_tmp4_ = (g_free (_tmp4_), NULL);
	_tmp5_ = NULL;
	g_assert (_vala_strcmp0 (_tmp5_ = (char*) gee_abstract_map_get ((GeeAbstractMap*) hashmap, "30"), "333") == 0);
	_tmp5_ = (g_free (_tmp5_), NULL);
	/* Clear and recheck*/
	gee_abstract_map_clear ((GeeAbstractMap*) hashmap);
	g_assert (!gee_abstract_map_contains ((GeeAbstractMap*) hashmap, "10"));
	g_assert (!gee_abstract_map_contains ((GeeAbstractMap*) hashmap, "20"));
	g_assert (!gee_abstract_map_contains ((GeeAbstractMap*) hashmap, "30"));
	hashmapOfInt = gee_hash_map_new (G_TYPE_INT, NULL, NULL, G_TYPE_INT, NULL, NULL, g_direct_hash, g_direct_equal, g_direct_equal);
	/* Check items*/
	gee_abstract_map_set ((GeeAbstractMap*) hashmapOfInt, GINT_TO_POINTER (10), GINT_TO_POINTER (111));
	g_assert (gee_abstract_map_contains ((GeeAbstractMap*) hashmapOfInt, GINT_TO_POINTER (10)));
	g_assert (!gee_abstract_map_contains ((GeeAbstractMap*) hashmapOfInt, GINT_TO_POINTER (20)));
	g_assert (!gee_abstract_map_contains ((GeeAbstractMap*) hashmapOfInt, GINT_TO_POINTER (30)));
	g_assert (GPOINTER_TO_INT (gee_abstract_map_get ((GeeAbstractMap*) hashmapOfInt, GINT_TO_POINTER (10))) == 111);
	gee_abstract_map_set ((GeeAbstractMap*) hashmapOfInt, GINT_TO_POINTER (20), GINT_TO_POINTER (222));
	g_assert (gee_abstract_map_contains ((GeeAbstractMap*) hashmapOfInt, GINT_TO_POINTER (10)));
	g_assert (gee_abstract_map_contains ((GeeAbstractMap*) hashmapOfInt, GINT_TO_POINTER (20)));
	g_assert (!gee_abstract_map_contains ((GeeAbstractMap*) hashmapOfInt, GINT_TO_POINTER (30)));
	g_assert (GPOINTER_TO_INT (gee_abstract_map_get ((GeeAbstractMap*) hashmapOfInt, GINT_TO_POINTER (10))) == 111);
	g_assert (GPOINTER_TO_INT (gee_abstract_map_get ((GeeAbstractMap*) hashmapOfInt, GINT_TO_POINTER (20))) == 222);
	gee_abstract_map_set ((GeeAbstractMap*) hashmapOfInt, GINT_TO_POINTER (30), GINT_TO_POINTER (333));
	g_assert (gee_abstract_map_contains ((GeeAbstractMap*) hashmapOfInt, GINT_TO_POINTER (10)));
	g_assert (gee_abstract_map_contains ((GeeAbstractMap*) hashmapOfInt, GINT_TO_POINTER (20)));
	g_assert (gee_abstract_map_contains ((GeeAbstractMap*) hashmapOfInt, GINT_TO_POINTER (30)));
	g_assert (GPOINTER_TO_INT (gee_abstract_map_get ((GeeAbstractMap*) hashmapOfInt, GINT_TO_POINTER (10))) == 111);
	g_assert (GPOINTER_TO_INT (gee_abstract_map_get ((GeeAbstractMap*) hashmapOfInt, GINT_TO_POINTER (20))) == 222);
	g_assert (GPOINTER_TO_INT (gee_abstract_map_get ((GeeAbstractMap*) hashmapOfInt, GINT_TO_POINTER (30))) == 333);
	(hashmap == NULL) ? NULL : (hashmap = (g_object_unref (hashmap), NULL));
	(hashmapOfInt == NULL) ? NULL : (hashmapOfInt = (g_object_unref (hashmapOfInt), NULL));
}


void test_hashmap_size (void) {
	GeeHashMap* hashmap;
	hashmap = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_str_equal);
	/* Check empty map*/
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 0);
	/* Check when one item*/
	gee_abstract_map_set ((GeeAbstractMap*) hashmap, "1", "1");
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 1);
	/* Check when more items*/
	gee_abstract_map_set ((GeeAbstractMap*) hashmap, "2", "2");
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 2);
	/* Check when items cleared*/
	gee_abstract_map_clear ((GeeAbstractMap*) hashmap);
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 0);
	(hashmap == NULL) ? NULL : (hashmap = (g_object_unref (hashmap), NULL));
}


void test_hashmap_get_keys (void) {
	GeeHashMap* hashmap;
	GeeSet* keySet;
	GeeSet* _tmp0_;
	GeeSet* _tmp1_;
	GeeSet* _tmp2_;
	hashmap = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_str_equal);
	/* Check keys on empty map*/
	keySet = gee_abstract_map_get_keys ((GeeAbstractMap*) hashmap);
	g_assert (gee_collection_get_size ((GeeCollection*) keySet) == 0);
	/* Check keys on map with one item*/
	gee_abstract_map_set ((GeeAbstractMap*) hashmap, "aaa", "111");
	g_assert (gee_collection_get_size ((GeeCollection*) keySet) == 1);
	g_assert (gee_collection_contains ((GeeCollection*) keySet, "aaa"));
	_tmp0_ = NULL;
	keySet = (_tmp0_ = gee_abstract_map_get_keys ((GeeAbstractMap*) hashmap), (keySet == NULL) ? NULL : (keySet = (g_object_unref (keySet), NULL)), _tmp0_);
	g_assert (gee_collection_get_size ((GeeCollection*) keySet) == 1);
	g_assert (gee_collection_contains ((GeeCollection*) keySet, "aaa"));
	/* Check modify key set directly*/
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		gee_collection_add ((GeeCollection*) keySet, "ccc");
		(hashmap == NULL) ? NULL : (hashmap = (g_object_unref (hashmap), NULL));
		(keySet == NULL) ? NULL : (keySet = (g_object_unref (keySet), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	g_test_trap_assert_stderr (CODE_NOT_REACHABLE);
	/* Check keys on map with multiple items*/
	gee_abstract_map_set ((GeeAbstractMap*) hashmap, "bbb", "222");
	g_assert (gee_collection_get_size ((GeeCollection*) keySet) == 2);
	g_assert (gee_collection_contains ((GeeCollection*) keySet, "aaa"));
	g_assert (gee_collection_contains ((GeeCollection*) keySet, "bbb"));
	_tmp1_ = NULL;
	keySet = (_tmp1_ = gee_abstract_map_get_keys ((GeeAbstractMap*) hashmap), (keySet == NULL) ? NULL : (keySet = (g_object_unref (keySet), NULL)), _tmp1_);
	g_assert (gee_collection_get_size ((GeeCollection*) keySet) == 2);
	g_assert (gee_collection_contains ((GeeCollection*) keySet, "aaa"));
	g_assert (gee_collection_contains ((GeeCollection*) keySet, "bbb"));
	/* Check keys on map clear*/
	gee_abstract_map_clear ((GeeAbstractMap*) hashmap);
	g_assert (gee_collection_get_size ((GeeCollection*) keySet) == 0);
	_tmp2_ = NULL;
	keySet = (_tmp2_ = gee_abstract_map_get_keys ((GeeAbstractMap*) hashmap), (keySet == NULL) ? NULL : (keySet = (g_object_unref (keySet), NULL)), _tmp2_);
	g_assert (gee_collection_get_size ((GeeCollection*) keySet) == 0);
	(hashmap == NULL) ? NULL : (hashmap = (g_object_unref (hashmap), NULL));
	(keySet == NULL) ? NULL : (keySet = (g_object_unref (keySet), NULL));
}


void test_hashmap_get_values (void) {
	GeeHashMap* hashmap;
	GeeCollection* valueCollection;
	GeeCollection* _tmp0_;
	GeeCollection* _tmp1_;
	GeeCollection* _tmp2_;
	hashmap = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_str_equal);
	/* Check keys on empty map*/
	valueCollection = gee_abstract_map_get_values ((GeeAbstractMap*) hashmap);
	g_assert (gee_collection_get_size (valueCollection) == 0);
	/* Check keys on map with one item*/
	gee_abstract_map_set ((GeeAbstractMap*) hashmap, "aaa", "111");
	g_assert (gee_collection_get_size (valueCollection) == 1);
	g_assert (gee_collection_contains (valueCollection, "111"));
	_tmp0_ = NULL;
	valueCollection = (_tmp0_ = gee_abstract_map_get_values ((GeeAbstractMap*) hashmap), (valueCollection == NULL) ? NULL : (valueCollection = (g_object_unref (valueCollection), NULL)), _tmp0_);
	g_assert (gee_collection_get_size (valueCollection) == 1);
	g_assert (gee_collection_contains (valueCollection, "111"));
	/* Check modify key set directly*/
	if (g_test_trap_fork ((guint64) 0, G_TEST_TRAP_SILENCE_STDOUT | G_TEST_TRAP_SILENCE_STDERR)) {
		gee_collection_add (valueCollection, "ccc");
		(hashmap == NULL) ? NULL : (hashmap = (g_object_unref (hashmap), NULL));
		(valueCollection == NULL) ? NULL : (valueCollection = (g_object_unref (valueCollection), NULL));
		return;
	}
	g_test_trap_assert_failed ();
	g_test_trap_assert_stderr (CODE_NOT_REACHABLE);
	/* Check keys on map with multiple items*/
	gee_abstract_map_set ((GeeAbstractMap*) hashmap, "bbb", "222");
	g_assert (gee_collection_get_size (valueCollection) == 2);
	g_assert (gee_collection_contains (valueCollection, "111"));
	g_assert (gee_collection_contains (valueCollection, "222"));
	_tmp1_ = NULL;
	valueCollection = (_tmp1_ = gee_abstract_map_get_values ((GeeAbstractMap*) hashmap), (valueCollection == NULL) ? NULL : (valueCollection = (g_object_unref (valueCollection), NULL)), _tmp1_);
	g_assert (gee_collection_get_size (valueCollection) == 2);
	g_assert (gee_collection_contains (valueCollection, "111"));
	g_assert (gee_collection_contains (valueCollection, "222"));
	/* Check keys on map clear*/
	gee_abstract_map_clear ((GeeAbstractMap*) hashmap);
	g_assert (gee_collection_get_size (valueCollection) == 0);
	_tmp2_ = NULL;
	valueCollection = (_tmp2_ = gee_abstract_map_get_values ((GeeAbstractMap*) hashmap), (valueCollection == NULL) ? NULL : (valueCollection = (g_object_unref (valueCollection), NULL)), _tmp2_);
	g_assert (gee_collection_get_size (valueCollection) == 0);
	(hashmap == NULL) ? NULL : (hashmap = (g_object_unref (hashmap), NULL));
	(valueCollection == NULL) ? NULL : (valueCollection = (g_object_unref (valueCollection), NULL));
}


void test_hashmap_clear (void) {
	GeeHashMap* hashmap;
	hashmap = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_str_equal);
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 0);
	/* Check clear on empty map*/
	gee_abstract_map_clear ((GeeAbstractMap*) hashmap);
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 0);
	/* Check clear one item*/
	gee_abstract_map_set ((GeeAbstractMap*) hashmap, "1", "1");
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 1);
	gee_abstract_map_clear ((GeeAbstractMap*) hashmap);
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 0);
	/* Check clear multiple items*/
	gee_abstract_map_set ((GeeAbstractMap*) hashmap, "1", "1");
	gee_abstract_map_set ((GeeAbstractMap*) hashmap, "2", "2");
	gee_abstract_map_set ((GeeAbstractMap*) hashmap, "3", "3");
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 3);
	gee_abstract_map_clear ((GeeAbstractMap*) hashmap);
	g_assert (gee_map_get_size ((GeeMap*) hashmap) == 0);
	(hashmap == NULL) ? NULL : (hashmap = (g_object_unref (hashmap), NULL));
}


void test_hashmap_empty (void) {
	GeeHashMap* hashmap;
	hashmap = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_str_equal);
	/* Check empty map*/
	g_assert (gee_abstract_map_get_is_empty ((GeeAbstractMap*) hashmap));
	/* Check when one item*/
	gee_abstract_map_set ((GeeAbstractMap*) hashmap, "1", "1");
	g_assert (!gee_abstract_map_get_is_empty ((GeeAbstractMap*) hashmap));
	/* Check when more items*/
	gee_abstract_map_set ((GeeAbstractMap*) hashmap, "2", "2");
	g_assert (!gee_abstract_map_get_is_empty ((GeeAbstractMap*) hashmap));
	/* Check when items cleared*/
	gee_abstract_map_clear ((GeeAbstractMap*) hashmap);
	g_assert (gee_abstract_map_get_is_empty ((GeeAbstractMap*) hashmap));
	(hashmap == NULL) ? NULL : (hashmap = (g_object_unref (hashmap), NULL));
}


void test_hashmap_set_all (void) {
	GeeHashMap* hashmap1;
	GeeHashMap* hashmap2;
	char* _tmp0_;
	char* _tmp1_;
	char* _tmp2_;
	char* _tmp3_;
	char* _tmp4_;
	char* _tmp5_;
	hashmap1 = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_str_equal);
	hashmap2 = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_str_equal);
	gee_abstract_map_set ((GeeAbstractMap*) hashmap1, "a", "1");
	gee_abstract_map_set ((GeeAbstractMap*) hashmap1, "b", "2");
	gee_abstract_map_set ((GeeAbstractMap*) hashmap1, "c", "3");
	gee_abstract_map_set ((GeeAbstractMap*) hashmap2, "d", "4");
	gee_abstract_map_set ((GeeAbstractMap*) hashmap2, "e", "5");
	gee_abstract_map_set ((GeeAbstractMap*) hashmap2, "f", "6");
	gee_abstract_map_set_all ((GeeAbstractMap*) hashmap1, (GeeMap*) hashmap2);
	g_assert (gee_map_get_size ((GeeMap*) hashmap1) == 6);
	g_assert (gee_abstract_map_contains ((GeeAbstractMap*) hashmap1, "a"));
	g_assert (gee_abstract_map_contains ((GeeAbstractMap*) hashmap1, "b"));
	g_assert (gee_abstract_map_contains ((GeeAbstractMap*) hashmap1, "c"));
	g_assert (gee_abstract_map_contains ((GeeAbstractMap*) hashmap1, "d"));
	g_assert (gee_abstract_map_contains ((GeeAbstractMap*) hashmap1, "e"));
	g_assert (gee_abstract_map_contains ((GeeAbstractMap*) hashmap1, "f"));
	_tmp0_ = NULL;
	g_assert (_vala_strcmp0 (_tmp0_ = (char*) gee_abstract_map_get ((GeeAbstractMap*) hashmap1, "a"), "1") == 0);
	_tmp0_ = (g_free (_tmp0_), NULL);
	_tmp1_ = NULL;
	g_assert (_vala_strcmp0 (_tmp1_ = (char*) gee_abstract_map_get ((GeeAbstractMap*) hashmap1, "b"), "2") == 0);
	_tmp1_ = (g_free (_tmp1_), NULL);
	_tmp2_ = NULL;
	g_assert (_vala_strcmp0 (_tmp2_ = (char*) gee_abstract_map_get ((GeeAbstractMap*) hashmap1, "c"), "3") == 0);
	_tmp2_ = (g_free (_tmp2_), NULL);
	_tmp3_ = NULL;
	g_assert (_vala_strcmp0 (_tmp3_ = (char*) gee_abstract_map_get ((GeeAbstractMap*) hashmap1, "d"), "4") == 0);
	_tmp3_ = (g_free (_tmp3_), NULL);
	_tmp4_ = NULL;
	g_assert (_vala_strcmp0 (_tmp4_ = (char*) gee_abstract_map_get ((GeeAbstractMap*) hashmap1, "e"), "5") == 0);
	_tmp4_ = (g_free (_tmp4_), NULL);
	_tmp5_ = NULL;
	g_assert (_vala_strcmp0 (_tmp5_ = (char*) gee_abstract_map_get ((GeeAbstractMap*) hashmap1, "f"), "6") == 0);
	_tmp5_ = (g_free (_tmp5_), NULL);
	(hashmap1 == NULL) ? NULL : (hashmap1 = (g_object_unref (hashmap1), NULL));
	(hashmap2 == NULL) ? NULL : (hashmap2 = (g_object_unref (hashmap2), NULL));
}


void test_hashmap_remove_all (void) {
	GeeHashMap* map1;
	GeeHashMap* map2;
	map1 = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_str_equal);
	map2 = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_str_equal);
	/* Check remove all on empty maps.*/
	g_assert (gee_abstract_map_get_is_empty ((GeeAbstractMap*) map1));
	g_assert (gee_abstract_map_get_is_empty ((GeeAbstractMap*) map2));
	gee_abstract_map_remove_all ((GeeAbstractMap*) map1, (GeeMap*) map2);
	g_assert (gee_abstract_map_get_is_empty ((GeeAbstractMap*) map1));
	g_assert (gee_abstract_map_get_is_empty ((GeeAbstractMap*) map2));
	gee_abstract_map_clear ((GeeAbstractMap*) map1);
	gee_abstract_map_clear ((GeeAbstractMap*) map2);
	/* Map1 is empty, map2 has entries. -> no change*/
	gee_abstract_map_set ((GeeAbstractMap*) map2, "a", "1");
	gee_abstract_map_set ((GeeAbstractMap*) map2, "b", "2");
	g_assert (gee_abstract_map_get_is_empty ((GeeAbstractMap*) map1));
	g_assert (gee_map_get_size ((GeeMap*) map2) == 2);
	gee_abstract_map_remove_all ((GeeAbstractMap*) map1, (GeeMap*) map2);
	g_assert (gee_abstract_map_get_is_empty ((GeeAbstractMap*) map1));
	g_assert (gee_map_get_size ((GeeMap*) map2) == 2);
	gee_abstract_map_clear ((GeeAbstractMap*) map1);
	gee_abstract_map_clear ((GeeAbstractMap*) map2);
	/* Map1 has entries, map2 is empty. -> no change*/
	gee_abstract_map_set ((GeeAbstractMap*) map1, "a", "1");
	gee_abstract_map_set ((GeeAbstractMap*) map1, "b", "2");
	g_assert (gee_map_get_size ((GeeMap*) map1) == 2);
	g_assert (gee_abstract_map_get_is_empty ((GeeAbstractMap*) map2));
	gee_abstract_map_remove_all ((GeeAbstractMap*) map1, (GeeMap*) map2);
	g_assert (gee_map_get_size ((GeeMap*) map1) == 2);
	g_assert (gee_abstract_map_get_is_empty ((GeeAbstractMap*) map2));
	gee_abstract_map_clear ((GeeAbstractMap*) map1);
	gee_abstract_map_clear ((GeeAbstractMap*) map2);
	/* Map1 and map2 have the same entries -> map1 is cleared*/
	gee_abstract_map_set ((GeeAbstractMap*) map1, "a", "0");
	gee_abstract_map_set ((GeeAbstractMap*) map1, "b", "1");
	gee_abstract_map_set ((GeeAbstractMap*) map2, "a", "1");
	gee_abstract_map_set ((GeeAbstractMap*) map2, "b", "0");
	g_assert (gee_map_get_size ((GeeMap*) map1) == 2);
	g_assert (gee_map_get_size ((GeeMap*) map2) == 2);
	gee_abstract_map_remove_all ((GeeAbstractMap*) map1, (GeeMap*) map2);
	g_assert (gee_abstract_map_get_is_empty ((GeeAbstractMap*) map1));
	g_assert (gee_map_get_size ((GeeMap*) map2) == 2);
	gee_abstract_map_clear ((GeeAbstractMap*) map1);
	gee_abstract_map_clear ((GeeAbstractMap*) map2);
	/* Map1 has some common keys with map2 but both have also unique keys -> common key are cleared from map1*/
	gee_abstract_map_set ((GeeAbstractMap*) map1, "x", "2");
	gee_abstract_map_set ((GeeAbstractMap*) map1, "a", "1");
	gee_abstract_map_set ((GeeAbstractMap*) map1, "b", "1");
	gee_abstract_map_set ((GeeAbstractMap*) map1, "y", "2");
	gee_abstract_map_set ((GeeAbstractMap*) map2, "i", "100");
	gee_abstract_map_set ((GeeAbstractMap*) map2, "a", "200");
	gee_abstract_map_set ((GeeAbstractMap*) map2, "j", "300");
	gee_abstract_map_set ((GeeAbstractMap*) map2, "b", "400");
	gee_abstract_map_set ((GeeAbstractMap*) map2, "k", "500");
	g_assert (gee_map_get_size ((GeeMap*) map1) == 4);
	g_assert (gee_map_get_size ((GeeMap*) map2) == 5);
	gee_abstract_map_remove_all ((GeeAbstractMap*) map1, (GeeMap*) map2);
	g_assert (gee_map_get_size ((GeeMap*) map1) == 2);
	g_assert (gee_map_get_size ((GeeMap*) map2) == 5);
	g_assert (gee_abstract_map_contains ((GeeAbstractMap*) map1, "x"));
	g_assert (gee_abstract_map_contains ((GeeAbstractMap*) map1, "y"));
	gee_abstract_map_clear ((GeeAbstractMap*) map1);
	gee_abstract_map_clear ((GeeAbstractMap*) map2);
	(map1 == NULL) ? NULL : (map1 = (g_object_unref (map1), NULL));
	(map2 == NULL) ? NULL : (map2 = (g_object_unref (map2), NULL));
}


void test_hashmap_contains_all (void) {
	GeeHashMap* map1;
	GeeHashMap* map2;
	map1 = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_str_equal);
	map2 = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_str_equal);
	/* Check empty.*/
	g_assert (gee_abstract_map_contains_all ((GeeAbstractMap*) map1, (GeeMap*) map2));
	/* Map1 has items, map2 is empty.*/
	gee_abstract_map_set ((GeeAbstractMap*) map1, "1", "1");
	g_assert (gee_abstract_map_contains_all ((GeeAbstractMap*) map1, (GeeMap*) map2));
	gee_abstract_map_clear ((GeeAbstractMap*) map1);
	gee_abstract_map_clear ((GeeAbstractMap*) map2);
	/* Map1 is empty, map2 has items.*/
	gee_abstract_map_set ((GeeAbstractMap*) map2, "1", "1");
	g_assert (!gee_abstract_map_contains_all ((GeeAbstractMap*) map1, (GeeMap*) map2));
	gee_abstract_map_clear ((GeeAbstractMap*) map1);
	gee_abstract_map_clear ((GeeAbstractMap*) map2);
	/* Map1 and map2 are the same.*/
	gee_abstract_map_set ((GeeAbstractMap*) map1, "1", "a");
	gee_abstract_map_set ((GeeAbstractMap*) map1, "2", "b");
	gee_abstract_map_set ((GeeAbstractMap*) map2, "1", "c");
	gee_abstract_map_set ((GeeAbstractMap*) map2, "2", "d");
	g_assert (gee_abstract_map_contains_all ((GeeAbstractMap*) map1, (GeeMap*) map2));
	gee_abstract_map_clear ((GeeAbstractMap*) map1);
	gee_abstract_map_clear ((GeeAbstractMap*) map2);
	/* Map1 and map2 are not the same*/
	gee_abstract_map_set ((GeeAbstractMap*) map1, "1", "a");
	gee_abstract_map_set ((GeeAbstractMap*) map2, "2", "b");
	g_assert (!gee_abstract_map_contains_all ((GeeAbstractMap*) map1, (GeeMap*) map2));
	gee_abstract_map_clear ((GeeAbstractMap*) map1);
	gee_abstract_map_clear ((GeeAbstractMap*) map2);
	/* Map1 has a subset of map2*/
	gee_abstract_map_set ((GeeAbstractMap*) map1, "1", "a");
	gee_abstract_map_set ((GeeAbstractMap*) map1, "2", "b");
	gee_abstract_map_set ((GeeAbstractMap*) map1, "3", "c");
	gee_abstract_map_set ((GeeAbstractMap*) map1, "4", "d");
	gee_abstract_map_set ((GeeAbstractMap*) map1, "5", "e");
	gee_abstract_map_set ((GeeAbstractMap*) map1, "6", "f");
	gee_abstract_map_set ((GeeAbstractMap*) map2, "2", "g");
	gee_abstract_map_set ((GeeAbstractMap*) map2, "4", "h");
	gee_abstract_map_set ((GeeAbstractMap*) map2, "6", "i");
	g_assert (gee_abstract_map_contains_all ((GeeAbstractMap*) map1, (GeeMap*) map2));
	gee_abstract_map_clear ((GeeAbstractMap*) map1);
	gee_abstract_map_clear ((GeeAbstractMap*) map2);
	/* Map1 has a subset of map2 in all but one element map2*/
	gee_abstract_map_set ((GeeAbstractMap*) map1, "1", "a");
	gee_abstract_map_set ((GeeAbstractMap*) map1, "2", "b");
	gee_abstract_map_set ((GeeAbstractMap*) map1, "3", "c");
	gee_abstract_map_set ((GeeAbstractMap*) map1, "4", "d");
	gee_abstract_map_set ((GeeAbstractMap*) map1, "5", "e");
	gee_abstract_map_set ((GeeAbstractMap*) map1, "6", "f");
	gee_abstract_map_set ((GeeAbstractMap*) map2, "2", "g");
	gee_abstract_map_set ((GeeAbstractMap*) map2, "4", "h");
	gee_abstract_map_set ((GeeAbstractMap*) map2, "6", "i");
	gee_abstract_map_set ((GeeAbstractMap*) map2, "7", "j");
	g_assert (!gee_abstract_map_contains_all ((GeeAbstractMap*) map1, (GeeMap*) map2));
	gee_abstract_map_clear ((GeeAbstractMap*) map1);
	gee_abstract_map_clear ((GeeAbstractMap*) map2);
	(map1 == NULL) ? NULL : (map1 = (g_object_unref (map1), NULL));
	(map2 == NULL) ? NULL : (map2 = (g_object_unref (map2), NULL));
}


static void _test_hashmap_get_gcallback (void) {
	test_hashmap_get ();
}


static void _test_hashmap_set_gcallback (void) {
	test_hashmap_set ();
}


static void _test_hashmap_remove_gcallback (void) {
	test_hashmap_remove ();
}


static void _test_hashmap_contains_gcallback (void) {
	test_hashmap_contains ();
}


static void _test_hashmap_size_gcallback (void) {
	test_hashmap_size ();
}


static void _test_hashmap_get_keys_gcallback (void) {
	test_hashmap_get_keys ();
}


static void _test_hashmap_get_values_gcallback (void) {
	test_hashmap_get_values ();
}


static void _test_hashmap_clear_gcallback (void) {
	test_hashmap_clear ();
}


static void _test_hashmap_empty_gcallback (void) {
	test_hashmap_empty ();
}


static void _test_hashmap_set_all_gcallback (void) {
	test_hashmap_set_all ();
}


static void _test_hashmap_remove_all_gcallback (void) {
	test_hashmap_remove_all ();
}


static void _test_hashmap_contains_all_gcallback (void) {
	test_hashmap_contains_all ();
}


void _main (char** args, int args_length1) {
	g_test_init (&args_length1, &args, NULL);
	/* Methods of Map interface*/
	g_test_add_func ("/HashMap/Map/get", (GCallback) _test_hashmap_get_gcallback);
	g_test_add_func ("/HashMap/Map/set", (GCallback) _test_hashmap_set_gcallback);
	g_test_add_func ("/HashMap/Map/remove", (GCallback) _test_hashmap_remove_gcallback);
	g_test_add_func ("/HashMap/Map/contains", (GCallback) _test_hashmap_contains_gcallback);
	g_test_add_func ("/HashMap/Map/size", (GCallback) _test_hashmap_size_gcallback);
	g_test_add_func ("/HashMap/Map/get_keys", (GCallback) _test_hashmap_get_keys_gcallback);
	g_test_add_func ("/HashMap/Map/get_values", (GCallback) _test_hashmap_get_values_gcallback);
	g_test_add_func ("/HashMap/Map/clear", (GCallback) _test_hashmap_clear_gcallback);
	g_test_add_func ("/HashMap/Map/empty", (GCallback) _test_hashmap_empty_gcallback);
	g_test_add_func ("/HashMap/Map/set_all", (GCallback) _test_hashmap_set_all_gcallback);
	g_test_add_func ("/HashMap/Map/remove_all", (GCallback) _test_hashmap_remove_all_gcallback);
	g_test_add_func ("/HashMap/Map/contains_all", (GCallback) _test_hashmap_contains_all_gcallback);
	g_test_run ();
}


int main (int argc, char ** argv) {
	g_type_init ();
	_main (argv, argc);
	return 0;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




