#include <config.h>
#include "gnome-cups-request.h"

#include <cups/cups.h>
#include <cups/language.h>
#include <cups/http.h>
#include <cups/ipp.h>

#include "gnome-cups-util.h"
#include "gnome-cups-i18n.h"

static const char *
get_error_string (ipp_status_t error)
{  
	static const char *status_oks[] =     /* "OK" status codes */
                {
			N_("Success"),
			N_("Success, some attributes were ignored"),
			N_("Success, some attributes confliced"),
			N_("Success, ignored subscriptions"),
			N_("Success, ignored notifications"),
			N_("Success, too many events"),
			N_("Success, cancel subscription"),
                };
	static const char *status_400s[] =        /* Client errors */
		{
			N_("Bad request"),
			N_("Forbidden"),
			N_("The client has not been authenticated"),
			N_("You are not authorized to perform this operation"),
			N_("This operation cannot be completed"),
			N_("Timeout"),
			N_("The requested file was not found"),
			N_("The requested resource no longer exists"),
			N_("The request was too large"),
			N_("The request was too long"),
			N_("The document format is not supported"),
			N_("The value or attributes in this request are not supported"),
			N_("The URI scheme is not supported"),
			N_("The requested character set is not supported"),
			N_("There were conflicting attributes in the request"),
			N_("Compression is not supported"),
			N_("There was a compression error"),
			N_("There was an error in the format of the document"),
			N_("There was an error accessing the document"),
			N_("Some attributes could not be set"),
			N_("All subscriptions were ignored"),
			N_("Too many subscriptions"),
			N_("All notifications were ignored."),
			N_("A print support file was not found.")
		};
	
	static const char *status_500s[] =        /* Server errors */
                {
			N_("Internal server error"),
			N_("Operation not supported"),
			N_("Service unavailable"),
			N_("Version not supported"),
			N_("Device error"),
			N_("Temporary error"),
			N_("The printer is not accepting jobs"),
			N_("The printer is busy"),
			N_("The job has been cancelled"),
			N_("Multiple-document jobs are not supported"),
			N_("The printer is deactivated"),
		};                                                                             
	if (error >= IPP_OK && error <= IPP_OK_BUT_CANCEL_SUBSCRIPTION) {
		return _(status_oks[error]);
	} else if (error == IPP_REDIRECTION_OTHER_SITE) {
		return _("Redirected to another site");
	} else if (error >= IPP_BAD_REQUEST && error <= IPP_PRINT_SUPPORT_FILE_NOT_FOUND) {
		return _(status_400s[error - IPP_BAD_REQUEST]);
	} else if (error >= IPP_INTERNAL_ERROR && error <= IPP_PRINTER_IS_DEACTIVATED) {
		return _(status_500s[error - IPP_INTERNAL_ERROR]);
	}                                                                       
	return _("Unknown error");
}

ipp_t *
gnome_cups_request_new (int operation_id)
{
	ipp_t *request;
	cups_lang_t *language;
	
	language = cupsLangDefault ();
	request = ippNew ();
	request->request.op.operation_id = operation_id;
	request->request.op.request_id = 1;
	
	ippAddString(request, IPP_TAG_OPERATION, IPP_TAG_CHARSET,
		     "attributes-charset", 
		     NULL, "utf-8");
	
	ippAddString(request, IPP_TAG_OPERATION, IPP_TAG_LANGUAGE,
		     "attributes-natural-language", 
		     NULL, language->language);
	
	return request;
}

ipp_t *
gnome_cups_request_new_for_printer (int operation_id, 
				    const char *printer_name)
{
	ipp_t *request;
	char *printer_uri;
	
	request = gnome_cups_request_new (operation_id);

	printer_uri = gnome_cups_get_printer_uri (printer_name);
	ippAddString (request, IPP_TAG_OPERATION, IPP_TAG_URI,
		      "printer-uri", NULL, printer_uri);
	g_free (printer_uri);

	return request;
}

ipp_t *
gnome_cups_request_new_for_job (int operation_id, int job)
{
	ipp_t *request;
	char *job_uri;
	
	request = gnome_cups_request_new (operation_id);

	job_uri = g_strdup_printf ("ipp://localhost/jobs/%d", job);

	ippAddString (request, IPP_TAG_OPERATION, IPP_TAG_URI,
		      "job-uri", NULL, gnome_cups_strdup (job_uri));

	/* FIXME: need a way to act as another user.  I guess. */
	ippAddString (request, 
		      IPP_TAG_OPERATION,
		      IPP_TAG_NAME, 
		      "requesting-user-name", NULL, 
		      gnome_cups_strdup (g_get_user_name ()));

	g_free (job_uri);

	return request;
}

void
gnome_cups_request_add_requested_attributes (ipp_t *request, 
					     ipp_tag_t group,
					     int n_attributes,
					     char **attributes)
{
	ipp_attribute_t *attr;
	int i;
	
	attr = ippAddStrings (request, 
			      group,
			      IPP_TAG_KEYWORD,
			      "requested-attributes",
			      n_attributes, NULL, NULL);
	
	for (i = 0; i < n_attributes; i++) {
		attr->values[i].string.text = gnome_cups_strdup (attributes[i]);
	}
}

static gboolean request_executing = FALSE;

gboolean 
_gnome_cups_request_is_executing (void) 
{
	return request_executing;
}

ipp_t *
gnome_cups_request_execute (ipp_t *request, const char *path, GError **err)
{
	static http_t *main_http = NULL;
	static gboolean main_http_in_use = FALSE;
	http_t *single_http = NULL;
	http_t *http;
	ipp_t *response;
	ipp_status_t status;

	g_return_val_if_fail (err == NULL || *err == NULL, NULL);

	/* FIXME: This sucks.  We want to try to keep everything on
	 * one http connection, but due to reentrancy problems, we
	 * can't always do that (if the request goes back to the main
	 * loop during authentication, we can be called again, and
	 * can't issue a new request on that connection. 
	 * 
	 * For right now, we keep a main http connection around, and use that 
	 * if it's available.  Otherwise we do a connection per request. */

	cupsSetUser (g_get_user_name ());

	if (!main_http) {
		main_http = httpConnectEncrypt (cupsServer(), ippPort(), cupsEncryption());
	}

	if (request_executing) {
		http = main_http;
	} else {
		http = single_http = httpConnectEncrypt (cupsServer (),
							 ippPort(),
							 cupsEncryption());
	}

	request_executing = TRUE;
	
	response = cupsDoRequest (http, request, path ? path : "/");

	status = cupsLastError ();

	if (single_http) {
		httpClose (single_http);
	}

	request_executing = FALSE;

	if (status > IPP_OK_CONFLICT) {
		*err = g_error_new (GNOME_CUPS_ERROR, 
				    status,
				    get_error_string (status));
	}
	
	return response;
}
