/* GNOME DB library
 * Copyright (C) 1999-2001 The Free Software Foundation
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <libgda/gda-data-model-list.h>
#include <libgnomedb/gnome-db-grid.h>
#include <libgnomedb/gnome-db-list.h>
#include <libgnomedb/gnome-db-util.h>

#define PARENT_TYPE GTK_TYPE_VBOX

struct _GnomeDbListPrivate {
	GdaDataModel *data_model;
	GtkWidget *grid;
};

static void gnome_db_list_class_init (GnomeDbListClass *klass);
static void gnome_db_list_init       (GnomeDbList *list, GnomeDbListClass *klass);
static void gnome_db_list_finalize   (GObject *object);

static GObjectClass *parent_class = NULL;

/*
 * GnomeDbList class implementation
 */

static void
gnome_db_list_class_init (GnomeDbListClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = gnome_db_list_finalize;
}

static void
gnome_db_list_init (GnomeDbList *list, GnomeDbListClass *klass)
{
	g_return_if_fail (GNOME_DB_IS_LIST (list));

	/* allocate private structure */
	list->priv = g_new0 (GnomeDbListPrivate, 1);
	list->priv->data_model = NULL;

	/* set up widgets */
	list->priv->grid = gnome_db_new_grid_widget (NULL);
	gnome_db_grid_hide_column_titles (GNOME_DB_GRID (list->priv->grid));
	gtk_box_pack_start (GTK_BOX (list), list->priv->grid, TRUE, TRUE, 0);
}

static void
gnome_db_list_finalize (GObject *object)
{
	GnomeDbList *list = (GnomeDbList *) object;

	g_return_if_fail (GNOME_DB_IS_LIST (list));

	/* free memory */
	if (GDA_IS_DATA_MODEL (list->priv->data_model)) {
		g_object_unref (G_OBJECT (list->priv->data_model));
		list->priv->data_model = NULL;
	}

	g_free (list->priv);
	list->priv = NULL;

	/* chain to parent class */
	parent_class->finalize (object);
}

GType
gnome_db_list_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbListClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_list_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbList),
			0,
			(GInstanceInitFunc) gnome_db_list_init
		};
		type = g_type_register_static (PARENT_TYPE, "GnomeDbList", &info, 0);
	}
	return type;
}

/**
 * gnome_db_list_new
 *
 * Create a new #GnomeDbList widget, which lets you display lists
 * of columns.
 *
 * Returns: the newly created object
 */
GtkWidget *
gnome_db_list_new (void)
{
	GnomeDbList *list;

	list = g_object_new (GNOME_DB_TYPE_LIST, NULL);
	return GTK_WIDGET (list);
}

/**
 * gnome_db_list_new_with_model
 */
GtkWidget *
gnome_db_list_new_with_model (GdaDataModel *model, gint col)
{
	GtkWidget *list;

	g_return_val_if_fail (GDA_IS_DATA_MODEL (model), NULL);

	list = gnome_db_list_new ();
	gnome_db_list_set_model (GNOME_DB_LIST (list), model, col);

	return list;
}

/**
 * gnome_db_list_set_model
 * @list: a #GnomeDbList widget.
 * @model: a #GdaDataModel object.
 * @col: column to be shown from the data model.
 *
 * Set the data model to be displayed by the #GnomeDbList widget.
 * As this widget just shows single-columns data models, you have
 * to specify, along with the #GdaDataModel object, the column position
 * to be shown.
 */
void
gnome_db_list_set_model (GnomeDbList *list, GdaDataModel *model, gint col)
{
	gint n;
	gint row_count;

	g_return_if_fail (GNOME_DB_IS_LIST (list));

	/* free previous data model */
	if (GDA_IS_DATA_MODEL (list->priv->data_model)) {
		g_object_unref (G_OBJECT (list->priv->data_model));
		list->priv->data_model = NULL;
	}

	/* fill in our private data model */
	list->priv->data_model = gda_data_model_list_new ();

	row_count = gda_data_model_get_n_rows (model);
	for (n = 0; n < row_count; n++) {
		GdaValue *value;

		value = gda_data_model_get_value_at (model, col, n);
		if (value != NULL) {
			gda_data_model_list_append_value (
				GDA_DATA_MODEL_LIST (list->priv->data_model),
				(const GdaValue *) value);
		}
	}

	gnome_db_grid_set_model (GNOME_DB_GRID (list->priv->grid), list->priv->data_model);
	gnome_db_grid_hide_column_titles (GNOME_DB_GRID (list->priv->grid));
}
