#include <libgnomedb/libgnomedb.h>
#include <libgnomedb/libgnomedb.h>
#include <gtk/gtk.h>

#define SQL_QUERY_ROWS "SELECT c.name, o.creation_date FROM orders o INNER JOIN customers c ON (c.id=o.customer) ORDER BY c.name, o.creation_date"
#define SQL_QUERY_COLS "SELECT p.name, p.price FROM products p ORDER BY p.name, p.price"

/* 
 * Global configuration 
 */ 
typedef struct {
	/* General section */
	GnomeDbDict        *dict;
	gchar         *filename;
	gchar         *dsn;
	gchar         *username;
	gchar         *password;
	GtkWidget     *mainwin;
	GtkWidget     *tests_nb;

	GSList        *tests; /* list of ATest structures */

	/* Menu */
	GtkWidget     *menu_save;
	GtkWidget     *menu_db_selectds;
	GtkWidget     *menu_db_action;

	GtkWidget     *selector;
	GtkWidget     *del_obj_button;
	GtkWidget     *del_test_button;
	GtkWidget     *nb;
} MainConfig;


/*
 * Structure for each test type
 */
typedef struct _ATest {
	gchar        *test_name;
	gchar        *test_descr;
	GtkWidget  *(*make_test_widget)     (MainConfig *config);
	void        (*clean_data)           (MainConfig *config, GtkWidget *test_widget);
	gboolean    (*signal_conn_changed)  (MainConfig *config, GtkWidget *test_widget);
} ATest;


/*
 * Functions for all tests
 */
static void       build_tests (MainConfig *dict);
static void       signal_tests(MainConfig *dict);

static GnomeDbResultSet *get_ready_result_set_from_query (GnomeDbQuery *query, const gchar *ident);

/*
 * list of tests' functions
 */
static GtkWidget *matrix_test_make_with_data   (MainConfig *config);
static GtkWidget *matrix_test_make_without_data   (MainConfig *config);
static void       matrix_test_clean  (MainConfig *config, GtkWidget *test_widget);
static gboolean   matrix_test_signal (MainConfig *config, GtkWidget *test_widget);

static GtkWidget *rels_graph_test_make   (MainConfig *config);
static void       rels_graph_test_clean  (MainConfig *config, GtkWidget *test_widget);
static gboolean   rels_graph_test_signal (MainConfig *config, GtkWidget *test_widget);

static GtkWidget *work_core_test_make_form (MainConfig *config);
static GtkWidget *work_core_test_make_grid (MainConfig *config);
static void       work_core_test_clean     (MainConfig *config, GtkWidget *test_widget);
static gboolean   work_core_test_signal    (MainConfig *config, GtkWidget *test_widget);

static GtkWidget *forms_test_make   (MainConfig *config);
static void       forms_test_clean  (MainConfig *config, GtkWidget *test_widget);
static gboolean   forms_test_signal (MainConfig *config, GtkWidget *test_widget);

static GtkWidget *selector_test_make   (MainConfig *config);
static void       selector_test_clean  (MainConfig *config, GtkWidget *test_widget);
static gboolean   selector_test_signal (MainConfig *config, GtkWidget *test_widget);

static GtkWidget *query_graph_test_make   (MainConfig *config);
static void       query_graph_test_clean  (MainConfig *config, GtkWidget *test_widget);
static gboolean   query_graph_test_signal (MainConfig *config, GtkWidget *test_widget);

static GtkWidget *dsn_selector_test_make   (MainConfig *config);

static GtkWidget *editor_test_make   (MainConfig *config);

static GtkWidget *combo_test_make (MainConfig *config);
static void       combo_test_clean (MainConfig *config, GtkWidget *test_widget);
static gboolean   combo_test_signal (MainConfig *config, GtkWidget *test_widget);

/*
 * tests decalaration
 */
static void
build_tests (MainConfig *dict)
{
	GSList *list = NULL;
	ATest *test;

	/* GnomeDbCanvasDbRelations test */
	test = g_new0 (ATest, 1);
	test->test_name = "GnomeDbCanvasDbRelations";
	test->test_descr = "Requires a selected graph";
	test->make_test_widget = rels_graph_test_make;
	test->clean_data = rels_graph_test_clean;
	test->signal_conn_changed = rels_graph_test_signal;
	list = g_slist_prepend (list, test);

	/* GnomeDbMatrix test with data */
	test = g_new0 (ATest, 1);
	test->test_name = "GnomeDbMatrix (with data)";
	test->test_descr = "Requires a selected table, and an opened connection";
	test->make_test_widget = matrix_test_make_with_data;
	test->clean_data = matrix_test_clean;
	test->signal_conn_changed = matrix_test_signal;
	list = g_slist_prepend (list, test);

	/* GnomeDbMatrix test without data */
	test = g_new0 (ATest, 1);
	test->test_name = "GnomeDbMatrix (no data)";
	test->test_descr = "Matrix test with the following queries:\nROWS:\t" SQL_QUERY_ROWS "\nCOLS:\t" SQL_QUERY_COLS;
	test->make_test_widget = matrix_test_make_without_data;
	test->clean_data = matrix_test_clean;
	test->signal_conn_changed = matrix_test_signal;
	list = g_slist_prepend (list, test);

	/* GnomeDbLayout test */
	test = g_new0 (ATest, 1);
	test->test_name = "Custom layout";
	test->test_descr = "Requires a custom form, and an opened connection";
	test->make_test_widget = forms_test_make;
	test->clean_data = forms_test_clean;
	test->signal_conn_changed = forms_test_signal;
	list = g_slist_prepend (list, test);

	/* GnomeDbSelector test */
	test = g_new0 (ATest, 1);
	test->test_name = "GnomeDbSelector";
	test->test_descr = "";
	test->make_test_widget = selector_test_make;
	test->clean_data = selector_test_clean;
	test->signal_conn_changed = selector_test_signal;
	list = g_slist_prepend (list, test);

	/* GnomeDbCanvasQueryStruct test */
	test = g_new0 (ATest, 1);
	test->test_name = "GnomeDbCanvasQueryStruct";
	test->test_descr = "Requires a selected query";
	test->make_test_widget = query_graph_test_make;
	test->clean_data = query_graph_test_clean;
	test->signal_conn_changed = query_graph_test_signal;
	list = g_slist_prepend (list, test);

	/* GnomeDbDataSourceSelector test */
	test = g_new0 (ATest, 1);
	test->test_name = "GnomeDbDataSourceSelector";
	test->test_descr = "";
	test->make_test_widget = dsn_selector_test_make;
	test->clean_data = NULL;
	test->signal_conn_changed = NULL;
	list = g_slist_prepend (list, test);

	/* GnomeDbEditor test */
	test = g_new0 (ATest, 1);
	test->test_name = "GnomeDbEditor";
	test->test_descr = "";
	test->make_test_widget = editor_test_make;
	test->clean_data = NULL;
	test->signal_conn_changed = NULL;
	list = g_slist_prepend (list, test);

	/* GnomeDbCombo test */
	test = g_new0 (ATest, 1);
	test->test_name = "Combo";
	test->test_descr = "Requires a selected table, and an opened connection";
	test->make_test_widget = combo_test_make;
	test->clean_data = combo_test_clean;
	test->signal_conn_changed = combo_test_signal;
	list = g_slist_prepend (list, test);

	/* GnomeDbForm test */
	test = g_new0 (ATest, 1);
	test->test_name = "Form";
	test->test_descr = "Requires a selected table, and an opened connection";
	test->make_test_widget = work_core_test_make_form;
	test->clean_data = work_core_test_clean;
	test->signal_conn_changed = work_core_test_signal;
	list = g_slist_prepend (list, test);

	/* GnomeDbGrid test */
        test = g_new0 (ATest, 1);
        test->test_name = "Grid";
        test->test_descr = "Requires a selected table, and an opened connection";
        test->make_test_widget = work_core_test_make_grid;
        test->clean_data = work_core_test_clean;
        test->signal_conn_changed = work_core_test_signal;
        list = g_slist_prepend (list, test);

	dict->tests = list;
}

static void
signal_tests (MainConfig *dict)
{
	gint nb_pages, i;

	nb_pages = gtk_notebook_get_n_pages (GTK_NOTEBOOK (dict->tests_nb));
	for (i=0; i<nb_pages; i++) {
		GtkWidget *page = gtk_notebook_get_nth_page (GTK_NOTEBOOK (dict->tests_nb), i);
		ATest *test = g_object_get_data (G_OBJECT (page), "atest");
		if (test->signal_conn_changed)
			(test->signal_conn_changed) (dict, page);
	}
}


/*
 * MAIN
 */
static void set_filename (MainConfig *dict, const gchar *filename);
static void destroy (GtkWidget *widget, gpointer data);
static gboolean delete_event (GtkWidget *widget, GdkEvent  *event, gpointer data);
static GtkWidget *build_menu (MainConfig *dict, GtkWidget *mainwin);
static GtkWidget *build_page (MainConfig *dict);
static void       load_file (MainConfig *dict);
int 
main (int argc, char **argv)
{
	GtkWidget *mainwin, *vbox, *menu, *page;
	GnomeDbDict *dict;
	MainConfig *config;
	
	/* Initialize i18n support */
	gtk_set_locale ();
	
	/* Initialize the widget set */
	gtk_init (&argc, &argv);

	/* Test Configuration */
	config = g_new0 (MainConfig, 1);
	dict = GNOME_DB_DICT (gnome_db_dict_new ());
	config->dict = dict;
	build_tests (config);
	g_object_set (G_OBJECT (gnome_db_dict_get_server (dict)), "with_functions", TRUE, NULL);
	
	/* Create the main window */
	mainwin = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_container_set_border_width (GTK_CONTAINER (mainwin), 0);
	g_signal_connect (G_OBJECT (mainwin), "delete_event",
			  G_CALLBACK (delete_event), NULL);
	g_signal_connect (G_OBJECT (mainwin), "destroy",
			  G_CALLBACK (destroy), NULL);
	config->mainwin = mainwin;

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (mainwin), vbox);
	gtk_widget_show (vbox);

	/* menu */
	menu = build_menu (config, mainwin);
	gtk_widget_show (menu);
	gtk_box_pack_start (GTK_BOX (vbox), menu, FALSE, FALSE, 0);

	/* tables and views page */
	page = build_page (config);
	gtk_box_pack_start (GTK_BOX (vbox), page, TRUE, TRUE, 0);

	/* Show the application window */
	gtk_widget_set_size_request (mainwin, 800, 600);
	gtk_widget_show_all (mainwin);

	/* Application init */
	if (argc > 1)
		set_filename (config, argv[1]);

	load_file (config);
	
	gtk_main ();
	g_object_unref (G_OBJECT (dict));

	return 0;
}



/*
 * Common callbacks
 */
static gboolean delete_event (GtkWidget *widget, GdkEvent  *event, gpointer data)
{
    g_print ("Leaving DB test...\n");

    return FALSE;
}

static void destroy (GtkWidget *widget, gpointer data)
{
    gtk_main_quit ();
}

static void
set_filename (MainConfig *dict, const gchar *filename)
{
	if (dict->filename) {
		g_free (dict->filename);
		dict->filename = NULL;
	}
	if (filename)
		dict->filename = g_strdup (filename);

	if (filename)
		gtk_window_set_title (GTK_WINDOW (dict->mainwin), filename);
	else
		gtk_window_set_title (GTK_WINDOW (dict->mainwin), _("No File"));
}

static void
load_file (MainConfig *config)
{
	GError *error = NULL;
	if (!config->filename)
		return;

	gnome_db_server_reset (gnome_db_dict_get_server (config->dict));
	/* actual file loading */
	if (!gnome_db_dict_load_xml_file (config->dict, config->filename, &error)) {
		GtkWidget *msg;
		
		msg = gtk_message_dialog_new (GTK_WINDOW (config->mainwin), 
					      GTK_DIALOG_DESTROY_WITH_PARENT,
					      GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, 
					      _("Error loading file '%s':\n%s\n"), config->filename,
					      error ? error->message : _("unreported error"));
		
		gtk_dialog_run (GTK_DIALOG (msg));
		gtk_widget_destroy (msg);
		g_error_free (error);
		error = NULL;
	}
	else 
		signal_tests (config);
}


/*
 * Menu building
 */
static void add_test_cb (GtkWidget *wid, MainConfig *config);
static void open_file_cb (GtkWidget *wid, MainConfig *dict);
static void save_file_cb (GtkWidget *wid, MainConfig *dict);
static void save_as_file_cb (GtkWidget *wid, MainConfig *dict);
static void select_ds_cb (GtkWidget *wid, MainConfig *dict);
static void sync_action_cb (GtkWidget *wid, MainConfig *dict);
static void open_conn_cb (GtkWidget *wid, MainConfig *dict);
GtkWidget *
build_menu (MainConfig *dict, GtkWidget *mainwin)
{
	GtkWidget *menubar1, *menuitem1, *menuitem1_menu, *entry;
	GtkAccelGroup *accel_group;
	GSList *list;

	accel_group = gtk_accel_group_new ();

	menubar1 = gtk_menu_bar_new ();
	gtk_widget_show (menubar1);

	/* File menu */
	menuitem1 = gtk_menu_item_new_with_mnemonic (_("_File"));
	gtk_widget_show (menuitem1);
	gtk_container_add (GTK_CONTAINER (menubar1), menuitem1);
	
	menuitem1_menu = gtk_menu_new ();
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem1), menuitem1_menu);
	
	entry = gtk_image_menu_item_new_from_stock (GTK_STOCK_OPEN, accel_group);
	gtk_widget_show (entry);
	gtk_container_add (GTK_CONTAINER (menuitem1_menu), entry);

	g_signal_connect (G_OBJECT (entry), "activate",
			  G_CALLBACK (open_file_cb), dict);

	entry = gtk_image_menu_item_new_from_stock (GTK_STOCK_SAVE, accel_group);
	gtk_widget_show (entry);
	gtk_container_add (GTK_CONTAINER (menuitem1_menu), entry);

	g_signal_connect (G_OBJECT (entry), "activate",
			  G_CALLBACK (save_file_cb), dict);

	entry = gtk_image_menu_item_new_from_stock (GTK_STOCK_SAVE_AS, accel_group);
	gtk_widget_show (entry);
	gtk_container_add (GTK_CONTAINER (menuitem1_menu), entry);

	g_signal_connect (G_OBJECT (entry), "activate",
			  G_CALLBACK (save_as_file_cb), dict);

	entry = gtk_image_menu_item_new_from_stock (GTK_STOCK_QUIT, accel_group);
	gtk_widget_show (entry);
	gtk_container_add (GTK_CONTAINER (menuitem1_menu), entry);

	g_signal_connect (G_OBJECT (entry), "activate",
			  G_CALLBACK (destroy), dict);

	/* Database menu */
	menuitem1 = gtk_menu_item_new_with_mnemonic (_("_Database"));
	gtk_widget_show (menuitem1);
	gtk_container_add (GTK_CONTAINER (menubar1), menuitem1);
	
	menuitem1_menu = gtk_menu_new ();
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem1), menuitem1_menu);
	
	entry = gtk_menu_item_new_with_mnemonic (_("Open connection"));
	gtk_widget_show (entry);
	gtk_container_add (GTK_CONTAINER (menuitem1_menu), entry);
	g_signal_connect (G_OBJECT (entry), "activate",
			  G_CALLBACK (open_conn_cb), dict);

	entry = gtk_menu_item_new_with_mnemonic (_("Select datasource"));
	gtk_widget_show (entry);
	gtk_container_add (GTK_CONTAINER (menuitem1_menu), entry);

	g_signal_connect (G_OBJECT (entry), "activate",
			  G_CALLBACK (select_ds_cb), dict);

	entry = gtk_menu_item_new_with_mnemonic (_("Synchronise metadata with DBMS"));
	gtk_widget_show (entry);
	gtk_container_add (GTK_CONTAINER (menuitem1_menu), entry);

	g_signal_connect (G_OBJECT (entry), "activate",
			  G_CALLBACK (sync_action_cb), dict);

	/* Tests menu */
	menuitem1 = gtk_menu_item_new_with_mnemonic (_("_New Test"));
	gtk_widget_show (menuitem1);
	gtk_container_add (GTK_CONTAINER (menubar1), menuitem1);

	menuitem1_menu = gtk_menu_new ();
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem1), menuitem1_menu);

	list = dict->tests;
	while (list) {
		ATest *test = (ATest*) (list->data);
		entry = gtk_menu_item_new_with_mnemonic (test->test_name);
		gtk_widget_show (entry);
		gtk_container_add (GTK_CONTAINER (menuitem1_menu), entry);
		g_object_set_data (G_OBJECT (entry), "atest", test);
		g_signal_connect (G_OBJECT (entry), "activate",
				  G_CALLBACK (add_test_cb), dict);
		
		list = g_slist_next (list);
	}

	gtk_window_add_accel_group (GTK_WINDOW (mainwin), accel_group);

	return menubar1;
}

static void add_test_page (MainConfig *config, ATest *test);
static void
add_test_cb (GtkWidget *wid, MainConfig *config)
{
	ATest *test;

	test = g_object_get_data (G_OBJECT (wid), "atest");
	add_test_page (config, test);
}

static void
open_file_cb (GtkWidget *wid, MainConfig *dict)
{
	GtkWidget *file_selector;
	gint response;

	/* Create the selector */
	file_selector = gtk_file_selection_new (_("Select a file to load"));
	response = gtk_dialog_run (GTK_DIALOG (file_selector));
	if (response == GTK_RESPONSE_OK) {
		set_filename (dict, gtk_file_selection_get_filename (GTK_FILE_SELECTION (file_selector)));
		load_file (dict);
	}
	gtk_widget_destroy (file_selector);
}

static void
save_file_cb (GtkWidget *wid, MainConfig *dict)
{
	if (dict->filename) {
		GError *error = NULL;

		if (!gnome_db_dict_save_xml_file (dict->dict, dict->filename, &error)) {
			GtkWidget *msg;
			
			msg = gtk_message_dialog_new (GTK_WINDOW (dict->mainwin), 
						      GTK_DIALOG_DESTROY_WITH_PARENT,
						      GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, 
						      _("Error saving file '%s':\n%s\n"), dict->filename,
						      error->message);
			gtk_dialog_run (GTK_DIALOG (msg));
			gtk_widget_destroy (msg);
			g_error_free (error);
			error = NULL;
		}
	}
	else
		save_as_file_cb (NULL, dict);	
}

static void
save_as_file_cb (GtkWidget *wid, MainConfig *dict)
{
	GtkWidget *file_selector;
	gint response;

	/* Create the selector */
	file_selector = gtk_file_selection_new (_("Select a file to save to"));
	response = gtk_dialog_run (GTK_DIALOG (file_selector));
	if (response == GTK_RESPONSE_OK) {
		set_filename (dict, gtk_file_selection_get_filename (GTK_FILE_SELECTION (file_selector)));
		save_file_cb (NULL, dict);
	}
	gtk_widget_destroy (file_selector);	
}

static gboolean conn_open (MainConfig *config);
static void stop_dbms_update_cb (GtkWidget *dlg, gint response, MainConfig *config);
static void
sync_action_cb (GtkWidget *wid, MainConfig *config)
{
	GtkWidget *dlg, *updview;
	GnomeDbDatabase *db;
	GnomeDbServer *srv;
	GError *error = NULL;

	db = gnome_db_dict_get_database (config->dict);
	srv = gnome_db_dict_get_server (config->dict);
	if (!conn_open (config)) {
		g_warning ("Can't open connection!\n");
		return;
	}

	dlg = gtk_dialog_new_with_buttons (_("Metadata synchronisation"),
					   GTK_WINDOW (config->mainwin), 
					   GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
					   GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, NULL);
	g_signal_connect (G_OBJECT (dlg), "response",
			  G_CALLBACK (stop_dbms_update_cb), config);
	updview = gnome_db_dbms_update_viewer_new (config->dict);
	gtk_widget_show (updview);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG(dlg)->vbox), updview, TRUE, TRUE, 0);
	gtk_widget_show (dlg);
	/* data types, functions and aggregates */
	if (!gnome_db_server_update_dbms_data (srv, &error)) {
		if (error->code != GNOME_DB_SERVER_META_DATA_UPDATE_USER_STOPPED) {
			GtkWidget *msg;
			
			msg = gtk_message_dialog_new (GTK_WINDOW (config->mainwin), 
						      GTK_DIALOG_DESTROY_WITH_PARENT,
						      GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE, 
						      _("Error updating Server metadata:\n%s\n"), error->message);
			gtk_dialog_run (GTK_DIALOG (msg));
			gtk_widget_destroy (msg);
		}
		g_error_free (error);
		error = NULL;	
	}
	/* database */
	if (!gnome_db_database_update_dbms_data (db, &error)) {
		if (error->code != GNOME_DB_DATABASE_META_DATA_UPDATE_USER_STOPPED) {
			GtkWidget *msg;
			
			msg = gtk_message_dialog_new (GTK_WINDOW (config->mainwin), 
						      GTK_DIALOG_DESTROY_WITH_PARENT,
						      GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE, 
						      _("Error updating Database metadata:\n%s\n"), 
						      error->message);
			gtk_dialog_run (GTK_DIALOG (msg));
			gtk_widget_destroy (msg);
		}
		g_error_free (error);
		error = NULL;	
	}

	signal_tests (config);
	gtk_widget_destroy (dlg);
}

static void
stop_dbms_update_cb (GtkWidget *dlg, gint response, MainConfig *config)
{
	gnome_db_server_stop_update_dbms_data (gnome_db_dict_get_server (config->dict));
}

static void
select_ds_cb (GtkWidget *wid, MainConfig *dict)
{
	GtkWidget *props;
                                                                                                                    
        props = gnome_db_login_dialog_new (_("Connection's configuration"));
        if (gnome_db_login_dialog_run (GNOME_DB_LOGIN_DIALOG (props))) {
		GnomeDbServer *srv;
		
		srv = gnome_db_dict_get_server (dict->dict);
		gnome_db_server_set_datasource (srv, gnome_db_login_dialog_get_dsn (GNOME_DB_LOGIN_DIALOG (props)));
		if (!gnome_db_login_dialog_get_username (GNOME_DB_LOGIN_DIALOG (props)))
			gnome_db_server_set_user_name (srv, "");
		else
			gnome_db_server_set_user_name (srv, gnome_db_login_dialog_get_username (GNOME_DB_LOGIN_DIALOG (props)));
		gnome_db_server_set_user_password (srv, gnome_db_login_dialog_get_password (GNOME_DB_LOGIN_DIALOG (props)));
		gtk_widget_destroy (props);
		sync_action_cb (NULL, dict);
	}
	else
		gtk_widget_destroy (props);
}


static gboolean 
conn_open (MainConfig *config)
{
	GnomeDbServer *srv;
	GError *error = NULL;

	srv = gnome_db_dict_get_server (config->dict);
	if (!gnome_db_server_conn_is_opened (srv)) {
		if (!gnome_db_server_open_connect (srv, &error)) {
			GtkWidget *msg;
			
			msg = gtk_message_dialog_new (GTK_WINDOW (config->mainwin), 
						      GTK_DIALOG_DESTROY_WITH_PARENT,
						      GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, 
						      _("Error opening the connection '%s':\n%s\n"), 
						      config->filename,
						      error->message);
			gtk_dialog_run (GTK_DIALOG (msg));
			gtk_widget_destroy (msg);
			g_error_free (error);
			error = NULL;	
			return FALSE;
		}
	}

	return TRUE;
}

static void
open_conn_cb (GtkWidget *wid, MainConfig *dict)
{
	if (!conn_open (dict)) {
		g_warning ("Can't open connection!\n");
		return;
	}
}



/*
 * Main page building
 */
static void selection_changed_cb (GnomeDbSelector *mgsel, GObject *obj, MainConfig *config);
static void left_del_clicked_cb (GtkWidget *button, MainConfig *config);
static void right_del_clicked_cb (GtkWidget *button, MainConfig *config);
static GtkWidget *
build_page (MainConfig *dict)
{
	GtkWidget *wid, *label, *paned, *vb, *bb, *button, *nb;
	gchar *str;
	ATest *test = dict->tests->data;

	paned = gtk_hpaned_new ();
	gtk_container_set_border_width (GTK_CONTAINER (paned), 5);

	/* left part */
	vb = gtk_vbox_new (FALSE, 5);
	gtk_container_set_border_width (GTK_CONTAINER (vb), 5);
	gtk_paned_add1 (GTK_PANED (paned), vb);

	label = gtk_label_new (NULL);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
	str = g_strdup_printf ("<b>%s</b>", _("Available dictionnary objects:"));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_box_pack_start (GTK_BOX (vb), label, FALSE, TRUE, 0);

	wid = gnome_db_selector_new (dict->dict, NULL, 
			       GNOME_DB_SELECTOR_TABLES | GNOME_DB_SELECTOR_FIELDS |
			       /* GNOME_DB_SELECTOR_FUNCTIONS | */
			       GNOME_DB_SELECTOR_QUERIES | GNOME_DB_SELECTOR_QVIS_FIELDS |
			       GNOME_DB_SELECTOR_GRAPHS |
			       GNOME_DB_SELECTOR_FORMS, 0);
	gnome_db_selector_set_headers_visible (GNOME_DB_SELECTOR (wid), FALSE);
	gtk_box_pack_start (GTK_BOX (vb), wid, TRUE, TRUE, 0);
	g_signal_connect (G_OBJECT (wid), "selection_changed",
			  G_CALLBACK (selection_changed_cb), dict);
	gtk_widget_set_size_request (wid, 200, 300);
	dict->selector = wid;

	bb = gtk_vbutton_box_new ();
	gtk_box_pack_start (GTK_BOX (vb), bb, FALSE, FALSE, 0);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bb), GTK_BUTTONBOX_SPREAD);
	button = gtk_button_new_from_stock (GTK_STOCK_DELETE);
	gtk_container_add (GTK_CONTAINER (bb), button);
	dict->del_obj_button = button;
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (left_del_clicked_cb), dict);

	/* right part */
	vb = gtk_vbox_new (FALSE, 5);
	gtk_container_set_border_width (GTK_CONTAINER (vb), 5);
	gtk_paned_add2 (GTK_PANED (paned), vb);

	label = gtk_label_new (NULL);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
	str = g_strdup_printf ("<b>%s</b>", _("Tested widget:"));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_box_pack_start (GTK_BOX (vb), label, FALSE, FALSE, 0);

	nb = gtk_notebook_new ();
	dict->tests_nb = nb;
	gtk_box_pack_start (GTK_BOX (vb), nb, TRUE, TRUE, 0);
	gtk_widget_show (nb);

	bb = gtk_hbutton_box_new ();
	gtk_box_pack_start (GTK_BOX (vb), bb, FALSE, FALSE, 0);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bb), GTK_BUTTONBOX_SPREAD);
	button = gtk_button_new_from_stock (GTK_STOCK_DELETE);
	dict->del_test_button = button;
	gtk_container_add (GTK_CONTAINER (bb), button);
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (right_del_clicked_cb), dict);

	/* start with the first test */
	add_test_page (dict, test);

	return paned;
}

/*
 * utility function to provide a new GnomeDbResultSet
 *
 * Returns: NULL if an error occured, or if the user did not enter any required parameter
 */
static GnomeDbResultSet *
get_ready_result_set_from_query (GnomeDbQuery *query, const gchar *ident)
{
	GnomeDbResultSet *rs = NULL;

	if (IS_GNOME_DB_QUERY (query) && gnome_db_query_is_select_query (GNOME_DB_QUERY (query))) {
		GnomeDbDataSet *dset;

		rs = GNOME_DB_RESULT_SET (gnome_db_result_set_new (query, NULL));
		dset = gnome_db_data_model_get_params (GNOME_DB_DATA_MODEL (rs));
		if (dset && gnome_db_data_set_needs_user_input (dset)) {
			GtkWidget *dlg;
			gint status;
			gchar *str;
			GnomeDbDict *dict;

			dict = gnome_db_base_get_dict (GNOME_DB_BASE (query));
			str = g_strdup_printf (_("%s: parameters required"), ident);
			dlg = gnome_db_basic_form_new_in_dialog (dict, dset, NULL, str,
								 _("Some parameters are required to execute this query"));
			g_free (str);
			status = gtk_dialog_run (GTK_DIALOG (dlg));
			gtk_widget_destroy (dlg);
			if (status == GTK_RESPONSE_REJECT) {
				g_object_unref (rs);
				rs = NULL;
			}
		}
		if (rs)
			gnome_db_data_model_refresh  (GNOME_DB_DATA_MODEL (rs), NULL);
	}

	return rs;
}


/* Add a new tested widget */
static void
add_test_page (MainConfig *config, ATest *test)
{
	GtkWidget *wid, *label;
	gint page;

	if (!test)
		return;

	wid = (test->make_test_widget) (config);
	g_object_set_data (G_OBJECT (wid), "atest", test);
	label = gtk_label_new (test->test_name);
	page = gtk_notebook_append_page (GTK_NOTEBOOK (config->tests_nb), wid, label);
	gtk_widget_show (wid);
	gtk_widget_show (label);
	gtk_widget_set_sensitive (config->del_test_button, TRUE);
	gtk_notebook_set_current_page (GTK_NOTEBOOK (config->tests_nb), page);
}

static void
selection_changed_cb (GnomeDbSelector *mgsel, GObject *obj, MainConfig *config)
{
	gtk_widget_set_sensitive (config->del_obj_button, obj ? TRUE : FALSE);
}

static void
left_del_clicked_cb (GtkWidget *button, MainConfig *config)
{
	GObject *obj = gnome_db_selector_get_selected_object (GNOME_DB_SELECTOR (config->selector));

	if (obj) 
		gnome_db_base_nullify (GNOME_DB_BASE (obj));
}

static void
right_del_clicked_cb (GtkWidget *button, MainConfig *config)
{
	GtkWidget *page;
	ATest *test;
	gint pageno = gtk_notebook_get_current_page (GTK_NOTEBOOK (config->tests_nb));

	page = gtk_notebook_get_nth_page (GTK_NOTEBOOK (config->tests_nb), pageno);
	test = g_object_get_data (G_OBJECT (page), "atest");
	if (test->clean_data) 
		(test->clean_data) (config, page);		
	gtk_notebook_remove_page (GTK_NOTEBOOK (config->tests_nb), pageno);
	gtk_widget_set_sensitive (config->del_test_button,
				  gtk_notebook_get_n_pages (GTK_NOTEBOOK (config->tests_nb)) != 0);
}








/*
 * GnomeDbMatrix test
 */
typedef struct {
	gboolean    with_data;
	GtkWidget  *vbox;
	GtkWidget  *wid;
	GtkWidget  *toggle_view_buttons[3];
} MatrixTestData;
#define MATRIX_TEST_DATA(x) ((MatrixTestData *) (x))

static void matrix_test_real_make (MainConfig *config, MatrixTestData *data);
static GtkWidget *
matrix_test_make_without_data (MainConfig *config)
{
	GtkWidget *ret;
	MatrixTestData *data;

	ret = gtk_vbox_new (FALSE, 0);

	/* private test data */
	data = g_new0 (MatrixTestData, 1);
	data->with_data = FALSE;
	data->vbox = ret;
	g_object_set_data (G_OBJECT (ret), "private_data", data);
	matrix_test_real_make (config, data);

	return ret;
}

static GtkWidget *
matrix_test_make_with_data (MainConfig *config)
{
	GtkWidget *ret;
	MatrixTestData *data;

	ret = gtk_vbox_new (FALSE, 0);

	/* private test data */
	data = g_new0 (MatrixTestData, 1);
	data->with_data = TRUE;
	data->vbox = ret;
	g_object_set_data (G_OBJECT (ret), "private_data", data);
	matrix_test_real_make (config, data);

	return ret;
}

static void matrix_test_view_type_toggled_cb (GtkToggleButton *button, MatrixTestData *data);
static void matrix_test_data_only_toggled_cb (GtkToggleButton *button, MatrixTestData *data);
static void
matrix_test_real_make (MainConfig *config, MatrixTestData *data)
{
	GtkWidget *wid, *bbox;
	GnomeDbQuery *query_rows, *query_cols;
	GnomeDbTable *table;
	GnomeDbTableField *field1, *field2;

	g_assert (data);

	if (!data->toggle_view_buttons[GNOME_DB_MATRIX_TABULAR_SYNTHETIC]) {
		/* buttons to choose the type of view */
		bbox = gtk_hbutton_box_new ();
		gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox), GTK_BUTTONBOX_SPREAD);
		gtk_box_pack_start (GTK_BOX (data->vbox), bbox, FALSE, FALSE, 0);
		gtk_widget_show (bbox);
		
		wid = gtk_toggle_button_new_with_label ("Tabular");
		gtk_box_pack_start (GTK_BOX (bbox), wid, TRUE, TRUE, 0);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (wid), TRUE);
		gtk_widget_show (wid);
		data->toggle_view_buttons[GNOME_DB_MATRIX_TABULAR_SYNTHETIC] = wid;
		g_signal_connect (G_OBJECT (wid), "toggled",
				  G_CALLBACK (matrix_test_view_type_toggled_cb), data);
		
		wid = gtk_toggle_button_new_with_label ("Synthetic list");
		gtk_box_pack_start (GTK_BOX (bbox), wid, TRUE, TRUE, 0);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (wid), FALSE);
		gtk_widget_show (wid);
		data->toggle_view_buttons[GNOME_DB_MATRIX_LIST_SYNTHETIC] = wid;
		g_signal_connect (G_OBJECT (wid), "toggled",
				  G_CALLBACK (matrix_test_view_type_toggled_cb), data);
		
		wid = gtk_toggle_button_new_with_label ("Detailled list");
		gtk_box_pack_start (GTK_BOX (bbox), wid, TRUE, TRUE, 0);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (wid), FALSE);
		gtk_widget_show (wid);
		data->toggle_view_buttons[GNOME_DB_MATRIX_LIST_DETAILLED] = wid;
		g_signal_connect (G_OBJECT (wid), "toggled",
				  G_CALLBACK (matrix_test_view_type_toggled_cb), data);
		
		if (data->with_data) {
			wid = gtk_check_button_new_with_label ("Assoc data only");
			gtk_box_pack_start (GTK_BOX (bbox), wid, TRUE, TRUE, 0);
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (wid), FALSE);
			gtk_widget_show (wid);
			g_signal_connect (G_OBJECT (wid), "toggled",
					  G_CALLBACK (matrix_test_data_only_toggled_cb), data);
		}
	}

	field1 = gnome_db_database_get_field_by_name (gnome_db_dict_get_database (config->dict), 
						      "order_contents.quantity");
	field2 = gnome_db_database_get_field_by_name (gnome_db_dict_get_database (config->dict), 
						      "order_contents.discount");

	query_rows = GNOME_DB_QUERY (gnome_db_query_new_from_sql (config->dict, SQL_QUERY_ROWS, NULL));
	query_cols = GNOME_DB_QUERY (gnome_db_query_new_from_sql (config->dict, SQL_QUERY_COLS, NULL));
	table = gnome_db_database_get_table_by_name (gnome_db_dict_get_database (config->dict), "order_contents");

	if (table && field1 && field2 &&
	    (gnome_db_query_get_query_type (query_rows) == GNOME_DB_QUERY_TYPE_SELECT) &&
	    (gnome_db_query_get_query_type (query_cols) == GNOME_DB_QUERY_TYPE_SELECT)) {
		/* open the connexion if not opened */
		if (gnome_db_server_open_connect (gnome_db_dict_get_server (config->dict), NULL)) {
			GnomeDbTarget *t1, *t2;
			GSList *additional_fields = NULL;
			guint mode = GNOME_DB_ACTION_ASK_CONFIRM_UPDATE |
				GNOME_DB_ACTION_ASK_CONFIRM_DELETE | GNOME_DB_ACTION_ASK_CONFIRM_INSERT |
				GNOME_DB_ACTION_REPORT_ERROR;

			t1 = gnome_db_query_get_target_by_alias (query_rows, "o");
			t2 = gnome_db_query_get_target_by_alias (query_cols, "p");

			additional_fields = g_slist_append (NULL, field1);
			additional_fields = g_slist_append (additional_fields, field2);

			wid = gnome_db_matrix_new (config->dict, query_rows, t1, query_cols, t2,
						  table, 
						  data->with_data ? additional_fields : NULL);
			/* gnome_db_data_widget_run (GNOME_DB_DATA_WIDGET (wid), mode); */

			g_slist_free (additional_fields);
		}
		else 
			wid = gtk_label_new (_("Can't open connection"));
	}
	else 
		wid = gtk_label_new (_("Missing some data for this test"));

	data->wid = wid;
	gtk_box_pack_start (GTK_BOX (data->vbox), wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);

	g_object_unref (G_OBJECT (query_rows));
	g_object_unref (G_OBJECT (query_cols));
}

static void
matrix_test_view_type_toggled_cb (GtkToggleButton *button, MatrixTestData *data)
{
	if (gtk_toggle_button_get_active (button)) {
		gint i;
		for (i=0; i<3; i++) {
			if (data->toggle_view_buttons[i] == (GtkWidget *) button) {
				if (IS_GNOME_DB_MATRIX (data->wid))
					gnome_db_matrix_set_view_type (GNOME_DB_MATRIX (data->wid), i);
			}
			else
				gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->toggle_view_buttons[i]), 
							      FALSE);
		}
	}
}

static void 
matrix_test_data_only_toggled_cb (GtkToggleButton *button, MatrixTestData *data)
{
	if (IS_GNOME_DB_MATRIX (data->wid))
		g_object_set (G_OBJECT (data->wid), "assoc_data_only", gtk_toggle_button_get_active (button), NULL);
	else
		gtk_toggle_button_set_active (button, FALSE);
}

static void
matrix_test_clean (MainConfig *config, GtkWidget *test_widget)
{
	MatrixTestData *data = g_object_get_data (G_OBJECT (test_widget), "private_data");

	if (data) 
		g_free (data);
}

static gboolean
matrix_test_signal (MainConfig *config, GtkWidget *test_widget)
{
	MatrixTestData *data = g_object_get_data (G_OBJECT (test_widget), "private_data");

	if (data->wid) {
		gtk_widget_destroy (data->wid);
		data->wid = NULL;
	}

	matrix_test_real_make (config, data);

	return TRUE;
}







/*
 * GnomeDbCanvasDbRelations test
 */
typedef struct {
	GtkWidget  *vbox;
	GtkWidget  *canvas;
	gulong      signal_handler;
} RelsGraphTestData;
#define RELS_GRAPH_TEST_DATA(x) ((RelsGraphTestData *) (x))

static void rels_graph_test_graph_changed (GnomeDbSelector *mgsel, GObject *obj, RelsGraphTestData *data);
static GtkWidget *
rels_graph_test_make (MainConfig *config)
{
	GtkWidget *ret, *wid = NULL;
	RelsGraphTestData *data;
	
	ret = gtk_vbox_new (FALSE, 0);
	wid = gnome_db_canvas_db_relations_new (config->dict, NULL);
	gtk_box_pack_start (GTK_BOX (ret), wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);

	/* private test data */
	data = g_new0 (RelsGraphTestData, 1);
	data->vbox = ret;
	data->canvas = wid;
	data->signal_handler = g_signal_connect (G_OBJECT (config->selector), "selection_changed",
						 G_CALLBACK (rels_graph_test_graph_changed), data);
	g_object_set_data (G_OBJECT (ret), "private_data", data);

	return ret;
}

static void
rels_graph_test_graph_changed (GnomeDbSelector *mgsel, GObject *obj, RelsGraphTestData *data)
{
	if (IS_GNOME_DB_GRAPH (obj))
		g_object_set (G_OBJECT (data->canvas), "graph", obj, NULL);
}

static void
rels_graph_test_clean (MainConfig *config, GtkWidget *test_widget)
{
	RelsGraphTestData *data = g_object_get_data (G_OBJECT (test_widget), "private_data");

	if (data) {
		g_signal_handler_disconnect (G_OBJECT (config->selector), data->signal_handler);
		g_free (data);
	}
}

static gboolean
rels_graph_test_signal (MainConfig *config, GtkWidget *test_widget)
{
	GObject *obj;
	RelsGraphTestData *data = g_object_get_data (G_OBJECT (test_widget), "private_data");

	obj = gnome_db_selector_get_selected_object (GNOME_DB_SELECTOR (config->selector));
	if (IS_GNOME_DB_GRAPH (obj))
		g_object_set (G_OBJECT (data->canvas), "graph", obj, NULL);

	return TRUE;
}



/*
 * GnomeDbCanvasQueryStruct test
 */
typedef struct {
	GtkWidget  *vbox;
	GtkWidget  *canvas;
	gulong      signal_handler;
} QueryGraphTestData;
#define QUERY_GRAPH_TEST_DATA(x) ((QueryGraphTestData *) (x))

static void query_graph_test_graph_changed (GnomeDbSelector *mgsel, GObject *obj, QueryGraphTestData *data);
static GtkWidget *
query_graph_test_make (MainConfig *config)
{
	GtkWidget *ret;
	QueryGraphTestData *data;
	
	ret = gtk_vbox_new (FALSE, 0);

	/* private test data */
	data = g_new0 (QueryGraphTestData, 1);
	data->vbox = ret;
	data->canvas = NULL;
	data->signal_handler = g_signal_connect (G_OBJECT (config->selector), "selection_changed",
						 G_CALLBACK (query_graph_test_graph_changed), data);
	g_object_set_data (G_OBJECT (ret), "private_data", data);

	return ret;
}

static void
query_graph_test_graph_changed (GnomeDbSelector *mgsel, GObject *obj, QueryGraphTestData *data)
{
	if (IS_GNOME_DB_QUERY (obj)) {
		GtkWidget *wid;
		GnomeDbDict *dict;
		GnomeDbGraph *graph;

		dict = gnome_db_base_get_dict (GNOME_DB_BASE (obj));
		graph = gnome_db_dict_get_graph_for_object (dict, obj);
		if (!graph) {
			/* the query did not have any graph: create one */
			graph = GNOME_DB_GRAPH (gnome_db_graph_query_new (GNOME_DB_QUERY (obj)));
			gnome_db_dict_assume_graph (dict, graph);
			g_object_unref (graph);
		}
		if (data->canvas)
			gtk_widget_destroy (data->canvas);
		
		wid = gnome_db_canvas_query_struct_new (GNOME_DB_QUERY (obj), graph);
		gtk_box_pack_start (GTK_BOX (data->vbox), wid, TRUE, TRUE, 0);
		gtk_widget_show (wid);
		data->canvas = wid;
	}
}

static void
query_graph_test_clean (MainConfig *config, GtkWidget *test_widget)
{
	QueryGraphTestData *data = g_object_get_data (G_OBJECT (test_widget), "private_data");

	if (data) {
		g_signal_handler_disconnect (G_OBJECT (config->selector), data->signal_handler);
		g_free (data);
	}
}

static gboolean
query_graph_test_signal (MainConfig *config, GtkWidget *test_widget)
{
	GObject *obj;
	QueryGraphTestData *data = g_object_get_data (G_OBJECT (test_widget), "private_data");

	obj = gnome_db_selector_get_selected_object (GNOME_DB_SELECTOR (config->selector));
	query_graph_test_graph_changed (GNOME_DB_SELECTOR (config->selector), obj, data);

	return TRUE;
}





/*
 * GnomeDbForm and GnomeDbGrid tests
 */
typedef struct {
	GtkWidget  *vbox;
	GtkWidget  *nodata;
	GtkWidget  *wid;
	gulong      signal_handler;
	gboolean    is_grid;
} CoreTestData;
#define WORK_CORE_TEST_DATA(x) ((CoreTestData *) (x))

static void work_core_test_graph_changed (GnomeDbSelector *mgsel, GObject *obj, CoreTestData *data);

static GtkWidget *
work_core_test_make_form (MainConfig *config)
{
	GtkWidget *ret, *wid = NULL;
	CoreTestData *data;
	
	ret = gtk_vbox_new (FALSE, 0);
	wid = gtk_label_new ("Select a table");
	gtk_box_pack_start (GTK_BOX (ret), wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);

	/* private test data */
	data = g_new0 (CoreTestData, 1);
	data->vbox = ret;
	data->nodata = wid;
	data->wid = NULL;
	data->is_grid = FALSE;
	data->signal_handler = g_signal_connect (G_OBJECT (config->selector), "selection_changed",
						 G_CALLBACK (work_core_test_graph_changed), data);
	g_object_set_data (G_OBJECT (ret), "private_data", data);
	work_core_test_signal (config, ret);

	return ret;
}
static GtkWidget *
work_core_test_make_grid (MainConfig *config)
{
	GtkWidget *ret, *wid = NULL;
	CoreTestData *data;
	
	ret = gtk_vbox_new (FALSE, 0);
	wid = gtk_label_new ("Select a table");
	gtk_box_pack_start (GTK_BOX (ret), wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);

	/* private test data */
	data = g_new0 (CoreTestData, 1);
	data->vbox = ret;
	data->nodata = wid;
	data->wid = NULL;
	data->is_grid = TRUE;
	data->signal_handler = g_signal_connect (G_OBJECT (config->selector), "selection_changed",
						 G_CALLBACK (work_core_test_graph_changed), data);
	g_object_set_data (G_OBJECT (ret), "private_data", data);
	work_core_test_signal (config, ret);

	return ret;	
}

static void
work_core_test_graph_changed (GnomeDbSelector *mgsel, GObject *obj, CoreTestData *data)
{
	if (data->wid) {
		gtk_widget_destroy (data->wid);
		data->wid = NULL;
		gtk_widget_show (data->nodata);
	}

	if (IS_GNOME_DB_TABLE (obj)) {
		gchar *str;
		GnomeDbQuery *query;
		GnomeDbDict *dict = gnome_db_base_get_dict (GNOME_DB_BASE (obj));

		str = g_strdup_printf ("SELECT * FROM %s", gnome_db_base_get_name (GNOME_DB_BASE (obj)));
		query = GNOME_DB_QUERY (gnome_db_query_new_from_sql (dict, str, NULL));
		g_free (str);
		if (gnome_db_query_get_query_type (query) == GNOME_DB_QUERY_TYPE_SELECT) {
			GnomeDbTarget *target = gnome_db_query_get_target_by_alias (query, 
						gnome_db_base_get_name (GNOME_DB_BASE (obj)));
			GtkWidget *wid;
			guint mode = GNOME_DB_ACTION_ASK_CONFIRM_DELETE |
				GNOME_DB_ACTION_ASK_CONFIRM_UPDATE |
				GNOME_DB_ACTION_ASK_CONFIRM_INSERT |
				GNOME_DB_ACTION_NAVIGATION_ARROWS | GNOME_DB_ACTION_NAVIGATION_SCROLL;
			
			if (data->is_grid)
				wid = gnome_db_grid_new_with_select_query (query, target);
			else {
				wid = gnome_db_form_new_with_select_query (query, target);
				gnome_db_data_widget_column_show_actions (GNOME_DB_DATA_WIDGET (wid), 0, TRUE);
			}
			
			gnome_db_data_widget_set_mode (GNOME_DB_DATA_WIDGET (wid), mode);
			gtk_box_pack_start (GTK_BOX (data->vbox), wid, TRUE, TRUE, 0);
			gtk_widget_show (wid);
			data->wid = wid;
			gtk_widget_hide (data->nodata);
		}
		else
			gtk_widget_show (data->nodata);
		g_object_unref (G_OBJECT (query));
	}
	
	if (IS_GNOME_DB_QUERY (obj) && gnome_db_query_is_select_query (GNOME_DB_QUERY (obj))) {
		GSList *targets;
		GnomeDbTarget *target = NULL;
		GnomeDbQuery *query = GNOME_DB_QUERY (obj);
		GtkWidget *wid;
		GnomeDbDataSet *dset;
		gboolean cancelled = FALSE;
		guint mode = GNOME_DB_ACTION_ASK_CONFIRM_DELETE |
			GNOME_DB_ACTION_ASK_CONFIRM_UPDATE |
			GNOME_DB_ACTION_ASK_CONFIRM_INSERT |
			GNOME_DB_ACTION_NAVIGATION_ARROWS | GNOME_DB_ACTION_NAVIGATION_SCROLL;
		
		targets = gnome_db_query_get_targets (query);
		if (targets) {
			target = GNOME_DB_TARGET (targets->data);
			g_slist_free (targets);
		}

		if (data->is_grid)
			wid = gnome_db_grid_new_with_select_query (query, target);
		else {
			wid = gnome_db_form_new_with_select_query (query, target);
			gnome_db_data_widget_column_show_actions (GNOME_DB_DATA_WIDGET (wid), 0, TRUE);
		}

		/* refresh the contents of the widget */
		dset = gnome_db_data_widget_get_params (GNOME_DB_DATA_WIDGET (wid));
		if (dset && gnome_db_data_set_needs_user_input (dset)) {
			GtkWidget *dlg;
			gint status;
			gchar *str;
			GnomeDbDict *dict;
			
			dict = gnome_db_base_get_dict (GNOME_DB_BASE (query));
			if (data->is_grid)
				str =_("Grid: parameters required");
			else
				str =_("Form: parameters required");
			dlg = gnome_db_basic_form_new_in_dialog (dict, dset, NULL, str,
								 _("Some parameters are required to execute this query"));
			status = gtk_dialog_run (GTK_DIALOG (dlg));
			gtk_widget_destroy (dlg);
			if (status == GTK_RESPONSE_REJECT) 
				cancelled = TRUE;
		}

		if (cancelled) {
			gtk_widget_destroy (wid);
		}
		else {
			gnome_db_data_widget_set_mode (GNOME_DB_DATA_WIDGET (wid), mode);
			gtk_box_pack_start (GTK_BOX (data->vbox), wid, TRUE, TRUE, 0);
			gtk_widget_show (wid);
			data->wid = wid;
			gtk_widget_hide (data->nodata);
		}
	}
}
static void
work_core_test_clean (MainConfig *config, GtkWidget *test_widget)
{
	CoreTestData *data = g_object_get_data (G_OBJECT (test_widget), "private_data");
	
	if (data) {
		g_signal_handler_disconnect (G_OBJECT (config->selector), data->signal_handler);
		g_free (data);
	}
}

static gboolean
work_core_test_signal (MainConfig *config, GtkWidget *test_widget)
{
	GObject *obj;
	CoreTestData *data = g_object_get_data (G_OBJECT (test_widget), "private_data");

	obj = gnome_db_selector_get_selected_object (GNOME_DB_SELECTOR (config->selector));
	work_core_test_graph_changed (NULL, obj, data);

	return TRUE;
}



/*
 * GnomeDbLayout test
 */
typedef struct {
	GtkWidget  *vbox;
	GtkWidget  *nodata;
	GtkWidget  *wid;
	gulong      signal_handler;
} FormsTestData;
#define FORMS_TEST_DATA(x) ((FormsTestData *) (x))

static void forms_test_changed (GnomeDbSelector *mgsel, GObject *obj, FormsTestData *data);
static GtkWidget *
forms_test_make (MainConfig *config)
{
	GtkWidget *ret, *wid = NULL;
	FormsTestData *data;
	
	ret = gtk_vbox_new (FALSE, 0);
	wid = gtk_label_new ("Select a Form");
	gtk_box_pack_start (GTK_BOX (ret), wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);

	/* private test data */
	data = g_new0 (FormsTestData, 1);
	data->vbox = ret;
	data->nodata = wid;
	data->wid = NULL;
	data->signal_handler = g_signal_connect (G_OBJECT (config->selector), "selection_changed",
						 G_CALLBACK (forms_test_changed), data);
	g_object_set_data (G_OBJECT (ret), "private_data", data);
	forms_test_signal (config, ret);

	return ret;
}

static void
forms_test_changed (GnomeDbSelector *mgsel, GObject *obj, FormsTestData *data)
{
	if (data->wid) {
		gtk_widget_destroy (data->wid);
		data->wid = NULL;
		gtk_widget_show (data->nodata);
	}

	if (IS_GNOME_DB_CUSTOM_LAYOUT (obj)) {
		GtkWidget *wid;

		wid = gnome_db_layout_new (GNOME_DB_CUSTOM_LAYOUT (obj));
		/* gnome_db_data_widget_run (GNOME_DB_DATA_WIDGET (wid), 0); */
		gtk_box_pack_start (GTK_BOX (data->vbox), wid, TRUE, TRUE, 0);
		gtk_widget_show (wid);
		data->wid = wid;
		gtk_widget_hide (data->nodata);

#ifdef debug
		gnome_db_base_dump (GNOME_DB_BASE (obj), 0);
#endif
	}
}

static void
forms_test_clean (MainConfig *config, GtkWidget *test_widget)
{
	FormsTestData *data = g_object_get_data (G_OBJECT (test_widget), "private_data");
	
	if (data) {
		g_signal_handler_disconnect (G_OBJECT (config->selector), data->signal_handler);
		g_free (data);
	}
}

static gboolean
forms_test_signal (MainConfig *config, GtkWidget *test_widget)
{
	GObject *obj;
	FormsTestData *data = g_object_get_data (G_OBJECT (test_widget), "private_data");

	obj = gnome_db_selector_get_selected_object (GNOME_DB_SELECTOR (config->selector));
	forms_test_changed (NULL, obj, data);

	return TRUE;	
}


/*
 * GnomeDbSelector test
 */
typedef struct {
	MainConfig *config;
	GtkWidget  *vbox;
	GtkWidget  *selector;
	gulong      signal_handler;
	guint       sel_mode;
	guint       sel_columns;
} SelectorTestData;
#define SELECTOR_TEST_DATA(x) ((SelectorTestData *) (x))

static void selector_test_mode_toggled_cb (GtkCheckButton *cb, SelectorTestData *data);
static void selector_test_changed (GnomeDbSelector *mgsel, GObject *obj, SelectorTestData *data);
static GtkWidget *
selector_test_make (MainConfig *config)
{
	GtkWidget *ret, *vb, *cb, *label, *hbox;
	SelectorTestData *data;
	
	/* private test data */
	data = g_new0 (SelectorTestData, 1);

	ret = gtk_table_new (10, 2, FALSE);

	/* toggle buttons for selector mode */
	label = gtk_label_new ("<b>Mode(s) for the\nGnomeDbSelector widget:</b>");
	gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
	gtk_table_attach (GTK_TABLE (ret), label, 0, 1, 0, 1, 0, 0, 0, 0);
	gtk_widget_show (label);

	vb = gtk_vbox_new (FALSE, 0);
	gtk_table_attach (GTK_TABLE (ret), vb, 0, 1, 1, 2, 0, GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
	gtk_widget_show (vb);

	cb = gtk_check_button_new_with_label ("Data types");
	g_object_set_data (G_OBJECT (cb), "mode", GUINT_TO_POINTER (GNOME_DB_SELECTOR_DATA_TYPES));
	gtk_box_pack_start (GTK_BOX (vb), cb, FALSE, FALSE, 0);
	gtk_widget_show (cb);
	g_signal_connect (G_OBJECT (cb), "toggled", 
			  G_CALLBACK (selector_test_mode_toggled_cb), data);


	cb = gtk_check_button_new_with_label ("Functions");
	g_object_set_data (G_OBJECT (cb), "mode", GUINT_TO_POINTER (GNOME_DB_SELECTOR_FUNCTIONS));
	gtk_box_pack_start (GTK_BOX (vb), cb, FALSE, FALSE, 0);
	gtk_widget_show (cb);
	g_signal_connect (G_OBJECT (cb), "toggled", 
			  G_CALLBACK (selector_test_mode_toggled_cb), data);

	cb = gtk_check_button_new_with_label ("Aggregates");
	g_object_set_data (G_OBJECT (cb), "mode", GUINT_TO_POINTER (GNOME_DB_SELECTOR_AGGREGATES));
	gtk_box_pack_start (GTK_BOX (vb), cb, FALSE, FALSE, 0);
	gtk_widget_show (cb);
	g_signal_connect (G_OBJECT (cb), "toggled", 
			  G_CALLBACK (selector_test_mode_toggled_cb), data);


	cb = gtk_check_button_new_with_label ("Tables");
	g_object_set_data (G_OBJECT (cb), "mode", GUINT_TO_POINTER (GNOME_DB_SELECTOR_TABLES));
	gtk_box_pack_start (GTK_BOX (vb), cb, FALSE, FALSE, 0);
	gtk_widget_show (cb);
	g_signal_connect (G_OBJECT (cb), "toggled", 
			  G_CALLBACK (selector_test_mode_toggled_cb), data);

	hbox = gtk_hbox_new (FALSE, 0); /* Offset */
	gtk_box_pack_start (GTK_BOX (vb), hbox, FALSE, FALSE, 0);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	cb = gtk_check_button_new_with_label ("Fields");
	g_object_set_data (G_OBJECT (cb), "mode", GUINT_TO_POINTER (GNOME_DB_SELECTOR_FIELDS));
	gtk_box_pack_start (GTK_BOX (hbox), cb, FALSE, FALSE, 0);
	gtk_widget_show (cb);
	g_signal_connect (G_OBJECT (cb), "toggled", 
			  G_CALLBACK (selector_test_mode_toggled_cb), data);


	cb = gtk_check_button_new_with_label ("Queries");
	g_object_set_data (G_OBJECT (cb), "mode", GUINT_TO_POINTER (GNOME_DB_SELECTOR_QUERIES));
	gtk_box_pack_start (GTK_BOX (vb), cb, FALSE, FALSE, 0);
	gtk_widget_show (cb);
	g_signal_connect (G_OBJECT (cb), "toggled", 
			  G_CALLBACK (selector_test_mode_toggled_cb), data);

	hbox = gtk_hbox_new (FALSE, 0); /* Offset */
	gtk_box_pack_start (GTK_BOX (vb), hbox, FALSE, FALSE, 0);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	vb = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), vb, FALSE, FALSE, 0);
	gtk_widget_show (vb);

	cb = gtk_check_button_new_with_label ("Targets");
	g_object_set_data (G_OBJECT (cb), "mode", GUINT_TO_POINTER (GNOME_DB_SELECTOR_TARGETS));
	gtk_box_pack_start (GTK_BOX (vb), cb, FALSE, FALSE, 0);
	gtk_widget_show (cb);
	g_signal_connect (G_OBJECT (cb), "toggled", 
			  G_CALLBACK (selector_test_mode_toggled_cb), data);

	cb = gtk_check_button_new_with_label ("Joins");
	g_object_set_data (G_OBJECT (cb), "mode", GUINT_TO_POINTER (GNOME_DB_SELECTOR_JOINS));
	gtk_box_pack_start (GTK_BOX (vb), cb, FALSE, FALSE, 0);
	gtk_widget_show (cb);
	g_signal_connect (G_OBJECT (cb), "toggled", 
			  G_CALLBACK (selector_test_mode_toggled_cb), data);

	cb = gtk_check_button_new_with_label ("Visible fields");
	g_object_set_data (G_OBJECT (cb), "mode", GUINT_TO_POINTER (GNOME_DB_SELECTOR_QVIS_FIELDS));
	gtk_box_pack_start (GTK_BOX (vb), cb, FALSE, FALSE, 0);
	gtk_widget_show (cb);
	g_signal_connect (G_OBJECT (cb), "toggled", 
			  G_CALLBACK (selector_test_mode_toggled_cb), data);

	cb = gtk_check_button_new_with_label ("All fields");
	g_object_set_data (G_OBJECT (cb), "mode", GUINT_TO_POINTER (GNOME_DB_SELECTOR_QALL_FIELDS));
	gtk_box_pack_start (GTK_BOX (vb), cb, FALSE, FALSE, 0);
	gtk_widget_show (cb);
	g_signal_connect (G_OBJECT (cb), "toggled", 
			  G_CALLBACK (selector_test_mode_toggled_cb), data);

	cb = gtk_check_button_new_with_label ("Sub queries");
	g_object_set_data (G_OBJECT (cb), "mode", GUINT_TO_POINTER (GNOME_DB_SELECTOR_SUB_QUERIES));
	gtk_box_pack_start (GTK_BOX (vb), cb, FALSE, FALSE, 0);
	gtk_widget_show (cb);
	g_signal_connect (G_OBJECT (cb), "toggled", 
			  G_CALLBACK (selector_test_mode_toggled_cb), data);


	/* vbox to hold the actual GnomeDbSelector widget */
	label = gtk_label_new ("<b>GnomeDbSelector widget:</b>");
	gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
	gtk_table_attach (GTK_TABLE (ret), label, 1, 2, 0, 1, 0, 0, 0, 0);
	gtk_widget_show (label);

	vb = gtk_vbox_new (FALSE, 0);
	gtk_table_attach_defaults (GTK_TABLE (ret), vb, 1, 2, 1, 2);
	gtk_widget_show (vb);

	/* private test data */
	data->config = config;
	data->vbox = vb;
	data->selector = NULL;
	data->signal_handler = g_signal_connect (G_OBJECT (config->selector), "selection_changed",
						 G_CALLBACK (selector_test_changed), data);
	data->sel_mode = 0;
	data->sel_columns = GNOME_DB_SELECTOR_COLUMN_OWNER |
		GNOME_DB_SELECTOR_COLUMN_COMMENTS |
		GNOME_DB_SELECTOR_COLUMN_TYPE |
		GNOME_DB_SELECTOR_COLUMN_FIELD_LENGTH |
		GNOME_DB_SELECTOR_COLUMN_FIELD_NNUL |
		GNOME_DB_SELECTOR_COLUMN_FIELD_DEFAULT |
		GNOME_DB_SELECTOR_COLUMN_QFIELD_VALUE |
		GNOME_DB_SELECTOR_COLUMN_QFIELD_TYPE;
	g_object_set_data (G_OBJECT (ret), "private_data", data);
	selector_test_signal (config, ret);

	return ret;
}

static void
selector_test_mode_toggled_cb (GtkCheckButton *cb, SelectorTestData *data)
{
	guint cb_mode = GPOINTER_TO_UINT (g_object_get_data (G_OBJECT (cb), "mode"));
	GObject *obj = NULL;

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (cb))) 
		data->sel_mode = data->sel_mode | cb_mode;
	else
		data->sel_mode = data->sel_mode & ~cb_mode;

	if (data->selector)
		obj = gnome_db_selector_get_selected_object (GNOME_DB_SELECTOR (data->config->selector));
	selector_test_changed (NULL, obj, data);
}

static void
selector_test_changed (GnomeDbSelector *mgsel, GObject *obj, SelectorTestData *data)
{
	GtkWidget *wid;

	/* don't care if the mode does not require the selection of an object */
	if (mgsel && 
	    ((data->sel_mode & GNOME_DB_SELECTOR_TABLES) ||
	     (data->sel_mode & GNOME_DB_SELECTOR_QUERIES) ||
	     (data->sel_mode & GNOME_DB_SELECTOR_DATA_TYPES) ||
	     (data->sel_mode & GNOME_DB_SELECTOR_FUNCTIONS) ||
	     (data->sel_mode & GNOME_DB_SELECTOR_AGGREGATES)))
	    return;

	if (data->selector) {
		gtk_widget_destroy (data->selector);
		data->selector = NULL;
	}

	wid = gnome_db_selector_new (data->config->dict, obj, data->sel_mode, data->sel_columns);
	gtk_box_pack_start (GTK_BOX (data->vbox), wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);
	data->selector = wid;
}

static void
selector_test_clean (MainConfig *config, GtkWidget *test_widget)
{
	SelectorTestData *data = g_object_get_data (G_OBJECT (test_widget), "private_data");
	
	if (data) {
		g_signal_handler_disconnect (G_OBJECT (config->selector), data->signal_handler);
		g_free (data);
	}
}

static gboolean
selector_test_signal (MainConfig *config, GtkWidget *test_widget)
{
	GObject *obj;
	SelectorTestData *data = g_object_get_data (G_OBJECT (test_widget), "private_data");

	obj = gnome_db_selector_get_selected_object (GNOME_DB_SELECTOR (config->selector));
	selector_test_changed (NULL, obj, data);

	return TRUE;	
}



/*
 * GnomeDbDataSourceSelector test
 */
static GtkWidget *
dsn_selector_test_make (MainConfig *config)
{
	GtkWidget *ret, *wid;
	
	ret = gtk_table_new (3, 3, FALSE);
	wid = gnome_db_data_source_selector_new (NULL);
	gtk_table_attach (GTK_TABLE (ret), wid, 1, 2, 1, 2, 0, 0, 0, 0);
	gtk_widget_show (wid);

	return ret;
}


/*
 * GnomeDbEditor test
 */
static GtkWidget *
editor_test_make (MainConfig *config)
{
	GtkWidget *ret;
	
	ret = gnome_db_editor_new ();
	gnome_db_editor_set_highlight (GNOME_DB_EDITOR (ret), TRUE);

	return ret;
}

/*
 * GnomeDbCombo test
 */
typedef struct {
	GtkWidget  *combo_vbox;
	GtkWidget  *combo;
	gulong      signal_handler;
} ComboTestData;
#define COMBO_TEST_DATA(x) ((ComboTestData *) (x))

static void combo_test_combo_changed (GnomeDbSelector *mgsel, GObject *obj, ComboTestData *data);
static GtkWidget *
combo_test_make (MainConfig *config)
{
	GtkWidget *ret, *wid;
	ComboTestData *data;
	
	ret = gtk_vbox_new (FALSE, 0);

	/* private test data */
	data = g_new0 (ComboTestData, 1);
	data->combo = NULL;
	data->signal_handler = g_signal_connect (G_OBJECT (config->selector), "selection_changed",
						 G_CALLBACK (combo_test_combo_changed), data);
	g_object_set_data (G_OBJECT (ret), "private_data", data);

	wid = gtk_label_new ("Combo box test, select a SELECT query");
	gtk_box_pack_start (GTK_BOX (ret), wid, FALSE, FALSE, 10);
	gtk_widget_show (wid);

	wid = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (ret), wid, FALSE, FALSE, 10);
	gtk_widget_show (wid);
	data->combo_vbox = wid;
	
	return ret;
}

static void
combo_test_combo_changed (GnomeDbSelector *mgsel, GObject *obj, ComboTestData *data)
{
	if (IS_GNOME_DB_QUERY (obj) && gnome_db_query_is_select_query (GNOME_DB_QUERY (obj))) {
		GtkWidget *wid;
		GnomeDbDict *dict;
		GnomeDbQuery *query = GNOME_DB_QUERY (obj);
		GnomeDbResultSet *rs;
		gint i, n_cols;
		gint *cols_index;

		dict = gnome_db_base_get_dict (GNOME_DB_BASE (obj));
		if (data->combo) {
			gtk_widget_destroy (data->combo);
			data->combo = NULL;
		}
		
		rs = get_ready_result_set_from_query (query, "Combo");
		if (!rs)
			return;

		n_cols = gda_data_model_get_n_columns (GDA_DATA_MODEL (rs));
		cols_index = g_new0 (gint, n_cols);
		for (i=0; i<n_cols; i++)
			cols_index [i] = i;
		wid = gnome_db_combo_new_with_model (dict, GDA_DATA_MODEL (rs), n_cols, cols_index);
		g_free (cols_index);
		g_object_unref (rs);

		gtk_box_pack_start (GTK_BOX (data->combo_vbox), wid, FALSE, FALSE, 0);
		gtk_widget_show (wid);
		data->combo = wid;
	}
}

static void
combo_test_clean (MainConfig *config, GtkWidget *test_widget)
{
	ComboTestData *data = g_object_get_data (G_OBJECT (test_widget), "private_data");

	if (data) {
		g_signal_handler_disconnect (G_OBJECT (config->selector), data->signal_handler);
		g_free (data);
	}
}

static gboolean
combo_test_signal (MainConfig *config, GtkWidget *test_widget)
{
	GObject *obj;
	ComboTestData *data = g_object_get_data (G_OBJECT (test_widget), "private_data");

	obj = gnome_db_selector_get_selected_object (GNOME_DB_SELECTOR (config->selector));
	combo_test_combo_changed (GNOME_DB_SELECTOR (config->selector), obj, data);

	return TRUE;
}
