/* gnome-db-table.c
 *
 * Copyright (C) 2003 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-table.h"
#include "gnome-db-database.h"
#include "gnome-db-table-field.h"
#include "gnome-db-constraint.h"
#include "gnome-db-xml-storage.h"
#include "gnome-db-field.h"
#include "gnome-db-entity.h"
#include "marshal.h"
#include "gnome-db-data-handler.h"
#include "gnome-db-server.h"
#include "gnome-db-server-data-type.h"
#include "gnome-db-result-set.h"
#include <string.h>
#include "gnome-db-ref-base.h"

/* 
 * Main static functions 
 */
static void gnome_db_table_class_init (GnomeDbTableClass * class);
static void gnome_db_table_init (GnomeDbTable * srv);
static void gnome_db_table_dispose (GObject   * object);
static void gnome_db_table_finalize (GObject   * object);

static void gnome_db_table_set_property (GObject              *object,
				    guint                 param_id,
				    const GValue         *value,
				    GParamSpec           *pspec);
static void gnome_db_table_get_property (GObject              *object,
				    guint                 param_id,
				    GValue               *value,
				    GParamSpec           *pspec);

/* XML storage interface */
static void        gnome_db_table_xml_storage_init (GnomeDbXmlStorageIface *iface);
static gchar      *gnome_db_table_get_xml_id (GnomeDbXmlStorage *iface);
static xmlNodePtr  gnome_db_table_save_to_xml (GnomeDbXmlStorage *iface, GError **error);
static gboolean    gnome_db_table_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error);

/* Entity interface */
static void             gnome_db_table_entity_init         (GnomeDbEntityIface *iface);
static gboolean         gnome_db_table_has_field           (GnomeDbEntity *iface, GnomeDbField *field);
static GSList          *gnome_db_table_get_fields          (GnomeDbEntity *iface);
static GnomeDbField    *gnome_db_table_get_field_by_name   (GnomeDbEntity *iface, const gchar *name);
static GnomeDbField    *gnome_db_table_get_field_by_xml_id (GnomeDbEntity *iface, const gchar *xml_id);
static GnomeDbField    *gnome_db_table_get_field_by_index  (GnomeDbEntity *iface, gint index);
static gint             gnome_db_table_get_field_index     (GnomeDbEntity *iface, GnomeDbField *field);
static void             gnome_db_table_add_field           (GnomeDbEntity *iface, GnomeDbField *field);
static void             gnome_db_table_add_field_before    (GnomeDbEntity *iface, GnomeDbField *field, GnomeDbField *field_before);
static void             gnome_db_table_swap_fields         (GnomeDbEntity *iface, GnomeDbField *field1, GnomeDbField *field2);
static void             gnome_db_table_remove_field        (GnomeDbEntity *iface, GnomeDbField *field);
static gboolean         gnome_db_table_is_writable         (GnomeDbEntity *iface);
static GSList          *gnome_db_table_get_parameters      (GnomeDbEntity *iface);


static void        gnome_db_table_set_database        (GnomeDbTable *table, GnomeDbDatabase *db);
static void        gnome_db_table_add_field_at_pos    (GnomeDbTable *table, GnomeDbTableField *field, gint pos);

/* When the Database is nullified */
static void        nullified_object_cb (GObject *obj, GnomeDbTable *table);
static void        nullified_field_cb  (GObject *obj, GnomeDbTable *table);
static void        nullified_parent_cb (GObject *obj, GnomeDbTable *table);

static void        changed_field_cb    (GObject *obj, GnomeDbTable *table);


#ifdef debug
static void        gnome_db_table_dump    (GnomeDbTable *table, guint offset);
#endif


/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	TEMPL_SIGNAL,
	LAST_SIGNAL
};

static gint gnome_db_table_signals[LAST_SIGNAL] = { 0 };

/* properties */
enum
{
	PROP_0,
	PROP_DB
};


/* private structure */
struct _GnomeDbTablePrivate
{
	GnomeDbDatabase *db;
	GSList     *fields;
	gboolean    is_view;
	GSList     *parents;     /* list of other GnomeDbTable objects which are parents */
	GHashTable *fields_hash; /* to improve fields retreival performances */
};
#define TABLE_SRV(table) (gnome_db_dict_get_server (gnome_db_base_get_dict (GNOME_DB_BASE (table))))


/* module error */
GQuark gnome_db_table_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_table_error");
	return quark;
}


guint
gnome_db_table_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbTableClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_table_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbTable),
			0,
			(GInstanceInitFunc) gnome_db_table_init
		};

		static const GInterfaceInfo xml_storage_info = {
			(GInterfaceInitFunc) gnome_db_table_xml_storage_init,
			NULL,
			NULL
		};

		static const GInterfaceInfo entity_info = {
			(GInterfaceInitFunc) gnome_db_table_entity_init,
			NULL,
			NULL
		};
		
		type = g_type_register_static (GNOME_DB_BASE_TYPE, "GnomeDbTable", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_XML_STORAGE_TYPE, &xml_storage_info);
		g_type_add_interface_static (type, GNOME_DB_ENTITY_TYPE, &entity_info);
	}
	return type;
}

static void 
gnome_db_table_xml_storage_init (GnomeDbXmlStorageIface *iface)
{
	iface->get_xml_id = gnome_db_table_get_xml_id;
	iface->save_to_xml = gnome_db_table_save_to_xml;
	iface->load_from_xml = gnome_db_table_load_from_xml;
}

static void
gnome_db_table_entity_init (GnomeDbEntityIface *iface)
{
	iface->has_field = gnome_db_table_has_field;
	iface->get_fields = gnome_db_table_get_fields;
	iface->get_field_by_name = gnome_db_table_get_field_by_name;
	iface->get_field_by_xml_id = gnome_db_table_get_field_by_xml_id;
	iface->get_field_by_index = gnome_db_table_get_field_by_index;
	iface->get_field_index = gnome_db_table_get_field_index;
	iface->add_field = gnome_db_table_add_field;
	iface->add_field_before = gnome_db_table_add_field_before;
	iface->swap_fields = gnome_db_table_swap_fields;
	iface->remove_field = gnome_db_table_remove_field;
	iface->is_writable = gnome_db_table_is_writable;
	iface->get_parameters = gnome_db_table_get_parameters;
}


static void
gnome_db_table_class_init (GnomeDbTableClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	gnome_db_table_signals[TEMPL_SIGNAL] =
		g_signal_new ("templ_signal",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbTableClass, templ_signal),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	class->templ_signal = NULL;

	object_class->dispose = gnome_db_table_dispose;
	object_class->finalize = gnome_db_table_finalize;

	/* Properties */
	object_class->set_property = gnome_db_table_set_property;
	object_class->get_property = gnome_db_table_get_property;
	g_object_class_install_property (object_class, PROP_DB,
					 g_param_spec_pointer ("database", NULL, NULL, 
							       (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	/* virtual functions */
#ifdef debug
        GNOME_DB_BASE_CLASS (class)->dump = (void (*)(GnomeDbBase *, guint)) gnome_db_table_dump;
#endif

}

static void
gnome_db_table_init (GnomeDbTable * gnome_db_table)
{
	gnome_db_table->priv = g_new0 (GnomeDbTablePrivate, 1);
	gnome_db_table->priv->db = NULL;
	gnome_db_table->priv->fields = NULL;
	gnome_db_table->priv->is_view = FALSE;
	gnome_db_table->priv->parents = NULL;
	gnome_db_table->priv->fields_hash = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);
}

/**
 * gnome_db_table_new
 * @dict: a #GnomeDbDict object
 *
 * Creates a new GnomeDbTable object
 *
 * Returns: the new object
 */
GObject*
gnome_db_table_new (GnomeDbDict *dict)
{
	GObject   *obj;
	GnomeDbTable *gnome_db_table;

	g_return_val_if_fail (!dict || IS_GNOME_DB_DICT (dict), NULL);

	obj = g_object_new (GNOME_DB_TABLE_TYPE, "dict", ASSERT_DICT (dict), NULL);
	gnome_db_table = GNOME_DB_TABLE (obj);
	gnome_db_base_set_id (GNOME_DB_BASE (gnome_db_table), 0);

	return obj;
}

static void 
nullified_field_cb (GObject *obj, GnomeDbTable *table)
{
	gchar *str;
	g_assert (g_slist_find (table->priv->fields, obj));

	table->priv->fields = g_slist_remove (table->priv->fields, obj);
	g_signal_handlers_disconnect_by_func (G_OBJECT (obj), 
					      G_CALLBACK (nullified_field_cb), table);
	g_signal_handlers_disconnect_by_func (G_OBJECT (obj), 
					      G_CALLBACK (changed_field_cb), table);
	str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (obj));
	g_hash_table_remove (table->priv->fields_hash, str);
	g_free (str);

#ifdef debug_signal
	g_print (">> 'FIELD_REMOVED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit_by_name (G_OBJECT (table), "field_removed", obj);
#ifdef debug_signal
	g_print ("<< 'FIELD_REMOVED' from %s\n", __FUNCTION__);
#endif

	g_object_set (obj, "db_table", NULL, NULL);	
	g_object_unref (obj);
}

static void 
nullified_parent_cb (GObject *obj, GnomeDbTable *table)
{
	g_assert (g_slist_find (table->priv->parents, obj));
	g_signal_handlers_disconnect_by_func (G_OBJECT (obj), 
					      G_CALLBACK (nullified_parent_cb), table);
	table->priv->parents = g_slist_remove (table->priv->parents, obj);
}

static void
gnome_db_table_dispose (GObject *object)
{
	GnomeDbTable *gnome_db_table;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_TABLE (object));

	gnome_db_table = GNOME_DB_TABLE (object);
	if (gnome_db_table->priv) {
		GSList *list;

		gnome_db_base_nullify_check (GNOME_DB_BASE (object));

		gnome_db_table_set_database (gnome_db_table, NULL);

		if (gnome_db_table->priv->fields_hash) {
			g_hash_table_destroy (gnome_db_table->priv->fields_hash);
			gnome_db_table->priv->fields_hash = NULL;
		}

		while (gnome_db_table->priv->fields)
			gnome_db_base_nullify (GNOME_DB_BASE (gnome_db_table->priv->fields->data));

		list = gnome_db_table->priv->parents;
		while (list) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (list->data),
							      G_CALLBACK (nullified_parent_cb), gnome_db_table);
			list = g_slist_next (list);
		}
		if (gnome_db_table->priv->parents) {
			g_slist_free (gnome_db_table->priv->parents);
			gnome_db_table->priv->parents = NULL;
		}
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_table_finalize (GObject   * object)
{
	GnomeDbTable *gnome_db_table;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_TABLE (object));

	gnome_db_table = GNOME_DB_TABLE (object);
	if (gnome_db_table->priv) {

		g_free (gnome_db_table->priv);
		gnome_db_table->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_table_set_property (GObject              *object,
			guint                 param_id,
			const GValue         *value,
			GParamSpec           *pspec)
{
	gpointer ptr;
	GnomeDbTable *gnome_db_table;

	gnome_db_table = GNOME_DB_TABLE (object);
	if (gnome_db_table->priv) {
		switch (param_id) {
		case PROP_DB:
			ptr = g_value_get_pointer (value);
			gnome_db_table_set_database (gnome_db_table, GNOME_DB_DATABASE (ptr));
			break;
		}
	}
}

static void
gnome_db_table_get_property (GObject              *object,
			  guint                 param_id,
			  GValue               *value,
			  GParamSpec           *pspec)
{
	GnomeDbTable *gnome_db_table;
	gnome_db_table = GNOME_DB_TABLE (object);
	
	if (gnome_db_table->priv) {
		switch (param_id) {
		case PROP_DB:
			g_value_set_pointer (value, gnome_db_table->priv->db);
			break;
		}	
	}
}

static void
gnome_db_table_set_database (GnomeDbTable *table, GnomeDbDatabase *db)
{
	if (table->priv->db) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (table->priv->db),
						      G_CALLBACK (nullified_object_cb), table);
		table->priv->db = NULL;
	}
	
	if (db && IS_GNOME_DB_DATABASE (db)) {
		table->priv->db = GNOME_DB_DATABASE (db);
		gnome_db_base_connect_nullify (db, G_CALLBACK (nullified_object_cb), table);
	}
}

static void
nullified_object_cb (GObject *obj, GnomeDbTable *table)
{
	gnome_db_base_nullify (GNOME_DB_BASE (table));
}

/**
 * gnome_db_table_get_database
 * @table: a #GnomeDbTable object
 *
 * Get the database to which the table belongs
 *
 * Returns: a #GnomeDbDatabase pointer
 */
GnomeDbDatabase
*gnome_db_table_get_database (GnomeDbTable *table)
{
	g_return_val_if_fail (table && IS_GNOME_DB_TABLE (table), NULL);
	g_return_val_if_fail (table->priv, NULL);

	return table->priv->db;
}

/**
 * gnome_db_table_is_view
 * @table: a #GnomeDbTable object
 *
 * Does the object represent a view rather than a table?
 *
 * Returns: TRUE if it is a view
 */
gboolean
gnome_db_table_is_view (GnomeDbTable *table)
{
	g_return_val_if_fail (table && IS_GNOME_DB_TABLE (table), FALSE);
	g_return_val_if_fail (table->priv, FALSE);

	return table->priv->is_view;
}

/**
 * gnome_db_table_get_parents
 * @table: a #GnomeDbTable object
 *
 * Get the parent tables of the table given as argument. This is significant only
 * for DBMS which support tables inheritance (like PostgreSQL for example).
 *
 * Returns: a constant list of #GnomeDbTable objects
 */
const GSList *
gnome_db_table_get_parents (GnomeDbTable *table)
{
	g_return_val_if_fail (table && IS_GNOME_DB_TABLE (table), NULL);
	g_return_val_if_fail (table->priv, NULL);

	return table->priv->parents;
}

/**
 * gnome_db_table_get_constraints
 * @table: a #GnomeDbTable object
 *
 * Get all the constraints which apply to the given table (each constraint
 * can represent a NOT NULL, a primary key or foreign key or a check constraint.
 *
 * Returns: a new list of #GnomeDbConstraint objects
 */
GSList *
gnome_db_table_get_constraints (GnomeDbTable *table)
{
	g_return_val_if_fail (table && IS_GNOME_DB_TABLE (table), NULL);
	g_return_val_if_fail (table->priv, NULL);

	return gnome_db_database_get_table_constraints (table->priv->db, table);
}

/**
 * gnome_db_table_get_pk_constraint
 * @table: a #GnomeDbTable object
 *
 * Get the primary key constraint of @table, if there is any. If several
 * #GnomeDbConstraint represent a primary key constraint for @table, then
 * the first one in the list of constraints is returned.
 *
 * Returns: a #GnomeDbConstraint object or %NULL.
 */
GnomeDbConstraint *
gnome_db_table_get_pk_constraint (GnomeDbTable *table)
{
	GnomeDbConstraint *pkcons = NULL;
	GSList *db_constraints, *list;

	g_return_val_if_fail (table && IS_GNOME_DB_TABLE (table), NULL);
	g_return_val_if_fail (GNOME_DB_TABLE (table)->priv, NULL);

	db_constraints = gnome_db_database_get_all_constraints (table->priv->db);
	list = db_constraints;
	while (list && !pkcons) {
		if ((gnome_db_constraint_get_table (GNOME_DB_CONSTRAINT (list->data)) == table) &&
		    (gnome_db_constraint_get_constraint_type (GNOME_DB_CONSTRAINT (list->data)) == 
		     CONSTRAINT_PRIMARY_KEY))
			pkcons = GNOME_DB_CONSTRAINT (list->data);

		list = g_slist_next (list);
	}
	g_slist_free (db_constraints);

	return pkcons;
}


/**
 * gnome_db_table_get_dict
 * @table: a #GnomeDbTable object
 *
 * Get the #GnomeDbDict to which the @table is associated
 *
 * Returns: the #GnomeDbDict object
 */
GnomeDbDict *
gnome_db_table_get_dict (GnomeDbTable *table)
{
	g_return_val_if_fail (table && IS_GNOME_DB_TABLE (table), NULL);
	g_return_val_if_fail (GNOME_DB_TABLE (table)->priv, NULL);

	return gnome_db_base_get_dict (GNOME_DB_BASE (table));
}


/**
 * gnome_db_database_update_dbms_data
 * @table: a #GnomeDbTable object
 * @error: location to store error, or %NULL
 *
 * Synchronises the Table representation with the table structure which is stored in
 * the DBMS. For this operation to succeed, the connection to the DBMS server MUST be opened
 * (using the corresponding #GnomeDbServer object).
 *
 * Returns: TRUE if no error
 */
gboolean
gnome_db_table_update_dbms_data (GnomeDbTable *table, GError **error)
{
	GnomeDbDict *dict;
	GSList *fields;
	GdaDataModel *rs;
	gchar *str;
	guint now, total;
	GSList *updated_fields = NULL;
	GnomeDbServer *srv;
	GnomeDbTableField *field;
	GdaParameterList *paramlist;
        GdaParameter *param;
	gint current_position = 0;
	GSList *constraints;
	GHashTable *fk_hash;
	
	g_return_val_if_fail (table && IS_GNOME_DB_TABLE (table), FALSE);
	g_return_val_if_fail (GNOME_DB_TABLE (table)->priv, FALSE);

	/* g_print ("################ TABLE %s\n", gnome_db_base_get_name (GNOME_DB_BASE (table))); */
	dict = gnome_db_base_get_dict (GNOME_DB_BASE (table));
	srv = gnome_db_dict_get_server (dict);
	if (!gnome_db_server_conn_is_opened (srv)) {
		g_set_error (error, GNOME_DB_TABLE_ERROR, GNOME_DB_TABLE_META_DATA_UPDATE,
			     _("Connection is not opened!"));
		return FALSE;
	}

	/* In this procedure, we are creating some new GnomeDbConstraint objects to express
	 * the constraints the database has on this table. This list is "attached" to the GnomeDbTable
	 * object by the "pending_constraints" keyword, and is later normally fetched by the GnomeDbDatabase object.
	 * If we still have one at this point, we must emit a warning and free that list.
	 */
	if ((constraints = g_object_get_data (G_OBJECT (table), "pending_constraints"))) {
		GSList *list = constraints;

		g_warning ("GnomeDbTable object %p has a non empty list of pending constraints", table);
		while (list) {
			g_object_unref (G_OBJECT (list->data));
			list = g_slist_next (list);
		}
		g_slist_free (constraints);
		constraints = NULL;
		g_object_set_data (G_OBJECT (table), "pending_constraints", NULL);
	}
	fk_hash = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);

	/* parameters list */
	paramlist = gda_parameter_list_new ();
        param = gda_parameter_new_string ("name", gnome_db_base_get_name (GNOME_DB_BASE (table)));
        gda_parameter_list_add_parameter (paramlist, param);
	rs = gnome_db_server_get_gda_schema (srv, GDA_CONNECTION_SCHEMA_FIELDS, paramlist);
	gda_parameter_list_free (paramlist);

	/* Result set analysis */
	if (!gnome_db_result_set_check_data_model (rs, 9, 
						   GDA_VALUE_TYPE_STRING,
						   GDA_VALUE_TYPE_STRING,
						   GDA_VALUE_TYPE_INTEGER,
						   GDA_VALUE_TYPE_INTEGER,
						   GDA_VALUE_TYPE_BOOLEAN,
						   GDA_VALUE_TYPE_BOOLEAN,
						   GDA_VALUE_TYPE_BOOLEAN,
						   GDA_VALUE_TYPE_STRING, -1)) {
		g_set_error (error, GNOME_DB_TABLE_ERROR, GNOME_DB_TABLE_FIELDS_ERROR,
			     _("Schema for list of fields is wrong"));
		g_object_unref (G_OBJECT (rs));
		return FALSE;
	}

	/* Resultset parsing */
	total = gda_data_model_get_n_rows (rs);
	now = 0;		
	while (now < total) {
		const GdaValue *value;
		gboolean newfield = FALSE;
		GnomeDbField *mgf;

		value = gda_data_model_get_value_at (rs, 0, now);
		str = gda_value_stringify (value);
		mgf = gnome_db_table_get_field_by_name (GNOME_DB_ENTITY (table), str);
		if (!mgf) {
			/* field name */
			field = GNOME_DB_TABLE_FIELD (gnome_db_table_field_new (gnome_db_base_get_dict (GNOME_DB_BASE (table)), NULL));
			gnome_db_base_set_name (GNOME_DB_BASE (field), str);
			newfield = TRUE;
		}
		else {
			field = GNOME_DB_TABLE_FIELD (mgf);
			current_position = g_slist_index (table->priv->fields, field) + 1;
		}
		g_free (str);
		
		updated_fields = g_slist_append (updated_fields, field);

		/* FIXME: No description for fields in the schema. */
		/* FIXME: No owner for fields in the schema. */
		
		/* Data type */
		value = gda_data_model_get_value_at (rs, 1, now);
		if (value && !gda_value_is_null (value) && (* gda_value_get_string(value))) {
			GnomeDbServerDataType *type;

			str = gda_value_stringify (value);
			type = gnome_db_server_get_data_type_by_name (srv, str);
			if (type)
				gnome_db_table_field_set_data_type (field, type);
			g_free (str);
		}
		if (!gnome_db_field_get_data_type (GNOME_DB_FIELD (field))) {
			if (value)
				str = gda_value_stringify (value);
			else
				str = g_strdup ("NULL");
			g_set_error (error, GNOME_DB_TABLE_ERROR, GNOME_DB_TABLE_FIELDS_ERROR,
				     _("Can't find data type %s"), str);
			g_free (str);
			g_object_unref (G_OBJECT (rs));
			return FALSE;
		}

		/* Size */
		value = gda_data_model_get_value_at (rs, 2, now);
		if (value && !gda_value_is_null (value)) 
			gnome_db_table_field_set_length (field, gda_value_get_integer (value));

		/* Scale */
		value = gda_data_model_get_value_at (rs, 3, now);
		if (value && !gda_value_is_null (value)) 
			gnome_db_table_field_set_scale (field, gda_value_get_integer (value));

		/* Default value */
		value = gda_data_model_get_value_at (rs, 8, now);
		if (value && !gda_value_is_null (value)) {
			gchar *defv = gda_value_stringify (value);
			if (defv && *defv)
				gnome_db_table_field_set_default_value (field, value);
			if (defv)
				g_free (defv);
		}
				
		/* signal if the field is new */
		if (newfield) {
			g_object_set (G_OBJECT (field), "db_table", table, NULL);
			gnome_db_table_add_field_at_pos (table, field, current_position++);
			g_object_unref (G_OBJECT (field));
		}
		
		/* NOT NULL constraint */
		value = gda_data_model_get_value_at (rs, 4, now);
		if (value && !gda_value_is_null (value) && gda_value_get_boolean (value)) {
			GnomeDbConstraint *cstr = GNOME_DB_CONSTRAINT (gnome_db_constraint_new (table, CONSTRAINT_NOT_NULL));
			gnome_db_constraint_not_null_set_field (cstr, field);
			constraints = g_slist_append (constraints, cstr);
		}

		/* Other constraints:
		 * For constraints other than the NOT NULL constraint, this is a temporary solution to
		 * get the constraints before we get a real constraints request schema in libgda.
		 *
		 * THE FOLLOWING ASSUMPTIONS ARE MADE:
		 * PRIMARY KEY: there is only ONE primary key per table even if it is a composed primary key
		 * UNIQUE: each field with the UNIQUE attribute is considered unique itself (=> no constraint
		 * for unique couple (or more) fields)
		 * FOREIGN KEY: we don't have the associated actions to ON UPDATE and ON DELETE actions
		 */

		/* PRIMARY KEY constraint */
		value = gda_data_model_get_value_at (rs, 5, now);
		if (value && !gda_value_is_null (value) && gda_value_get_boolean (value)) {
			GnomeDbConstraint *cstr = NULL;
			GSList *list = constraints, *nlist;
			
			/* find the primary key constraint if it already exists */
			while (list && !cstr) {
				if (gnome_db_constraint_get_constraint_type (GNOME_DB_CONSTRAINT (list->data)) ==
				    CONSTRAINT_PRIMARY_KEY)
					cstr = GNOME_DB_CONSTRAINT (list->data);
				list = g_slist_next (list);
			}

			if (!cstr) {
				cstr = GNOME_DB_CONSTRAINT (gnome_db_constraint_new (table, CONSTRAINT_PRIMARY_KEY));
				constraints = g_slist_append (constraints, cstr);
			}

			/* set the fields */
			nlist = gnome_db_constraint_pkey_get_fields (cstr);
			nlist = g_slist_append (nlist, field);
			gnome_db_constraint_pkey_set_fields (cstr, nlist);
			g_slist_free (nlist);
		}

		/* UNIQUE constraint */
		value = gda_data_model_get_value_at (rs, 6, now);
		if (value && !gda_value_is_null (value) && gda_value_get_boolean (value)) {
			GnomeDbConstraint *cstr;
			GSList *nlist;

			cstr = GNOME_DB_CONSTRAINT (gnome_db_constraint_new (table, CONSTRAINT_UNIQUE));
			constraints = g_slist_append (constraints, cstr);

			nlist = g_slist_append (NULL, field);
			gnome_db_constraint_unique_set_fields (cstr, nlist);
			g_slist_free (nlist);
		}

		/* FOREIGN KEY constraint */
		value = gda_data_model_get_value_at (rs, 7, now);
		if (value && !gda_value_is_null (value) && (* gda_value_get_string (value))) {
			gchar *ref_table, *str, *tok;
			GnomeDbRefBase *ref;
			GnomeDbConstraint *cstr = NULL;
			GSList *list, *nlist;
			GnomeDbConstraintFkeyPair *pair;

			/* ref table */
			str = g_strdup (gda_value_get_string (value));
			ref_table = g_strdup (strtok_r (str, ".", &tok));
			g_free (str);
			
			ref = GNOME_DB_REF_BASE (gnome_db_ref_base_new (gnome_db_base_get_dict (GNOME_DB_BASE (table))));
			gnome_db_ref_base_set_ref_name (ref, GNOME_DB_TABLE_FIELD_TYPE, 
						  REFERENCE_BY_NAME, gda_value_get_string (value));
			
			/* find the foreign key constraint if it already exists */
			cstr = g_hash_table_lookup (fk_hash, ref_table);
			if (!cstr) {
				cstr = GNOME_DB_CONSTRAINT (gnome_db_constraint_new (table, CONSTRAINT_FOREIGN_KEY));
				constraints = g_slist_append (constraints, cstr);
				g_hash_table_insert (fk_hash, ref_table, cstr);
			}
			else
				g_free (ref_table);
			
			nlist = gnome_db_constraint_fkey_get_fields (cstr);
			pair = g_new0 (GnomeDbConstraintFkeyPair, 1);
			pair->fkey = field;
			pair->ref_pkey = NULL;
			pair->ref_pkey_repl = ref;
			nlist = g_slist_append (nlist, pair);
			gnome_db_constraint_fkey_set_fields (cstr, nlist);
			
			/* memory libreation */
			list = nlist;
			while (list) {
				g_free (list->data);
				list = g_slist_next (list);
			}
			g_object_unref (G_OBJECT (ref));
			g_slist_free (nlist);
		}
		
		now++;
	}
	    
	g_object_unref (G_OBJECT (rs));
	
	/* remove the fields not existing anymore */
	fields = table->priv->fields;
	while (fields) {
		if (!g_slist_find (updated_fields, fields->data)) {
			gnome_db_base_nullify (GNOME_DB_BASE (fields->data));
			fields = table->priv->fields;
		}
		else
			fields = g_slist_next (fields);
	}
	g_slist_free (updated_fields);

	/* stick the constraints list to the GnomeDbTable object */
	g_object_set_data (G_OBJECT (table), "pending_constraints", constraints);
	g_hash_table_destroy (fk_hash);
	
	return TRUE;
}


/*
 * pos = -1 to append
 */
static void
gnome_db_table_add_field_at_pos (GnomeDbTable *table, GnomeDbTableField *field, gint pos)
{
	gchar *str;
	table->priv->fields = g_slist_insert (table->priv->fields, field, pos);
	
	str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (field));
	g_hash_table_insert (table->priv->fields_hash, str, field);

	g_object_ref (G_OBJECT (field));
	gnome_db_base_connect_nullify (field, 
				 G_CALLBACK (nullified_field_cb), table);
	g_signal_connect (G_OBJECT (field), "changed",
			  G_CALLBACK (changed_field_cb), table);

#ifdef debug_signal
	g_print (">> 'FIELD_ADDED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit_by_name (G_OBJECT (table), "field_added", field);
#ifdef debug_signal
	g_print ("<< 'FIELD_ADDED' from %s\n", __FUNCTION__);
#endif	
}


static void
changed_field_cb (GObject *obj, GnomeDbTable *table)
{
#ifdef debug_signal
	g_print (">> 'FIELD_UPDATED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit_by_name (G_OBJECT (table), "field_updated", obj);
#ifdef debug_signal
	g_print ("<< 'FIELD_UPDATED' from %s\n", __FUNCTION__);
#endif	
}


#ifdef debug
static void
gnome_db_table_dump (GnomeDbTable *table, guint offset)
{
	gchar *str;
        guint i;
        GSList *list;
	
	g_return_if_fail (table && IS_GNOME_DB_TABLE (table));

        /* string for the offset */
        str = g_new0 (gchar, offset+1);
        for (i=0; i<offset; i++)
                str[i] = ' ';
        str[offset] = 0;

        /* dump */
        if (table->priv)
                g_print ("%s" D_COL_H1 "GnomeDbTable" D_COL_NOR  " %s (%p)\n",
                         str, gnome_db_base_get_name (GNOME_DB_BASE (table)), table);
        else
                g_print ("%s" D_COL_ERR "Using finalized object %p" D_COL_NOR, str, table);

	/* fields */
        list = table->priv->fields;
        if (list) {
                g_print ("%sFields:\n", str);
                while (list) {
                        gnome_db_base_dump (GNOME_DB_BASE (list->data), offset+5);
                        list = g_slist_next (list);
                }
        }
        else
                g_print ("%sContains no field\n", str);

}
#endif





/* 
 * GnomeDbEntity interface implementation
 */
static gboolean
gnome_db_table_has_field (GnomeDbEntity *iface, GnomeDbField *field)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_TABLE (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_TABLE (iface)->priv, FALSE);

	return g_slist_find (GNOME_DB_TABLE (iface)->priv->fields, field) ? TRUE : FALSE;
}

static GSList *
gnome_db_table_get_fields (GnomeDbEntity *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_TABLE (iface), NULL);
	g_return_val_if_fail (GNOME_DB_TABLE (iface)->priv, NULL);

	return g_slist_copy (GNOME_DB_TABLE (iface)->priv->fields);
}

static GnomeDbField *
gnome_db_table_get_field_by_name (GnomeDbEntity *iface, const gchar *name)
{
	GnomeDbField *field = NULL;
	GSList *list;
	gchar *lcname = g_utf8_strdown (name, -1);

	g_return_val_if_fail (iface && IS_GNOME_DB_TABLE (iface), NULL);
	g_return_val_if_fail (GNOME_DB_TABLE (iface)->priv, NULL);

	list = GNOME_DB_TABLE (iface)->priv->fields;
	while (list && !field) {
		if (!strcmp (gnome_db_field_get_name (GNOME_DB_FIELD (list->data)), lcname) ||
		    !strcmp (gnome_db_field_get_name (GNOME_DB_FIELD (list->data)), name))
			field = GNOME_DB_FIELD (list->data);
		list = g_slist_next (list);
	}
	g_free (lcname);

	return field;
}

static GnomeDbField *
gnome_db_table_get_field_by_xml_id (GnomeDbEntity *iface, const gchar *xml_id)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_TABLE (iface), NULL);
	g_return_val_if_fail (GNOME_DB_TABLE (iface)->priv, NULL);

	return g_hash_table_lookup (GNOME_DB_TABLE (iface)->priv->fields_hash, xml_id);
}

static GnomeDbField *
gnome_db_table_get_field_by_index (GnomeDbEntity *iface, gint index)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_TABLE (iface), NULL);
	g_return_val_if_fail (GNOME_DB_TABLE (iface)->priv, NULL);
	g_return_val_if_fail (index >= 0, NULL);
	g_return_val_if_fail (index < g_slist_length (GNOME_DB_TABLE (iface)->priv->fields), NULL);
	
	return GNOME_DB_FIELD (g_slist_nth_data (GNOME_DB_TABLE (iface)->priv->fields, index));
}

static gint
gnome_db_table_get_field_index (GnomeDbEntity *iface, GnomeDbField *field)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_TABLE (iface), -1);
	g_return_val_if_fail (GNOME_DB_TABLE (iface)->priv, -1);

	return g_slist_index (GNOME_DB_TABLE (iface)->priv->fields, field);
}

static void
gnome_db_table_add_field (GnomeDbEntity *iface, GnomeDbField *field)
{
	g_return_if_fail (iface && IS_GNOME_DB_TABLE (iface));
	g_return_if_fail (GNOME_DB_TABLE (iface)->priv);
	g_return_if_fail (field && IS_GNOME_DB_TABLE_FIELD (field));
	g_return_if_fail (!g_slist_find (GNOME_DB_TABLE (iface)->priv->fields, field));
	g_return_if_fail (gnome_db_field_get_entity (field) == iface);
	
	gnome_db_table_add_field_at_pos (GNOME_DB_TABLE (iface), GNOME_DB_TABLE_FIELD (field), -1);
}

static void
gnome_db_table_add_field_before (GnomeDbEntity *iface, GnomeDbField *field, GnomeDbField *field_before)
{
	GnomeDbTable *table;
	gint pos = -1;

	g_return_if_fail (iface && IS_GNOME_DB_TABLE (iface));
	g_return_if_fail (GNOME_DB_TABLE (iface)->priv);
	table = GNOME_DB_TABLE (iface);

	g_return_if_fail (field && IS_GNOME_DB_TABLE_FIELD (field));
	g_return_if_fail (!g_slist_find (GNOME_DB_TABLE (iface)->priv->fields, field));
	g_return_if_fail (gnome_db_field_get_entity (field) == iface);
	if (field_before) {
		g_return_if_fail (field_before && IS_GNOME_DB_TABLE_FIELD (field_before));
		g_return_if_fail (g_slist_find (GNOME_DB_TABLE (iface)->priv->fields, field_before));
		pos = g_slist_index (table->priv->fields, field_before);
	}

	gnome_db_table_add_field_at_pos (table, GNOME_DB_TABLE_FIELD (field), pos);
}

static void
gnome_db_table_swap_fields (GnomeDbEntity *iface, GnomeDbField *field1, GnomeDbField *field2)
{
	GSList *ptr1, *ptr2;

	g_return_if_fail (iface && IS_GNOME_DB_TABLE (iface));
	g_return_if_fail (GNOME_DB_TABLE (iface)->priv);
	g_return_if_fail (field1 && IS_GNOME_DB_TABLE_FIELD (field1));
	g_return_if_fail (field2 && IS_GNOME_DB_TABLE_FIELD (field2));
	ptr1 = g_slist_find (GNOME_DB_TABLE (iface)->priv->fields, field1);
	ptr2 = g_slist_find (GNOME_DB_TABLE (iface)->priv->fields, field2);
	g_return_if_fail (ptr1);
	g_return_if_fail (ptr2);
	
	ptr1->data = field2;
	ptr2->data = field1;

#ifdef debug_signal
	g_print (">> 'FIELDS_ORDER_CHANGED' from %s\n", __FUNCTION__);
#endif
	g_signal_emit_by_name (G_OBJECT (iface), "fields_order_changed");
#ifdef debug_signal
	g_print ("<< 'FIELDS_ORDER_CHANGED' from %s\n", __FUNCTION__);
#endif

}

static void
gnome_db_table_remove_field (GnomeDbEntity *iface, GnomeDbField *field)
{
	g_return_if_fail (iface && IS_GNOME_DB_TABLE (iface));
	g_return_if_fail (GNOME_DB_TABLE (iface)->priv);
	g_return_if_fail (field && IS_GNOME_DB_TABLE_FIELD (field));
	g_return_if_fail (g_slist_find (GNOME_DB_TABLE (iface)->priv->fields, field));

	nullified_field_cb (G_OBJECT (field), GNOME_DB_TABLE (iface));
}

static gboolean
gnome_db_table_is_writable (GnomeDbEntity *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_TABLE (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_TABLE (iface)->priv, FALSE);
	
	return GNOME_DB_TABLE (iface)->priv->is_view ? FALSE : TRUE;
}

static GSList *
gnome_db_table_get_parameters (GnomeDbEntity *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_TABLE (iface), NULL);
	g_return_val_if_fail (GNOME_DB_TABLE (iface)->priv, NULL);
	
	return NULL;
}


/* 
 * GnomeDbXmlStorage interface implementation
 */
static gchar *
gnome_db_table_get_xml_id (GnomeDbXmlStorage *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_TABLE (iface), NULL);
	g_return_val_if_fail (GNOME_DB_TABLE (iface)->priv, NULL);

	return g_strdup_printf ("TV%s", gnome_db_base_get_name (GNOME_DB_BASE (iface)));
}

static xmlNodePtr
gnome_db_table_save_to_xml (GnomeDbXmlStorage *iface, GError **error)
{
	xmlNodePtr node = NULL;
	GnomeDbTable *table;
	gchar *str;
	const gchar *cstr;
	GSList *list;
	gint i;

	g_return_val_if_fail (iface && IS_GNOME_DB_TABLE (iface), NULL);
	g_return_val_if_fail (GNOME_DB_TABLE (iface)->priv, NULL);

	table = GNOME_DB_TABLE (iface);

	node = xmlNewNode (NULL, "GNOME_DB_TABLE");
	
	str = gnome_db_table_get_xml_id (iface);
	xmlSetProp (node, "id", str);
	g_free (str);
	xmlSetProp (node, "name", gnome_db_base_get_name (GNOME_DB_BASE (table)));
	cstr = gnome_db_base_get_owner (GNOME_DB_BASE (table));
	if (cstr && *cstr)
		xmlSetProp (node, "owner", cstr);
	xmlSetProp (node, "descr", gnome_db_base_get_description (GNOME_DB_BASE (table)));

	xmlSetProp (node, "is_view", table->priv->is_view ? "t" : "f");

	/* parent tables */
	i = 0;
	list = table->priv->parents;
	while (list) {
		xmlNodePtr parent;
		gchar *str;

		parent = xmlNewChild (node, NULL, "GNOME_DB_PARENT_TABLE", NULL);
		str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (list->data));
		xmlSetProp (parent, "table", str);
		g_free (str);

		str = g_strdup_printf ("%d", i);
		xmlSetProp (parent, "order", str);
		g_free (str);

		list = g_slist_next (list);
	}
	
	/* fields */
	list = table->priv->fields;
	while (list) {
		xmlNodePtr field;
		
		field = gnome_db_xml_storage_save_to_xml (GNOME_DB_XML_STORAGE (list->data), error);

		if (field)
			xmlAddChild (node, field);
		else {
			xmlFreeNode (node);
			return NULL;
		}
		list = g_slist_next (list);
	}

	return node;
}

static gboolean
gnome_db_table_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error)
{
	GnomeDbTable *table;
	gchar *prop;
	gboolean name = FALSE;
	xmlNodePtr children;

	g_return_val_if_fail (iface && IS_GNOME_DB_TABLE (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_TABLE (iface)->priv, FALSE);
	g_return_val_if_fail (node, FALSE);

	table = GNOME_DB_TABLE (iface);
	if (strcmp (node->name, "GNOME_DB_TABLE")) {
		g_set_error (error,
			     GNOME_DB_TABLE_ERROR,
			     GNOME_DB_TABLE_XML_LOAD_ERROR,
			     _("XML Tag is not <GNOME_DB_TABLE>"));
		return FALSE;
	}

	prop = xmlGetProp (node, "name");
	if (prop) {
		name = TRUE;
		gnome_db_base_set_name (GNOME_DB_BASE (table), prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "descr");
	if (prop) {
		gnome_db_base_set_description (GNOME_DB_BASE (table), prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "owner");
	if (prop) {
		gnome_db_base_set_owner (GNOME_DB_BASE (table), prop);
		g_free (prop);
	}

	table->priv->is_view = FALSE;
	prop = xmlGetProp (node, "is_view");
	if (prop) {
		table->priv->is_view = (*prop == 't') ? TRUE : FALSE;
		g_free (prop);
	}

	children = node->children;
	while (children) {
		gboolean done = FALSE;

		/* parent table */
		if (!strcmp (children->name, "GNOME_DB_PARENT_TABLE")) {
			TO_IMPLEMENT;
			done = TRUE;
		}
		/* fields */
		if (!done && !strcmp (children->name, "GNOME_DB_FIELD")) {
			GnomeDbTableField *field;
			field = GNOME_DB_TABLE_FIELD (gnome_db_table_field_new (gnome_db_base_get_dict (GNOME_DB_BASE (iface)), NULL));
			if (gnome_db_xml_storage_load_from_xml (GNOME_DB_XML_STORAGE (field), children, error)) {
				g_object_set (G_OBJECT (field), "db_table", table, NULL);
				gnome_db_table_add_field (GNOME_DB_ENTITY (table), GNOME_DB_FIELD (field));
				g_object_unref (G_OBJECT (field));
			}
			else
				return FALSE;
		}
		
		children = children->next;
	}

	if (name)
		return TRUE;
	else {
		g_set_error (error,
			     GNOME_DB_TABLE_ERROR,
			     GNOME_DB_TABLE_XML_LOAD_ERROR,
			     _("Missing required attributes for <GNOME_DB_TABLE>"));
		return FALSE;
	}
}

