/* gnome-db-query-parsing.c
 *
 * Copyright (C) 2003 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-query.h"
#include "gnome-db-field.h"
#include "gnome-db-qfield.h"
#include "gnome-db-qf-field.h"
#include "gnome-db-qf-value.h"
#include "gnome-db-qf-all.h"
#include "gnome-db-qf-func.h"
#include "gnome-db-qf-agg.h"
#include "gnome-db-ref-base.h"
#include "gnome-db-target.h"
#include "gnome-db-entity.h"
#include "gnome-db-table.h"
#include "gnome-db-database.h"
#include "gnome-db-constraint.h"
#include "gnome-db-xml-storage.h"
#include "gnome-db-referer.h"
#include "gnome-db-renderer.h"
#include "gnome-db-parameter.h"
#include "marshal.h"
#include "gnome-db-join.h"
#include <string.h>
#include "gnome-db-data-set.h"
#include "gnome-db-condition.h"
#include "gnome-db-server.h"
#include "gnome-db-server-function.h"
#include "gnome-db-server-aggregate.h"
#include "gnome-db-data-handler.h"

#include "gnome-db-query-private.h"
#include "gnome-db-query-parsing.h"


static void       clean_old_targets (GnomeDbQuery *query, ParseData *pdata);
static void       clean_old_fields (GnomeDbQuery *query, ParseData *pdata);

static GnomeDbTarget    *parsed_create_target_sql_table  (GnomeDbQuery *query, ParseData *pdata, sql_table *table, GError **error);
static GnomeDbTarget    *parsed_create_target_db_table   (GnomeDbQuery *query, ParseData *pdata,
						     const gchar *table, const gchar *as, GError **error);
static GnomeDbTarget    *parsed_create_target_sub_select (GnomeDbQuery *query, ParseData *pdata, 
						     sql_select_statement *select, const gchar *as, GError **error);
static gboolean     parsed_create_join_sql_table    (GnomeDbQuery *query, ParseData *pdata, sql_table *table, GError **error);

static GnomeDbCondition *parsed_create_simple_condition  (GnomeDbQuery *query, ParseData *pdata, sql_condition *sqlcond, 
						     gboolean try_existing_field, GSList **targets_return, 
						     GError **error);
static GnomeDbField     *parsed_create_field_query_field  (GnomeDbQuery *query, gboolean add_to_query, 
						      ParseData *pdata, GList *field_names,
						      gboolean try_existing_field, gboolean *new_field, 
						      GnomeDbTarget **target_return, GError **error);
static GnomeDbField     *parsed_create_value_query_field  (GnomeDbQuery *query, gboolean add_to_query,
						      ParseData *pdata, const gchar *value, GList *param_specs, 
						      gboolean *new_field, GError **error);
static GnomeDbField     *parsed_create_func_query_field   (GnomeDbQuery *query, gboolean add_to_query, 
						      ParseData *pdata, const gchar *funcname, 
						      GList *funcargs, 
						      gboolean try_existing_field, gboolean *new_field, GError **error);

/*
 * Conversion between sql_condition_operator enum and GnomeDbConditionType type.
 * WARNING: the converstion is not straight forward, and some more testing is required on
 * sql_condition_operator.
 */
static GnomeDbConditionType parse_condop_converter[] = {GNOME_DB_CONDITION_LEAF_EQUAL, 
						   GNOME_DB_CONDITION_LEAF_EQUAL,
						   GNOME_DB_CONDITION_LEAF_IN, 
						   GNOME_DB_CONDITION_LEAF_LIKE,
						   GNOME_DB_CONDITION_LEAF_BETWEEN,
						   GNOME_DB_CONDITION_LEAF_SUP, 
						   GNOME_DB_CONDITION_LEAF_INF,
						   GNOME_DB_CONDITION_LEAF_SUPEQUAL, 
						   GNOME_DB_CONDITION_LEAF_INFEQUAL,
						   GNOME_DB_CONDITION_LEAF_DIFF, 
						   GNOME_DB_CONDITION_LEAF_REGEX,
						   GNOME_DB_CONDITION_LEAF_REGEX_NOCASE, 
						   GNOME_DB_CONDITION_LEAF_NOT_REGEX,
						   GNOME_DB_CONDITION_LEAF_NOT_REGEX_NOCASE, 
						   GNOME_DB_CONDITION_LEAF_SIMILAR};

/*
 * Conversion between sql_join_type enum and GnomeDbJoinType type.
 */
static GnomeDbJoinType parse_join_converter[] = {GNOME_DB_JOIN_TYPE_CROSS, GNOME_DB_JOIN_TYPE_INNER,
					    GNOME_DB_JOIN_TYPE_LEFT_OUTER, GNOME_DB_JOIN_TYPE_RIGHT_OUTER,
					    GNOME_DB_JOIN_TYPE_FULL_OUTER};

/*
 * Creates and initialises a new ParseData structure
 */
ParseData *
parse_data_new (GnomeDbQuery *query)
{
	ParseData *pdata = g_new0 (ParseData, 1);
	pdata->prev_targets = g_slist_copy (query->priv->targets);
	pdata->prev_fields = NULL;
	pdata->parsed_targets = NULL;
	pdata->new_targets = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);
	pdata->sql_table_targets = g_hash_table_new (NULL, NULL);

	return pdata;
}

void
parse_data_destroy (ParseData *pdata)
{
	if (pdata->prev_targets)
		g_slist_free (pdata->prev_targets);
	if (pdata->prev_fields)
		g_slist_free (pdata->prev_fields);
	if (pdata->parsed_targets)
		g_slist_free (pdata->parsed_targets);
	if (pdata->new_targets)
		g_hash_table_destroy (pdata->new_targets);
	if (pdata->sql_table_targets)
		g_hash_table_destroy (pdata->sql_table_targets);
	g_free (pdata);
}

/*
 * Fills @targets_hash with what's already present in @query.
 * For each target, the target's alias is inserted as a key to the target object, and if there
 * is only one target representing an entity, then the entity name is also inserted as a key
 * to the target object.
 */
void
parse_data_compute_targets_hash (GnomeDbQuery *query, ParseData *pdata)
{
	GSList *list = query->priv->targets;
	GHashTable *tmp_hash;
	gchar *str;
	gint i;
	GnomeDbEntity *ent;
	const gchar *name;

	tmp_hash = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);
	list = query->priv->targets;
	while (list) {
		str = g_utf8_strdown (gnome_db_target_get_alias (GNOME_DB_TARGET (list->data)), -1);
		g_hash_table_insert (pdata->new_targets, str, list->data);

		ent = gnome_db_target_get_represented_entity (GNOME_DB_TARGET (list->data));
		name = gnome_db_base_get_name (GNOME_DB_BASE (ent));
		if (name && *name) {
			str = g_utf8_strdown (name, -1);
			i = GPOINTER_TO_INT (g_hash_table_lookup (tmp_hash, str));
			g_hash_table_insert (tmp_hash, str, GINT_TO_POINTER (i+1));
		}

		list = g_slist_next (list);
	}
	
	list = query->priv->targets;
	while (list) {
		ent = gnome_db_target_get_represented_entity (GNOME_DB_TARGET (list->data));
		name = gnome_db_base_get_name (GNOME_DB_BASE (ent));
		if (name && *name) {
			str = g_utf8_strdown (name, -1);
			i = GPOINTER_TO_INT (g_hash_table_lookup (tmp_hash, str));
			if (i == 1) 
				g_hash_table_insert (pdata->new_targets, str, list->data);
			else
				g_free (str);
		}

		list = g_slist_next (list);
	}
	g_hash_table_destroy (tmp_hash);
}

static void
clean_old_targets (GnomeDbQuery *query, ParseData *pdata)
{
	GSList *list;

	/* cleaning old non parsed targets */
	list = pdata->prev_targets;
	while (list) {
		gnome_db_base_nullify (GNOME_DB_BASE (list->data));
		list = g_slist_next (list);
	}
	g_slist_free (pdata->prev_targets);
	pdata->prev_targets = NULL;

	/* compute targets hash */
	parse_data_compute_targets_hash (query, pdata);

	/* new list of fields */
	pdata->prev_fields = g_slist_copy (query->priv->fields);
}

static void
clean_old_fields (GnomeDbQuery *query, ParseData *pdata)
{
	GSList *list;

	/* cleaning old non parsed targets */
	list = pdata->prev_fields;
	while (list) {
		if (g_slist_find (query->priv->fields, list->data))
			gnome_db_base_nullify (GNOME_DB_BASE (list->data));
		list = g_slist_next (list);
	}
	g_slist_free (pdata->prev_fields);
	pdata->prev_fields = NULL;
}

/*
 * main SELECT analysis
 */
gboolean 
parsed_create_select_query (GnomeDbQuery *query, sql_select_statement *select, GError **error)
{
	gboolean has_error = FALSE;
	ParseData *pdata = parse_data_new (query);

	gnome_db_query_set_query_type (query, GNOME_DB_QUERY_TYPE_SELECT);
	/* FIXME: UNION, EXCEPT, etc forms of query are not yet handled by
	   the libgdasql library, it should be done there before we can do anything here */

	/*
	 * DISTINCT flag
	 */
	query->priv->global_distinct = select->distinct ? TRUE : FALSE;

	/*
	 * Targets: creating #GnomeDbTarget objects
	 */
	if (select->from) {
		GList *list = select->from;
		GnomeDbTarget *target;
		while (list && !has_error) {
			target = parsed_create_target_sql_table (query, pdata, (sql_table *) list->data, error);
			if (target)
				g_hash_table_insert (pdata->sql_table_targets, list->data, target);
			else 
				has_error = TRUE;
			list = g_list_next (list);
		}
	}
	clean_old_targets (query, pdata);

	/*
	 * Joins: creating #GnomeDbJoin objects
	 */
	while (query->priv->joins_flat)
		gnome_db_base_nullify (GNOME_DB_BASE (query->priv->joins_flat->data));
	if (!has_error && select->from) {
		GList *list = select->from;
		while (list && !has_error) {
			has_error = ! parsed_create_join_sql_table (query, pdata, (sql_table *) list->data, error);
			list = g_list_next (list);
		}
	}

       	/*
	 * Fields: creating #GnomeDbField objects
	 */
	if (!has_error && select->fields) {
		GList *list = select->fields;
		gint i = 0;

		while (list && !has_error) {
			sql_field *field = (sql_field *) (list->data);
			GnomeDbField *qfield;

			qfield = parsed_create_global_query_field (query, TRUE, pdata, field, FALSE, NULL, NULL, error);
			if (!qfield)
				has_error = TRUE;
			else {
				/* place the field at position 'i' */
				if (g_slist_index (query->priv->fields, qfield) != i) {
					query->priv->fields = g_slist_remove (query->priv->fields, qfield);
					query->priv->fields = g_slist_insert (query->priv->fields, qfield, i);
				}

				/* set that field visible in case it was already there but not visible */
				gnome_db_qfield_set_visible (GNOME_DB_QFIELD (qfield), TRUE);
			}
			list = g_list_next (list);
			i++;
		}
	}
	clean_old_fields (query, pdata);

	/* 
	 * Make sure that for each join condition, all the fields participating in the condition are groupped
	 * together where the first visible field is.
	 */
	gnome_db_query_order_fields_using_join_conds (query);

	/*
	 * WHERE clause: creating a #GnomeDbCondition object
	 */
	if (query->priv->cond)
		gnome_db_base_nullify (GNOME_DB_BASE (query->priv->cond));
	if (!has_error && select->where) {
		GnomeDbCondition *cond = parsed_create_complex_condition (query, pdata, select->where, TRUE, NULL, error);
		if (!cond)
			has_error = TRUE;
		else {
			gnome_db_query_set_condition (query, cond);
			g_object_unref (G_OBJECT (cond));
		}
	}

	/*
	 * ORDER BY clause
	 */
	if (!has_error && select->order) {
		GList *list = select->order;
		gint i = 0;

		while (list && !has_error) {
			sql_order_field *of = (sql_order_field *) list->data;
			GnomeDbField *qfield;
			gboolean created;

			qfield = parsed_create_field_query_field (query, TRUE, pdata, of->name, TRUE, &created, 
								  NULL, error);
			if (qfield) {
				if (created)
					gnome_db_qfield_set_visible (GNOME_DB_QFIELD (qfield), FALSE);
				gnome_db_query_set_order_by_field (query, GNOME_DB_QFIELD (qfield), i, of->order_type == SQL_asc);
			}
			else {
				if (error && *error) {
					g_error_free (*error);
					*error = NULL;
				}

				if (g_list_length (of->name) == 1) {
					gint j = atoi ((char *) (of->name->data));
					gchar *str = g_strdup_printf ("%d", j);
					if (!strcmp ((char *) (of->name->data), str)) {
						qfield = (GnomeDbField *) gnome_db_entity_get_field_by_index (GNOME_DB_ENTITY (query), j-1);
						if (qfield) 
							gnome_db_query_set_order_by_field (query, GNOME_DB_QFIELD (qfield), i, 
										     of->order_type == SQL_asc);
						else {
							g_set_error (error,
								     GNOME_DB_QUERY_ERROR,
								     GNOME_DB_QUERY_SQL_ANALYSE_ERROR,
								     _("Invalid ORDER BY clause: can't find field number %d"), j);
							has_error = TRUE;
						}
					}
					else {
						g_set_error (error,
							     GNOME_DB_QUERY_ERROR,
							     GNOME_DB_QUERY_SQL_ANALYSE_ERROR,
							     _("Invalid ORDER BY clause"));
						has_error = TRUE;
					}
					g_free (str);
				}
				else {
					g_set_error (error,
						     GNOME_DB_QUERY_ERROR,
						     GNOME_DB_QUERY_SQL_ANALYSE_ERROR,
						     _("Invalid ORDER BY clause"));
					has_error = TRUE;
				}
			}
			list = g_list_next (list);
			i++;
		}
	}

	parse_data_destroy (pdata);

#ifdef debug_NO
	if (has_error) {
		if (error && *error)
			g_print ("Analysed SELECT query (ERROR: %s):\n", (*error)->message);
		else
			g_print ("Analysed SELECT query (ERROR: NO_MSG):\n");
	}
	else
		g_print ("Analysed SELECT query:\n");
	/* if (!gnome_db_referer_is_active (GNOME_DB_REFERER (query))) */
		gnome_db_base_dump (GNOME_DB_BASE (query), 10);
#endif
	return !has_error;
}


/*
 * main UPDATE analysis
 */
gboolean 
parsed_create_update_query (GnomeDbQuery *query, sql_update_statement *update, GError **error)
{
	gboolean has_error = FALSE;
	ParseData *pdata = parse_data_new (query);

	gnome_db_query_set_query_type (query, GNOME_DB_QUERY_TYPE_UPDATE);

	/*
	 * Target: creating the #GnomeDbTarget object
	 */
	if (update->table) {
		has_error = parsed_create_target_sql_table (query, pdata, update->table, error) ? FALSE : TRUE;
		
		clean_old_targets (query, pdata);
	}
	else {
		g_set_error (error,
			     GNOME_DB_QUERY_ERROR,
			     GNOME_DB_QUERY_SQL_ANALYSE_ERROR,
			     _("Missing UPDATE target entity"));
		has_error = TRUE;
	}

       	/*
	 * Fields: creating #GnomeDbField objects
	 */
	if (!has_error) {
		if (update->set) {
			GList *list = update->set;
			
			while (list && !has_error) {
				GnomeDbCondition *cond;

				cond = parsed_create_simple_condition (query, pdata, (sql_condition *) (list->data),
								       FALSE, NULL, error);
				if (!cond)
					has_error = TRUE;
				else {
					GnomeDbQfield *field_left, *field_right;

					g_assert (gnome_db_condition_get_cond_type (cond) == GNOME_DB_CONDITION_LEAF_EQUAL);

					field_left = gnome_db_condition_leaf_get_operator (cond, GNOME_DB_CONDITION_OP_LEFT);
					field_right = gnome_db_condition_leaf_get_operator (cond, GNOME_DB_CONDITION_OP_RIGHT);

					if (IS_GNOME_DB_QF_FIELD (field_left)) {
						g_object_set (G_OBJECT (field_left), "value_provider", field_right, NULL);
						gnome_db_qfield_set_visible (field_left, TRUE);
						gnome_db_qfield_set_visible (field_right, FALSE);
					}
					else {
						g_set_error (error,
							     GNOME_DB_QUERY_ERROR,
							     GNOME_DB_QUERY_SQL_ANALYSE_ERROR,
							     _("UPDATE target field is not an entity's field"));
						has_error = TRUE;
					}
					g_object_unref (G_OBJECT (cond));
				}
				list = g_list_next (list);
			}
		}
		else {
			g_set_error (error,
				     GNOME_DB_QUERY_ERROR,
				     GNOME_DB_QUERY_SQL_ANALYSE_ERROR,
				     _("Missing target fields to update"));
			has_error = TRUE;	
		}
	}
	clean_old_fields (query, pdata);

	/*
	 * WHERE clause: creating a #GnomeDbCondition object
	 */
	if (query->priv->cond)
		gnome_db_base_nullify (GNOME_DB_BASE (query->priv->cond));
	if (!has_error && update->where) {
		GnomeDbCondition *cond = parsed_create_complex_condition (query, pdata, update->where, TRUE, NULL, error);
		if (!cond)
			has_error = TRUE;
		else {
			gnome_db_query_set_condition (query, cond);
			g_object_unref (G_OBJECT (cond));
		}
	}

	parse_data_destroy (pdata);

#ifdef debug_NO
	if (has_error) {
		if (error && *error)
			g_print ("Analysed UPDATE query (ERROR: %s):\n", (*error)->message);
		else
			g_print ("Analysed UPDATE query (ERROR: NO_MSG):\n");
	}
	else
	g_print ("Analysed UPDATE query:\n");
	gnome_db_base_dump (GNOME_DB_BASE (query), 10);
#endif
	return !has_error;
}


/*
 * main INSERT analysis
 */
gboolean
parsed_create_insert_query (GnomeDbQuery *query, sql_insert_statement *insert, GError **error)
{
	gboolean has_error = FALSE;
	GSList *fields = NULL;
	GnomeDbTarget *target = NULL;
	ParseData *pdata = parse_data_new (query);

	gnome_db_query_set_query_type (query, GNOME_DB_QUERY_TYPE_INSERT);

	/*
	 * Target: creating the #GnomeDbTarget object
	 */
	if (insert->table) {
		has_error = parsed_create_target_sql_table (query, pdata, insert->table, error) ? FALSE : TRUE;
		if (!has_error)
			target = query->priv->targets->data;

		clean_old_targets (query, pdata);
	}
	else {
		g_set_error (error,
			     GNOME_DB_QUERY_ERROR,
			     GNOME_DB_QUERY_SQL_ANALYSE_ERROR,
			     _("Missing INSERT target entity"));
		has_error = TRUE;
	}

       	/*
	 * Fields: creating visible #GnomeDbField objects
	 */
	if (!has_error && insert->fields) {
		GList *list = insert->fields;
		
		while (list && !has_error) {
			sql_field *field = (sql_field *) (list->data);
			GnomeDbField *qfield;
			
			qfield = parsed_create_global_query_field (query, TRUE, pdata, field, FALSE, NULL, NULL, error);
			if (!qfield)
				has_error = TRUE;
			else {
				if (IS_GNOME_DB_QF_FIELD (qfield)) {
					gnome_db_qfield_set_visible (GNOME_DB_QFIELD (qfield), TRUE);
					fields = g_slist_append (fields, qfield);
				}
				else {
					g_set_error (error,
						     GNOME_DB_QUERY_ERROR,
						     GNOME_DB_QUERY_SQL_ANALYSE_ERROR,
						     _("INSERT target is not a field"));
					has_error = TRUE;
				}
			}
			
			list = g_list_next (list);
		}
	}
	clean_old_fields (query, pdata);

	/*
	 * Values: creating hidden #GnomeDbField objects
	 */
	if (!has_error && insert->values) {
		GList *list = insert->values;
		GSList *entity_fields = NULL;
		gint pos = 0;

		if (fields) {
			if (g_slist_length (fields) < g_list_length (insert->values)) {
				g_set_error (error,
					     GNOME_DB_QUERY_ERROR,
					     GNOME_DB_QUERY_SQL_ANALYSE_ERROR,
					     _("INSERT has more expressions than target columns"));
				has_error = TRUE;
			}
			if (g_slist_length (fields) > g_list_length (insert->values)) {
				g_set_error (error,
					     GNOME_DB_QUERY_ERROR,
					     GNOME_DB_QUERY_SQL_ANALYSE_ERROR,
					     _("INSERT has more target columns than expressions"));
				has_error = TRUE;
			}
		}
		else {
			entity_fields = gnome_db_entity_get_fields (gnome_db_target_get_represented_entity (target));
			if (g_slist_length (entity_fields) < g_list_length (insert->values)) {
				g_set_error (error,
					     GNOME_DB_QUERY_ERROR,
					     GNOME_DB_QUERY_SQL_ANALYSE_ERROR,
					     _("INSERT has more expressions than target columns"));
				has_error = TRUE;
			}
		}
		
		while (list && !has_error) {
			sql_field *field = (sql_field *) (list->data);
			GnomeDbField *qfield;
			
			qfield = parsed_create_global_query_field (query, TRUE, pdata, field, FALSE, NULL, NULL, error);
			if (!qfield)
				has_error = TRUE;
			else {
				if (!IS_GNOME_DB_QF_FIELD (qfield)) {
					gnome_db_qfield_set_visible (GNOME_DB_QFIELD (qfield), FALSE);
					if (fields)
						g_object_set (G_OBJECT (g_slist_nth_data (fields, pos)), "value_provider",
							      qfield, NULL);
					else {
						GnomeDbField *field;

						field = (GnomeDbField *) gnome_db_qf_field_new_with_objects (query, target,
												  g_slist_nth_data (entity_fields,
														    pos));
						gnome_db_qfield_set_visible (GNOME_DB_QFIELD (field), TRUE);
						gnome_db_entity_add_field (GNOME_DB_ENTITY (query), field);
						g_object_set (G_OBJECT (field), "value_provider", qfield, NULL);
						g_object_unref (G_OBJECT (field));
					}
				}
				else {
					g_set_error (error,
						     GNOME_DB_QUERY_ERROR,
						     GNOME_DB_QUERY_SQL_ANALYSE_ERROR,
						     _("INSERT expression is a target field"));
					has_error = TRUE;
				}
			}
			
			list = g_list_next (list);
			pos ++;
		}

		if (entity_fields)
			g_slist_free (entity_fields);
	}

	parse_data_destroy (pdata);

#ifdef debug_NO
	if (has_error) {
		if (error && *error)
			g_print ("Analysed INSERT query (ERROR: %s):\n", (*error)->message);
		else
			g_print ("Analysed INSERT query (ERROR: NO_MSG):\n");
	}
	else
	g_print ("Analysed INSERT query:\n");
	gnome_db_base_dump (GNOME_DB_BASE (query), 10);
#endif
	return !has_error;	
}

/*
 * main DELETE analysis
 */
gboolean
parsed_create_delete_query (GnomeDbQuery *query, sql_delete_statement *delete, GError **error)
{
	gboolean has_error = FALSE;
	ParseData *pdata = parse_data_new (query);

	gnome_db_query_set_query_type (query, GNOME_DB_QUERY_TYPE_DELETE);

	/*
	 * Target: creating the #GnomeDbTarget object
	 */
	if (delete->table) {
		has_error = parsed_create_target_sql_table (query, pdata, delete->table, error) ? FALSE : TRUE;
		clean_old_targets (query, pdata);
	}
	else {
		g_set_error (error,
			     GNOME_DB_QUERY_ERROR,
			     GNOME_DB_QUERY_SQL_ANALYSE_ERROR,
			     _("Missing DELETE target entity"));
		has_error = TRUE;
	}
	clean_old_fields (query, pdata);


	/*
	 * WHERE clause: creating a #GnomeDbCondition object
	 */
	if (query->priv->cond)
		gnome_db_base_nullify (GNOME_DB_BASE (query->priv->cond));
	if (!has_error && delete->where) {
		GnomeDbCondition *cond = parsed_create_complex_condition (query, pdata, delete->where, TRUE, NULL, error);
		if (!cond)
			has_error = TRUE;
		else {
			gnome_db_query_set_condition (query, cond);
			g_object_unref (G_OBJECT (cond));
		}
	}

	parse_data_destroy (pdata);

#ifdef debug_NO
	if (has_error) {
		if (error && *error)
			g_print ("Analysed DELETE query (ERROR: %s):\n", (*error)->message);
		else
			g_print ("Analysed DELETE query (ERROR: NO_MSG):\n");
	}
	else
	g_print ("Analysed DELETE query:\n");
	gnome_db_base_dump (GNOME_DB_BASE (query), 10);
#endif
	return !has_error;

}


/*
 * parsed_create_target_sql_table
 * @query: the query to work on
 * @pdata:
 * @table: the sql_table to analyse
 * @error: place to store the error, or %NULL
 *
 * Analyse a sql_table struct and makes a #GnomeDbTarget for it, or use an already existing one.
 *
 * Returns: a new GnomeDbTarget corresponding to the 'right' part of the join if we have a join.
 */
static GnomeDbTarget *
parsed_create_target_sql_table (GnomeDbQuery *query, ParseData *pdata, sql_table *table, GError **error)
{
	GnomeDbTarget *target = NULL;

	/* GnomeDbTarget creation */
	switch (table->type) {
	case SQL_simple:
		target = parsed_create_target_db_table (query, pdata, table->d.simple, table->as, error);
		break;
	case SQL_nestedselect:
		target = parsed_create_target_sub_select (query, pdata, table->d.select, table->as, error);
		break;
	case SQL_tablefunction:
		TO_IMPLEMENT;
		break;
	default:
		g_assert_not_reached ();
	}
	
	if (target)
		pdata->parsed_targets = g_slist_prepend (pdata->parsed_targets, target);
	
	return target;
}

/* 
 * parsed_create_join_sql_table
 * @query: the query to work on
 * @pdata:
 * @table: the sql_table to analyse
 * @error: place to store the error, or %NULL
 *
 * Analyse a sql_table struct and makes the #GnomeDbJoin object for it
 *
 * Returns: a new GnomeDbTarget corresponding to the 'right' part of the join if we have a join.
 */
static gboolean
parsed_create_join_sql_table (GnomeDbQuery *query, ParseData *pdata,
			      sql_table *table, GError **error)
{
	gboolean has_error = FALSE;
	GnomeDbTarget *target = g_hash_table_lookup (pdata->sql_table_targets, table);
	
	if (!target) {
		g_set_error (error,
			     GNOME_DB_QUERY_ERROR,
			     GNOME_DB_QUERY_SQL_ANALYSE_ERROR,
			     _("Internal error: can't find target"));
		has_error = TRUE;
	}

	if (!has_error && (table->join_type != SQL_cross_join)) {
		if (table->join_cond) { /* we have a join with a condition */
			GnomeDbCondition *cond = NULL;
			GSList *targets = NULL;
			
			cond = parsed_create_complex_condition (query, pdata,
								table->join_cond, FALSE, &targets, error);
			if (cond) {
				/* test nb of targets */
				if (g_slist_length (targets) != 2) {
					g_set_error (error,
						     GNOME_DB_QUERY_ERROR,
						     GNOME_DB_QUERY_SQL_ANALYSE_ERROR,
						     _("Join condition must be between two entities"));
					has_error = TRUE;
				}
				else {
					GnomeDbJoin *join;
					GnomeDbTarget *left_target = NULL;
					left_target = (GnomeDbTarget *) targets->data;
					if (left_target == target)
						left_target = (GnomeDbTarget *) g_slist_next (targets)->data;
					
					join = GNOME_DB_JOIN (gnome_db_join_new_with_targets (query, left_target, target));
					has_error = !gnome_db_query_add_join (query, join);
					gnome_db_join_set_join_type (join, parse_join_converter [table->join_type]);
					g_object_unref (G_OBJECT (join));
					
					if (!has_error && cond)
						has_error = !gnome_db_join_set_condition (join, cond);
					
				}
				
				if (targets)
					g_slist_free (targets);
				g_object_unref (G_OBJECT (cond));
			}
		}
		else { /* we have a join without any condition; try to find the correct other target */
			GSList *targets = g_slist_find (pdata->parsed_targets, target);
			GnomeDbJoin *join;
			GnomeDbTarget *left_target = NULL;
			GnomeDbEntity *ent = gnome_db_target_get_represented_entity (target);

			while (targets) {
				GnomeDbEntity *left_ent;
				GSList *list;
				left_ent = gnome_db_target_get_represented_entity (GNOME_DB_TARGET (targets->data));
				list = gnome_db_dict_get_entities_fk_constraints (gnome_db_base_get_dict ((GnomeDbBase*) query),
									    ent, left_ent, FALSE);
				if (list) {
					if (g_slist_length (list) != 1) {
						g_set_error (error,
							     GNOME_DB_QUERY_ERROR,
							     GNOME_DB_QUERY_SQL_ANALYSE_ERROR,
							     _("Ambiguous join"));
						has_error = TRUE;
					}
					else {
						if (left_target) {
							g_set_error (error,
								     GNOME_DB_QUERY_ERROR,
								     GNOME_DB_QUERY_SQL_ANALYSE_ERROR,
								     _("Ambiguous join"));
							has_error = TRUE;
						}
						else
							left_target = GNOME_DB_TARGET (targets->data);
					}
					g_slist_free (list);
				}
				targets = g_slist_next (targets);
			}
			
			if (left_target && !has_error) {
				join = GNOME_DB_JOIN (gnome_db_join_new_with_targets (query, left_target, target));
				has_error = !gnome_db_query_add_join (query, join);
				gnome_db_join_set_join_type (join, parse_join_converter [table->join_type]);
				g_object_unref (G_OBJECT (join));
			}
		}
	}

	return ! has_error;
}

/*
 * Create a new target within query, from a sub query
 * @query: the query to add the target to
 * @pdata:
 * @select: corresponding SELECT sub query for the target
 * @as: corresponding alias for the target
 * @error: place to store the error, or %NULL
 *
 * Creates a new target from the @select and @as entries. However, if a target was already present in @prev_targets_hash,
 * then no new target is created and the existing target is returned, after having been removed from @prev_targets_hash
 */
static GnomeDbTarget *
parsed_create_target_sub_select (GnomeDbQuery *query, ParseData *pdata, sql_select_statement *select, const gchar *as, GError **error)
{
	GnomeDbTarget *target = NULL;
	GnomeDbQuery *subq;
	gboolean err = FALSE;
	gchar *str;

	/* if there is a suitable target in @prev_targets_hash, then don't create a new one */
	if (as && *as) {
		GSList *list = pdata->prev_targets;
		while (list && !target) {
			GnomeDbTarget *tmp = GNOME_DB_TARGET (list->data);
			const gchar *tmpstr = gnome_db_target_get_alias (GNOME_DB_TARGET (tmp));
			
			if (!strcmp (tmpstr, as))
				target = GNOME_DB_TARGET (list->data);
			if (!target) {
				str = g_utf8_strdown (as, -1);
				if (!strcmp (tmpstr, str))
					target = GNOME_DB_TARGET (list->data);
				g_free (str);
			}
		}
		
		list = g_slist_next (list);
	}
	
	if (target) {
		GnomeDbEntity *ent = gnome_db_target_get_represented_entity (target);
		
		if (!IS_GNOME_DB_QUERY (ent))
			target = NULL; /* the target found MUST reference a sub query, otherwise we don't want it */
	}
	
	if (!target) {
		/* create new sub query and target */
		subq = GNOME_DB_QUERY (gnome_db_query_new (gnome_db_base_get_dict (GNOME_DB_BASE (query))));
		err = !parsed_create_select_query (subq, select, error);
		if (!err) {
			gnome_db_query_add_sub_query (query, subq);
			target = GNOME_DB_TARGET (gnome_db_target_new_with_entity (query, GNOME_DB_ENTITY (subq)));
			if (as && *as) {
				gnome_db_base_set_name (GNOME_DB_BASE (subq), as);
				gnome_db_target_set_alias (target, as);
			}
			if (! gnome_db_query_add_target (query, target, error)) {
				g_object_unref (target);
				target = NULL;
			}
			g_object_unref (target);
		}
		g_object_unref (subq);
	}
	else {
		/* only parse the existing target's sub query */
		subq = GNOME_DB_QUERY (gnome_db_target_get_represented_entity (target));
		err = !parsed_create_select_query (subq, select, error);
		if (!err) 
			pdata->prev_targets = g_slist_remove (pdata->prev_targets, target);			
		else
			target = NULL;
	}

	return target;
}
			       


/*
 * Create a new target within query, from a table name
 * @query: the query to add the target to
 * @pdata:
 * @table: corresponding table name for the target
 * @as: corresponding alias for the target
 * @error: place to store the error, or %NULL
 *
 * Creates a new target from the @name and @as entries. However, if a target was already present in @prev_targets_hash,
 * then no new target is created and the existing target is returned, after having been removed from @prev_targets_hash
 */
static GnomeDbTarget *
parsed_create_target_db_table (GnomeDbQuery *query, ParseData *pdata,
			       const gchar *table, const gchar *as, GError **error)
{
	GnomeDbTarget *target = NULL;
	GnomeDbTable *db_table;
	GnomeDbDatabase *db = gnome_db_dict_get_database (gnome_db_base_get_dict (GNOME_DB_BASE (query)));
	gboolean has_error = FALSE;
	gchar *str;
	GSList *list;

	/* referenced entity: first try to find the table corresponding to a lower-case version of @table, 
	 * and if not found, try with the unchanged @table */
	db_table = gnome_db_database_get_table_by_name (db, table);
	if (!db_table) {
		g_set_error (error,
			     GNOME_DB_QUERY_ERROR,
			     GNOME_DB_QUERY_SQL_ANALYSE_ERROR,
			     _("Can't find target '%s'"), table);
		return NULL;
	}

	/* if there is a suitable target in @prev_targets_hash, then don't create a new one */
	list = pdata->prev_targets;
	while (list) {
		GnomeDbTarget *tmp = NULL;
		
		/* compare referenced db table */
		if (gnome_db_target_get_represented_entity (GNOME_DB_TARGET (list->data)) == (GnomeDbEntity *) db_table) {
			tmp = GNOME_DB_TARGET (list->data);
			if (!target)
				target = tmp;
		}
		
		if (as && *as && tmp) {
			/* look for target alias */
			const gchar *tmpstr = gnome_db_target_get_alias (GNOME_DB_TARGET (tmp));
			
			if (!strcmp (tmpstr, as))
				target = GNOME_DB_TARGET (list->data);
			if (!target) {
				str = g_utf8_strdown (as, -1);
				if (!strcmp (tmpstr, str))
					target = GNOME_DB_TARGET (list->data);
				g_free (str);
			}
		}
		
		list = g_slist_next (list);
	}
	
	if (target) {
		if (as && *as)
			gnome_db_target_set_alias (target, as);
		pdata->prev_targets = g_slist_remove (pdata->prev_targets, target);
		return target;
	}

	target = (GnomeDbTarget *) gnome_db_target_new_with_entity (query, GNOME_DB_ENTITY (db_table));
	has_error = ! gnome_db_query_add_target (query, target, error);
	if (as && *as)
		gnome_db_target_set_alias (target, as);
	gnome_db_base_set_name (GNOME_DB_BASE (target), gnome_db_base_get_name (GNOME_DB_BASE (db_table)));
	g_object_unref (G_OBJECT (target));

	if (has_error && target)
		target = NULL;

	return target;
}

/*
 * Create a new GnomeDbQfield object from the sql_field_item structure, and put it into @query.
 * Returns NULL if the creation failed.
 * @query: the query to add the field to
 * @pdata:
 * @field: sql_field_item to be analysed
 * @new_field: a place to store if a new field has been created, or one already existing has
 *             been returned
 * @target_return: place to store the target which has been used, or %NULL (on error, it is unchanged)
 * @error: place to store the error, or %NULL
 *
 * Returns: NULL if value has not been interpreted as a valid field name
 */
GnomeDbField *
parsed_create_global_query_field (GnomeDbQuery *query, gboolean add_to_query, 
				  ParseData *pdata, sql_field *field, 
				  gboolean try_existing_field, gboolean *new_field,
				  GnomeDbTarget **target_return, GError **error)
{
	sql_field_item *field_item = (sql_field_item *) (field->item);
	GnomeDbField *qfield = NULL;
	GError *my_error = NULL;
	GError **error_ptr;

	if (error)
		error_ptr = error;
	else
		error_ptr = &my_error;

	switch (field_item->type) {
	case SQL_name:
		/* first try a value QueryField */
		if (g_list_length (field_item->d.name) == 1)
			qfield = parsed_create_value_query_field (query, add_to_query, pdata,
								  (gchar*) field_item->d.name->data, 
								  field->param_spec, new_field, error_ptr);
		if (!qfield && !(*error_ptr))
			/* then try an entity.field QueryField */
			qfield = parsed_create_field_query_field (query, add_to_query, pdata,
								  field_item->d.name, try_existing_field, new_field, 
								  target_return, error_ptr);
		if (qfield && field->as && *(field->as) && !IS_GNOME_DB_QF_ALL (qfield)) {
			gnome_db_qfield_set_alias (GNOME_DB_QFIELD (qfield), field->as);
			gnome_db_base_set_name (GNOME_DB_BASE (qfield), field->as);
		}
			
		break;
	case SQL_equation:
		/* Example: SELECT ..., a+b, ... FROM table */
		/* not yet doable, need the creation of a GnomeDbQfEquation object which inherits GnomeDbQfield */
		TO_IMPLEMENT;
		break;
	case SQL_inlineselect:
		/* Example: SELECT ..., (SELECT a FROM table WHERE...), ... FROM table */
		/* not yet doable, need the creation of a GnomeDbQfSelect object which inherits GnomeDbQfield */
		TO_IMPLEMENT;
		break;
	case SQL_function:
		qfield = parsed_create_func_query_field (query, add_to_query, pdata,
							 field_item->d.function.funcname,
							 field_item->d.function.funcarglist, 
							 try_existing_field, new_field, error_ptr);
		break;
	}

	if (*error_ptr && !error) 
		g_error_free (*error_ptr);

	return qfield;
}


/*
 * Create a GnomeDbQfField object for the provided name.
 * @query: the query to add the field to
 * @pdata:
 * @field_names: a list of string for each dot separated field: table.field => 2 strings 
 * @new_field: place to store if a new field has been created, or if the returned value is an existing field
 * @target_return: place to store the target which has been used, or %NULL (on error, it is unchanged)
 * @error: place to store the error, or %NULL
 *
 * Returns: NULL if value has not been interpreted as a valid field name
 */
static GnomeDbField *
parsed_create_field_query_field (GnomeDbQuery *query, gboolean add_to_query, 
				 ParseData *pdata, GList *field_names,
				 gboolean try_existing_field, gboolean *new_field, 
				 GnomeDbTarget **target_return, GError **error)
{
	GnomeDbField *qfield = NULL;
	GList *list;
	gint i;
	gboolean has_error = FALSE;
	GString *sql_name;

	/* build complete SQL name */
	sql_name = g_string_new ("");
	list = field_names;
	while (list) {
		if (list != field_names)
			g_string_append (sql_name, ".");
		g_string_append (sql_name, (gchar *)list->data);
		list = g_list_next (list);
	}

	/* try to find an existing field 
	 * if try_existing_field is TRUE, try it, beware with the value_provider which need updating
	 * which might be a problem.
	 */
	if (try_existing_field && * (sql_name->str)) {
		if (pdata->prev_fields)
			qfield = (GnomeDbField *) gnome_db_query_get_field_by_sql_naming_fields (query, sql_name->str, 
										      pdata->prev_fields);
		else
			qfield = (GnomeDbField *) gnome_db_query_get_field_by_sql_naming (query, sql_name->str);
		if (qfield) {
			g_string_free (sql_name, TRUE);
			if (new_field)
				*new_field = FALSE;
			pdata->prev_fields = g_slist_remove (pdata->prev_fields, qfield);
			return qfield;
		}
	}

	/* no pre-existing field found */
	list = field_names;
	i = g_list_length (list);
	if ((i>2) || (i==0)) {
		if (i==0) 
			g_set_error (error,
				     GNOME_DB_QUERY_ERROR,
				     GNOME_DB_QUERY_SYNTAX_ERROR,
				     _("Invalid empty field name"));
		else 
			g_set_error (error,
				     GNOME_DB_QUERY_ERROR,
				     GNOME_DB_QUERY_SYNTAX_ERROR,
				     _("Invalid field name '%s'"), sql_name->str);
		
		has_error = TRUE;
	}
	else {
		GnomeDbTarget *target = NULL;
		GnomeDbField *field = NULL;
		
		gchar *table_alias = NULL;
		gchar *field_name = NULL;
		
		if (i==2) { /* Like: "table.field" or "table_alias.field" */
			table_alias = g_utf8_strdown ((gchar *) (list->data), -1);
			field_name = (gchar *) (g_list_next (list)->data);
			target = g_hash_table_lookup (pdata->new_targets, table_alias);
			if (!target) {
				g_set_error (error,
					     GNOME_DB_QUERY_ERROR,
					     GNOME_DB_QUERY_SQL_ANALYSE_ERROR,
					     _("Can't find target '%s'"), table_alias);
				has_error = TRUE;
			}
			else {
				if (*field_name != '*') {
					if (!strcmp (field_name, "null")) {
						TO_IMPLEMENT;
					}
					field = gnome_db_entity_get_field_by_name 
						(gnome_db_target_get_represented_entity (target), field_name);
					if (field) {
						/* try to see if such a field already exist */
						/* FIXME */
						qfield = (GnomeDbField *) gnome_db_qf_field_new_with_objects (query, 
												   target, field);
						if (target_return)
							*target_return = target;
					}
					else {
						g_set_error (error,
							     GNOME_DB_QUERY_ERROR,
							     GNOME_DB_QUERY_SQL_ANALYSE_ERROR,
							     _("Can't find field '%s'"), field_name);
						has_error = TRUE;
					}
				}
				else {
					qfield = (GnomeDbField *) gnome_db_qf_all_new_with_target (query, target);
					if (target_return)
						*target_return = target;
				}
			}
			g_free (table_alias);
		}
		else { /* Like: "field" */
			GSList *targets = query->priv->targets;
			field_name = (gchar *) (list->data);
			if (*field_name != '*') {
				if (!strcmp (field_name, "null")) {
					TO_IMPLEMENT;
					/* PB: we don't know the exact data type of the value! */
				}
				else {
					while (targets && !has_error) {
						GnomeDbField *tmp;
						
						tmp = gnome_db_entity_get_field_by_name 
							(gnome_db_target_get_represented_entity (GNOME_DB_TARGET (targets->data)),
							 field_name);
						if (tmp) {
							if (field) {
								g_set_error (error,
									     GNOME_DB_QUERY_ERROR,
									     GNOME_DB_QUERY_SQL_ANALYSE_ERROR,
									     _("Ambiguous field '%s'"), field_name);
								has_error = TRUE;
							}
							else {
								target = (GnomeDbTarget *) targets->data;
								field = tmp;
							}
						}
						targets = g_slist_next (targets);
					}
					if (field) {
						qfield = (GnomeDbField *) gnome_db_qf_field_new_with_objects (query, 
												   target, field);
						if (target_return)
							*target_return = target;
					}
					else {
						g_set_error (error,
							     GNOME_DB_QUERY_ERROR,
							     GNOME_DB_QUERY_SQL_ANALYSE_ERROR,
							     _("Can't find field '%s'"), field_name);
						has_error = TRUE;
					}
				}
			}
			else {
				/* field_name == '*' */
				if (g_slist_length (query->priv->targets) != 1) {
					g_set_error (error,
						     GNOME_DB_QUERY_ERROR,
						     GNOME_DB_QUERY_SQL_ANALYSE_ERROR,
						     _("Ambiguous field '*'"));
					has_error = TRUE;	
				}
				else {
					target = GNOME_DB_TARGET (query->priv->targets->data);
					qfield = (GnomeDbField *) gnome_db_qf_all_new_with_target (query, target);
					if (target_return)
						*target_return = target;
				}
			}
		}
		
		if (qfield) {
			/* actual field association to the query */
			gnome_db_base_set_name (GNOME_DB_BASE (qfield), field_name);
			
			if (add_to_query) {
				gnome_db_entity_add_field (GNOME_DB_ENTITY (query), qfield);
				g_object_unref (G_OBJECT (qfield));
			}
		}
	}

	g_string_free (sql_name, TRUE);
	if (new_field)
		*new_field = TRUE;

	return qfield;
}

/*
 * Create a GnomeDbQfValue object for the provided value.
 * @query: the query to add the field to
 * @pdata:
 * @value: string SQL representation of the value
 * @error: place to store the error, or %NULL
 *
 * Returns: NULL if value has not been interpreted as a valid value
 */
static GnomeDbField *
parsed_create_value_query_field (GnomeDbQuery *query, gboolean add_to_query, ParseData *pdata,
				 const gchar *value, GList *param_specs, gboolean *new_field, GError **error)
{
	GnomeDbField *qfield = NULL;
	GList *list;
	gint i = 0;
	GnomeDbDataHandler *dh;
	GdaValue *gdaval;
	GnomeDbServerDataType *real_type = NULL;
	GnomeDbServer *srv;
	GdaValueType gdatype;
	gboolean unspecvalue = FALSE;
	gboolean created = TRUE; /* FIXME: compute that value */
	gboolean isparam;

	srv = gnome_db_dict_get_server (gnome_db_base_get_dict (GNOME_DB_BASE (query)));	
	
	if (*value == 0)
		unspecvalue = TRUE; /* the value is not specified, it needs to be a parameter */

	/*
	 * try to determine a suitable GnomeDbServerDataType for the value 
	 */
	list = param_specs;
	while (list && !real_type) {
		if (((param_spec*) list->data)->type == PARAM_type) {
			real_type = gnome_db_server_get_data_type_by_name (srv, ((param_spec*) list->data)->content);
			if (!real_type) {
				g_set_error (error,
					     GNOME_DB_QUERY_ERROR,
					     GNOME_DB_QUERY_SQL_ANALYSE_ERROR,
					     _("Can't find data type '%s'"), ((param_spec*) list->data)->content);
				return NULL;
			}
		}
		
		list = g_list_next (list);
	}

	if (!real_type && unspecvalue) {
		g_set_error (error,
			     GNOME_DB_QUERY_ERROR,
			     GNOME_DB_QUERY_SQL_ANALYSE_ERROR,
			     _("Missing data type for empty value"));
		return NULL;
	}

	if (!real_type) {
		/* Try to find a nice type for the value, as none was provided ! */
		GnomeDbServerInfo *sinfo = gnome_db_server_get_server_info (srv);

		if (sinfo) {
			gboolean keep_type = FALSE;
			
			while ((i < sinfo->value_nb_tests_types) && !real_type) {
				dh = gnome_db_server_get_handler_by_gda (srv, sinfo->value_test_gda_types [i]);
				gdaval = gnome_db_data_handler_get_value_from_sql (dh, value, sinfo->value_test_gda_types [i]); 
				if (gdaval) {
					gchar *sql = gnome_db_data_handler_get_sql_from_value (dh, gdaval);
					if (sql) {
						keep_type = !strcmp (value, sql);
						/*g_print ("TEST : %s => %s => %s\n", value, sql, keep_type ? "OK": "NOK");*/
						g_free (sql);
					}
					gda_value_free (gdaval);
				}
				
				if (keep_type) 
					real_type = gnome_db_server_get_data_type_by_name (srv, sinfo->value_test_data_types [i]);
				i++;
			}

			if (!real_type) {
				/* Don't set error here because not finding any real type can mean that we 
				   don't have a value in the first place */
				return NULL;
			}
		}
		else {
			/* no server info, display warning and return */
			g_warning ("No ServerInfo available for the current provider, can't try to "
				   "create a value query field for field '%s'. Please contact GnomeDb "
				   "maintainer to add missing ServerInfo.", value);
			return NULL;	
		}
	}

	
	/*
	 * Actual GnomeDbField creation
	 * here, real_type can't be NULL 
	 */
	qfield = (GnomeDbField *) gnome_db_qf_value_new (query, real_type);
	if (unspecvalue) 
		gnome_db_qf_value_set_is_parameter (GNOME_DB_QF_VALUE (qfield), TRUE); /* we really need it to be a param */
	else {
		gdatype = gnome_db_server_data_type_get_gda_type (real_type);
		dh = gnome_db_server_get_handler_by_gda (srv, gdatype);
		gdaval = gnome_db_data_handler_get_value_from_sql (dh, value, gdatype);
		gnome_db_qf_value_set_value (GNOME_DB_QF_VALUE (qfield), gdaval);
		gda_value_free (gdaval);
		gnome_db_qf_value_set_is_parameter (GNOME_DB_QF_VALUE (qfield), FALSE);
	}

	if (add_to_query) {
		gnome_db_entity_add_field (GNOME_DB_ENTITY (query), qfield);
		g_object_unref (G_OBJECT (qfield));
	}
	
	/*
	 * Using the param spec to set attributes
	 */
	isparam = g_list_length (param_specs) > 0 ? TRUE : FALSE;
	list = param_specs;
	while (list) {
		param_spec *pspec = (param_spec*) list->data;

		switch (pspec->type) {
		case PARAM_name:
			gnome_db_base_set_name (GNOME_DB_BASE (qfield), pspec->content);
			break;
		case PARAM_descr:
			gnome_db_base_set_description (GNOME_DB_BASE (qfield), pspec->content);
			break;
		case PARAM_isparam:
			isparam = ! ((* pspec->content == 'f') || * pspec->content == 'F');
			break;
		case PARAM_nullok:
			gnome_db_qf_value_set_not_null (GNOME_DB_QF_VALUE (qfield), 
						  ! ((* pspec->content == 't') || (* pspec->content == 'T')));
			break;
		default:
			break;
		}		
		list = g_list_next (list);
	}
	if (! gnome_db_qf_value_is_parameter (GNOME_DB_QF_VALUE (qfield)))
		gnome_db_qf_value_set_is_parameter (GNOME_DB_QF_VALUE (qfield), isparam);

	if (new_field)
		*new_field = created;
	return qfield;
}


/*
 * Create a GnomeDbQfFunc or GnomeDbQfAgg object correspponding to @funcname
 * @query: the query to add the field to
 * @pdata:
 * @funcname: the name of the function or the aggregate
 * @funcargs: arguments of the function or aggregate
 * @error: place to store the error, or %NULL
 *
 * Returns: NULL if value has not been interpreted as a valid value
 */
static GnomeDbField *
parsed_create_func_query_field (GnomeDbQuery *query, gboolean add_to_query, ParseData *pdata,
				const gchar *funcname, GList *funcargs, 
				gboolean try_existing_field, gboolean *new_field, GError **error)
{
	GnomeDbField *qfield = NULL;
	GSList *list;
	GSList *args = NULL, *arg_types = NULL;
	GList *dlist;
	GnomeDbServerFunction *function = NULL;
	GnomeDbServerAggregate *agg = NULL;
 	gboolean has_error = FALSE;
	gboolean created = TRUE; /* FIXME: compute that value */

	GnomeDbServer *srv;	
	srv = gnome_db_dict_get_server (gnome_db_base_get_dict (GNOME_DB_BASE (query)));

	/* Add the fields corresponding to the list of arguments */
	dlist = funcargs;
	while (dlist && !has_error) {
		gboolean new_field;

		GnomeDbField *qfield = parsed_create_global_query_field (query, TRUE, pdata, 
									 (sql_field *)dlist->data, 
									 try_existing_field, &new_field, 
									 NULL, error);

		if (!qfield) 
			return NULL;
		else {
			args = g_slist_append (args, qfield);
			arg_types = g_slist_append (arg_types, gnome_db_qfield_get_data_type (GNOME_DB_QFIELD (qfield)));

			if (new_field)
				gnome_db_qfield_set_visible (GNOME_DB_QFIELD (qfield), FALSE);
		}
		dlist = g_list_next (dlist);
	}

	/* find the right GnomeDbServerFunction object */
	function = gnome_db_server_get_function_by_name_arg (srv, funcname, arg_types);
	
	/* find the right GnomeDbServerAggregate object */
	if (!has_error && !function && (g_slist_length (arg_types) == 1)) 
		agg = gnome_db_server_get_aggregate_by_name_arg (srv, funcname, arg_types->data);
	
	if (function || agg) {
		if (function) {
			/* actual object creation for GnomeDbServerFunction */
			qfield = GNOME_DB_FIELD (gnome_db_qf_func_new_with_func (query, 
										 GNOME_DB_SERVER_FUNCTION (function)));
			has_error = ! gnome_db_qf_func_set_args (GNOME_DB_QF_FUNC (qfield), args);
		}
		else {
			GnomeDbQfield *arg = NULL;
			if (args && (g_slist_length (args) == 1))
				arg = GNOME_DB_QFIELD (args->data);
				
			/* actual object creation for GnomeDbServerAggregate */
			qfield = GNOME_DB_FIELD (gnome_db_qf_agg_new_with_agg (query, GNOME_DB_SERVER_AGGREGATE (agg)));
			if (arg)
				has_error = ! gnome_db_qf_agg_set_arg (GNOME_DB_QF_AGG (qfield), arg);
			else
				has_error = TRUE;
		}

		gnome_db_base_set_name (GNOME_DB_BASE (qfield), funcname);
		
		if (has_error) {
			if (qfield)
				g_object_unref (qfield);
			qfield = NULL;
			g_set_error (error,
				     GNOME_DB_QUERY_ERROR,
				     GNOME_DB_QUERY_SQL_ANALYSE_ERROR,
				     _("Can't set aggregate or function's arguments"));
		}
		else {
			if (add_to_query) {
				gnome_db_entity_add_field (GNOME_DB_ENTITY (query), qfield);
				g_object_unref (G_OBJECT (qfield));
			}
		}
	}
	else {
		GString *string = g_string_new ("");
		
		g_string_append_printf (string, "%s (", funcname);
		list = arg_types;
		while (list) {
			g_string_append (string, (list == arg_types) ? "" : ", ");
			g_string_append_printf (string, "%s", gnome_db_base_get_name (GNOME_DB_BASE (list->data)));
			list = g_slist_next (list);
		}
		g_string_append (string, ")");
		g_set_error (error,
			     GNOME_DB_QUERY_ERROR,
			     GNOME_DB_QUERY_SQL_ANALYSE_ERROR,
			     _("Can't find function or aggregate '%s'"), string->str);
		g_string_free (string, TRUE);
	}

	/* memory de-allocation */
	g_slist_free (args);
	g_slist_free (arg_types);

	if (new_field)
		*new_field = created;
	return qfield;
}


GnomeDbCondition *
parsed_create_complex_condition (GnomeDbQuery *query, ParseData *pdata, sql_where *where,
				 gboolean try_existing_field, GSList **targets_return, GError **error)
{
	GnomeDbCondition *cond = NULL, *tmpcond, *tmpcond2;

	g_return_val_if_fail (where, NULL);

	switch (where->type) {
	case SQL_single:
		cond = parsed_create_simple_condition (query, pdata, where->d.single, try_existing_field, 
						       targets_return, error);
		break;
	case SQL_negated:
		tmpcond = parsed_create_complex_condition (query, pdata, where->d.negated, 
							   try_existing_field, targets_return, error);
		if (tmpcond) {
			cond = (GnomeDbCondition*) gnome_db_condition_new (query, GNOME_DB_CONDITION_NODE_NOT);
			if (! gnome_db_condition_node_add_child (cond, tmpcond, error)) {
				g_object_unref (G_OBJECT (cond));
				cond = NULL;
			}
			g_object_unref (G_OBJECT (tmpcond));
		}
		break;
	case SQL_pair:
		tmpcond = parsed_create_complex_condition (query, pdata, where->d.pair.left, 
							   try_existing_field, targets_return, error);
		if (tmpcond) {
			tmpcond2 = parsed_create_complex_condition (query, pdata, where->d.pair.right, 
								    try_existing_field, targets_return, error);
			if (tmpcond2) {
				switch (where->d.pair.op) {
				case SQL_and:
					cond = (GnomeDbCondition*) gnome_db_condition_new (query, GNOME_DB_CONDITION_NODE_AND);
					break;
				case SQL_or:
					cond = (GnomeDbCondition*) gnome_db_condition_new (query, GNOME_DB_CONDITION_NODE_OR);
					break;
				default:
					g_assert_not_reached ();
				}
				if (! gnome_db_condition_node_add_child (cond, tmpcond, error)) {
					g_object_unref (G_OBJECT (cond));
					cond = NULL;
				}
				if (cond && ! gnome_db_condition_node_add_child (cond, tmpcond2, error)) {
					g_object_unref (G_OBJECT (cond));
					cond = NULL;
				}
				g_object_unref (G_OBJECT (tmpcond));
				g_object_unref (G_OBJECT (tmpcond2));
			}
		}
		break;
	}

	return cond;
}


/*
 *  Always return a LEAF GnomeDbCondition object
 */
static GnomeDbCondition *
parsed_create_simple_condition (GnomeDbQuery *query, ParseData *pdata, sql_condition *sqlcond, 
				gboolean try_existing_field, GSList **targets_return, GError **error)
{
	GnomeDbCondition *cond = NULL;
	GnomeDbConditionType condtype = parse_condop_converter [sqlcond->op];
	gboolean has_error = FALSE;
	GnomeDbField *fields[] = {NULL, NULL, NULL};
	GnomeDbTarget *target;

	/* getting GnomeDbQfield objects */
	if (condtype == GNOME_DB_CONDITION_LEAF_BETWEEN) {
		gboolean created;
		target = NULL;
		fields[0] = parsed_create_global_query_field (query, TRUE, pdata, 
							      sqlcond->d.between.field, 
							      try_existing_field , &created, 
							      &target, error);
		if (created && fields[0])
			gnome_db_qfield_set_visible (GNOME_DB_QFIELD (fields[0]), FALSE);
		if (fields[0]) {
			if (targets_return && target && !g_slist_find (*targets_return, target))
				*targets_return = g_slist_append (*targets_return, target);

			target = NULL;
			fields[1] = parsed_create_global_query_field (query, TRUE, pdata, 
								      sqlcond->d.between.lower, 
								      try_existing_field, &created, 
								      &target, error);
		}

		if (created && fields[1])
			gnome_db_qfield_set_visible (GNOME_DB_QFIELD (fields[1]), FALSE);
		
		if (fields[1]) {
			if (targets_return && target && !g_slist_find (*targets_return, target))
				*targets_return = g_slist_append (*targets_return, target);
			
			target = NULL;
			fields[2] = parsed_create_global_query_field (query, TRUE, pdata, 
								      sqlcond->d.between.upper, 
								      try_existing_field, &created, 
								      &target, error);
		}
		if (created && fields[2])
			gnome_db_qfield_set_visible (GNOME_DB_QFIELD (fields[2]), FALSE);

		if (fields[2]) {
			if (targets_return && target && !g_slist_find (*targets_return, target))
				*targets_return = g_slist_append (*targets_return, target);
		}

		if (! fields[0] || ! fields[1] || ! fields[2])
			has_error = TRUE;
	}
	else {
		gboolean created;
		target = NULL;
		fields[0] = parsed_create_global_query_field (query, TRUE, pdata, 
							      sqlcond->d.pair.left, 
							      try_existing_field, &created, 
							      &target, error);
		if (created && fields[0])
			gnome_db_qfield_set_visible (GNOME_DB_QFIELD (fields[0]), FALSE);
		
		if (fields[0]) {
			if (targets_return && target && !g_slist_find (*targets_return, target))
				*targets_return = g_slist_append (*targets_return, target);

			target = NULL;
			fields[1] = parsed_create_global_query_field (query, TRUE, pdata, 
								      sqlcond->d.pair.right, 
								      try_existing_field, &created, 
								      &target, error);
		}
		if (created && fields[1])
			gnome_db_qfield_set_visible (GNOME_DB_QFIELD (fields[1]), FALSE);

		if (fields[1]) {
			if (targets_return && target && !g_slist_find (*targets_return, target))
				*targets_return = g_slist_append (*targets_return, target);
		}

		if (! fields[0] || ! fields[1])
			has_error = TRUE;
	}
	if (has_error) 
		return NULL;

	/* from now on, all the required GnomeDbQfield objects do exist */
	/* cond creation */
	cond = (GnomeDbCondition*) gnome_db_condition_new (query, condtype);
	gnome_db_condition_leaf_set_operator (cond, GNOME_DB_CONDITION_OP_LEFT, GNOME_DB_QFIELD (fields[0]));
	gnome_db_condition_leaf_set_operator (cond, GNOME_DB_CONDITION_OP_RIGHT, GNOME_DB_QFIELD (fields[1]));
	if (condtype == GNOME_DB_CONDITION_LEAF_BETWEEN)
		gnome_db_condition_leaf_set_operator (cond, GNOME_DB_CONDITION_OP_RIGHT2, GNOME_DB_QFIELD (fields[2]));

	if (sqlcond->negated) {
		GnomeDbCondition *cond2 = (GnomeDbCondition*) gnome_db_condition_new (query, GNOME_DB_CONDITION_NODE_NOT);
		gnome_db_condition_node_add_child (cond2, cond, NULL);
		g_object_unref (G_OBJECT (cond));
		cond = cond2;
	}

	return cond;
}

