/* gnome-db-server-info.c
 *
 * Copyright (C) 2004 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include "gnome-db-server.h"
#include "gnome-db-server-info.h"

static GnomeDbServerInfo *none_info_build ();
static GnomeDbServerInfo *postgres_info_build ();
static GnomeDbServerInfo *mysql_info_build ();
static GnomeDbServerInfo *sqlite_info_build ();

GnomeDbServerInfo *
gnome_db_server_info_get (const gchar *provider_id)
{
	if (!provider_id || ! (*provider_id) )
		return none_info_build ();

	if (!strcmp (provider_id, "SQLite")) 
		return sqlite_info_build ();

	if (!strcmp (provider_id, "PostgreSQL")) 
		return postgres_info_build ();

	if (!strcmp (provider_id, "MySQL")) 
		return mysql_info_build ();

	return NULL;
}

void
gnome_db_server_info_free (GnomeDbServerInfo *info)
{
	g_free (info->value_test_data_types);
	g_free (info->value_test_gda_types);
	g_free (info);
}

typedef struct {
	GdaValueType  gda_type;
	gchar        *dbms_type;
} Assoc;

static GnomeDbServerInfo *
postgres_info_build ()
{
	GnomeDbServerInfo *info;
	gint i, max, current;

	/* WARNING: the order in which the data types appear in this array is important:
	 * the array is scanned starting from the first entry when the query parser needs to
	 * find the type for a value (represented by its SQL string) and it stops at the first
	 * occurence of a data type which seems to fit
	 *
	 * The empty ("") entries mean that the DBMS can't produce that kind of data type
	 */
	Assoc transl[] = {
		{GDA_VALUE_TYPE_SMALLINT,        "int2"},
		{GDA_VALUE_TYPE_INTEGER,         "int4"},
		{GDA_VALUE_TYPE_BIGINT,          "int8"},
		{GDA_VALUE_TYPE_SINGLE,          "float4"},
		{GDA_VALUE_TYPE_DOUBLE,          "float8"},
		{GDA_VALUE_TYPE_BOOLEAN,         "bool"},
		{GDA_VALUE_TYPE_TIME,            "time"},
		{GDA_VALUE_TYPE_DATE,            "date"},
		{GDA_VALUE_TYPE_TIMESTAMP,       "timestamp"},
		{GDA_VALUE_TYPE_GEOMETRIC_POINT, "point"},
		{GDA_VALUE_TYPE_STRING,          "varchar"},
		{GDA_VALUE_TYPE_NUMERIC,         "numeric"},
		{GDA_VALUE_TYPE_BINARY,          "bytea"},
		{GDA_VALUE_TYPE_BIGUINT,         ""},
		{GDA_VALUE_TYPE_BLOB,            ""},
		{GDA_VALUE_TYPE_GOBJECT,         ""},
		{GDA_VALUE_TYPE_LIST,            ""},
		{GDA_VALUE_TYPE_MONEY,           ""},
		{GDA_VALUE_TYPE_SMALLUINT,       ""},
		{GDA_VALUE_TYPE_TINYINT,         ""},
		{GDA_VALUE_TYPE_TINYUINT,        ""},
		{GDA_VALUE_TYPE_TYPE,            ""},
		{GDA_VALUE_TYPE_UINTEGER,        ""},
	};

	/* Assoc transl[] = { */
/* 		{GDA_VALUE_TYPE_INTEGER,         "int4"}, */
/* 		{GDA_VALUE_TYPE_BOOLEAN,         "bool"}, */
/* 		{GDA_VALUE_TYPE_TIME,            "time"}, */
/* 		{GDA_VALUE_TYPE_DATE,            "date"}, */
/* 		{GDA_VALUE_TYPE_TIMESTAMP,       "timestamp"}, */
/* 		{GDA_VALUE_TYPE_STRING,          "varchar"}, */
/* 	}; */

	info = g_new0 (GnomeDbServerInfo, 1);
	info->provider_name = "PostgreSQL";

	info->is_case_insensitive = TRUE;

	info->implicit_data_types_casts = TRUE;
	
	/* mem allocation */
	info->value_nb_tests_types = 0;
	max = sizeof (transl) / sizeof (Assoc);
	for (i = 0; i < max; i++) 
		if (transl[i].dbms_type && ( *(transl[i].dbms_type)))
			info->value_nb_tests_types ++;
	info->value_test_data_types = g_new0 (gchar *, info->value_nb_tests_types);
	info->value_test_gda_types = g_new0 (GdaValueType, info->value_nb_tests_types);

	/* mem copy */
	current = 0;
	for (i = 0; i < max; i++) 
		if (transl[i].dbms_type && ( *(transl[i].dbms_type))) {
			info->value_test_data_types [current] = transl[i].dbms_type;
			info->value_test_gda_types [current] = transl[i].gda_type;
			current++;
		}

#ifdef debug
	g_print ("%s default data types: %d\n", info->provider_name, current);
	for (i = 0; i < current; i++) 
		g_print ("\tGDA %02d --> %s\n", info->value_test_gda_types [i], info->value_test_data_types [i]);
#endif
	
	return info;
}

static GnomeDbServerInfo *
mysql_info_build ()
{
	GnomeDbServerInfo *info;
	gint i, max, current;

	/* WARNING: the order in which the data types appear in this array is important:
	 * the array is scanned starting from the first entry when the query parser needs to
	 * find the type for a value (represented by its SQL string) and it stops at the first
	 * occurence of a data type which seems to fit
	 *
	 * The empty ("") entries mean that the DBMS can't produce that kind of data type
	 */
	Assoc transl[] = {
		{GDA_VALUE_TYPE_INTEGER,         "int"},
		{GDA_VALUE_TYPE_BOOLEAN,         "tinyint"},
		{GDA_VALUE_TYPE_TIME,            "time"},
		{GDA_VALUE_TYPE_TIMESTAMP,       "timestamp"},
		{GDA_VALUE_TYPE_STRING,          "varchar"},
		{GDA_VALUE_TYPE_BIGINT,          ""},
		{GDA_VALUE_TYPE_BIGUINT,         ""},
		{GDA_VALUE_TYPE_BINARY,          ""},
		{GDA_VALUE_TYPE_BLOB,            ""},
		{GDA_VALUE_TYPE_DATE,            ""},
		{GDA_VALUE_TYPE_DOUBLE,          ""},
		{GDA_VALUE_TYPE_GEOMETRIC_POINT, ""},
		{GDA_VALUE_TYPE_GOBJECT,         ""},
		{GDA_VALUE_TYPE_LIST,            ""},
		{GDA_VALUE_TYPE_MONEY,           ""},
		{GDA_VALUE_TYPE_NUMERIC,         ""},
		{GDA_VALUE_TYPE_SINGLE,          ""},
		{GDA_VALUE_TYPE_SMALLINT,        ""},
		{GDA_VALUE_TYPE_SMALLUINT,       ""},
		{GDA_VALUE_TYPE_TINYINT,         ""},
		{GDA_VALUE_TYPE_TINYUINT,        ""},
		{GDA_VALUE_TYPE_TYPE,            ""},
		{GDA_VALUE_TYPE_UINTEGER,        ""},
	};

	info = g_new0 (GnomeDbServerInfo, 1);
	info->provider_name = "MySQL";

	info->is_case_insensitive = TRUE;

	info->implicit_data_types_casts = TRUE;
	
	/* mem allocation */
	info->value_nb_tests_types = 0;
	max = sizeof (transl) / sizeof (Assoc);
	for (i = 0; i < max; i++) 
		if (transl[i].dbms_type && ( *(transl[i].dbms_type)))
			info->value_nb_tests_types ++;
	info->value_test_data_types = g_new0 (gchar *, info->value_nb_tests_types);
	info->value_test_gda_types = g_new0 (GdaValueType, info->value_nb_tests_types);

	/* mem copy */
	current = 0;
	for (i = 0; i < max; i++) 
		if (transl[i].dbms_type && ( *(transl[i].dbms_type))) {
			info->value_test_data_types [current] = transl[i].dbms_type;
			info->value_test_gda_types [current] = transl[i].gda_type;
			current++;
		}

#ifdef debug
	g_print ("%s default data types: %d\n", info->provider_name, current);
	for (i = 0; i < current; i++) 
		g_print ("\tGDA %02d --> %s\n", info->value_test_gda_types [i], info->value_test_data_types [i]);
#endif
	
	return info;
}

static GnomeDbServerInfo *
sqlite_info_build ()
{
	GnomeDbServerInfo *info;
	gint i, max, current;

	/* WARNING: the order in which the data types appear in this array is important:
	 * the array is scanned starting from the first entry when the query parser needs to
	 * find the type for a value (represented by its SQL string) and it stops at the first
	 * occurence of a data type which seems to fit
	 *
	 * The empty ("") entries mean that the DBMS can't produce that kind of data type
	 */
	Assoc transl[] = {
		{GDA_VALUE_TYPE_INTEGER,         "int"},
		{GDA_VALUE_TYPE_BOOLEAN,         ""},
		{GDA_VALUE_TYPE_TIME,            ""},
		{GDA_VALUE_TYPE_TIMESTAMP,       ""},
		{GDA_VALUE_TYPE_STRING,          "varchar"},
		{GDA_VALUE_TYPE_BIGINT,          ""},
		{GDA_VALUE_TYPE_BIGUINT,         ""},
		{GDA_VALUE_TYPE_BINARY,          ""},
		{GDA_VALUE_TYPE_BLOB,            "blob"},
		{GDA_VALUE_TYPE_DATE,            ""},
		{GDA_VALUE_TYPE_DOUBLE,          "real"},
		{GDA_VALUE_TYPE_GEOMETRIC_POINT, ""},
		{GDA_VALUE_TYPE_GOBJECT,         ""},
		{GDA_VALUE_TYPE_LIST,            ""},
		{GDA_VALUE_TYPE_MONEY,           ""},
		{GDA_VALUE_TYPE_NUMERIC,         ""},
		{GDA_VALUE_TYPE_SINGLE,          ""},
		{GDA_VALUE_TYPE_SMALLINT,        ""},
		{GDA_VALUE_TYPE_SMALLUINT,       ""},
		{GDA_VALUE_TYPE_TINYINT,         ""},
		{GDA_VALUE_TYPE_TINYUINT,        ""},
		{GDA_VALUE_TYPE_TYPE,            ""},
		{GDA_VALUE_TYPE_UINTEGER,        ""},
	};

	info = g_new0 (GnomeDbServerInfo, 1);
	info->provider_name = "SQLite";

	info->is_case_insensitive = TRUE;

	info->implicit_data_types_casts = TRUE;
	
	/* mem allocation */
	info->value_nb_tests_types = 0;
	max = sizeof (transl) / sizeof (Assoc);
	for (i = 0; i < max; i++) 
		if (transl[i].dbms_type && ( *(transl[i].dbms_type)))
			info->value_nb_tests_types ++;
	info->value_test_data_types = g_new0 (gchar *, info->value_nb_tests_types);
	info->value_test_gda_types = g_new0 (GdaValueType, info->value_nb_tests_types);

	/* mem copy */
	current = 0;
	for (i = 0; i < max; i++) 
		if (transl[i].dbms_type && ( *(transl[i].dbms_type))) {
			info->value_test_data_types [current] = transl[i].dbms_type;
			info->value_test_gda_types [current] = transl[i].gda_type;
			current++;
		}

#ifdef debug
	g_print ("%s default data types: %d\n", info->provider_name, current);
	for (i = 0; i < current; i++) 
		g_print ("\tGDA %02d --> %s\n", info->value_test_gda_types [i], info->value_test_data_types [i]);
#endif
	
	return info;
}


/*
 * Creates a GnomeDbServerInfo structure to be used by a GnomeDbServer when
 * no data source has been set
 */
static
GnomeDbServerInfo *none_info_build ()
{
	GnomeDbServerInfo *info;
	GdaValueType gda_type;

	info = g_new0 (GnomeDbServerInfo, 1);
	info->provider_name = NULL;

	info->is_case_insensitive = TRUE;
	info->implicit_data_types_casts = TRUE;
	
	info->value_nb_tests_types = GDA_VALUE_TYPE_UNKNOWN - 1;
	info->value_test_data_types = g_new0 (gchar *, GDA_VALUE_TYPE_UNKNOWN);
	info->value_test_gda_types = g_new0 (GdaValueType, GDA_VALUE_TYPE_UNKNOWN);

	gda_type = GDA_VALUE_TYPE_NULL;
	for (gda_type ++; gda_type < GDA_VALUE_TYPE_UNKNOWN; gda_type ++) {
		info->value_test_data_types [gda_type - 1] = gda_type_to_string (gda_type);
		info->value_test_gda_types [gda_type - 1] = gda_type;
	}
	
	return info;	
}
