/* gnome-db-handler-boolean.c
 *
 * Copyright (C) 2003 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-handler-boolean.h"
#include "gnome-db-entry-boolean.h"
#include "gnome-db-data-cell-renderer-boolean.h"
#include <libgnomedb/gnome-db-server.h>

static void gnome_db_handler_boolean_class_init (GnomeDbHandlerBooleanClass * class);
static void gnome_db_handler_boolean_init (GnomeDbHandlerBoolean * wid);
static void gnome_db_handler_boolean_dispose (GObject   * object);


/* GnomeDbDataHandler interface */
static void         gnome_db_handler_boolean_data_handler_init      (GnomeDbDataHandlerIface *iface);
static GnomeDbDataEntry *gnome_db_handler_boolean_get_entry_from_value   (GnomeDbDataHandler *dh, const GdaValue *value, 
							       GdaValueType type);
static GtkCellRenderer *gnome_db_handler_boolean_get_cell_renderer  (GnomeDbDataHandler * dh, GdaValueType type);
static gchar       *gnome_db_handler_boolean_get_sql_from_value     (GnomeDbDataHandler *dh, const GdaValue *value);
static gchar       *gnome_db_handler_boolean_get_str_from_value     (GnomeDbDataHandler *dh, const GdaValue *value);
static GdaValue    *gnome_db_handler_boolean_get_value_from_sql     (GnomeDbDataHandler *dh, const gchar *sql, 
							       GdaValueType type);
static GdaValue    *gnome_db_handler_boolean_get_sane_init_value    (GnomeDbDataHandler * dh, GdaValueType type);

static guint        gnome_db_handler_boolean_get_nb_gda_types       (GnomeDbDataHandler *dh);
static GdaValueType gnome_db_handler_boolean_get_gda_type_index     (GnomeDbDataHandler *dh, guint index);
static gboolean     gnome_db_handler_boolean_accepts_gda_type       (GnomeDbDataHandler * dh, GdaValueType type);

static const gchar *gnome_db_handler_boolean_get_descr              (GnomeDbDataHandler *dh);
static const gchar *gnome_db_handler_boolean_get_descr_detail       (GnomeDbDataHandler *dh);
static const gchar *gnome_db_handler_boolean_get_version            (GnomeDbDataHandler *dh);
static gboolean     gnome_db_handler_boolean_is_plugin              (GnomeDbDataHandler *dh);
static const gchar *gnome_db_handler_boolean_get_plugin_name        (GnomeDbDataHandler *dh);
static const gchar *gnome_db_handler_boolean_get_plugin_file        (GnomeDbDataHandler *dh);
static gchar       *gnome_db_handler_boolean_get_key                (GnomeDbDataHandler *dh);


/* signals */
enum
{
	LAST_SIGNAL
};

static gint gnome_db_handler_boolean_signals[LAST_SIGNAL] = { };

struct  _GnomeDbHandlerBooleanPriv {
	gchar          *detailled_descr;
	guint           nb_gda_types;
	GdaValueType   *valid_gda_types;
	GnomeDbServer       *srv;
};

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

guint
gnome_db_handler_boolean_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbHandlerBooleanClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_handler_boolean_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbHandlerBoolean),
			0,
			(GInstanceInitFunc) gnome_db_handler_boolean_init
		};		

		static const GInterfaceInfo data_entry_info = {
			(GInterfaceInitFunc) gnome_db_handler_boolean_data_handler_init,
			NULL,
			NULL
		};

		type = g_type_register_static (GNOME_DB_BASE_TYPE, "GnomeDbHandlerBoolean", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_DATA_HANDLER_TYPE, &data_entry_info);
	}
	return type;
}

static void
gnome_db_handler_boolean_data_handler_init (GnomeDbDataHandlerIface *iface)
{
	iface->get_entry_from_value = gnome_db_handler_boolean_get_entry_from_value;
	iface->get_cell_renderer = gnome_db_handler_boolean_get_cell_renderer;
	iface->get_sql_from_value = gnome_db_handler_boolean_get_sql_from_value;
	iface->get_str_from_value = gnome_db_handler_boolean_get_str_from_value;
	iface->get_value_from_sql = gnome_db_handler_boolean_get_value_from_sql;
	iface->get_value_from_str = NULL;
	iface->get_sane_init_value = gnome_db_handler_boolean_get_sane_init_value;
	iface->get_nb_gda_types = gnome_db_handler_boolean_get_nb_gda_types;
	iface->accepts_gda_type = gnome_db_handler_boolean_accepts_gda_type;
	iface->get_gda_type_index = gnome_db_handler_boolean_get_gda_type_index;
	iface->get_descr = gnome_db_handler_boolean_get_descr;
	iface->get_descr_detail = gnome_db_handler_boolean_get_descr_detail;
	iface->get_version = gnome_db_handler_boolean_get_version;
	iface->is_plugin = gnome_db_handler_boolean_is_plugin;
	iface->get_plugin_name = gnome_db_handler_boolean_get_plugin_name;
	iface->get_plugin_file = gnome_db_handler_boolean_get_plugin_file;
	iface->get_key = gnome_db_handler_boolean_get_key;
}


static void
gnome_db_handler_boolean_class_init (GnomeDbHandlerBooleanClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	
	parent_class = g_type_class_peek_parent (class);

	object_class->dispose = gnome_db_handler_boolean_dispose;
}

static void
gnome_db_handler_boolean_init (GnomeDbHandlerBoolean * hdl)
{
	/* Private structure */
	hdl->priv = g_new0 (GnomeDbHandlerBooleanPriv, 1);
	hdl->priv->detailled_descr = _("Boolean values handler");
	hdl->priv->nb_gda_types = 1;
	hdl->priv->valid_gda_types = g_new0 (GdaValueType, 1);
	hdl->priv->valid_gda_types[0] = GDA_VALUE_TYPE_BOOLEAN;
	hdl->priv->srv = NULL;

	gnome_db_base_set_name (GNOME_DB_BASE (hdl), _("InternalBoolean"));
	gnome_db_base_set_description (GNOME_DB_BASE (hdl), _("Booleans representation"));
}

static void
gnome_db_handler_boolean_dispose (GObject *object)
{
	GnomeDbHandlerBoolean *hdl;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_HANDLER_BOOLEAN (object));

	hdl = GNOME_DB_HANDLER_BOOLEAN (object);

	if (hdl->priv) {
		gnome_db_base_nullify_check (GNOME_DB_BASE (object));

		g_free (hdl->priv->valid_gda_types);
		hdl->priv->valid_gda_types = NULL;

		if (hdl->priv->srv)
			g_object_remove_weak_pointer (G_OBJECT (hdl->priv->srv),
						      (gpointer *) & (hdl->priv->srv));

		g_free (hdl->priv);
		hdl->priv = NULL;
	}

	/* for the parent class */
	parent_class->dispose (object);
}

/**
 * gnome_db_handler_boolean_new
 * @srv: a #GnomeDbServer object
 *
 * Creates a data handler for booleans
 *
 * Returns: the new object
 */
GObject *
gnome_db_handler_boolean_new (GnomeDbServer *srv)
{
	GObject *obj;
	GnomeDbHandlerBoolean *hdl;

	g_return_val_if_fail (srv && IS_GNOME_DB_SERVER (srv), NULL);
	obj = g_object_new (GNOME_DB_HANDLER_BOOLEAN_TYPE, NULL);
	hdl = GNOME_DB_HANDLER_BOOLEAN (obj);

	g_object_add_weak_pointer (G_OBJECT (srv), (gpointer *) &(hdl->priv->srv));
	hdl->priv->srv = srv;

	return obj;
}


/* Interface implementation */
static GnomeDbDataEntry *
gnome_db_handler_boolean_get_entry_from_value (GnomeDbDataHandler *iface, const GdaValue *value, GdaValueType type)
{
	GnomeDbHandlerBoolean *hdl;
	GnomeDbDataEntry *de;
	GdaValueType real_type;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_BOOLEAN (iface), NULL);
	hdl = GNOME_DB_HANDLER_BOOLEAN (iface);
	g_return_val_if_fail (hdl->priv, NULL);

	if (value && (gda_value_get_type (value) != GDA_VALUE_TYPE_NULL)) {
		real_type = gda_value_get_type (value);
		g_return_val_if_fail (gnome_db_data_handler_accepts_gda_type (iface, type), NULL);
	}
	else
		real_type = type;

	de = GNOME_DB_DATA_ENTRY (gnome_db_entry_boolean_new (iface, real_type));
	if (value && (gda_value_get_type (value) != GDA_VALUE_TYPE_NULL))
		gnome_db_data_entry_set_value (de, value);
	else 
		gnome_db_data_entry_set_value (de, NULL);

	return de;
}

static GtkCellRenderer *
gnome_db_handler_boolean_get_cell_renderer  (GnomeDbDataHandler *iface, GdaValueType type)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_BOOLEAN (iface), NULL);
	g_return_val_if_fail (GNOME_DB_HANDLER_BOOLEAN (iface)->priv, NULL);
	
	return gnome_db_data_cell_renderer_boolean_new (iface, type);
}

static gchar *
gnome_db_handler_boolean_get_sql_from_value (GnomeDbDataHandler *iface, const GdaValue *value)
{
	gchar *retval;
	GnomeDbHandlerBoolean *hdl;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_BOOLEAN (iface), NULL);
	hdl = GNOME_DB_HANDLER_BOOLEAN (iface);
	g_return_val_if_fail (hdl->priv, NULL);

	if (gda_value_get_boolean (value)) 
		retval = g_strdup ("TRUE");
	else
		retval = g_strdup ("FALSE");

	return retval;
}

static gchar *
gnome_db_handler_boolean_get_str_from_value (GnomeDbDataHandler *iface, const GdaValue *value)
{
	GnomeDbHandlerBoolean *hdl;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_BOOLEAN (iface), NULL);
	hdl = GNOME_DB_HANDLER_BOOLEAN (iface);
	g_return_val_if_fail (hdl->priv, NULL);

	return gnome_db_server_stringify_value (hdl->priv->srv, value);
}

static GdaValue *
gnome_db_handler_boolean_get_value_from_sql (GnomeDbDataHandler *iface, const gchar *sql, GdaValueType type)
{
	GnomeDbHandlerBoolean *hdl;
	GdaValue *value;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_BOOLEAN (iface), NULL);
	hdl = GNOME_DB_HANDLER_BOOLEAN (iface);
	g_return_val_if_fail (hdl->priv, NULL);

	if ((*sql == 't') || (*sql == 'T'))
		value = gda_value_new_boolean (TRUE);
	else
		value = gda_value_new_boolean (FALSE);
	return value;
}

static GdaValue *
gnome_db_handler_boolean_get_sane_init_value (GnomeDbDataHandler *iface, GdaValueType type)
{
	GnomeDbHandlerBoolean *hdl;
	GdaValue *value;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_BOOLEAN (iface), NULL);
	hdl = GNOME_DB_HANDLER_BOOLEAN (iface);
	g_return_val_if_fail (hdl->priv, NULL);

	value = gda_value_new_boolean (FALSE);
	return value;
}

static guint
gnome_db_handler_boolean_get_nb_gda_types (GnomeDbDataHandler *iface)
{
	GnomeDbHandlerBoolean *hdl;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_BOOLEAN (iface), 0);
	hdl = GNOME_DB_HANDLER_BOOLEAN (iface);
	g_return_val_if_fail (hdl->priv, 0);

	return hdl->priv->nb_gda_types;
}


static gboolean
gnome_db_handler_boolean_accepts_gda_type (GnomeDbDataHandler *iface, GdaValueType type)
{
	GnomeDbHandlerBoolean *hdl;
	guint i = 0;
	gboolean found = FALSE;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_BOOLEAN (iface), FALSE);
	g_return_val_if_fail (type != GDA_VALUE_TYPE_UNKNOWN, FALSE);
	hdl = GNOME_DB_HANDLER_BOOLEAN (iface);
	g_return_val_if_fail (hdl->priv, 0);

	while (!found && (i < hdl->priv->nb_gda_types)) {
		if (hdl->priv->valid_gda_types [i] == type)
			found = TRUE;
		i++;
	}

	return found;
}

static GdaValueType
gnome_db_handler_boolean_get_gda_type_index (GnomeDbDataHandler *iface, guint index)
{
	GnomeDbHandlerBoolean *hdl;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_BOOLEAN (iface), GDA_VALUE_TYPE_UNKNOWN);
	hdl = GNOME_DB_HANDLER_BOOLEAN (iface);
	g_return_val_if_fail (hdl->priv, GDA_VALUE_TYPE_UNKNOWN);
	g_return_val_if_fail (index < hdl->priv->nb_gda_types, GDA_VALUE_TYPE_UNKNOWN);

	return hdl->priv->valid_gda_types[index];
}

static const gchar *
gnome_db_handler_boolean_get_descr (GnomeDbDataHandler *iface)
{
	GnomeDbHandlerBoolean *hdl;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_BOOLEAN (iface), NULL);
	hdl = GNOME_DB_HANDLER_BOOLEAN (iface);
	g_return_val_if_fail (hdl->priv, NULL);

	return gnome_db_base_get_description (GNOME_DB_BASE (hdl));
}

static const gchar *
gnome_db_handler_boolean_get_descr_detail (GnomeDbDataHandler *iface)
{
	GnomeDbHandlerBoolean *hdl;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_BOOLEAN (iface), NULL);
	hdl = GNOME_DB_HANDLER_BOOLEAN (iface);
	g_return_val_if_fail (hdl->priv, NULL);

	return hdl->priv->detailled_descr;
}

static const gchar *
gnome_db_handler_boolean_get_version (GnomeDbDataHandler *iface)
{
	GnomeDbHandlerBoolean *hdl;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_BOOLEAN (iface), NULL);
	hdl = GNOME_DB_HANDLER_BOOLEAN (iface);
	g_return_val_if_fail (hdl->priv, NULL);
	
	return g_strdup ("Internal");
}

static gboolean
gnome_db_handler_boolean_is_plugin (GnomeDbDataHandler *iface)
{
	GnomeDbHandlerBoolean *hdl;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_BOOLEAN (iface), FALSE);
	hdl = GNOME_DB_HANDLER_BOOLEAN (iface);
	g_return_val_if_fail (hdl->priv, FALSE);

	return FALSE;
}

static const gchar *
gnome_db_handler_boolean_get_plugin_name (GnomeDbDataHandler *iface)
{
	GnomeDbHandlerBoolean *hdl;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_BOOLEAN (iface), NULL);
	hdl = GNOME_DB_HANDLER_BOOLEAN (iface);
	g_return_val_if_fail (hdl->priv, NULL);
	
	return NULL;
}

static const gchar *
gnome_db_handler_boolean_get_plugin_file (GnomeDbDataHandler *iface)
{
	GnomeDbHandlerBoolean *hdl;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_BOOLEAN (iface), NULL);
	hdl = GNOME_DB_HANDLER_BOOLEAN (iface);
	g_return_val_if_fail (hdl->priv, NULL);
	
	return NULL;
}

static gchar *
gnome_db_handler_boolean_get_key (GnomeDbDataHandler *iface)
{
	GnomeDbHandlerBoolean *hdl;

	g_return_val_if_fail (iface && IS_GNOME_DB_HANDLER_BOOLEAN (iface), NULL);
	hdl = GNOME_DB_HANDLER_BOOLEAN (iface);
	g_return_val_if_fail (hdl->priv, NULL);
	
	return g_strdup (gnome_db_base_get_name (GNOME_DB_BASE (hdl)));
}

