/* gnome-db-parameter.c
 *
 * Copyright (C) 2003 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include "gnome-db-parameter.h"
#include "marshal.h"
#include "gnome-db-dict.h"
#include "gnome-db-query.h"
#include "gnome-db-entity.h"
#include "gnome-db-field.h"
#include "gnome-db-server-data-type.h"
#include "gnome-db-xml-storage.h"
#include "gnome-db-referer.h"
#include "gnome-db-data-model.h"

/* 
 * Main static functions 
 */
static void gnome_db_parameter_class_init (GnomeDbParameterClass * class);
static void gnome_db_parameter_init (GnomeDbParameter * srv);
static void gnome_db_parameter_dispose (GObject   * object);
static void gnome_db_parameter_finalize (GObject   * object);

static void gnome_db_parameter_set_property    (GObject              *object,
						guint                 param_id,
						const GValue         *value,
						GParamSpec           *pspec);
static void gnome_db_parameter_get_property    (GObject              *object,
						guint                 param_id,
						GValue               *value,
						GParamSpec           *pspec);

/* Referer interface */
static void        gnome_db_parameter_referer_init        (GnomeDbRefererIface *iface);
static GSList     *gnome_db_parameter_get_ref_objects     (GnomeDbReferer *iface);
static void        gnome_db_parameter_replace_refs        (GnomeDbReferer *iface, GHashTable *replacements);

static void gnome_db_parameter_add_for_field (GnomeDbParameter *param, GnomeDbField *for_field);
static void gnome_db_parameter_del_for_field (GnomeDbParameter *param, GnomeDbField *for_field);
static void gnome_db_parameter_set_data_type (GnomeDbParameter *param, GnomeDbServerDataType *type);

static void nullified_object_cb (GnomeDbBase *obj, GnomeDbParameter *param);
static void nullified_for_field_cb (GnomeDbBase *obj, GnomeDbParameter *param);
static void nullified_depend_on_cb (GnomeDbBase *obj, GnomeDbParameter *param);
static void nullified_source_field_cb (GnomeDbBase *obj, GnomeDbParameter *param);
static void nullified_alias_of_cb (GnomeDbBase *obj, GnomeDbParameter *param);

static void alias_of_changed_cb (GnomeDbParameter *alias_of, GnomeDbParameter *param);

static void gnome_db_parameter_signal_changed (GnomeDbBase *base, gboolean block_changed_signal);
#ifdef debug
static void gnome_db_parameter_dump (GnomeDbParameter *parameter, guint offset);
#endif

static void gnome_db_parameter_set_full_bind_param (GnomeDbParameter *param, GnomeDbParameter *alias_of);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	LAST_SIGNAL
};

static gint gnome_db_parameter_signals[LAST_SIGNAL] = {  };

/* properties */
enum
{
	PROP_0,
	PROP_PLUGIN,
	PROP_QF_INTERNAL,
	PROP_USE_DEFAULT_VALUE,
	PROP_SIMPLE_BIND,
	PROP_FULL_BIND
};


struct _GnomeDbParameterPrivate
{
	GSList                *for_fields;
	GnomeDbServerDataType *type;
	GnomeDbParameter      *alias_of;     /* FULL bind to param */
	GnomeDbParameter      *change_with;  /* SIMPLE bind to param */
	
	gboolean               invalid_forced;
	gboolean               valid;
	gboolean               default_forced;
	GdaValue              *value;
	GdaValue              *default_value; /* CAN be either NULL or of any type */
	gboolean               not_null;      /* TRUE if 'value' must not be NULL when passed to destination fields */

	GnomeDbField          *in_field;      /* if not NULL, limits the possible values for the parameter */
	GSList                *dependencies;  /* other GnomeDbParameter objects on which the parameter depends on */

	gchar                 *plugin;        /* plugin to be used for user interaction */

	gboolean               user_input_required;
};

/* module error */
GQuark gnome_db_parameter_error_quark (void)
{
        static GQuark quark;
        if (!quark)
                quark = g_quark_from_static_string ("gnome_db_parameter_error");
        return quark;
}

guint
gnome_db_parameter_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbParameterClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_parameter_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbParameter),
			0,
			(GInstanceInitFunc) gnome_db_parameter_init
		};

		static const GInterfaceInfo referer_info = {
			(GInterfaceInitFunc) gnome_db_parameter_referer_init,
			NULL,
			NULL
		};
		
		type = g_type_register_static (GNOME_DB_BASE_TYPE, "GnomeDbParameter", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_REFERER_TYPE, &referer_info);
	}
	return type;
}

static void
gnome_db_parameter_referer_init (GnomeDbRefererIface *iface)
{
	iface->activate = NULL;
	iface->deactivate = NULL;
	iface->is_active = NULL;
	iface->get_ref_objects = gnome_db_parameter_get_ref_objects;
	iface->replace_refs = gnome_db_parameter_replace_refs;
}

static void
gnome_db_parameter_class_init (GnomeDbParameterClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);


	/* virtual functions */
	GNOME_DB_BASE_CLASS (class)->signal_changed = gnome_db_parameter_signal_changed;
#ifdef debug
        GNOME_DB_BASE_CLASS (class)->dump = (void (*)(GnomeDbBase *, guint)) gnome_db_parameter_dump;
#endif

	object_class->dispose = gnome_db_parameter_dispose;
	object_class->finalize = gnome_db_parameter_finalize;

	/* Properties */
	object_class->set_property = gnome_db_parameter_set_property;
	object_class->get_property = gnome_db_parameter_get_property;
	g_object_class_install_property (object_class, PROP_PLUGIN,
					 g_param_spec_string ("handler_plugin", NULL, NULL, NULL, 
							      (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	g_object_class_install_property (object_class, PROP_USE_DEFAULT_VALUE,
					 g_param_spec_boolean ("use_default_value", NULL, NULL, FALSE,
							       (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	g_object_class_install_property (object_class, PROP_SIMPLE_BIND,
					 g_param_spec_pointer ("simple_bind", NULL, NULL, (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	g_object_class_install_property (object_class, PROP_FULL_BIND,
					 g_param_spec_pointer ("full_bind", NULL, NULL, (G_PARAM_READABLE | G_PARAM_WRITABLE)));
}

static void
gnome_db_parameter_init (GnomeDbParameter *parameter)
{
	parameter->priv = g_new0 (GnomeDbParameterPrivate, 1);
	parameter->priv->for_fields = NULL;
	parameter->priv->type = NULL;
	parameter->priv->alias_of = NULL;
	parameter->priv->change_with = NULL;

	parameter->priv->invalid_forced = FALSE;
	parameter->priv->valid = TRUE;
	parameter->priv->default_forced = FALSE;
	parameter->priv->value = NULL;
	parameter->priv->default_value = NULL;

	parameter->priv->not_null = FALSE;
	parameter->priv->in_field = NULL;
	parameter->priv->dependencies = NULL;
	parameter->priv->plugin = NULL;
	parameter->priv->user_input_required = FALSE;
}

/**
 * gnome_db_parameter_new
 * @dict: a #GnomeDbDict object
 * @type: the #GnomeDbServerDataType requested
 *
 * Creates a new parameter of type @type
 *
 * Returns: a new #GnomeDbParameter object
 */
GObject *
gnome_db_parameter_new (GnomeDbDict *dict, GnomeDbServerDataType *type)
{
	GObject   *obj;
	GnomeDbParameter *param;

        g_return_val_if_fail (!dict || IS_GNOME_DB_DICT (dict), NULL);
	g_return_val_if_fail (type && IS_GNOME_DB_SERVER_DATA_TYPE (type), NULL);

        obj = g_object_new (GNOME_DB_PARAMETER_TYPE, "dict", ASSERT_DICT (dict), NULL);
	param = GNOME_DB_PARAMETER (obj);

	gnome_db_parameter_set_data_type (param, type);

        return obj;
}

/**
 * gnome_db_parameter_new_copy
 * @orig: a #GnomeDbParameter object to copy
 *
 * Copy constructor.
 *
 * Returns: a new #GnomeDbParameter object
 */
GObject *
gnome_db_parameter_new_copy (GnomeDbParameter *orig)
{
	GObject   *obj;
	GnomeDbParameter *param;
	GSList *list;

	g_return_val_if_fail (orig && IS_GNOME_DB_PARAMETER (orig), NULL);
	g_return_val_if_fail (orig->priv, NULL);

	obj = g_object_new (GNOME_DB_PARAMETER_TYPE, "dict", gnome_db_base_get_dict (GNOME_DB_BASE (orig)), NULL);
	param = GNOME_DB_PARAMETER (obj);

	gnome_db_base_set_name (GNOME_DB_BASE (param), gnome_db_base_get_name (GNOME_DB_BASE (orig)));
	gnome_db_base_set_description (GNOME_DB_BASE (param), gnome_db_base_get_description (GNOME_DB_BASE (orig)));

	/* setting which require calling another function */
	gnome_db_parameter_set_data_type (param, orig->priv->type);
	list = orig->priv->for_fields;
	while (list) {
		gnome_db_parameter_add_for_field (param, GNOME_DB_FIELD (list->data));
		list = g_slist_next (list);
	}
	if (orig->priv->alias_of)
		gnome_db_parameter_set_full_bind_param (param, orig->priv->alias_of);
	if (orig->priv->change_with)
		gnome_db_parameter_bind_to_param (param, orig->priv->change_with);
	
	if (orig->priv->in_field) {
		/*g_print ("set source field to %p for param %p\n", orig->priv->in_field, param);*/
		gnome_db_parameter_set_source_field (param, orig->priv->in_field, NULL);
	}

	list = orig->priv->dependencies;
	while (list) {
		gnome_db_parameter_add_dependency (param, GNOME_DB_PARAMETER (list->data));
		list = g_slist_next (list);
	}

	/* direct settings */
	param->priv->invalid_forced = orig->priv->invalid_forced;
	param->priv->valid = orig->priv->valid;
	param->priv->default_forced = orig->priv->default_forced;	
	if (orig->priv->value)
		param->priv->value = gda_value_copy (orig->priv->value);
	if (orig->priv->default_value)
		param->priv->default_value = gda_value_copy (orig->priv->default_value);
	param->priv->not_null = orig->priv->not_null;
	if (orig->priv->plugin)
		param->priv->plugin = g_strdup (orig->priv->plugin);
	param->priv->user_input_required = orig->priv->user_input_required;

	return obj;
}

/**
 * gnome_db_parameter_new_with_dest_field
 * @field: the #GnomeDbField object the parameter is for
 * @type: the #GnomeDbServerDataType requested
 *
 * Creates a new parameter to be passed to @field for the operations where a parameter is required;
 * other #GnomeDbField can also be added by using the gnome_db_parameter_add_dest_field() method.
 *
 * Returns: a new #GnomeDbParameter object
 */
GObject *
gnome_db_parameter_new_with_dest_field (GnomeDbField *field, GnomeDbServerDataType *type)
{
	GObject   *obj;
	GnomeDbParameter *param;
	GnomeDbDict *dict;

        g_return_val_if_fail (field && IS_GNOME_DB_FIELD (field), NULL);
	g_return_val_if_fail (type && IS_GNOME_DB_SERVER_DATA_TYPE (type), NULL);

	dict = gnome_db_base_get_dict (GNOME_DB_BASE (field));
	g_return_val_if_fail (dict, NULL);

        obj = g_object_new (GNOME_DB_PARAMETER_TYPE, "dict", dict, NULL);
	param = GNOME_DB_PARAMETER (obj);

	gnome_db_parameter_add_for_field (param, field);
	gnome_db_parameter_set_data_type (param, type);

        return obj;
}

static void
gnome_db_parameter_add_for_field (GnomeDbParameter *param, GnomeDbField *for_field)
{
	if (!g_slist_find (param->priv->for_fields, for_field)) {
		param->priv->for_fields = g_slist_append (param->priv->for_fields, for_field);
		gnome_db_base_connect_nullify (for_field, G_CALLBACK (nullified_for_field_cb), param);
		g_object_ref (G_OBJECT (for_field));
	}
}

static void
gnome_db_parameter_del_for_field (GnomeDbParameter *param, GnomeDbField *for_field)
{
	if (g_slist_find (param->priv->for_fields, for_field)) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (for_field),
						      G_CALLBACK (nullified_for_field_cb), param);
		param->priv->for_fields = g_slist_remove (param->priv->for_fields, for_field);
		g_object_unref (G_OBJECT (for_field));
	}
}

static void
gnome_db_parameter_set_data_type (GnomeDbParameter *param, GnomeDbServerDataType *type)
{
	if (param->priv->type) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (param->priv->type),
						      G_CALLBACK (nullified_object_cb), param);
		g_object_unref (G_OBJECT (param->priv->type));
		param->priv->type = NULL;
	}
	
	if (type) {
		param->priv->type = type;
		gnome_db_base_connect_nullify (type, 
					       G_CALLBACK (nullified_object_cb), param);
		g_object_ref (G_OBJECT (param->priv->type));
	}
}

static void
nullified_for_field_cb (GnomeDbBase *obj, GnomeDbParameter *param)
{
	gnome_db_parameter_del_for_field (param, GNOME_DB_FIELD (obj));
}

static void
nullified_object_cb (GnomeDbBase *obj, GnomeDbParameter *param)
{
	gnome_db_base_nullify (GNOME_DB_BASE (param));
}

static void
gnome_db_parameter_dispose (GObject *object)
{
	GnomeDbParameter *parameter;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_PARAMETER (object));

	parameter = GNOME_DB_PARAMETER (object);
	if (parameter->priv) {
		gnome_db_base_nullify_check (GNOME_DB_BASE (object));

		gnome_db_parameter_bind_to_param (parameter, NULL);
		gnome_db_parameter_set_full_bind_param (parameter, NULL);

		if (parameter->priv->in_field)
			gnome_db_parameter_set_source_field (parameter, NULL, NULL);

		while (parameter->priv->for_fields)
			gnome_db_parameter_del_for_field (parameter, GNOME_DB_FIELD (parameter->priv->for_fields->data));

		while (parameter->priv->dependencies)
			gnome_db_parameter_del_dependency (parameter, GNOME_DB_PARAMETER (parameter->priv->dependencies->data));

		gnome_db_parameter_set_data_type (parameter, NULL);

		if (parameter->priv->value) {
			gda_value_free (parameter->priv->value);
			parameter->priv->value = NULL;
		}

		if (parameter->priv->default_value) {
			gda_value_free (parameter->priv->default_value);
			parameter->priv->default_value = NULL;
		}
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_parameter_finalize (GObject   * object)
{
	GnomeDbParameter *parameter;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_PARAMETER (object));

	parameter = GNOME_DB_PARAMETER (object);
	if (parameter->priv) {
		if (parameter->priv->plugin)
			g_free (parameter->priv->plugin);

		g_free (parameter->priv);
		parameter->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_parameter_set_property    (GObject              *object,
				    guint                 param_id,
				    const GValue         *value,
				    GParamSpec           *pspec)
{
	const gchar *ptr;
	GnomeDbParameter *parameter;

	parameter = GNOME_DB_PARAMETER (object);
	if (parameter->priv) {
		switch (param_id) {
		case PROP_PLUGIN:
			ptr = g_value_get_string (value);
			if (parameter->priv->plugin) {
				g_free (parameter->priv->plugin);
				parameter->priv->plugin = NULL;
			}
			if (ptr)
				parameter->priv->plugin = g_strdup (ptr);
			break;
		case PROP_USE_DEFAULT_VALUE:
			parameter->priv->default_forced = g_value_get_boolean (value);
			parameter->priv->invalid_forced = FALSE;
			break;
		case PROP_SIMPLE_BIND:
			gnome_db_parameter_bind_to_param (parameter, g_value_get_pointer (value));
			break;
		case PROP_FULL_BIND:
			gnome_db_parameter_set_full_bind_param (parameter, g_value_get_pointer (value));
			break;
		}
	}
}

static void
gnome_db_parameter_get_property    (GObject              *object,
				    guint                 param_id,
				    GValue               *value,
				    GParamSpec           *pspec)
{
	GnomeDbParameter *parameter;

	parameter = GNOME_DB_PARAMETER (object);
	if (parameter->priv) {
		switch (param_id) {
		case PROP_PLUGIN:
			g_value_set_string (value, parameter->priv->plugin);
			break;
		case PROP_USE_DEFAULT_VALUE:
			g_value_set_boolean (value, parameter->priv->default_forced);
			break;
		case PROP_SIMPLE_BIND:
			g_value_set_pointer (value, parameter->priv->change_with);
			break;
		case PROP_FULL_BIND:
			g_value_set_pointer (value, parameter->priv->alias_of);
			break;
		}
	}
}


/**
 * gnome_db_parameter_add_dest_field
 * @param: a #GnomeDbParameter object
 * @field: the #GnomeDbField object the parameter is for
 *
 * Adds a #GnomeDbField object for which the parameter is for
 */
void
gnome_db_parameter_add_dest_field (GnomeDbParameter *param, GnomeDbField *field)
{
	g_return_if_fail (param && IS_GNOME_DB_PARAMETER (param));
	g_return_if_fail (param->priv);
	g_return_if_fail (field && IS_GNOME_DB_FIELD (field));

	gnome_db_parameter_add_for_field (param, field);	
}

/**
 * gnome_db_parameter_get_dest_fields
 * @param: a #GnomeDbParameter object
 * 
 * Get the #GnomeDbField objects which created @param (and which will use its value)
 *
 * Returns: the list of #GnomeDbField object; it must not be changed or free'd
 */
GSList *
gnome_db_parameter_get_dest_fields (GnomeDbParameter *param)
{
	g_return_val_if_fail (param && IS_GNOME_DB_PARAMETER (param), NULL);
	g_return_val_if_fail (param->priv, NULL);

	return param->priv->for_fields;
}

/**
 * gnome_db_parameter_get_data_type
 * @param: a #GnomeDbParameter object
 * 
 * Get the requested data type for @param.
 *
 * Returns: the data type
 */
GnomeDbServerDataType *
gnome_db_parameter_get_data_type (GnomeDbParameter *param)
{
	g_return_val_if_fail (param && IS_GNOME_DB_PARAMETER (param), NULL);
	g_return_val_if_fail (param->priv, NULL);

	return param->priv->type;
}


/**
 * gnome_db_parameter_get_value
 * @param: a #GnomeDbParameter object
 *
 * Get the value held into the parameter
 *
 * Returns: the value (a NULL value returns a GDA_VALUE_TYPE_NULL GdaValue)
 */
const GdaValue *
gnome_db_parameter_get_value (GnomeDbParameter *param)
{
	g_return_val_if_fail (param && IS_GNOME_DB_PARAMETER (param), NULL);
	g_return_val_if_fail (param->priv, NULL);

	if (!param->priv->alias_of) {
		if (!param->priv->value)
			param->priv->value = gda_value_new_null ();
		return param->priv->value;
	}
	else 
		return gnome_db_parameter_get_value (param->priv->alias_of);
}


/*
 * gnome_db_parameter_set_value
 * @param: a #GnomeDbParameter object
 * @value: a value to set the parameter to
 *
 * Sets the value within the parameter. If @param is an alias for another
 * parameter, then the value is also set for that other parameter.
 *
 * The action of any call to gnome_db_parameter_declare_invalid() is cancelled
 * as soon as this method is called, even if @param's value does not change.
 * 
 * If the value is not different from the one already contained within @param,
 * then @param is not chaged and no signal is emitted.
 */
void
gnome_db_parameter_set_value (GnomeDbParameter *param, const GdaValue *value)
{
	gboolean changed = TRUE;
	const GdaValue *current_val;

	g_return_if_fail (param && IS_GNOME_DB_PARAMETER (param));
	g_return_if_fail (param->priv);

	param->priv->invalid_forced = FALSE;

	/* param will be changed? */
	current_val = gnome_db_parameter_get_value (param);
	if (current_val == value)
		changed = FALSE;

	if (changed && gda_value_is_null (current_val) && ((value && gda_value_is_null (value)) || !value)) 
		changed = FALSE;
	
	if (changed && value && (gda_value_get_type (value) == gda_value_get_type (current_val)))
		changed = gda_value_compare (value, current_val);
		
	/* param's validity */
	param->priv->valid = TRUE;
	if (!value || gda_value_is_null (value))
		if (param->priv->not_null)
			param->priv->valid = FALSE;

	if (value &&
	    (gda_value_get_type (value) != GDA_VALUE_TYPE_NULL) &&
	    (gda_value_get_type (value) != gnome_db_server_data_type_get_gda_type (param->priv->type)))
		param->priv->valid = FALSE;

#ifdef debug_NO
	g_print ("Changed param %p (%s): value %s --> %s \t(type %d -> %d) VALID: %d CHANGED: %d\n", param, gnome_db_base_get_name (param),
		 current_val ? gda_value_stringify (current_val) : "_NULL_",
		 value ? gda_value_stringify (value) : "_NULL_",
		 current_val ? gda_value_get_type (current_val) : 0,
		 value ? gda_value_get_type (value) : 0, gnome_db_parameter_is_valid (param), changed);
#endif

	/* end of procedure if the value has not been changed, after calculating the param's validity */
	if (!changed) {
		if (GPOINTER_TO_INT (g_object_get_data (G_OBJECT (param), "changed_pending"))) {
			gboolean changed_blocked;
			g_object_get (G_OBJECT (param), "changed_blocked", &changed_blocked, NULL);
			if (!changed_blocked) {
				g_object_set_data (G_OBJECT (param), "changed_pending", NULL);
				gnome_db_base_changed (GNOME_DB_BASE (param));
			}
		}
		return;
	}

	/* the parameter does not have the default value forced since it has changed */
	param->priv->default_forced = FALSE;

	/* real setting of the value */
	if (param->priv->alias_of) {
#ifdef debug_NO
		g_print ("Param %p is alias of param %p => propagating changes to param %p\n",
			 param, param->priv->alias_of, param->priv->alias_of);
#endif
		gnome_db_parameter_set_value (param->priv->alias_of, value);
	}
	else {
		gboolean changed_blocked;
		if (param->priv->value) {
			gda_value_free (param->priv->value);
			param->priv->value = NULL;
		}

		if (value)
			param->priv->value = gda_value_copy (value);

		/* if the GnomeDbBase has "changed_blocked" property TRUE, then store that the next time we need
		 * to emit the "changed" signal even if the stored value has not changed "
		 */
		g_object_get (G_OBJECT (param), "changed_blocked", &changed_blocked, NULL);
		if (changed_blocked)
			g_object_set_data (G_OBJECT (param), "changed_pending", GINT_TO_POINTER (TRUE));
		else
			gnome_db_base_changed (GNOME_DB_BASE (param));
	}
}


/**
 * gnome_db_parameter_declare_invalid
 * @param: a #GnomeDbParameter object
 *
 * Forces a parameter to be invalid; to set it valid again, a new value must be assigned
 * to it using gnome_db_parameter_set_value().
 */
void
gnome_db_parameter_declare_invalid (GnomeDbParameter *param)
{
	g_return_if_fail (param && IS_GNOME_DB_PARAMETER (param));
	g_return_if_fail (param->priv);

#ifdef debug_NO
	g_print ("Param %p (%s): declare invalid\n", param, gnome_db_base_get_name (param));
#endif

	if (param->priv->invalid_forced)
		return;

	param->priv->invalid_forced = TRUE;
	param->priv->valid = FALSE;
	
	if (param->priv->value) {
		gda_value_free (param->priv->value);
		param->priv->value = NULL;
	}

	/* if we are an alias, then we forward the value setting to the master */
	if (param->priv->alias_of) 
		gnome_db_parameter_declare_invalid (param->priv->alias_of);
	else 
		gnome_db_base_changed (GNOME_DB_BASE (param));
}


/**
 * gnome_db_parameter_is_valid
 * @param: a #GnomeDbParameter object
 *
 * Get the validity of @param (that is, of the value held by @param)
 *
 * Returns: TRUE if @param's value can safely be used
 */
gboolean
gnome_db_parameter_is_valid (GnomeDbParameter *param)
{
	g_return_val_if_fail (param && IS_GNOME_DB_PARAMETER (param), FALSE);
	g_return_val_if_fail (param->priv, FALSE);

	if (param->priv->alias_of) {
		return gnome_db_parameter_is_valid (param->priv->alias_of);
	}
	else {
		if (param->priv->invalid_forced) {
			return FALSE;
		}

		if (param->priv->default_forced) {
			return param->priv->default_value ? TRUE : FALSE;
		}
		else {
			return param->priv->valid;
		}
	}
}


/**
 * gnome_db_parameter_get_default_value
 * @param: a #GnomeDbParameter object
 *
 * Get the default value held into the parameter. WARNING: the default value does not need to be of 
 * the same type as the one required by @param.
 *
 * Returns: the default value
 */
const GdaValue *
gnome_db_parameter_get_default_value (GnomeDbParameter *param)
{
	g_return_val_if_fail (param && IS_GNOME_DB_PARAMETER (param), NULL);
	g_return_val_if_fail (param->priv, NULL);

	return param->priv->default_value;	
}


/*
 * gnome_db_parameter_set_default_value
 * @param: a #GnomeDbParameter object
 * @value: a value to set the parameter's default value to
 *
 * Sets the default value within the parameter. WARNING: the default value does not need to be of 
 * the same type as the one required by @param.
 */
void
gnome_db_parameter_set_default_value (GnomeDbParameter *param, const GdaValue *value)
{
	g_return_if_fail (param && IS_GNOME_DB_PARAMETER (param));
	g_return_if_fail (param->priv);

	if (param->priv->default_value) {
		gda_value_free (param->priv->default_value);
		param->priv->default_value = NULL;
	}

	if (value) 
		param->priv->default_value = gda_value_copy (value);

	gnome_db_base_changed (GNOME_DB_BASE (param));
}

/**
 * gnome_db_parameter_set_not_null
 * @param: a #GnomeDbParameter object
 * @not_null:
 *
 * Sets if the parameter can have a NULL value. If @not_null is TRUE, then that won't be allowed
 */
void
gnome_db_parameter_set_not_null (GnomeDbParameter *param, gboolean not_null)
{
	g_return_if_fail (param && IS_GNOME_DB_PARAMETER (param));
	g_return_if_fail (param->priv);

	if (not_null != param->priv->not_null) {
		param->priv->not_null = not_null;

		/* updating the parameter's validity regarding the NULL value */
		if (!not_null && 
		    (!param->priv->value || gda_value_is_null (param->priv->value)))
			param->priv->valid = TRUE;

		if (not_null && 
		    (!param->priv->value || gda_value_is_null (param->priv->value)))
			param->priv->valid = FALSE;

		gnome_db_base_changed (GNOME_DB_BASE (param));
	}
}

/**
 * gnome_db_parameter_get_not_null
 * @param: a #GnomeDbParameter object
 *
 * Get wether the parameter can be NULL or not
 *
 * Returns: TRUE if the parameter cannot be NULL
 */
gboolean
gnome_db_parameter_get_not_null (GnomeDbParameter *param)
{
	g_return_val_if_fail (param && IS_GNOME_DB_PARAMETER (param), FALSE);
	g_return_val_if_fail (param->priv, FALSE);

	return param->priv->not_null;
}

/**
 * gnome_db_parameter_set_source_field
 * @param: a #GnomeDbParameter object
 * @field: a #GnomeDbField object or NULL
 * @error: location to store error, or %NULL
 *
 * Sets a limit on the possible values for the @param parameter: the value must be among the values of @field within
 * the #GnomeDbEntity to which @field belongs (the #GnomeDbEntity can either be a #GnomeDbDataModel which already has
 * some data, or a #GnomeDbQuery which will be converted to a #GnomeDbResultset (which itself is a #GnomeDbDataModel)
 * to have some actual data).
 *
 * @field must be of the same type as the requested type for @param.
 *
 * @field is not copied, just referenced.
 *
 * Returns: TRUE if no error occurred
 */
gboolean
gnome_db_parameter_set_source_field (GnomeDbParameter *param, GnomeDbField *field, GError **error)
{
	g_return_val_if_fail (param && IS_GNOME_DB_PARAMETER (param), FALSE);
	g_return_val_if_fail (param->priv, FALSE);
	
	if (param->priv->in_field) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (param->priv->in_field),
						      G_CALLBACK (nullified_source_field_cb), param);
		g_object_unref (G_OBJECT (param->priv->in_field));
		param->priv->in_field = NULL;
	}
	
	if (field) {
		GnomeDbQuery *query;
		GnomeDbEntity *ent;

		g_return_val_if_fail (IS_GNOME_DB_FIELD (field), FALSE);
		ent = gnome_db_field_get_entity (GNOME_DB_FIELD (field));
		if (! IS_GNOME_DB_QUERY (ent) && ! IS_GNOME_DB_DATA_MODEL (ent)) {
			g_set_error (error,
				     GNOME_DB_PARAMETER_ERROR,
				     GNOME_DB_PARAMETER_QUERY_LIMIT_ERROR,
				     _("A parameter can only get its value within a GnomeDbQuery or a GnomeDbDataModel"));
			return FALSE;
		}

		if (IS_GNOME_DB_QUERY (ent)) {
			query = GNOME_DB_QUERY (ent);
			
			/* test to see if @query is well formed for the job */
			if (!gnome_db_query_is_select_query (query)) {
				g_set_error (error,
					     GNOME_DB_PARAMETER_ERROR,
					     GNOME_DB_PARAMETER_QUERY_LIMIT_ERROR,
					     _("Parameter: query to limit range is not a selection query"));
				return FALSE;
			}
		}
		
		param->priv->in_field = field;
		gnome_db_base_connect_nullify (field, G_CALLBACK (nullified_source_field_cb), param);
		g_object_ref (G_OBJECT (param->priv->in_field));
	}
	return TRUE;
}

static void
nullified_source_field_cb (GnomeDbBase *obj, GnomeDbParameter *param)
{
	gnome_db_parameter_set_source_field (param, NULL, NULL);
}

/**
 * gnome_db_parameter_get_limit_query
 * @param: a #GnomeDbParameter object
 *
 * Get the query field which limits the possible value for the parameter.
 *
 * Returns: the #GnomeDbField, or %NULL
 */
GnomeDbField *
gnome_db_parameter_get_source_field (GnomeDbParameter *param)
{
 	g_return_val_if_fail (param && IS_GNOME_DB_PARAMETER (param), NULL);
	g_return_val_if_fail (param->priv, NULL);
	
	return param->priv->in_field;
}

/**
 * gnome_db_parameter_add_dependency
 * @param: a #GnomeDbParameter object
 * @depend_on: a #GnomeDbParameter object
 *
 * Tell @param that its value will depend on the value of @depend_on
 */
void
gnome_db_parameter_add_dependency (GnomeDbParameter *param, GnomeDbParameter *depend_on)
{
 	g_return_if_fail (param && IS_GNOME_DB_PARAMETER (param));
	g_return_if_fail (param->priv);
 	g_return_if_fail (depend_on && IS_GNOME_DB_PARAMETER (depend_on));
	g_return_if_fail (depend_on->priv);
	g_return_if_fail (!g_slist_find (depend_on->priv->dependencies, param));

	if (!g_slist_find (param->priv->dependencies, depend_on) && (depend_on != param)) {
		param->priv->dependencies = g_slist_append (param->priv->dependencies, depend_on);
		gnome_db_base_connect_nullify (depend_on, 
					       G_CALLBACK (nullified_depend_on_cb), param);
		g_object_ref (G_OBJECT (depend_on));
	}	
}

/**
 * gnome_db_parameter_del_dependency
 * @param: a #GnomeDbParameter object
 * @depend_on: a #GnomeDbParameter object
 *
 * Remove the @depend_on dependency from @param
 */
void
gnome_db_parameter_del_dependency (GnomeDbParameter *param, GnomeDbParameter *depend_on)
{
 	g_return_if_fail (param && IS_GNOME_DB_PARAMETER (param));
	g_return_if_fail (param->priv);
 	g_return_if_fail (depend_on && IS_GNOME_DB_PARAMETER (depend_on));
	g_return_if_fail (depend_on->priv);

	if (g_slist_find (param->priv->dependencies, depend_on)) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (depend_on),
						      G_CALLBACK (nullified_depend_on_cb), param);
		g_object_unref (G_OBJECT (depend_on));
		param->priv->dependencies = g_slist_remove (param->priv->dependencies, depend_on);
	}
}

static void
nullified_depend_on_cb (GnomeDbBase *obj, GnomeDbParameter *param)
{
	gnome_db_parameter_del_dependency (param, GNOME_DB_PARAMETER (obj));
}

/**
 * gnome_db_parameter_get_dependencies
 * @param: a #GnomeDbParameter object
 *
 * Get the list of #GnomeDbParameter @param depends on
 *
 * Returns: the list of parameters (the list should not be modified)
 */
GSList *
gnome_db_parameter_get_dependencies (GnomeDbParameter *param)
{
 	g_return_val_if_fail (param && IS_GNOME_DB_PARAMETER (param), NULL);
	g_return_val_if_fail (param->priv, NULL);
	
	return param->priv->dependencies;
}

/**
 * gnome_db_parameter_replace_ref
 * @param: a #GnomeDbParameter object
 * @replacements: the (objects to be replaced, replacing object) pairs
 *
 * Replace references to some objects by references to some other objects,
 * as listed in @replacements.
 */
void
gnome_db_parameter_replace_ref (GnomeDbParameter *param, GHashTable *replacements)
{
	GSList *list;
	gpointer ref;

	g_return_if_fail (param && IS_GNOME_DB_PARAMETER (param));
	g_return_if_fail (param->priv);

	/* Destination fields */
	list = param->priv->for_fields;
	while (list) {
		ref = g_hash_table_lookup (replacements, list->data);
		if (ref) /* we don't actually replace the ref, but simply add one destination field */
			gnome_db_parameter_add_dest_field (param, ref);

		list = g_slist_next (list);
	}

	/* dependencies */ 
	list = param->priv->dependencies;
	while (list) {
		ref = g_hash_table_lookup (replacements, list->data);
		if (ref) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (list->data),
							      G_CALLBACK (nullified_depend_on_cb), param);
			g_object_unref (G_OBJECT (list->data));
			list->data = ref;
			gnome_db_base_connect_nullify (ref, G_CALLBACK (nullified_depend_on_cb), param);
			g_object_ref (G_OBJECT (ref));
		}
		list = g_slist_next (list);
	}
	
	/* in_field */
	if (param->priv->in_field) {
		ref = g_hash_table_lookup (replacements, param->priv->in_field);
		if (ref) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (param->priv->in_field),
							      G_CALLBACK (nullified_source_field_cb), param);
			g_object_unref (G_OBJECT (param->priv->in_field));
			param->priv->in_field = GNOME_DB_FIELD (ref);
			gnome_db_base_connect_nullify (param->priv->in_field, 
						       G_CALLBACK (nullified_source_field_cb), param);
			g_object_ref (G_OBJECT (param->priv->in_field));
		}
	}
}

/**
 * gnome_db_parameter_bind_to_param
 * @param: a #GnomeDbParameter
 * @bind_to: a #GnomeDbParameter or %NULL
 *
 * Sets @param to change when @bind_to changes (and does not make @bind_to change when @param changes)
 */
void
gnome_db_parameter_bind_to_param (GnomeDbParameter *param, GnomeDbParameter *bind_to)
{
	const GdaValue *cvalue;
	GdaValue *value1 = NULL, *value2 = NULL;

	g_return_if_fail (param && IS_GNOME_DB_PARAMETER (param));
	g_return_if_fail (param->priv);
	g_return_if_fail (param != bind_to);

	if (param->priv->change_with == bind_to)
		return;

	/* get a copy of the current values of @param and @bind_to */
	if (bind_to) {
		g_return_if_fail (bind_to && IS_GNOME_DB_PARAMETER (bind_to));
		g_return_if_fail (bind_to->priv);
		g_return_if_fail (param->priv->type == bind_to->priv->type);
		cvalue = gnome_db_parameter_get_value (bind_to);
		if (cvalue && !gda_value_is_null (cvalue))
			value2 = gda_value_copy (cvalue);
	}

	cvalue = gnome_db_parameter_get_value (param);
	if (cvalue && !gda_value_is_null (cvalue))
		value1 = gda_value_copy (cvalue);

	/* get rid of the old alias */
	if (param->priv->change_with) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (param->priv->change_with),
						      G_CALLBACK (nullified_alias_of_cb), param);
		g_signal_handlers_disconnect_by_func (G_OBJECT (param->priv->change_with),
						      G_CALLBACK (alias_of_changed_cb), param);
		param->priv->change_with = NULL;
	}

	/* setting the new alias or reseting the value if there is no new alias */
	if (bind_to) {
		gboolean equal = FALSE;

		param->priv->change_with = bind_to;
		gnome_db_base_connect_nullify (param->priv->change_with, 
					       G_CALLBACK (nullified_alias_of_cb), param);
		g_signal_connect (G_OBJECT (param->priv->change_with), "changed",
				  G_CALLBACK (alias_of_changed_cb), param);

		/* if alias_of has a different value than param, then we set param to the new value */
		if (value1 && value2 &&
		    (gda_value_get_type (value1) == gda_value_get_type (value2)))
			equal = !gda_value_compare (value1, value2);
		else {
			if (!value1 && !value2)
				equal = TRUE;
		}
		if (!equal)
			gnome_db_parameter_set_value (param, value2);
	}
		
	if (value1) gda_value_free (value1);
	if (value2) gda_value_free (value2);
}

/*
 * gnome_db_parameter_set_full_bind_param
 * @param: a #GnomeDbParameter
 * @alias_of: a #GnomeDbParameter or %NULL
 *
 * Sets @param to change when @alias_of changes and makes @alias_of change when @param changes.
 * The difference with gnome_db_parameter_bind_to_param is that when @param changes, then @alias_of also
 * changes.
 */
static void
gnome_db_parameter_set_full_bind_param (GnomeDbParameter *param, GnomeDbParameter *alias_of)
{
	const GdaValue *cvalue;
	GdaValue *value1 = NULL, *value2 = NULL;

	g_return_if_fail (param && IS_GNOME_DB_PARAMETER (param));
	g_return_if_fail (param->priv);

	if (param->priv->alias_of == alias_of)
		return;

	/* get a copy of the current values of @param and @alias_of */
	if (alias_of) {
		g_return_if_fail (alias_of && IS_GNOME_DB_PARAMETER (alias_of));
		g_return_if_fail (alias_of->priv);
		g_return_if_fail (param->priv->type == alias_of->priv->type);
		cvalue = gnome_db_parameter_get_value (alias_of);
		if (cvalue && !gda_value_is_null (cvalue))
			value2 = gda_value_copy (cvalue);
	}

	cvalue = gnome_db_parameter_get_value (param);
	if (cvalue && !gda_value_is_null (cvalue))
		value1 = gda_value_copy (cvalue);
		
	
	/* get rid of the old alias */
	if (param->priv->alias_of) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (param->priv->alias_of),
						      G_CALLBACK (nullified_alias_of_cb), param);
		g_signal_handlers_disconnect_by_func (G_OBJECT (param->priv->alias_of),
						      G_CALLBACK (alias_of_changed_cb), param);
		param->priv->alias_of = NULL;
	}

	/* setting the new alias or reseting the value if there is no new alias */
	if (alias_of) {
		gboolean equal = FALSE;

		/* get rid of the internal param's value */
		if (param->priv->value) {
			gda_value_free (param->priv->value);
			param->priv->value = NULL;
		}

		param->priv->alias_of = alias_of;
		gnome_db_base_connect_nullify (param->priv->alias_of, 
					       G_CALLBACK (nullified_alias_of_cb), param);
		g_signal_connect (G_OBJECT (param->priv->alias_of), "changed",
				  G_CALLBACK (alias_of_changed_cb), param);

		/* if alias_of has a different value than param, then we emit a CHANGED signal */
		if (value1 && value2 &&
		    (gda_value_get_type (value1) == gda_value_get_type (value2)))
			equal = !gda_value_compare (value1, value2);
		else {
			if (!value1 && !value2)
				equal = TRUE;
		}

		if (!equal)
			gnome_db_base_changed (GNOME_DB_BASE (param));
	}
	else {
		/* restore the value that was in the previous alias parameter, 
		 * if there was such a value, and don't emit a signal */
		g_assert (! param->priv->value);
		if (value1)
			param->priv->value = value1;
		value1 = NULL;
	}

	if (value1) gda_value_free (value1);
	if (value2) gda_value_free (value2);
}

static void
nullified_alias_of_cb (GnomeDbBase *obj, GnomeDbParameter *param)
{
	if ((gpointer) obj == (gpointer) param->priv->alias_of)
		gnome_db_parameter_set_full_bind_param (param, NULL);
	else
		gnome_db_parameter_bind_to_param (param, NULL);
}

static void
alias_of_changed_cb (GnomeDbParameter *alias_of, GnomeDbParameter *param)
{
	if ((gpointer) alias_of == (gpointer) param->priv->alias_of) {
		/* callback used as full bind */
		gnome_db_base_changed (GNOME_DB_BASE (param));
	}
	else {
		/* callback used as simple bind */
		gnome_db_parameter_set_value (param, gnome_db_parameter_get_value (alias_of));
	}
}

static void
gnome_db_parameter_signal_changed (GnomeDbBase *base, gboolean block_changed_signal)
{
	GnomeDbParameter *param = GNOME_DB_PARAMETER (base);

	if (param->priv->alias_of) {
		if (block_changed_signal)
			gnome_db_base_block_changed (GNOME_DB_BASE (param->priv->alias_of));
		else
			gnome_db_base_unblock_changed (GNOME_DB_BASE (param->priv->alias_of));
	}
}

/**
 * gnome_db_parameter_get_bind_param
 * @param: a #GnomeDbParameter
 *
 * Get the parameter which makes @param change its value when the param's value is changed.
 *
 * Returns: the #GnomeDbParameter or %NULL
 */
GnomeDbParameter *
gnome_db_parameter_get_bind_param (GnomeDbParameter *param)
{
	g_return_val_if_fail (param && IS_GNOME_DB_PARAMETER (param), NULL);
	g_return_val_if_fail (param->priv, NULL);

	return param->priv->change_with;
}


/**
 * gnome_db_parameter_requires_user_input
 * @param: a #GnomeDbParameter
 *
 * Tells if the parameter is configured in a way that even if there is a value, it requires
 * that the user at least validates that value, or change it.
 *
 * Returns: TRUE if user input is required
 */
gboolean
gnome_db_parameter_requires_user_input (GnomeDbParameter *param)
{
	g_return_val_if_fail (param && IS_GNOME_DB_PARAMETER (param), FALSE);
	g_return_val_if_fail (param->priv, FALSE);

	return param->priv->user_input_required;
}

/**
 * gnome_db_parameter_set_user_input_required
 * @param: a #GnomeDbParameter
 * @input_required:
 * 
 * Sets if the user input is required for @param (even though it may already have
 * a value) and be valid.
 */
void
gnome_db_parameter_set_user_input_required (GnomeDbParameter *param, gboolean input_required)
{
	g_return_if_fail (param && IS_GNOME_DB_PARAMETER (param));
	g_return_if_fail (param->priv);

	param->priv->user_input_required = input_required;
}


#ifdef debug
static void
gnome_db_parameter_dump (GnomeDbParameter *parameter, guint offset)
{
	gchar *str;
	guint i;

	g_return_if_fail (parameter);
	g_return_if_fail (IS_GNOME_DB_PARAMETER (parameter));

	/* string for the offset */
	str = g_new0 (gchar, offset+1);
        for (i=0; i<offset; i++)
                str[i] = ' ';
        str[offset] = 0;
	
	/* dump */
	if (parameter->priv) {
		GSList *list;
		g_print ("%s" D_COL_H1 "GnomeDbParameter %p (%s), type=%s\n" D_COL_NOR, str, parameter,
			 gnome_db_base_get_name (GNOME_DB_BASE (parameter)), 
			 gnome_db_base_get_name (GNOME_DB_BASE (parameter->priv->type)));
		
		list = parameter->priv->for_fields;
		while (list) {
			gchar *xmlid = NULL;

			if (IS_GNOME_DB_XML_STORAGE (list->data))
				xmlid = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (list->data));

			if (list == parameter->priv->for_fields)
				g_print ("%sFor fields: ", str);
			else
				g_print (", ");
			if (xmlid) {
				g_print ("%p (%s)", list->data, xmlid);
				g_free (xmlid);
			}
			else
				g_print ("%p", list->data);

			list = g_slist_next (list);
		}
		if (parameter->priv->for_fields)
			g_print ("\n");

		if (parameter->priv->in_field) {
			if (IS_GNOME_DB_XML_STORAGE (parameter->priv->in_field)) {
				gchar *xmlid = gnome_db_xml_storage_get_xml_id (
					GNOME_DB_XML_STORAGE (parameter->priv->in_field));
				g_print ("%sDepends on field %p (%s)\n", str, parameter->priv->in_field, xmlid);
				g_free (xmlid);
			}
			else
				g_print ("%sDepends on field %p\n", str, parameter->priv->in_field);	
		}

		if (parameter->priv->alias_of) {
			g_print ("%sAlias of parameter %p (%s)\n", str, parameter->priv->alias_of,
				 gnome_db_base_get_name (GNOME_DB_BASE (parameter->priv->alias_of)));
			gnome_db_parameter_dump (parameter->priv->alias_of, offset + 5);
		}

		list = parameter->priv->dependencies;
		while (list) {
			if (list == parameter->priv->dependencies)
				g_print ("%sParam dependencies:", str);
			else
				g_print (", ");
			g_print ("%p", list->data);
			list = g_slist_next (list);
		}
		if (parameter->priv->dependencies)
			g_print ("\n");

		list = parameter->priv->dependencies;
		while (list) {
			gnome_db_parameter_dump (GNOME_DB_PARAMETER (list->data), offset + 5);
			list = g_slist_next (list);
		}
	}
	else
		g_print ("%s" D_COL_ERR "Using finalized object %p" D_COL_NOR, str, parameter);

	g_free (str);
}
#endif


/* 
 * GnomeDbReferer interface implementation
 */

static GSList *
gnome_db_parameter_get_ref_objects (GnomeDbReferer *iface)
{
	GSList *list = NULL;
	GnomeDbBase *base;
	g_return_val_if_fail (iface && IS_GNOME_DB_PARAMETER (iface), NULL);
	g_return_val_if_fail (GNOME_DB_PARAMETER (iface)->priv, NULL);

	TO_IMPLEMENT;

	return list;
}

static void
gnome_db_parameter_replace_refs (GnomeDbReferer *iface, GHashTable *replacements)
{
	GnomeDbParameter *parameter;
	gpointer repl;
	GSList *list;

	g_return_if_fail (iface && IS_GNOME_DB_PARAMETER (iface));
	g_return_if_fail (GNOME_DB_PARAMETER (iface)->priv);

	parameter = GNOME_DB_PARAMETER (iface);
	
	repl = g_hash_table_lookup (replacements, parameter->priv->type);
	if (repl)
		gnome_db_parameter_set_data_type (parameter, GNOME_DB_SERVER_DATA_TYPE (repl));

	list = parameter->priv->for_fields;
	while (list) {
		/* FIXME: the parameter->priv->for_fields is changed during the loop! */
		repl = g_hash_table_lookup (replacements, list->data);
		if (repl) {
			gnome_db_parameter_del_for_field (parameter, GNOME_DB_FIELD (list->data));
			gnome_db_parameter_add_for_field (parameter, GNOME_DB_FIELD (repl));
		}
		list = g_slist_next (list);
	}
	
	if (parameter->priv->alias_of) {
		repl = g_hash_table_lookup (replacements, parameter->priv->alias_of);
		if (repl) 
			gnome_db_parameter_set_full_bind_param (parameter, repl);
	}

	if (parameter->priv->change_with) {
		repl = g_hash_table_lookup (replacements, parameter->priv->change_with);
		if (repl)
			gnome_db_parameter_bind_to_param (parameter, GNOME_DB_PARAMETER (repl));
	}
	
	if (parameter->priv->in_field) {
		repl = g_hash_table_lookup (replacements, parameter->priv->in_field);
		if (repl)
			gnome_db_parameter_set_source_field (parameter, GNOME_DB_FIELD (repl), NULL);
	}

	list = parameter->priv->dependencies;
	while (list) {
		repl = g_hash_table_lookup (replacements, list->data);
		if (repl) {
			gnome_db_parameter_del_dependency (parameter, GNOME_DB_PARAMETER (list->data));
			gnome_db_parameter_add_dependency (parameter, GNOME_DB_PARAMETER (repl));
		}
		list = g_slist_next (list);
	}
}
