/* gnome-db-qf-func.c
 *
 * Copyright (C) 2003 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-db-qf-func.h"
#include "gnome-db-xml-storage.h"
#include "gnome-db-field.h"
#include "gnome-db-entity.h"
#include "gnome-db-renderer.h"
#include "gnome-db-referer.h"
#include "gnome-db-ref-base.h"
#include "marshal.h"
#include "gnome-db-query.h"
#include "gnome-db-server.h"
#include "gnome-db-server-data-type.h"
#include <string.h>

/* 
 * Main static functions 
 */
static void gnome_db_qf_func_class_init (GnomeDbQfFuncClass * class);
static void gnome_db_qf_func_init (GnomeDbQfFunc *qf);
static void gnome_db_qf_func_dispose (GObject *object);
static void gnome_db_qf_func_finalize (GObject *object);

static void gnome_db_qf_func_set_property (GObject              *object,
				      guint                 param_id,
				      const GValue         *value,
				      GParamSpec           *pspec);
static void gnome_db_qf_func_get_property (GObject              *object,
				      guint                 param_id,
				      GValue               *value,
				      GParamSpec           *pspec);

/* XML storage interface */
static void        gnome_db_qf_func_xml_storage_init (GnomeDbXmlStorageIface *iface);
static gchar      *gnome_db_qf_func_get_xml_id (GnomeDbXmlStorage *iface);
static xmlNodePtr  gnome_db_qf_func_save_to_xml (GnomeDbXmlStorage *iface, GError **error);
static gboolean    gnome_db_qf_func_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error);

/* Field interface */
static void              gnome_db_qf_func_field_init      (GnomeDbFieldIface *iface);
static GnomeDbEntity         *gnome_db_qf_func_get_entity      (GnomeDbField *iface);
static GnomeDbServerDataType *gnome_db_qf_func_get_data_type   (GnomeDbField *iface);

/* Renderer interface */
static void            gnome_db_qf_func_renderer_init   (GnomeDbRendererIface *iface);
static GdaXqlItem     *gnome_db_qf_func_render_as_xql   (GnomeDbRenderer *iface, GnomeDbDataSet *context, GError **error);
static gchar          *gnome_db_qf_func_render_as_sql   (GnomeDbRenderer *iface, GnomeDbDataSet *context, guint options, GError **error);
static gchar          *gnome_db_qf_func_render_as_str   (GnomeDbRenderer *iface, GnomeDbDataSet *context);

/* Referer interface */
static void        gnome_db_qf_func_referer_init        (GnomeDbRefererIface *iface);
static gboolean    gnome_db_qf_func_activate            (GnomeDbReferer *iface);
static void        gnome_db_qf_func_deactivate          (GnomeDbReferer *iface);
static gboolean    gnome_db_qf_func_is_active           (GnomeDbReferer *iface);
static GSList     *gnome_db_qf_func_get_ref_objects     (GnomeDbReferer *iface);
static void        gnome_db_qf_func_replace_refs        (GnomeDbReferer *iface, GHashTable *replacements);

/* virtual functions */
static GObject    *gnome_db_qf_func_copy           (GnomeDbQfield *orig);
static gboolean    gnome_db_qf_func_is_equal       (GnomeDbQfield *qfield1, GnomeDbQfield *qfield2);


#ifdef debug
static void        gnome_db_qf_func_dump           (GnomeDbQfFunc *func, guint offset);
#endif

/* When the GnomeDbQuery or GnomeDbTarget is nullified */
static void nullified_object_cb (GObject *obj, GnomeDbQfFunc *func);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	TEMPL_SIGNAL,
	LAST_SIGNAL
};

static gint gnome_db_qf_func_signals[LAST_SIGNAL] = { 0 };

/* properties */
enum
{
	PROP_0,
	PROP_QUERY,
};


/* private structure */
struct _GnomeDbQfFuncPrivate
{
	GnomeDbQuery    *query;
	GnomeDbRefBase  *func_ref;  /* references a GnomeDbServerFunction */
	GSList     *args;      /* list of GnomeDbRefBase objects */
};


/* module error */
GQuark gnome_db_qf_func_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_qf_func_error");
	return quark;
}


GType
gnome_db_qf_func_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbQfFuncClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_qf_func_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbQfFunc),
			0,
			(GInstanceInitFunc) gnome_db_qf_func_init
		};

		static const GInterfaceInfo xml_storage_info = {
			(GInterfaceInitFunc) gnome_db_qf_func_xml_storage_init,
			NULL,
			NULL
		};

		static const GInterfaceInfo field_info = {
			(GInterfaceInitFunc) gnome_db_qf_func_field_init,
			NULL,
			NULL
		};

		static const GInterfaceInfo renderer_info = {
			(GInterfaceInitFunc) gnome_db_qf_func_renderer_init,
			NULL,
			NULL
		};

		static const GInterfaceInfo referer_info = {
			(GInterfaceInitFunc) gnome_db_qf_func_referer_init,
			NULL,
			NULL
		};
		
		
		type = g_type_register_static (GNOME_DB_QFIELD_TYPE, "GnomeDbQfFunc", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_XML_STORAGE_TYPE, &xml_storage_info);
		g_type_add_interface_static (type, GNOME_DB_FIELD_TYPE, &field_info);
		g_type_add_interface_static (type, GNOME_DB_RENDERER_TYPE, &renderer_info);
		g_type_add_interface_static (type, GNOME_DB_REFERER_TYPE, &referer_info);
	}
	return type;
}

static void 
gnome_db_qf_func_xml_storage_init (GnomeDbXmlStorageIface *iface)
{
	iface->get_xml_id = gnome_db_qf_func_get_xml_id;
	iface->save_to_xml = gnome_db_qf_func_save_to_xml;
	iface->load_from_xml = gnome_db_qf_func_load_from_xml;
}

static void
gnome_db_qf_func_field_init (GnomeDbFieldIface *iface)
{
	iface->get_entity = gnome_db_qf_func_get_entity;
	iface->get_data_type = gnome_db_qf_func_get_data_type;
}

static void
gnome_db_qf_func_renderer_init (GnomeDbRendererIface *iface)
{
	iface->render_as_xql = gnome_db_qf_func_render_as_xql;
	iface->render_as_sql = gnome_db_qf_func_render_as_sql;
	iface->render_as_str = gnome_db_qf_func_render_as_str;
	iface->is_valid = NULL;
}

static void
gnome_db_qf_func_referer_init (GnomeDbRefererIface *iface)
{
        iface->activate = gnome_db_qf_func_activate;
        iface->deactivate = gnome_db_qf_func_deactivate;
        iface->is_active = gnome_db_qf_func_is_active;
        iface->get_ref_objects = gnome_db_qf_func_get_ref_objects;
        iface->replace_refs = gnome_db_qf_func_replace_refs;
}

static void
gnome_db_qf_func_class_init (GnomeDbQfFuncClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	gnome_db_qf_func_signals[TEMPL_SIGNAL] =
		g_signal_new ("templ_signal",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbQfFuncClass, templ_signal),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE,
			      0);
	class->templ_signal = NULL;

	object_class->dispose = gnome_db_qf_func_dispose;
	object_class->finalize = gnome_db_qf_func_finalize;

	/* Properties */
	object_class->set_property = gnome_db_qf_func_set_property;
	object_class->get_property = gnome_db_qf_func_get_property;
	g_object_class_install_property (object_class, PROP_QUERY,
					 g_param_spec_pointer ("query", NULL, NULL, 
							       (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	/* virtual functions */
#ifdef debug
        GNOME_DB_BASE_CLASS (class)->dump = (void (*)(GnomeDbBase *, guint)) gnome_db_qf_func_dump;
#endif
	GNOME_DB_QFIELD_CLASS (class)->copy = gnome_db_qf_func_copy;
	GNOME_DB_QFIELD_CLASS (class)->is_equal = gnome_db_qf_func_is_equal;
	GNOME_DB_QFIELD_CLASS (class)->is_list = NULL;
	GNOME_DB_QFIELD_CLASS (class)->get_params = NULL;
}

static void
gnome_db_qf_func_init (GnomeDbQfFunc *gnome_db_qf_func)
{
	gnome_db_qf_func->priv = g_new0 (GnomeDbQfFuncPrivate, 1);
	gnome_db_qf_func->priv->query = NULL;
	gnome_db_qf_func->priv->func_ref = NULL;
	gnome_db_qf_func->priv->args = NULL;
}


/**
 * gnome_db_qf_func_new_with_func
 * @query: a #GnomeDbQuery in which the new object will be
 * @func: a #GnomeDbServerFunction object
 *
 * Creates a new GnomeDbQfFunc object which represents the @func function
 *
 * Returns: the new object
 */
GObject*
gnome_db_qf_func_new_with_func (GnomeDbQuery *query, GnomeDbServerFunction *func)
{
	GObject   *obj;
	GnomeDbQfFunc *gnome_db_qf_func;
	GnomeDbDict *dict;
	guint id;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (func && IS_GNOME_DB_SERVER_FUNCTION (func), NULL);

	dict = gnome_db_base_get_dict (GNOME_DB_BASE (query));

	obj = g_object_new (GNOME_DB_QF_FUNC_TYPE, "dict", dict, NULL);
	gnome_db_qf_func = GNOME_DB_QF_FUNC (obj);
	g_object_get (G_OBJECT (query), "field_serial", &id, NULL);
	gnome_db_base_set_id (GNOME_DB_BASE (gnome_db_qf_func), id);

	gnome_db_qf_func->priv->query = query;
	gnome_db_base_connect_nullify (query, 
				 G_CALLBACK (nullified_object_cb), gnome_db_qf_func);

	gnome_db_qf_func->priv->func_ref = GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));
	gnome_db_ref_base_set_ref_object (gnome_db_qf_func->priv->func_ref, GNOME_DB_BASE (func));
	
	return obj;
}

/**
 * gnome_db_qf_func_new_with_xml_id
 * @query: a #GnomeDbQuery in which the new object will be
 * @func_xml_id: the XML Id of a #GnomeDbServerFunction object
 *
 * Creates a new GnomeDbQfFunc object which represents a given function
 *
 * Returns: the new object
 */
GObject*
gnome_db_qf_func_new_with_xml_id (GnomeDbQuery *query, const gchar *func_xml_id)
{
	GObject   *obj;
	GnomeDbQfFunc *gnome_db_qf_func;
	GnomeDbDict *dict;
	guint id;

	g_return_val_if_fail (query && IS_GNOME_DB_QUERY (query), NULL);
	g_return_val_if_fail (func_xml_id && *func_xml_id, NULL);

	dict = gnome_db_base_get_dict (GNOME_DB_BASE (query));
	obj = g_object_new (GNOME_DB_QF_FUNC_TYPE, "dict", dict, NULL);
	gnome_db_qf_func = GNOME_DB_QF_FUNC (obj);
	g_object_get (G_OBJECT (query), "field_serial", &id, NULL);
	gnome_db_base_set_id (GNOME_DB_BASE (gnome_db_qf_func), id);

	gnome_db_qf_func->priv->query = query;
	gnome_db_base_connect_nullify (query, 
				 G_CALLBACK (nullified_object_cb), gnome_db_qf_func);

	gnome_db_qf_func->priv->func_ref = GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));
	gnome_db_ref_base_set_ref_name (gnome_db_qf_func->priv->func_ref, GNOME_DB_SERVER_FUNCTION_TYPE, 
				  REFERENCE_BY_XML_ID, func_xml_id);
	
	return obj;
}

static void 
nullified_object_cb (GObject *obj, GnomeDbQfFunc *func)
{
	gnome_db_base_nullify (GNOME_DB_BASE (func));
}

static void
gnome_db_qf_func_dispose (GObject *object)
{
	GnomeDbQfFunc *gnome_db_qf_func;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_QF_FUNC (object));

	gnome_db_qf_func = GNOME_DB_QF_FUNC (object);
	if (gnome_db_qf_func->priv) {
		gnome_db_base_nullify_check (GNOME_DB_BASE (object));

		if (gnome_db_qf_func->priv->args) {
			GSList *list = gnome_db_qf_func->priv->args;
			while (list) {
				g_object_unref (G_OBJECT (list->data));
				list = g_slist_next (list);
			}
			g_slist_free (gnome_db_qf_func->priv->args);
			gnome_db_qf_func->priv->args = NULL;
		}

		if (gnome_db_qf_func->priv->query) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (gnome_db_qf_func->priv->query),
							      G_CALLBACK (nullified_object_cb), gnome_db_qf_func);
			gnome_db_qf_func->priv->query = NULL;
		}

		if (gnome_db_qf_func->priv->func_ref) {
			g_object_unref (G_OBJECT (gnome_db_qf_func->priv->func_ref));
			gnome_db_qf_func->priv->func_ref = NULL;
		}
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_qf_func_finalize (GObject   * object)
{
	GnomeDbQfFunc *gnome_db_qf_func;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_QF_FUNC (object));

	gnome_db_qf_func = GNOME_DB_QF_FUNC (object);
	if (gnome_db_qf_func->priv) {
		g_free (gnome_db_qf_func->priv);
		gnome_db_qf_func->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_qf_func_set_property (GObject              *object,
			guint                 param_id,
			const GValue         *value,
			GParamSpec           *pspec)
{
	GnomeDbQfFunc *gnome_db_qf_func;
	gpointer ptr;

	gnome_db_qf_func = GNOME_DB_QF_FUNC (object);
	if (gnome_db_qf_func->priv) {
		switch (param_id) {
		case PROP_QUERY:
			ptr = g_value_get_pointer (value);
			g_return_if_fail (ptr && IS_GNOME_DB_QUERY (ptr));

			if (gnome_db_qf_func->priv->query) {
				if (gnome_db_qf_func->priv->query == GNOME_DB_QUERY (ptr))
					return;

				g_signal_handlers_disconnect_by_func (G_OBJECT (gnome_db_qf_func->priv->query),
								      G_CALLBACK (nullified_object_cb), gnome_db_qf_func);
			}

			gnome_db_qf_func->priv->query = GNOME_DB_QUERY (ptr);
			gnome_db_base_connect_nullify (ptr,
						 G_CALLBACK (nullified_object_cb), gnome_db_qf_func);
			break;
		}
	}
}

static void
gnome_db_qf_func_get_property (GObject              *object,
			guint                 param_id,
			GValue               *value,
			GParamSpec           *pspec)
{
	GnomeDbQfFunc *gnome_db_qf_func;
	gnome_db_qf_func = GNOME_DB_QF_FUNC (object);
	
	if (gnome_db_qf_func->priv) {
		switch (param_id) {
		case PROP_QUERY:
			g_value_set_pointer (value, gnome_db_qf_func->priv->query);
			break;
		}	
	}
}

static GObject *
gnome_db_qf_func_copy (GnomeDbQfield *orig)
{
	GnomeDbQfFunc *qf;
	GObject *obj;
	GSList *list;
	GnomeDbDict *dict;

	g_assert (IS_GNOME_DB_QF_FUNC (orig));
	qf = GNOME_DB_QF_FUNC (orig);

	obj = gnome_db_qf_func_new_with_xml_id (qf->priv->query, 
					  gnome_db_ref_base_get_ref_name (qf->priv->func_ref, NULL, NULL));
	if (gnome_db_base_get_name (GNOME_DB_BASE (orig)))
		gnome_db_base_set_name (GNOME_DB_BASE (obj), gnome_db_base_get_name (GNOME_DB_BASE (orig)));

	if (gnome_db_base_get_description (GNOME_DB_BASE (orig)))
		gnome_db_base_set_description (GNOME_DB_BASE (obj), gnome_db_base_get_description (GNOME_DB_BASE (orig)));


	/* arguments */
	dict = gnome_db_base_get_dict (GNOME_DB_BASE (orig));
	list = qf->priv->args;
	while (list) {
		GnomeDbRefBase *ref;
		const gchar *refname;
		GType type;

		refname = gnome_db_ref_base_get_ref_name (GNOME_DB_REF_BASE (list->data), &type, NULL);
		ref = GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));
		gnome_db_ref_base_set_ref_name (ref, type, REFERENCE_BY_XML_ID, refname);
		GNOME_DB_QF_FUNC (obj)->priv->args = g_slist_append (GNOME_DB_QF_FUNC (obj)->priv->args, ref);
		list = g_slist_next (list);
	}

	return obj;
}

static gboolean
gnome_db_qf_func_is_equal (GnomeDbQfield *qfield1, GnomeDbQfield *qfield2)
{
	const gchar *ref1, *ref2;
	gboolean retval;
	g_assert (IS_GNOME_DB_QF_FUNC (qfield1));
	g_assert (IS_GNOME_DB_QF_FUNC (qfield2));
	
	/* it is here assumed that qfield1 and qfield2 are of the same type and refer to the same
	   query */
	ref1 = gnome_db_ref_base_get_ref_name (GNOME_DB_QF_FUNC (qfield1)->priv->func_ref, NULL, NULL);
	ref2 = gnome_db_ref_base_get_ref_name (GNOME_DB_QF_FUNC (qfield2)->priv->func_ref, NULL, NULL);

	retval = !strcmp (ref1, ref2) ? TRUE : FALSE;
	if (retval) {
		TO_IMPLEMENT; /* arguments */
	}

	return retval;
}

/**
 * gnome_db_qf_func_get_ref_func
 * @func: a #GnomeDbQfFunc object
 *
 * Get the real #GnomeDbServerFunction object used by @func
 *
 * Returns: the #GnomeDbServerFunction object, or NULL if @func is not active
 */
GnomeDbServerFunction *
gnome_db_qf_func_get_ref_func (GnomeDbQfFunc *func)
{
	GnomeDbBase *base;
	g_return_val_if_fail (func && IS_GNOME_DB_QF_FUNC (func), NULL);
	g_return_val_if_fail (func->priv, NULL);

	base = gnome_db_ref_base_get_ref_object (func->priv->func_ref);
	if (base)
		return GNOME_DB_SERVER_FUNCTION (base);
	else
		return NULL;
}

/**
 * gnome_db_qf_func_set_args
 * @func: a #GnomeDbQfFunc object
 * @args: a list of #GnomeDbQfield objects
 *
 * Sets the argument(s) of @func. If @args is %NULL, then
 * all the arguments (if there was any) are removed.
 *
 * If @func is not active, then no check on the provided args
 * is performed.
 *
 * Returns: TRUE if no error occurred
 */
gboolean
gnome_db_qf_func_set_args (GnomeDbQfFunc *func, GSList *args)
{
	gboolean args_ok = TRUE;
	g_return_val_if_fail (func && IS_GNOME_DB_QF_FUNC (func), FALSE);
	g_return_val_if_fail (func->priv, FALSE);

	if (args && gnome_db_ref_base_activate (func->priv->func_ref)) {
		/* test for the right arguments */
		GnomeDbServerInfo *sinfo = gnome_db_server_get_server_info (gnome_db_dict_get_server (gnome_db_base_get_dict (GNOME_DB_BASE (func))));
		GnomeDbServerFunction *sfunc = GNOME_DB_SERVER_FUNCTION (gnome_db_ref_base_get_ref_object (func->priv->func_ref));
		GSList *arg = args, *list = gnome_db_server_function_get_arg_types (sfunc);
		
		if (g_slist_length (arg) != g_slist_length (list))
			args_ok = FALSE;

		while (GNOME_DB_FUNCTIONS_TEST_PARAMS_DO_TEST && arg && list && args_ok) {
			if (!sinfo || !sinfo->implicit_data_types_casts) {
				/* Strict tests */
				if (arg->data && list->data &&
				    list->data &&
				    (gnome_db_qfield_get_data_type (GNOME_DB_QFIELD (arg->data)) != GNOME_DB_SERVER_DATA_TYPE (list->data)))
					args_ok = FALSE;
			}
			else {
				/* GdaValueType compatibility test */
				if (arg->data && list->data &&
				    list->data &&
				    (gnome_db_server_data_type_get_gda_type (gnome_db_qfield_get_data_type (GNOME_DB_QFIELD (arg->data))) !=
				     gnome_db_server_data_type_get_gda_type (GNOME_DB_SERVER_DATA_TYPE (list->data))))
					args_ok = FALSE;
			}

			arg = g_slist_next (arg);
			list = g_slist_next (list);
		}
	}

	if (!args || args_ok) {
		/* Delete any pre-existing argument */
		if (func->priv->args) {
			GSList *list = func->priv->args;
			while (list) {
				g_object_unref (G_OBJECT (list->data));
				list = g_slist_next (list);
			}
			g_slist_free (func->priv->args);
			func->priv->args = NULL;
		}
	}

	if (args && args_ok) {
		/* add the new arguments */
		GSList *list = args;
		while (list) {
			GnomeDbRefBase *ref;
				
			ref = GNOME_DB_REF_BASE (gnome_db_ref_base_new (gnome_db_base_get_dict (GNOME_DB_BASE (func))));
			if (list->data)
				gnome_db_ref_base_set_ref_object (ref, GNOME_DB_BASE (list->data));
			func->priv->args = g_slist_append (func->priv->args, ref);
			list = g_slist_next (list);
		}
	}

	if (args_ok)
		gnome_db_referer_activate (GNOME_DB_REFERER (func));

	return args_ok;
}

/**
 * gnome_db_qf_func_get_args
 * @func: a #GnomeDbQfFunc object
 *
 * Get a list of the other #GnomeDbQfield objects which are arguments of @func. If some
 * of them are missing, then a %NULL is inserted where it should have been.
 *
 * Returns: a new list of arguments
 */
GSList *
gnome_db_qf_func_get_args (GnomeDbQfFunc *func)
{
	GSList *retval = NULL, *list;

	g_return_val_if_fail (func && IS_GNOME_DB_QF_FUNC (func), NULL);
	g_return_val_if_fail (func->priv, NULL);
	
	list = func->priv->args;
	while (list) {
		GnomeDbBase *base = NULL;

		if (list->data)
			base = gnome_db_ref_base_get_ref_object (GNOME_DB_REF_BASE (list->data));

		retval = g_slist_append (retval, base);
		list = g_slist_next (list);
	}

	return retval;
}

#ifdef debug
static void
gnome_db_qf_func_dump (GnomeDbQfFunc *func, guint offset)
{
	gchar *str;
	gint i;

	g_return_if_fail (func && IS_GNOME_DB_QF_FUNC (func));
	
        /* string for the offset */
        str = g_new0 (gchar, offset+1);
        for (i=0; i<offset; i++)
                str[i] = ' ';
        str[offset] = 0;

        /* dump */
        if (func->priv) {
                g_print ("%s" D_COL_H1 "GnomeDbQfFunc" D_COL_NOR " \"%s\" (%p, id=%d) ",
                         str, gnome_db_base_get_name (GNOME_DB_BASE (func)), func, gnome_db_base_get_id (GNOME_DB_BASE (func)));
		if (gnome_db_qf_func_is_active (GNOME_DB_REFERER (func)))
			g_print ("Active");
		else
			g_print (D_COL_ERR "Inactive" D_COL_NOR);
		if (gnome_db_qfield_is_visible (GNOME_DB_QFIELD (func)))
			g_print (", Visible\n");
		g_print ("\n");
	}
        else
                g_print ("%s" D_COL_ERR "Using finalized object %p" D_COL_NOR, str, func);
}
#endif


/* 
 * GnomeDbEntity interface implementation
 */
static GnomeDbEntity *
gnome_db_qf_func_get_entity (GnomeDbField *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_QF_FUNC (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_FUNC (iface)->priv, NULL);

	return GNOME_DB_ENTITY (GNOME_DB_QF_FUNC (iface)->priv->query);
}

static GnomeDbServerDataType *
gnome_db_qf_func_get_data_type (GnomeDbField *iface)
{
	g_return_val_if_fail (iface && IS_GNOME_DB_QF_FUNC (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_FUNC (iface)->priv, NULL);
	
	if (gnome_db_qf_func_activate (GNOME_DB_REFERER (iface))) {
		GnomeDbServerFunction *func;
		func = GNOME_DB_SERVER_FUNCTION (gnome_db_ref_base_get_ref_object (GNOME_DB_QF_FUNC (iface)->priv->func_ref));
		return gnome_db_server_function_get_ret_type (func);
	}

	return NULL;
}

/* 
 * GnomeDbXmlStorage interface implementation
 */
static gchar *
gnome_db_qf_func_get_xml_id (GnomeDbXmlStorage *iface)
{
	gchar *q_xml_id, *xml_id;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_FUNC (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_FUNC (iface)->priv, NULL);

	q_xml_id = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (GNOME_DB_QF_FUNC (iface)->priv->query));
	xml_id = g_strdup_printf ("%s:QF%d", q_xml_id, gnome_db_base_get_id (GNOME_DB_BASE (iface)));
	g_free (q_xml_id);
	
	return xml_id;
}

static xmlNodePtr
gnome_db_qf_func_save_to_xml (GnomeDbXmlStorage *iface, GError **error)
{
	xmlNodePtr node = NULL;
	GnomeDbQfFunc *func;
	gchar *str;
	GSList *list;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_FUNC (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_FUNC (iface)->priv, NULL);

	func = GNOME_DB_QF_FUNC (iface);

	node = xmlNewNode (NULL, "GNOME_DB_QF");
	
	str = gnome_db_qf_func_get_xml_id (iface);
	xmlSetProp (node, "id", str);
	g_free (str);

	xmlSetProp (node, "type", "FUNC");
	xmlSetProp (node, "name", gnome_db_base_get_name (GNOME_DB_BASE (func)));
	if (gnome_db_base_get_description (GNOME_DB_BASE (func)) && *gnome_db_base_get_description (GNOME_DB_BASE (func)))
		xmlSetProp (node, "descr", gnome_db_base_get_description (GNOME_DB_BASE (func)));
	xmlSetProp (node, "object", gnome_db_ref_base_get_ref_name (func->priv->func_ref, NULL, NULL));
	if (! gnome_db_qfield_is_visible (GNOME_DB_QFIELD (func)))
		xmlSetProp (node, "is_visible",  "f");
	if (gnome_db_qfield_is_internal (GNOME_DB_QFIELD (func)))
		xmlSetProp (node, "is_internal", "t");

	str = gnome_db_qfield_get_alias (GNOME_DB_QFIELD (func));
	if (str && *str) 
		xmlSetProp (node, "alias", str);

	/* function's arguments */
	list = func->priv->args;
	while (list) {
		xmlNodePtr argnode;

		argnode = xmlNewChild (node, NULL, "GNOME_DB_QF_REF", NULL);
		xmlSetProp (argnode, "object", gnome_db_ref_base_get_ref_name (GNOME_DB_REF_BASE (list->data), NULL, NULL));
		list = g_slist_next (list);
	}

	return node;
}

static gboolean
gnome_db_qf_func_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error)
{
	GnomeDbQfFunc *func;
	gchar *prop;
	gboolean funcref = FALSE;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_FUNC (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_QF_FUNC (iface)->priv, FALSE);
	g_return_val_if_fail (node, FALSE);

	func = GNOME_DB_QF_FUNC (iface);
	if (strcmp (node->name, "GNOME_DB_QF")) {
		g_set_error (error,
			     GNOME_DB_QF_FUNC_ERROR,
			     GNOME_DB_QF_FUNC_XML_LOAD_ERROR,
			     _("XML Tag is not <GNOME_DB_QF>"));
		return FALSE;
	}

	prop = xmlGetProp (node, "type");
	if (prop) {
		if (strcmp (prop, "FUNC")) {
			g_set_error (error,
				     GNOME_DB_QF_FUNC_ERROR,
				     GNOME_DB_QF_FUNC_XML_LOAD_ERROR,
				     _("Wrong type of function in <GNOME_DB_QF>"));
			return FALSE;
		}
		g_free (prop);
	}

	prop = xmlGetProp (node, "id");
	if (prop) {
		gchar *ptr, *tok;
		ptr = strtok_r (prop, ":", &tok);
		ptr = strtok_r (NULL, ":", &tok);
		if (strlen (ptr) < 3) {
			g_set_error (error,
				     GNOME_DB_QF_FUNC_ERROR,
				     GNOME_DB_QF_FUNC_XML_LOAD_ERROR,
				     _("Wrong 'id' attribute in <GNOME_DB_QF>"));
			return FALSE;
		}
		gnome_db_base_set_id (GNOME_DB_BASE (func), atoi (ptr+2));
		g_free (prop);
	}

	prop = xmlGetProp (node, "name");
	if (prop) {
		gnome_db_base_set_name (GNOME_DB_BASE (func), prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "descr");
	if (prop) {
		gnome_db_base_set_description (GNOME_DB_BASE (func), prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "object");
	if (prop) {
		funcref = TRUE;
		gnome_db_ref_base_set_ref_name (func->priv->func_ref, GNOME_DB_SERVER_FUNCTION_TYPE, REFERENCE_BY_XML_ID, prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "is_visible");
	if (prop) {
		gnome_db_qfield_set_visible (GNOME_DB_QFIELD (func), (*prop == 't') ? TRUE : FALSE);
		g_free (prop);
	}

	prop = xmlGetProp (node, "is_internal");
	if (prop) {
		gnome_db_qfield_set_internal (GNOME_DB_QFIELD (func), (*prop == 't') ? TRUE : FALSE);
		g_free (prop);
	}

	prop = xmlGetProp (node, "alias");
	if (prop) {
		gnome_db_qfield_set_alias (GNOME_DB_QFIELD (func), prop);
		g_free (prop);
	}
	
	/* function's arguments */
	if (node->children) {
		GnomeDbDict *dict;
		xmlNodePtr argnode = node->children;
		dict = gnome_db_base_get_dict (GNOME_DB_BASE (func));
		while (argnode) {
			if (!strcmp (argnode->name, "GNOME_DB_QF_REF")) {
				prop = xmlGetProp (argnode, "object");
				if (prop) {
					GnomeDbRefBase *ref;

					ref = GNOME_DB_REF_BASE (gnome_db_ref_base_new (dict));
					gnome_db_ref_base_set_ref_name (ref, GNOME_DB_FIELD_TYPE, REFERENCE_BY_XML_ID, prop);
					g_free (prop);
					func->priv->args = g_slist_append (func->priv->args, ref);
				}
			}
			argnode = argnode->next;
		}
	}
	
	if (funcref) {
		/* test the right number of arguments */
		GnomeDbBase *ref;
		
		ref = gnome_db_ref_base_get_ref_object (func->priv->func_ref);
		if (ref && 
		    (g_slist_length (func->priv->args) != 
		     g_slist_length (gnome_db_server_function_get_arg_types (GNOME_DB_SERVER_FUNCTION (ref))))) {
			g_set_error (error,
				     GNOME_DB_QF_FUNC_ERROR,
				     GNOME_DB_QF_FUNC_XML_LOAD_ERROR,
				     _("Wrong number of arguments for function %s"), gnome_db_base_get_name (ref));
			return FALSE;
		}
		return TRUE;
	}
	else {
		g_set_error (error,
			     GNOME_DB_QF_FUNC_ERROR,
			     GNOME_DB_QF_FUNC_XML_LOAD_ERROR,
			     _("Missing required attributes for <GNOME_DB_QF>"));
		return FALSE;
	}
}


/*
 * GnomeDbRenderer interface implementation
 */
static GdaXqlItem *
gnome_db_qf_func_render_as_xql (GnomeDbRenderer *iface, GnomeDbDataSet *context, GError **error)
{
	GdaXqlItem *node = NULL;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_FUNC (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_FUNC (iface)->priv, NULL);
	
	TO_IMPLEMENT;
	return node;
}

static gchar *
gnome_db_qf_func_render_as_sql (GnomeDbRenderer *iface, GnomeDbDataSet *context, guint options, GError **error)
{
	gchar *str = NULL;
	GnomeDbBase *base;
	GnomeDbQfFunc *func;
	gboolean err = FALSE;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_FUNC (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_FUNC (iface)->priv, NULL);
	func = GNOME_DB_QF_FUNC (iface);

	base = gnome_db_ref_base_get_ref_object (func->priv->func_ref);

	if (base) {
		GString *string;
		GSList *list;

		string = g_string_new (gnome_db_base_get_name (base));
		g_string_append (string, " (");
		list = func->priv->args;
		while (list && !err) {
			gchar *argstr;
			GnomeDbBase *argbase;

			if (list != func->priv->args)
				g_string_append (string, ", ");

			argbase = gnome_db_ref_base_get_ref_object (GNOME_DB_REF_BASE (list->data));
			if (argbase) {
				argstr = gnome_db_renderer_render_as_sql (GNOME_DB_RENDERER (argbase), context, 
								    options, error);
				if (argstr) {
					g_string_append (string, argstr);
					g_free (argstr);
				}
				else
					err = TRUE;
			}
			else {
				const gchar *tmpstr;
				tmpstr = gnome_db_ref_base_get_ref_name (GNOME_DB_REF_BASE (list->data), NULL, NULL);
				g_set_error (error,
					     GNOME_DB_QF_FUNC_ERROR,
					     GNOME_DB_QF_FUNC_RENDER_ERROR,
					     _("Can't find referenced field '%s'"), tmpstr);
				err = TRUE;
			}

			list = g_slist_next (list);
		}
		g_string_append (string, ")");
		str = string->str;
		g_string_free (string, FALSE);
	}
	else {
		g_set_error (error,
			     GNOME_DB_QF_FUNC_ERROR,
			     GNOME_DB_QF_FUNC_RENDER_ERROR,
			     _("Can't find function '%s'"), gnome_db_ref_base_get_ref_name (func->priv->func_ref,
										      NULL, NULL));
		err = TRUE;
	}

	if (err) {
		if (str)
			g_free (str);
		return NULL;
	}

	return str;
}

static gchar *
gnome_db_qf_func_render_as_str (GnomeDbRenderer *iface, GnomeDbDataSet *context)
{
	gchar *str = NULL;
	GnomeDbBase *base;
	GnomeDbQfFunc *func;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_FUNC (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_FUNC (iface)->priv, NULL);
	func = GNOME_DB_QF_FUNC (iface);

	base = gnome_db_ref_base_get_ref_object (func->priv->func_ref);

	if (base) {
		GString *string;
		GSList *list;

		string = g_string_new (gnome_db_base_get_name (base));
		g_string_append (string, " (");
		list = func->priv->args;
		while (list) {
			gchar *argstr;
			GnomeDbBase *argbase;

			if (list != func->priv->args)
				g_string_append (string, ", ");

			argbase = gnome_db_ref_base_get_ref_object (GNOME_DB_REF_BASE (list->data));
			if (argbase) {
				argstr = gnome_db_renderer_render_as_str (GNOME_DB_RENDERER (argbase), context);
				g_assert (argstr);
				g_string_append (string, argstr);
				g_free (argstr);
			}
			else {
				const gchar *tmpstr;
				tmpstr = gnome_db_ref_base_get_ref_name (GNOME_DB_REF_BASE (list->data), NULL, NULL);
				g_string_append (string, tmpstr);
			}

			list = g_slist_next (list);
		}
		g_string_append (string, ")");

		str = string->str;
		g_string_free (string, FALSE);

	}
	else 
		str = g_strdup (_("Non activated function"));
	
	return str;
}


/*
 * GnomeDbReferer interface implementation
 */
static gboolean
gnome_db_qf_func_activate (GnomeDbReferer *iface)
{
	gboolean active = FALSE;
	GnomeDbQfFunc *func;
	GSList *list;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_FUNC (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_QF_FUNC (iface)->priv, FALSE);
	func = GNOME_DB_QF_FUNC (iface);

	active = gnome_db_ref_base_activate (func->priv->func_ref);
	list = func->priv->args;
	while (list) {
		active = active && gnome_db_ref_base_activate (GNOME_DB_REF_BASE (list->data));
		list = g_slist_next (list);
	}

	return active;
}

static void
gnome_db_qf_func_deactivate (GnomeDbReferer *iface)
{
	GnomeDbQfFunc *func;
	GSList *list;

	g_return_if_fail (iface && IS_GNOME_DB_QF_FUNC (iface));
	g_return_if_fail (GNOME_DB_QF_FUNC (iface)->priv);
	func = GNOME_DB_QF_FUNC (iface);

	gnome_db_ref_base_deactivate (func->priv->func_ref);
	list = func->priv->args;
	while (list) {
		gnome_db_ref_base_deactivate (GNOME_DB_REF_BASE (list->data));
		list = g_slist_next (list);
	}
}

static gboolean
gnome_db_qf_func_is_active (GnomeDbReferer *iface)
{
	gboolean active;
	GnomeDbQfFunc *func;
	GSList *list;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_FUNC (iface), FALSE);
	g_return_val_if_fail (GNOME_DB_QF_FUNC (iface)->priv, FALSE);
	func = GNOME_DB_QF_FUNC (iface);

	active = gnome_db_ref_base_is_active (func->priv->func_ref);
	list = func->priv->args;
	while (list && active) {
		active = gnome_db_ref_base_is_active (GNOME_DB_REF_BASE (list->data)) && active;
		list = g_slist_next (list);
	}

	return active;
}

static GSList *
gnome_db_qf_func_get_ref_objects (GnomeDbReferer *iface)
{
	GSList *list = NULL;
        GnomeDbBase *base;
	GnomeDbQfFunc *func;
	GSList *args;

	g_return_val_if_fail (iface && IS_GNOME_DB_QF_FUNC (iface), NULL);
	g_return_val_if_fail (GNOME_DB_QF_FUNC (iface)->priv, NULL);
	func = GNOME_DB_QF_FUNC (iface);

        base = gnome_db_ref_base_get_ref_object (func->priv->func_ref);
        if (base)
                list = g_slist_append (list, base);

	args = func->priv->args;
	while (args) {
		base = gnome_db_ref_base_get_ref_object (GNOME_DB_REF_BASE (args->data));
		if (base)
			list = g_slist_append (list, base);
		args = g_slist_next (args);
	}

        return list;
}

static void
gnome_db_qf_func_replace_refs (GnomeDbReferer *iface, GHashTable *replacements)
{
	GnomeDbQfFunc *func;
	GSList *list;

        g_return_if_fail (iface && IS_GNOME_DB_QF_FUNC (iface));
        g_return_if_fail (GNOME_DB_QF_FUNC (iface)->priv);

        func = GNOME_DB_QF_FUNC (iface);
        if (func->priv->query) {
                GnomeDbQuery *query = g_hash_table_lookup (replacements, func->priv->query);
                if (query) {
                        g_signal_handlers_disconnect_by_func (G_OBJECT (func->priv->query),
                                                              G_CALLBACK (nullified_object_cb), func);
                        func->priv->query = query;
			gnome_db_base_connect_nullify (query, 
						 G_CALLBACK (nullified_object_cb), func);
                }
        }

        gnome_db_ref_base_replace_ref_object (func->priv->func_ref, replacements);
	list = func->priv->args;
	while (list) {
		gnome_db_ref_base_replace_ref_object (GNOME_DB_REF_BASE (list->data), replacements);
		list = g_slist_next (list);
	}
}
