/* gnome-db-graph-item.c
 *
 * Copyright (C) 2004 - 2005 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include "gnome-db-graph-item.h"
#include <libgnomedb/gnome-db-ref-base.h>
#include <libgnomedb/gnome-db-xml-storage.h>
#include "marshal.h"

/* 
 * Main static functions 
 */
static void gnome_db_graph_item_class_init (GnomeDbGraphItemClass *class);
static void gnome_db_graph_item_init (GnomeDbGraphItem *graph);
static void gnome_db_graph_item_dispose (GObject *object);
static void gnome_db_graph_item_finalize (GObject *object);

static void gnome_db_graph_item_set_property (GObject              *object,
					guint                 param_id,
					const GValue         *value,
					GParamSpec           *pspec);
static void gnome_db_graph_item_get_property (GObject              *object,
					guint                 param_id,
					GValue               *value,
					GParamSpec           *pspec);

/* XML storage interface */
static void        gnome_db_graph_item_xml_storage_init (GnomeDbXmlStorageIface *iface);
static xmlNodePtr  gnome_db_graph_item_save_to_xml (GnomeDbXmlStorage *iface, GError **error);
static gboolean    gnome_db_graph_item_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error);


#ifdef debug
static void        gnome_db_graph_item_dump                (GnomeDbGraphItem *graph, guint offset);
#endif

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	MOVED,
	LAST_SIGNAL
};

static gint gnome_db_graph_item_signals[LAST_SIGNAL] = { 0 };

/* properties */
enum
{
	PROP_0,
	PROP_REF_OBJECT
};


struct _GnomeDbGraphItemPrivate
{
	GnomeDbRefBase    *ref_object;
	gdouble       x;
	gdouble       y;
};

/* module error */
GQuark gnome_db_graph_item_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("gnome_db_graph_item_error");
	return quark;
}


GType
gnome_db_graph_item_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbGraphItemClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_graph_item_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbGraphItem),
			0,
			(GInstanceInitFunc) gnome_db_graph_item_init
		};
		
		static const GInterfaceInfo xml_storage_info = {
                        (GInterfaceInitFunc) gnome_db_graph_item_xml_storage_init,
                        NULL,
                        NULL
                };

		type = g_type_register_static (GNOME_DB_BASE_TYPE, "GnomeDbGraphItem", &info, 0);
		g_type_add_interface_static (type, GNOME_DB_XML_STORAGE_TYPE, &xml_storage_info);
	}
	return type;
}

static void
gnome_db_graph_item_xml_storage_init (GnomeDbXmlStorageIface *iface)
{
        iface->get_xml_id = NULL;
        iface->save_to_xml = gnome_db_graph_item_save_to_xml;
        iface->load_from_xml = gnome_db_graph_item_load_from_xml;
}


static void
gnome_db_graph_item_class_init (GnomeDbGraphItemClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);
	
	gnome_db_graph_item_signals[MOVED] =
		g_signal_new ("moved",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbGraphItemClass, moved),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE, 0);

	object_class->dispose = gnome_db_graph_item_dispose;
	object_class->finalize = gnome_db_graph_item_finalize;

	/* Properties */
	object_class->set_property = gnome_db_graph_item_set_property;
	object_class->get_property = gnome_db_graph_item_get_property;

	g_object_class_install_property (object_class, PROP_REF_OBJECT,
					 g_param_spec_pointer ("ref_object", NULL, NULL, 
							       (G_PARAM_READABLE | G_PARAM_WRITABLE)));

	/* virtual functions */
#ifdef debug
        GNOME_DB_BASE_CLASS (class)->dump = (void (*)(GnomeDbBase *, guint)) gnome_db_graph_item_dump;
#endif
}

static void
gnome_db_graph_item_init (GnomeDbGraphItem *graph)
{
	graph->priv = g_new0 (GnomeDbGraphItemPrivate, 1);
	graph->priv->ref_object = NULL;
	graph->priv->x = 0.;
	graph->priv->y = 0.;
}

static void ref_lost_cb (GnomeDbRefBase *ref, GnomeDbGraphItem *graph);

/**
 * gnome_db_graph_item_new
 * @dict: a #GnomeDbDict object
 * @ref_obj: a #GnomeDbBase object which the new item will reference, or %NULL.
 *
 * Creates a new #GnomeDbGraphItem object
 *
 * Returns: the newly created object
 */
GObject   *
gnome_db_graph_item_new (GnomeDbDict *dict, GnomeDbBase *ref_obj)
{
	GObject *obj = NULL;
	GnomeDbGraphItem *item;

	g_return_val_if_fail (!dict || IS_GNOME_DB_DICT (dict), NULL);
	if (ref_obj)
		g_return_val_if_fail (IS_GNOME_DB_BASE (ref_obj), NULL);

	obj = g_object_new (GNOME_DB_GRAPH_ITEM_TYPE, "dict", ASSERT_DICT (dict), NULL);
	item = GNOME_DB_GRAPH_ITEM (obj);

	item->priv->ref_object = GNOME_DB_REF_BASE (gnome_db_ref_base_new (ASSERT_DICT (dict)));
	if (ref_obj) 
		gnome_db_ref_base_set_ref_object (item->priv->ref_object, ref_obj);

	g_signal_connect (G_OBJECT (item->priv->ref_object), "ref-lost",
			  G_CALLBACK (ref_lost_cb), item);

	return obj;
}

static void
ref_lost_cb (GnomeDbRefBase *ref, GnomeDbGraphItem *graph)
{
	gnome_db_base_nullify (GNOME_DB_BASE (graph));
}

static void
gnome_db_graph_item_dispose (GObject *object)
{
	GnomeDbGraphItem *graph;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_GRAPH_ITEM (object));

	graph = GNOME_DB_GRAPH_ITEM (object);
	if (graph->priv) {		
		if (graph->priv->ref_object) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (graph->priv->ref_object),
							      G_CALLBACK (ref_lost_cb), graph);
			g_object_unref (G_OBJECT (graph->priv->ref_object));
                        graph->priv->ref_object = NULL;
                }
	}

	/* parent class */
	parent_class->dispose (object);
}


static void
gnome_db_graph_item_finalize (GObject   * object)
{
	GnomeDbGraphItem *graph;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_GNOME_DB_GRAPH_ITEM (object));

	graph = GNOME_DB_GRAPH_ITEM (object);
	if (graph->priv) {
		g_free (graph->priv);
		graph->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
gnome_db_graph_item_set_property (GObject              *object,
			   guint                 param_id,
			   const GValue         *value,
			   GParamSpec           *pspec)
{
	GnomeDbGraphItem *graph;
	gpointer ptr;

	graph = GNOME_DB_GRAPH_ITEM (object);
	if (graph->priv) {
		switch (param_id) {
		case PROP_REF_OBJECT:
			g_assert (graph->priv->ref_object);
			if (graph->priv->ref_object) {
				ptr = g_value_get_pointer (value);
				gnome_db_ref_base_set_ref_object (graph->priv->ref_object, ptr);
			}
			break;
		}
	}
}

static void
gnome_db_graph_item_get_property (GObject              *object,
			   guint                 param_id,
			   GValue               *value,
			   GParamSpec           *pspec)
{
	GnomeDbGraphItem *graph;

	graph = GNOME_DB_GRAPH_ITEM (object);
        if (graph->priv) {
                switch (param_id) {
                case PROP_REF_OBJECT:
			g_assert (graph->priv->ref_object);
			g_value_set_pointer (value, 
					     gnome_db_ref_base_get_ref_object (graph->priv->ref_object));
                        break;
                }
        }
}

/**
 * gnome_db_graph_item_set_position
 * @item: a #GnomeDbGraphItemItem object
 * @x:
 * @y:
 *
 * Sets the position to be remembered for @item.
 */
void
gnome_db_graph_item_set_position (GnomeDbGraphItem *item, gdouble x, gdouble y)
{
	g_return_if_fail (item && IS_GNOME_DB_GRAPH_ITEM (item));
	g_return_if_fail (item->priv);
	
	if ((item->priv->x == x) && (item->priv->y == y))
		return;

	item->priv->x = x;
	item->priv->y = y;
	
#ifdef debug_signal
	g_print (">> 'MOVED' from %s::%s()\n", __FILE__, __FUNCTION__);
#endif
	g_signal_emit (G_OBJECT (item), gnome_db_graph_item_signals[MOVED], 0);
#ifdef debug_signal
	g_print ("<< 'MOVED' from %s::%s()\n", __FILE__, __FUNCTION__);
#endif
}

/**
 * gnome_db_graph_item_get_position
 * @item: a #GnomeDbGraphItemItem object
 * @x: a place where to store the X part of the position, or %NULL
 * @y: a place where to store the Y part of the position, or %NULL
 *
 * Get @item's position.
 */
void
gnome_db_graph_item_get_position (GnomeDbGraphItem *item, gdouble *x, gdouble *y)
{
	g_return_if_fail (item && IS_GNOME_DB_GRAPH_ITEM (item));
	g_return_if_fail (item->priv);
	
	if (x)
		*x = item->priv->x;
	if (y)
		*y = item->priv->y;
}


/**
 * gnome_db_graph_item_get_ref_object
 * @item: a #GnomeDbGraphItem object
 *
 * Get the referenced #GnomeDbBase object, if it exists.
 *
 * Returns: the referenced object, or %NULL
 */
GnomeDbBase *
gnome_db_graph_item_get_ref_object (GnomeDbGraphItem *item)
{
	g_return_val_if_fail (item && IS_GNOME_DB_GRAPH_ITEM (item), NULL);
	g_return_val_if_fail (item->priv, NULL);
	
	return gnome_db_ref_base_get_ref_object (item->priv->ref_object);
}

#ifdef debug
static void
gnome_db_graph_item_dump (GnomeDbGraphItem *item, guint offset)
{
	gchar *str;
	gint i;

	g_return_if_fail (item && IS_GNOME_DB_GRAPH_ITEM (item));
	
        /* string for the offset */
        str = g_new0 (gchar, offset+1);
        for (i=0; i<offset; i++)
                str[i] = ' ';
        str[offset] = 0;

        /* dump */
        if (item->priv) {
		GnomeDbBase *base = gnome_db_ref_base_get_ref_object (item->priv->ref_object);
		g_print ("%s" D_COL_H1 "GnomeDbGraphItem" D_COL_NOR " for %p at (%.3f, %.3f) ", str, base,
			 item->priv->x, item->priv->y);
		
	}
        else
                g_print ("%s" D_COL_ERR "Using finalized object %p" D_COL_NOR, str, item);
}
#endif

/* 
 * GnomeDbXmlStorage interface implementation
 */
static xmlNodePtr
gnome_db_graph_item_save_to_xml (GnomeDbXmlStorage *iface, GError **error)
{
        xmlNodePtr node = NULL;
	GnomeDbGraphItem *item;
        gchar *str;
	GnomeDbBase *base;

        g_return_val_if_fail (iface && IS_GNOME_DB_GRAPH_ITEM (iface), NULL);
        g_return_val_if_fail (GNOME_DB_GRAPH_ITEM (iface)->priv, NULL);

        item = GNOME_DB_GRAPH_ITEM (iface);

        node = xmlNewNode (NULL, "GNOME_DB_GRAPH_ITEM");

	g_assert (item->priv->ref_object);
	base = gnome_db_ref_base_get_ref_object (item->priv->ref_object);
	if (base) {
		str = gnome_db_xml_storage_get_xml_id (GNOME_DB_XML_STORAGE (base));
		xmlSetProp (node, "object", str);
		g_free (str);
	}
	
	str = g_strdup_printf ("%.3f", item->priv->x);
	xmlSetProp (node, "xpos", str);
	g_free (str);

	str = g_strdup_printf ("%.3f", item->priv->y);
	xmlSetProp (node, "ypos", str);
	g_free (str);

        return node;
}

static gboolean
gnome_db_graph_item_load_from_xml (GnomeDbXmlStorage *iface, xmlNodePtr node, GError **error)
{
	GnomeDbGraphItem *item;
        gchar *prop;

        g_return_val_if_fail (iface && IS_GNOME_DB_GRAPH_ITEM (iface), FALSE);
        g_return_val_if_fail (GNOME_DB_GRAPH_ITEM (iface)->priv, FALSE);
        g_return_val_if_fail (node, FALSE);

	item = GNOME_DB_GRAPH_ITEM (iface);

	if (strcmp (node->name, "GNOME_DB_GRAPH_ITEM")) {
                g_set_error (error,
                             GNOME_DB_GRAPH_ITEM_ERROR,
                             GNOME_DB_GRAPH_ITEM_XML_LOAD_ERROR,
                             _("XML Tag is not <GNOME_DB_GRAPH_ITEM>"));
                return FALSE;
        }

	prop = xmlGetProp (node, "object");
	if (prop) {
		g_assert (item->priv->ref_object);
		gnome_db_ref_base_set_ref_name (item->priv->ref_object, 0/* FIXME */, REFERENCE_BY_XML_ID, prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "xpos");
	if (prop) {
		item->priv->x = atof (prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "ypos");
	if (prop) {
		item->priv->y = atof (prop);
		g_free (prop);
	}

        return TRUE;
}

